#!/bin/bash

# Color definitions
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
NC='\033[0m'

echo -e "${YELLOW}AIRUN 설치를 위해 비밀번호를 입력해주세요.${NC}"
sudo -v
echo -e "${GREEN}Starting AI.RUN installation...${NC}"

# PostgreSQL Docker 설치 및 설정
setup_postgresql_docker() {
    echo -e "${GREEN}PostgreSQL Docker 설치 및 설정을 시작합니다...${NC}"

    # Docker 설치 확인
    if ! command -v docker &>/dev/null; then
        echo -e "${YELLOW}Docker가 설치되어 있지 않습니다. Docker를 설치합니다...${NC}"

        # 운영체제 확인
        if [ -f /etc/debian_version ]; then
            # Debian/Ubuntu 기반
            sudo apt-get update
            sudo apt-get install -y apt-transport-https ca-certificates curl gnupg lsb-release

            # GPG 키 설정
            curl -fsSL https://download.docker.com/linux/ubuntu/gpg |
                sudo gpg --dearmor -o /usr/share/keyrings/docker-archive-keyring.gpg

            # 배포판 이름 확인
            distro_codename=$(lsb_release -cs)

            # HamoniKR wilma → jammy로 처리
            if [[ "$distro_codename" == "wilma" ]]; then
                docker_codename="jammy"
            elif [[ "$distro_codename" == "noble" ]]; then
                docker_codename="noble"
            else
                docker_codename="$distro_codename"
            fi

            # Docker 저장소 등록
            echo "deb [arch=$(dpkg --print-architecture) signed-by=/usr/share/keyrings/docker-archive-keyring.gpg] \
https://download.docker.com/linux/ubuntu $docker_codename stable" |
                sudo tee /etc/apt/sources.list.d/docker.list >/dev/null

            sudo apt-get update
            sudo apt-get install -y docker-ce docker-ce-cli containerd.io

        elif [ -f /etc/redhat-release ]; then
            # RHEL/CentOS
            sudo yum install -y yum-utils
            sudo yum-config-manager --add-repo https://download.docker.com/linux/centos/docker-ce.repo
            sudo yum install -y docker-ce docker-ce-cli containerd.io

        else
            echo -e "${RED}지원되지 않는 운영체제입니다. Docker를 수동으로 설치해주세요.${NC}"
            return 1
        fi

        # Docker 서비스 시작 및 사용자 그룹 추가
        sudo systemctl start docker
        sudo systemctl enable docker
        sudo usermod -aG docker $USER

        echo -e "${YELLOW}Docker 설치 완료. 로그아웃 후 다시 로그인하세요.${NC}"
        exit 0

    else
        echo -e "${GREEN}Docker가 이미 설치되어 있습니다.${NC}"
    fi
}

setup_airun_postgres() {
    # PostgreSQL Docker 컨테이너 설정
    CONTAINER_NAME="airun-postgres"
    PG_HOST="localhost"
    PG_PORT="5433"
    PG_USER="ivs"
    PG_PASSWORD="1234"
    PG_DATABASE="airun"

    # 기존 컨테이너 확인 및 제거
    if docker ps -a | grep -q $CONTAINER_NAME; then
        echo -e "${YELLOW}기존 PostgreSQL 컨테이너가 발견되었습니다. 제거합니다...${NC}"
        docker stop $CONTAINER_NAME 2>/dev/null || true
        docker rm $CONTAINER_NAME 2>/dev/null || true
    fi

    # PostgreSQL 데이터 볼륨 생성
    POSTGRES_DATA_DIR="$HOME/.airun/postgres-data"
    mkdir -p $POSTGRES_DATA_DIR

    # PostgreSQL Docker 컨테이너 실행 (pgvector 포함)
    echo -e "${YELLOW}PostgreSQL Docker 컨테이너를 시작합니다 (pgvector 포함)...${NC}"
    docker run --name $CONTAINER_NAME \
        -e POSTGRES_USER=$PG_USER \
        -e POSTGRES_PASSWORD=$PG_PASSWORD \
        -e POSTGRES_DB=$PG_DATABASE \
        -p $PG_PORT:5432 \
        -v $POSTGRES_DATA_DIR:/var/lib/postgresql/data \
        -d pgvector/pgvector:0.8.0-pg17

    # 컨테이너 실행 확인
    if docker ps | grep -q $CONTAINER_NAME; then
        echo -e "${GREEN}PostgreSQL Docker 컨테이너가 성공적으로 실행되었습니다.${NC}"

        # 컨테이너가 완전히 시작될 때까지 기다림
        echo -e "${YELLOW}PostgreSQL 서버가 시작될 때까지 잠시 기다립니다...${NC}"
        sleep 10

        # 연결 테스트
        if docker exec $CONTAINER_NAME pg_isready -h localhost -U $PG_USER >/dev/null 2>&1; then
            echo -e "${GREEN}PostgreSQL 서버에 성공적으로 연결되었습니다.${NC}"
        else
            echo -e "${RED}PostgreSQL 서버 연결 테스트에 실패했습니다.${NC}"
            return 1
        fi

        echo -e "${GREEN}pgvector 확장 활성화 중...${NC}"
        
        # pgvector 확장 활성화 (이미 설치되어 있으므로 활성화만 필요)
        docker exec $CONTAINER_NAME psql -U $PG_USER -d $PG_DATABASE -c "CREATE EXTENSION IF NOT EXISTS vector;"
        
        echo -e "${GREEN}pgvector 확장 활성화 완료!${NC}"

        # .env 파일 생성
        ENV_FILE=".env"

        echo -e "${YELLOW}.env 파일을 생성합니다: $ENV_FILE${NC}"
        cat >$ENV_FILE <<EOF
PG_HOST=$PG_HOST
PG_PORT=$PG_PORT
PG_USER=$PG_USER
PG_PASSWORD=$PG_PASSWORD
PG_DATABASE=$PG_DATABASE
EOF
        echo -e "${GREEN}PostgreSQL 환경 설정 파일이 생성되었습니다: $ENV_FILE${NC}"

        # 컨테이너 자동 시작 설정
        echo -e "${YELLOW}시스템 부팅 시 PostgreSQL Docker 컨테이너가 자동으로 시작되도록 설정합니다...${NC}"
        docker update --restart always $CONTAINER_NAME

        echo -e "${GREEN}PostgreSQL Docker 설정이 완료되었습니다.${NC}"
        echo -e "${GREEN}호스트: $PG_HOST${NC}"
        echo -e "${GREEN}포트: $PG_PORT${NC}"
        echo -e "${GREEN}사용자: $PG_USER${NC}"
        echo -e "${GREEN}데이터베이스: $PG_DATABASE${NC}"

        # 기존 .env 파일에 데이터베이스 설정 추가
        echo -e "${GREEN}프로젝트 .env 파일에 데이터베이스 설정을 추가합니다...${NC}"
        CURRENT_DIR="$(cd "$(dirname "$0")" && pwd)"
        ENV_FILE="$CURRENT_DIR/.env"
        
        # .env 파일이 없으면 생성
        if [ ! -f "$ENV_FILE" ]; then
            touch "$ENV_FILE"
        fi
        
        # 데이터베이스 설정이 이미 있는지 확인
        if ! grep -q "PG_HOST" "$ENV_FILE" 2>/dev/null; then
            echo "" >> "$ENV_FILE"
            echo "# 데이터베이스 설정 (자동 추가됨)" >> "$ENV_FILE"
            echo "PG_HOST=$PG_HOST" >> "$ENV_FILE"
            echo "PG_PORT=$PG_PORT" >> "$ENV_FILE"
            echo "PG_USER=$PG_USER" >> "$ENV_FILE"
            echo "PG_PASSWORD=$PG_PASSWORD" >> "$ENV_FILE"
            echo "PG_DATABASE=$PG_DATABASE" >> "$ENV_FILE"
            echo "" >> "$ENV_FILE"
            echo "# Redis 설정" >> "$ENV_FILE"
            echo "REDIS_HOST=localhost" >> "$ENV_FILE"
            echo "REDIS_PORT=6379" >> "$ENV_FILE"
            echo "" >> "$ENV_FILE"
            echo "# 웹검색 서버 설정" >> "$ENV_FILE"
            echo "WEBSEARCH_SERVER_PORT=5610" >> "$ENV_FILE"
            echo "WEBSEARCH_SERVER_HOST=0.0.0.0" >> "$ENV_FILE"
            echo "WEBSEARCH_SERVER_WORKERS=1" >> "$ENV_FILE"
            echo "" >> "$ENV_FILE"
            echo "# Report 서버 설정" >> "$ENV_FILE"
            echo "REPORT_SERVER_PORT=5620" >> "$ENV_FILE"
            echo "REPORT_SERVER_HOST=0.0.0.0" >> "$ENV_FILE"
            echo "REPORT_SERVER_WORKERS=1" >> "$ENV_FILE"
            echo "MAX_CONCURRENT_JOBS=10" >> "$ENV_FILE"
            echo "CACHE_EXPIRY_HOURS=24" >> "$ENV_FILE"
            echo "" >> "$ENV_FILE"
            echo "# 로그 레벨" >> "$ENV_FILE"
            echo "LOG_LEVEL=INFO" >> "$ENV_FILE"
            
            echo -e "${GREEN}데이터베이스 설정이 .env 파일에 추가되었습니다: $ENV_FILE${NC}"
        else
            echo -e "${YELLOW}데이터베이스 설정이 이미 .env 파일에 존재합니다.${NC}"
        fi

        return 0
    else
        echo -e "${RED}PostgreSQL Docker 컨테이너 실행에 실패했습니다.${NC}"
        return 1
    fi
}

# PostgreSQL Docker 설치
setup_postgresql_docker

# PostgreSQL 설정
setup_airun_postgres

# Remove existing completion files
echo -e "${YELLOW}Removing existing completion files...${NC}"
# Remove from system directories (requires sudo)
sudo rm -f /etc/bash_completion.d/airun 2>/dev/null || true
sudo rm -f /usr/share/bash-completion/completions/airun 2>/dev/null || true
sudo rm -f /usr/local/share/bash-completion/completions/airun 2>/dev/null || true
sudo rm -f /usr/share/zsh/site-functions/_airun 2>/dev/null || true
sudo rm -f /usr/local/share/zsh/site-functions/_airun 2>/dev/null || true

# Remove from user directories
rm -f ~/.local/share/bash-completion/completions/airun 2>/dev/null || true
rm -f ~/.zsh/completion/_airun 2>/dev/null || true
rm -rf ~/.zsh/completion 2>/dev/null || true

# Remove completion related lines from shell config files
if [ -f ~/.bashrc ]; then
    sed -i '/# AI.RUN bash completion/,+5d' ~/.bashrc 2>/dev/null || true
fi

if [ -f ~/.zshrc ]; then
    sed -i '/# AI.RUN zsh completion/,+5d' ~/.zshrc 2>/dev/null || true
fi

# Remove existing airun installation
echo -e "${YELLOW}Removing existing airun installation...${NC}"

# Remove global npm packages and links
echo -e "${YELLOW}Removing global npm packages...${NC}"
# Remove global installation from current directory
sudo npm uninstall -g . 2>/dev/null || true
# Remove by package name
sudo npm uninstall -g airun 2>/dev/null || true
# Remove any remaining links
sudo npm unlink -g airun 2>/dev/null || true
# Clean npm cache
npm cache clean --force 2>/dev/null || true
# Remove from global node_modules
sudo rm -rf /usr/local/lib/node_modules/airun 2>/dev/null || true
sudo rm -rf /usr/lib/node_modules/airun 2>/dev/null || true

# Remove local installation files
echo -e "${YELLOW}Removing local installation files...${NC}"
rm -rf node_modules package-lock.json 2>/dev/null || true

# Remove Python virtual environment
echo -e "${YELLOW}Removing Python virtual environment...${NC}"
deactivate 2>/dev/null || true
rm -rf "$HOME/.airun_venv" 2>/dev/null || true

# Remove configuration and cache directories
echo -e "${YELLOW}Removing configuration and cache directories...${NC}"
rm -rf "$HOME/.airun" 2>/dev/null || true

# Remove desktop integration files
echo -e "${YELLOW}Removing desktop integration files...${NC}"
rm -f "$HOME/Desktop/airun.desktop" "$HOME/바탕화면/airun.desktop" 2>/dev/null || true
rm -f "$HOME/.local/share/applications/airun.desktop" 2>/dev/null || true
rm -f "$HOME/.local/share/icons/airun.png" 2>/dev/null || true
update-desktop-database "$HOME/.local/share/applications" 2>/dev/null || true

# Remove any remaining global installations
echo -e "${YELLOW}Checking for remaining global installations...${NC}"
if command -v pip &>/dev/null; then
    pip uninstall -y airun 2>/dev/null || true
fi

# Check and install Chrome
echo -e "${GREEN}Checking Chrome installation...${NC}"
# dpkg가 있는 경우(주로 Debian/Ubuntu)
if command -v dpkg &>/dev/null; then
    if ! dpkg -l | grep -q "^ii.*google-chrome-stable"; then
        echo -e "${YELLOW}Installing Chrome...${NC}"
        if [ -f /etc/debian_version ]; then
            wget -q https://dl.google.com/linux/direct/google-chrome-stable_current_amd64.deb
            sudo apt-get install -y ./google-chrome-stable_current_amd64.deb >/dev/null 2>&1
            rm -f google-chrome-stable_current_amd64.deb
        fi
    else
        echo -e "${GREEN}Chrome is already installed${NC}"
    fi
fi

# /etc/redhat-release가 있는 경우(RHEL/CentOS 등)
if [ -f /etc/redhat-release ]; then
    # Red Hat 계열은 dpkg 대신 rpm/yum/dnf를 사용하므로 별도 체크 필요
    if ! rpm -q google-chrome-stable &>/dev/null; then
        echo -e "${YELLOW}Installing Chrome...${NC}"
        if command -v dnf &>/dev/null; then
            sudo dnf install -y chromium chromium-headless chromedriver >/dev/null 2>&1
        else
            sudo yum install -y chromium chromium-headless chromedriver >/dev/null 2>&1
        fi
    else
        echo -e "${GREEN}Chrome is already installed${NC}"
    fi
fi

# Check Chrome version
CHROME_VERSION=$(google-chrome --version 2>/dev/null || echo "Chrome not found")
echo -e "${GREEN}Chrome version: $CHROME_VERSION${NC}"

# Install system packages (RAG related)
echo -e "${GREEN}Installing system packages...${NC}"
if [ -f /etc/debian_version ]; then
    #  hamonikr: libreoffice-hwp2odt 추가
    wget -qO- https://repo.hamonikr.org/hamonikr-app.apt | sudo /bin/bash -

    sudo apt update
    sudo apt install -y \
        build-essential \
        pkg-config \
        python3.12 \
        python3.12-dev \
        libpython3.12-dev \
        python3-dev \
        python3-venv \
        python3-pip \
        python3-bs4 \
        python3-lxml \
        python-is-python3 \
        libxml2-dev \
        libxslt1-dev \
        libcairo2-dev \
        libgirepository1.0-dev \
        poppler-utils \
        libreoffice \
        libreoffice-hwp2odt \
        unoconv \
        pandoc \
        wkhtmltopdf \
        ghostscript \
        tesseract-ocr \
        tesseract-ocr-kor \
        fonts-nanum* \
        fonts-noto-cjk \
        curl \
        inotify-tools \
        graphviz \
        imagemagick \
        redis-server \
        python3-tk \
        ghostscript \
        antiword

elif [ -f /etc/redhat-release ]; then
    sudo yum install -y \
        gcc \
        gcc-c++ \
        pkg-config \
        python3-devel \
        python3-pip \
        python3-venv \
        python3-bs4 \
        python3-lxml \
        libxml2-devel \
        libxslt-devel \
        cairo-devel \
        gobject-introspection-devel \
        poppler-utils \
        libreoffice \
        unoconv \
        pandoc \
        wkhtmltopdf \
        ghostscript \
        tesseract \
        tesseract-langpack-kor \
        nanum-fonts* \
        google-noto-cjk-fonts \
        curl \
        inotify-tools \
        graphviz \
        ImageMagick \
        redis-server \
        python3-tk \
        ghostscript \
        antiword
fi

# Redis 설치 및 검증 함수
setup_redis() {
    echo -e "${GREEN}Redis 설정을 확인합니다...${NC}"

    # Redis 서버 설치 확인 및 설치
    if ! command -v redis-server &>/dev/null; then
        echo -e "${YELLOW}Redis 서버를 설치합니다...${NC}"
        if [ -f /etc/debian_version ]; then
            sudo apt-get update
            sudo apt-get install -y redis-server
        elif [ -f /etc/redhat-release ]; then
            sudo yum install -y redis
        else
            echo -e "${RED}지원되지 않는 운영체제입니다.${NC}"
            return 1
        fi
    fi

    # Redis 서비스 상태 확인 및 시작
    if command -v systemctl &>/dev/null; then
        if ! systemctl is-active --quiet redis-server; then
            echo -e "${YELLOW}Redis 서버를 시작합니다...${NC}"
            sudo systemctl start redis-server
            sudo systemctl enable redis-server
        fi
    else
        # systemd가 없는 경우 수동으로 Redis 시작
        if ! pgrep redis-server >/dev/null; then
            echo -e "${YELLOW}Redis 서버를 수동으로 시작합니다...${NC}"
            sudo redis-server --daemonize yes
        fi
    fi

    # Redis 퍼시스턴스 설정 추가
    setup_redis_persistence

    # Redis 연결 테스트
    local max_attempts=5
    local attempt=1
    while [ $attempt -le $max_attempts ]; do
        if redis-cli ping | grep -q "PONG"; then
            echo -e "${GREEN}Redis 서버가 정상적으로 실행 중입니다.${NC}"
            return 0
        fi
        echo -e "${YELLOW}Redis 서버 연결 시도 중... ($attempt/$max_attempts)${NC}"
        sleep 2
        attempt=$((attempt + 1))
    done

    echo -e "${RED}Redis 서버 연결에 실패했습니다.${NC}"
    return 1
}

# Redis 퍼시스턴스 설정 추가 함수
setup_redis_persistence() {
    echo -e "${GREEN}Redis 퍼시스턴스 설정을 추가합니다...${NC}"

    # Redis 설정 파일 백업
    sudo cp /etc/redis/redis.conf /etc/redis/redis.conf.backup

    # 퍼시스턴스 설정 추가
    sudo sed -i 's/^# save 900 1/save 900 1/' /etc/redis/redis.conf
    sudo sed -i 's/^# save 300 10/save 300 10/' /etc/redis/redis.conf
    sudo sed -i 's/^# save 60 10000/save 60 10000/' /etc/redis/redis.conf

    # AOF 설정 추가
    sudo sed -i 's/^# appendonly no/appendonly yes/' /etc/redis/redis.conf
    sudo sed -i 's/^# appendfilename "appendonly.aof"/appendfilename "rag_queue.aof"/' /etc/redis/redis.conf
    sudo sed -i 's/^# appendfsync everysec/appendfsync everysec/' /etc/redis/redis.conf

    # 메모리 설정
    sudo sed -i 's/^# maxmemory <bytes>/maxmemory 1gb/' /etc/redis/redis.conf
    sudo sed -i 's/^# maxmemory-policy noeviction/maxmemory-policy allkeys-lru/' /etc/redis/redis.conf

    # Redis 서비스 재시작
    sudo systemctl restart redis-server

    echo -e "${GREEN}Redis 퍼시스턴스 설정이 완료되었습니다.${NC}"
}

# Check and install nvm
if ! command -v nvm &>/dev/null; then
    echo -e "${YELLOW}Installing nvm...${NC}"
    curl -o- https://raw.githubusercontent.com/nvm-sh/nvm/v0.40.2/install.sh | bash
    export NVM_DIR="$HOME/.nvm"
    [ -s "$NVM_DIR/nvm.sh" ] && \. "$NVM_DIR/nvm.sh"
fi

# NVM 환경 설정 강화
export NVM_DIR="$HOME/.nvm"
[ -s "$NVM_DIR/nvm.sh" ] && \. "$NVM_DIR/nvm.sh"                   # NVM 로드
[ -s "$NVM_DIR/bash_completion" ] && \. "$NVM_DIR/bash_completion" # NVM bash completion 로드

# Install and configure Node.js for this project
echo -e "${GREEN}Installing Node.js 22.18.0 for this project...${NC}"
nvm install 22.18.0
nvm use 22.18.0

# Create .nvmrc file for this project
echo "22.18.0" >.nvmrc

# Node.js 버전 확인 및 재시도
NODE_VERSION=$(node -v)
if [[ "$NODE_VERSION" != "v22.18.0" ]]; then
    echo -e "${YELLOW}Node.js 버전이 올바르지 않습니다. 재시도합니다...${NC}"
    # 기존 Node.js 제거
    nvm deactivate 2>/dev/null || true
    nvm uninstall 22.18.0 2>/dev/null || true
    # 다시 설치
    nvm install 22.18.0
    nvm use 22.18.0

    # 최종 버전 확인
    NODE_VERSION=$(node -v)
    if [[ "$NODE_VERSION" != "v22.18.0" ]]; then
        echo -e "${RED}Node.js 버전 설정 실패. 현재 버전: $NODE_VERSION${NC}"
        echo -e "${RED}수동으로 다음 명령어를 실행해주세요:${NC}"
        echo -e "${YELLOW}nvm install 22.18.0${NC}"
        echo -e "${YELLOW}nvm use 22.18.0${NC}"
        exit 1
    fi
fi
echo -e "${GREEN}Project Node.js version: $NODE_VERSION${NC}"

# Add nvm use to .bashrc and .zshrc for this project
echo -e "${YELLOW}프로젝트 디렉토리에서 자동으로 Node.js 버전을 사용하도록 설정합니다...${NC}"

# # .bashrc에 추가
# if ! grep -q "cd \"\$HOME/workspaces/airun\" && nvm use" "$HOME/.bashrc" 2>/dev/null; then
#     echo "" >>"$HOME/.bashrc"
#     echo "# AI.RUN Node.js version" >>"$HOME/.bashrc"
#     echo "if [ -d \"\$HOME/workspaces/airun\" ]; then" >>"$HOME/.bashrc"
#     echo "    cd \"\$HOME/workspaces/airun\" && nvm use" >>"$HOME/.bashrc"
#     echo "fi" >>"$HOME/.bashrc"
# fi

# # .zshrc에 추가
# if ! grep -q "cd \"\$HOME/workspaces/airun\" && nvm use" "$HOME/.zshrc" 2>/dev/null; then
#     echo "" >>"$HOME/.zshrc"
#     echo "# AI.RUN Node.js version" >>"$HOME/.zshrc"
#     echo "if [ -d \"\$HOME/workspaces/airun\" ]; then" >>"$HOME/.zshrc"
#     echo "    cd \"\$HOME/workspaces/airun\" && nvm use" >>"$HOME/.zshrc"
#     echo "fi" >>"$HOME/.zshrc"
# fi

# Install pm2 globally
echo -e "${GREEN}Installing pm2...${NC}"
if ! command -v pm2 &>/dev/null; then
    npm install -g pm2

    # Check pm2 installation
    if command -v pm2 &>/dev/null; then
        PM2_VERSION=$(pm2 -v)
        echo -e "${GREEN}pm2 installed successfully. Version: $PM2_VERSION${NC}"
    else
        echo -e "${RED}Failed to install pm2${NC}"
        exit 1
    fi
else
    PM2_VERSION=$(pm2 -v)
    echo -e "${GREEN}pm2 is already installed. Version: $PM2_VERSION${NC}"
fi

# Setup pm2 startup
echo -e "${GREEN}Setting up pm2 startup...${NC}"
pm2 startup

# Install npm 9.2.0
echo -e "${GREEN}Installing npm 10.8.2...${NC}"
npm install -g npm@10.8.2

# Check npm version
NPM_VERSION=$(npm -v)
if [[ "$NPM_VERSION" != "10.8.2" ]]; then
    echo -e "${RED}npm version is not 10.8.2. Current version: $NPM_VERSION${NC}"
    exit 1
fi
echo -e "${GREEN}npm version: $NPM_VERSION${NC}"

# Install WebSocket package globally
echo -e "${GREEN}Installing WebSocket package globally...${NC}"
npm install -g ws

# Install Mermaid CLI
echo -e "${GREEN}Installing Mermaid CLI...${NC}"

# 기존 설치 제거
echo -e "${YELLOW}Removing existing Mermaid CLI installation...${NC}"
npm uninstall -g @mermaid-js/mermaid-cli 2>/dev/null || true
npm uninstall -g puppeteer 2>/dev/null || true

# puppeteer 먼저 설치
echo -e "${YELLOW}Installing puppeteer first...${NC}"
npm config set fetch-timeout 120000
npm config set fetch-retries 5
npm config set fetch-retry-factor 10
npm config set fetch-retry-mintimeout 20000
npm config set fetch-retry-maxtimeout 120000
npm install -g puppeteer

# Mermaid CLI 설치
echo -e "${YELLOW}Installing @mermaid-js/mermaid-cli globally...${NC}"
npm install -g @mermaid-js/mermaid-cli

# Check installation
if command -v mmdc &>/dev/null; then
    # 실제 실행 테스트
    if mmdc --version 2>/dev/null; then
        echo -e "${GREEN}Mermaid CLI installed successfully${NC}"
    else
        echo -e "${RED}Mermaid CLI installation verification failed${NC}"
        exit 1
    fi
else
    echo -e "${RED}Failed to install Mermaid CLI${NC}"
    exit 1
fi

# Check Python
if ! command -v python3 &>/dev/null; then
    echo -e "${RED}Python3가 설치되어 있지 않습니다. Python3를 먼저 설치해 주세요.${NC}"
    exit 1
fi

# Check Python version
PYTHON_VERSION=$(python3 --version | awk '{print $2}')
PYTHON_MAJOR=$(echo $PYTHON_VERSION | cut -d. -f1)
PYTHON_MINOR=$(echo $PYTHON_VERSION | cut -d. -f2)

if [ "$PYTHON_MAJOR" != "3" ] || [ "$PYTHON_MINOR" != "12" ]; then
    echo -e "${RED}Python 버전이 3.12가 아닙니다. 현재 버전: $PYTHON_VERSION${NC}"
    exit 1
fi

echo -e "${GREEN}Python version: $PYTHON_VERSION${NC}"

# Check and recreate virtual environment
VENV_PATH="$HOME/.airun_venv"
if [ -d "$VENV_PATH" ]; then
    echo -e "${YELLOW}기존 가상 환경을 제거하고 새로 생성합니다...${NC}"
    rm -rf "$VENV_PATH"
fi

echo -e "${GREEN}Python 3.12 가상 환경을 생성합니다...${NC}"
# 명시적으로 python3를 사용하여 가상 환경 생성
if ! python3 -m venv "$VENV_PATH"; then
    echo -e "${RED}가상 환경 생성에 실패했습니다. 다시 시도합니다...${NC}"
    # 디렉토리 생성 확인
    mkdir -p "$VENV_PATH"
    chmod 755 "$VENV_PATH"
    # 다시 시도
    if ! python3 -m venv "$VENV_PATH"; then
        echo -e "${RED}가상 환경 생성에 실패했습니다. 스크립트를 종료합니다.${NC}"
        exit 1
    fi
fi

# 가상 환경 활성화 스크립트 확인
if [ ! -f "$VENV_PATH/bin/activate" ]; then
    echo -e "${RED}가상 환경 활성화 스크립트를 찾을 수 없습니다. 스크립트를 종료합니다.${NC}"
    exit 1
fi

# 가상 환경 활성화
source "$VENV_PATH/bin/activate"

# Python 경로 확인
VENV_PYTHON=$(which python)
echo -e "${GREEN}가상 환경 Python 경로: $VENV_PYTHON${NC}"

# Python 버전 확인
VENV_PYTHON_VERSION=$("$VENV_PATH/bin/python" --version)
echo -e "${GREEN}가상 환경 Python 버전: $VENV_PYTHON_VERSION${NC}"

# 가상 환경 권한 설정
echo -e "${GREEN}가상 환경 권한 설정...${NC}"
chmod -R 755 "$VENV_PATH" || true
chmod -R u+w "$VENV_PATH" || true

# pip 업그레이드
echo -e "${GREEN}pip 업그레이드 중...${NC}"
"$VENV_PATH/bin/python" -m pip install --upgrade pip

# GPU 환경 감지
echo -e "${GREEN}GPU 환경을 감지합니다...${NC}"
CUDA_MAJOR="$(nvidia-smi | sed -n 's/.*CUDA Version: \([0-9]\+\)\..*/\1/p' | head -n1)"
echo -e "${GREEN}CUDA 메이저 버전: $CUDA_MAJOR${NC}"  # 변수명에 맞게 수정

# GPU 환경 감지 및 PyTorch 설치
if command -v nvidia-smi &>/dev/null; then
    echo -e "${GREEN}NVIDIA GPU가 감지되었습니다.${NC}"
    
    CUDA_MAJOR="$(nvidia-smi | sed -n 's/.*CUDA Version: \([0-9]\+\)\..*/\1/p' | head -n1)"
    echo -e "${GREEN}CUDA 메이저 버전: $CUDA_MAJOR${NC}"

    if [ "$CUDA_MAJOR" = "12" ] || [ "$CUDA_MAJOR" = "13" ]; then
        echo -e "${YELLOW}CUDA 12/13 환경을 감지했습니다. PyTorch CUDA 12.9 버전을 설치합니다.${NC}"
        "$VENV_PATH/bin/pip" install torch torchvision torchaudio --index-url https://download.pytorch.org/whl/cu129
    elif [ "$CUDA_MAJOR" = "11" ]; then
        echo -e "${YELLOW}CUDA 11.x 환경을 감지했습니다. PyTorch CUDA 11.8 버전을 설치합니다.${NC}"
        "$VENV_PATH/bin/pip" install torch torchvision torchaudio --index-url https://download.pytorch.org/whl/cu118
    else
        echo -e "${YELLOW}CUDA 버전이 11 미만입니다. CPU 버전을 설치합니다.${NC}"
        "$VENV_PATH/bin/pip" install torch torchvision torchaudio --index-url https://download.pytorch.org/whl/cpu
    fi
else
    echo -e "${YELLOW}NVIDIA GPU가 감지되지 않았습니다. CPU 버전을 설치합니다.${NC}"
    "$VENV_PATH/bin/pip" install torch torchvision torchaudio
fi

# PyTorch 설치 확인
echo -e "${GREEN}PyTorch 설치를 확인합니다...${NC}"
if "$VENV_PATH/bin/python" -c "import torch; print('CUDA 사용 가능:', torch.cuda.is_available()); print('PyTorch 버전:', torch.__version__)" 2>/dev/null; then
    echo -e "${GREEN}PyTorch가 성공적으로 설치되었습니다.${NC}"
else
    echo -e "${RED}PyTorch 설치에 실패했습니다.${NC}"
    exit 1
fi

# requirements-linux.txt 수정 (torch는 위에서 CUDA 버전에 맞게 설치됨)
# Note: PyTorch with CUDA support is installed above and fixed in requirements-linux.txt
# echo -e "${GREEN}requirements-linux.txt를 업데이트합니다...${NC}"
# sed -i '/^torch==/d' requirements-linux.txt
# PyTorch 버전은 requirements-linux.txt에 torch==2.8.0+cu129로 고정됨

# Install system-level Python packages (build dependencies)
if [ -f "requirements-system.txt" ]; then
    echo -e "${GREEN}Installing system-level Python packages...${NC}"
    pip3 install -r requirements-system.txt --break-system-package
else
    echo -e "${YELLOW}requirements-system.txt not found. Installing basic build packages only...${NC}"
    pip3 install Cython setuptools "lxml[html_clean]" --break-system-package
fi

# 운영체제 감지
if [[ "$OSTYPE" == "darwin"* ]]; then
    REQUIREMENTS_FILE="requirements-mac.txt"
elif [[ "$OSTYPE" == "msys" || "$OSTYPE" == "cygwin" ]]; then
    REQUIREMENTS_FILE="requirements-windows.txt"
else
    REQUIREMENTS_FILE="requirements-linux.txt"
fi

# OS별 의존성 설치 (pyhwpx 패키지 설치 전에 수행)
if [ -f "$REQUIREMENTS_FILE" ]; then
    echo -e "${GREEN}Installing OS-specific Python packages...${NC}"
    export PIP_PREFER_BINARY=1
    # PyTorch CUDA 12.9 지원을 위한 index URL 추가
    "$VENV_PATH/bin/pip" install -r "$REQUIREMENTS_FILE" --extra-index-url https://download.pytorch.org/whl/cu129
else
    echo -e "${YELLOW}requirements-$REQUIREMENTS_FILE not found. Skipping OS-specific packages installation...${NC}"
fi

# Install pyhwpx package
# echo -e "${GREEN}pyhwpx 패키지 설치 중...${NC}"
# if [ -d "pyhwpx" ]; then
#     # 이전 설치 제거
#     "$VENV_PATH/bin/pip" uninstall -y pyhwpx 2>/dev/null || true

#     # 가상환경 사이트 패키지 경로 확인
#     SITE_PACKAGES=$("$VENV_PATH/bin/python" -c "import site; print(site.getsitepackages()[0])")

#     if [ -n "$SITE_PACKAGES" ]; then
#         echo -e "${GREEN}pyhwpx 패키지를 설치합니다...${NC}"

#         # pyhwpx 디렉토리 생성
#         mkdir -p "$SITE_PACKAGES/pyhwpx"

#         # pyhwpx 디렉토리 복사
#         if [ -d "pyhwpx/pyhwpx" ]; then
#             # 원본 파일 복사
#             cp "pyhwpx/pyhwpx/__init__.py" "$SITE_PACKAGES/pyhwpx/" 2>/dev/null || true
#             echo -e "${GREEN}__init__.py 파일을 복사했습니다.${NC}"

#             # 복사된 파일 내용 확인
#             if [ ! -s "$SITE_PACKAGES/pyhwpx/__init__.py" ]; then
#                 echo -e "${YELLOW}__init__.py 파일이 비어 있습니다. 원본 파일을 다시 복사합니다...${NC}"
#                 cat "pyhwpx/pyhwpx/__init__.py" > "$SITE_PACKAGES/pyhwpx/__init__.py"
#                 echo -e "${GREEN}__init__.py 파일을 다시 복사했습니다.${NC}"
#             else
#                 echo -e "${GREEN}__init__.py 파일이 성공적으로 복사되었습니다.${NC}"
#             fi

#         else
#             echo -e "${RED}pyhwpx/pyhwpx 디렉토리를 찾을 수 없습니다.${NC}"
#             exit 1
#         fi

#         # 운영체제에 맞는 바이너리 파일 복사
#         if [[ "$OSTYPE" == "msys" || "$OSTYPE" == "cygwin" ]]; then
#             # Windows - .pyd 파일 복사
#             cp "pyhwpx/pyhwpx/"*.pyd "$SITE_PACKAGES/pyhwpx/" 2>/dev/null || true
#             echo -e "${GREEN}.pyd 파일을 복사했습니다.${NC}"
#         else
#             # Linux/macOS - .so 파일 복사
#             cp "pyhwpx/pyhwpx/"*.so "$SITE_PACKAGES/pyhwpx/" 2>/dev/null || true
#             echo -e "${GREEN}.so 파일을 복사했습니다.${NC}"
#         fi

#         # templates 디렉토리 복사
#         if [ -d "pyhwpx/pyhwpx/templates" ]; then
#             mkdir -p "$SITE_PACKAGES/pyhwpx/templates"
#             cp -r "pyhwpx/pyhwpx/templates/"* "$SITE_PACKAGES/pyhwpx/templates/" 2>/dev/null || true
#             echo -e "${GREEN}templates 디렉토리를 복사했습니다.${NC}"
#         fi

#         # examples 디렉토리 복사
#         if [ -d "pyhwpx/pyhwpx/examples" ]; then
#             mkdir -p "$SITE_PACKAGES/pyhwpx/examples"
#             cp -r "pyhwpx/pyhwpx/examples/"* "$SITE_PACKAGES/pyhwpx/examples/" 2>/dev/null || true
#             echo -e "${GREEN}examples 디렉토리를 복사했습니다.${NC}"
#         fi

#         # 패키지 정보 파일 생성
#         mkdir -p "$SITE_PACKAGES/pyhwpx-1.0.0.dist-info"
#         cat > "$SITE_PACKAGES/pyhwpx-1.0.0.dist-info/METADATA" << EOF
# Metadata-Version: 2.1
# Name: pyhwpx
# Version: 1.0.0
# Summary: 한글 HWPX 문서 생성 라이브러리
# Author: Kevin Kim
# Author-email: chaeya@gmail.com
# License: MIT
# Platform: UNKNOWN
# Classifier: Development Status :: 4 - Beta
# Classifier: Intended Audience :: Developers
# Classifier: License :: OSI Approved :: MIT License
# Classifier: Operating System :: OS Independent
# Classifier: Programming Language :: Python :: 3
# Classifier: Programming Language :: Python :: 3.12
# Classifier: Topic :: Office/Business
# Classifier: Topic :: Software Development :: Libraries :: Python Modules
# Requires-Python: ==3.12.*
# EOF

#         echo -e "${GREEN}pyhwpx 패키지가 $SITE_PACKAGES/pyhwpx에 성공적으로 설치되었습니다.${NC}"
#         echo -e "${GREEN}이제 'import pyhwpx'로 패키지를 임포트할 수 있습니다.${NC}"

#         # 설치 확인
#         echo -e "${YELLOW}패키지 설치 확인 중...${NC}"
#         # 더 자세한 오류 정보를 위해 임시 스크립트 생성
#         TEMP_SCRIPT=$(mktemp)
#         cat > "$TEMP_SCRIPT" << EOF
# try:
#     import pyhwpx
#     print(f"pyhwpx 패키지가 성공적으로 임포트되었습니다.")

#     # 패키지 버전 확인 (안전하게)
#     version = getattr(pyhwpx, '__version__', '버전 정보 없음')
#     print(f"패키지 버전: {version}")

#     # document 모듈 확인
#     try:
#         if hasattr(pyhwpx, 'document'):
#             print(f"document 모듈 확인: 성공")
#         elif hasattr(pyhwpx, 'HWPDocument'):
#             print(f"HWPDocument 클래스 확인: 성공")
#         else:
#             print(f"document 모듈 또는 HWPDocument 클래스를 찾을 수 없습니다.")
#     except Exception as e:
#         print(f"document 모듈 확인 중 오류: {str(e)}")

#     # 모듈 경로 확인
#     #print(f"pyhwpx 패키지 경로: {pyhwpx.__file__}")

#     # 디렉토리 내용 확인
#     import os
#     pkg_dir = os.path.dirname(pyhwpx.__file__)
#     #print(f"패키지 디렉토리 내용: {os.listdir(pkg_dir)}")

#     # .so 파일 확인
#     so_files = [f for f in os.listdir(pkg_dir) if f.endswith('.so')]
#     #if so_files:
#     #    print(f"발견된 .so 파일: {so_files}")
#     #else:
#     #    print("패키지 디렉토리에 .so 파일이 없습니다.")

# except ImportError as e:
#     print(f"패키지 임포트 실패: {str(e)}")
# except Exception as e:
#     import traceback
#     print(f"예상치 못한 오류: {str(e)}")
#     print(traceback.format_exc())
# EOF

#         # 스크립트 실행
#         echo -e "${YELLOW}상세 패키지 확인 결과:${NC}"
#         "$VENV_PATH/bin/python" "$TEMP_SCRIPT"

#         # 임시 스크립트 삭제
#         rm -f "$TEMP_SCRIPT"

#         # 간단한 임포트 테스트
#         if "$VENV_PATH/bin/python" -c "import pyhwpx" 2>/dev/null; then
#             echo -e "${GREEN}패키지 설치 확인 완료!${NC}"
#         else
#             echo -e "${RED}패키지 설치 확인 실패. 위의 상세 정보를 확인하세요.${NC}"
#         fi
#     else
#         echo -e "${RED}사이트 패키지 경로를 찾을 수 없습니다.${NC}"
#         exit 1
#     fi
# else
#     echo -e "${YELLOW}pyhwpx 디렉토리를 찾을 수 없습니다. 설치를 건너뜁니다...${NC}"
# fi

# Install Node.js dependencies
echo -e "${GREEN}Installing Node.js dependencies...${NC}"
if [ ! -f "package.json" ]; then
    echo -e "${YELLOW}package.json not found. Creating default package.json...${NC}"
    cat >package.json <<EOF
{
  "name": "airun",
  "version": "1.0.0",
  "description": "AI.RUN Application",
  "main": "index.js",
  "scripts": {
    "start": "electron ."
  },
  "author": "",
  "license": "ISC",
  "dependencies": {},
  "devDependencies": {}
}
EOF
fi

# Add execute permission to index.js
echo -e "${GREEN}Adding execute permission to index.js...${NC}"
chmod +x index.js

rm -rf node_modules package-lock.json
npm cache clean --force

# Change npm configuration
npm config set ignore-scripts false
npm config set legacy-peer-deps true

# Local installation
echo -e "${YELLOW}Installing local dependencies...${NC}"
npm install --legacy-peer-deps

# 큐 시스템 관련 패키지 설치
echo -e "${GREEN}Installing queue system packages...${NC}"
npm install --save bull ioredis

# Global installation through npm link
echo -e "${GREEN}Creating global link...${NC}"
if [ -f "package.json" ]; then
    # 현재 디렉토리의 절대 경로 가져오기
    CURRENT_DIR=$(pwd)
    echo -e "${YELLOW}현재 설치 경로: ${CURRENT_DIR}${NC}"

    # 기존 링크 확인 및 제거 (시스템 전역)
    AIRUN_PATH=$(which airun 2>/dev/null || echo "")
    if [ -n "$AIRUN_PATH" ]; then
        echo -e "${YELLOW}기존 airun 링크 발견: ${AIRUN_PATH}${NC}"
        if [ -L "$AIRUN_PATH" ]; then
            EXISTING_TARGET=$(readlink -f "$AIRUN_PATH" 2>/dev/null || readlink "$AIRUN_PATH")
            echo -e "${YELLOW}기존 링크 대상: ${EXISTING_TARGET}${NC}"

            if [ "$EXISTING_TARGET" != "$CURRENT_DIR/index.js" ]; then
                echo -e "${YELLOW}기존 링크가 다른 경로를 가리키고 있습니다. 링크를 제거합니다...${NC}"
                # 기존 전역 패키지 제거
                npm uninstall -g airun 2>/dev/null || true
                npm unlink -g airun 2>/dev/null || true

                # 시스템 전역 링크 직접 제거
                if [ -f "/usr/local/bin/airun" ]; then
                    sudo rm -f "/usr/local/bin/airun" 2>/dev/null || true
                fi
                if [ -d "/usr/local/lib/node_modules/airun" ] || [ -L "/usr/local/lib/node_modules/airun" ]; then
                    sudo rm -rf "/usr/local/lib/node_modules/airun" 2>/dev/null || true
                fi
            else
                echo -e "${GREEN}기존 링크가 현재 디렉토리를 가리키고 있습니다. 재설치를 건너뜁니다.${NC}"
            fi
        fi
    fi

    # NVM 환경 확인 및 링크 제거
    NVM_DIR="$HOME/.nvm"
    if [ -d "$NVM_DIR" ]; then
        echo -e "${YELLOW}NVM 환경이 감지되었습니다. NVM 링크를 확인합니다...${NC}"

        # 활성화된 Node.js 버전 확인
        NODE_VERSION=$(node -v 2>/dev/null || echo "")
        if [ -n "$NODE_VERSION" ]; then
            NODE_VERSION=${NODE_VERSION#v} # v 접두사 제거
            echo -e "${YELLOW}활성화된 Node.js 버전: ${NODE_VERSION}${NC}"

            # NVM 링크 경로
            NVM_BIN_PATH="$NVM_DIR/versions/node/v$NODE_VERSION/bin/airun"
            NVM_LIB_PATH="$NVM_DIR/versions/node/v$NODE_VERSION/lib/node_modules/airun"

            # NVM 링크 확인 및 제거
            if [ -L "$NVM_BIN_PATH" ]; then
                echo -e "${YELLOW}NVM bin 링크 발견: ${NVM_BIN_PATH}${NC}"
                NVM_TARGET=$(readlink -f "$NVM_BIN_PATH" 2>/dev/null || readlink "$NVM_BIN_PATH")
                echo -e "${YELLOW}NVM 링크 대상: ${NVM_TARGET}${NC}"

                if [ "$NVM_TARGET" != "$CURRENT_DIR/index.js" ]; then
                    echo -e "${YELLOW}NVM 링크가 다른 경로를 가리키고 있습니다. 링크를 제거합니다...${NC}"
                    rm -f "$NVM_BIN_PATH" 2>/dev/null || true
                fi
            fi

            if [ -L "$NVM_LIB_PATH" ]; then
                echo -e "${YELLOW}NVM lib 링크 발견: ${NVM_LIB_PATH}${NC}"
                NVM_LIB_TARGET=$(readlink -f "$NVM_LIB_PATH" 2>/dev/null || readlink "$NVM_LIB_PATH")
                echo -e "${YELLOW}NVM lib 링크 대상: ${NVM_LIB_TARGET}${NC}"

                if [ "$NVM_LIB_TARGET" != "$CURRENT_DIR" ]; then
                    echo -e "${YELLOW}NVM lib 링크가 다른 경로를 가리키고 있습니다. 링크를 제거합니다...${NC}"
                    rm -f "$NVM_LIB_PATH" 2>/dev/null || true
                fi
            fi
        fi
    fi

    # 전역 npm 모듈 경로 확인
    if command -v npm &>/dev/null; then
        GLOBAL_PATH=$(npm root -g)
        echo -e "${YELLOW}전역 npm 모듈 경로: ${GLOBAL_PATH}${NC}"

        # 전역 모듈 디렉토리에서 airun 링크 확인 및 제거
        if [ -L "${GLOBAL_PATH}/airun" ]; then
            EXISTING_MODULE_TARGET=$(readlink -f "${GLOBAL_PATH}/airun" 2>/dev/null || readlink "${GLOBAL_PATH}/airun")
            echo -e "${YELLOW}기존 모듈 링크 대상: ${EXISTING_MODULE_TARGET}${NC}"

            if [ "$EXISTING_MODULE_TARGET" != "$CURRENT_DIR" ]; then
                echo -e "${YELLOW}기존 모듈 링크가 다른 경로를 가리키고 있습니다. 링크를 제거합니다...${NC}"
                sudo rm -f "${GLOBAL_PATH}/airun" 2>/dev/null || true
            fi
        fi
    fi

    # 전역 링크 생성 (npm link 사용)
    echo -e "${YELLOW}전역 링크 생성 중...${NC}"
    npm link
    npm link electron

    # 링크 확인
    LINK_PATH=$(which airun 2>/dev/null || echo "링크를 찾을 수 없습니다")
    echo -e "${YELLOW}생성된 링크 경로: ${LINK_PATH}${NC}"

    # 링크 대상 확인
    if [ -L "$LINK_PATH" ]; then
        LINK_TARGET=$(readlink -f "$LINK_PATH" 2>/dev/null || readlink "$LINK_PATH")
        echo -e "${YELLOW}링크 대상: ${LINK_TARGET}${NC}"

        # 링크가 올바른 경로를 가리키는지 확인
        if [ "$LINK_TARGET" = "$CURRENT_DIR/index.js" ]; then
            echo -e "${GREEN}링크가 올바르게 설정되었습니다!${NC}"
        else
            echo -e "${RED}링크가 올바른 경로를 가리키지 않습니다. 수동으로 확인이 필요합니다.${NC}"
            echo -e "${RED}예상 경로: ${CURRENT_DIR}/index.js${NC}"
            echo -e "${RED}실제 경로: ${LINK_TARGET}${NC}"
        fi
    fi

    # 전역 npm 모듈 경로 확인
    if command -v npm &>/dev/null; then
        GLOBAL_PATH=$(npm root -g)
        echo -e "${YELLOW}전역 npm 모듈 경로: ${GLOBAL_PATH}${NC}"

        if [ -L "${GLOBAL_PATH}/airun" ]; then
            AIRUN_LINK=$(readlink -f "${GLOBAL_PATH}/airun" 2>/dev/null || readlink "${GLOBAL_PATH}/airun")
            echo -e "${GREEN}airun 모듈이 ${GLOBAL_PATH}/airun에 링크되었습니다.${NC}"
            echo -e "${GREEN}링크 대상: ${AIRUN_LINK}${NC}"

            # 모듈 링크가 올바른 경로를 가리키는지 확인
            if [ "$AIRUN_LINK" = "$CURRENT_DIR" ]; then
                echo -e "${GREEN}모듈 링크가 올바르게 설정되었습니다!${NC}"
            else
                echo -e "${RED}모듈 링크가 올바른 경로를 가리키지 않습니다. 수동으로 확인이 필요합니다.${NC}"
                echo -e "${RED}예상 경로: ${CURRENT_DIR}${NC}"
                echo -e "${RED}실제 경로: ${AIRUN_LINK}${NC}"
            fi
        fi
    fi

    # 설치 완료 메시지
    echo -e "${GREEN}설치가 완료되었습니다!${NC}"
    echo -e "${YELLOW}주의: airun 명령어는 현재 디렉토리(${CURRENT_DIR})에 링크되어 있습니다.${NC}"
    echo -e "${YELLOW}이 디렉토리를 삭제하거나 이동하면 명령어가 작동하지 않을 수 있습니다.${NC}"
else
    echo -e "${YELLOW}No package.json found. Skipping global link...${NC}"
fi

# Install electron
echo -e "${GREEN}Installing Electron...${NC}"
npm install electron --save-dev

# chrome-sandbox 파일의 소유자 및 권한 설정
SANDBOX_PATH="./node_modules/electron/dist/chrome-sandbox"
if [ -f "${SANDBOX_PATH}" ]; then
    echo "Setting correct permissions for chrome-sandbox..."
    sudo chown root:root "${SANDBOX_PATH}"
    sudo chmod 4755 "${SANDBOX_PATH}"
else
    echo "chrome-sandbox not found at ${SANDBOX_PATH}"
fi

# Create .airun directory and set permissions
AIRUN_DIR="$HOME/.airun"
TRANSFORMERS_CACHE_DIR="$AIRUN_DIR/cache/transformers"
QUEUES_DIR="$AIRUN_DIR/queues"
MODELS_DIR="$AIRUN_DIR/models"
echo -e "${GREEN}Setting AI.RUN directory permissions...${NC}"
mkdir -p "$TRANSFORMERS_CACHE_DIR"
mkdir -p "$QUEUES_DIR"
mkdir -p "$MODELS_DIR"
chmod -R 755 "$AIRUN_DIR"

# 모델 다운로드
echo -e "${GREEN}Downloading KURE-v1 model...${NC}"
MODEL_NAME="nlpai-lab/KURE-v1"
MODEL_PATH="$MODELS_DIR/models--nlpai-lab--KURE-v1"

# 모델이 이미 있는지 확인
if [ ! -d "$MODEL_PATH" ]; then
    echo -e "${YELLOW}Model not found. Downloading...${NC}"
    # 가상환경 활성화
    source "$VENV_PATH/bin/activate"

    # Python 스크립트를 사용하여 모델 다운로드
    cat >download_model.py <<EOF
from sentence_transformers import SentenceTransformer
import os

model_name = "nlpai-lab/KURE-v1"
model_path = os.path.expanduser("~/.airun/models/models--nlpai-lab--KURE-v1")

print(f"Downloading model {model_name} to {model_path}...")
model = SentenceTransformer(model_name, cache_folder=model_path)
print("Model download completed successfully!")
EOF

    # 모델 다운로드 스크립트 실행
    "$VENV_PATH/bin/python" download_model.py

    # 임시 스크립트 삭제
    rm download_model.py

    # 모델 다운로드 확인
    if [ -d "$MODEL_PATH" ]; then
        echo -e "${GREEN}Model downloaded successfully!${NC}"
    else
        echo -e "${RED}Failed to download model. Please check the error messages above.${NC}"
    fi
else
    echo -e "${GREEN}Model already exists. Skipping download.${NC}"
fi

# 시맨틱 청킹 모델 다운로드
echo -e "${GREEN}Downloading Korean Semantic Chunking model...${NC}"
SEMANTIC_MODEL_NAME="snunlp/KR-SBERT-V40K-klueNLI-augSTS"
SEMANTIC_MODEL_PATH="$MODELS_DIR/models--snunlp--KR-SBERT-V40K-klueNLI-augSTS"

# 모델이 이미 있는지 확인
if [ ! -d "$SEMANTIC_MODEL_PATH" ]; then
    echo -e "${YELLOW}Semantic model not found. Downloading...${NC}"
    # 가상환경 활성화
    source "$VENV_PATH/bin/activate"

    # Python 스크립트를 사용하여 모델 다운로드
    cat >download_semantic_model.py <<EOF
from sentence_transformers import SentenceTransformer
import os

model_name = "snunlp/KR-SBERT-V40K-klueNLI-augSTS"
model_path = os.path.expanduser("~/.airun/models/models--snunlp--KR-SBERT-V40K-klueNLI-augSTS")

print(f"Downloading semantic model {model_name} to {model_path}...")
model = SentenceTransformer(model_name, cache_folder=model_path)
print("Semantic model download completed successfully!")
EOF

    # 모델 다운로드 스크립트 실행
    "$VENV_PATH/bin/python" download_semantic_model.py

    # 임시 스크립트 삭제
    rm download_semantic_model.py

    # 모델 다운로드 확인
    if [ -d "$SEMANTIC_MODEL_PATH" ]; then
        echo -e "${GREEN}Semantic model downloaded successfully!${NC}"
    else
        echo -e "${RED}Failed to download semantic model. Please check the error messages above.${NC}"
    fi
else
    echo -e "${GREEN}Semantic model already exists. Skipping download.${NC}"
fi

# 큐 설정 디렉토리 생성 확인
if [ -d "$QUEUES_DIR" ]; then
    echo -e "${GREEN}Queue directory created: $QUEUES_DIR${NC}"

    # 기본 큐 설정 파일 생성
    echo -e "${GREEN}Creating default queue configuration...${NC}"
    cat >"$QUEUES_DIR/config.json" <<EOF
{
  "redis": {
    "host": "127.0.0.1",
    "port": 6379,
    "db": 0
  },
  "queues": {
    "apiUsage": {
      "name": "api-usage-queue",
      "options": {
        "attempts": 3,
        "backoff": {
          "type": "exponential",
          "delay": 1000
        },
        "removeOnComplete": 100,
        "removeOnFail": 1000
      }
    },
    "licenseCheck": {
      "name": "license-check-queue",
      "options": {
        "attempts": 5,
        "backoff": {
          "type": "exponential",
          "delay": 2000
        },
        "removeOnComplete": 100,
        "removeOnFail": 1000
      }
    }
  }
}
EOF

    echo -e "${GREEN}Default queue configuration created at $QUEUES_DIR/config.json${NC}"
else
    echo -e "${RED}Failed to create queue directory: $QUEUES_DIR${NC}"
fi

# 기본 큐 처리 인덱스 파일 생성
echo -e "${GREEN}Creating default queue processor file...${NC}"
cat >"$QUEUES_DIR/index.js" <<EOF
const Queue = require('bull');
const fs = require('fs');
const path = require('path');
const os = require('os');

// 설정 파일 로드
const configPath = path.join(__dirname, 'config.json');
let config = { redis: { host: '127.0.0.1', port: 6379 }, queues: {} };

try {
  if (fs.existsSync(configPath)) {
    const configData = fs.readFileSync(configPath, 'utf8');
    config = JSON.parse(configData);
  } else {
    console.warn('큐 설정 파일을 찾을 수 없습니다. 기본 설정을 사용합니다.');
  }
} catch (error) {
  console.error('설정 파일 로드 중 오류:', error);
}

// Redis 연결 설정
const redisConfig = config.redis || { host: '127.0.0.1', port: 6379 };

// 라이선스 파일 경로
const LICENSE_FILE = path.join(os.homedir(), '.airun', 'license.key');

// API 사용량 큐 생성
const apiUsageQueue = new Queue('api-usage-queue', {
  redis: redisConfig,
  ...(config.queues?.apiUsage?.options || {})
});

// 라이선스 체크 큐 생성
const licenseCheckQueue = new Queue('license-check-queue', {
  redis: redisConfig,
  ...(config.queues?.licenseCheck?.options || {})
});

// 라이선스 파일 읽기 함수
function readLicenseFile() {
  try {
    if (fs.existsSync(LICENSE_FILE)) {
      const data = fs.readFileSync(LICENSE_FILE, 'utf8');
      return JSON.parse(data);
    }
    return null;
  } catch (error) {
    console.error('라이선스 파일 읽기 오류:', error);
    return null;
  }
}

// 라이선스 파일 저장 함수
function saveLicenseFile(licenseData) {
  try {
    fs.writeFileSync(LICENSE_FILE, JSON.stringify(licenseData, null, 2), 'utf8');
    return true;
  } catch (error) {
    console.error('라이선스 파일 저장 오류:', error);
    return false;
  }
}

// API 사용량 처리 작업자
apiUsageQueue.process(async (job) => {
  console.log('API 사용량 처리 작업 시작 - Job ID:', job.id);
  console.log('작업 데이터:', job.data);
  
  try {
    // 라이선스 데이터 읽기
    console.log('라이선스 데이터 읽기 시도...');
    const licenseData = readLicenseFile();
    
    if (!licenseData) {
      throw new Error('라이선스 데이터를 읽을 수 없습니다');
    }
    
    // API 사용량 업데이트
    const currentUsage = licenseData.apiUsage || 0;
    console.log('라이선스 데이터 읽기 성공 - 현재 API 사용량:', currentUsage);
    
    const usageCount = job.data.count || 1;
    const newUsage = currentUsage + usageCount;
    
    console.log('API 사용량 업데이트 - 새 사용량:', newUsage);
    
    // 라이선스 데이터 업데이트 및 저장
    licenseData.apiUsage = newUsage;
    console.log('라이선스 데이터 저장 시도...');
    
    if (saveLicenseFile(licenseData)) {
      console.log('API 사용량 업데이트');
      console.log('라이선스 저장 성공');
      console.log('라이선스 데이터 저장 성공');
      return { success: true, count: usageCount, totalUsed: newUsage };
    } else {
      throw new Error('라이선스 데이터 저장 실패');
    }
  } catch (error) {
    console.error('API 사용량 처리 오류:', error);
    throw error; // 작업 실패로 처리
  }
});

// 라이선스 체크 작업자
licenseCheckQueue.process(async (job) => {
  console.log('라이선스 체크 작업 시작 - Job ID:', job.id);
  
  try {
    // 라이선스 데이터 읽기
    const licenseData = readLicenseFile();
    
    if (!licenseData) {
      return { valid: false, reason: 'license_not_found' };
    }
    
    // 라이선스 유효성 검사 로직
    const isValid = licenseData.valid === true;
    const apiLimit = licenseData.apiLimit || 1000;
    const apiUsage = licenseData.apiUsage || 0;
    
    return {
      valid: isValid,
      apiLimit,
      apiUsage,
      remaining: apiLimit - apiUsage,
      resetDate: licenseData.resetDate || null
    };
  } catch (error) {
    console.error('라이선스 체크 오류:', error);
    throw error;
  }
});

// 이벤트 리스너
apiUsageQueue.on('completed', (job, result) => {
  console.log('API 사용량 저장 완료 [Job ' + job.id + ']:', result);
});

apiUsageQueue.on('failed', (job, error) => {
  console.error('API 사용량 저장 실패 [Job ' + job.id + ']:', error);
});

licenseCheckQueue.on('completed', (job, result) => {
  console.log('라이선스 체크 완료 [Job ' + job.id + ']:', result);
});

licenseCheckQueue.on('failed', (job, error) => {
  console.error('라이선스 체크 실패 [Job ' + job.id + ']:', error);
});

// 큐 객체 내보내기
module.exports = {
  apiUsageQueue,
  licenseCheckQueue,
  
  // API 사용량 기록 함수
  recordApiUsage: async (count = 1) => {
    try {
      const job = await apiUsageQueue.add({ 
        count, 
        timestamp: Date.now() 
      });
      return { success: true, jobId: job.id };
    } catch (error) {
      console.error('API 사용량 작업 추가 실패:', error);
      return { success: false, error: error.message };
    }
  },
  
  // 라이선스 체크 함수
  checkLicense: async () => {
    try {
      const job = await licenseCheckQueue.add({ timestamp: Date.now() });
      const result = await job.finished();
      return result;
    } catch (error) {
      console.error('라이선스 체크 작업 실패:', error);
      return { valid: false, error: error.message };
    }
  }
};
EOF

echo -e "${GREEN}Default queue processor file created at $QUEUES_DIR/index.js${NC}"


# 폰트 파일 복사
echo -e "${GREEN}폰트 파일 복사 중...${NC}"
ASSETS_DIR="$(pwd)/assets"
FONT_FILES=("Pretendard-Regular.ttf" "Pretendard-Bold.ttf")

# 운영체제별 파일 크기 확인 함수
get_file_size() {
    local file_path="$1"
    if [[ "$OSTYPE" == "darwin"* ]]; then
        # macOS
        stat -f%z "$file_path" 2>/dev/null || echo 0
    elif [[ "$OSTYPE" == "msys" || "$OSTYPE" == "cygwin" ]]; then
        # Windows (MSYS/Cygwin)
        stat -c%s "$file_path" 2>/dev/null || echo 0
    else
        # Linux
        stat -c%s "$file_path" 2>/dev/null || echo 0
    fi
}

# 폰트 파일 복사 성공 여부
FONTS_COPIED=false

for FONT_FILE in "${FONT_FILES[@]}"; do
    SOURCE_PATH="$ASSETS_DIR/$FONT_FILE"
    TARGET_PATH="$AIRUN_DIR/$FONT_FILE"

    if [ -f "$SOURCE_PATH" ]; then
        # 대상 파일이 없거나 크기가 다른 경우에만 복사
        SOURCE_SIZE=$(get_file_size "$SOURCE_PATH")
        TARGET_SIZE=$(get_file_size "$TARGET_PATH")

        if [ ! -f "$TARGET_PATH" ] || [ "$SOURCE_SIZE" != "$TARGET_SIZE" ]; then
            if cp "$SOURCE_PATH" "$TARGET_PATH" 2>/dev/null; then
                # 파일 권한 설정 (읽기 권한 추가)
                chmod 644 "$TARGET_PATH"
                echo -e "${GREEN}폰트 파일 복사 완료: $FONT_FILE${NC}"
                FONTS_COPIED=true
            else
                echo -e "${RED}폰트 파일 복사 실패: $FONT_FILE${NC}"
            fi
        else
            echo -e "${YELLOW}폰트 파일이 이미 존재합니다: $FONT_FILE${NC}"
        fi
    else
        echo -e "${YELLOW}소스 폰트 파일을 찾을 수 없습니다: $SOURCE_PATH${NC}"
    fi
done

if [ "$FONTS_COPIED" = true ]; then
    echo -e "${GREEN}폰트 파일이 $AIRUN_DIR 디렉토리에 성공적으로 복사되었습니다.${NC}"
else
    echo -e "${YELLOW}폰트 파일 복사가 필요하지 않거나 실패했습니다.${NC}"
fi

echo -e "${GREEN}Installation completed!${NC}"

# Install desktop file and icon for Linux systems
if [[ "$(uname)" == "Linux" ]]; then
    echo -e "${GREEN}Installing desktop files...${NC}"

    # Check assets directory
    if [ -d "assets" ] && [ -f "assets/airun.desktop" ] && [ -f "assets/airun.png" ]; then
        # Check XDG environment variable and set default
        if [[ -z "$XDG_DESKTOP_DIR" ]]; then
            # Try to detect Korean environment desktop directory
            if [ -d "$HOME/바탕화면" ]; then
                XDG_DESKTOP_DIR="$HOME/바탕화면"
            else
                XDG_DESKTOP_DIR="$HOME/Desktop"
            fi
        fi

        # Install desktop file
        mkdir -p "$XDG_DESKTOP_DIR"
        cp assets/airun.desktop "$XDG_DESKTOP_DIR/" 2>/dev/null || echo -e "${YELLOW}Skipping desktop file copy...${NC}"

        # Install icon
        ICON_DIR="$HOME/.local/share/icons"
        mkdir -p "$ICON_DIR"
        cp assets/airun.png "$ICON_DIR/" 2>/dev/null || echo -e "${YELLOW}Skipping icon file copy...${NC}"

        # Modify icon path in desktop file
        if [ -f "$XDG_DESKTOP_DIR/airun.desktop" ]; then
            sed -i "s|Icon=airun.png|Icon=$ICON_DIR/airun.png|g" "$XDG_DESKTOP_DIR/airun.desktop"
            chmod +x "$XDG_DESKTOP_DIR/airun.desktop"
        fi

        # Register in application menu
        APP_DIR="$HOME/.local/share/applications"
        mkdir -p "$APP_DIR"
        cp "$XDG_DESKTOP_DIR/airun.desktop" "$APP_DIR/" 2>/dev/null || true
        update-desktop-database ~/.local/share/applications 2>/dev/null || true
    else
        echo -e "${YELLOW}Assets directory or required files not found. Skipping desktop file installation...${NC}"
    fi
fi

# Install shell completion (bash and zsh)
echo -e "${GREEN}Installing shell completion...${NC}"
if [[ "$(uname)" == "Linux" ]]; then
    COMPLETION_SCRIPT="scripts/airun-completion.bash"

    # completion 스크립트 존재 확인
    if [ ! -f "$COMPLETION_SCRIPT" ]; then
        echo -e "${YELLOW}Completion script not found at: $COMPLETION_SCRIPT${NC}"
        return
    fi

    # Bash completion 설치
    BASH_COMPLETION_DIRS=(
        "/etc/bash_completion.d"
        "/usr/share/bash-completion/completions"
        "$HOME/.local/share/bash-completion/completions"
    )

    COMPLETION_INSTALLED=false

    # Bash completion 파일 복사
    for dir in "${BASH_COMPLETION_DIRS[@]}"; do
        if [ -d "$dir" ]; then
            if [ -w "$dir" ]; then
                cp "$COMPLETION_SCRIPT" "$dir/airun"
                chmod 644 "$dir/airun"
                COMPLETION_INSTALLED=true
                echo -e "${GREEN}Bash completion installed to: $dir/airun${NC}"
                break
            elif [ "$dir" = "/etc/bash_completion.d" ] || [ "$dir" = "/usr/share/bash-completion/completions" ]; then
                sudo cp "$COMPLETION_SCRIPT" "$dir/airun"
                sudo chmod 644 "$dir/airun"
                COMPLETION_INSTALLED=true
                echo -e "${GREEN}Bash completion installed to: $dir/airun${NC}"
                break
            fi
        fi
    done

    # Bash completion 폴백 설치
    if [ "$COMPLETION_INSTALLED" = false ]; then
        COMPLETION_DIR="$HOME/.local/share/bash-completion/completions"
        mkdir -p "$COMPLETION_DIR"
        cp "$COMPLETION_SCRIPT" "$COMPLETION_DIR/airun"
        chmod 644 "$COMPLETION_DIR/airun"
        echo -e "${GREEN}Bash completion installed to: $COMPLETION_DIR/airun${NC}"

        # .bashrc에 completion 디렉토리 추가
        if ! grep -q "$COMPLETION_DIR" "$HOME/.bashrc" 2>/dev/null; then
            echo "# AI.RUN bash completion" >>"$HOME/.bashrc"
            echo "if [ -d $COMPLETION_DIR ]; then" >>"$HOME/.bashrc"
            echo "    for completion in $COMPLETION_DIR/*; do" >>"$HOME/.bashrc"
            echo "        [[ -f \$completion ]] && source \$completion" >>"$HOME/.bashrc"
            echo "    done" >>"$HOME/.bashrc"
            echo "fi" >>"$HOME/.bashrc"
        fi
    fi

    # Zsh completion 설치
    ZSH_COMPLETION_DIR="$HOME/.zsh/completion"
    mkdir -p "$ZSH_COMPLETION_DIR"
    # Zsh용 completion 스크립트 복사
    cp "scripts/airun-completion.zsh" "$ZSH_COMPLETION_DIR/_airun"
    chmod 644 "$ZSH_COMPLETION_DIR/_airun"
    echo -e "${GREEN}Zsh completion installed to: $ZSH_COMPLETION_DIR/_airun${NC}"

    # .zshrc에 completion 설정 추가
    if ! grep -q "fpath=($ZSH_COMPLETION_DIR" "$HOME/.zshrc" 2>/dev/null; then
        echo "" >>"$HOME/.zshrc"
        echo "# AI.RUN zsh completion" >>"$HOME/.zshrc"
        echo "# 기본 completion 시스템 설정" >>"$HOME/.zshrc"
        echo "autoload -Uz compinit" >>"$HOME/.zshrc"
        echo "compinit" >>"$HOME/.zshrc"
        echo "" >>"$HOME/.zshrc"
        echo "# completion 스타일 설정" >>"$HOME/.zshrc"
        echo "zstyle ':completion:*' menu select" >>"$HOME/.zshrc"
        echo "zstyle ':completion:*' matcher-list 'm:{a-zA-Z}={A-Za-z}'" >>"$HOME/.zshrc"
        echo "zstyle ':completion:*' list-colors \${(s.:.)LS_COLORS}" >>"$HOME/.zshrc"
        echo "zstyle ':completion:*' verbose yes" >>"$HOME/.zshrc"
        echo "zstyle ':completion:*:descriptions' format '%F{green}-- %d --%f'" >>"$HOME/.zshrc"
        echo "zstyle ':completion:*:messages' format '%F{yellow}-- %d --%f'" >>"$HOME/.zshrc"
        echo "zstyle ':completion:*:warnings' format '%F{red}-- no matches found --%f'" >>"$HOME/.zshrc"
        echo "" >>"$HOME/.zshrc"
        echo "# AI.RUN completion 설정" >>"$HOME/.zshrc"
        echo "fpath=($ZSH_COMPLETION_DIR \$fpath)" >>"$HOME/.zshrc"
    fi

    # 설치 결과 확인
    echo -e "\n${GREEN}Checking installation results:${NC}"
    echo -e "${YELLOW}Bash completion files:${NC}"
    if [ -f "$COMPLETION_DIR/airun" ]; then
        echo -e "${GREEN}✓ Found at: $COMPLETION_DIR/airun${NC}"
    fi

    echo -e "\n${YELLOW}Zsh completion files:${NC}"
    if [ -f "$ZSH_COMPLETION_DIR/_airun" ]; then
        echo -e "${GREEN}✓ Found at: $ZSH_COMPLETION_DIR/_airun${NC}"
    fi

    # 사용자에게 안내
    echo -e "${YELLOW}Please restart your shell or run:${NC}"
    echo -e "For bash: ${GREEN}source ~/.bashrc${NC}"
    echo -e "For zsh: ${GREEN}source ~/.zshrc${NC}"
fi

echo -e "${YELLOW}Usage:${NC}"
echo -e "1. Run with 'airun' command in terminal"
echo -e "2. Use 'airun -g' command for GUI mode"

# Redis 및 큐 시스템 상태 확인
echo -e "\n${GREEN}Queue System Status:${NC}"
if redis-cli ping | grep -q "PONG" 2>/dev/null; then
    echo -e "${GREEN}✓ Redis is running${NC}"
else
    echo -e "${RED}✗ Redis is not running. API usage tracking may not work properly.${NC}"
    echo -e "${YELLOW}  Start Redis with: sudo systemctl start redis-server${NC}"
fi

echo -e "\n${GREEN}Installation completed successfully!${NC}"
echo -e "\n${YELLOW}환경 설정:${NC}"
echo -e "AI.RUN 서비스 환경 설정은 다음 파일에서 관리됩니다:"
echo -e "${GREEN}$(pwd)/.env${NC}"
echo -e "데이터베이스 비밀번호 등 민감한 정보 변경 시 이 파일을 수정하세요."

# systemd 서비스 설치
echo -e "${GREEN}시스템 서비스 설치를 시작합니다...${NC}"

# 시스템 환경 체크
INIT_SYSTEM="unknown"
IN_CONTAINER=false

# Docker 컨테이너 내부인지 확인
if [ -f "/.dockerenv" ] || grep -q docker /proc/1/cgroup 2>/dev/null; then
    IN_CONTAINER=true
    echo -e "${YELLOW}Docker 컨테이너 환경이 감지되었습니다.${NC}"
fi

# init 시스템 확인
if command -v systemctl &>/dev/null && systemctl list-units &>/dev/null; then
    INIT_SYSTEM="systemd"
    echo -e "${GREEN}systemd가 감지되었습니다.${NC}"
elif [ -d "/etc/init.d" ]; then
    INIT_SYSTEM="sysvinit"
    echo -e "${YELLOW}전통적인 SysV init 시스템이 감지되었습니다.${NC}"
else
    echo -e "${RED}지원되는 init 시스템을 찾을 수 없습니다.${NC}"
fi

# 현재 설치 경로와 사용자 가져오기
CURRENT_DIR=$(pwd)

# 사용자 정보 가져오기 (sudo 포함)
CURRENT_USER=""
if [ -n "$SUDO_USER" ]; then
    CURRENT_USER="$SUDO_USER"
else
    if command -v logname &>/dev/null; then
        CURRENT_USER=$(logname 2>/dev/null)
    fi
    if [ -z "$CURRENT_USER" ]; then
        CURRENT_USER=$(whoami)
    fi
    if [ -z "$CURRENT_USER" ]; then
        CURRENT_USER=$(id -un)
    fi
    if [ -z "$CURRENT_USER" ]; then
        CURRENT_USER=$(basename "$HOME")
    fi
fi

if [ -z "$CURRENT_USER" ]; then
    echo -e "${RED}사용자 정보를 가져올 수 없습니다.${NC}"
    exit 1
fi

echo -e "${GREEN}설치 사용자: $CURRENT_USER${NC}"

# 로그 디렉토리 생성
echo -e "${YELLOW}시스템 로그 디렉토리를 생성합니다...${NC}"
sudo mkdir -p /var/log/airun
sudo chown $CURRENT_USER:$CURRENT_USER /var/log/airun
sudo chmod 755 /var/log/airun

if [ "$INIT_SYSTEM" = "systemd" ] && [ "$IN_CONTAINER" = false ]; then
    # systemd 서비스 파일 생성
    echo -e "${YELLOW}systemd 서비스 파일을 생성합니다...${NC}"

    # API 서버 서비스 파일 생성
    cat >airun-api.service <<EOF
[Unit]
Description=Airun API Server
After=network.target redis.service

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR
Environment=NODE_ENV=production
Environment=PATH=/usr/local/bin:/usr/bin:/bin:$HOME/.nvm/versions/node/v22.18.0/bin
Environment=AIRUN_PATH=$CURRENT_DIR
Environment=HOME=$HOME
Environment=NODE_PATH=$CURRENT_DIR/node_modules
Environment=PYTHONPATH=$CURRENT_DIR
# Node.js로 직접 서버 실행
ExecStart=/usr/bin/env node api_server.js
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-api.log
StandardError=append:/var/log/airun/airun-api.error.log

[Install]
WantedBy=multi-user.target
EOF

    # RAG 모니터링 서비스 파일 생성
    cat >airun-rag.service <<EOF
[Unit]
Description=AI.RUN RAG Watch Service
After=network.target

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
RuntimeDirectory=airun
RuntimeDirectoryMode=0755
WorkingDirectory=$CURRENT_DIR
Environment=PYTHONPATH=$CURRENT_DIR
Environment=PYTHONUNBUFFERED=1
Environment=HOME=$HOME
Environment=AIRUN_LOG_DIR=/var/log/airun
# 가상 환경의 Python 인터프리터를 직접 사용
ExecStart=$HOME/.airun_venv/bin/python3 -u $CURRENT_DIR/plugins/rag/airun-rag.py
ExecStop=/bin/kill -TERM \$MAINPID
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-rag.log
StandardError=append:/var/log/airun/airun-rag.error.log

# 보안 설정
ProtectSystem=full
NoNewPrivileges=true
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

    # RQ worker 설정 추가
    echo -e "${GREEN}RQ worker 설정을 시작합니다...${NC}"

    # Python 가상환경에 RQ 설치
    "$VENV_PATH/bin/pip" install rq

    # RQ worker 시작을 위한 systemd 서비스 파일 생성
    cat >airun-rq-worker.service <<EOF
[Unit]
Description=AI.RUN RQ Worker Service
After=network.target redis.service

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR
Environment=HOME=$HOME
Environment=NODE_PATH=$CURRENT_DIR/node_modules
Environment=PYTHONPATH=$CURRENT_DIR:$CURRENT_DIR/plugins
Environment=PYTHONUNBUFFERED=1
Environment=AIRUN_HOME=$CURRENT_DIR
Environment=AIRUN_VENV=$HOME/.airun_venv
ExecStart=/bin/bash -c 'cd $CURRENT_DIR && . $HOME/.airun_venv/bin/activate && exec rq worker image_processing'
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-rq-worker.log
StandardError=append:/var/log/airun/airun-rq-worker.error.log

[Install]
WantedBy=multi-user.target
EOF

    # 웹검색 서비스 파일 생성
    echo -e "${GREEN}웹검색 서비스 설정을 시작합니다...${NC}"
    cat >airun-websearch.service <<EOF
[Unit]
Description=AI.RUN WebSearch Server (Port 5610)
After=network.target

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR/services/websearch
Environment=PYTHONPATH=$CURRENT_DIR
Environment=PYTHONUNBUFFERED=1
Environment=HOME=$HOME
Environment=WEBSEARCH_SERVER_HOST=0.0.0.0
Environment=WEBSEARCH_SERVER_PORT=5610
Environment=WEBSEARCH_SERVER_WORKERS=1
ExecStart=$HOME/.airun_venv/bin/python3 -u airun-websearch-server.py
ExecStop=/bin/kill -TERM \$MAINPID
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-websearch.log
StandardError=append:/var/log/airun/airun-websearch.error.log

# 보안 설정
ProtectSystem=full
NoNewPrivileges=true
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

    # 보고서 생성 서비스 파일 생성
    echo -e "${GREEN}보고서 생성 서비스 설정을 시작합니다...${NC}"
    cat >airun-report.service <<EOF
[Unit]
Description=AI.RUN Report Generation Server (Port 5620)
After=network.target airun-rag.service airun-websearch.service
Wants=airun-rag.service airun-websearch.service

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR/services/report
Environment=PYTHONPATH=$CURRENT_DIR
Environment=PYTHONUNBUFFERED=1
Environment=HOME=$HOME
Environment=REPORT_SERVER_HOST=0.0.0.0
Environment=REPORT_SERVER_PORT=5620
Environment=REPORT_SERVER_WORKERS=1
ExecStart=$HOME/.airun_venv/bin/python3 -u airun-report-server.py
ExecStop=/bin/kill -TERM \$MAINPID
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-report.log
StandardError=append:/var/log/airun/airun-report.error.log

# 보안 설정
ProtectSystem=full
NoNewPrivileges=true
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

    # 사업분석 서비스 파일 생성
    echo -e "${GREEN}사업분석 서비스 설정을 시작합니다...${NC}"
    cat >airun-biz.service <<EOF
[Unit]
Description=AI.RUN Business Analysis Server (Port 5640)
After=network.target
Wants=network.target

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR/services/biz-announcement
Environment=PYTHONPATH=$CURRENT_DIR
Environment=PYTHONUNBUFFERED=1
Environment=HOME=$HOME
Environment=BUSINESS_SERVER_HOST=0.0.0.0
Environment=BUSINESS_SERVER_PORT=5640
Environment=BUSINESS_SERVER_WORKERS=1
ExecStart=$HOME/.airun_venv/bin/python3 -u main.py
ExecStop=/bin/kill -TERM \$MAINPID
Restart=always
RestartSec=3
StandardOutput=append:/var/log/airun/airun-biz.log
StandardError=append:/var/log/airun/airun-biz.error.log

# 보안 설정
ProtectSystem=full
NoNewPrivileges=true
PrivateTmp=true

[Install]
WantedBy=multi-user.target
EOF

    # LLM Fine-tuning 서비스 파일 생성
    echo -e "${GREEN}LLM Fine-tuning 서비스 설정을 시작합니다...${NC}"
    cat >airun-llm-finetune.service <<EOF
[Unit]
Description=AIRUN LLM Fine-tuning Service
Documentation=https://github.com/airun/llm-finetune
After=network.target postgresql.service redis.service
Wants=postgresql.service redis.service

[Service]
Type=simple
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR/services/llm-finetune
Environment=NODE_ENV=production
Environment=LLM_FINETUNE_PORT=5630
Environment=PATH=/usr/local/bin:/usr/bin:/bin:$HOME/.nvm/versions/node/v22.18.0/bin
Environment=HOME=$HOME
Environment=NODE_PATH=$CURRENT_DIR/node_modules:$CURRENT_DIR/services/llm-finetune/node_modules
ExecStart=/usr/bin/env node llm-finetune-server.js
Restart=always
RestartSec=10
StandardOutput=append:$HOME/.airun/logs/airun-llm-finetune.log
StandardError=append:$HOME/.airun/logs/airun-llm-finetune.error.log

[Install]
WantedBy=multi-user.target
EOF

    # Business Daily Collection 서비스 파일 생성
    echo -e "${GREEN}Business Daily Collection 서비스 설정을 시작합니다...${NC}"
    cat >airun-biz-daily-collection.service <<EOF
[Unit]
Description=AIRUN Business Announcement Daily Data Collection
Wants=airun-biz.service
After=airun-biz.service

[Service]
Type=oneshot
User=$CURRENT_USER
Group=$CURRENT_USER
WorkingDirectory=$CURRENT_DIR/services/biz-announcement/scripts
Environment=PATH=/usr/local/sbin:/usr/local/bin:/usr/sbin:/usr/bin:/sbin:/bin
Environment=BIZ_SERVICE_URL=http://localhost:5640
ExecStart=/bin/bash $CURRENT_DIR/services/biz-announcement/scripts/daily_collection.sh
StandardOutput=append:/var/log/airun/airun-biz-daily-collection.log
StandardError=append:/var/log/airun/airun-biz-daily-collection.error.log

[Install]
WantedBy=multi-user.target
EOF

    # Business Daily Collection 타이머 파일 생성
    echo -e "${GREEN}Business Daily Collection 타이머 설정을 시작합니다...${NC}"
    cat >airun-biz-daily-collection.timer <<EOF
[Unit]
Description=AIRUN Business Announcement Daily Data Collection Timer
Requires=airun-biz.service

[Timer]
OnCalendar=*-*-* 09:00:00
Persistent=true
RandomizedDelaySec=300

[Install]
WantedBy=timers.target
EOF

    # 서비스 파일들 설치
    echo -e "${YELLOW}systemd 서비스 파일을 설치합니다...${NC}"
    sudo cp airun-api.service /etc/systemd/system/
    sudo cp airun-rag.service /etc/systemd/system/
    sudo cp airun-rq-worker.service /etc/systemd/system/
    sudo cp airun-websearch.service /etc/systemd/system/
    sudo cp airun-report.service /etc/systemd/system/
    sudo cp airun-biz.service /etc/systemd/system/
    sudo cp airun-llm-finetune.service /etc/systemd/system/
    sudo cp airun-biz-daily-collection.service /etc/systemd/system/
    sudo cp airun-biz-daily-collection.timer /etc/systemd/system/
    sudo chmod 644 /etc/systemd/system/airun-api.service
    sudo chmod 644 /etc/systemd/system/airun-rag.service
    sudo chmod 644 /etc/systemd/system/airun-rq-worker.service
    sudo chmod 644 /etc/systemd/system/airun-websearch.service
    sudo chmod 644 /etc/systemd/system/airun-report.service
    sudo chmod 644 /etc/systemd/system/airun-biz.service
    sudo chmod 644 /etc/systemd/system/airun-llm-finetune.service
    sudo chmod 644 /etc/systemd/system/airun-biz-daily-collection.service
    sudo chmod 644 /etc/systemd/system/airun-biz-daily-collection.timer

    # systemd 데몬 리로드
    echo -e "${YELLOW}systemd 데몬을 리로드합니다...${NC}"
    sudo systemctl daemon-reload

    # 서비스 활성화 및 시작
    echo -e "${YELLOW}서비스를 활성화하고 시작합니다...${NC}"
    sudo systemctl enable airun-api airun-rag airun-rq-worker airun-websearch airun-report airun-biz airun-llm-finetune airun-biz-daily-collection.timer
    sudo systemctl start airun-api airun-rag airun-rq-worker airun-websearch airun-report airun-biz airun-llm-finetune airun-biz-daily-collection.timer

    # 서비스 상태 확인
    echo -e "${GREEN}서비스 상태를 확인합니다...${NC}"
    if systemctl is-active --quiet airun-api && systemctl is-active --quiet airun-rag && systemctl is-active --quiet airun-rq-worker && systemctl is-active --quiet airun-websearch && systemctl is-active --quiet airun-report && systemctl is-active --quiet airun-biz && systemctl is-active --quiet airun-llm-finetune; then
        echo -e "${GREEN}✓ 모든 서비스가 실행 중입니다${NC}"
    else
        echo -e "${RED}✗ 일부 서비스 시작 실패${NC}"
        echo -e "${YELLOW}로그 확인:${NC}"
        echo -e "${YELLOW}- API 서버: sudo journalctl -u airun-api -f${NC}"
        echo -e "${YELLOW}- RAG 모니터: sudo journalctl -u airun-rag -f${NC}"
        echo -e "${YELLOW}- RQ Worker: sudo journalctl -u airun-rq-worker -f${NC}"
        echo -e "${YELLOW}- 웹검색 서비스: sudo journalctl -u airun-websearch -f${NC}"
        echo -e "${YELLOW}- 보고서 생성: sudo journalctl -u airun-report -f${NC}"
        echo -e "${YELLOW}- 사업분석 서비스: sudo journalctl -u airun-biz -f${NC}"
    fi

    echo -e "${GREEN}systemd 서비스 설치가 완료되었습니다.${NC}"
    echo -e "${YELLOW}서비스 제어 명령어:${NC}"
    echo -e "API 서버 (5500포트):"
    echo -e "  시작: ${GREEN}sudo systemctl start airun-api${NC}"
    echo -e "  중지: ${GREEN}sudo systemctl stop airun-api${NC}"
    echo -e "  재시작: ${GREEN}sudo systemctl restart airun-api${NC}"
    echo -e "  상태: ${GREEN}sudo systemctl status airun-api${NC}"
    echo -e "RAG 서비스 (5600포트):"
    echo -e "  시작: ${GREEN}sudo systemctl start airun-rag${NC}"
    echo -e "  중지: ${GREEN}sudo systemctl stop airun-rag${NC}"
    echo -e "  재시작: ${GREEN}sudo systemctl restart airun-rag${NC}"
    echo -e "  상태: ${GREEN}sudo systemctl status airun-rag${NC}"
    echo -e "웹검색 서비스 (5610포트):"
    echo -e "  시작: ${GREEN}sudo systemctl start airun-websearch${NC}"
    echo -e "  중지: ${GREEN}sudo systemctl stop airun-websearch${NC}"
    echo -e "  재시작: ${GREEN}sudo systemctl restart airun-websearch${NC}"
    echo -e "  상태: ${GREEN}sudo systemctl status airun-websearch${NC}"
    echo -e "보고서 생성 (5620포트):"
    echo -e "  시작: ${GREEN}sudo systemctl start airun-report${NC}"
    echo -e "  중지: ${GREEN}sudo systemctl stop airun-report${NC}"
    echo -e "  재시작: ${GREEN}sudo systemctl restart airun-report${NC}"
    echo -e "  상태: ${GREEN}sudo systemctl status airun-report${NC}"
    echo -e "LLM Fine-tuning (5630포트):"
    echo -e "  시작: ${GREEN}sudo systemctl start airun-llm-finetune${NC}"
    echo -e "  중지: ${GREEN}sudo systemctl stop airun-llm-finetune${NC}"
    echo -e "  재시작: ${GREEN}sudo systemctl restart airun-llm-finetune${NC}"
    echo -e "  상태: ${GREEN}sudo systemctl status airun-llm-finetune${NC}"
    echo -e "사업분석 (5640포트):"
    echo -e "  시작: ${GREEN}sudo systemctl start airun-biz${NC}"
    echo -e "  중지: ${GREEN}sudo systemctl stop airun-biz${NC}"
    echo -e "  재시작: ${GREEN}sudo systemctl restart airun-biz${NC}"
    echo -e "  상태: ${GREEN}sudo systemctl status airun-biz${NC}"
    echo -e "RQ Worker:"
    echo -e "  시작: ${GREEN}sudo systemctl start airun-rq-worker${NC}"
    echo -e "  중지: ${GREEN}sudo systemctl stop airun-rq-worker${NC}"
    echo -e "  재시작: ${GREEN}sudo systemctl restart airun-rq-worker${NC}"
    echo -e "  상태: ${GREEN}sudo systemctl status airun-rq-worker${NC}"

else
    # init.d 스크립트 생성
    echo -e "${YELLOW}init.d 스크립트를 생성합니다...${NC}"

    # API 서버 init.d 스크립트
    cat >airun-api <<EOF
#!/bin/sh
### BEGIN INIT INFO
# Provides:          airun-api
# Required-Start:    \$remote_fs \$syslog \$network
# Required-Stop:     \$remote_fs \$syslog \$network
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: Airun API Server
# Description:       Airun API Server service
### END INIT INFO

USER=$CURRENT_USER
HOME=$HOME
DAEMON="$HOME/.nvm/versions/node/v22.18.0/bin/node"
DAEMON_ARGS="$CURRENT_DIR/api_server.js"
PIDFILE=/var/run/airun-api.pid
LOGFILE=/var/log/airun/airun-api.log
ERRFILE=/var/log/airun/airun-api.error.log

export NODE_ENV=production
export PATH=/usr/local/bin:/usr/bin:/bin:$HOME/.nvm/versions/node/v22.18.0/bin
export AIRUN_PATH=$CURRENT_DIR
export HOME=$HOME
export NODE_PATH=$CURRENT_DIR/node_modules

start() {
    echo "Starting airun-api..."
    if [ -f \$PIDFILE ]; then
        echo "PID file exists. Service may be running already"
        return 1
    fi
    
    start-stop-daemon --start --background \\
        --make-pidfile --pidfile \$PIDFILE \\
        --chuid \$USER \\
        --exec \$DAEMON -- \$DAEMON_ARGS >> \$LOGFILE 2>> \$ERRFILE
}

stop() {
    echo "Stopping airun-api..."
    start-stop-daemon --stop --pidfile \$PIDFILE --retry 10
    rm -f \$PIDFILE
}

case "\$1" in
    start)
        start
        ;;
    stop)
        stop
        ;;
    restart)
        stop
        start
        ;;
    status)
        if [ -f \$PIDFILE ]; then
            echo "Service is running"
        else
            echo "Service is stopped"
        fi
        ;;
    *)
        echo "Usage: \$0 {start|stop|restart|status}"
        exit 1
        ;;
esac

exit 0
EOF

    # RAG 서비스 init.d 스크립트
    cat >airun-rag <<EOF
#!/bin/sh
### BEGIN INIT INFO
# Provides:          airun-rag
# Required-Start:    \$remote_fs \$syslog \$network
# Required-Stop:     \$remote_fs \$syslog \$network
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: Airun RAG Service
# Description:       Airun RAG monitoring service
### END INIT INFO

USER=$CURRENT_USER
HOME=$HOME
DAEMON="$HOME/.airun_venv/bin/python3"
DAEMON_ARGS="$CURRENT_DIR/plugins/rag/airun-rag.py"
PIDFILE=/var/run/airun-rag.pid
LOGFILE=/var/log/airun/airun-rag.log
ERRFILE=/var/log/airun/airun-rag.error.log

export PYTHONPATH=$CURRENT_DIR
export PYTHONUNBUFFERED=1
export HOME=$HOME
export AIRUN_LOG_DIR=/var/log/airun

start() {
    echo "Starting airun-rag..."
    if [ -f \$PIDFILE ]; then
        echo "PID file exists. Service may be running already"
        return 1
    fi
    
    start-stop-daemon --start --background \\
        --make-pidfile --pidfile \$PIDFILE \\
        --chuid \$USER \\
        --exec \$DAEMON -- \$DAEMON_ARGS >> \$LOGFILE 2>> \$ERRFILE
}

stop() {
    echo "Stopping airun-rag..."
    start-stop-daemon --stop --pidfile \$PIDFILE --retry 10
    rm -f \$PIDFILE
}

case "\$1" in
    start)
        start
        ;;
    stop)
        stop
        ;;
    restart)
        stop
        start
        ;;
    status)
        if [ -f \$PIDFILE ]; then
            echo "Service is running"
        else
            echo "Service is stopped"
        fi
        ;;
    *)
        echo "Usage: \$0 {start|stop|restart|status}"
        exit 1
        ;;
esac

exit 0
EOF

    # RQ Worker init.d 스크립트
    cat >airun-rq-worker <<EOF
#!/bin/sh
### BEGIN INIT INFO
# Provides:          airun-rq-worker
# Required-Start:    \$remote_fs \$syslog \$network redis-server
# Required-Stop:     \$remote_fs \$syslog \$network redis-server
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: Airun RQ Worker
# Description:       Airun RQ Worker service
### END INIT INFO

USER=$CURRENT_USER
HOME=$HOME
DAEMON="$HOME/.airun_venv/bin/rq"
DAEMON_ARGS="worker image_processing"
PIDFILE=/var/run/airun-rq-worker.pid
LOGFILE=/var/log/airun/airun-rq-worker.log
ERRFILE=/var/log/airun/airun-rq-worker.error.log

export HOME=$HOME
export NODE_PATH=$CURRENT_DIR/node_modules
export PYTHONPATH=$CURRENT_DIR:$CURRENT_DIR/plugins
export PYTHONUNBUFFERED=1
export AIRUN_HOME=$CURRENT_DIR
export AIRUN_VENV=$HOME/.airun_venv

start() {
    echo "Starting airun-rq-worker..."
    if [ -f \$PIDFILE ]; then
        echo "PID file exists. Service may be running already"
        return 1
    fi
    
    cd $CURRENT_DIR
    . $HOME/.airun_venv/bin/activate
    
    start-stop-daemon --start --background \\
        --make-pidfile --pidfile \$PIDFILE \\
        --chuid \$USER \\
        --exec \$DAEMON -- \$DAEMON_ARGS >> \$LOGFILE 2>> \$ERRFILE
}

stop() {
    echo "Stopping airun-rq-worker..."
    start-stop-daemon --stop --pidfile \$PIDFILE --retry 10
    rm -f \$PIDFILE
}

case "\$1" in
    start)
        start
        ;;
    stop)
        stop
        ;;
    restart)
        stop
        start
        ;;
    status)
        if [ -f \$PIDFILE ]; then
            echo "Service is running"
        else
            echo "Service is stopped"
        fi
        ;;
    *)
        echo "Usage: \$0 {start|stop|restart|status}"
        exit 1
        ;;
esac

exit 0
EOF

    # 웹검색 서비스 init.d 스크립트
    cat >airun-websearch <<EOF
#!/bin/sh
### BEGIN INIT INFO
# Provides:          airun-websearch
# Required-Start:    \$remote_fs \$syslog \$network
# Required-Stop:     \$remote_fs \$syslog \$network
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: Airun WebSearch Server
# Description:       Airun WebSearch service (Port 5610)
### END INIT INFO

USER=$CURRENT_USER
HOME=$HOME
DAEMON="$HOME/.airun_venv/bin/python3"
DAEMON_ARGS="$CURRENT_DIR/services/websearch/airun-websearch-server.py"
PIDFILE=/var/run/airun-websearch.pid
LOGFILE=/var/log/airun/airun-websearch.log
ERRFILE=/var/log/airun/airun-websearch.error.log

export PYTHONPATH=$CURRENT_DIR
export PYTHONUNBUFFERED=1
export HOME=$HOME
export WEBSEARCH_SERVER_HOST=0.0.0.0
export WEBSEARCH_SERVER_PORT=5610
export WEBSEARCH_SERVER_WORKERS=1

start() {
    echo "Starting airun-websearch..."
    if [ -f \$PIDFILE ]; then
        echo "PID file exists. Service may be running already"
        return 1
    fi
    
    cd $CURRENT_DIR/services/websearch
    
    start-stop-daemon --start --background \\
        --make-pidfile --pidfile \$PIDFILE \\
        --chuid \$USER \\
        --chdir $CURRENT_DIR/services/websearch \\
        --exec \$DAEMON -- -u \$DAEMON_ARGS >> \$LOGFILE 2>> \$ERRFILE
}

stop() {
    echo "Stopping airun-websearch..."
    start-stop-daemon --stop --pidfile \$PIDFILE --retry 10
    rm -f \$PIDFILE
}

case "\$1" in
    start)
        start
        ;;
    stop)
        stop
        ;;
    restart)
        stop
        start
        ;;
    status)
        if [ -f \$PIDFILE ]; then
            echo "Service is running"
        else
            echo "Service is stopped"
        fi
        ;;
    *)
        echo "Usage: \$0 {start|stop|restart|status}"
        exit 1
        ;;
esac

exit 0
EOF

    # 보고서 생성 서비스 init.d 스크립트
    cat >airun-report <<EOF
#!/bin/sh
### BEGIN INIT INFO
# Provides:          airun-report
# Required-Start:    \$remote_fs \$syslog \$network airun-rag airun-websearch
# Required-Stop:     \$remote_fs \$syslog \$network airun-rag airun-websearch
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: Airun Report Generation Server
# Description:       Airun Report Generation service (Port 5620)
### END INIT INFO

USER=$CURRENT_USER
HOME=$HOME
DAEMON="$HOME/.airun_venv/bin/python3"
DAEMON_ARGS="$CURRENT_DIR/services/report/airun-report-server.py"
PIDFILE=/var/run/airun-report.pid
LOGFILE=/var/log/airun/airun-report.log
ERRFILE=/var/log/airun/airun-report.error.log

export PYTHONPATH=$CURRENT_DIR
export PYTHONUNBUFFERED=1
export HOME=$HOME
export REPORT_SERVER_HOST=0.0.0.0
export REPORT_SERVER_PORT=5620
export REPORT_SERVER_WORKERS=1

start() {
    echo "Starting airun-report..."
    if [ -f \$PIDFILE ]; then
        echo "PID file exists. Service may be running already"
        return 1
    fi
    
    cd $CURRENT_DIR/services/report
    
    start-stop-daemon --start --background \\
        --make-pidfile --pidfile \$PIDFILE \\
        --chuid \$USER \\
        --chdir $CURRENT_DIR/services/report \\
        --exec \$DAEMON -- -u \$DAEMON_ARGS >> \$LOGFILE 2>> \$ERRFILE
}

stop() {
    echo "Stopping airun-report..."
    start-stop-daemon --stop --pidfile \$PIDFILE --retry 10
    rm -f \$PIDFILE
}

case "\$1" in
    start)
        start
        ;;
    stop)
        stop
        ;;
    restart)
        stop
        start
        ;;
    status)
        if [ -f \$PIDFILE ]; then
            echo "Service is running"
        else
            echo "Service is stopped"
        fi
        ;;
    *)
        echo "Usage: \$0 {start|stop|restart|status}"
        exit 1
        ;;
esac

exit 0
EOF

    # 스크립트에 실행 권한 부여
    chmod +x airun-api airun-rag airun-rq-worker airun-websearch airun-report

    # init.d 디렉토리로 스크립트 복사
    echo -e "${YELLOW}서비스 스크립트를 /etc/init.d로 복사합니다...${NC}"
    sudo cp airun-api airun-rag airun-rq-worker airun-websearch airun-report /etc/init.d/
    sudo chmod 755 /etc/init.d/airun-api /etc/init.d/airun-rag /etc/init.d/airun-rq-worker /etc/init.d/airun-websearch /etc/init.d/airun-report

    # 서비스 등록 (update-rc.d 사용)
    if command -v update-rc.d &>/dev/null; then
        echo -e "${YELLOW}서비스를 시스템에 등록합니다...${NC}"
        sudo update-rc.d airun-api defaults
        sudo update-rc.d airun-rag defaults
        sudo update-rc.d airun-rq-worker defaults
        sudo update-rc.d airun-websearch defaults
        sudo update-rc.d airun-report defaults
    fi

    # 서비스 시작
    echo -e "${YELLOW}서비스를 시작합니다...${NC}"
    sudo service airun-api start
    sudo service airun-rag start
    sudo service airun-rq-worker start
    sudo service airun-websearch start
    sudo service airun-report start

    # 서비스 상태 확인
    echo -e "${GREEN}서비스 상태를 확인합니다...${NC}"
    sudo service airun-api status
    sudo service airun-rag status
    sudo service airun-rq-worker status
    sudo service airun-websearch status
    sudo service airun-report status

    echo -e "${GREEN}서비스 설치가 완료되었습니다.${NC}"
    echo -e "${YELLOW}서비스 제어 명령어:${NC}"
    echo -e "API 서버 (5500포트):"
    echo -e "  시작: ${GREEN}sudo service airun-api start${NC}"
    echo -e "  중지: ${GREEN}sudo service airun-api stop${NC}"
    echo -e "  재시작: ${GREEN}sudo service airun-api restart${NC}"
    echo -e "  상태: ${GREEN}sudo service airun-api status${NC}"
    echo -e "RAG 서비스 (5600포트):"
    echo -e "  시작: ${GREEN}sudo service airun-rag start${NC}"
    echo -e "  중지: ${GREEN}sudo service airun-rag stop${NC}"
    echo -e "  재시작: ${GREEN}sudo service airun-rag restart${NC}"
    echo -e "  상태: ${GREEN}sudo service airun-rag status${NC}"
    echo -e "웹검색 서비스 (5610포트):"
    echo -e "  시작: ${GREEN}sudo service airun-websearch start${NC}"
    echo -e "  중지: ${GREEN}sudo service airun-websearch stop${NC}"
    echo -e "  재시작: ${GREEN}sudo service airun-websearch restart${NC}"
    echo -e "  상태: ${GREEN}sudo service airun-websearch status${NC}"
    echo -e "보고서 생성 (5620포트):"
    echo -e "  시작: ${GREEN}sudo service airun-report start${NC}"
    echo -e "  중지: ${GREEN}sudo service airun-report stop${NC}"
    echo -e "  재시작: ${GREEN}sudo service airun-report restart${NC}"
    echo -e "  상태: ${GREEN}sudo service airun-report status${NC}"
    echo -e "사업분석 (5640포트):"
    echo -e "  시작: ${GREEN}sudo service airun-biz start${NC}"
    echo -e "  중지: ${GREEN}sudo service airun-biz stop${NC}"
    echo -e "  재시작: ${GREEN}sudo service airun-biz restart${NC}"
    echo -e "  상태: ${GREEN}sudo service airun-biz status${NC}"
    echo -e "RQ Worker:"
    echo -e "  시작: ${GREEN}sudo service airun-rq-worker start${NC}"
    echo -e "  중지: ${GREEN}sudo service airun-rq-worker stop${NC}"
    echo -e "  재시작: ${GREEN}sudo service airun-rq-worker restart${NC}"
    echo -e "  상태: ${GREEN}sudo service airun-rq-worker status${NC}"
fi

# setup_chromadb_directory() 함수는 PostgreSQL 사용으로 인해 더 이상 필요하지 않음
# ChromaDB 디렉토리 설정이 PostgreSQL로 대체됨

# 로깅 설정
setup_logging() {
    local LOG_DIR="$HOME/.airun/logs"
    local LOG_FILE="$LOG_DIR/install.log"
    local ERROR_LOG="$LOG_DIR/install.error.log"

    mkdir -p "$LOG_DIR"
    touch "$LOG_FILE" "$ERROR_LOG"

    # 로그 파일 로테이션 (이전 로그 백업)
    if [ -f "$LOG_FILE" ]; then
        mv "$LOG_FILE" "$LOG_FILE.$(date +%Y%m%d-%H%M%S)"
    fi

    # 로그 및 에러 출력 리다이렉션
    exec 1> >(tee -a "$LOG_FILE")
    exec 2> >(tee -a "$ERROR_LOG" >&2)

    echo "=== Installation started at $(date) ==="
}

# 설치 검증 함수
verify_installation() {
    local errors=0
    echo -e "${GREEN}설치 상태를 검증합니다...${NC}"

    # 가상환경 검증
    if [ ! -f "$VENV_PATH/bin/python" ]; then
        echo -e "${RED}가상환경이 올바르게 설치되지 않았습니다.${NC}"
        errors=$((errors + 1))
    fi

    # 필수 Python 패키지 검증
    local required_packages=("redis" "torch" "transformers")
    for package in "${required_packages[@]}"; do
        if ! "$VENV_PATH/bin/pip" show "$package" &>/dev/null; then
            echo -e "${RED}패키지 $package가 설치되지 않았습니다.${NC}"
            errors=$((errors + 1))
        fi
    done

    # Redis 연결 검증
    if ! redis-cli ping &>/dev/null; then
        echo -e "${RED}Redis 서버에 연결할 수 없습니다.${NC}"
        errors=$((errors + 1))
    fi

    # PostgreSQL 기반 RAG 시스템 사용으로 ChromaDB 디렉토리 검증 불필요

    # systemd 서비스 검증 (systemd가 지원되는 환경에서만)
    if command -v systemctl &>/dev/null && [ "$INIT_SYSTEM" = "systemd" ]; then
        local services=("airun-api" "airun-rag" "airun-rq-worker" "airun-websearch" "airun-report" "airun-biz" "airun-llm-finetune")
        for service in "${services[@]}"; do
            if ! systemctl is-active --quiet "$service"; then
                echo -e "${RED}$service 서비스가 실행되지 않았습니다.${NC}"
                errors=$((errors + 1))
            fi
        done
    else
        echo -e "${YELLOW}systemd가 지원되지 않는 환경입니다. 서비스 상태 확인을 건너뜁니다.${NC}"
    fi

    if [ $errors -eq 0 ]; then
        echo -e "${GREEN}모든 구성 요소가 정상적으로 설치되었습니다.${NC}"
        return 0
    else
        echo -e "${RED}설치 중 $errors개의 문제가 발견되었습니다.${NC}"
        return 1
    fi
}

# 롤백 함수
rollback_installation() {
    echo -e "${YELLOW}설치 실패로 인한 롤백을 시작합니다...${NC}"

    # 프로세스 종료
    if pgrep -f "airun-rag.py" >/dev/null; then
        pkill -f "airun-rag.py"
    fi

    # Redis 데이터 초기화
    if redis-cli ping &>/dev/null; then
        redis-cli FLUSHALL
    fi

    # ChromaDB 디렉토리 초기화
    if [ -d "$HOME/.airun/chromadb" ]; then
        rm -rf "$HOME/.airun/chromadb"/*
    fi

    echo -e "${YELLOW}롤백이 완료되었습니다.${NC}"
}

# vLLM 서비스 설치 함수
install_vllm_service() {
    echo -e "${GREEN}vLLM 서비스를 설치합니다...${NC}"

    if [ "$INIT_SYSTEM" = "systemd" ] && [ "$IN_CONTAINER" = false ]; then
        # systemd 서비스 파일 생성
        echo -e "${YELLOW}vLLM systemd 서비스 파일을 생성합니다...${NC}"
        cat >airun-vllm.service <<EOF
[Unit]
Description=AI.RUN vLLM Server
After=network.target
Wants=network.target

[Service]
Type=exec
User=$CURRENT_USER
Group=$CURRENT_USER
Environment=PYTHONUNBUFFERED=1
Environment=HOME=$HOME
Environment=AIRUN_PATH=$CURRENT_DIR
Environment=PYTHONPATH=$CURRENT_DIR
EnvironmentFile=$HOME/.airun/airun.conf
WorkingDirectory=$CURRENT_DIR/services/vllm
ExecStart=$HOME/.airun_venv/bin/python3 -m vllm.entrypoints.openai.api_server --model naver-hyperclovax/HyperCLOVAX-SEED-Text-Instruct-0.5B --download-dir $HOME/.airun/models --port 11400 --host 0.0.0.0
StandardOutput=append:/var/log/airun/airun-vllm.log
StandardError=append:/var/log/airun/airun-vllm.error.log
Restart=always
RestartSec=3

[Install]
WantedBy=multi-user.target
EOF

        # 서비스 파일 설치
        echo -e "${YELLOW}vLLM systemd 서비스를 설치합니다...${NC}"
        sudo cp airun-vllm.service /etc/systemd/system/
        sudo chmod 644 /etc/systemd/system/airun-vllm.service
        sudo chown root:root /etc/systemd/system/airun-vllm.service

        # 임시 파일 삭제
        rm -f airun-vllm.service

        sudo systemctl daemon-reload
        sudo systemctl enable airun-vllm
        sudo systemctl start airun-vllm

        # 서비스 상태 확인
        if systemctl is-active --quiet airun-vllm; then
            echo -e "${GREEN}vLLM 서비스가 성공적으로 시작되었습니다.${NC}"
        else
            echo -e "${RED}vLLM 서비스 시작에 실패했습니다.${NC}"
            echo -e "${YELLOW}로그 확인: sudo journalctl -u airun-vllm -f${NC}"
        fi

        echo -e "${GREEN}vLLM 서비스 설치가 완료되었습니다.${NC}"
        echo -e "${YELLOW}서비스 제어 명령어:${NC}"
        echo -e "  시작: ${GREEN}sudo systemctl start airun-vllm${NC}"
        echo -e "  중지: ${GREEN}sudo systemctl stop airun-vllm${NC}"
        echo -e "  재시작: ${GREEN}sudo systemctl restart airun-vllm${NC}"
        echo -e "  상태: ${GREEN}sudo systemctl status airun-vllm${NC}"
    else
        # init.d 스크립트 생성
        echo -e "${YELLOW}vLLM init.d 스크립트를 생성합니다...${NC}"
        cat >airun-vllm <<EOF
#!/bin/sh
### BEGIN INIT INFO
# Provides:          airun-vllm
# Required-Start:    \$remote_fs \$syslog \$network
# Required-Stop:     \$remote_fs \$syslog \$network
# Default-Start:     2 3 4 5
# Default-Stop:      0 1 6
# Short-Description: AI.RUN vLLM Server
# Description:       AI.RUN vLLM OpenAI-compatible Server service
### END INIT INFO

USER=$CURRENT_USER
HOME=$HOME
DAEMON="$HOME/.airun_venv/bin/python3"
DAEMON_ARGS="-m vllm.entrypoints.openai.api_server --model naver-hyperclovax/HyperCLOVAX-SEED-Text-Instruct-0.5B --download-dir $HOME/.airun/models --port 11400 --host 0.0.0.0"
PIDFILE=/var/run/airun-vllm.pid
LOGFILE=/var/log/airun/airun-vllm.log
ERRFILE=/var/log/airun/airun-vllm.error.log
WORKDIR="$CURRENT_DIR/services/vllm"

export PYTHONUNBUFFERED=1
export HOME=$HOME
export AIRUN_PATH=$CURRENT_DIR
export PYTHONPATH=$CURRENT_DIR

start() {
    echo "Starting airun-vllm..."
    if [ -f \$PIDFILE ]; then
        echo "PID file exists. Service may be running already"
        return 1
    fi
    
    cd \$WORKDIR
    start-stop-daemon --start --background \\
        --make-pidfile --pidfile \$PIDFILE \\
        --chuid \$USER \\
        --chdir \$WORKDIR \\
        --exec \$DAEMON -- \$DAEMON_ARGS >> \$LOGFILE 2>> \$ERRFILE
}

stop() {
    echo "Stopping airun-vllm..."
    start-stop-daemon --stop --pidfile \$PIDFILE --retry 10
    rm -f \$PIDFILE
}

case "\$1" in
    start)
        start
        ;;
    stop)
        stop
        ;;
    restart)
        stop
        start
        ;;
    status)
        if [ -f \$PIDFILE ]; then
            echo "Service is running"
        else
            echo "Service is stopped"
        fi
        ;;
    *)
        echo "Usage: \$0 {start|stop|restart|status}"
        exit 1
        ;;
esac

exit 0
EOF

        # init.d 스크립트 설치
        echo -e "${YELLOW}vLLM init.d 스크립트를 설치합니다...${NC}"
        sudo cp airun-vllm /etc/init.d/
        sudo chmod 755 /etc/init.d/airun-vllm
        sudo chown root:root /etc/init.d/airun-vllm

        # 임시 파일 삭제
        rm -f airun-vllm

        if command -v update-rc.d &>/dev/null; then
            sudo update-rc.d airun-vllm defaults
        fi

        # 서비스 시작
        sudo service airun-vllm start

        echo -e "${GREEN}vLLM 서비스 설치가 완료되었습니다.${NC}"
        echo -e "${YELLOW}서비스 제어 명령어:${NC}"
        echo -e "  시작: ${GREEN}sudo service airun-vllm start${NC}"
        echo -e "  중지: ${GREEN}sudo service airun-vllm stop${NC}"
        echo -e "  재시작: ${GREEN}sudo service airun-vllm restart${NC}"
        echo -e "  상태: ${GREEN}sudo service airun-vllm status${NC}"
    fi
}

# 메인 설치 함수에 트랩 추가
main() {
    # 실패 시 롤백을 위한 트랩 설정
    trap 'rollback_installation' ERR

    setup_logging

    # Redis 설정
    if ! setup_redis; then
        echo -e "${RED}Redis 설정 실패${NC}"
        return 1
    fi

    # PostgreSQL 기반 RAG 시스템 사용으로 ChromaDB 디렉토리 설정 불필요

    # 설치 검증
    if ! verify_installation; then
        echo -e "${RED}설치 검증 실패${NC}"
        return 1
    fi

    # vLLM 서비스 설치
    # install_vllm_service

    echo -e "${GREEN}설치가 성공적으로 완료되었습니다.${NC}"
}

# MCP 에이전트 도구 빌드
# echo -e "${GREEN}Building MCP agent tools...${NC}"

# 다른 도구 패키지 빌드가 필요한 경우 여기에 추가...

# 원래 나머지 설치 과정 진행
echo -e "${GREEN}Installation completed!${NC}"

echo -e "\n${GREEN}🌐 AIRUN 서비스 포트 구성:${NC}"
echo -e "${YELLOW}  메인 API (5500포트):      http://localhost:5500${NC}"
echo -e "${YELLOW}  문서검색 RAG (5600포트):   http://localhost:5600${NC}"
echo -e "${YELLOW}  웹검색 (5610포트):        http://localhost:5610${NC}"
echo -e "${YELLOW}  보고서 생성 (5620포트):     http://localhost:5620${NC}"
echo -e "\n${GREEN}🔧 서비스 관리:${NC}"
echo -e "${YELLOW}  전체 상태 확인: sudo systemctl status airun-*${NC}"
echo -e "${YELLOW}  전체 재시작:   sudo systemctl restart airun-*${NC}"
echo -e "${YELLOW}  로그 확인:     sudo journalctl -u airun-* -f${NC}"
