import { requireAuth } from './auth.js';

// reportPage.js - 보고서 생성 페이지
export async function reportPage(chatMessages, translations) {
    // 사용자 인증 체크
    const hasAccess = await requireAuth(reqAPI);
    if (!hasAccess) {
        return;
    }

    // 필요한 스크립트 및 스타일 로드
    if (!document.querySelector('link[href="./pages/report.css"]')) {
        const link = document.createElement('link');
        link.rel = 'stylesheet';
        link.href = './pages/report.css';
        document.head.appendChild(link);
    }

    // 입력 컨테이너 숨기기
    const inputContainer = document.querySelector('.input-container');
    if (inputContainer) {
        inputContainer.style.display = 'none';
    }

    // chatMessages 컨테이너 비우기
    if (!chatMessages) {
        console.error('chatMessages 컨테이너를 찾을 수 없습니다.');
        return;
    }
    
    // 보고서 페이지 HTML 생성
    chatMessages.innerHTML = `
        <div class="report-page-container">
            <h1>${translations.report?.title || '보고서 생성'}</h1>
            <p class="report-page-description">${translations.report?.description || 'AI를 활용하여 전문적인 보고서를 생성합니다. 템플릿을 선택하고 주제를 입력하세요.'}</p>
            
            <div class="report-section template-format-section">
                <div class="template-format-container">
                    <div class="template-select-container">
                        <h2>${translations.report?.template || '템플릿 선택'}</h2>
                        <div class="template-select-wrapper">
                            <select id="templateSelect" class="template-select">
                                <option value="">기본 템플릿</option>
                                <option value="loading" disabled>템플릿 로딩 중...</option>
                            </select>
                            <p class="template-description">사용자 입력에 따라 자동으로 구성되는 기본 템플릿입니다.</p>
                        </div>
                    </div>
                    
                    <div class="format-select-container">
                        <h2>${translations.report?.format || '출력 형식'}</h2>
                        <div class="format-options-wrapper">
                            <div class="format-option selected" data-format="pdf">
                                <h3>PDF</h3>
                                <p>Adobe PDF 형식</p>
                            </div>
                            <div class="format-option" data-format="hwpx">
                                <h3>HWPX</h3>
                                <p>한글 문서 형식</p>
                            </div>
                            <div class="format-option" data-format="docx">
                                <h3>DOCX</h3>
                                <p>Microsoft Word 형식</p>
                            </div>
                            <div class="format-option" data-format="pptx">
                                <h3>PPTX</h3>
                                <p>Microsoft PowerPoint 형식</p>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="report-section prompt-section">
                <h2>${translations.report?.prompt || '문서작성 요청사항'}</h2>
                <p>${translations.report?.promptDescription || '생성하고 싶은 문서의 주제나 내용을 자세히 입력하세요.'}</p>
                <div class="prompt-container">
                    <textarea class="prompt-textarea" id="reportPrompt" placeholder="${translations.report?.promptPlaceholder || '예: AI 스타트업의 시장 분석 보고서 작성'}" rows="15"></textarea>
                    <div class="example-container">
                        <h4>${translations.report?.examples || '간단한 요청사항 예시'}</h4>
                        <div class="example-items">
                            <div class="example-item">2024년 글로벌 클라우드 시장 동향 분석</div>
                            <div class="example-item">AI 기반 개인화 학습 플랫폼 개발 사업계획서</div>
                            <div class="example-item">스마트 도시 사업 계획서 작성</div>
                            <div class="example-item">2025년 1분기 사업 계획서 작성</div>
                        </div>
                        <div class="detailed-examples">
                            <h4>${translations.report?.detailedExamples || '상세 요청사항 예시'}</h4>
                            <div class="detailed-example-tabs">
                                <div class="detailed-example-tab active" data-tab="marketing">마케팅</div>
                                <div class="detailed-example-tab" data-tab="proposal">제안서</div>
                                <div class="detailed-example-tab" data-tab="khnp">KHNP</div>
                            </div>
                            <div class="detailed-example-content active" id="marketing-example">
                                <pre>MZ세대를 타겟으로 한 친환경 패션 브랜드의 디지털 마케팅 전략 보고서를 작성해주세요.

브랜드 현황:
- 브랜드명: EcoStyle
- 주요 제품: 재활용 소재를 활용한 캐주얼 의류 및 액세서리
- 타겟: 20-30대 환경 의식이 높은 MZ세대
- 현재 매출: 월 평균 5천만원 (온라인 채널 80%)

시장 현황:
- 친환경 패션 시장 연간 20% 성장
- MZ세대의 환경 친화적 소비 트렌드 강화
- 소셜미디어를 통한 브랜드 가치 전파 중요성 증가

마케팅 목표:
- 6개월 내 매출 50% 증가
- 인스타그램 팔로워 10만명 달성
- 브랜드 인지도 30% 향상

현재 마케팅 예산:
- 월 1천만원 (디지털 마케팅 집중)</pre>
                                <button class="use-example-btn" data-example="marketing">이 예시 사용하기</button>
                            </div>
                            <div class="detailed-example-content" id="proposal-example">
                                <pre>도시형 스마트팜 실증단지 조성 기획안을 작성해주세요.

사업 개요:
- 사업명: 서울 도시형 스마트팜 실증단지 조성사업
- 위치: 서울시 마곡지구 내 유휴부지
- 면적: 약 10,000㎡
- 총사업비: 300억원
- 사업기간: 2025-2027 (3년)

주요 시설:
- 수직형 스마트팜 실증동
- R&D 센터
- 창업 인큐베이팅 센터
- 체험·교육 시설
- 농산물 직매장

기대효과:
- 도시농업 기술혁신 거점 조성
- 청년 창업 활성화
- 식량자급률 향상
- 도시 환경 개선

추진 방향:
- 민관 협력 모델 구축
- 첨단 농업기술 실증
- 창업 생태계 조성
- 시민 참여형 프로그램 운영</pre>
                                <button class="use-example-btn" data-example="proposal">이 예시 사용하기</button>
                            </div>
                            <div class="detailed-example-content" id="khnp-example">
                                <pre>1. 사업명:
스마트 공장 모니터링 시스템 개발 사업

2. 사업기간:
2024.06.01 ~ 2024.12.31 (7개월)

3. 사업배경:
- 제조 현장의 실시간 모니터링 필요성 증대
- 생산 효율성 향상을 위한 데이터 기반 의사결정 요구
- 스마트 공장 구축을 위한 기반 시스템 필요

4. 사업목적:
- 실시간 생산 현황 모니터링 체계 구축
- 생산 설비 이상 징후 조기 감지
- 생산성 향상을 위한 데이터 분석 기반 마련

5. 사업범위:
- 생산 설비 데이터 수집 시스템 구축
- 실시간 모니터링 대시보드 개발
- 이상 징후 감지 알고리즘 개발
- 데이터 분석 및 리포팅 시스템 구축</pre>
                                <button class="use-example-btn" data-example="khnp">이 예시 사용하기</button>
                            </div>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="generate-button-container">
                <button class="generate-report-button" id="generateReportBtn">
                    ${translations.report?.generate || '보고서 생성'}
                </button>
            </div>
            
            <div class="report-result-section" id="reportResultSection"></div>
        </div>
    `;

    // 이벤트 리스너 등록
    document.querySelectorAll('.format-option').forEach(option => {
        option.addEventListener('click', () => {
            document.querySelectorAll('.format-option').forEach(opt => {
                opt.classList.remove('selected');
            });
            option.classList.add('selected');
        });
    });

    document.querySelectorAll('.example-item').forEach(example => {
        example.addEventListener('click', () => {
            document.getElementById('reportPrompt').value = example.textContent;
        });
    });

    // 상세 예시 탭 이벤트 리스너
    document.querySelectorAll('.detailed-example-tab').forEach(tab => {
        tab.addEventListener('click', () => {
            // 모든 탭 비활성화
            document.querySelectorAll('.detailed-example-tab').forEach(t => {
                t.classList.remove('active');
            });
            // 클릭한 탭 활성화
            tab.classList.add('active');
            
            // 모든 콘텐츠 숨기기
            document.querySelectorAll('.detailed-example-content').forEach(content => {
                content.classList.remove('active');
            });
            // 선택한 탭에 해당하는 콘텐츠 표시
            const tabId = tab.dataset.tab;
            document.getElementById(`${tabId}-example`).classList.add('active');
        });
    });

    // 예시 사용하기 버튼 이벤트 리스너
    document.querySelectorAll('.use-example-btn').forEach(button => {
        button.addEventListener('click', () => {
            const exampleType = button.dataset.example;
            const exampleContent = document.querySelector(`#${exampleType}-example pre`).textContent;
            document.getElementById('reportPrompt').value = exampleContent;
        });
    });

    document.getElementById('generateReportBtn').addEventListener('click', () => generateReport(translations));

    // 진행 상태 이벤트 리스너 등록
    window.addEventListener('report-progress', handleProgressUpdate);

    // 템플릿 데이터 로드
    await loadTemplateData();
}

// 페이지 언로드 시 이벤트 리스너 제거
function cleanupEventListeners() {
    window.removeEventListener('report-progress', handleProgressUpdate);
}

// 진행 상태 업데이트 처리 함수
function handleProgressUpdate(event) {
    const { data } = event.detail;
    
    // 결과 섹션 가져오기
    const resultSection = document.getElementById('reportResultSection');
    if (!resultSection) return;
    
    // 로딩 컨테이너 가져오기 또는 생성
    let loadingContainer = document.querySelector('.report-loading');
    if (!loadingContainer) {
        // 이미 결과가 표시된 경우 무시
        if (resultSection.querySelector('.report-success') || resultSection.querySelector('.report-error')) {
            return;
        }
        
        // 로딩 컨테이너 생성
        loadingContainer = document.createElement('div');
        loadingContainer.className = 'report-loading';
        resultSection.innerHTML = '';
        resultSection.appendChild(loadingContainer);
    }

    // 마지막 메시지 저장을 위한 변수
    const lastMessageKey = 'lastProgressMessage';
    const lastMessage = loadingContainer.getAttribute(lastMessageKey);

    // 동일한 메시지가 연속으로 들어오는 경우 무시
    if (lastMessage === data.message) {
        return;
    }

    // 새 메시지 저장
    loadingContainer.setAttribute(lastMessageKey, data.message);
    
    // 진행 상태 메시지 업데이트
    const messageElement = loadingContainer.querySelector('.progress-message');
    if (messageElement) {
        messageElement.textContent = data.message || '보고서 생성 중...';
    } else {
        const newMessageElement = document.createElement('p');
        newMessageElement.className = 'progress-message';
        newMessageElement.textContent = data.message || '보고서 생성 중...';
        newMessageElement.style.textAlign = 'left';
        
        // 스피너 요소 생성
        const spinner = document.createElement('div');
        spinner.className = 'loading-spinner';
        
        // 진행 상태 표시줄 컨테이너
        const progressBarContainer = document.createElement('div');
        progressBarContainer.className = 'progress-bar-container';
        
        // 진행 상태 표시줄
        const progressBar = document.createElement('div');
        progressBar.className = 'progress-bar';
        progressBar.style.width = `${data.percentage || 0}%`;
        
        // 진행률 텍스트
        const progressText = document.createElement('div');
        progressText.className = 'progress-text';
        progressText.textContent = `${data.percentage || 0}%`;
        
        // 로그 컨테이너
        const logContainer = document.createElement('div');
        logContainer.className = 'progress-log-container';
        logContainer.style.textAlign = 'left';
        
        // 요소들 추가
        progressBarContainer.appendChild(progressBar);
        loadingContainer.appendChild(spinner);
        loadingContainer.appendChild(newMessageElement);
        loadingContainer.appendChild(progressBarContainer);
        loadingContainer.appendChild(progressText);
        loadingContainer.appendChild(logContainer);
    }
    
    // 진행률 업데이트
    if (data.percentage !== null && data.percentage !== undefined) {
        const progressBar = loadingContainer.querySelector('.progress-bar');
        if (progressBar) {
            progressBar.style.width = `${data.percentage}%`;
        }
        
        const progressText = loadingContainer.querySelector('.progress-text');
        if (progressText) {
            progressText.textContent = `${Math.round(data.percentage)}%`;
        }
    }
    
    // 로그 메시지 추가
    if (data.message) {
        const logContainer = loadingContainer.querySelector('.progress-log-container');
        if (logContainer) {
            // 마지막 로그 메시지 확인
            const lastLogEntry = logContainer.lastElementChild;
            if (lastLogEntry && lastLogEntry.textContent.includes(data.message)) {
                return; // 동일한 메시지가 이미 있으면 추가하지 않음
            }

            const logEntry = document.createElement('div');
            logEntry.className = 'log-entry';
            
            const timestamp = new Date().toLocaleTimeString();
            logEntry.innerHTML = `<span class="log-time">[${timestamp}]</span> ${data.message}`;
            
            // 최대 로그 항목 수 제한 (최신 10개만 유지)
            if (logContainer.children.length >= 10) {
                logContainer.removeChild(logContainer.firstChild);
            }
            
            logContainer.appendChild(logEntry);
            logContainer.scrollTop = logContainer.scrollHeight;
        }
    }
}

// 템플릿 데이터 로드
async function loadTemplateData() {
    try {
        const response = await window.reqAPI('getTemplateList', {});
        if (response && response.success) {
            updateTemplateList(response.templates);
        } else {
            showError('템플릿 목록을 불러오는데 실패했습니다.');
        }
    } catch (error) {
        console.error('템플릿 로드 오류:', error);
        showError('템플릿 목록을 불러오는데 실패했습니다.');
    }
}

// 템플릿 목록 업데이트
function updateTemplateList(templates) {
    const templateSelect = document.getElementById('templateSelect');
    if (!templateSelect) return;

    // 로딩 옵션 제거
    templateSelect.querySelector('option[value="loading"]')?.remove();

    // 템플릿 목록 추가
    if (templates && templates.length > 0) {
        templates.forEach(template => {
            const option = document.createElement('option');
            option.value = template.id;
            option.textContent = template.name;
            option.dataset.description = template.description || '템플릿 설명이 없습니다.';
            templateSelect.appendChild(option);
        });
    }

    // 템플릿 선택 이벤트 리스너 추가
    templateSelect.addEventListener('change', () => {
        const selectedOption = templateSelect.options[templateSelect.selectedIndex];
        const description = selectedOption.dataset.description || '템플릿 설명이 없습니다.';
        document.querySelector('.template-description').textContent = description;
    });
}

// 템플릿 선택 처리
function selectTemplate(selectedOption) {
    // 셀렉트박스로 변경되어 이 함수는 더 이상 사용하지 않음
}

// 보고서 생성 함수
async function generateReport(translations) {
    // 버튼 상태 관리
    const generateButton = document.getElementById('generateReportBtn');
    if (!generateButton || generateButton.disabled) {
        return; // 이미 실행 중이면 중복 실행 방지
    }

    // 버튼 비활성화 및 텍스트 변경
    generateButton.disabled = true;
    generateButton.textContent = translations.report?.generating || '생성 중...';
    generateButton.classList.add('generating');

    try {
        // 입력 검증
        const promptText = document.getElementById('reportPrompt').value.trim();
        if (!promptText) {
            showError(translations.report?.promptRequired || '보고서 주제를 입력해주세요.');
            resetButton();
            return;
        }
        
        // 선택된 템플릿과 형식 가져오기
        const templateSelect = document.getElementById('templateSelect');
        const selectedTemplate = templateSelect ? templateSelect.value : '';
        const selectedFormat = document.querySelector('.format-option.selected')?.dataset.format || 'pdf';
        
        // PDF 형식인 경우 차트 렌더링을 위한 대기 시간 추가
        if (selectedFormat === 'pdf') {
            await new Promise(resolve => setTimeout(resolve, 2000));
        }
        
        // 결과 섹션 초기화 및 로딩 표시
        const resultSection = document.getElementById('reportResultSection');
        resultSection.innerHTML = '';
        
        const loadingElement = document.createElement('div');
        loadingElement.className = 'report-loading';
        
        // 스피너 요소 생성
        const spinner = document.createElement('div');
        spinner.className = 'loading-spinner';
        
        // 진행 상태 메시지
        const messageElement = document.createElement('p');
        messageElement.className = 'progress-message';
        messageElement.textContent = translations.report?.generating || '보고서 생성 중...';
        
        // 진행 상태 표시줄 컨테이너
        const progressBarContainer = document.createElement('div');
        progressBarContainer.className = 'progress-bar-container';
        
        // 진행 상태 표시줄
        const progressBar = document.createElement('div');
        progressBar.className = 'progress-bar';
        progressBar.style.width = '0%';
        
        // 진행률 텍스트
        const progressText = document.createElement('div');
        progressText.className = 'progress-text';
        progressText.textContent = '0%';
        
        // 설명 텍스트
        const descriptionElement = document.createElement('p');
        descriptionElement.className = 'loading-description';
        descriptionElement.textContent = translations.report?.generatingDescription || '이 작업은 몇 분 정도 소요될 수 있습니다.';
        
        // 로그 컨테이너
        const logContainer = document.createElement('div');
        logContainer.className = 'progress-log-container';
        
        // 요소들 추가
        progressBarContainer.appendChild(progressBar);
        loadingElement.appendChild(spinner);
        loadingElement.appendChild(messageElement);
        loadingElement.appendChild(progressBarContainer);
        loadingElement.appendChild(progressText);
        loadingElement.appendChild(descriptionElement);
        loadingElement.appendChild(logContainer);
        
        resultSection.appendChild(loadingElement);
        
        // 고유 작업 ID 생성
        const taskId = Date.now().toString();
        
        // API 호출
        const result = await window.reqAPI('generateReport', {
            prompt: promptText,
            template: selectedTemplate,
            format: selectedFormat,
            taskId
        }, true);
        
        // 로딩 제거
        resultSection.innerHTML = '';
        
        if (result.success) {
            // 성공 결과 표시
            const successElement = document.createElement('div');
            successElement.className = 'report-success';
            
            const successTitle = document.createElement('h3');
            successTitle.textContent = translations.report?.success || '보고서 생성 완료';
            successElement.appendChild(successTitle);
            
            const filePath = document.createElement('p');
            filePath.className = 'file-path';
            filePath.textContent = result.file_path;
            successElement.appendChild(filePath);
            
            const buttonContainer = document.createElement('div');
            buttonContainer.className = 'result-button-container';
            
            const openButton = document.createElement('button');
            openButton.className = 'open-file-button';
            openButton.textContent = translations.report?.open || '파일 열기';
            openButton.addEventListener('click', () => {
                window.reqAPI('openFile', { path: result.file_path });
            });
            buttonContainer.appendChild(openButton);
            
            successElement.appendChild(buttonContainer);
            resultSection.appendChild(successElement);
            
            // 이벤트 리스너 제거
            cleanupEventListeners();
        } else {
            // 오류 표시
            showError(result.error || (translations.report?.error || '보고서 생성 중 오류가 발생했습니다.'));
            
            // 이벤트 리스너 제거
            cleanupEventListeners();
        }
    } catch (error) {
        // 오류 표시
        resultSection.innerHTML = '';
        // error.message가 객체인 경우를 처리
        let errorMessage = error?.message;
        if (typeof errorMessage === 'object') {
            errorMessage = JSON.stringify(errorMessage);
        }
        showError(errorMessage || (translations.report?.error || '보고서 생성 중 오류가 발생했습니다.'));
    } finally {
        // 버튼 상태 복원
        resetButton();
    }

    // 버튼 상태 복원 함수
    function resetButton() {
        if (generateButton) {
            generateButton.disabled = false;
            generateButton.textContent = translations.report?.generate || '보고서 생성';
            generateButton.classList.remove('generating');
        }
    }
}

// 오류 표시 함수
function showError(message) {
    const resultSection = document.getElementById('reportResultSection');
    if (resultSection) {
        resultSection.innerHTML = '';
        
        const errorElement = document.createElement('div');
        errorElement.className = 'report-error';
        errorElement.textContent = message;
        resultSection.appendChild(errorElement);
    } else {
        alert(message);
    }
}

// CSS 스타일 추가를 위한 스타일 태그 생성
const style = document.createElement('style');
style.textContent = `
    .generate-report-button {
        transition: all 0.3s ease;
    }
    
    .generate-report-button:disabled {
        opacity: 0.7;
        cursor: not-allowed;
    }
    
    .generate-report-button.generating {
        background-color: #666;
    }
`;
document.head.appendChild(style); 