import { shell_exec } from '../../codeExecution.js';
import { promisify } from 'util';
import { exec as execCallback, spawn } from 'child_process';
import { getPythonPipPath } from '../../envLoaders.js';
import { makeVEnvCmd } from '../../codeModifiers.js';
import { getVarVal } from '../../configuration.js';
import path from 'path';
import { fileURLToPath } from 'url';
import fs from 'fs';
import os from 'os';
import { glob } from 'glob';
import chalk from 'chalk';
import { createHash } from 'crypto';

const exec = promisify(execCallback);
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// 프로젝트 루트 디렉토리 찾기 (plugins/websearch 상위 디렉토리의 상위 디렉토리)
const projectRoot = path.resolve(__dirname, '../..');

const isWindows = process.platform === 'win32';

class SearchCache {
    constructor() {
        this.cacheDir = path.join(os.homedir(), '.airun', 'cache', 'web_search');
        this.maxSize = 1000;
        this.ttl = 72 * 60 * 60 * 1000; // 3일
        this.cache = new Map();
        
        // 캐시 디렉토리 생성
        if (!fs.existsSync(this.cacheDir)) {
            fs.mkdirSync(this.cacheDir, { recursive: true });
        }
        
        // NO_CACHE 옵션이 활성화되지 않은 경우에만 캐시 로드
        if (!global.NO_CACHE) {
            this.loadCache();
        }
    }

    generateCacheFilePath(key) {
        // key 값 검증
        if (!key) {
            console.error('Invalid key provided to generateCacheFilePath:', key);
            return path.join(this.cacheDir, 'invalid_cache.json');
        }
        
        try {
            // 키를 해시화하여 파일명으로 사용
            const hash = createHash('md5').update(String(key)).digest('hex');
            return path.join(this.cacheDir, `${hash}.json`);
        } catch (error) {
            console.error('Error generating cache file path:', error);
            return path.join(this.cacheDir, 'error_cache.json');
        }
    }

    loadCache() {
        try {
            if (fs.existsSync(this.cacheDir)) {
                const files = fs.readdirSync(this.cacheDir);
                files.forEach(file => {
                    if (file.endsWith('.json')) {
                        const filePath = path.join(this.cacheDir, file);
                        try {
                            const data = JSON.parse(fs.readFileSync(filePath, 'utf8'));
                            const key = data.key;
                            this.cache.set(key, {
                                data: data.data,
                                timestamp: data.timestamp
                            });
                        } catch (err) {
                            // 개별 파일 로드 실패는 무시하고 계속 진행
                            console.error(`Failed to load cache file ${file}:`, err);
                        }
                    }
                });
                // if (!global.NO_CACHE) {
                //     console.log(`Cache loaded: ${this.cache.size} items`);
                // }
            }
        } catch (error) {
            console.error('Cache load failed:', error);
        }
    }

    saveCache() {
        try {
            // 캐시 디렉토리가 없으면 생성
            if (!fs.existsSync(this.cacheDir)) {
                fs.mkdirSync(this.cacheDir, { recursive: true });
            }

            // 기존 캐시 파일 삭제
            const files = fs.readdirSync(this.cacheDir);
            files.forEach(file => {
                if (file.endsWith('.json')) {
                    fs.unlinkSync(path.join(this.cacheDir, file));
                }
            });

            // 새로운 캐시 파일들 저장
            for (const [key, value] of this.cache.entries()) {
                const filePath = this.generateCacheFilePath(key);
                const cacheData = {
                    key: key,
                    data: value.data,
                    timestamp: value.timestamp
                };
                fs.writeFileSync(filePath, JSON.stringify(cacheData), 'utf8');
            }
        } catch (error) {
            console.error('Cache save failed:', error);
        }
    }

    generateKey(query) {
        // 쿼리가 undefined나 null인 경우 처리
        if (!query) {
            console.error('Invalid query provided to generateKey:', query);
            return 'invalid_query';
        }
        
        // 공백과 특수문자 정규화
        return query.toLowerCase()
            .trim()
            .replace(/\s+/g, ' ')
            .replace(/[?!.,]/g, '')
            .replace(/[은는이가을를의]/g, '');  // 조사 제거
    }

    get(query) {
        // 쿼리가 undefined나 null인 경우 처리
        if (!query) {
            console.error('Invalid query provided to get:', query);
            return null;
        }
        
        const key = this.generateKey(query);
        const item = this.cache.get(key);
        
        if (!item) return null;
        
        // TTL 체크
        if (Date.now() - item.timestamp > this.ttl) {
            this.cache.delete(key);
            this.saveCache();
            return null;
        }
        
        return item.data;
    }

    set(query, results) {
        // 쿼리나 결과가 undefined나 null인 경우 처리
        if (!query || !results) {
            console.error('Invalid query or results provided to set:', { query, results });
            return;
        }
        
        const key = this.generateKey(query);
        
        // 캐시 크기 제한 체크
        if (this.cache.size >= this.maxSize) {
            // 가장 오래된 항목 제거
            const oldestKey = Array.from(this.cache.entries())
                .sort(([,a], [,b]) => a.timestamp - b.timestamp)[0][0];
            this.cache.delete(oldestKey);
        }
        
        this.cache.set(key, {
            data: results,
            timestamp: Date.now()
        });

        // 캐시를 파일에 저장
        this.saveCache();
    }

    clear() {
        this.cache.clear();
        this.saveCache();
    }
}

export class WebSearchService {
    static cache = new SearchCache();
    static isEnabled = false;
    static newSearchServiceUrl = 'http://localhost:5610';

    static async checkIfEnabled(forceEnabled) {
        // forceEnabled가 명시적으로 전달된 경우 해당 값을 사용
        if (forceEnabled !== undefined) {
            WebSearchService.isEnabled = forceEnabled;
            return forceEnabled;
        }

        try {
            const useWebSearch = await getVarVal('USE_WEB_SEARCH');
            if (!useWebSearch) return false;
            
            const value = typeof useWebSearch === 'string' ? useWebSearch.toLowerCase() : useWebSearch;
            WebSearchService.isEnabled = value === 'true' || value === true || value === 'yes';
            return WebSearchService.isEnabled;
        } catch (error) {
            console.error('Error checking web search status:', error);
            return false;
        }
    }

    static clearCache() {
        console.log('Web search cache is cleared...');
        this.cache.clear();
        console.log('Cache initialization is complete.');
    }

    // 검색 쿼리 최적화를 위한 새로운 메서드들
    static optimizeSearchQuery(query) {
        // 불필요한 조사와 특수문자 제거
        const cleanQuery = query
            .replace(/[은는이가을를의]/g, ' ')
            .replace(/[?!.,]/g, ' ')
            .replace(/\s+/g, ' ')
            .trim();

        // 주요 키워드 추출
        const keywords = cleanQuery.split(' ')
            .filter(word => word.length > 1) // 1글자 단어 제외
            .filter(word => !['정보', '검색', '찾기', '알려', '알고', '싶', '무엇', '어떤', '방법'].includes(word)); // 불필요한 단어 제외

        // 검색 쿼리 생성
        const searchQueries = [];
        
        // 1. 전체 쿼리
        searchQueries.push(query);
        
        // 2. 키워드 조합 (2-3개 키워드)
        if (keywords.length > 1) {
            // 가장 관련성 높은 키워드 조합 우선
            const keywordPairs = [];
            for (let i = 0; i < keywords.length - 1; i++) {
                for (let j = i + 1; j < keywords.length; j++) {
                    keywordPairs.push([keywords[i], keywords[j]]);
                }
            }
            
            // 키워드 쌍의 관련성 점수 계산 및 정렬
            const scoredPairs = keywordPairs.map(pair => {
                const combined = pair.join(' ');
                const score = this.calculateKeywordPairScore(combined);
                return { pair, score };
            }).sort((a, b) => b.score - a.score);
            
            // 상위 3개의 키워드 조합만 사용
            scoredPairs.slice(0, 3).forEach(({ pair }) => {
                searchQueries.push(pair.join(' '));
            });
        }
        
        // 3. 시간 정보가 포함된 경우 "최신" 키워드 추가
        if (query.includes('최신') || query.includes('최근') || query.includes('새로운')) {
            const timeKeywords = ['2024', '최신', '순위'];
            keywords.forEach(keyword => {
                timeKeywords.forEach(timeKeyword => {
                    if (!searchQueries.includes(`${keyword} ${timeKeyword}`)) {
                        searchQueries.push(`${keyword} ${timeKeyword}`);
                    }
                });
            });
        }

        // 4. 도메인 특화 키워드 추가
        const domainKeywords = this.getDomainSpecificKeywords(query);
        if (domainKeywords.length > 0) {
            domainKeywords.forEach(domainKeyword => {
                const baseKeywords = keywords.slice(0, 2); // 기본 키워드 2개만 사용
                searchQueries.push([...baseKeywords, domainKeyword].join(' '));
            });
        }

        // 중복 제거 및 정렬
        return [...new Set(searchQueries)]
            .sort((a, b) => b.length - a.length); // 긴 쿼리를 먼저 검색
    }

    // 키워드 쌍의 관련성 점수 계산
    static calculateKeywordPairScore(combined) {
        let score = 0;
        
        // 1. 길이 기반 점수
        score += combined.length > 8 ? 2 : 1;
        
        // 2. 특정 도메인 키워드 포함 여부
        const domainKeywords = ['순위', '차트', '랭킹', '인기', '화제'];
        domainKeywords.forEach(keyword => {
            if (combined.includes(keyword)) score += 2;
        });
        
        // 3. 숫자 포함 여부 (연도나 순위 관련 가능성)
        if (/\d+/.test(combined)) score += 1;
        
        return score;
    }

    // 도메인 특화 키워드 생성
    static getDomainSpecificKeywords(query) {
        const keywords = [];
        
        // 음악/아이돌 관련
        if (query.includes('음악') || query.includes('노래') || query.includes('아이돌') || 
            query.includes('걸그룹') || query.includes('보이그룹')) {
            keywords.push('음원차트', '멜론차트', 'TOP100', '인기곡');
        }
        
        // 연예/방송 관련
        if (query.includes('방송') || query.includes('예능') || query.includes('드라마')) {
            keywords.push('시청률', '화제성', '순위');
        }
        
        // 게임 관련
        if (query.includes('게임') || query.includes('플레이')) {
            keywords.push('인기게임', '게임순위', '이용자수');
        }

        return keywords;
    }

    static async searchWeb(query, engine = 'auto', maxResults = 5, useRawQuery = true) {
        const startTime = Date.now();
        try {
            await WebSearchService.checkIfEnabled(true);

            console.log('\n🔍 웹 검색 시작...');
            console.log(`- Start time: ${new Date(startTime).toISOString()}`);
            
            // 실제 질문 추출
            let actualQuery = query;
            const questionMatch = query.match(/\[질문\]\s*(.*?)(\s*$|\[|$)/);
            if (questionMatch) {
                actualQuery = questionMatch[1].trim();
            } else {
                actualQuery = query.trim();
            }

            console.log('- Original query:', actualQuery);
            console.log('- Engine:', engine);
            console.log('- Maximum results:', maxResults);

            // 🚀 1단계: 5610 포트의 새로운 웹검색 서비스 확인 및 사용
            const isNewServiceAvailable = await WebSearchService.checkNewWebSearchService();
            
            if (isNewServiceAvailable) {
                try {
                    // 엔진 매핑 (기존 방식 → 새 서비스 방식)
                    let newServiceEngine = engine;
                    if (engine === 'parallel') {
                        newServiceEngine = 'parallel';  // 병렬 검색
                    } else if (engine === 'auto') {
                        newServiceEngine = 'auto';      // 스마트 검색 (구글+네이버)
                    }
                    // google, naver, daum은 그대로 사용

                    const newServiceResults = await WebSearchService.searchWithNewService(actualQuery, newServiceEngine, maxResults);
                    
                    if (newServiceResults && newServiceResults.length > 0) {
                        const totalTime = ((Date.now() - startTime) / 1000).toFixed(3);
                        console.log(`✓ Search completed using new service`);
                        console.log(`- Total execution time: ${totalTime} seconds`);
                        return newServiceResults;
                    } else {
                        console.log('⚠️ New service returned no results, falling back to legacy method');
                    }
                } catch (error) {
                    console.log(`⚠️ New service failed (${error.message}), falling back to legacy method`);
                }
            } else {
                console.log('⚠️ New web search service not available, using legacy method');
            }

            // 🔄 2단계: 새 서비스가 없거나 실패한 경우 기존 방식 사용
            console.log('\n📚 Using legacy web search method...');

            // 검색 쿼리 배열 설정
            let searchQueries = [];
            
            // useRawQuery가 true면 쿼리 최적화 없이 원본 쿼리만 사용
            if (useRawQuery) {
                searchQueries = [actualQuery];
                console.log('- Using raw query only:', actualQuery);
            } else {
                // 검색 쿼리 최적화
                searchQueries = this.optimizeSearchQuery(actualQuery);
                console.log('- Optimized queries:', searchQueries);
            }

            let allResults = [];
            let processedUrls = new Set(); // 중복 URL 방지
            let totalEngineResults = { google: 0, naver: 0, daum: 0 }; // 전체 검색 엔진별 결과 수

            // 각 최적화된 쿼리에 대해 검색 수행
            for (const searchQuery of searchQueries) {
                // 캐시 확인
                if (!global.NO_CACHE && this.cache) {
                    const cachedResults = this.cache.get(searchQuery);
                    if (cachedResults) {
                        console.log(`✓ Found cached results for: ${searchQuery}`);
                        // 중복되지 않은 결과만 추가
                        cachedResults.forEach(result => {
                            if (!processedUrls.has(result.url)) {
                                processedUrls.add(result.url);
                                allResults.push(result);
                                // 캐시된 결과의 검색 엔진 정보 업데이트
                                if (result.engine) {
                                    totalEngineResults[result.engine.toLowerCase()]++;
                                }
                            }
                        });
                        continue;
                    }
                }

                // 검색 수행
                const queryResults = await this.executeSearch(searchQuery, engine, maxResults);
                
                // 중복되지 않은 결과만 추가
                queryResults.forEach(result => {
                    if (!processedUrls.has(result.url)) {
                        processedUrls.add(result.url);
                        allResults.push(result);
                        // 검색 엔진별 결과 수 업데이트
                        if (result.engine) {
                            totalEngineResults[result.engine.toLowerCase()]++;
                        }
                    }
                });

                // 충분한 결과를 얻었거나 useRawQuery가 true인 경우 중단
                if (allResults.length >= maxResults * 2 || useRawQuery) {
                    break;
                }
            }

            // 결과 정렬 및 제한
            allResults = allResults
                .sort((a, b) => {
                    // 제목에 원본 쿼리의 키워드가 더 많이 포함된 결과를 우선
                    const aScore = this.calculateRelevanceScore(a, actualQuery);
                    const bScore = this.calculateRelevanceScore(b, actualQuery);
                    return bScore - aScore;
                })
                .slice(0, maxResults);

            if (allResults.length > 0) {
                const totalTime = ((Date.now() - startTime) / 1000).toFixed(3);
                // console.log(`=== Search Engine Summary ===`);
                Object.entries(totalEngineResults).forEach(([engine, count]) => {
                    if (count > 0) {
                        console.log(`- ${engine.charAt(0).toUpperCase() + engine.slice(1)}: ${count} total results`);
                    }
                });
                console.log(`✓ Found ${allResults.length} unique search results`);
                console.log(`- Total execution time: ${totalTime} seconds`);
                return allResults;
            }

            console.log('⚠️ No search results found');
            console.log(`- Total execution time: ${((Date.now() - startTime) / 1000).toFixed(3)} seconds`);
            return [];

        } catch (error) {
            console.error('✗ Web search error:', error);
            console.log(`- Total execution time: ${((Date.now() - startTime) / 1000).toFixed(3)} seconds`);
            return [];
        }
    }

    // 검색 실행을 위한 새로운 메서드
    static async executeSearch(query, engine, maxResults) {
        const startTime = Date.now();
        const escapedQuery = query
            .replace(/'/g, "\\'")
            .replace(/"/g, '\\"')
            .replace(/\n/g, ' ');

        const python_interpreter = await getPythonPipPath();
        if (!python_interpreter) {
            throw new Error('Python interpreter not found in virtual environment');
        }

        // 프로젝트 루트 디렉토리를 기준으로 경로 설정
        const websearchPluginPath = path.join(projectRoot, 'plugins', 'websearch');
        
        const compiledModulePath = path.join(websearchPluginPath, process.platform === 'win32' ? 'web_search.cp*-win_amd64.pyd' : 'web_search.cpython-*-x86_64-linux-gnu.so');

        // Python 스크립트 생성
        const pythonScript = `
import sys
import os
import json
import importlib.util
import glob
import asyncio
import warnings

# 모든 경고 메시지 무시
warnings.filterwarnings('ignore')

try:
    # 현재 디렉토리를 Python 경로에 추가
    script_dir = "${websearchPluginPath.replace(/\\/g, '/')}"
    if script_dir not in sys.path:
        sys.path.insert(0, script_dir)

    # utils.py가 있는 디렉토리를 sys.path에 추가
    root_dir = os.path.dirname(os.path.dirname(os.path.dirname("${__dirname.replace(/\\/g, '/')}")))
    if root_dir not in sys.path:
        sys.path.append(root_dir)

    # 컴파일된 모듈 찾기
    module_pattern = os.path.join(script_dir, "web_search.cpython-*-x86_64-linux-gnu.so")
    module_files = glob.glob(module_pattern)
    
    if not module_files:
        # 정확한 파일명으로 시도
        exact_file = os.path.join(script_dir, "web_search.cpython-312-x86_64-linux-gnu.so")
        if os.path.exists(exact_file):
            module_files = [exact_file]
            print(f"Using compiled module: {os.path.basename(exact_file)}", file=sys.stderr)
        else:
            # 컴파일된 모듈을 찾지 못한 경우 에러
            raise ImportError(f"No compiled module found matching pattern: {module_pattern}")
    
    module_path = module_files[0]
    module_name = "web_search"
    
    # 모듈 로드
    spec = importlib.util.spec_from_file_location(module_name, module_path)
    if spec is None:
        raise ImportError(f"Failed to create module spec for {module_path}")
    
    module = importlib.util.module_from_spec(spec)
    sys.modules[module_name] = module
    spec.loader.exec_module(module)
    
    # 이벤트 루프 설정
    IS_WINDOWS = os.name == 'nt'
    if IS_WINDOWS:
        loop = asyncio.ProactorEventLoop()
        asyncio.set_event_loop(loop)
        results = loop.run_until_complete(module.parallel_search_windows("${escapedQuery}", ${maxResults}))
    else:
        loop = asyncio.new_event_loop()
        asyncio.set_event_loop(loop)
        results = loop.run_until_complete(module.parallel_search_unix("${escapedQuery}", ${maxResults}))
    
    loop.close()
    
    # 결과 출력 (JSON만 출력)
    print(json.dumps(results, ensure_ascii=False))

except Exception as e:
    import traceback
    error_info = {
        "error": str(e),
        "traceback": traceback.format_exc()
    }
    print(json.dumps(error_info), file=sys.stderr)
    sys.exit(1)
`;

        // 검색 엔진별로 별도의 프로세스 실행
        const runSearch = async () => {
            const pythonStartTime = Date.now();  // pythonStartTime 변수를 여기로 이동
            
            if (process.platform === 'win32') {
                // Windows용 실행 방식
                return new Promise((resolve, reject) => {
                    const childProcess = spawn(python_interpreter, ['-c', pythonScript], {
                        env: {
                            ...process.env,
                            PYTHONPATH: path.dirname(path.dirname(path.dirname(__dirname))),
                            PYTHONUNBUFFERED: '1',
                            PYTHONIOENCODING: 'utf-8',
                            VIRTUAL_ENV: path.join(os.homedir(), '.airun_venv'),
                            PATH: `${path.join(os.homedir(), '.airun_venv', 'bin')}:${process.env.PATH}`
                        },
                        windowsHide: true
                    });

                    let stdoutData = '';
                    let stderrData = '';

                    childProcess.stdout.on('data', (data) => {
                        const text = data.toString();
                        // JSON 데이터만 저장하고 출력하지 않음
                        stdoutData += text;
                    });

                    childProcess.stderr.on('data', (data) => {
                        const text = data.toString();
                        // 검색 결과 수 관련 메시지만 출력
                        if (text.includes('[DEBUG] google Search results:') ||
                            text.includes('[DEBUG] naver Search results:') ||
                            text.includes('[DEBUG] daum Search results:')) {
                            const resultCount = text.match(/results: (\d+)/);
                            if (resultCount) {
                                const engine = text.includes('google') ? 'Google' :
                                             text.includes('naver') ? 'Naver' : 'Daum';
                                console.log(`${engine}: ${resultCount[1]} results`);
                            }
                        }
                        // 중요한 에러만 출력
                        else if (text.includes('Error:') || text.includes('Exception:')) {
                            console.error('[Error]:', text);
                        }
                        stderrData += text;
                    });

                    childProcess.on('error', (error) => {
                        console.error('[Process Error]:', error);
                        reject(error);
                    });

                    childProcess.on('close', (code) => {
                        if (code === 0) {
                            resolve({
                                stdout: stdoutData,
                                stderr: stderrData,
                                pythonStartTime
                            });
                        } else {
                            console.error('Process exit code:', code);
                            console.error('stderr:', stderrData);
                            reject(new Error(`Process exited with code ${code}`));
                        }
                    });
                });
            } else {
                // Linux/Mac용 실행 방식
                return new Promise((resolve, reject) => {
                    const childProcess = spawn(python_interpreter, ['-c', pythonScript], {
                        env: {
                            ...process.env,
                            PYTHONPATH: path.dirname(path.dirname(path.dirname(__dirname))),
                            PYTHONUNBUFFERED: '1',
                            PYTHONIOENCODING: 'utf-8',
                            VIRTUAL_ENV: path.join(os.homedir(), '.airun_venv'),
                            PATH: `${path.join(os.homedir(), '.airun_venv', 'bin')}:${process.env.PATH}`,
                            // 서브프로세스 stderr 출력 방지
                            SUPPRESS_DEBUG_OUTPUT: '1'
                        },
                        windowsHide: true
                    });

                    let stdoutData = '';
                    let stderrData = '';
                    
                    // 중복 메시지 방지를 위한 로깅 상태 추적
                    const loggedEngines = {
                        google: false,
                        naver: false,
                        daum: false
                    };

                    childProcess.stdout.on('data', (data) => {
                        const text = data.toString();
                        // JSON 데이터만 저장하고 출력하지 않음
                        stdoutData += text;
                    });

                    childProcess.on('error', (error) => {
                        console.error('[Process Error]:', error);
                        reject(error);
                    });

                    childProcess.on('close', (code) => {
                        if (code === 0) {
                            resolve({
                                stdout: stdoutData,
                                stderr: stderrData,
                                pythonStartTime
                            });
                        } else {
                            console.error('Process exit code:', code);
                            console.error('stderr:', stderrData);
                            reject(new Error(`Process exited with code ${code}`));
                        }
                    });
                });
            }
        };
        
        const { stdout, stderr, pythonStartTime } = await runSearch();
        
        // 검색 엔진별 결과 수 추적을 위한 객체
        const engineResults = {
            google: 0,
            naver: 0,
            daum: 0
        };

        // stderr에서 검색 엔진별 결과 수 파싱
        if (stderr) {
            const lines = stderr.split('\n');
            lines.forEach(line => {
                if (line.includes('[DEBUG]')) {
                    if (line.includes('google Search results:')) {
                        const match = line.match(/results: (\d+)/);
                        if (match) engineResults.google = parseInt(match[1], 10);
                    } else if (line.includes('naver Search results:')) {
                        const match = line.match(/results: (\d+)/);
                        if (match) engineResults.naver = parseInt(match[1], 10);
                    } else if (line.includes('daum Search results:')) {
                        const match = line.match(/results: (\d+)/);
                        if (match) engineResults.daum = parseInt(match[1], 10);
                    }
                }
            });
        }

        // 검색 엔진별 결과 수 출력
        Object.entries(engineResults).forEach(([engine, count]) => {
            if (count > 0) {
                console.log(`${engine.charAt(0).toUpperCase() + engine.slice(1)}: ${count} results`);
            }
        });
        
        // 결과 처리
        if (stdout && stdout.trim()) {
            try {
                const jsonMatch = stdout.match(/\[.*\]/s);
                if (!jsonMatch) {
                    console.error('No JSON data found in output');
                    return [];
                }
                
                const cleanedOutput = jsonMatch[0].replace(/\\u200b/g, '').replace(/\\'/g, "'").trim();

                if (cleanedOutput.startsWith('[') && cleanedOutput.endsWith(']')) {
                    const results = JSON.parse(cleanedOutput);
                    if (results && !results.error && results.length > 0) {
                        // NO_CACHE 옵션이 false인 경우에만 캐시에 저장
                        if (!global.NO_CACHE) {
                            this.cache?.set(query, results);
                        }
                        
                        const executionTime = ((Date.now() - startTime) / 1000).toFixed(3);
                        // console.log(`✓ Found ${results.length} search results for query: "${query}"`);
                        // console.log(`Execution time: ${executionTime} seconds`);
                        
                        // 엔진별 결과 수 계산
                        const engineCounts = {
                            google: results.filter(r => r.engine === 'google').length,
                            naver: results.filter(r => r.engine === 'naver').length,
                            daum: results.filter(r => r.engine === 'daum').length
                        };
                        
                        // 엔진별 결과가 있는 경우에만 출력
                        // Object.entries(engineCounts).forEach(([engine, count]) => {
                        //     if (count > 0) {
                        //         console.log(`${engine.charAt(0).toUpperCase() + engine.slice(1)}: ${count} total results`);
                        //     }
                        // });
                        
                        return results;
                    } else if (results.error) {
                        console.error('Python error:', results.error);
                        if (results.traceback) {
                            console.error('Error traceback:', results.traceback);
                        }
                    }
                } else {
                    console.error('Invalid JSON format:', cleanedOutput);
                    return [];
                }
            } catch (error) {
                console.error('Error parsing web search results:', error);
                console.log('Raw output:', stdout);
                return [];
            }
        }

        return [];
    }

    // 결과 관련성 점수 계산을 위한 새로운 메서드
    static calculateRelevanceScore(result, originalQuery) {
        let score = 0;
        const keywords = originalQuery.toLowerCase().split(' ').filter(word => word.length > 1);
        
        // 제목에 키워드가 포함된 경우
        if (result.title) {
            const title = result.title.toLowerCase();
            keywords.forEach(keyword => {
                if (title.includes(keyword)) score += 2;
            });
        }
        
        // 설명에 키워드가 포함된 경우
        if (result.description) {
            const description = result.description.toLowerCase();
            keywords.forEach(keyword => {
                if (description.includes(keyword)) score += 1;
            });
        }
        
        return score;
    }

    // 검색 결과를 구조화된 형식으로 출력하는 메서드 추가
    static displaySearchResults(results) {
        if (!results || results.length === 0) return;

        console.log('\n========== Web Search Results ==========');
        results.forEach((result, index) => {
            console.log(`\n[검색 결과 ${index + 1}]`);
            if (result.title) console.log(`제목: ${result.title}`);
            if (result.url) console.log(`출처: ${result.url}`);
            if (result.type) console.log(`유형: ${result.type}`);
            if (result.description) {
                console.log('내용:');
                console.log(result.description);
            }
            console.log('─'.repeat(50));
        });
        console.log('\n✓ 검색이 완료되었습니다.');
    }

    // 5610 포트의 새로운 웹검색 서비스가 구동 중인지 확인
    static async checkNewWebSearchService() {
        try {
            const response = await fetch(`${this.newSearchServiceUrl}/health`, {
                method: 'GET',
                timeout: 2000  // 2초 타임아웃
            });
            
            if (response.ok) {
                const health = await response.json();
                return health.status === 'healthy';
            }
            return false;
        } catch (error) {
            // 연결 실패는 정상적인 상황 (서비스가 꺼져 있음)
            return false;
        }
    }

    // 5610 포트의 새로운 웹검색 서비스를 사용한 검색
    static async searchWithNewService(query, engine = 'auto', maxResults = 5) {
        try {
            const requestData = {
                query: query,
                max_results: maxResults,
                engine: engine,
                use_cache: !global.NO_CACHE
            };

            const response = await fetch(`${this.newSearchServiceUrl}/search`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify(requestData),
                timeout: 30000  // 30초 타임아웃
            });

            if (!response.ok) {
                throw new Error(`HTTP ${response.status}: ${response.statusText}`);
            }

            const result = await response.json();
            
            if (result.success) {
                // 엔진별 결과 수 출력
                if (result.engine_stats) {
                    Object.entries(result.engine_stats).forEach(([engine, count]) => {
                        if (count > 0) {
                            console.log(`- ${engine.charAt(0).toUpperCase() + engine.slice(1)}: ${count} results`);
                        }
                    });
                }

                console.log(`✓ Found ${result.total_results} search results`);
                console.log(`- Execution time: ${result.execution_time.toFixed(3)} seconds`);
                
                if (result.cached) {
                    console.log('- Results from cache');
                }

                return result.results || [];
            } else {
                throw new Error(result.error || 'Search failed');
            }
        } catch (error) {
            console.error('❌ New web search service error:', error.message);
            throw error;
        }
    }
}

async function processSearchResults(rawOutput) {
    try {
        const searchData = JSON.parse(rawOutput);
        
        // 새로운 형식 확인 (메타데이터 포함)
        const results = searchData.results || searchData;
        const metadata = searchData.metadata || {};
        
        // 엔진별 결과 수 계산
        const engineResults = metadata.engine_counts || {
            google: results.filter(r => r.engine === 'google').length,
            naver: results.filter(r => r.engine === 'naver').length,
            daum: results.filter(r => r.engine === 'daum').length
        };

        // 검색 결과 요약 출력
        if (results.length > 0) {
            console.log(`✓ Found ${results.length} search results`);
            // console.log('\n=== Search Engine Summary ===');
            Object.entries(engineResults).forEach(([engine, count]) => {
                if (count > 0) {
                    console.log(`- ${engine.charAt(0).toUpperCase() + engine.slice(1)}: ${count} total results`);
                }
            });

            // 메타데이터 정보 출력 (있는 경우)
            if (metadata.duration) {
                console.log(`\nSearch Duration: ${metadata.duration.toFixed(2)} seconds`);
            }
            if (metadata.keywords_used && metadata.keywords_used.length > 1) {
                console.log('\nSearch Keywords Used:');
                metadata.keywords_used.forEach(keyword => console.log(`- ${keyword}`));
            }
        } else {
            console.log('No search results found');
        }

        return results;
    } catch (error) {
        console.error('Error processing search results:', error);
        return [];
    }
}

async function executeWebSearch(query, maxResults = 5, useRawQueryOnly = false) {
    const startTime = new Date();
    console.log('\n🔍 웹 검색 시작...');
    console.log('- Start time:', startTime.toISOString());
    console.log('- Original query (using raw query only):', query);
    console.log('- Maximum results per query:', maxResults);
    console.log('Using raw query only:', useRawQueryOnly);

    try {
        const pythonScript = path.join(__dirname, 'web_search.py');
        const command = `${pythonPath} "${pythonScript}" "${query}" ${maxResults}`;
        
        return new Promise((resolve, reject) => {
            let stdoutData = '';
            let stderrData = '';
            
            const childProcess = spawn(pythonPath, [pythonScript, query, maxResults.toString()], {
                env: process.env
            });

            childProcess.stdout.on('data', (data) => {
                const text = data.toString();
                stdoutData += text;
            });

            childProcess.stderr.on('data', (data) => {
                const text = data.toString();
                if (text.includes('[DEBUG]')) {
                    console.log(text.trim());
                }
                stderrData += text;
            });

            childProcess.on('error', (error) => {
                console.error('Error executing Python script:', error);
                reject(error);
            });

            childProcess.on('close', async (code) => {
                const endTime = new Date();
                const executionTime = (endTime - startTime) / 1000;

                if (code === 0) {
                    try {
                        // 검색 결과 처리
                        const results = await processSearchResults(stdoutData);
                        
                        console.log(`- Total execution time: ${executionTime.toFixed(3)} seconds`);
                        
                        resolve(results);
                    } catch (error) {
                        console.error('Error processing results:', error);
                        reject(error);
                    }
                } else {
                    console.error(`Process exited with code ${code}`);
                    console.error('stderr:', stderrData);
                    reject(new Error(`Process exited with code ${code}`));
                }
            });
        });
    } catch (error) {
        console.error('Error in web search:', error);
        throw error;
    }
} 