import Queue from 'bull';
import path from 'path';
import fs from 'fs';
import os from 'os';
import { fileURLToPath } from 'url';
import { licenseManager } from '../license.js';
import { getVarVal, setVarVal } from '../configuration.js';
import axios from 'axios';
import chalk from 'chalk';
import Redis from 'ioredis';
import crypto from 'crypto';
import { debugLog, logger } from '../utils/logger.js';

// ES 모듈에서 __dirname 대체
const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// fetch with timeout 유틸리티 함수
const fetchWithTimeout = async (url, options = {}) => {
    const { timeout = 5000, ...fetchOptions } = options;
    
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), timeout);
    
    try {
        const response = await fetch(url, {
            ...fetchOptions,
            signal: controller.signal
        });
        clearTimeout(timeoutId);
        return response;
    } catch (error) {
        clearTimeout(timeoutId);
        throw error;
    }
};

// 설정 파일 로드
const configPath = path.join(__dirname, 'config.json');
let config = { redis: { host: '127.0.0.1', port: 6379 }, queues: {} };

try {
  if (fs.existsSync(configPath)) {
    const configData = fs.readFileSync(configPath, 'utf8');
    config = JSON.parse(configData);
  } else {
    logger.warn('큐 설정 파일을 찾을 수 없습니다. 기본 설정을 사용합니다.');
  }
} catch (error) {
  logger.error('설정 파일 로드 중 오류:', error);
}

// Redis 연결 설정
const redisConfig = config.redis || { host: '127.0.0.1', port: 6379 };

// 큐 설정
const QUEUE_CONFIG = {
    defaultJobOptions: {
        timeout: 5 * 60 * 1000,  // 기본 타임아웃 5분
        attempts: 3,             // 실패 시 최대 3번 재시도
        backoff: {
            type: 'exponential',
            delay: 500           // 재시도 간격 1000 -> 500ms로 단축
        },
        removeOnComplete: 100,   // 완료된 작업 100개까지만 보관
        removeOnFail: 200        // 실패한 작업 200개까지만 보관
    },
    limiter: {
        max: 150,               // 동시 처리 가능한 최대 작업 수 100 -> 150으로 증가
        duration: 3000          // 5초 -> 3초로 단축
    },
    settings: {
        lockDuration: 20000,    // 작업 잠금 시간 30초 -> 20초로 단축
        stalledInterval: 20000, // 작업 정체 확인 간격 30초 -> 20초로 단축
        maxStalledCount: 3,     // 최대 정체 허용 횟수 2 -> 3으로 증가
        guardInterval: 3000,    // 가드 프로세스 실행 간격 5초 -> 3초로 단축
        retryProcessDelay: 2000 // 재시도 지연 시간 5초 -> 2초로 단축
    }
};

// 큐 처리 설정
const CONCURRENCY = {
    chat: process.env.CHAT_CONCURRENCY || 5,
    code: process.env.CODE_CONCURRENCY || 3,
    agent: process.env.AGENT_CONCURRENCY || 3,
    report: process.env.REPORT_CONCURRENCY || 3,
    api: process.env.API_CONCURRENCY || 8
};

// 우선순위 설정
const PRIORITY = {
    HIGH: 1,
    NORMAL: 2,
    LOW: 3
};

// 라이선스 파일 경로
const LICENSE_FILE = path.join(os.homedir(), '.airun', 'license.key');

// Redis 연결 재시도 로직
const connectWithRetry = async (retries = 5, delay = 1000) => {
    for (let i = 0; i < retries; i++) {
        try {
            const redis = new Redis(redisConfig);
            await redis.ping();
            console.log('Redis 연결 성공');
            return redis;
        } catch (error) {
            logger.error(`Redis 연결 실패 (시도 ${i + 1}/${retries}):`, error);
            if (i < retries - 1) {
                await new Promise(resolve => setTimeout(resolve, delay));
            }
        }
    }
    throw new Error('Redis 연결 실패');
};

// Redis 연결 초기화
let redisClient;
try {
    redisClient = await connectWithRetry();
} catch (error) {
    logger.error('Redis 연결 초기화 실패:', error);
    process.exit(1);
}

// 큐 상태 모니터링
const monitorQueue = async (queue) => {
    try {
        const [active, waiting, completed, failed] = await Promise.all([
            queue.getActiveCount(),
            queue.getWaitingCount(),
            queue.getCompletedCount(),
            queue.getFailedCount()
        ]);

        // 실패한 작업이 있는 경우에만 경고 로그 출력
        if (failed > 0) {
            logger.warn(`[${queue.name}] 실패한 작업이 있습니다: ${failed}개`);
        }

    } catch (error) {
        logger.error(`[${queue.name}] 모니터링 오류:`, error);
    }
};

// 큐 생성 함수
const createQueue = (name, options = {}) => {
    const queue = new Queue(name, {
        redis: {
            ...redisConfig,
            retryStrategy: (times) => {
                if (times > 10) {
                    logger.error(`[${name}] Redis 재연결 최대 시도 횟수 초과`);
                    return null;
                }
                const delay = Math.min(times * 100, 3000);
                logger.warn(`[${name}] Redis 재연결 시도 ${times}회, ${delay}ms 후 재시도`);
                return delay;
            },
            reconnectOnError: (err) => {
                const targetError = 'READONLY';
                if (err.message.includes(targetError)) {
                    return true;
                }
                return 1;
            }
        },
        ...QUEUE_CONFIG,
        ...options
    });

    // 큐 이벤트 리스너
    queue.on('error', error => {
        if (error.code === 'ECONNREFUSED') {
            logger.warn(`[${name}] Redis 연결 거부됨. 재연결 시도 중...`);
        } else {
            logger.error(`[${name}] 오류:`, error);
        }
    });

    queue.on('failed', async (job, error) => {
        const errorDetails = {
            jobId: job.id,
            timestamp: new Date().toISOString(),
            error: error.message,
            stack: error.stack,
            data: job.data
        };

        logger.error(`[${name}] 작업 실패 [Job ${job.id}]:`, error.message);
        
        // 실패한 작업 정보를 Redis에 저장
        try {
            await redisClient.lpush(`failed:${name}`, JSON.stringify(errorDetails));
            // 최대 1000개의 실패 기록만 유지
            await redisClient.ltrim(`failed:${name}`, 0, 999);
        } catch (redisError) {
            logger.error(`[${name}] 실패 기록 저장 오류:`, redisError);
        }

        // 실패한 작업이 10개 이상 쌓이면 경고
        const failedCount = await queue.getFailedCount();
        if (failedCount >= 10) {
            logger.warn(`[${name}] 경고: 실패한 작업이 ${failedCount}개 이상 쌓였습니다.`);
        }
    });

    queue.on('completed', (job, result) => {
        console.log(`[${name}] 작업 완료 [Job ${job.id}]`);
    });

    queue.on('stalled', (jobId) => {
        logger.warn(`[${name}] 작업 정체 감지 [Job ${jobId}]`);
    });

    // Redis 연결 이벤트 처리
    queue.on('waiting', () => {
        // Redis가 재연결되면 대기 중인 작업 처리 재개
        debugLog(`[${name}] 큐 대기 상태 - Redis 연결 확인 중`);
    });

    // Redis 클라이언트 이벤트 처리 (Bull은 내부적으로 ioredis 사용)
    if (queue.client) {
        queue.client.on('connect', () => {
            logger.info(`[${name}] Redis 연결됨`);
        });

        queue.client.on('reconnecting', (delay) => {
            logger.warn(`[${name}] Redis 재연결 중... (${delay}ms 후)`);
        });

        queue.client.on('error', (error) => {
            if (error.code === 'ECONNREFUSED') {
                logger.warn(`[${name}] Redis 서버에 연결할 수 없습니다. 서비스가 실행 중인지 확인하세요.`);
            } else if (error.code === 'ENOTFOUND') {
                logger.error(`[${name}] Redis 호스트를 찾을 수 없습니다: ${redisConfig.host}`);
            } else {
                logger.error(`[${name}] Redis 클라이언트 오류:`, error.message);
            }
        });

        queue.client.on('end', () => {
            logger.warn(`[${name}] Redis 연결이 종료되었습니다`);
        });
    }

    // 주기적 모니터링 설정
    setInterval(() => monitorQueue(queue), 30000);

    // 실패한 작업 정리 (매일 자정에 실행)
    setInterval(async () => {
        try {
            const failedCount = await queue.getFailedCount();
            if (failedCount > 0) {
                console.log(`[${name}] 실패한 작업 정리 시작 (${failedCount}개)`);
                await queue.clean(24 * 60 * 60 * 1000, 'failed'); // 24시간 이상 된 실패 작업 정리
                console.log(`[${name}] 실패한 작업 정리 완료`);
            }
        } catch (error) {
            logger.error(`[${name}] 실패한 작업 정리 중 오류:`, error);
        }
    }, 24 * 60 * 60 * 1000); // 24시간마다 실행

    return queue;
};

// 큐 생성
export const apiUsageQueue = createQueue('api-usage-queue', {
    ...config.queues.apiUsage.options,
    concurrency: CONCURRENCY.api
});

export const licenseCheckQueue = createQueue('license-check-queue', {
    ...config.queues.licenseCheck.options,
    concurrency: CONCURRENCY.api
});

export const apiRequestQueue = createQueue('api-request-queue', {
    ...config.queues.apiRequest.options,
    concurrency: CONCURRENCY.api
});

export const chatRequestQueue = createQueue('chat-request-queue', {
    ...config.queues.chatRequest.options,
    concurrency: CONCURRENCY.chat
});

export const reportGenerationQueue = createQueue('report-generation-queue', {
    ...config.queues.reportGeneration.options,
    concurrency: CONCURRENCY.report
});

export const healthBatchQueue = createQueue('health-batch-queue', {
    defaultJobOptions: {
        timeout: 10 * 60 * 1000,  // 건강검진 분석은 10분 타임아웃
        attempts: 2,              // 실패 시 최대 2번 재시도
        backoff: {
            type: 'exponential',
            delay: 1000
        },
        removeOnComplete: 50,     // 완료된 작업 50개까지만 보관
        removeOnFail: 100         // 실패한 작업 100개까지만 보관
    },
    concurrency: 8                // 동시에 8개 환자 처리
});

// API 사용량 기록 함수
export async function queueApiUsage(endpoint) {
    try {
        const job = await apiUsageQueue.add({
            endpoint,
            timestamp: Date.now()
        });
        return { success: true, jobId: job.id };
    } catch (error) {
        logger.error('API 사용량 기록 작업 추가 실패:', error);
        return { success: false, error: error.message };
    }
}

// 채팅 요청 큐에 작업 추가
export async function enqueueChatRequest(prompt, options = {}, handlerEndpoint) {
    // 현재 설정값들을 가져옵니다
    const currentSettings = {
        USE_LLM: await getVarVal('USE_LLM'),
        AUTO_PROVIDER_SELECTION: await getVarVal('AUTO_PROVIDER_SELECTION'),
        temperature: options.temperature ?? options.parameters?.temperature ?? 0
    };

    // 이미지 데이터 표준화
    let standardizedImage = options.image;
    let standardizedImages = options.images || [];

    // 단일 이미지가 있고 이미지 배열이 없는 경우, 단일 이미지를 배열로 변환
    if (standardizedImage && !standardizedImages.length) {
        standardizedImages = [standardizedImage];
    }

    // console.log(chalk.cyan('[Queue] enqueueChatRequest 이미지 데이터:'), {
    //     hasImage: !!standardizedImage,
    //     hasImages: !!standardizedImages.length,
    //     imageCount: standardizedImages.length,
    //     imageType: standardizedImage ? (typeof standardizedImage) : 'none',
    //     imageIsPath: standardizedImage ? !standardizedImage.startsWith('data:') : false,
    //     imageLength: standardizedImage ? standardizedImage.length : 0
    // });

    try {
        // chat/sync와 동일한 구조로 데이터 구성
        const jobData = {
            prompt,
            sessionId: options.sessionId || undefined,
            userId: options.userId || undefined,
            username: options.username || undefined, // username 정보 추가
            rag: options.rag === true || false,
            web: options.web === true || false,
            enableSmartResearch: options.enableSmartResearch === true || false, // AI프로 버튼 상태 추가
            hideCode: options.hideCode || false,
            autoExecute: options.autoExecute || false,
            handlerEndpoint,
            settings: currentSettings,
            temperature: options.temperature ?? options.parameters?.temperature ?? 0,
            ragSearchScope: options.ragSearchScope || 'personal', // RAG 검색 범위 추가
            messages_: options.messages || [],
            promptSession: {
                prompt,
                image: standardizedImage || null,
                images: standardizedImages
            },
            useReview: options.USE_REVIEW === true || false, // 코드리뷰 옵션 추가
            // 멘션 기능 관련 필드 추가
            mentionedDocumentContent: options.mentionedDocumentContent || '',
            mentionedDocuments: options.mentionedDocuments || [],
        };

        // console.log(chalk.cyan('[Queue] 작업 데이터 생성:'), {
        //     hasPromptSession: !!jobData.promptSession,
        //     promptSessionImage: !!jobData.promptSession.image,
        //     imageCount: jobData.promptSession.images.length,
        //     imageType: jobData.promptSession.image ? (typeof jobData.promptSession.image) : 'none',
        //     imageIsPath: jobData.promptSession.image ? !jobData.promptSession.image.startsWith('data:') : false
        // });

        const job = await chatRequestQueue.add(jobData);
        
        // logger.debug(chalk.green('[Queue] 작업 추가 완료:'), {
        //     jobId: job.id,
        //     hasPromptSession: !!job.data.promptSession,
        //     hasImage: !!job.data.promptSession.image,
        //     imageCount: job.data.promptSession.images.length
        // });
        
        return {
            success: true,
            jobId: job.id,
            status: 'queued',
            message: '작업이 큐에 추가되었습니다.',
            downloadInfo: []
        };
    } catch (error) {
        logger.error(chalk.red('[Queue] 요청 큐 추가 실패:'), error);
        throw error;
    }
}

// 보고서 생성 요청 큐에 작업 추가
export async function enqueueReportGeneration(options = {}, handlerEndpoint) {
    // filepath가 있으면 파일 내용을 우선 사용하고, prompt가 있으면 추가
    let promptText = '';
    
    if (options.filepath) {
        try {
            const fs = await import('fs');
            logger.debug(`[Queue] 파일 경로에서 내용 읽기: ${options.filepath}`);
            const fileContent = fs.readFileSync(options.filepath, 'utf8');
            
            if (!fileContent || fileContent.trim().length === 0) {
                throw new Error('파일이 비어 있습니다.');
            }
            
            promptText = fileContent;
            logger.debug(`[Queue] 파일 내용 읽기 성공: ${fileContent.length}바이트`);
            
            // 추가 프롬프트가 있으면 파일 내용 뒤에 추가
            if (options.prompt && options.prompt.trim()) {
                promptText += '\n\n' + options.prompt;
                logger.debug(`[Queue] 추가 프롬프트 포함: ${options.prompt.length}바이트`);
            }
        } catch (fileError) {
            logger.error(`[Queue] 파일 읽기 실패: ${fileError.message}`);
            throw new Error(`파일을 읽을 수 없습니다: ${fileError.message}`);
        }
    } else {
        // filepath가 없으면 prompt만 사용
        promptText = options.prompt || '';
    }
    
    // 미리 hash, job_id, log_file 생성
    const hash = crypto.createHash('md5').update(promptText).digest('hex').slice(0, 8);
    const timestamp = new Date().toISOString().replace(/[-:TZ.]/g, '').slice(0, 15);
    const jobId = `${hash}_${timestamp}`;
    const logFilePath = path.join(os.homedir(), '.airun', 'logs', `report_generator_${jobId}.log`);

    const job = await reportGenerationQueue.add({
        data: {
            ...options,
            prompt: promptText, // 파일에서 읽은 내용 또는 원래 prompt 사용
            project_hash: hash,
            job_id: jobId,
            log_file: logFilePath,
            userId: options.userId, // 사용자 ID 명시적 전달
            username: options.username, // 사용자명 명시적 전달
        },
        handlerEndpoint
    });

    return {
        success: true,
        jobId: jobId,  // Bull 큐 ID 대신 실제 프로젝트 해시 반환
        bullJobId: job.id,  // Bull 큐 ID는 별도 필드로 저장
        message: '보고서 생성 작업이 큐에 추가되었습니다.',
        downloadInfo: []
    };
}

// 건강검진 배치 분석 요청 큐에 작업 추가
export async function enqueueHealthBatchAnalysis(jobId, patients) {
    try {
        console.log(`건강검진 배치 분석 큐에 작업 추가: ${jobId}, 환자 수: ${patients.length}`);
        
        // 각 환자별로 개별 작업 생성 (병렬 처리)
        const jobs = await Promise.all(
            patients.map(async (patient, index) => {
                return await healthBatchQueue.add('analyze-patient', {
                    jobId,
                    patientId: patient.id,
                    patientData: patient,
                    priority: index < 10 ? 1 : 2 // 처음 10명은 높은 우선순위
                }, {
                    priority: index < 10 ? 1 : 2
                });
            })
        );
        
        console.log(`건강검진 배치 분석 작업 ${jobs.length}개가 큐에 추가되었습니다.`);
        
        return {
            success: true,
            jobId,
            queuedJobs: jobs.length,
            message: `${jobs.length}개 환자 분석 작업이 큐에 추가되었습니다.`
        };
    } catch (error) {
        logger.error('건강검진 배치 분석 작업 추가 실패:', error);
        throw error;
    }
}

// API 요청 큐에 작업 추가
export async function enqueueApiRequest(endpoint, method, params, body, headers, handlerEndpoint, priority = 'normal') {
    try {
        const options = { priority: priority === 'high' ? 1 : priority === 'low' ? 10 : 5 };
        const job = await apiRequestQueue.add({
            endpoint,
            method,
            params,
            body,
            headers,
            handlerEndpoint,
            timestamp: Date.now()
        }, options);
        return { success: true, jobId: job.id, downloadInfo: [] };
    } catch (error) {
        logger.error('API 요청 작업 추가 실패:', error);
        return { success: false, error: error.message };
    }
}

// 큐 정리 함수
export async function cleanup() {
    try {
        await Promise.all([
            apiUsageQueue.close(),
            licenseCheckQueue.close(),
            apiRequestQueue.close(),
            chatRequestQueue.close(),
            reportGenerationQueue.close()
        ]);
        
        // Redis 연결 종료
        if (redisClient) {
            await redisClient.quit();
        }
        
        console.log('모든 큐와 Redis 연결이 정상적으로 종료되었습니다.');
    } catch (error) {
        logger.error('큐 정리 중 오류 발생:', error);
    }
}

// API 사용량 처리 작업자
apiUsageQueue.process(async (job) => {
    const { endpoint } = job.data;
    console.log(`API 사용량 기록 처리 - Endpoint: ${endpoint}`);
    
    try {
        // API 사용량 증가 로직은 apiUsageManager에서 처리
        return { success: true };
    } catch (error) {
        logger.error('API 사용량 기록 처리 중 오류:', error);
        throw error;
    }
});

// 채팅 요청 처리 작업자
chatRequestQueue.process(async (job) => {
    const { 
        prompt, 
        sessionId, 
        userId, 
        username, // username 정보 추가
        rag, 
        web, 
        enableSmartResearch, // AI프로 버튼 상태 추가
        hideCode, 
        autoExecute, 
        handlerEndpoint, 
        settings, 
        temperature, 
        ragSearchScope, // RAG 검색 범위 추가
        messages_,
        promptSession,
        useReview,
        // 멘션 기능 관련 필드 추가
        mentionedDocumentContent,
        mentionedDocuments
    } = job.data;

    // console.log(chalk.cyan('[Queue] 작업 처리 시작:'), {
    //     jobId: job.id,
    //     handlerEndpoint,
    //     hasPromptSession: !!promptSession,
    //     hasImage: !!promptSession?.image,
    //     imageType: promptSession?.image ? (typeof promptSession.image) : 'none',
    //     imageIsPath: promptSession?.image ? !promptSession.image.startsWith('data:') : false,
    //     imageLength: promptSession?.image ? promptSession.image.length : 0
    // });

    try {
        // 초기 검증을 먼저 수행하여 빠른 실패 처리
        if (!handlerEndpoint) {
            throw new Error('채팅 핸들러 엔드포인트가 제공되지 않았습니다');
        }
        
        const handler = global.apiHandlers[handlerEndpoint];
        if (!handler) {
            throw new Error(`엔드포인트 '${handlerEndpoint}'에 대한 핸들러 함수를 찾을 수 없습니다`);
        }
        
        // 세션 객체 가져오기 (병렬로 처리 가능한 부분은 병렬 처리)
        let currentSession = null;
        if (sessionId) {
            // SharedContext에서 기존 세션만 가져오기
            const { sharedContext } = await import('../aiFeatures.js');
            currentSession = await sharedContext.getSession(sessionId);
            
            if (!currentSession) {
                logger.warn(`[Queue] 세션을 찾을 수 없습니다: ${sessionId}`);
                // 큐에서는 세션을 생성하지 않고 에러 처리
                throw new Error(`세션을 찾을 수 없습니다: ${sessionId}. 세션은 API 서버에서 먼저 생성되어야 합니다.`);
            }
            
            // console.log(`[Queue] 기존 세션 사용: ${sessionId}, 타입: ${currentSession.getSessionType()}`);
        }

        // 작업 진행 상태 업데이트 (검증 완료 후)
        await job.progress(20);

        // chat/sync와 동일한 구조로 데이터 구성
        const handlerOptions = {
            sessionId,      // 세션 식별자
            userId,      // 사용자 식별자
            username,    // 사용자명 추가
            rag,
            web,
            enableSmartResearch, // AI프로 버튼 상태 전달
            hideCode,
            autoExecute,
            settings,
            temperature,
            ragSearchScope, // RAG 검색 범위 전달
            messages_,
            promptSession,
            useReview,
            // 멘션 기능 관련 필드 추가
            mentionedDocumentContent,
            mentionedDocuments,
            session: currentSession, // 세션 객체 전달
            job: job // job 객체 전달하여 progressMessages 지원!
        };

        // console.log(chalk.cyan('[Queue] 핸들러 호출 구조:'), {
        //     hasPromptSession: !!handlerOptions.promptSession,
        //     promptSessionImage: !!handlerOptions.promptSession?.image,
        //     imageCount: handlerOptions.promptSession?.images?.length || 0,
        //     imageType: handlerOptions.promptSession?.image ? (typeof handlerOptions.promptSession.image) : 'none'
        // });

        await job.progress(40);

        // 핸들러 호출 시 타임아웃 설정 (큐 설정값 사용)
        const queueTimeout = config.queues.chatRequest.options.timeout || 3600000; // 기본 1시간
        const timeoutPromise = new Promise((_, reject) => {
            setTimeout(() => reject(new Error('Handler timeout')), queueTimeout);
        });

        const handlerPromise = handler(prompt, handlerOptions);
        const result = await Promise.race([handlerPromise, timeoutPromise]);

        await job.progress(100);
        
        // 다운로드 정보와 웹 참조 정보를 결과에 포함하여 반환
        if (result && typeof result === 'object') {
            // chat() 함수는 { success: true, data: { ... downloadInfo: [] } } 형태로 응답하므로
            // data 내부에서 downloadInfo 추출
            const downloadInfo = result.data?.downloadInfo || result.downloadInfo || [];
            
            logger.debug(`[Queue] 작업 완료 - 다운로드 정보: ${downloadInfo.length}개`);
            downloadInfo.forEach((info, index) => {
                logger.debug(`[Queue] 다운로드 ${index + 1}: ${info.filename} (페이지: ${info.pages?.join(', ') || 'N/A'})`);
            });
            
            // 작업 완료 후 세션 저장
            if (currentSession) {
                try {
                    const dbManager = await import('../services/database/index.js');
                    await dbManager.default.saveSession(currentSession);
                    logger.debug(`[Queue] 세션 저장 완료: ${sessionId}`);
                } catch (saveError) {
                    logger.error(`[Queue] 세션 저장 실패:`, saveError);
                }
            }
            
            return {
                ...result,
                downloadInfo: downloadInfo
            };
        }
        
        return result;
    } catch (error) {
        logger.error(chalk.red('[Queue] 작업 처리 실패:'), {
            jobId: job.id,
            error: error.message,
            stack: error.stack
        });
        throw error;
    }
});

// 코드 요청 처리 작업자도 추가
chatRequestQueue.process('code', async (job) => {
    const { prompt, sessionId, userId, rag, web, hideCode, autoExecute, handlerEndpoint, settings, temperature, ragSearchScope, messages_ } = job.data;
    console.log(`코드 요청 처리 시작 - Job ID: ${job.id}, 핸들러 엔드포인트: ${handlerEndpoint}`);
    
    try {
        if (!handlerEndpoint) {
            throw new Error('코드 핸들러 엔드포인트가 제공되지 않았습니다');
        }
        
        const handler = global.apiHandlers[handlerEndpoint];
        if (!handler) {
            throw new Error(`엔드포인트 '${handlerEndpoint}'에 대한 핸들러 함수를 찾을 수 없습니다`);
        }
        
        console.log(`[Queue Code] 코드 핸들러 호출됨:`, { 
            promptLength: prompt?.length, 
            sessionId,
            rag,
            web,
            hideCode,
            autoExecute,
            settings,
            temperature,
            messagesCount: messages_?.length || 0
        });
        
        // 세션 객체 가져오기 (큐에서는 생성하지 않음)
        let currentSession = null;
        if (sessionId) {
            const { sharedContext } = await import('../aiFeatures.js');
            currentSession = await sharedContext.getSession(sessionId);
            
            if (!currentSession) {
                logger.warn(`[Queue Code] 세션을 찾을 수 없습니다: ${sessionId}`);
                throw new Error(`세션을 찾을 수 없습니다: ${sessionId}. 세션은 API 서버에서 먼저 생성되어야 합니다.`);
            }
            
            console.log(`[Queue Code] 기존 세션 사용: ${sessionId}, 타입: ${currentSession.getSessionType()}`);
        }

        const result = await handler(prompt, {
            sessionId,
            userId,
            rag: rag === true,
            web: web === true,
            hideCode,
            autoExecute,
            ragSearchScope, // RAG 검색 범위 전달
            parameters: {
                ...settings,
                temperature: temperature ?? settings?.temperature ?? 0
            },
            messages_: messages_ || [],
            _fromQueue: true,
            session: currentSession // 세션 객체 전달
        });
        
        if (!result || !result.success) {
            throw new Error(result?.error?.message || '코드 요청 처리 실패');
        }
        
        // if (result?.data?.choices?.[0]?.message?.content) {
        //     console.log('\n[AI 응답]');
        //     console.log('------------------------');
        //     console.log(result.data.choices[0].message.content);
        //     console.log('------------------------\n');
        // }
        
        return result;
    } catch (error) {
        logger.error('코드 요청 처리 오류:', error);
        throw new Error(error.message || '알 수 없는 오류가 발생했습니다');
    }
});

// 보고서 생성 처리 작업자
console.log(chalk.blue('🔧 [큐 워커] 보고서 생성 큐 워커 등록 중...'));
reportGenerationQueue.process(async (job) => {
    const data = job.data.data || job.data; // 중첩 구조 또는 직접 구조 모두 지원
    const userInfo = data.username ? `(사용자: ${data.username})` : data.userId ? `(사용자 ID: ${data.userId})` : '(사용자: 미지정)';
    console.log(chalk.blue(`[보고서 생성 시작] Job ID: ${job.id} ${userInfo}`));
    console.log(chalk.cyan('처리할 데이터:'), {
        ...data,
        userId: data.userId,
        username: data.username,
        prompt: data.prompt ? `${data.prompt.substring(0, 100)}...` : 'N/A'
    });
    
    try {
        // 먼저 airun-report-server.py 서비스 상태 확인
        let useReportServer = false;
        try {
            const healthResponse = await fetchWithTimeout(`http://localhost:${process.env.REPORT_SERVER_PORT || 5620}/health`, {
                method: 'GET',
                timeout: 3000 // 3초 타임아웃
            });
            
            if (healthResponse.ok) {
                const healthData = await healthResponse.json();
                useReportServer = healthData.status === 'healthy';
                console.log(chalk.green(`[보고서 서버 상태] 서비스 사용 가능: ${useReportServer}`));
            }
        } catch (healthError) {
            console.log(chalk.yellow(`[보고서 서버 상태] 서비스 연결 실패, 폴백 모드 사용: ${healthError.message}`));
        }

        if (useReportServer) {
            // airun-report-server.py로 요청 전달
            console.log(chalk.blue('[보고서 생성] 전용 서버 사용'));
            
            const response = await fetchWithTimeout(`http://localhost:${process.env.REPORT_SERVER_PORT || 5620}/generate`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json'
                },
                body: JSON.stringify({
                    prompt: data.prompt,
                    format: data.format || 'pdf',
                    template: data.template || 'simple',
                    project_hash: data.project_hash,
                    job_id: data.job_id, // 큐에서 생성한 실제 Job ID 전달 (job.id는 Bull 큐 내부 ID)
                    username: data.username || 'system', // 사용자명 전달 (username 필드만 사용)
                    email: data.email, // 이메일 주소 전달
                    // AI 모델 설정 전달
                    provider: data.provider,
                    model: data.model
                }),
                timeout: 10000 // 10초 타임아웃
            });

            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const result = await response.json();
            
            if (!result.success) {
                throw new Error(result.message || '보고서 생성 서버 오류');
            }

            // 작업 상태 확인 및 완료 대기
            const reportJobId = result.job_id;
            let attempts = 0;
            const maxAttempts = 180; // 3분 (1초 간격)
            
            // 초기 진행률 설정
            await job.progress(10);
            console.log(chalk.cyan(`[보고서 진행률] Job ID: ${job.id} - 10% (서버 작업 시작)`));
            
            while (attempts < maxAttempts) {
                const statusResponse = await fetchWithTimeout(`http://localhost:${process.env.REPORT_SERVER_PORT || 5620}/status/${reportJobId}`, {
                    timeout: 5000 // 5초 타임아웃
                });
                const statusData = await statusResponse.json();
                
                if (statusData.status === 'completed') {
                    // 파일 경로가 있는 경우 파일 존재 여부 추가 확인
                                         if (statusData.file_path) {
                         if (!fs.existsSync(statusData.file_path)) {
                            console.log(chalk.red(`[보고서 생성 오류] 파일이 존재하지 않음: ${statusData.file_path}`));
                            throw new Error(`생성된 파일을 찾을 수 없습니다: ${statusData.file_path}`);
                        }
                        
                        const fileSize = fs.statSync(statusData.file_path).size;
                        if (fileSize === 0) {
                            console.log(chalk.red(`[보고서 생성 오류] 파일이 비어있음: ${statusData.file_path}`));
                            throw new Error(`생성된 파일이 비어있습니다: ${statusData.file_path}`);
                        }
                        
                        console.log(chalk.green(`[보고서 생성 완료] Job ID: ${job.id} ${userInfo} (전용 서버)`));
                        console.log(chalk.cyan('생성 결과:'), { ...statusData, file_size: fileSize });
                    } else {
                        console.log(chalk.green(`[보고서 생성 완료] Job ID: ${job.id} ${userInfo} (전용 서버)`));
                        console.log(chalk.cyan('생성 결과:'), statusData);
                    }
                    
                    await job.progress(100);
                    
                    return {
                        success: true,
                        file_path: statusData.file_path,
                        job_id: reportJobId,
                        status: 'completed',
                        method: 'report-server'
                    };
                } else if (statusData.status === 'failed') {
                    throw new Error(statusData.error || '보고서 생성 실패');
                }
                
                // 진행률 업데이트 (더 세밀한 처리)
                let currentProgress = 10; // 기본 시작 진행률
                
                if (statusData.progress !== undefined && statusData.progress !== null) {
                    // 서버에서 제공하는 진행률 사용 (10-95% 범위로 매핑)
                    currentProgress = Math.min(95, Math.max(10, 10 + (statusData.progress * 0.85)));
                } else {
                    // 진행률 정보가 없으면 시간 기반으로 추정 (10-80% 범위)
                    const timeProgress = Math.min(70, (attempts / maxAttempts) * 70);
                    currentProgress = 10 + timeProgress;
                }
                
                await job.progress(Math.round(currentProgress));
                
                if (attempts % 10 === 0) { // 10초마다 로그 출력
                    console.log(chalk.cyan(`[보고서 진행률] Job ID: ${job.id} - ${Math.round(currentProgress)}% (${attempts}초 경과)`));
                }
                
                await new Promise(resolve => setTimeout(resolve, 1000));
                attempts++;
            }
            
            throw new Error('보고서 생성 시간 초과');
            
        } else {
            // 폴백: 기존 ReportManager 사용
            console.log(chalk.blue('[보고서 생성] 폴백 모드 - ReportManager 사용'));
            
            // 폴백 모드 진행률 시뮬레이션
            await job.progress(5);
            console.log(chalk.cyan(`[보고서 진행률] Job ID: ${job.id} - 5% (폴백 모드 시작)`));
            
            const ReportManager = (await import('../services/report/ReportManager.js')).default;
            const reportManager = new ReportManager({ debug: true });
            await reportManager.initialize();
            
            await job.progress(20);
            console.log(chalk.cyan(`[보고서 진행률] Job ID: ${job.id} - 20% (ReportManager 초기화 완료)`));

            // 진행률 콜백 함수 정의
            const progressCallback = async (message, progress) => {
                if (progress >= 0) {
                    // 20-95% 범위로 매핑
                    const mappedProgress = Math.min(95, Math.max(20, 20 + (progress * 0.75)));
                    await job.progress(Math.round(mappedProgress));
                    console.log(chalk.cyan(`[보고서 진행률] Job ID: ${job.id} - ${Math.round(mappedProgress)}% (${message})`));
                }
            };

            // 문서 생성 실행 (project_hash, job_id, log_file을 명시적으로 전달)
            const result = await reportManager.generateBusinessPlan({
                executive_summary: data.prompt,
                format: data.format,
                template: data.template,
                project_hash: data.project_hash,
                job_id: data.job_id,
                log_file: data.log_file,
                progressCallback: progressCallback // 진행률 콜백 추가
            });

            // 파일 경로가 있는 경우 파일 존재 여부 확인
                         if (result.file_path) {
                 if (!fs.existsSync(result.file_path)) {
                    console.log(chalk.red(`[보고서 생성 오류] 파일이 존재하지 않음: ${result.file_path}`));
                    throw new Error(`생성된 파일을 찾을 수 없습니다: ${result.file_path}`);
                }
                
                const fileSize = fs.statSync(result.file_path).size;
                if (fileSize === 0) {
                    console.log(chalk.red(`[보고서 생성 오류] 파일이 비어있음: ${result.file_path}`));
                    throw new Error(`생성된 파일이 비어있습니다: ${result.file_path}`);
                }
                
                console.log(chalk.green(`[보고서 생성 완료] Job ID: ${job.id} ${userInfo} (폴백 모드)`));
                console.log(chalk.cyan('생성 결과:'), { ...result, file_size: fileSize });
            } else {
                console.log(chalk.green(`[보고서 생성 완료] Job ID: ${job.id} ${userInfo} (폴백 모드)`));
                console.log(chalk.cyan('생성 결과:'), result);
            }

            await job.progress(100);

            return {
                success: true,
                ...result,
                method: 'report-manager'
            };
        }
        
    } catch (error) {
        console.error(chalk.red('\n[보고서 생성 오류]'));
        console.error(chalk.red('─'.repeat(50)));
        console.error(chalk.cyan('- 상태: '), '오류 발생');
        console.error(chalk.cyan('- Job ID: '), job.id);
        console.error(chalk.cyan('- 오류 메시지: '), error.message);
        console.error(chalk.cyan('- 스택 트레이스: '), error.stack);
        console.error(chalk.red('─'.repeat(50)));
        
        // 보고서 서버 오류인 경우 폴백 시도
        if (error.message.includes('HTTP error') || error.message.includes('서버 오류')) {
            console.log(chalk.yellow('[보고서 생성] 서버 오류로 인한 폴백 모드 시도'));
            
            try {
                await job.progress(5);
                console.log(chalk.cyan(`[보고서 진행률] Job ID: ${job.id} - 5% (폴백 복구 시작)`));
                
                const ReportManager = (await import('../services/report/ReportManager.js')).default;
                const reportManager = new ReportManager({ debug: true });
                await reportManager.initialize();
                
                await job.progress(20);
                console.log(chalk.cyan(`[보고서 진행률] Job ID: ${job.id} - 20% (폴백 복구 초기화 완료)`));

                // 진행률 콜백 함수 정의
                const fallbackProgressCallback = async (message, progress) => {
                    if (progress >= 0) {
                        // 20-95% 범위로 매핑
                        const mappedProgress = Math.min(95, Math.max(20, 20 + (progress * 0.75)));
                        await job.progress(Math.round(mappedProgress));
                        console.log(chalk.cyan(`[보고서 진행률] Job ID: ${job.id} - ${Math.round(mappedProgress)}% (폴백 복구: ${message})`));
                    }
                };

                const result = await reportManager.generateBusinessPlan({
                    executive_summary: data.prompt,
                    format: data.format,
                    template: data.template,
                    project_hash: data.project_hash,
                    job_id: data.job_id,
                    log_file: data.log_file,
                    progressCallback: fallbackProgressCallback // 진행률 콜백 추가
                });

                // 파일 경로가 있는 경우 파일 존재 여부 확인
                if (result.file_path) {
                    if (!fs.existsSync(result.file_path)) {
                        console.log(chalk.red(`[보고서 생성 오류] 파일이 존재하지 않음: ${result.file_path}`));
                        throw new Error(`생성된 파일을 찾을 수 없습니다: ${result.file_path}`);
                    }
                    
                    const fileSize = fs.statSync(result.file_path).size;
                    if (fileSize === 0) {
                        console.log(chalk.red(`[보고서 생성 오류] 파일이 비어있음: ${result.file_path}`));
                        throw new Error(`생성된 파일이 비어있습니다: ${result.file_path}`);
                    }
                    
                    console.log(chalk.green(`[보고서 생성 완료] Job ID: ${job.id} ${userInfo} (폴백 복구)`));
                    console.log(chalk.cyan('생성 결과:'), { ...result, file_size: fileSize });
                } else {
                    console.log(chalk.green(`[보고서 생성 완료] Job ID: ${job.id} ${userInfo} (폴백 복구)`));
                    console.log(chalk.cyan('생성 결과:'), result);
                }

                await job.progress(100);
                
                return {
                    success: true,
                    ...result,
                    method: 'report-manager-fallback'
                };
            } catch (fallbackError) {
                console.error(chalk.red('[보고서 생성] 폴백 모드도 실패:'), fallbackError.message);
                return {
                    success: false,
                    error: `원본 오류: ${error.message}, 폴백 오류: ${fallbackError.message}`
                };
            }
        }
        
        return {
            success: false,
            error: error.message || '문서 생성 중 알 수 없는 오류 발생'
        };
    }
});

// API 요청 처리 작업자
apiRequestQueue.process(async (job) => {
    const { endpoint, method, params, body, headers, handlerEndpoint } = job.data;
    console.log(`API 요청 처리 시작 - Endpoint: ${endpoint}, Job ID: ${job.id}`);
    
    try {
        if (!handlerEndpoint) {
            throw new Error('API 핸들러 엔드포인트가 제공되지 않았습니다');
        }
        
        const handler = global.apiHandlers[handlerEndpoint];
        if (!handler) {
            throw new Error(`엔드포인트 '${handlerEndpoint}'에 대한 핸들러 함수를 찾을 수 없습니다`);
        }

        // RAG 관련 엔드포인트 처리
        if (handlerEndpoint.startsWith('/api/v1/rag/')) {
            console.log(chalk.blue(`[RAG] ${handlerEndpoint} 처리 시작`));
            // console.log(chalk.cyan('요청 데이터:'), body);

            const result = await handler(body);
            
            if (result.success) {
                console.log(chalk.green(`[RAG] ${handlerEndpoint} 처리 완료`));
                // console.log(chalk.cyan('처리 결과:'), result);
            } else {
                console.log(chalk.red(`[RAG] ${handlerEndpoint} 처리 실패`));
                console.log(chalk.red('오류:'), result.error);
            }
            
            return result;
        }
        
        // 기타 API 요청 처리
        return await handler(body);
    } catch (error) {
        logger.error('API 요청 처리 오류:', error);
        throw error;
    }
});

// 라이선스 파일 직접 읽기 함수
function readLicenseFile() {
  try {
    if (fs.existsSync(LICENSE_FILE)) {
      const data = fs.readFileSync(LICENSE_FILE, 'utf8');
      return JSON.parse(data);
    }
    return null;
  } catch (error) {
    logger.error('라이선스 파일 읽기 오류:', error);
    return null;
  }
}

// 라이선스 파일 직접 저장 함수
function saveLicenseFile(licenseData) {
  try {
    fs.writeFileSync(LICENSE_FILE, JSON.stringify(licenseData, null, 2), 'utf8');
    return true;
  } catch (error) {
    logger.error('라이선스 파일 저장 오류:', error);
    return false;
  }
}

// 라이선스 체크 작업자
licenseCheckQueue.process(async (job) => {
  console.log('라이선스 체크 작업 시작 - Job ID:', job.id);
  
  try {
    // licenseManager를 통한 라이선스 체크
    const licenseData = await licenseManager.readLicense();
    
    if (!licenseData) {
      return { valid: false, reason: 'license_not_found' };
    }
    
    // 라이선스 키 검증
    const validationResult = await licenseManager.validateLicense(licenseData.key);
    if (!validationResult || !validationResult.valid) {
      return { valid: false, reason: 'license_not_valid' };
    }
    
    return {
      valid: true,
      apiPlan: licenseData.apiPlan || 'basic',
      apiLimit: licenseData.apiUsage?.limit || 1000,
      apiUsage: licenseData.apiUsage?.used || 0,
      remaining: (licenseData.apiUsage?.limit || 1000) - (licenseData.apiUsage?.used || 0),
      resetDate: licenseData.apiUsage?.resetDate || null
    };
  } catch (error) {
    logger.error('라이선스 체크 오류:', error);
    // 오류 발생 시 기본 응답 반환 (전체 시스템 중단 방지)
    return { valid: false, reason: 'license_check_error', error: error.message };
  }
});

// Redis 클라이언트 export 추가
export { redisClient };

// 큐 워커 등록 상태 확인
console.log(chalk.green('✅ [큐 시스템] 모든 큐 워커 등록 완료'));
console.log(chalk.cyan('- API 사용량 큐 워커: 등록됨'));
console.log(chalk.cyan('- 채팅 요청 큐 워커: 등록됨'));
console.log(chalk.cyan('- 보고서 생성 큐 워커: 등록됨'));
console.log(chalk.cyan('- API 요청 큐 워커: 등록됨'));
console.log(chalk.cyan('- 라이선스 체크 큐 워커: 등록됨'));
