#!/usr/bin/env node

import axios from 'axios';
import { Pool } from 'pg';
import fs from 'fs/promises';
import path from 'path';
import os from 'os';
import { config } from 'dotenv';

// 환경 변수 로드
config({ path: path.join(process.cwd(), '.env') });

// PostgreSQL 연결 설정
const dbPool = new Pool({
  host: process.env.DB_HOST || process.env.POSTGRES_HOST || 'localhost',
  port: process.env.DB_PORT || process.env.POSTGRES_PORT || 5433,
  user: process.env.DB_USER || process.env.POSTGRES_USER || 'ivs',
  password: process.env.DB_PASSWORD || process.env.POSTGRES_PASSWORD || '1234',
  database: process.env.DB_NAME || process.env.POSTGRES_DB || 'airun',
  max: 10,
  idleTimeoutMillis: 30000,
  connectionTimeoutMillis: 10000,
});

// 공공데이터포털 API 키
const PUBLIC_DATA_API_KEY = '156a84b10f1664ffa60f7c9836badf761fc72eaacb24130ed433c569a5bf9b89';

// 로그 관리
const logDir = path.join(os.homedir(), '.airun', 'logs');

const writeLog = async (message) => {
  const timestamp = new Date().toISOString();
  const logMessage = `[${timestamp}] ${message}\n`;
  
  console.log(logMessage.trim());
  
  try {
    await fs.mkdir(logDir, { recursive: true });
    await fs.appendFile(path.join(logDir, 'daily-business-collector.log'), logMessage);
  } catch (error) {
    console.error('로그 파일 쓰기 실패:', error);
  }
};

// 데이터베이스 헬퍼 함수
async function insertDataCollectionLog(source, collectionType, totalItems, newItems, updatedItems, failedItems, startTime, endTime, status, errorMessage = null, details = {}) {
  const duration = Math.floor((endTime - startTime) / 1000);
  
  const query = `
    INSERT INTO data_collection_logs (
      source, collection_type, total_items, new_items, updated_items, failed_items,
      start_time, end_time, duration_seconds, status, error_message, details
    ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12)
    RETURNING id
  `;
  
  const values = [
    source, collectionType, totalItems, newItems, updatedItems, failedItems,
    startTime, endTime, duration, status, errorMessage, JSON.stringify(details)
  ];
  
  try {
    const result = await dbPool.query(query, values);
    return result.rows[0].id;
  } catch (error) {
    await writeLog(`수집 로그 저장 실패: ${error.message}`);
    throw error;
  }
}

async function upsertBusinessAnnouncement(data) {
  const query = `
    INSERT INTO business_announcements (
      id, source, external_id, title, description, category, subcategory,
      budget, support_amount, support_ratio, announcement_date, application_start,
      application_end, project_start, project_end, eligibility, requirements,
      evaluation, contact, documents, status, raw_data, created_at, updated_at
    ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10, $11, $12, $13, $14, $15, $16, $17, $18, $19, $20, $21, $22, $23, $24)
    ON CONFLICT (id) DO UPDATE SET
      title = EXCLUDED.title,
      description = EXCLUDED.description,
      category = EXCLUDED.category,
      subcategory = EXCLUDED.subcategory,
      budget = EXCLUDED.budget,
      support_amount = EXCLUDED.support_amount,
      support_ratio = EXCLUDED.support_ratio,
      announcement_date = EXCLUDED.announcement_date,
      application_start = EXCLUDED.application_start,
      application_end = EXCLUDED.application_end,
      project_start = EXCLUDED.project_start,
      project_end = EXCLUDED.project_end,
      eligibility = EXCLUDED.eligibility,
      requirements = EXCLUDED.requirements,
      evaluation = EXCLUDED.evaluation,
      contact = EXCLUDED.contact,
      documents = EXCLUDED.documents,
      status = EXCLUDED.status,
      raw_data = EXCLUDED.raw_data,
      updated_at = CURRENT_TIMESTAMP
    RETURNING (xmax = 0) AS was_inserted
  `;

  try {
    const result = await dbPool.query(query, [
      data.id, data.source, data.external_id, data.title, data.description,
      data.category, data.subcategory, data.budget, data.support_amount,
      data.support_ratio, data.announcement_date, data.application_start,
      data.application_end, data.project_start, data.project_end,
      JSON.stringify(data.eligibility), JSON.stringify(data.requirements),
      JSON.stringify(data.evaluation), JSON.stringify(data.contact),
      JSON.stringify(data.documents), data.status, JSON.stringify(data.raw_data),
      new Date(), new Date()
    ]);

    return result.rows[0].was_inserted;
  } catch (error) {
    await writeLog(`DB 저장 실패: ${error.message}`);
    throw error;
  }
}

// 날짜 파싱 함수
function parseDate(dateStr, format) {
  if (!dateStr) return null;
  
  try {
    if (format === 'YYYYMMDD') {
      const year = dateStr.substring(0, 4);
      const month = dateStr.substring(4, 6) - 1;
      const day = dateStr.substring(6, 8);
      return new Date(year, month, day);
    } else if (format === 'YYYY-MM-DD') {
      return new Date(dateStr + 'T00:00:00Z');
    } else if (format === 'YYYY-MM-DD HH:mm:ss') {
      return new Date(dateStr.replace(' ', 'T') + 'Z');
    }
    return new Date(dateStr);
  } catch (error) {
    return null;
  }
}

// API 호출 함수 (에러 핸들링 개선)
async function callAPI(endpoint, params, retries = 3) {
  for (let i = 0; i < retries; i++) {
    try {
      const response = await axios.get(endpoint, {
        params: {
          serviceKey: PUBLIC_DATA_API_KEY,
          ...params
        },
        timeout: 30000,
        headers: {
          'Accept': 'application/json',
          'User-Agent': 'AIRUN-DailyBusinessCollector/1.0'
        }
      });

      return response.data;
    } catch (error) {
      if (i === retries - 1) {
        if (error.response) {
          throw new Error(`API 응답 오류: ${error.response.status} - ${error.response.statusText}`);
        } else if (error.request) {
          throw new Error('API 요청 실패: 응답 없음');
        } else {
          throw new Error(`API 요청 오류: ${error.message}`);
        }
      }
      
      // 재시도 전 대기
      await new Promise(resolve => setTimeout(resolve, 2000 * (i + 1)));
    }
  }
}

// K-Startup 데이터 처리
function parseKStartupData(item) {
  return {
    id: `KSTARTUP_${item.pbanc_sn}`,
    source: 'KSTARTUP',
    external_id: item.pbanc_sn.toString(),
    title: item.biz_pbanc_nm || '',
    description: item.pbanc_ctnt || '',
    category: item.supt_biz_clsfc || '',
    subcategory: item.intg_pbanc_biz_nm || '',
    budget: null,
    support_amount: null,
    support_ratio: null,
    announcement_date: item.pbanc_rcpt_bgng_dt ? parseDate(item.pbanc_rcpt_bgng_dt, 'YYYYMMDD') : null,
    application_start: item.pbanc_rcpt_bgng_dt ? parseDate(item.pbanc_rcpt_bgng_dt, 'YYYYMMDD') : null,
    application_end: item.pbanc_rcpt_end_dt ? parseDate(item.pbanc_rcpt_end_dt, 'YYYYMMDD') : null,
    project_start: null,
    project_end: null,
    eligibility: {
      target: item.aply_trgt || '',
      business_years: item.biz_enyy || '',
      age_range: item.biz_trgt_age || '',
      region: item.supt_regin || ''
    },
    requirements: {
      content: item.aply_trgt_ctnt || '',
      exclusions: item.aply_excl_trgt_ctnt || ''
    },
    evaluation: {},
    contact: {
      department: item.biz_prch_dprt_nm || '',
      phone: item.prch_cnpl_no || ''
    },
    documents: {
      detail_url: item.detl_pg_url || '',
      guidance_url: item.biz_gdnc_url || '',
      application_url: item.aply_mthd_onli_rcpt_istc || ''
    },
    status: item.rcrt_prgs_yn === 'Y' ? 'active' : 'closed',
    raw_data: item
  };
}

// MSIT 데이터 처리
function parseMSITData(item) {
  return {
    id: `MSIT_${item.pressDt}_${item.subject.replace(/[^a-zA-Z0-9]/g, '').substring(0, 20)}`,
    source: 'MSIT',
    external_id: `${item.pressDt}_${item.subject.substring(0, 50)}`,
    title: item.subject || '',
    description: '',
    category: '과학기술정보통신부',
    subcategory: item.deptName || '',
    budget: null,
    support_amount: null,
    support_ratio: null,
    announcement_date: item.pressDt ? parseDate(item.pressDt, 'YYYY-MM-DD') : null,
    application_start: null,
    application_end: null,
    project_start: null,
    project_end: null,
    eligibility: {},
    requirements: {},
    evaluation: {},
    contact: {
      name: item.managerName || '',
      phone: item.managerTel || ''
    },
    documents: {
      view_url: item.viewUrl || '',
      files: item.files || []
    },
    status: 'active',
    raw_data: item
  };
}

// SME 데이터 처리
function parseSMEData(item) {
  return {
    id: `SME_${item.itemId}`,
    source: 'SME',
    external_id: item.itemId.toString(),
    title: item.title || '',
    description: item.dataContents || '',
    category: '중소벤처기업부',
    subcategory: item.writerPosition || '',
    budget: null,
    support_amount: null,
    support_ratio: null,
    announcement_date: null,
    application_start: item.applicationStartDate ? parseDate(item.applicationStartDate, 'YYYY-MM-DD') : null,
    application_end: item.applicationEndDate ? parseDate(item.applicationEndDate, 'YYYY-MM-DD') : null,
    project_start: null,
    project_end: null,
    eligibility: {},
    requirements: {},
    evaluation: {},
    contact: {
      name: item.writerName || '',
      position: item.writerPosition || '',
      phone: item.writerPhone || '',
      email: item.writerEmail || ''
    },
    documents: {
      view_url: item.viewUrl || '',
      file_name: item.fileName || '',
      file_url: item.fileUrl || ''
    },
    status: 'active',
    raw_data: item
  };
}

// PPS 데이터 처리  
function parsePPSData(item) {
  return {
    id: `PPS_${item.bidNtceNo}`,
    source: 'PPS',
    external_id: item.bidNtceNo.toString(),
    title: item.bidNtceNm || '',
    description: item.ntceKindNm || '',
    category: '조달청',
    subcategory: item.srvceDivNm || '',
    budget: item.asignBdgtAmt ? parseInt(item.asignBdgtAmt) : null,
    support_amount: item.presmptPrce ? parseInt(item.presmptPrce) : null,
    support_ratio: null,
    announcement_date: item.bidNtceDt ? parseDate(item.bidNtceDt, 'YYYY-MM-DD HH:mm:ss') : null,
    application_start: item.bidBeginDt ? parseDate(item.bidBeginDt, 'YYYY-MM-DD HH:mm:ss') : null,
    application_end: item.bidClseDt ? parseDate(item.bidClseDt, 'YYYY-MM-DD HH:mm:ss') : null,
    project_start: null,
    project_end: null,
    eligibility: {
      method: item.cntrctCnclsMthdNm || '',
      region_limit: item.rgnLmtBidLocplcJdgmBssNm || ''
    },
    requirements: {
      bid_method: item.bidMethdNm || '',
      qualification_date: item.bidQlfctRgstDt || ''
    },
    evaluation: {
      opening_date: item.opengDt || '',
      opening_place: item.opengPlce || ''
    },
    contact: {
      name: item.ntceInsttOfclNm || '',
      phone: item.ntceInsttOfclTelNo || '',
      email: item.ntceInsttOfclEmailAdrs || '',
      institution: item.ntceInsttNm || ''
    },
    documents: {
      detail_url: item.bidNtceDtlUrl || '',
      standard_doc_url: item.stdNtceDocUrl || ''
    },
    status: 'active',
    raw_data: item
  };
}

// 각 API 수집 함수들
async function collectKStartupData() {
  const source = 'KSTARTUP';
  const startTime = new Date();
  let totalItems = 0, newItems = 0, updatedItems = 0, failedItems = 0;
  
  try {
    await writeLog(`${source} 신규/활성 데이터 수집 시작`);
    
    // 현재 모집 중인 공고만 가져오기
    const params = {
      page: 1,
      perPage: 100,
      'cond[rcrt_prgs_yn::EQ]': 'Y', // 모집 진행 중인 것만
      returnType: 'json'
    };

    const data = await callAPI('https://apis.data.go.kr/B552735/kisedKstartupService01/getAnnouncementInformation01', params);
    
    if (data && data.data) {
      totalItems = data.data.length;
      
      for (const item of data.data) {
        try {
          const parsedData = parseKStartupData(item);
          const wasInserted = await upsertBusinessAnnouncement(parsedData);
          
          if (wasInserted) {
            newItems++;
          } else {
            updatedItems++;
          }
        } catch (error) {
          failedItems++;
          await writeLog(`${source} 항목 처리 실패: ${error.message}`);
        }
      }
    }

    const endTime = new Date();
    await insertDataCollectionLog(source, 'api', totalItems, newItems, updatedItems, failedItems, startTime, endTime, 'success');
    await writeLog(`${source} 수집 완료: 총 ${totalItems}개, 신규 ${newItems}개, 업데이트 ${updatedItems}개, 실패 ${failedItems}개`);
    
    return { success: true, totalItems, newItems, updatedItems, failedItems };
  } catch (error) {
    const endTime = new Date();
    await insertDataCollectionLog(source, 'api', totalItems, newItems, updatedItems, failedItems, startTime, endTime, 'failed', error.message);
    await writeLog(`${source} 수집 실패: ${error.message}`);
    return { success: false, error: error.message };
  }
}

async function collectMSITData() {
  const source = 'MSIT';
  const startTime = new Date();
  let totalItems = 0, newItems = 0, updatedItems = 0, failedItems = 0;
  
  try {
    await writeLog(`${source} 최신 데이터 수집 시작`);
    
    // 최근 공고만 가져오기
    const params = {
      pageNo: 1,
      numOfRows: 50,
      returnType: 'json'
    };

    const data = await callAPI('https://apis.data.go.kr/1721000/msitannouncementinfo/businessAnnouncMentList', params);
    
    if (data && data.response && data.response[1] && data.response[1].body && data.response[1].body.items) {
      const items = data.response[1].body.items;
      totalItems = items.length;
      
      for (const itemWrapper of items) {
        try {
          const item = itemWrapper.item;
          const parsedData = parseMSITData(item);
          const wasInserted = await upsertBusinessAnnouncement(parsedData);
          
          if (wasInserted) {
            newItems++;
          } else {
            updatedItems++;
          }
        } catch (error) {
          failedItems++;
          await writeLog(`${source} 항목 처리 실패: ${error.message}`);
        }
      }
    }

    const endTime = new Date();
    await insertDataCollectionLog(source, 'api', totalItems, newItems, updatedItems, failedItems, startTime, endTime, 'success');
    await writeLog(`${source} 수집 완료: 총 ${totalItems}개, 신규 ${newItems}개, 업데이트 ${updatedItems}개, 실패 ${failedItems}개`);
    
    return { success: true, totalItems, newItems, updatedItems, failedItems };
  } catch (error) {
    const endTime = new Date();
    await insertDataCollectionLog(source, 'api', totalItems, newItems, updatedItems, failedItems, startTime, endTime, 'failed', error.message);
    await writeLog(`${source} 수집 실패: ${error.message}`);
    return { success: false, error: error.message };
  }
}

async function collectSMEData() {
  const source = 'SME';
  const startTime = new Date();
  let totalItems = 0, newItems = 0, updatedItems = 0, failedItems = 0;
  
  try {
    await writeLog(`${source} 최근 3일 데이터 수집 시작`);
    
    // 최근 3일간의 데이터만 가져오기
    const today = new Date();
    const threeDaysAgo = new Date(today.getTime() - 3 * 24 * 60 * 60 * 1000);
    
    const params = {
      pageNo: 1,
      numOfRows: 30,
      startDate: threeDaysAgo.toISOString().split('T')[0],
      endDate: today.toISOString().split('T')[0]
    };

    const data = await callAPI('https://apis.data.go.kr/1421000/mssBizService_v2/getbizList_v2', params);
    
    if (data && data.response && data.response.body && data.response.body.items) {
      const items = data.response.body.items.item || [];
      const itemList = Array.isArray(items) ? items : [items];
      totalItems = itemList.length;
      
      for (const item of itemList) {
        try {
          const parsedData = parseSMEData(item);
          const wasInserted = await upsertBusinessAnnouncement(parsedData);
          
          if (wasInserted) {
            newItems++;
          } else {
            updatedItems++;
          }
        } catch (error) {
          failedItems++;
          await writeLog(`${source} 항목 처리 실패: ${error.message}`);
        }
      }
    }

    const endTime = new Date();
    await insertDataCollectionLog(source, 'api', totalItems, newItems, updatedItems, failedItems, startTime, endTime, 'success');
    await writeLog(`${source} 수집 완료: 총 ${totalItems}개, 신규 ${newItems}개, 업데이트 ${updatedItems}개, 실패 ${failedItems}개`);
    
    return { success: true, totalItems, newItems, updatedItems, failedItems };
  } catch (error) {
    const endTime = new Date();
    await insertDataCollectionLog(source, 'api', totalItems, newItems, updatedItems, failedItems, startTime, endTime, 'failed', error.message);
    await writeLog(`${source} 수집 실패: ${error.message}`);
    return { success: false, error: error.message };
  }
}

async function collectPPSData() {
  const source = 'PPS';
  const startTime = new Date();
  let totalItems = 0, newItems = 0, updatedItems = 0, failedItems = 0;
  
  try {
    await writeLog(`${source} 최근 2일 데이터 수집 시작`);
    
    // 최근 2일간의 입찰공고만 가져오기
    const today = new Date();
    const endDate = today.toISOString().replace(/T.*/, '').replace(/-/g, '') + '2359';
    const startDate = new Date(today.getTime() - 2 * 24 * 60 * 60 * 1000)
      .toISOString().replace(/T.*/, '').replace(/-/g, '') + '0000';
    
    const params = {
      pageNo: 1,
      numOfRows: 100,
      type: 'json',
      inqryDiv: 1,
      inqryBgnDt: startDate,
      inqryEndDt: endDate
    };

    const data = await callAPI('https://apis.data.go.kr/1230000/ad/BidPublicInfoService/getBidPblancListInfoServc', params);
    
    if (data && data.response && data.response.body && data.response.body.items) {
      const items = data.response.body.items;
      totalItems = items.length;
      
      for (const item of items) {
        try {
          const parsedData = parsePPSData(item);
          const wasInserted = await upsertBusinessAnnouncement(parsedData);
          
          if (wasInserted) {
            newItems++;
          } else {
            updatedItems++;
          }
        } catch (error) {
          failedItems++;
          await writeLog(`${source} 항목 처리 실패: ${error.message}`);
        }
      }
    }

    const endTime = new Date();
    await insertDataCollectionLog(source, 'api', totalItems, newItems, updatedItems, failedItems, startTime, endTime, 'success');
    await writeLog(`${source} 수집 완료: 총 ${totalItems}개, 신규 ${newItems}개, 업데이트 ${updatedItems}개, 실패 ${failedItems}개`);
    
    return { success: true, totalItems, newItems, updatedItems, failedItems };
  } catch (error) {
    const endTime = new Date();
    await insertDataCollectionLog(source, 'api', totalItems, newItems, updatedItems, failedItems, startTime, endTime, 'failed', error.message);
    await writeLog(`${source} 수집 실패: ${error.message}`);
    return { success: false, error: error.message };
  }
}

// 메인 수집 함수
async function runDailyCollection() {
  await writeLog('========================================');
  await writeLog('매일 사업공고 데이터 수집 시작');
  await writeLog('========================================');
  
  const overallStartTime = new Date();
  const results = {
    kstartup: null,
    msit: null,
    sme: null,
    pps: null,
    summary: {
      total_sources: 4,
      successful_sources: 0,
      total_items: 0,
      total_new_items: 0,
      total_updated_items: 0,
      total_failed_items: 0
    }
  };

  // 1. K-Startup 수집
  await writeLog('1/4 K-Startup 데이터 수집...');
  results.kstartup = await collectKStartupData();
  if (results.kstartup.success) {
    results.summary.successful_sources++;
    results.summary.total_items += results.kstartup.totalItems;
    results.summary.total_new_items += results.kstartup.newItems;
    results.summary.total_updated_items += results.kstartup.updatedItems;
    results.summary.total_failed_items += results.kstartup.failedItems;
  }

  // 2초 대기 (API 부하 방지)
  await new Promise(resolve => setTimeout(resolve, 2000));

  // 2. MSIT 수집
  await writeLog('2/4 과학기술정보통신부 데이터 수집...');
  results.msit = await collectMSITData();
  if (results.msit.success) {
    results.summary.successful_sources++;
    results.summary.total_items += results.msit.totalItems;
    results.summary.total_new_items += results.msit.newItems;
    results.summary.total_updated_items += results.msit.updatedItems;
    results.summary.total_failed_items += results.msit.failedItems;
  }

  await new Promise(resolve => setTimeout(resolve, 2000));

  // 3. SME 수집
  await writeLog('3/4 중소벤처기업부 데이터 수집...');
  results.sme = await collectSMEData();
  if (results.sme.success) {
    results.summary.successful_sources++;
    results.summary.total_items += results.sme.totalItems;
    results.summary.total_new_items += results.sme.newItems;
    results.summary.total_updated_items += results.sme.updatedItems;
    results.summary.total_failed_items += results.sme.failedItems;
  }

  await new Promise(resolve => setTimeout(resolve, 2000));

  // 4. PPS 수집
  await writeLog('4/4 조달청 데이터 수집...');
  results.pps = await collectPPSData();
  if (results.pps.success) {
    results.summary.successful_sources++;
    results.summary.total_items += results.pps.totalItems;
    results.summary.total_new_items += results.pps.newItems;
    results.summary.total_updated_items += results.pps.updatedItems;
    results.summary.total_failed_items += results.pps.failedItems;
  }

  const overallEndTime = new Date();
  const totalDuration = Math.floor((overallEndTime - overallStartTime) / 1000);

  // 결과 요약
  await writeLog('========================================');
  await writeLog('데일리 수집 완료 요약:');
  await writeLog(`성공한 소스: ${results.summary.successful_sources}/${results.summary.total_sources}`);
  await writeLog(`총 처리 항목: ${results.summary.total_items}개`);
  await writeLog(`신규 항목: ${results.summary.total_new_items}개`);
  await writeLog(`업데이트 항목: ${results.summary.total_updated_items}개`);
  await writeLog(`실패 항목: ${results.summary.total_failed_items}개`);
  await writeLog(`총 소요 시간: ${totalDuration}초`);
  await writeLog('========================================');

  return results;
}

// 메인 실행
(async () => {
  try {
    const results = await runDailyCollection();
    
    // 정상 종료 코드 (성공한 소스가 있으면 0, 모두 실패하면 1)
    const exitCode = results.summary.successful_sources > 0 ? 0 : 1;
    
    await dbPool.end();
    process.exit(exitCode);
  } catch (error) {
    await writeLog(`치명적 오류 발생: ${error.message}`);
    await dbPool.end();
    process.exit(1);
  }
})();