# Tool Development Guide

AI Tool System을 위한 새로운 도구 개발 가이드입니다.

## 🚀 빠른 시작

### 1. 새 도구 파일 생성

```bash
# tools/custom-tools/ 디렉토리에 새 파일 생성
touch tools/custom-tools/my-new-tools.js
```

### 2. 기본 도구 템플릿

```javascript
// tools/custom-tools/my-new-tools.js

const myTool = {
    name: 'my_tool',
    category: 'utility',
    description: '내 새로운 도구입니다.',
    version: '1.0.0',
    parameters: {
        input: { 
            type: 'string', 
            required: true, 
            description: '입력 값' 
        }
    },
    handler: async (args, context) => {
        const { input } = args;
        const { userId, job } = context;
        
        try {
            // 여기에 도구 로직 구현
            const result = `처리된 결과: ${input}`;
            
            return {
                success: true,
                data: { result },
                message: '작업 완료'
            };
        } catch (error) {
            return {
                success: false,
                error: error.message,
                message: '작업 실패'
            };
        }
    }
};

export default [myTool];
```

### 3. 도구 테스트

```bash
# CLI 도구로 테스트
node tools/cli-tools.js test my_tool
```

## 📋 도구 정의 세부사항

### 필수 필드

- **name**: 도구의 고유 이름 (문자열, 소문자, 언더스코어 사용)
- **handler**: 도구 실행 함수 (비동기 함수)

### 선택 필드

- **category**: 도구 카테고리 (기본값: 'general')
- **description**: 도구 설명
- **version**: 버전 (기본값: '1.0.0')
- **parameters**: 매개변수 정의 객체

### 매개변수 정의

```javascript
parameters: {
    param_name: {
        type: 'string',        // 'string', 'number', 'boolean', 'array', 'object'
        required: true,        // 필수 여부
        default: 'value',      // 기본값
        description: '설명',   // 매개변수 설명
        enum: ['a', 'b', 'c'], // 허용되는 값들 (선택사항)
        min: 0,               // 최소값 (number 타입)
        max: 100,             // 최대값 (number 타입)
        pattern: '^[a-z]+$'   // 정규식 패턴 (string 타입)
    }
}
```

## 🔧 핸들러 함수 작성

### 기본 구조

```javascript
handler: async (args, context) => {
    // 1. 매개변수 추출
    const { param1, param2 = 'default' } = args;
    
    // 2. 컨텍스트 정보 활용
    const { userId, job, toolName, version } = context;
    
    // 3. 진행 상황 업데이트 (장시간 작업 시)
    if (job && job.progress) {
        await job.progress(30, '작업 진행 중...');
    }
    
    try {
        // 4. 실제 작업 수행
        const result = await performTask(param1, param2);
        
        // 5. 성공 응답 반환
        return {
            success: true,
            data: result,
            message: '작업 완료'
        };
    } catch (error) {
        // 6. 실패 응답 반환
        return {
            success: false,
            error: error.message,
            message: '작업 실패'
        };
    }
}
```

### 컨텍스트 객체

```javascript
context = {
    userId: 'user123',           // 사용자 ID
    job: {                       // 작업 객체 (진행 상황 업데이트용)
        progress: async (percent, message) => {},
        update: async (data) => {}
    },
    toolName: 'my_tool',         // 도구 이름
    version: '1.0.0',            // 도구 버전
    registeredAt: '2024-01-01'   // 등록 시간
}
```

### 응답 형식

#### 성공 응답
```javascript
{
    success: true,
    data: {
        // 결과 데이터
    },
    message: '사용자에게 표시할 메시지'
}
```

#### 실패 응답
```javascript
{
    success: false,
    error: '기술적 오류 메시지',
    message: '사용자에게 표시할 친화적 메시지'
}
```

## 🎯 고급 기능

### 1. 진행 상황 업데이트

```javascript
handler: async (args, context) => {
    const { job } = context;
    
    if (job && job.progress) {
        await job.progress(0, '작업 시작...');
        
        // 작업 1
        await doStep1();
        await job.progress(33, '1단계 완료');
        
        // 작업 2
        await doStep2();
        await job.progress(66, '2단계 완료');
        
        // 작업 3
        await doStep3();
        await job.progress(100, '모든 작업 완료');
    }
}
```

### 2. 입력 검증

```javascript
handler: async (args, context) => {
    const { email, age } = args;
    
    // 이메일 형식 검증
    const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
    if (!emailRegex.test(email)) {
        return {
            success: false,
            error: '잘못된 이메일 형식',
            message: '올바른 이메일 주소를 입력해주세요.'
        };
    }
    
    // 나이 범위 검증
    if (age < 0 || age > 150) {
        return {
            success: false,
            error: '나이 범위 오류',
            message: '나이는 0-150 사이여야 합니다.'
        };
    }
    
    // 검증 통과 후 로직 실행
    // ...
}
```

### 3. 외부 API 호출

```javascript
handler: async (args, context) => {
    const { query } = args;
    const { job } = context;
    
    try {
        if (job) await job.progress(20, 'API 호출 중...');
        
        const response = await fetch(`https://api.example.com/search?q=${encodeURIComponent(query)}`, {
            headers: {
                'Authorization': `Bearer ${process.env.API_KEY}`,
                'Content-Type': 'application/json'
            },
            timeout: 10000 // 10초 타임아웃
        });
        
        if (!response.ok) {
            throw new Error(`API 오류: ${response.status} ${response.statusText}`);
        }
        
        const data = await response.json();
        
        if (job) await job.progress(100, 'API 응답 처리 완료');
        
        return {
            success: true,
            data: data,
            message: '검색 완료'
        };
    } catch (error) {
        return {
            success: false,
            error: error.message,
            message: 'API 호출 실패'
        };
    }
}
```

### 4. 파일 시스템 접근

```javascript
handler: async (args, context) => {
    const { filePath } = args;
    
    try {
        const fs = await import('fs/promises');
        const path = await import('path');
        
        // 보안을 위한 경로 검증
        const safePath = path.resolve(filePath);
        const allowedDir = path.resolve('./allowed-directory');
        
        if (!safePath.startsWith(allowedDir)) {
            return {
                success: false,
                error: '접근 권한 없음',
                message: '허용되지 않은 디렉토리입니다.'
            };
        }
        
        // 파일 읽기
        const content = await fs.readFile(safePath, 'utf8');
        
        return {
            success: true,
            data: { content },
            message: '파일 읽기 완료'
        };
    } catch (error) {
        return {
            success: false,
            error: error.message,
            message: '파일 읽기 실패'
        };
    }
}
```

## 📁 도구 배포 방법

### 방법 1: Custom Tools 디렉토리

```bash
# tools/custom-tools/ 에 파일 추가
cp my-tools.js tools/custom-tools/
```

### 방법 2: User Tools 디렉토리

```bash
# 프로젝트 루트에 user-tools 디렉토리 생성
mkdir user-tools
cp my-tools.js user-tools/
```

### 방법 3: 런타임 등록

```javascript
import toolRegistry from './tools/tools-registry.js';

// 개별 등록
toolRegistry.registerTool(myTool);

// 파일에서 로드
await toolRegistry.loadToolsFromFile('./my-tools.js');
```

## 🧪 테스트 및 디버깅

### CLI 도구 사용

```bash
# 도구 목록 보기
node tools/cli-tools.js list

# 도구 정보 확인
node tools/cli-tools.js info my_tool

# 도구 테스트
node tools/cli-tools.js test my_tool

# 도구 다시 로드
node tools/cli-tools.js reload
```

### 로그 출력

```javascript
handler: async (args, context) => {
    const { toolName } = context;
    
    console.log(`[${toolName}] 작업 시작`);
    console.log(`[${toolName}] 매개변수:`, args);
    
    // 작업 수행
    const result = await performTask();
    
    console.log(`[${toolName}] 작업 완료:`, result);
    
    return {
        success: true,
        data: result,
        message: '작업 완료'
    };
}
```

## 📝 예제 도구들

시스템에 포함된 예제 도구들을 참고하세요:

- `tools/custom-tools/example-tools.js` - 기본 예제들
- `user-tools-example/my-custom-tools.js` - 고급 예제들
- `tools/builtin-tools.js` - 시스템 내장 도구들

## 🔐 보안 고려사항

1. **입력 검증**: 모든 사용자 입력을 검증하세요
2. **경로 보안**: 파일 시스템 접근 시 경로를 검증하세요
3. **API 키 보호**: 환경 변수로 API 키를 관리하세요
4. **타임아웃 설정**: 외부 API 호출 시 타임아웃을 설정하세요
5. **오류 처리**: 민감한 정보가 오류 메시지에 노출되지 않도록 하세요

## 🚀 최적화 팁

1. **비동기 처리**: `async/await`를 적절히 사용하세요
2. **메모리 관리**: 큰 데이터를 처리할 때 스트림을 사용하세요
3. **캐싱**: 반복되는 계산 결과를 캐시하세요
4. **배치 처리**: 여러 작업을 배치로 처리하세요
5. **리소스 정리**: 파일 핸들이나 네트워크 연결을 적절히 정리하세요 