# Tool System Integration Guide

기존 AI 시스템에 Tool System을 통합하는 방법을 설명합니다.

## 🔧 시스템 통합 단계

### 1. 도구 로더 초기화

시스템 시작 시 도구 로더를 초기화합니다.

```javascript
// index.js 또는 main entry point
import toolLoader from './tools/tool-loader.js';

async function initializeSystem() {
    console.log('시스템 초기화 중...');
    
    try {
        // 도구 시스템 초기화
        await toolLoader.loadAllTools();
        console.log('✅ 도구 시스템 초기화 완료');
        
        // 기타 시스템 초기화...
        
    } catch (error) {
        console.error('❌ 시스템 초기화 실패:', error);
        process.exit(1);
    }
}

// 시스템 시작
initializeSystem();
```

### 2. mainLogic.js 통합

기존 executeToolCall 함수를 수정하여 레지스트리 시스템을 사용합니다.

```javascript
// mainLogic.js
import toolRegistry from './tools/tools-registry.js';

export async function executeToolCall(toolCall, userId, job = null) {
    try {
        const { name, arguments: args } = toolCall.function;
        
        console.log('[ToolCall] 호출 시작:', { name, args, userId });
        
        // arguments 파싱
        let parsedArgs;
        try {
            if (typeof args === 'string') {
                parsedArgs = JSON.parse(args);
            } else if (typeof args === 'object' && args !== null) {
                parsedArgs = args;
            } else {
                parsedArgs = {};
            }
        } catch (error) {
            console.error('Failed to parse tool arguments:', error);
            parsedArgs = {};
        }
        
        // 새로운 도구 레지스트리 시스템 사용
        try {
            const result = await toolRegistry.executeTool(name, parsedArgs, userId, job);
            console.log(`[ToolCall] 성공: ${name}`, result.success ? '✅' : '❌');
            return result;
        } catch (error) {
            console.error(`[ToolCall] 도구 실행 오류 ${name}:`, error);
            
            // 등록되지 않은 도구인 경우 레거시 fallback 시도
            if (error.message.includes('등록되지 않은 도구')) {
                console.log(`[ToolCall] 레거시 도구 fallback 시도: ${name}`);
                return await executeLegacyTool(name, parsedArgs, userId, job);
            }
            
            return {
                success: false,
                error: error.message,
                message: `도구 실행 실패: ${error.message}`
            };
        }
        
    } catch (error) {
        console.error('Tool execution error:', error);
        return {
            success: false,
            error: error.message,
            message: `도구 호출 처리 중 오류 발생: ${error.message}`
        };
    }
}

// 레거시 도구 실행 (기존 하드코딩된 도구들을 위한 fallback)
async function executeLegacyTool(name, parsedArgs, userId, job) {
    console.log(`[LegacyTool] 실행: ${name}`);
    
    // 기존 switch-case 문이 여기에 위치
    switch (name) {
        // 아직 마이그레이션되지 않은 도구들...
        default:
            return {
                success: false,
                error: `알 수 없는 도구: ${name}`,
                message: `지원하지 않는 도구입니다: ${name}`
            };
    }
}
```

### 3. API 서버 통합

```javascript
// api_server.js
import toolLoader from './tools/tool-loader.js';

async function startServer() {
    // 서버 시작 전 도구 로드
    await toolLoader.loadAllTools();
    
    // Express 앱 설정...
    const app = express();
    
    // 도구 관련 API 엔드포인트
    app.get('/api/tools', (req, res) => {
        const tools = toolRegistry.getToolsByCategory();
        res.json(tools);
    });
    
    app.get('/api/tools/:toolName', (req, res) => {
        const toolInfo = toolRegistry.getToolInfo(req.params.toolName);
        if (toolInfo) {
            res.json(toolInfo);
        } else {
            res.status(404).json({ error: '도구를 찾을 수 없습니다.' });
        }
    });
    
    // 서버 시작
    app.listen(port, () => {
        console.log(`서버가 포트 ${port}에서 실행 중입니다.`);
    });
}

startServer().catch(console.error);
```

## 🔄 기존 도구 마이그레이션

### 1. 기존 하드코딩된 도구 식별

```javascript
// 기존 executeToolCall에서 switch-case로 정의된 도구들을 찾습니다
switch (name) {
    case 'search_web':
        // 웹 검색 로직
        break;
    case 'search_documents':
        // 문서 검색 로직
        break;
    case 'email':
        // 이메일 발송 로직
        break;
    // ... 기타 도구들
}
```

### 2. 새 형식으로 변환

```javascript
// tools/builtin-tools.js
const searchWebTool = {
    name: 'search_web',
    category: 'search',
    description: '웹에서 실시간 정보를 검색합니다.',
    version: '1.0.0',
    parameters: {
        query: {
            type: 'string',
            required: true,
            description: '검색할 키워드나 질문'
        }
    },
    handler: async (args, context) => {
        const { query } = args;
        const { userId, job } = context;
        
        // 기존 웹 검색 로직을 여기로 이동
        try {
            const searchResults = await performWebSearch(query, userId, job);
            return {
                success: true,
                data: searchResults,
                message: '웹 검색 완료'
            };
        } catch (error) {
            return {
                success: false,
                error: error.message,
                message: '웹 검색 실패'
            };
        }
    }
};

export default [searchWebTool, /* 기타 도구들... */];
```

### 3. 점진적 마이그레이션

1. **Phase 1**: 새 도구부터 레지스트리 시스템 사용
2. **Phase 2**: 기존 도구들을 하나씩 새 형식으로 변환
3. **Phase 3**: 레거시 fallback 제거

## 📊 Function Calling Schema 생성

도구 레지스트리에서 자동으로 Function Calling Schema를 생성합니다.

```javascript
// mainLogic.js에서 tools 배열 생성
import toolRegistry from './tools/tools-registry.js';

export function generateToolsSchema() {
    const registeredTools = toolRegistry.getTools();
    const tools = [];
    
    for (const toolName of registeredTools) {
        const toolInfo = toolRegistry.getToolInfo(toolName);
        if (toolInfo) {
            tools.push({
                type: "function",
                function: {
                    name: toolInfo.name,
                    description: toolInfo.description,
                    parameters: {
                        type: "object",
                        properties: convertParametersToSchema(toolInfo.parameters),
                        required: getRequiredParameters(toolInfo.parameters)
                    }
                }
            });
        }
    }
    
    return tools;
}

function convertParametersToSchema(parameters) {
    const properties = {};
    
    for (const [paramName, paramDef] of Object.entries(parameters)) {
        properties[paramName] = {
            type: paramDef.type,
            description: paramDef.description
        };
        
        if (paramDef.enum) {
            properties[paramName].enum = paramDef.enum;
        }
        
        if (paramDef.default !== undefined) {
            properties[paramName].default = paramDef.default;
        }
    }
    
    return properties;
}

function getRequiredParameters(parameters) {
    return Object.entries(parameters)
        .filter(([name, def]) => def.required)
        .map(([name]) => name);
}

// 기존 tools 배열을 동적으로 생성
export const tools = generateToolsSchema();
```

## 🔄 런타임 도구 관리

### 도구 다시 로드

```javascript
// 런타임에 도구 다시 로드
import toolLoader from './tools/tool-loader.js';

export async function reloadTools() {
    try {
        await toolLoader.reloadTools();
        console.log('✅ 도구 다시 로드 완료');
        
        // Function Calling Schema 업데이트
        const newTools = generateToolsSchema();
        console.log(`🔧 ${newTools.length}개 도구 스키마 업데이트됨`);
        
        return {
            success: true,
            message: '도구 다시 로드 완료',
            toolCount: newTools.length
        };
    } catch (error) {
        console.error('❌ 도구 다시 로드 실패:', error);
        return {
            success: false,
            error: error.message,
            message: '도구 다시 로드 실패'
        };
    }
}
```

### 새 도구 동적 추가

```javascript
// 런타임에 새 도구 추가
export async function addTool(toolDefinition) {
    try {
        toolRegistry.registerTool(toolDefinition);
        console.log(`✅ 도구 추가됨: ${toolDefinition.name}`);
        
        return {
            success: true,
            message: `도구 '${toolDefinition.name}' 추가 완료`
        };
    } catch (error) {
        console.error('❌ 도구 추가 실패:', error);
        return {
            success: false,
            error: error.message,
            message: '도구 추가 실패'
        };
    }
}
```

## 🔍 모니터링 및 로깅

### 도구 실행 통계

```javascript
// 도구 실행 통계 수집
class ToolAnalytics {
    constructor() {
        this.stats = new Map();
    }
    
    recordExecution(toolName, success, duration) {
        if (!this.stats.has(toolName)) {
            this.stats.set(toolName, {
                totalCalls: 0,
                successCount: 0,
                failureCount: 0,
                totalDuration: 0,
                avgDuration: 0
            });
        }
        
        const stat = this.stats.get(toolName);
        stat.totalCalls++;
        stat.totalDuration += duration;
        stat.avgDuration = stat.totalDuration / stat.totalCalls;
        
        if (success) {
            stat.successCount++;
        } else {
            stat.failureCount++;
        }
    }
    
    getStats() {
        return Object.fromEntries(this.stats);
    }
}

const toolAnalytics = new ToolAnalytics();

// executeToolCall에서 통계 수집
export async function executeToolCall(toolCall, userId, job = null) {
    const startTime = Date.now();
    let success = false;
    
    try {
        const result = await toolRegistry.executeTool(name, parsedArgs, userId, job);
        success = result.success;
        return result;
    } catch (error) {
        success = false;
        throw error;
    } finally {
        const duration = Date.now() - startTime;
        toolAnalytics.recordExecution(toolCall.function.name, success, duration);
    }
}
```

## 🚀 성능 최적화

### 도구 캐싱

```javascript
// 자주 사용되는 도구 결과 캐싱
class ToolCache {
    constructor(maxSize = 1000, ttl = 300000) { // 5분 TTL
        this.cache = new Map();
        this.maxSize = maxSize;
        this.ttl = ttl;
    }
    
    generateKey(toolName, args) {
        return `${toolName}:${JSON.stringify(args)}`;
    }
    
    get(toolName, args) {
        const key = this.generateKey(toolName, args);
        const cached = this.cache.get(key);
        
        if (cached && Date.now() - cached.timestamp < this.ttl) {
            return cached.result;
        }
        
        if (cached) {
            this.cache.delete(key);
        }
        
        return null;
    }
    
    set(toolName, args, result) {
        const key = this.generateKey(toolName, args);
        
        if (this.cache.size >= this.maxSize) {
            const firstKey = this.cache.keys().next().value;
            this.cache.delete(firstKey);
        }
        
        this.cache.set(key, {
            result,
            timestamp: Date.now()
        });
    }
}

const toolCache = new ToolCache();
```

## 🔐 보안 강화

### 도구 접근 권한

```javascript
// 사용자별 도구 접근 권한 관리
class ToolPermissions {
    constructor() {
        this.permissions = new Map();
    }
    
    setUserPermissions(userId, allowedTools) {
        this.permissions.set(userId, new Set(allowedTools));
    }
    
    hasPermission(userId, toolName) {
        const userPermissions = this.permissions.get(userId);
        if (!userPermissions) {
            return true; // 기본적으로 모든 도구 허용
        }
        return userPermissions.has(toolName);
    }
}

const toolPermissions = new ToolPermissions();

// executeToolCall에서 권한 확인
export async function executeToolCall(toolCall, userId, job = null) {
    const toolName = toolCall.function.name;
    
    if (!toolPermissions.hasPermission(userId, toolName)) {
        return {
            success: false,
            error: '접근 권한 없음',
            message: `도구 '${toolName}'에 대한 접근 권한이 없습니다.`
        };
    }
    
    // 도구 실행...
}
```

이 통합 가이드를 따라하시면 기존 시스템을 손상시키지 않으면서도 확장 가능한 도구 시스템을 점진적으로 도입할 수 있습니다. 