/**
 * 공통 로그 유틸리티 모듈
 * 모든 파일에서 동일한 로그 설정을 사용할 수 있도록 제공
 */

import { getVarVal } from '../configuration.js';

// 로그 레벨 정의
const LOG_LEVELS = {
    ERROR: 0,
    WARN: 1,
    INFO: 2,
    DEBUG: 3
};

// 현재 로그 레벨 캐시
let currentLogLevel = null;
let debugMode = null;

/**
 * 환경변수에서 로그 레벨을 가져옵니다
 */
async function getLogLevel() {
    if (currentLogLevel !== null) {
        return currentLogLevel;
    }

    try {
        // 환경변수에서 LOG_LEVEL 확인
        const logLevelEnv = process.env.LOG_LEVEL || await getVarVal('LOG_LEVEL') || 'INFO';
        currentLogLevel = LOG_LEVELS[logLevelEnv.toUpperCase()] ?? LOG_LEVELS.INFO;
        return currentLogLevel;
    } catch (error) {
        currentLogLevel = LOG_LEVELS.INFO; // 기본값
        return currentLogLevel;
    }
}

/**
 * 디버그 모드 여부를 확인합니다
 */
async function isDebugMode() {
    if (debugMode !== null) {
        return debugMode;
    }

    try {
        const debugEnv = process.env.DEBUG || await getVarVal('DEBUG') || 'false';
        const logLevel = await getLogLevel();
        
        // DEBUG=true이거나 LOG_LEVEL=DEBUG인 경우 디버그 모드
        debugMode = debugEnv.toLowerCase() === 'true' || 
                   ['1', 'yes', 'y'].includes(debugEnv.toLowerCase()) ||
                   logLevel >= LOG_LEVELS.DEBUG;
        
        return debugMode;
    } catch (error) {
        debugMode = false; // 기본값
        return debugMode;
    }
}

/**
 * 로그 레벨 캐시를 초기화합니다
 */
function resetLogLevel() {
    currentLogLevel = null;
    debugMode = null;
}

/**
 * 조건부 로그 출력 함수
 * @param {string} level - 로그 레벨 (ERROR, WARN, INFO, DEBUG)
 * @param {string|any} tagOrMessage - 태그 또는 메시지 (파라미터가 1개면 메시지로 처리)
 * @param {string|any} message - 메시지 또는 데이터
 * @param {any} data - 추가 데이터 (선택사항)
 */
async function log(level, tagOrMessage, message, data = null) {
    const logLevel = await getLogLevel();
    const requiredLevel = LOG_LEVELS[level.toUpperCase()];
    
    // 현재 설정된 로그 레벨보다 높은 레벨의 로그는 출력하지 않음
    if (requiredLevel > logLevel) {
        return;
    }
    
    // 파라미터 개수에 따라 처리 방식 결정
    if (arguments.length === 2) {
        // log(level, message) - 태그 없이 메시지만
        console.log(tagOrMessage);
    } else if (arguments.length === 3) {
        if (typeof tagOrMessage === 'string' && typeof message === 'string') {
            // log(level, tag, message) - 기존 방식
            console.log(`[${tagOrMessage}] ${message}`);
        } else {
            // log(level, message, data) - 메시지와 데이터
            console.log(tagOrMessage, message);
        }
    } else if (arguments.length === 4) {
        // log(level, tag, message, data) - 완전한 형태
        const logMessage = `[${tagOrMessage}] ${message}`;
        if (data) {
            console.log(logMessage, data);
        } else {
            console.log(logMessage);
        }
    }
}

/**
 * 에러 로그 (항상 출력)
 * 사용 방법:
 * - error('메시지')
 * - error('태그', '메시지')
 * - error('메시지', 데이터)
 * - error('태그', '메시지', 데이터)
 */
async function error(...args) {
    await log('ERROR', ...args);
}

/**
 * 경고 로그
 * 사용 방법:
 * - warn('메시지')
 * - warn('태그', '메시지')
 * - warn('메시지', 데이터)
 * - warn('태그', '메시지', 데이터)
 */
async function warn(...args) {
    await log('WARN', ...args);
}

/**
 * 정보 로그
 * 사용 방법:
 * - info('메시지')
 * - info('태그', '메시지')
 * - info('메시지', 데이터)
 * - info('태그', '메시지', 데이터)
 */
async function info(...args) {
    await log('INFO', ...args);
}

/**
 * 디버그 로그 (DEBUG 모드에서만 출력)
 * 사용 방법:
 * - debug('메시지')
 * - debug('태그', '메시지')
 * - debug('메시지', 데이터)
 * - debug('태그', '메시지', 데이터)
 */
async function debug(...args) {
    const isDebug = await isDebugMode();
    if (isDebug) {
        await log('DEBUG', ...args);
    }
}

/**
 * 동기식 디버그 로그 (기존 호환성을 위해 제공)
 */
function debugLog(tag, message, data = null) {
    // DEBUG 로그 완전 비활성화
    return;
}

/**
 * 동기식 로그 함수들 (console.log와 완전 호환)
 */
function syncLog(level, ...args) {
    // 환경변수를 동기적으로 확인
    const logLevelEnv = process.env.LOG_LEVEL || 'INFO';
    const currentLevel = LOG_LEVELS[logLevelEnv.toUpperCase()] ?? LOG_LEVELS.INFO;
    const requiredLevel = LOG_LEVELS[level.toUpperCase()];
    
    if (requiredLevel > currentLevel) {
        return;
    }
    
    // 파라미터 개수에 따라 처리
    if (args.length === 1) {
        console.log(args[0]);
    } else if (args.length === 2) {
        if (typeof args[0] === 'string' && typeof args[1] === 'string') {
            console.log(`[${args[0]}] ${args[1]}`);
        } else {
            console.log(args[0], args[1]);
        }
    } else if (args.length === 3) {
        const logMessage = `[${args[0]}] ${args[1]}`;
        console.log(logMessage, args[2]);
    } else {
        console.log(...args);
    }
}

// 동기식 버전 (console.log와 완전 호환)
const logger = {
    error: (...args) => syncLog('ERROR', ...args),
    warn: (...args) => syncLog('WARN', ...args),
    info: (...args) => syncLog('INFO', ...args),
    debug: (...args) => {
        // LOG_LEVEL에 따라서만 제어 (다른 로그들과 일관성 유지)
        syncLog('DEBUG', ...args);
    },
    log: (...args) => syncLog('INFO', ...args) // 기본적으로 INFO 레벨
};

/**
 * 성능 측정 로그 (항상 출력되는 중요 로그)
 */
async function performance(...args) {
    if (args.length === 1) {
        await info(`PERF`, args[0]);
    } else {
        await info(`PERF-${args[0]}`, ...args.slice(1));
    }
}

/**
 * 사용자 액션 로그 (항상 출력되는 중요 로그)
 */
async function userAction(...args) {
    if (args.length === 1) {
        await info(`USER`, args[0]);
    } else {
        await info(`USER-${args[0]}`, ...args.slice(1));
    }
}

export {
    log,
    error,
    warn,
    info,
    debug,
    debugLog,
    performance,
    userAction,
    isDebugMode,
    getLogLevel,
    resetLogLevel,
    LOG_LEVELS,
    logger,
    syncLog
}; 