// API 서버 기본 URL - 상대 경로 사용으로 프로토콜 자동 매칭
const API_BASE_URL = process.env.NEXT_PUBLIC_API_SERVER_URL || process.env.NEXT_PUBLIC_API_URL || '/api/v1';

// API 요청 타입 정의
type ApiMethod = 'GET' | 'POST' | 'PUT' | 'DELETE';

interface ApiRequestOptions {
  method?: ApiMethod;
  headers?: Record<string, string>;
  body?: any;
  params?: Record<string, string>;
}

// API 응답 타입 정의
interface ApiResponse<T = any> {
  success: boolean;
  data?: T;
  error?: {
    code?: string;
    message: string;
    details?: any;
  };
  timestamp?: string;
}

// API 클라이언트 클래스
class ApiClient {
  private apiKey: string = '';
  private baseUrl: string;

  constructor(baseUrl: string = API_BASE_URL) {
    this.baseUrl = baseUrl;
  }

  // API 키 설정
  setApiKey(apiKey: string): void {
    this.apiKey = apiKey;
  }

  // 브라우저 환경인지 확인
  private isBrowser(): boolean {
    return typeof window !== 'undefined';
  }

  // 로컬 스토리지에서 API 키 가져오기
  private getApiKeyFromStorage(): string | null {
    if (!this.isBrowser()) return null;
    return localStorage.getItem('apiKey');
  }

  // 기본 헤더 생성
  private getHeaders(): Record<string, string> {
    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
    };

    // 1. 직접 설정된 API 키 사용
    if (this.apiKey) {
      headers['X-API-Key'] = this.apiKey;
    } 
    // 2. 로컬 스토리지에서 API 키 가져오기
    else {
      const storageApiKey = this.getApiKeyFromStorage();
      if (storageApiKey) {
        headers['X-API-Key'] = storageApiKey;
      }
    }

    // 토큰도 함께 전송 (인증 토큰이 있는 경우)
    if (this.isBrowser()) {
      const token = localStorage.getItem('userToken');
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }
    }

    return headers;
  }

  // API 요청 함수
  async request<T = any>(
    endpoint: string,
    options: ApiRequestOptions = {}
  ): Promise<ApiResponse<T>> {
    const {
      method = 'GET',
      headers = {},
      body,
      params = {},
    } = options;

    // URL 매개변수 처리
    const queryParams = new URLSearchParams();
    Object.entries(params).forEach(([key, value]) => {
      queryParams.append(key, value);
    });

    const queryString = queryParams.toString();
    const url = `${this.baseUrl}${endpoint}${queryString ? `?${queryString}` : ''}`;

    try {
      const response = await fetch(url, {
        method,
        headers: {
          ...this.getHeaders(),
          ...headers,
        },
        body: body ? JSON.stringify(body) : undefined,
      });

      const data = await response.json();

      if (!response.ok) {
        return {
          success: false,
          error: {
            code: data.error?.code || `HTTP_ERROR_${response.status}`,
            message: data.error?.message || response.statusText,
            details: data.error?.details,
          },
          timestamp: new Date().toISOString(),
        };
      }

      return data as ApiResponse<T>;
    } catch (error) {
      return {
        success: false,
        error: {
          code: 'NETWORK_ERROR',
          message: error instanceof Error ? error.message : '네트워크 오류가 발생했습니다.',
        },
        timestamp: new Date().toISOString(),
      };
    }
  }

  // 편의 메서드: GET
  async get<T = any>(endpoint: string, params?: Record<string, string>): Promise<ApiResponse<T>> {
    return this.request<T>(endpoint, { method: 'GET', params });
  }

  // 편의 메서드: POST
  async post<T = any>(endpoint: string, body?: any): Promise<ApiResponse<T>> {
    return this.request<T>(endpoint, { method: 'POST', body });
  }

  // 편의 메서드: PUT
  async put<T = any>(endpoint: string, body?: any): Promise<ApiResponse<T>> {
    return this.request<T>(endpoint, { method: 'PUT', body });
  }

  // 편의 메서드: DELETE
  async delete<T = any>(endpoint: string): Promise<ApiResponse<T>> {
    return this.request<T>(endpoint, { method: 'DELETE' });
  }
}

// API 클라이언트 인스턴스 생성
const apiClient = new ApiClient();

export default apiClient;
export type { ApiResponse }; 