import apiClient, { ApiResponse } from './apiClient';
import { ChatMessage, ChatOptions } from './chatService';

// 코드 응답 인터페이스
export interface CodeResponse {
  id: string;
  object: string;
  created: number;
  model: string;
  provider: string;
  sessionId: string;
  userId: string;
  choices: {
    index: number;
    message: {
      role: string;
      content: string;
      userId: string;
    };
    finish_reason: string;
  }[];
  usage: {
    prompt_tokens: number;
    completion_tokens: number;
    total_tokens: number;
  };
  rag_enabled: boolean;
  web_search_enabled: boolean;
  execution_result: {
    code?: string;
    executionResult?: {
      output?: string;
      error?: string;
    };
  };
}

// 코드 요청 옵션 인터페이스
export interface CodeOptions extends ChatOptions {
  language?: string;
  hideCode?: boolean;
  autoExecute?: boolean;
}

// 코드 작업 상태 인터페이스
export interface CodeJobStatus {
  jobId: string;
  state: string;
  message: string;
  data?: CodeResponse;
  timestamp: string;
}

// 코드 서비스 클래스
class CodeService {
  // 코드 생성 요청 (비동기)
  async generateCode(prompt: string, options: CodeOptions = {}): Promise<ApiResponse<{ jobId: string }>> {
    return apiClient.post('/code', {
      prompt,
      options,
    });
  }

  // 코드 생성 요청 (동기)
  async generateCodeSync(prompt: string, options: CodeOptions = {}): Promise<ApiResponse<CodeResponse>> {
    return apiClient.post('/code/sync', {
      prompt,
      options,
    });
  }

  // 코드 작업 상태 확인
  async checkCodeStatus(jobId: string): Promise<ApiResponse<CodeJobStatus>> {
    return apiClient.get(`/code/status/${jobId}`);
  }

  // 코드 실행 결과만 가져오기
  async getExecutionResult(code: string, language: string = 'python'): Promise<ApiResponse<{
    output: string;
    error?: string;
  }>> {
    return apiClient.post('/code/execute', {
      code,
      language,
    });
  }
}

// 코드 서비스 인스턴스 생성
const codeService = new CodeService();

export default codeService; 