import { spawn } from 'child_process';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// MCP 서버 테스트 클라이언트
class MCPTestClient {
  constructor() {
    this.messageId = 1;
    this.serverProcess = null;
  }

  async startServer() {
    const serverPath = path.join(__dirname, '../dist/index.js');
    this.serverProcess = spawn('node', [serverPath], {
      stdio: ['pipe', 'pipe', 'pipe'],
      cwd: path.dirname(serverPath)
    });

    this.serverProcess.stderr.on('data', (data) => {
      console.error(`서버 에러: ${data}`);
    });

    // 서버 시작 대기
    await new Promise(resolve => setTimeout(resolve, 2000));
    console.log('✅ MCP 서버 시작됨');
  }

  async sendRequest(method, params = {}) {
    const request = {
      jsonrpc: '2.0',
      id: this.messageId++,
      method,
      params
    };

    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error('요청 타임아웃'));
      }, 30000);

      let buffer = '';
      const onData = (data) => {
        buffer += data.toString();
        
        while (true) {
          const newlineIndex = buffer.indexOf('\n');
          if (newlineIndex === -1) break;
          
          const line = buffer.slice(0, newlineIndex);
          buffer = buffer.slice(newlineIndex + 1);
          
          if (line.trim()) {
            try {
              const response = JSON.parse(line);
              if (response.id === request.id) {
                clearTimeout(timeout);
                this.serverProcess.stdout.off('data', onData);
                
                if (response.error) {
                  reject(new Error(response.error.message));
                } else {
                  resolve(response.result);
                }
                return;
              }
            } catch (e) {
              // JSON 파싱 오류 무시
            }
          }
        }
      };

      this.serverProcess.stdout.on('data', onData);
      this.serverProcess.stdin.write(JSON.stringify(request) + '\n');
    });
  }

  async testTranslateText() {
    console.log('\n🧪 translate_text 도구 테스트 시작...');
    
    const startTime = Date.now();
    
    try {
      const result = await this.sendRequest('tools/call', {
        name: 'translate_text',
        arguments: {
          text: 'Hello, this is a test message for translation.',
          target_lang: 'KO'
        }
      });
      
      const endTime = Date.now();
      const duration = endTime - startTime;
      
      console.log(`✅ translate_text 테스트 성공 (${duration}ms)`);
      
      if (result && result.content && result.content.length > 0) {
        console.log(`📝 번역 결과: ${result.content[0].text}`);
      } else {
        console.log('⚠️  예상치 못한 응답 형식:', JSON.stringify(result, null, 2));
      }
      
      return { success: true, duration };
      
    } catch (error) {
      const endTime = Date.now();
      const duration = endTime - startTime;
      
      console.log(`❌ translate_text 테스트 실패 (${duration}ms)`);
      console.log(`   오류: ${error.message}`);
      
      return { success: false, duration, error: error.message };
    }
  }

  async cleanup() {
    if (this.serverProcess) {
      this.serverProcess.kill();
      console.log('🔚 MCP 서버 종료됨');
    }
  }
}

// 테스트 실행
async function runTest() {
  const client = new MCPTestClient();
  
  try {
    await client.startServer();
    
    const result = await client.testTranslateText();
    
    console.log('\n📊 테스트 결과:');
    console.log(`상태: ${result.success ? '성공 ✅' : '실패 ❌'}`);
    console.log(`소요 시간: ${result.duration}ms`);
    
    if (!result.success) {
      console.log(`오류: ${result.error}`);
    }
    
  } catch (error) {
    console.error('테스트 실행 중 오류:', error);
  } finally {
    await client.cleanup();
  }
}

// 테스트 실행
runTest().catch(console.error); 