import { spawn } from 'child_process';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

// 디버깅 테스트 클라이언트
class DebugTestClient {
  constructor() {
    this.messageId = 1;
    this.serverProcess = null;
  }

  async startServer() {
    const serverPath = path.join(__dirname, '../dist/index.js');
    this.serverProcess = spawn('node', [serverPath], {
      stdio: ['pipe', 'pipe', 'pipe'],
      cwd: path.dirname(serverPath)
    });

    this.serverProcess.stderr.on('data', (data) => {
      console.error(`서버 로그: ${data}`);
    });

    // 서버 시작 대기
    await new Promise(resolve => setTimeout(resolve, 2000));
    console.log('✅ MCP 서버 시작됨');
  }

  async sendRequest(method, params = {}) {
    const request = {
      jsonrpc: '2.0',
      id: this.messageId++,
      method,
      params
    };

    console.log(`📤 요청 보내기: ${JSON.stringify(request, null, 2)}`);

    return new Promise((resolve, reject) => {
      const timeout = setTimeout(() => {
        reject(new Error('요청 타임아웃'));
      }, 10000);

      let buffer = '';
      const onData = (data) => {
        buffer += data.toString();
        console.log(`📥 서버 응답 버퍼: ${buffer}`);
        
        while (true) {
          const newlineIndex = buffer.indexOf('\n');
          if (newlineIndex === -1) break;
          
          const line = buffer.slice(0, newlineIndex);
          buffer = buffer.slice(newlineIndex + 1);
          
          if (line.trim()) {
            try {
              const response = JSON.parse(line);
              console.log(`📥 파싱된 응답: ${JSON.stringify(response, null, 2)}`);
              
              if (response.id === request.id) {
                clearTimeout(timeout);
                this.serverProcess.stdout.off('data', onData);
                
                if (response.error) {
                  reject(new Error(response.error.message));
                } else {
                  resolve(response.result);
                }
                return;
              }
            } catch (e) {
              console.log(`❌ JSON 파싱 오류: ${e.message}, 라인: ${line}`);
            }
          }
        }
      };

      this.serverProcess.stdout.on('data', onData);
      this.serverProcess.stdin.write(JSON.stringify(request) + '\n');
    });
  }

  async testValidation() {
    console.log('\n🔍 파라미터 검증 디버깅 테스트');
    
    // 1. 유효한 파라미터로 테스트
    console.log('\n📝 테스트 1: 유효한 파라미터');
    try {
      const result1 = await this.sendRequest('tools/call', {
        name: 'translate_text',
        arguments: {
          text: 'Hello world',
          target_lang: 'KO'
        }
      });
      console.log('✅ 유효한 파라미터 테스트 결과:', result1);
    } catch (error) {
      console.log('❌ 유효한 파라미터 테스트 실패:', error.message);
    }
    
    // 2. 무효한 파라미터로 테스트
    console.log('\n📝 테스트 2: 무효한 파라미터 (text 누락)');
    try {
      const result2 = await this.sendRequest('tools/call', {
        name: 'translate_text',
        arguments: {
          target_lang: 'KO'
        }
      });
      console.log('⚠️ 무효한 파라미터 테스트 결과:', result2);
    } catch (error) {
      console.log('❌ 무효한 파라미터 테스트 실패:', error.message);
    }
    
    // 3. 빈 파라미터로 테스트
    console.log('\n📝 테스트 3: 빈 파라미터');
    try {
      const result3 = await this.sendRequest('tools/call', {
        name: 'translate_text',
        arguments: {}
      });
      console.log('⚠️ 빈 파라미터 테스트 결과:', result3);
    } catch (error) {
      console.log('❌ 빈 파라미터 테스트 실패:', error.message);
    }
  }

  async cleanup() {
    if (this.serverProcess) {
      this.serverProcess.kill();
      console.log('\n🔚 MCP 서버 종료됨');
    }
  }
}

// 디버깅 테스트 실행
async function runDebugTest() {
  const client = new DebugTestClient();
  
  try {
    await client.startServer();
    await client.testValidation();
    
  } catch (error) {
    console.error('디버깅 테스트 실행 중 오류:', error);
  } finally {
    await client.cleanup();
  }
}

// 테스트 실행
runDebugTest().catch(console.error); 