#!/bin/bash

# 더미 서버 관리 스크립트
# 사용법: ./run.sh [start|stop|restart|status|logs]

SERVER_NAME="poc_dummy_server"
SERVER_PORT=4000
SERVER_FILE="server.js"
PID_FILE="/tmp/${SERVER_NAME}.pid"
LOG_FILE="/tmp/${SERVER_NAME}.log"

# 색상 정의
RED='\033[0;31m'
GREEN='\033[0;32m'
YELLOW='\033[1;33m'
BLUE='\033[0;34m'
NC='\033[0m' # No Color

# 로그 함수
log_info() {
    echo -e "${BLUE}[INFO]${NC} $1"
}

log_success() {
    echo -e "${GREEN}[SUCCESS]${NC} $1"
}

log_warning() {
    echo -e "${YELLOW}[WARNING]${NC} $1"
}

log_error() {
    echo -e "${RED}[ERROR]${NC} $1"
}

# 서버 상태 확인
check_status() {
    if [ -f "$PID_FILE" ]; then
        PID=$(cat "$PID_FILE")
        if ps -p "$PID" > /dev/null 2>&1; then
            return 0  # 실행 중
        else
            rm -f "$PID_FILE"  # 죽은 PID 파일 제거
            return 1  # 실행 중이 아님
        fi
    else
        return 1  # PID 파일 없음
    fi
}

# 포트 사용 중인지 확인
check_port() {
    if lsof -Pi :$SERVER_PORT -sTCP:LISTEN -t >/dev/null ; then
        return 0  # 포트 사용 중
    else
        return 1  # 포트 사용 안 함
    fi
}

# 서버 시작
start_server() {
    log_info "더미 서버 시작 중..."
    
    # 이미 실행 중인지 확인
    if check_status; then
        PID=$(cat "$PID_FILE")
        log_warning "서버가 이미 실행 중입니다 (PID: $PID)"
        return 1
    fi
    
    # 포트가 사용 중인지 확인
    if check_port; then
        log_error "포트 $SERVER_PORT이 이미 사용 중입니다"
        log_info "다른 프로세스를 확인하세요: lsof -i :$SERVER_PORT"
        return 1
    fi
    
    # Node.js가 설치되어 있는지 확인
    if ! command -v node &> /dev/null; then
        log_error "Node.js가 설치되어 있지 않습니다"
        return 1
    fi
    
    # 서버 파일 존재 확인
    if [ ! -f "$SERVER_FILE" ]; then
        log_error "서버 파일 '$SERVER_FILE'을 찾을 수 없습니다"
        return 1
    fi
    
    # 백그라운드에서 서버 시작
    nohup node "$SERVER_FILE" > "$LOG_FILE" 2>&1 &
    SERVER_PID=$!
    
    # PID 저장
    echo $SERVER_PID > "$PID_FILE"
    
    # 시작 확인 (2초 대기)
    sleep 2
    
    if check_status; then
        log_success "더미 서버가 성공적으로 시작되었습니다"
        log_info "PID: $SERVER_PID"
        log_info "포트: $SERVER_PORT"
        log_info "로그 파일: $LOG_FILE"
        
        # 서버 응답 테스트
        if curl -s "http://localhost:$SERVER_PORT/api/complaints" > /dev/null; then
            log_success "서버 응답 테스트 성공"
        else
            log_warning "서버가 시작되었지만 응답 테스트에 실패했습니다"
        fi
    else
        log_error "서버 시작에 실패했습니다"
        log_info "로그를 확인하세요: tail -f $LOG_FILE"
        return 1
    fi
}

# 서버 중지
stop_server() {
    log_info "더미 서버 중지 중..."
    
    if check_status; then
        PID=$(cat "$PID_FILE")
        log_info "서버 프로세스 종료 중 (PID: $PID)..."
        
        # SIGTERM으로 정상 종료 시도
        kill "$PID"
        
        # 5초 대기
        for i in {1..5}; do
            if ! ps -p "$PID" > /dev/null 2>&1; then
                break
            fi
            sleep 1
        done
        
        # 여전히 실행 중이면 강제 종료
        if ps -p "$PID" > /dev/null 2>&1; then
            log_warning "정상 종료 실패, 강제 종료 중..."
            kill -9 "$PID"
            sleep 1
        fi
        
        # PID 파일 제거
        rm -f "$PID_FILE"
        
        if ! ps -p "$PID" > /dev/null 2>&1; then
            log_success "더미 서버가 성공적으로 중지되었습니다"
        else
            log_error "서버 중지에 실패했습니다"
            return 1
        fi
    else
        log_warning "실행 중인 서버가 없습니다"
    fi
    
    # 포트에서 실행 중인 다른 프로세스 확인
    if check_port; then
        log_warning "포트 $SERVER_PORT에서 다른 프로세스가 실행 중입니다"
        log_info "확인하세요: lsof -i :$SERVER_PORT"
    fi
}

# 서버 재시작
restart_server() {
    log_info "더미 서버 재시작 중..."
    stop_server
    sleep 2
    start_server
}

# 서버 상태 표시
show_status() {
    echo "=== 더미 서버 상태 ==="
    
    if check_status; then
        PID=$(cat "$PID_FILE")
        log_success "서버 실행 중 (PID: $PID)"
        
        # 프로세스 정보
        if ps -p "$PID" -o pid,ppid,cmd,etime > /dev/null 2>&1; then
            echo "프로세스 정보:"
            ps -p "$PID" -o pid,ppid,cmd,etime
        fi
        
        # 포트 확인
        if check_port; then
            log_success "포트 $SERVER_PORT 사용 중"
        else
            log_warning "포트 $SERVER_PORT 사용 안 함"
        fi
        
        # 서버 응답 테스트
        if curl -s "http://localhost:$SERVER_PORT/api/complaints" > /dev/null; then
            log_success "서버 응답 정상"
        else
            log_error "서버 응답 없음"
        fi
        
    else
        log_warning "서버 실행 중이 아님"
        
        # 포트 확인
        if check_port; then
            log_warning "포트 $SERVER_PORT에서 다른 프로세스 실행 중"
            echo "포트 사용 프로세스:"
            lsof -i :$SERVER_PORT
        fi
    fi
    
    echo "로그 파일: $LOG_FILE"
    echo "PID 파일: $PID_FILE"
}

# 로그 보기
show_logs() {
    if [ -f "$LOG_FILE" ]; then
        log_info "로그 파일 내용 (마지막 50줄):"
        tail -50 "$LOG_FILE"
        echo ""
        log_info "실시간 로그를 보려면: tail -f $LOG_FILE"
    else
        log_warning "로그 파일이 없습니다: $LOG_FILE"
    fi
}

# 도움말
show_help() {
    echo "더미 서버 관리 스크립트"
    echo ""
    echo "사용법: $0 [COMMAND]"
    echo ""
    echo "COMMANDS:"
    echo "  start     서버 시작"
    echo "  stop      서버 중지"
    echo "  restart   서버 재시작"
    echo "  status    서버 상태 확인"
    echo "  logs      로그 보기"
    echo "  help      도움말 보기"
    echo ""
    echo "예시:"
    echo "  $0 start    # 서버 시작"
    echo "  $0 status   # 상태 확인"
    echo "  $0 logs     # 로그 보기"
    echo "  $0 stop     # 서버 중지"
}

# 메인 로직
case "$1" in
    start)
        start_server
        ;;
    stop)
        stop_server
        ;;
    restart)
        restart_server
        ;;
    status)
        show_status
        ;;
    logs)
        show_logs
        ;;
    help|--help|-h)
        show_help
        ;;
    "")
        log_error "명령어를 지정해주세요"
        show_help
        exit 1
        ;;
    *)
        log_error "알 수 없는 명령어: $1"
        show_help
        exit 1
        ;;
esac 