import React, { useState, useEffect } from 'react';

interface UserStatsProps {
  userId: string;
  username: string;
  onClose: () => void;
}

interface UserStats {
  aiUsage: {
    totalCost: number;
    totalTokens: number;
    totalRequests: number;
    providerStats: Record<string, { cost: number; tokens: number; requests: number }>;
    modelStats: Record<string, { cost: number; tokens: number; requests: number }>;
    dailyStats: Record<string, { cost: number; tokens: number; requests: number }>;
  };
  apiUsage: {
    totalCalls: number;
    successCalls: number;
    errorCalls: number;
    avgResponseTime: number;
    endpointStats: Record<string, { calls: number; errors: number; totalResponseTime: number; responseCount: number }>;
  };
  recentActivity: Array<{
    type: string;
    timestamp: string;
    [key: string]: any;
  }>;
}

const UserStats: React.FC<UserStatsProps> = ({ userId, username, onClose }) => {
  const [stats, setStats] = useState<UserStats | null>(null);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [activeTab, setActiveTab] = useState<'ai' | 'api' | 'activity'>('ai');

  useEffect(() => {
    const fetchStats = async () => {
      try {
        setLoading(true);
        setError(null);
        
        const response = await fetch(`/api/admin/users/${userId}/stats`);
        const data = await response.json();
        
        if (data.success) {
          setStats(data.data);
        } else {
          setError(data.error?.message || '통계 조회 실패');
        }
      } catch (err) {
        setError('통계 조회 중 오류가 발생했습니다.');
        console.error('Stats fetch error:', err);
      } finally {
        setLoading(false);
      }
    };

    fetchStats();
  }, [userId]);

  const formatNumber = (num: number) => {
    return new Intl.NumberFormat('ko-KR').format(num);
  };

  const formatCurrency = (amount: number) => {
    return new Intl.NumberFormat('ko-KR', {
      style: 'currency',
      currency: 'USD',
      minimumFractionDigits: 4,
      maximumFractionDigits: 4
    }).format(amount);
  };

  if (loading) {
    return (
      <div className="mt-6 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-6">
        <div className="flex justify-between items-center mb-4">
          <p className="text-base font-bold">사용자 통계: {username}</p>
          <button
            onClick={onClose}
            className="text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
          >
            ✕
          </button>
        </div>
        <div className="flex items-center justify-center py-8">
          <div className="text-center">
            <div className="w-8 h-8 border-4 border-blue-500 border-t-transparent rounded-full animate-spin mx-auto mb-2"></div>
            <p className="text-gray-600 dark:text-gray-400">통계를 불러오는 중...</p>
          </div>
        </div>
      </div>
    );
  }

  if (error) {
    return (
      <div className="mt-6 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-6">
        <div className="flex justify-between items-center mb-4">
          <p className="text-base font-bold">사용자 통계: {username}</p>
          <button
            onClick={onClose}
            className="text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200"
          >
            ✕
          </button>
        </div>
        <div className="text-center py-8">
          <p className="text-red-600 dark:text-red-400">{error}</p>
        </div>
      </div>
    );
  }

  if (!stats) return null;

  return (
    <div className="mt-6 bg-white dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700 p-6">
      <div className="flex justify-between items-center mb-6">
        <p className="text-base font-bold">사용자 통계: {username}</p>
        <button
          onClick={onClose}
          className="text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 text-xl"
        >
          ✕
        </button>
      </div>

      {/* 탭 메뉴 */}
      <div className="flex space-x-4 mb-6 border-b border-gray-200 dark:border-gray-700">
        <button
          className={`py-2 px-4 font-medium ${
            activeTab === 'ai'
              ? 'text-blue-600 border-b-2 border-blue-600'
              : 'text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200'
          }`}
          onClick={() => setActiveTab('ai')}
        >
          AI 사용통계
        </button>
        <button
          className={`py-2 px-4 font-medium ${
            activeTab === 'api'
              ? 'text-blue-600 border-b-2 border-blue-600'
              : 'text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200'
          }`}
          onClick={() => setActiveTab('api')}
        >
          API 사용통계
        </button>
        <button
          className={`py-2 px-4 font-medium ${
            activeTab === 'activity'
              ? 'text-blue-600 border-b-2 border-blue-600'
              : 'text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200'
          }`}
          onClick={() => setActiveTab('activity')}
        >
          최근 활동
        </button>
      </div>

      {/* 탭 내용 영역 - 스크롤 가능 */}
      <div className="max-h-[600px] overflow-y-auto pr-2">

        {/* AI 사용통계 탭 */}
        {activeTab === 'ai' && (
          <div className="space-y-6">
            {/* 전체 통계 */}
            <div className="grid grid-cols-1 md:grid-cols-3 gap-4">
              <div className="bg-blue-50 dark:bg-blue-900/20 p-4 rounded-lg">
                <p className="text-base font-semibold text-blue-700 dark:text-blue-300 mb-2">총 비용</p>
                <p className="text-base font-bold text-blue-800 dark:text-blue-200">
                  {formatCurrency(stats.aiUsage.totalCost)}
                </p>
              </div>
              <div className="bg-green-50 dark:bg-green-900/20 p-4 rounded-lg">
                <p className="text-base font-semibold text-green-700 dark:text-green-300 mb-2">총 토큰</p>
                <p className="text-base font-bold text-green-800 dark:text-green-200">
                  {formatNumber(stats.aiUsage.totalTokens)}
                </p>
              </div>
              <div className="bg-purple-50 dark:bg-purple-900/20 p-4 rounded-lg">
                <p className="text-base font-semibold text-purple-700 dark:text-purple-300 mb-2">총 요청</p>
                <p className="text-base font-bold text-purple-800 dark:text-purple-200">
                  {formatNumber(stats.aiUsage.totalRequests)}
                </p>
              </div>
            </div>

            {/* 프로바이더별 통계 */}
            <div>
              <p className="text-base font-semibold mb-3">프로바이더별 사용량</p>
              <div className="bg-gray-50 dark:bg-gray-900/50 rounded-lg p-4">
                {Object.entries(stats.aiUsage.providerStats).length > 0 ? (
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                    {Object.entries(stats.aiUsage.providerStats).map(([provider, stat]) => (
                      <div key={provider} className="bg-white dark:bg-gray-800 p-3 rounded border">
                        <p className="font-medium mb-2">{provider}</p>
                        <div className="text-sm space-y-1">
                          <p>비용: {formatCurrency(stat.cost)}</p>
                          <p>토큰: {formatNumber(stat.tokens)}</p>
                          <p>요청: {formatNumber(stat.requests)}</p>
                        </div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <p className="text-gray-500 dark:text-gray-400">사용 데이터가 없습니다.</p>
                )}
              </div>
            </div>

            {/* 모델별 통계 */}
            <div>
              <p className="text-base font-semibold mb-3">모델별 사용량</p>
              <div className="bg-gray-50 dark:bg-gray-900/50 rounded-lg p-4">
                {Object.entries(stats.aiUsage.modelStats).length > 0 ? (
                  <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4">
                    {Object.entries(stats.aiUsage.modelStats).map(([model, stat]) => (
                      <div key={model} className="bg-white dark:bg-gray-800 p-3 rounded border">
                        <p className="font-medium mb-2">{model}</p>
                        <div className="text-sm space-y-1">
                          <p>비용: {formatCurrency(stat.cost)}</p>
                          <p>토큰: {formatNumber(stat.tokens)}</p>
                          <p>요청: {formatNumber(stat.requests)}</p>
                        </div>
                      </div>
                    ))}
                  </div>
                ) : (
                  <p className="text-gray-500 dark:text-gray-400">사용 데이터가 없습니다.</p>
                )}
              </div>
            </div>
          </div>
        )}

        {/* API 사용통계 탭 */}
        {activeTab === 'api' && (
          <div className="space-y-6">
            {/* 전체 API 통계 */}
            <div className="grid grid-cols-1 md:grid-cols-4 gap-4">
              <div className="bg-blue-50 dark:bg-blue-900/20 p-4 rounded-lg">
                <p className="text-base font-semibold text-blue-700 dark:text-blue-300 mb-2">총 호출</p>
                <p className="text-base font-bold text-blue-800 dark:text-blue-200">
                  {formatNumber(stats.apiUsage.totalCalls)}
                </p>
              </div>
              <div className="bg-green-50 dark:bg-green-900/20 p-4 rounded-lg">
                <p className="text-base font-semibold text-green-700 dark:text-green-300 mb-2">성공</p>
                <p className="text-base font-bold text-green-800 dark:text-green-200">
                  {formatNumber(stats.apiUsage.successCalls)}
                </p>
              </div>
              <div className="bg-red-50 dark:bg-red-900/20 p-4 rounded-lg">
                <p className="text-base font-semibold text-red-700 dark:text-red-300 mb-2">오류</p>
                <p className="text-base font-bold text-red-800 dark:text-red-200">
                  {formatNumber(stats.apiUsage.errorCalls)}
                </p>
              </div>
              <div className="bg-yellow-50 dark:bg-yellow-900/20 p-4 rounded-lg">
                <p className="text-base font-semibold text-yellow-700 dark:text-yellow-300 mb-2">평균 응답시간</p>
                <p className="text-base font-bold text-yellow-800 dark:text-yellow-200">
                  {Math.round(stats.apiUsage.avgResponseTime)}ms
                </p>
              </div>
            </div>

            {/* 엔드포인트별 통계 */}
            <div>
              <p className="text-base font-semibold mb-3">엔드포인트별 사용량</p>
              <div className="bg-gray-50 dark:bg-gray-900/50 rounded-lg p-4">
                {Object.entries(stats.apiUsage.endpointStats).length > 0 ? (
                  <div className="overflow-x-auto">
                    <table className="w-full text-sm">
                      <thead>
                        <tr className="border-b border-gray-200 dark:border-gray-700">
                          <th className="text-left py-2">엔드포인트</th>
                          <th className="text-right py-2">호출수</th>
                          <th className="text-right py-2">오류수</th>
                          <th className="text-right py-2">평균 응답시간</th>
                        </tr>
                      </thead>
                      <tbody>
                        {Object.entries(stats.apiUsage.endpointStats).map(([endpoint, stat]) => (
                          <tr key={endpoint} className="border-b border-gray-100 dark:border-gray-800">
                            <td className="py-2 font-mono text-xs">{endpoint}</td>
                            <td className="text-right py-2">{formatNumber(stat.calls)}</td>
                            <td className="text-right py-2">{formatNumber(stat.errors)}</td>
                            <td className="text-right py-2">
                              {stat.responseCount > 0 
                                ? Math.round(stat.totalResponseTime / stat.responseCount) + 'ms'
                                : 'N/A'
                              }
                            </td>
                          </tr>
                        ))}
                      </tbody>
                    </table>
                  </div>
                ) : (
                  <p className="text-gray-500 dark:text-gray-400">API 호출 데이터가 없습니다.</p>
                )}
              </div>
            </div>
          </div>
        )}

        {/* 최근 활동 탭 */}
        {activeTab === 'activity' && (
          <div>
            <p className="text-base font-semibold mb-3">최근 활동</p>
            <div className="bg-gray-50 dark:bg-gray-900/50 rounded-lg p-4">
              {stats.recentActivity.length > 0 ? (
                <div className="space-y-3">
                  {stats.recentActivity.map((activity, index) => (
                    <div key={index} className="bg-white dark:bg-gray-800 p-3 rounded border">
                      <div className="flex justify-between items-start">
                        <div>
                          <span className={`inline-block px-2 py-1 rounded text-xs font-medium ${
                            activity.type === 'ai_usage' 
                              ? 'bg-blue-100 text-blue-800 dark:bg-blue-900/50 dark:text-blue-300'
                              : 'bg-green-100 text-green-800 dark:bg-green-900/50 dark:text-green-300'
                          }`}>
                            {activity.type === 'ai_usage' ? 'AI 사용' : 'API 호출'}
                          </span>
                          <div className="mt-2 text-sm">
                            {activity.type === 'ai_usage' ? (
                              <div>
                                <p><strong>프로바이더:</strong> {activity.provider}</p>
                                <p><strong>모델:</strong> {activity.model}</p>
                                <p><strong>비용:</strong> {formatCurrency(activity.cost)}</p>
                                <p><strong>토큰:</strong> {formatNumber(activity.tokens)}</p>
                              </div>
                            ) : (
                              <div>
                                <p><strong>엔드포인트:</strong> {activity.endpoint}</p>
                                <p><strong>메서드:</strong> {activity.method}</p>
                                <p><strong>상태:</strong> {activity.status}</p>
                                <p><strong>응답시간:</strong> {activity.responseTime}ms</p>
                              </div>
                            )}
                          </div>
                        </div>
                        <div className="text-xs text-gray-500 dark:text-gray-400">
                          {new Date(activity.timestamp).toLocaleString('ko-KR')}
                        </div>
                      </div>
                    </div>
                  ))}
                </div>
              ) : (
                <p className="text-gray-500 dark:text-gray-400">최근 활동이 없습니다.</p>
              )}
            </div>
          </div>
        )}
        </div> {/* 탭 내용 영역 닫기 */}
    </div>
  );
};

export default UserStats; 