import { Pool } from 'pg';
import dbConfig from './dbconfig';

export const pg = new Pool(dbConfig);

// 초기화 상태를 추적하는 변수
let isInitializing = false;
let initializationPromise: Promise<void> | null = null;

export const initSupportTicketsTable = async () => {
  // 이미 초기화 중이거나 완료된 경우 기존 Promise를 반환
  if (isInitializing && initializationPromise) {
    return initializationPromise;
  }

  // 초기화 시작
  isInitializing = true;
  initializationPromise = performInitialization();
  
  try {
    await initializationPromise;
  } finally {
    isInitializing = false;
  }
};

async function performInitialization() {
  const client = await pg.connect();
  try {
    // 테이블 생성 전에 Advisory Lock 사용하여 동시 실행 방지
    await client.query('SELECT pg_advisory_lock(12345)');
    
    // Create tickets table if it doesn't exist
    await client.query(`
      CREATE TABLE IF NOT EXISTS support_tickets (
        id SERIAL PRIMARY KEY,
        title VARCHAR(255) NOT NULL,
        question TEXT NOT NULL,
        status VARCHAR(20) NOT NULL DEFAULT 'open',
        created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
        author_name VARCHAR(255) NOT NULL,
        author_role VARCHAR(20) NOT NULL,
        attachment_image BYTEA,
        attachment_image_mimetype VARCHAR(100)
      );
    `);

    // Create replies table if it doesn't exist
    await client.query(`
      CREATE TABLE IF NOT EXISTS support_ticket_replies (
        id SERIAL PRIMARY KEY,
        ticket_id INTEGER NOT NULL REFERENCES support_tickets(id) ON DELETE CASCADE,
        content TEXT NOT NULL,
        author_name VARCHAR(255) NOT NULL,
        author_role VARCHAR(20) NOT NULL,
        created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
        images BYTEA[],
        image_mimetypes VARCHAR(100)[],
        CONSTRAINT fk_ticket
          FOREIGN KEY(ticket_id) 
          REFERENCES support_tickets(id)
          ON DELETE CASCADE
      );
    `);

    // Create documents table if it doesn't exist
    await client.query(`
      CREATE TABLE IF NOT EXISTS support_documents (
        id SERIAL PRIMARY KEY,
        filename VARCHAR(255) NOT NULL,
        filepath TEXT NOT NULL,
        filesize BIGINT,
        mimetype VARCHAR(100),
        user_id VARCHAR(255),
        created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // Create chat documents table if it doesn't exist
    await client.query(`
      CREATE TABLE IF NOT EXISTS chat_documents (
        id SERIAL PRIMARY KEY,
        filename VARCHAR(255) NOT NULL,
        filepath VARCHAR(500) NOT NULL,
        filesize BIGINT,
        mimetype VARCHAR(100),
        user_id VARCHAR(100) NOT NULL,
        created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // Create indexes for chat_documents (각각 별도로 실행하여 오류 방지)
    try {
      await client.query(`CREATE INDEX IF NOT EXISTS idx_chat_documents_user_id ON chat_documents(user_id);`);
    } catch (err: any) {
      // 인덱스가 이미 존재하는 경우 무시
      console.log('Index idx_chat_documents_user_id already exists or creation failed:', err.message);
    }
    
    try {
      await client.query(`CREATE INDEX IF NOT EXISTS idx_chat_documents_created_at ON chat_documents(created_at);`);
    } catch (err: any) {
      // 인덱스가 이미 존재하는 경우 무시
      console.log('Index idx_chat_documents_created_at already exists or creation failed:', err.message);
    }

    // Create memories table if it doesn't exist
    await client.query(`
      CREATE TABLE IF NOT EXISTS user_memories (
        id SERIAL PRIMARY KEY,
        user_id VARCHAR(255) NOT NULL,
        title VARCHAR(255) NOT NULL,
        content TEXT NOT NULL,
        category VARCHAR(100) DEFAULT 'general',
        importance_level INTEGER DEFAULT 1 CHECK (importance_level BETWEEN 1 AND 5),
        tags TEXT[],
        created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
        last_accessed TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
        access_count INTEGER DEFAULT 0,
        is_active BOOLEAN DEFAULT true
      );
    `);

    // Create indexes for user_memories (각각 별도로 실행)
    const memoryIndexes = [
      'CREATE INDEX IF NOT EXISTS idx_user_memories_user_id ON user_memories(user_id);',
      'CREATE INDEX IF NOT EXISTS idx_user_memories_category ON user_memories(category);',
      'CREATE INDEX IF NOT EXISTS idx_user_memories_importance ON user_memories(importance_level);',
      'CREATE INDEX IF NOT EXISTS idx_user_memories_active ON user_memories(is_active);'
    ];

          for (const indexQuery of memoryIndexes) {
        try {
          await client.query(indexQuery);
        } catch (err: any) {
          console.log('Index creation failed or already exists:', err.message);
        }
      }

    // Add updated_at column if it doesn't exist
    const columns = await client.query(`
      SELECT column_name
      FROM information_schema.columns
      WHERE table_name = 'support_tickets';
    `);
    const columnNames = columns.rows.map(row => row.column_name);

    if (!columnNames.includes('updated_at')) {
      await client.query(`
        ALTER TABLE support_tickets 
        ADD COLUMN updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP;
      `);
    }

    // Add images and image_mimetypes columns to replies table if they don't exist
    const replyColumns = await client.query(`
      SELECT column_name
      FROM information_schema.columns
      WHERE table_name = 'support_ticket_replies';
    `);
    const replyColumnNames = replyColumns.rows.map(row => row.column_name);

    if (!replyColumnNames.includes('images')) {
      await client.query(`
        ALTER TABLE support_ticket_replies 
        ADD COLUMN images BYTEA[] DEFAULT '{}';
      `);
    }

    if (!replyColumnNames.includes('image_mimetypes')) {
      await client.query(`
        ALTER TABLE support_ticket_replies 
        ADD COLUMN image_mimetypes VARCHAR(100)[] DEFAULT '{}';
      `);
    }

    // FlowAI 테이블 생성
    await client.query(`
      CREATE TABLE IF NOT EXISTS flowai_workflows (
        id VARCHAR(255) PRIMARY KEY,
        name VARCHAR(255) NOT NULL,
        description TEXT,
        nodes TEXT NOT NULL DEFAULT '[]',
        connections TEXT NOT NULL DEFAULT '[]',
        variables TEXT DEFAULT '{}',
        user_id VARCHAR(255) NOT NULL,
        is_public BOOLEAN DEFAULT false,
        tags TEXT DEFAULT '[]',
        created_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
        updated_at TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP
      );
    `);

    // FlowAI 실행 기록 테이블 생성
    await client.query(`
      CREATE TABLE IF NOT EXISTS flowai_executions (
        id SERIAL PRIMARY KEY,
        workflow_id VARCHAR(255) NOT NULL REFERENCES flowai_workflows(id) ON DELETE CASCADE,
        user_id VARCHAR(255) NOT NULL,
        status VARCHAR(20) NOT NULL DEFAULT 'pending',
        results TEXT DEFAULT '[]',
        start_time TIMESTAMP WITH TIME ZONE DEFAULT CURRENT_TIMESTAMP,
        end_time TIMESTAMP WITH TIME ZONE,
        total_duration INTEGER,
        error_message TEXT,
        final_output TEXT
      );
    `);

    // FlowAI 인덱스 생성
    const flowaIndexes = [
      'CREATE INDEX IF NOT EXISTS idx_flowai_workflows_user_id ON flowai_workflows(user_id);',
      'CREATE INDEX IF NOT EXISTS idx_flowai_workflows_public ON flowai_workflows(is_public);',
      'CREATE INDEX IF NOT EXISTS idx_flowai_workflows_updated ON flowai_workflows(updated_at);',
      'CREATE INDEX IF NOT EXISTS idx_flowai_executions_workflow ON flowai_executions(workflow_id);',
      'CREATE INDEX IF NOT EXISTS idx_flowai_executions_user ON flowai_executions(user_id);',
      'CREATE INDEX IF NOT EXISTS idx_flowai_executions_status ON flowai_executions(status);'
    ];

    for (const indexQuery of flowaIndexes) {
      try {
        await client.query(indexQuery);
      } catch (err: any) {
        console.log('FlowAI index creation failed or already exists:', err.message);
      }
    }

    console.log('Support tables and FlowAI tables are up to date.');
  } catch (err) {
    console.error('Error initializing support tables:', err);
    throw err; // 에러를 다시 던져서 호출자가 알 수 있도록 함
  } finally {
    try {
      // Advisory Lock 해제
      await client.query('SELECT pg_advisory_unlock(12345)');
    } catch (unlockErr) {
      console.error('Error releasing advisory lock:', unlockErr);
    }
    client.release();
  }
} 