import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../middlewares';
import { Pool } from 'pg';

// 데이터베이스 연결 설정
const pool = new Pool({
  host: process.env.DB_HOST || process.env.POSTGRES_HOST || 'localhost',
  port: parseInt(process.env.DB_PORT || process.env.POSTGRES_PORT || '5433'),
  database: process.env.DB_NAME || process.env.POSTGRES_DB || 'airun',
  user: process.env.DB_USER || process.env.POSTGRES_USER || 'ivs',
  password: process.env.DB_PASSWORD || process.env.POSTGRES_PASSWORD || '1234',
});

// 문서 목록 조회 API
export async function GET(req: NextRequest) {
  return withAnyAuth(req, async (req, user, authType) => {
    try {
      const { searchParams } = new URL(req.url);
      const page = parseInt(searchParams.get('page') || '1');
      const limit = parseInt(searchParams.get('limit') || '50');
      const offset = (page - 1) * limit;
      const userId = searchParams.get('userId');
      const projectId = searchParams.get('projectId');
      
      // 문서 조회 쿼리 구성
      let query = `
        SELECT 
          id, filename, filepath, filesize, mimetype, user_id,
          upload_status, embedding_status, project_name, tags, metadata,
          error_message, processed_at, created_at, updated_at
        FROM chat_documents
      `;
      
      const queryParams: any[] = [];
      const whereConditions: string[] = [];
      
      // 사용자별 필터링
      if (userId) {
        // 특정 사용자의 문서 조회 (관리자만 가능)
        if (user.role === 'admin') {
          whereConditions.push(`user_id = $${queryParams.length + 1}`);
          queryParams.push(userId);
        } else if (userId === (user.username || user.id)) {
          // 자신의 문서만 조회
          whereConditions.push(`user_id = $${queryParams.length + 1}`);
          queryParams.push(userId);
        } else {
          // 권한 없음
          return NextResponse.json(
            {
              success: false,
              error: {
                code: 'FORBIDDEN',
                message: '해당 사용자의 문서에 접근할 권한이 없습니다.'
              }
            },
            { status: 403 }
          );
        }
      } else if (projectId) {
        // 프로젝트별 필터링 (projectId는 실제로 project_name)
        const projectName = decodeURIComponent(projectId);
        
        // 프로젝트에 문서가 있는지 확인
        const projectCheckResult = await pool.query(
          'SELECT DISTINCT user_id FROM chat_documents WHERE project_name = $1',
          [projectName]
        );
        
        if (projectCheckResult.rows.length === 0) {
          return NextResponse.json(
            {
              success: false,
              error: {
                code: 'NOT_FOUND',
                message: '프로젝트를 찾을 수 없습니다.'
              }
            },
            { status: 404 }
          );
        }
        
        const projectUserId = projectCheckResult.rows[0].user_id;
        
        // 프로젝트 접근 권한 확인
        if (user.role !== 'admin' && projectUserId !== (user.username || user.id)) {
          return NextResponse.json(
            {
              success: false,
              error: {
                code: 'FORBIDDEN',
                message: '해당 프로젝트에 접근할 권한이 없습니다.'
              }
            },
            { status: 403 }
          );
        }
        
        whereConditions.push(`project_name = $${queryParams.length + 1}`);
        queryParams.push(projectName);
      } else {
        // 기본: 일반 사용자는 자신의 문서만 조회
        if (user.role !== 'admin') {
          whereConditions.push(`user_id = $${queryParams.length + 1}`);
          queryParams.push(user.username || user.id);
        }
      }
      
      // WHERE 조건 추가
      if (whereConditions.length > 0) {
        query += ' WHERE ' + whereConditions.join(' AND ');
      }
      
      // 정렬 및 페이징
      query += ' ORDER BY created_at DESC LIMIT $' + (queryParams.length + 1) + ' OFFSET $' + (queryParams.length + 2);
      queryParams.push(limit, offset);
      
      const result = await pool.query(query, queryParams);
      
      // 전체 개수 조회 (동일한 조건 적용)
      let countQuery = 'SELECT COUNT(*) FROM chat_documents';
      const countParams: any[] = [];
      
      if (whereConditions.length > 0) {
        countQuery += ' WHERE ' + whereConditions.join(' AND ');
        countParams.push(...queryParams.slice(0, queryParams.length - 2)); // LIMIT, OFFSET 제외
      }
      
      const countResult = await pool.query(countQuery, countParams);
      const totalCount = parseInt(countResult.rows[0].count);
      
      return NextResponse.json({
        success: true,
        data: result.rows,
        pagination: {
          page,
          limit,
          total: totalCount,
          totalPages: Math.ceil(totalCount / limit)
        },
        timestamp: new Date().toISOString()
      });
      
    } catch (error) {
      console.error('문서 목록 조회 실패:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            code: 'INTERNAL_SERVER_ERROR',
            message: '문서 목록을 불러오는 중 오류가 발생했습니다.'
          },
          timestamp: new Date().toISOString()
        },
        { status: 500 }
      );
    }
  });
}

// 문서 업로드 API (파일 업로드 후 메타데이터 저장)
export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user, authType) => {
    try {
      const formData = await req.formData();
      const file = formData.get('file') as File;
      const projectName = formData.get('project_name') as string;
      const tags = formData.get('tags') as string;
      
      if (!file) {
        return NextResponse.json(
          {
            success: false,
            error: {
              code: 'INVALID_REQUEST',
              message: '파일이 제공되지 않았습니다.'
            }
          },
          { status: 400 }
        );
      }
      
      // 파일 정보 추출
      const filename = file.name;
      const filesize = file.size;
      const mimetype = file.type;
      const userId = user.username || user.id;
      
      // 태그 파싱 (쉼표로 구분된 문자열을 배열로 변환)
      const tagsArray = tags ? tags.split(',').map(tag => tag.trim()).filter(tag => tag.length > 0) : [];
      
      // 메타데이터 구성
      const metadata = {
        originalName: filename,
        uploadedBy: userId,
        uploadedAt: new Date().toISOString()
      };
      
      // 데이터베이스에 문서 정보 저장
      const insertQuery = `
        INSERT INTO chat_documents (
          filename, filepath, filesize, mimetype, user_id,
          upload_status, embedding_status, project_name, tags, metadata
        ) VALUES ($1, $2, $3, $4, $5, $6, $7, $8, $9, $10)
        RETURNING id, filename, upload_status, embedding_status, created_at
      `;
      
      // 임시 파일 경로 (실제로는 RAG 서버에서 처리)
      const filepath = `/tmp/${Date.now()}_${filename}`;
      
      const insertResult = await pool.query(insertQuery, [
        filename,
        filepath,
        filesize,
        mimetype,
        userId,
        'uploaded',
        'pending',
        projectName || null,
        tagsArray,
        JSON.stringify(metadata)
      ]);
      
      const document = insertResult.rows[0];
      
      // 실제 파일 업로드는 별도 처리 (RAG 서버로 전송)
      // TODO: RAG 서버로 파일 업로드 로직 구현
      
      return NextResponse.json({
        success: true,
        data: {
          ...document,
          message: '문서가 성공적으로 업로드되었습니다. 임베딩 처리를 시작합니다.'
        },
        timestamp: new Date().toISOString()
      });
      
    } catch (error) {
      console.error('문서 업로드 실패:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            code: 'INTERNAL_SERVER_ERROR',
            message: '문서 업로드 중 오류가 발생했습니다.'
          },
          timestamp: new Date().toISOString()
        },
        { status: 500 }
      );
    }
  });
}