import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../../middlewares';
import { Pool } from 'pg';

// 데이터베이스 연결 설정
const pool = new Pool({
  host: process.env.DB_HOST || process.env.POSTGRES_HOST || 'localhost',
  port: parseInt(process.env.DB_PORT || process.env.POSTGRES_PORT || '5433'),
  database: process.env.DB_NAME || process.env.POSTGRES_DB || 'airun',
  user: process.env.DB_USER || process.env.POSTGRES_USER || 'ivs',
  password: process.env.DB_PASSWORD || process.env.POSTGRES_PASSWORD || '1234',
});

interface TreeNode {
  id: string;
  type: 'user' | 'project' | 'all';
  name: string;
  userId?: string;
  projectId?: number;
  documentCount: number;
  children?: TreeNode[];
  expanded?: boolean;
}

// 트리 구조 데이터 조회 API
export async function GET(req: NextRequest) {
  return withAnyAuth(req, async (req, user, authType) => {
    try {
      const treeData: TreeNode[] = [];
      
      // 관리자인 경우 전체 트리 구조 생성
      if (user.role === 'admin') {
        // 전체 문서 수 조회
        const totalCountResult = await pool.query('SELECT COUNT(*) as count FROM chat_documents');
        const totalCount = parseInt(totalCountResult.rows[0].count);
        
        // 전체 노드 추가
        const allNode: TreeNode = {
          id: 'all',
          type: 'all',
          name: '전체 문서',
          documentCount: totalCount,
          children: [],
          expanded: true
        };
        
        // 사용자별 문서 수 조회
        const userStatsResult = await pool.query(`
          SELECT 
            user_id,
            COUNT(*) as document_count
          FROM chat_documents 
          GROUP BY user_id
          ORDER BY user_id
        `);
        
        // 사용자별 노드 생성
        for (const userStat of userStatsResult.rows) {
          const userNode: TreeNode = {
            id: `user_${userStat.user_id}`,
            type: 'user',
            name: userStat.user_id,
            userId: userStat.user_id,
            documentCount: parseInt(userStat.document_count),
            children: [],
            expanded: false
          };
          
          // 해당 사용자의 프로젝트 조회 (chat_documents의 project_name 기반)
          const projectsResult = await pool.query(`
            SELECT 
              project_name as name,
              COUNT(*) as document_count
            FROM chat_documents 
            WHERE user_id = $1 AND project_name IS NOT NULL AND project_name != ''
            GROUP BY project_name
            ORDER BY project_name
          `, [userStat.user_id]);
          
          // 프로젝트 노드 추가
          for (const project of projectsResult.rows) {
            const projectNode: TreeNode = {
              id: `project_${userStat.user_id}_${project.name}`,
              type: 'project',
              name: project.name,
              userId: userStat.user_id,
              projectId: project.name, // project_name을 ID로 사용
              documentCount: parseInt(project.document_count),
              expanded: false
            };
            userNode.children!.push(projectNode);
          }
          
          allNode.children!.push(userNode);
        }
        
        treeData.push(allNode);
        
      } else {
        // 일반 사용자는 자신의 프로젝트만 표시
        const userId = user.username || user.id;
        
        // 사용자의 총 문서 수 조회
        const userCountResult = await pool.query(
          'SELECT COUNT(*) as count FROM chat_documents WHERE user_id = $1',
          [userId]
        );
        const userCount = parseInt(userCountResult.rows[0].count);
        
        // 내 문서 노드
        const myDocsNode: TreeNode = {
          id: 'my_docs',
          type: 'user',
          name: '내 문서',
          userId: userId,
          documentCount: userCount,
          children: [],
          expanded: true
        };
        
        // 사용자의 프로젝트 조회 (chat_documents의 project_name 기반)
        const projectsResult = await pool.query(`
          SELECT 
            project_name as name,
            COUNT(*) as document_count
          FROM chat_documents 
          WHERE user_id = $1 AND project_name IS NOT NULL AND project_name != ''
          GROUP BY project_name
          ORDER BY project_name
        `, [userId]);
        
        // 프로젝트 노드 추가
        for (const project of projectsResult.rows) {
          const projectNode: TreeNode = {
            id: `project_${userId}_${project.name}`,
            type: 'project',
            name: project.name,
            userId: userId,
            projectId: project.name, // project_name을 ID로 사용
            documentCount: parseInt(project.document_count),
            expanded: false
          };
          myDocsNode.children!.push(projectNode);
        }
        
        treeData.push(myDocsNode);
      }
      
      return NextResponse.json({
        success: true,
        data: treeData,
        timestamp: new Date().toISOString()
      });
      
    } catch (error) {
      console.error('트리 데이터 조회 실패:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            code: 'INTERNAL_SERVER_ERROR',
            message: '트리 데이터를 불러오는 중 오류가 발생했습니다.'
          },
          timestamp: new Date().toISOString()
        },
        { status: 500 }
      );
    }
  });
}