import { NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';
import os from 'os';

interface UsageItem {
  timestamp: string;
  userId?: string;
  username?: string;
  feature?: string;
  [key: string]: any;
}

interface TopUser {
  user_id: string;
  username?: string;
  request_count: number;
}

function getAppDataPath() {
  switch (process.platform) {
    case 'win32':
      return path.join(process.env.APPDATA || '', 'airun');
    case 'darwin':
      return path.join(os.homedir(), 'Library', 'Application Support', 'airun');
    default:
      return path.join(os.homedir(), '.airun');
  }
}

async function getAllUsageData(): Promise<UsageItem[]> {
  try {
    const appDataDir = getAppDataPath();
    const archiveDir = path.join(appDataDir, 'archives');
    let allData: UsageItem[] = [];
    // 아카이브 데이터
    if (fs.existsSync(archiveDir)) {
      const archiveFiles = await fs.promises.readdir(archiveDir);
      for (const file of archiveFiles) {
        if (file.startsWith('usage_') && file.endsWith('.json')) {
          const archiveContent = await fs.promises.readFile(path.join(archiveDir, file), 'utf8');
          const archiveData = JSON.parse(archiveContent) as UsageItem[];
          allData.push(...archiveData);
        }
      }
    }
    // 현재 데이터
    const usageFilePath = path.join(appDataDir, 'usage.json');
    if (fs.existsSync(usageFilePath)) {
      const fileContent = await fs.promises.readFile(usageFilePath, 'utf8');
      if (fileContent.trim()) {
        const currentData = JSON.parse(fileContent) as UsageItem[];
        allData.push(...currentData);
      }
    }
    // 날짜순 정렬
    allData.sort((a, b) => new Date(a.timestamp).getTime() - new Date(b.timestamp).getTime());
    return allData;
  } catch (error) {
    return [];
  }
}

export async function GET() {
  try {
    const allUsage = await getAllUsageData();
    const now = new Date();
    const startOfDay = new Date(now.getFullYear(), now.getMonth(), now.getDate());
    const startOfWeek = new Date(now);
    startOfWeek.setDate(now.getDate() - now.getDay());
    const startOfMonth = new Date(now.getFullYear(), now.getMonth(), 1);

    const daily = allUsage.filter((u) => new Date(u.timestamp) >= startOfDay);
    const weekly = allUsage.filter((u) => new Date(u.timestamp) >= startOfWeek);
    const monthly = allUsage.filter((u) => new Date(u.timestamp) >= startOfMonth);

    // 비용/토큰 계산 함수 (provider/model별 가격 정책 적용 필요시 확장)
    function calcCost(u: any) {
      // 무료 제공자
      if (u.provider === 'vllm' || u.provider === 'ollama') return 0;
      if (u.cost) return parseFloat(u.cost);
      const promptTokens = parseInt(u.promptTokens) || 0;
      const completionTokens = parseInt(u.completionTokens) || 0;
      // 기본 단가(예시): $0.0001/token
      return (promptTokens + completionTokens) * 0.0001;
    }

    // 전체/월별 비용, 토큰
    let totalCost = 0, totalTokens = 0, monthlyCost = 0, monthlyTokens = 0;
    const currentMonth = now.getMonth(), currentYear = now.getFullYear();
    for (const u of allUsage) {
      const cost = calcCost(u);
      const promptTokens = parseInt(u.promptTokens) || 0;
      const completionTokens = parseInt(u.completionTokens) || 0;
      const tokens = promptTokens + completionTokens;
      totalCost += cost;
      totalTokens += tokens;
      const d = new Date(u.timestamp);
      if (d.getMonth() === currentMonth && d.getFullYear() === currentYear) {
        monthlyCost += cost;
        monthlyTokens += tokens;
      }
    }

    // 상위 사용자
    const userMap: Record<string, TopUser> = {};
    for (const u of allUsage) {
      if (!u.userId) continue;
      if (!userMap[u.userId]) userMap[u.userId] = { user_id: u.userId, username: u.username, request_count: 0 };
      userMap[u.userId].request_count++;
    }
    const top_users = Object.values(userMap).sort((a, b) => b.request_count - a.request_count).slice(0, 10);

    // 기능별 사용량
    const feature_usage: Record<string, number> = {};
    for (const u of allUsage) {
      if (!u.feature) continue;
      feature_usage[u.feature] = (feature_usage[u.feature] || 0) + 1;
    }

    // 제공자별 통계
    const providerStats: Record<string, { cost: number; promptTokens: number; completionTokens: number; requests: number; }> = {};
    for (const u of allUsage) {
      const provider = u.provider || 'Unknown';
      if (!providerStats[provider]) providerStats[provider] = { cost: 0, promptTokens: 0, completionTokens: 0, requests: 0 };
      providerStats[provider].cost += calcCost(u);
      providerStats[provider].promptTokens += parseInt(u.promptTokens) || 0;
      providerStats[provider].completionTokens += parseInt(u.completionTokens) || 0;
      providerStats[provider].requests++;
    }

    // 월별 추이
    const monthlyStats: Record<string, { cost: number; tokens: number; }> = {};
    for (const u of allUsage) {
      const d = new Date(u.timestamp);
      const monthKey = `${d.getFullYear()}-${String(d.getMonth() + 1).padStart(2, '0')}`;
      if (!monthlyStats[monthKey]) monthlyStats[monthKey] = { cost: 0, tokens: 0 };
      monthlyStats[monthKey].cost += calcCost(u);
      monthlyStats[monthKey].tokens += (parseInt(u.promptTokens) || 0) + (parseInt(u.completionTokens) || 0);
    }

    // 시간대별(요일/시간) 히트맵
    const heatmap = Array(7).fill(0).map(() => Array(24).fill(0));
    for (const u of allUsage) {
      const d = new Date(u.timestamp);
      heatmap[d.getDay()][d.getHours()]++;
    }

    // AI 호출 상세 내역(최신순)
    const callDetails = allUsage.map(u => ({
      timestamp: u.timestamp,
      provider: u.provider,
      model: u.model,
      promptTokens: parseInt(u.promptTokens) || 0,
      completionTokens: parseInt(u.completionTokens) || 0,
      cost: calcCost(u),
      sessionId: u.sessionId,
      prompt: typeof u.prompt === 'string' ? u.prompt : String(u.prompt || ''),
      userId: u.userId,
      username: u.username,
      feature: u.feature
    })).sort((a, b) => new Date(b.timestamp).getTime() - new Date(a.timestamp).getTime());

    // 세션별 사용량
    const sessionMap: Record<string, { timestamp: string; firstPrompt: string; calls: number; totalTokens: number; providers: Set<string>; }> = {};
    for (const u of allUsage) {
      const sessionId = u.sessionId || 'default';
      if (!sessionMap[sessionId] && u.prompt && typeof u.prompt === 'string' && !u.prompt.startsWith('I executed') && !u.prompt.startsWith('\n')) {
        sessionMap[sessionId] = {
          timestamp: u.timestamp,
          firstPrompt: u.prompt,
          calls: 0,
          totalTokens: 0,
          providers: new Set()
        };
      }
      if (sessionMap[sessionId]) {
        sessionMap[sessionId].calls++;
        sessionMap[sessionId].totalTokens += (parseInt(u.promptTokens) || 0) + (parseInt(u.completionTokens) || 0);
        sessionMap[sessionId].providers.add(u.provider);
      }
    }
    // 모델별 토큰/비용 효율성
    const modelStats: Record<string, { promptTokens: number; completionTokens: number; totalTokens: number; totalCost: number; }> = {};
    for (const u of allUsage) {
      const modelKey = `${u.provider || 'Unknown'}-${u.model || 'Unknown'}`;
      if (!modelStats[modelKey]) modelStats[modelKey] = { promptTokens: 0, completionTokens: 0, totalTokens: 0, totalCost: 0 };
      const promptTokens = parseInt(u.promptTokens) || 0;
      const completionTokens = parseInt(u.completionTokens) || 0;
      modelStats[modelKey].promptTokens += promptTokens;
      modelStats[modelKey].completionTokens += completionTokens;
      modelStats[modelKey].totalTokens += promptTokens + completionTokens;
      modelStats[modelKey].totalCost += calcCost(u);
    }
    // 오늘 세션별 버블차트용 데이터
    const today = new Date(); today.setHours(0, 0, 0, 0);
    const tomorrow = new Date(today); tomorrow.setDate(tomorrow.getDate() + 1);
    const todayData = allUsage.filter(u => {
      const d = new Date(u.timestamp);
      return d >= today && d < tomorrow;
    });
    const todaySessionMap: Record<string, { timestamp: string; calls: number; totalTokens: number; providers: Set<string>; }> = {};
    for (const u of todayData) {
      const sessionId = u.sessionId || 'default';
      if (!todaySessionMap[sessionId]) {
        todaySessionMap[sessionId] = {
          timestamp: u.timestamp,
          calls: 0,
          totalTokens: 0,
          providers: new Set()
        };
      }
      todaySessionMap[sessionId].calls++;
      todaySessionMap[sessionId].totalTokens += (parseInt(u.promptTokens) || 0) + (parseInt(u.completionTokens) || 0);
      todaySessionMap[sessionId].providers.add(u.provider);
    }

    return NextResponse.json({
      success: true,
      data: {
        daily_requests: daily.length,
        weekly_requests: weekly.length,
        monthly_requests: monthly.length,
        top_users,
        feature_usage,
        total_cost: totalCost,
        total_tokens: totalTokens,
        monthly_cost: monthlyCost,
        monthly_tokens: monthlyTokens,
        provider_stats: providerStats,
        monthly_stats: monthlyStats,
        heatmap,
        call_details: callDetails,
        session_stats: Object.entries(sessionMap).map(([sessionId, v]) => ({ sessionId, ...v, providers: Array.from(v.providers) })),
        model_stats: Object.entries(modelStats).map(([model, v]) => ({ model, ...v })),
        today_session_bubble: Object.entries(todaySessionMap).map(([sessionId, v]) => ({ sessionId, ...v, providers: Array.from(v.providers) })),
        all_usage: allUsage // 원본 전체 내역
      }
    });
  } catch (error) {
    return NextResponse.json({ success: false, error: { message: error instanceof Error ? error.message : String(error) } }, { status: 500 });
  }
} 