import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../middlewares';
import { Pool } from 'pg';

// 데이터베이스 연결 설정
const pool = new Pool({
  host: process.env.DB_HOST || process.env.POSTGRES_HOST || 'localhost',
  port: parseInt(process.env.DB_PORT || process.env.POSTGRES_PORT || '5433'),
  database: process.env.DB_NAME || process.env.POSTGRES_DB || 'airun',
  user: process.env.DB_USER || process.env.POSTGRES_USER || 'ivs',
  password: process.env.DB_PASSWORD || process.env.POSTGRES_PASSWORD || '1234',
});

// 프로젝트 목록 조회 API (실제 projects 테이블 기반)
export async function GET(req: NextRequest) {
  return withAnyAuth(req, async (req, user, authType) => {
    try {
      // projects 테이블에서 프로젝트 조회하고 문서 수 집계
      // project_id와 project_name 모두를 고려하여 JOIN (기존 데이터 호환성)
      let query = `
        SELECT
          p.id,
          p.name,
          p.description,
          p.user_id,
          p.created_at,
          p.updated_at,
          COALESCE(COUNT(cd.id), 0) as document_count
        FROM projects p
        LEFT JOIN chat_documents cd ON (
          p.id = cd.project_id 
          OR (cd.project_id IS NULL AND p.name = cd.project_name AND p.user_id = cd.user_id)
        )
      `;

      const queryParams: any[] = [];

      // 일반 사용자는 자신의 프로젝트만 조회
      if (user.role !== 'admin') {
        query += ' WHERE p.user_id = $1';
        queryParams.push(user.username || user.id);
      }

      query += ' GROUP BY p.id, p.name, p.description, p.user_id, p.created_at, p.updated_at';
      query += ' ORDER BY p.created_at DESC';

      const result = await pool.query(query, queryParams);

      // 결과를 프로젝트 형태로 변환
      const projects = result.rows.map(row => ({
        id: row.id,
        name: row.name,
        description: row.description,
        user_id: row.user_id,
        document_count: parseInt(row.document_count),
        created_at: row.created_at,
        updated_at: row.updated_at
      }));

      return NextResponse.json({
        success: true,
        data: projects,
        timestamp: new Date().toISOString()
      });
      
    } catch (error) {
      console.error('프로젝트 목록 조회 실패:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            code: 'INTERNAL_SERVER_ERROR',
            message: '프로젝트 목록을 불러오는 중 오류가 발생했습니다.'
          },
          timestamp: new Date().toISOString()
        },
        { status: 500 }
      );
    }
  });
}

// 프로젝트 생성 API (실제 projects 테이블에 저장)
export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user, authType) => {
    try {
      const body = await req.json();
      const { name, description } = body;

      if (!name || name.trim().length === 0) {
        return NextResponse.json(
          {
            success: false,
            error: {
              code: 'INVALID_REQUEST',
              message: '프로젝트명이 필요합니다.'
            }
          },
          { status: 400 }
        );
      }

      const userId = user.username || user.id;
      const projectName = name.trim();

      // 동일한 프로젝트명이 이미 존재하는지 확인
      const existingResult = await pool.query(
        'SELECT COUNT(*) as count FROM projects WHERE name = $1 AND user_id = $2',
        [projectName, userId]
      );

      if (parseInt(existingResult.rows[0].count) > 0) {
        return NextResponse.json(
          {
            success: false,
            error: {
              code: 'DUPLICATE_PROJECT',
              message: '동일한 이름의 프로젝트가 이미 존재합니다.'
            },
            timestamp: new Date().toISOString()
          },
          { status: 409 }
        );
      }

      // 실제 projects 테이블에 프로젝트 생성
      const insertResult = await pool.query(
        `INSERT INTO projects (name, description, user_id)
         VALUES ($1, $2, $3)
         RETURNING id, name, description, user_id, created_at, updated_at`,
        [projectName, description || null, userId]
      );

      const project = {
        ...insertResult.rows[0],
        document_count: 0
      };

      return NextResponse.json({
        success: true,
        data: {
          ...project,
          message: '프로젝트가 성공적으로 생성되었습니다.'
        },
        timestamp: new Date().toISOString()
      });
      
    } catch (error) {
      console.error('프로젝트 생성 실패:', error);
      
      return NextResponse.json(
        {
          success: false,
          error: {
            code: 'INTERNAL_SERVER_ERROR',
            message: '프로젝트 생성 중 오류가 발생했습니다.'
          },
          timestamp: new Date().toISOString()
        },
        { status: 500 }
      );
    }
  });
}