import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '../../middlewares';
import { getRagServerUrl } from '@/config/serverConfig';

const RAG_API_SERVER = getRagServerUrl();

// RAG 청크 조회 API
export async function GET(req: NextRequest) {
  return withAnyAuth(req, async (req, user) => {
    try {
      const { searchParams } = new URL(req.url);
      const filename = searchParams.get('filename');
      
      if (!filename) {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '파일명이 필요합니다.'
            }
          }, 
          { status: 400 }
        );
      }
      
      console.log('RAG 청크 조회 요청:', {
        filename,
        userId: user?.id
      });
      
      // RAG 서버에 청크 조회 요청
      const ragResponse = await fetch(`${RAG_API_SERVER}/list?filename=${encodeURIComponent(filename)}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        }
      });
      
      if (!ragResponse.ok) {
        const errorText = await ragResponse.text();
        console.error('RAG 서버 응답 오류:', {
          status: ragResponse.status,
          statusText: ragResponse.statusText,
          error: errorText
        });
        
        return NextResponse.json(
          {
            success: false,
            error: {
              message: `청크 조회 실패: ${ragResponse.statusText}`,
              details: errorText
            }
          }, 
          { status: ragResponse.status }
        );
      }
      
      const ragData = await ragResponse.json();
      console.log('RAG 서버 응답 성공:', {
        chunksCount: ragData.chunks?.length || 0,
        hasMetadata: ragData.chunks?.some((chunk: any) => chunk.metadata) || false
      });
      
      // 청크 데이터 구조 확인 및 메타데이터 파싱
      const processedChunks = ragData.chunks?.map((chunk: any, index: number) => {
        const metadata = chunk.metadata || {};
        
        // JSON 문자열로 저장된 메타데이터 파싱
        const parseJsonMetadata = (field: string) => {
          if (metadata[field] && typeof metadata[field] === 'string') {
            try {
              return JSON.parse(metadata[field]);
            } catch (e) {
              console.warn(`메타데이터 파싱 실패 (${field}):`, e);
              return [];
            }
          }
          return metadata[field] || [];
        };
        
        return {
          id: chunk.id || `chunk-${index}`,
          content: chunk.content || chunk.document || '',
          metadata: {
            ...metadata,
            extracted_questions: parseJsonMetadata('extracted_questions'),
            extracted_keywords: parseJsonMetadata('extracted_keywords'),
            extracted_topics: parseJsonMetadata('extracted_topics'),
            // 원본 메타데이터도 보존
            _raw_extracted_questions: metadata.extracted_questions,
            _raw_extracted_keywords: metadata.extracted_keywords,
            _raw_extracted_topics: metadata.extracted_topics,
          }
        };
      }) || [];
      
      return NextResponse.json({
        success: true,
        data: {
          filename,
          chunks: processedChunks,
          total: processedChunks.length,
          message: `${processedChunks.length}개의 청크를 조회했습니다.`
        }
      });
      
    } catch (error) {
      console.error('RAG 청크 조회 중 오류:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: error instanceof Error ? error.message : '청크 조회 중 오류가 발생했습니다.'
          }
        },
        { status: 500 }
      );
    }
  });
} 