import { NextRequest, NextResponse } from 'next/server';
import { withAnyAuth } from '@/app/api/middlewares';
import { pg } from '@/app/api/db';

const RAG_API_SERVER = process.env.NEXT_PUBLIC_RAG_API_SERVER || 'http://localhost:5600';

export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user) => {
    try {
      const formData = await req.formData();
      const file = formData.get('file') as File;
      
      if (!file) {
        return NextResponse.json({ success: false, error: 'File is required.' }, { status: 400 });
      }
      
      const apiFormData = new FormData();
      apiFormData.append('file', file);
      // support 사용자로 업로드
      apiFormData.append('user_id', 'support');
      
      // Call the new endpoint for page-based chunking
      const apiResponse = await fetch(`${RAG_API_SERVER}/upload-page-chunk`, {
        method: 'POST',
        body: apiFormData,
      });

      if (!apiResponse.ok) {
        const errorText = await apiResponse.text();
        return NextResponse.json({ success: false, error: `API Error: ${errorText}` }, { status: apiResponse.status });
      }

      const apiData = await apiResponse.json();

      // 파일이 support 폴더에 정상적으로 저장되었는지 확인하기 위해 documents API 호출
      try {
        const documentsResponse = await fetch(`${req.nextUrl.origin}/api/support/documents`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
          },
          body: JSON.stringify({
            filename: apiData.file || file.name,
            filepath: apiData.rag_docs_path || '',
            filesize: file.size,
            mimetype: file.type,
            user_id: 'support'
          })
        });

        if (!documentsResponse.ok) {
          console.warn('[DOCUMENTS_API_WARNING] Failed to register document in support folder');
        }
      } catch (docError) {
        console.error('[DOCUMENTS_API_ERROR]', docError);
        // 문서 등록 실패는 중요하지 않으므로 계속 진행
      }

      return NextResponse.json(apiData);

    } catch (error) {
      console.error('[SUPPORT_UPLOAD_ERROR]', error);
      return NextResponse.json({ success: false, error: 'Internal Server Error' }, { status: 500 });
    }
  });
} 