'use client';

// Promise.withResolvers polyfill 먼저 로드
import '@/lib/promise-polyfill';

import { useState, useEffect, useRef } from 'react';
import Image from 'next/image';
import { McpMarkdownRenderer } from '@/components/McpMarkdownRenderer';
import { externalApiClient, ApiClient } from '@/lib/apiClient';
import { getAuthHeaders } from '@/utils/api';

interface MCPTool {
  name: string;
  command?: string;
  args?: string[];
  url?: string;
  transport: 'stdio' | 'sse';
}

interface MCPConfig {
  [key: string]: MCPTool;
}

interface ChatMessage {
  role: 'user' | 'assistant' | 'tool';
  content: string;
  timestamp: Date;
  toolCalls?: ToolCall[];
  attachedFiles?: AttachedFile[];
}

interface AttachedFile {
  file: File;
  url?: string;
  name: string;
  type: string;
  size: number;
  extractedText?: string;
  base64?: string;
  isProcessing?: boolean;
  processingError?: string;
}

interface ToolCall {
  name: string;
  args: any;
  result?: any;
  error?: string;
  server?: string;
  duration?: number;
  isLoading?: boolean;
}

interface ModalProps {
  isOpen: boolean;
  onClose: () => void;
  title: string;
  message: string;
  type: 'success' | 'error' | 'info';
}

// 세션 관련 인터페이스 추가
interface AssistantSession {
  id: string;
  title: string;
  lastMessage: string;
  updatedAt: string;
  messageCount: number;
  provider: string;
  model: string;
  type: string;
  history: ChatMessage[];
}

interface ApiSession {
  id: string;
  title?: string;
  lastMessage?: string;
  updatedAt?: string;
  createdAt?: string;
  messageCount?: number;
  provider?: string;
  model?: string;
  type?: string;
  history?: any[];
}

// 프로바이더 인터페이스 추가
interface Provider {
  key: string;
  name: string;
  models: { id: string; name: string }[];
  requiresApiKey?: boolean;
  apiKeyConfigured?: boolean;
  isDynamic?: boolean;
  isCurrent?: boolean;
}

interface ServerToolInfo {
  serverName: string;
  toolCount: number;
  tools: Array<{
    name: string;
    description: string;
  }>;
}

// A2A 관련 인터페이스 추가
interface A2AAgent {
  id: string;
  name: string;
  capabilities: string[];
  endpoint: string;
  protocol: string;
  trustLevel: 'high' | 'medium' | 'low';
  status: 'connected' | 'disconnected' | 'error';
  lastSeen?: Date;
}

interface A2ANetworkStatus {
  isConnected: boolean;
  connectedAgents: number;
  exposedTools: number;
  networkHealth: 'good' | 'degraded' | 'poor';
  lastSync?: Date;
}

const Modal = ({ isOpen, onClose, title, message, type }: ModalProps) => {
  if (!isOpen) return null;

  const getIcon = () => {
    switch (type) {
      case 'success': return '✅';
      case 'error': return '❌';
      case 'info': return 'ℹ️';
      default: return 'ℹ️';
    }
  };

  const getColorClasses = () => {
    switch (type) {
      case 'success': return 'border-green-500 bg-white dark:bg-gray-800';
      case 'error': return 'border-red-500 bg-white dark:bg-gray-800';
      case 'info': return 'border-blue-500 bg-white dark:bg-gray-800';
      default: return 'border-gray-500 bg-white dark:bg-gray-800';
    }
  };

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center bg-black bg-opacity-50">
      <div className={`max-w-md w-full mx-4 rounded-lg border-2 ${getColorClasses()} p-6 shadow-xl`}>
        <div className="flex items-center space-x-3 mb-4">
          <span className="text-2xl">{getIcon()}</span>
          <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
            {title}
          </h3>
        </div>
        <p className="text-gray-700 dark:text-gray-300 mb-6 whitespace-pre-wrap">
          {message}
        </p>
        <div className="flex justify-end">
          <button
            onClick={onClose}
            className="px-4 py-2 bg-gray-600 hover:bg-gray-700 text-white rounded-md transition-colors"
          >
            확인
          </button>
        </div>
      </div>
    </div>
  );
};

// 사용자 친화적인 도구 호출 정보 표시 컴포넌트
const ToolCallsDisplay = ({ toolCalls }: { toolCalls: ToolCall[] }) => {
  const [isExpanded, setIsExpanded] = useState(false); // 기본적으로 닫혀 있도록 변경
  
  if (!toolCalls || toolCalls.length === 0) return null;

  const completedCalls = toolCalls.filter(tc => !tc.isLoading);
  const loadingCalls = toolCalls.filter(tc => tc.isLoading);
  const hasResults = completedCalls.length > 0;

  // 사용자 친화적인 도구 이름 변환
  const getFriendlyToolName = (toolName: string) => {
    const toolNameMap: { [key: string]: string } = {
      'web_search': '🌐 웹 검색',
      'execute_command': '⚡ 시스템 명령어 실행',
      'file_read': '📖 파일 읽기',
      'file_write': '📝 파일 쓰기',
      'database_query': '🗃️ 데이터베이스 쿼리',
      'api_call': '🔗 API 호출',
      'image_analysis': '🖼️ 이미지 분석',
      'text_analysis': '📊 텍스트 분석',
      'translation': '🌍 번역',
      'calculation': '🧮 계산',
      'code_execution': '💻 코드 실행',
      'document_generation': '📄 문서 생성'
    };
    return toolNameMap[toolName] || `🔧 ${toolName}`;
  };

  // 서버 이름을 사용자 친화적으로 변환
  const getFriendlyServerName = (serverName: string) => {
    // 실제 서버 이름을 그대로 사용하되, 일부 특수 케이스만 처리
    if (!serverName || serverName === 'unknown') {
      return '알 수 없음';
    }
    
    // 서버 이름의 첫 글자를 대문자로 변환
    return serverName.charAt(0).toUpperCase() + serverName.slice(1);
  };

  // 서버별 배지 색상 스타일
  const getServerBadgeStyle = (serverName: string) => {
    // 서버 이름에 따라 동적으로 색상 생성
    if (!serverName || serverName === 'unknown') {
      return 'bg-gray-100 dark:bg-gray-600 text-gray-800 dark:text-gray-200';
    }
    
    // 서버 이름의 해시값을 이용해 일관된 색상 선택
    const colors = [
      'bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-200',
      'bg-purple-100 dark:bg-purple-900/30 text-purple-800 dark:text-purple-200',
      'bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-200',
      'bg-orange-100 dark:bg-orange-900/30 text-orange-800 dark:text-orange-200',
      'bg-cyan-100 dark:bg-cyan-900/30 text-cyan-800 dark:text-cyan-200',
      'bg-red-100 dark:bg-red-900/30 text-red-800 dark:text-red-200',
      'bg-yellow-100 dark:bg-yellow-900/30 text-yellow-800 dark:text-yellow-200',
      'bg-indigo-100 dark:bg-indigo-900/30 text-indigo-800 dark:text-indigo-200',
      'bg-pink-100 dark:bg-pink-900/30 text-pink-800 dark:text-pink-200',
      'bg-teal-100 dark:bg-teal-900/30 text-teal-800 dark:text-teal-200'
    ];
    
    // 서버 이름의 문자열 해시값 계산
    let hash = 0;
    for (let i = 0; i < serverName.length; i++) {
      const char = serverName.charCodeAt(i);
      hash = ((hash << 5) - hash) + char;
      hash = hash & hash; // 32비트 정수로 변환
    }
    
    // 해시값을 이용해 색상 선택
    const colorIndex = Math.abs(hash) % colors.length;
    return colors[colorIndex];
  };

  // 입력 인자를 사용자 친화적으로 변환
  const formatInputArgs = (args: any) => {
    if (!args || typeof args !== 'object') return args;
    
    const formatted: string[] = [];
    for (const [key, value] of Object.entries(args)) {
      if (key === 'command' || key === 'query' || key === 'search_term' || key === 'input') {
        formatted.push(`${value}`);
      } else if (key === 'timeout_ms') {
        formatted.push(`타임아웃: ${value}ms`);
      } else if (key === 'file_path' || key === 'path') {
        formatted.push(`파일: ${value}`);
      } else if (key === 'url') {
        formatted.push(`URL: ${value}`);
      } else if (key === 'language') {
        formatted.push(`언어: ${value}`);
      } else if (typeof value === 'string' && value.length < 100) {
        formatted.push(`${key}: ${value}`);
      } else if (typeof value === 'number') {
        formatted.push(`${key}: ${value}`);
      } else if (typeof value === 'boolean') {
        formatted.push(`${key}: ${value ? '예' : '아니오'}`);
      }
    }
    return formatted.length > 0 ? formatted.join(' | ') : '입력 없음';
  };

  // 실행 결과를 사용자 친화적으로 변환
  const formatResult = (result: any) => {
    if (!result) return '결과 없음';
    
    if (typeof result === 'string') {
      try {
        const parsed = JSON.parse(result);
        return formatResult(parsed);
      } catch {
        return result.length > 500 ? result.substring(0, 500) + '...' : result;
      }
    }
    
    if (typeof result === 'object') {
      // 시스템 명령어 실행 결과
      if (result.content && Array.isArray(result.content)) {
        const textContent = result.content
          .filter((item: any) => item.type === 'text')
          .map((item: any) => item.text)
          .join('\n');
        return textContent.length > 500 ? textContent.substring(0, 500) + '...' : textContent;
      }
      
      // 웹 검색 결과
      if (result.results && Array.isArray(result.results)) {
        return `${result.results.length}개의 검색 결과를 찾았습니다.`;
      }
      
      // 파일 읽기 결과
      if (result.file_content || result.content) {
        const content = result.file_content || result.content;
        return typeof content === 'string' 
          ? (content.length > 500 ? content.substring(0, 500) + '...' : content)
          : '파일 내용을 읽었습니다.';
      }
      
      // 기타 객체 결과
      if (result.status === 'success' || result.success) {
        return result.message || '작업이 성공적으로 완료되었습니다.';
      }
      
      if (result.error) {
        return `오류: ${result.error}`;
      }
      
      // 일반 객체를 간단히 표시
      const keys = Object.keys(result);
      if (keys.length > 0) {
        return `${keys.length}개의 속성을 가진 결과가 반환되었습니다.`;
      }
    }
    
    return '결과를 반환했습니다.';
  };

  return (
    <div className="mt-4 w-full bg-gradient-to-r from-purple-50 to-blue-50 dark:from-purple-900/20 dark:to-blue-900/20 rounded-lg border-2 border-purple-200 dark:border-purple-700 overflow-hidden shadow-sm">
      {/* 헤더 - 항상 표시 */}
      <button
        onClick={() => setIsExpanded(!isExpanded)}
        className="w-full px-4 py-3 flex items-center justify-between hover:bg-purple-100 dark:hover:bg-purple-800/30 transition-colors"
      >
        <div className="flex items-center space-x-3">
          <span className="text-sm">🛠️</span>
          <div className="text-left">
            <p className="font-bold text-purple-900 dark:text-purple-100 text-sm">
              AI 도구 사용 내역
              {loadingCalls.length > 0 && (
                <span className="ml-2 text-sm text-blue-600 dark:text-blue-400 font-normal">
                  ({loadingCalls.length}개 실행 중)
                </span>
              )}
            </p>
            <p className="text-sm text-purple-700 dark:text-purple-300">
              {completedCalls.length > 0 && `${completedCalls.length}개 완료`}
              {loadingCalls.length > 0 && completedCalls.length > 0 && ', '}
              {loadingCalls.length > 0 && `${loadingCalls.length}개 실행 중`}
            </p>
          </div>
        </div>
        
        <div className="flex items-center space-x-2">
          {/* 로딩 중인 도구가 있으면 스피너 표시 */}
          {loadingCalls.length > 0 && (
            <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-blue-600"></div>
          )}
          
          {/* 화살표 아이콘 - 사용자 선호에 따라 up/down 화살표 사용 */}
          <svg 
            className={`w-5 h-5 text-purple-600 dark:text-purple-400 transition-transform ${
              isExpanded ? 'rotate-180' : ''
            }`} 
            fill="none" 
            stroke="currentColor" 
            viewBox="0 0 24 24"
          >
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
          </svg>
        </div>
      </button>
      
      {/* 펼쳐진 내용 */}
      {isExpanded && (
        <div className="w-full border-t border-purple-200 dark:border-purple-600 bg-white dark:bg-gray-900">
          <div className="w-full p-4 space-y-4">
            {toolCalls.map((toolCall, index) => (
              <div key={index} className="w-full bg-gradient-to-r from-gray-50 to-gray-100 dark:from-gray-800 dark:to-gray-700 rounded-lg p-4 border border-gray-200 dark:border-gray-600 shadow-sm">
                <div className="flex items-center justify-between mb-4">
                  <div className="flex items-center space-x-3">
                    <span className="text-sm">
                      {toolCall.isLoading ? (
                        <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-blue-600"></div>
                      ) : toolCall.error ? '❌' : '✅'}
                    </span>
                    <div>
                      <div className="flex items-center space-x-2">
                        <p className="font-bold text-gray-900 dark:text-white text-sm">
                          {getFriendlyToolName(toolCall.name)}
                        </p>
                        {toolCall.server && (
                          <span className={`inline-flex items-center px-2 py-1 rounded-full text-xs font-medium ${getServerBadgeStyle(toolCall.server)}`}>
                            {getFriendlyServerName(toolCall.server)}
                          </span>
                        )}
                      </div>
                    </div>
                  </div>
                </div>
                
                {/* 사용자 친화적인 도구 호출 정보 */}
                <div className="w-full space-y-4">
                  {/* 입력 정보 */}
                  <div>
                    <p className="text-sm font-bold text-gray-700 dark:text-gray-300 mb-2 flex items-center">
                      <span className="mr-2">📥</span>
                      입력
                    </p>
                    <div className="w-full bg-blue-50 dark:bg-blue-900/20 rounded-md p-3 border border-blue-200 dark:border-blue-700">
                      <p className="text-sm text-blue-800 dark:text-blue-200 font-medium">
                        {formatInputArgs(toolCall.args)}
                      </p>
                    </div>
                  </div>
                  
                  {/* 실행 결과 */}
                  {toolCall.result && (
                    <div>
                      <p className="text-sm font-bold text-gray-700 dark:text-gray-300 mb-2 flex items-center">
                        <span className="mr-2">📤</span>
                        실행 결과
                      </p>
                      <div className="w-full bg-green-50 dark:bg-green-900/20 rounded-md p-3 border border-green-200 dark:border-green-700">
                        <p className="text-sm text-green-800 dark:text-green-200 font-medium whitespace-pre-wrap">
                          {formatResult(toolCall.result)}
                        </p>
                        {/* 원본 데이터 보기 토글 */}
                        <details className="mt-2">
                          <summary className="text-sm text-gray-500 dark:text-gray-400 cursor-pointer hover:text-gray-700 dark:hover:text-gray-300">
                            원본 데이터 보기 (개발자용)
                          </summary>
                          <div className="mt-2 w-full bg-gray-100 dark:bg-gray-800 rounded p-2 text-sm text-gray-600 dark:text-gray-400 overflow-x-auto">
                            <pre className="whitespace-pre-wrap">
                              {typeof toolCall.result === 'string' 
                                ? toolCall.result 
                                : JSON.stringify(toolCall.result, null, 2)}
                            </pre>
                          </div>
                        </details>
                      </div>
                    </div>
                  )}
                  
                  {/* 오류 메시지 */}
                  {toolCall.error && (
                    <div>
                      <p className="text-sm font-bold text-red-600 dark:text-red-400 mb-2 flex items-center">
                        <span className="mr-2">❌</span>
                        오류 메시지
                      </p>
                      <div className="w-full bg-red-50 dark:bg-red-900/20 rounded-md p-3 border border-red-200 dark:border-red-700 overflow-x-auto shadow-inner">
                        <pre className="text-sm text-red-800 dark:text-red-200 whitespace-pre-wrap">
                          {toolCall.error}
                        </pre>
                      </div>
                    </div>
                  )}
                  
                  {/* 실행 중 상태 */}
                  {toolCall.isLoading && (
                    <div>
                      <p className="text-sm font-bold text-blue-600 dark:text-blue-400 mb-2 flex items-center">
                        <span className="mr-2">⏳</span>
                        실행 상태
                      </p>
                      <div className="w-full bg-blue-50 dark:bg-blue-900/20 rounded-md p-3 border border-blue-200 dark:border-blue-700">
                        <div className="flex items-center space-x-2">
                          <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-blue-600"></div>
                          <p className="text-sm text-blue-800 dark:text-blue-200 font-medium">
                            도구를 실행하는 중입니다...
                          </p>
                        </div>
                      </div>
                    </div>
                  )}
                </div>
              </div>
            ))}
          </div>
        </div>
      )}
    </div>
  );
};

interface ImageModalProps {
  isOpen: boolean;
  onClose: () => void;
  imageUrl: string;
  imageName: string;
}

const ImageModal = ({ isOpen, onClose, imageUrl, imageName }: ImageModalProps) => {
  if (!isOpen) return null;

  return (
    <div 
      className="fixed inset-0 bg-black bg-opacity-75 flex items-center justify-center z-50 p-4"
      onClick={onClose}
    >
      <div 
        className="relative max-w-4xl max-h-full bg-white dark:bg-gray-800 rounded-lg overflow-hidden shadow-2xl"
        onClick={(e) => e.stopPropagation()}
      >
        {/* 닫기 버튼 */}
        <button
          onClick={onClose}
          className="absolute top-4 right-4 z-10 bg-black bg-opacity-50 hover:bg-opacity-75 text-white rounded-full p-2 transition-all duration-200"
          title="닫기"
        >
          <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
          </svg>
        </button>
        
        {/* 이미지 */}
        <img
          src={imageUrl}
          alt={imageName}
          className="max-w-full max-h-[80vh] object-contain"
        />
        
        {/* 이미지 이름 */}
        <div className="absolute bottom-0 left-0 right-0 bg-black bg-opacity-50 text-white p-4">
          <p className="text-center text-sm">{imageName}</p>
        </div>
      </div>
    </div>
  );
};

export default function AssistantPage() {
  const [mcpConfig, setMcpConfig] = useState<MCPConfig>({});
  const [pendingConfig, setPendingConfig] = useState<MCPConfig>({});
  const [newToolJson, setNewToolJson] = useState(JSON.stringify({
    "hamonize": {
      "command": "node",
      "args": [
        "./mcp_tools/hamonize/dist/index.js"
      ],
      "transport": "stdio"
    }
  }, null, 2));
  const [chatMessages, setChatMessages] = useState<ChatMessage[]>([]);
  const [currentMessage, setCurrentMessage] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [isInitialized, setIsInitialized] = useState(false);
  const [isInitializing, setIsInitializing] = useState(false);
  const [uploadedFiles, setUploadedFiles] = useState<AttachedFile[]>([]);
  const [isDragOver, setIsDragOver] = useState(false);
  const [isProcessingFiles, setIsProcessingFiles] = useState(false);
  
  // 음성 인식 관련 상태
  const [isListening, setIsListening] = useState<boolean>(false);
  const [isProcessingAudio, setIsProcessingAudio] = useState<boolean>(false);
  const mediaRecorderRef = useRef<MediaRecorder | null>(null);
  const audioChunksRef = useRef<Blob[]>([]);
  const [serverCount, setServerCount] = useState(0);
  const [serverTools, setServerTools] = useState<ServerToolInfo[]>([]);
  const [totalToolCount, setTotalToolCount] = useState(0);
  const chatContainerRef = useRef<HTMLDivElement>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const abortControllerRef = useRef<AbortController | null>(null);

  // 세션 관리 상태 추가
  const [sessions, setSessions] = useState<AssistantSession[]>([]);
  const [currentSessionId, setCurrentSessionId] = useState<string | null>(null);
  const [isLoadingSessions, setIsLoadingSessions] = useState(false);

  // 프로바이더 및 모델 관련 상태 추가
  const [providers, setProviders] = useState<Record<string, Provider>>({});
  const [availableProviders, setAvailableProviders] = useState<Provider[]>([]);
  const [isLoadingProviders, setIsLoadingProviders] = useState(false);
  const [selectedProvider, setSelectedProvider] = useState<string>('ollama');
  const [selectedModel, setSelectedModel] = useState<string>('hamonize:latest');
  const [temperature, setTemperature] = useState<number>(0.7);

  // 폴딩 상태 관리
  const [chatSettingsCollapsed, setChatSettingsCollapsed] = useState(true);
  const [addToolCollapsed, setAddToolCollapsed] = useState(true);
  const [toolListCollapsed, setToolListCollapsed] = useState(true);
  const [a2aCollapsed, setA2aCollapsed] = useState(true);
  const [sessionListCollapsed, setSessionListCollapsed] = useState(false);

  // A2A 관련 상태 추가
  const [a2aNetworkStatus, setA2aNetworkStatus] = useState<A2ANetworkStatus>({
    isConnected: false,
    connectedAgents: 0,
    exposedTools: 0,
    networkHealth: 'poor',
    lastSync: undefined
  });
  const [a2aAgents, setA2aAgents] = useState<A2AAgent[]>([]);
  const [isA2aEnabled, setIsA2aEnabled] = useState(false);
  const [isA2aLoading, setIsA2aLoading] = useState(false);
  const [newAgentEndpoint, setNewAgentEndpoint] = useState('');
  const [newAgentName, setNewAgentName] = useState('');

  // 이미지 모달 상태
  const [imageModal, setImageModal] = useState<{
    isOpen: boolean;
    imageUrl: string;
    imageName: string;
  }>({
    isOpen: false,
    imageUrl: '',
    imageName: ''
  });

  // 토스트 알림 상태 관리
  const [toast, setToast] = useState<{
    message: string;
    type: 'success' | 'error' | 'info';
    isVisible: boolean;
  }>({
    message: '',
    type: 'info',
    isVisible: false
  });

  // 모달 상태
  const [modal, setModal] = useState<{
    isOpen: boolean;
    title: string;
    message: string;
    type: 'success' | 'error' | 'info';
  }>({
    isOpen: false,
    title: '',
    message: '',
    type: 'info'
  });

  // 사용자 설정 localStorage에 저장/로드
  const saveUserSettings = (provider: string, model: string, temp: number) => {
    const settings = {
      provider,
      model,
      temperature: temp
    };
    localStorage.setItem('mcp-assistant-settings', JSON.stringify(settings));
  };

  const loadUserSettings = () => {
    try {
      const savedSettings = localStorage.getItem('mcp-assistant-settings');
      if (savedSettings) {
        const settings = JSON.parse(savedSettings);
        return {
          provider: settings.provider || 'ollama',
          model: settings.model || 'hamonize:latest',
          temperature: settings.temperature || 0.7
        };
      }
    } catch (error) {
      console.error('사용자 설정 로드 실패:', error);
    }
    return null;
  };

  // 모달 표시 함수
  const showModal = (title: string, message: string, type: 'success' | 'error' | 'info' = 'info') => {
    setModal({
      isOpen: true,
      title,
      message,
      type
    });
  };

  // 모달 닫기 함수
  const closeModal = () => {
    setModal(prev => ({ ...prev, isOpen: false }));
  };

  // 토스트 표시 함수
  const showToast = (message: string, type: 'success' | 'error' | 'info' = 'info') => {
    setToast({
      message,
      type,
      isVisible: true
    });
  };

  // 토스트 자동 닫기 effect
  useEffect(() => {
    if (toast.isVisible) {
      const timer = setTimeout(() => {
        setToast(prev => ({ ...prev, isVisible: false }));
      }, 3000);
      
      return () => clearTimeout(timer);
    }
  }, [toast.isVisible]);

  // 토스트 닫기 함수
  const closeToast = () => {
    setToast(prev => ({ ...prev, isVisible: false }));
  };

  // 스미더리 형식의 MCP 설정 예시
  const defaultToolExample = {
    "hamonize": {
      "command": "node",
      "args": [
        "./mcp_tools/hamonize/dist/index.js"
      ],
      "transport": "stdio"
    }
  };

  // 프로바이더 목록 로드
  const loadProviders = async () => {
    setIsLoadingProviders(true);
    try {
      // 저장된 사용자 설정 먼저 로드
      const savedSettings = loadUserSettings();
      
      // 실제 API 서버에서 사용 가능한 프로바이더 목록 가져오기
      const availableResponse = await externalApiClient.getAvailableProviders();
      
      if (availableResponse.success && availableResponse.data) {
        setAvailableProviders(availableResponse.data);
        
        // 각 프로바이더의 모델 목록 로드
        const providersData: any = {};
        for (const provider of availableResponse.data) {
          try {
            const modelsResponse = await externalApiClient.getProviderModels(provider.key);
            if (modelsResponse.success && modelsResponse.data) {
              providersData[provider.key] = {
                name: provider.name,
                requiresApiKey: provider.requiresApiKey,
                apiKeyConfigured: provider.apiKeyConfigured !== false,
                isDynamic: provider.isDynamic,
                isAvailable: true,
                models: modelsResponse.data
              };
            }
          } catch (modelError) {
            console.warn(`${provider.key} 모델 목록 로드 실패:`, modelError);
            // 모델 목록 로드 실패해도 프로바이더는 표시
            providersData[provider.key] = {
              name: provider.name,
              requiresApiKey: provider.requiresApiKey,
              apiKeyConfigured: provider.apiKeyConfigured !== false,
              isDynamic: provider.isDynamic,
              isAvailable: true,
              models: []
            };
          }
        }
        
        setProviders(providersData);
        
        if (availableResponse.data.length > 0) {
          let selectedProviderKey = '';
          let selectedModelId = '';
          
          // 저장된 설정이 있으면 우선 사용
          if (savedSettings) {
            const savedProvider = availableResponse.data.find((p: any) => p.key === savedSettings.provider);
            if (savedProvider && providersData[savedProvider.key]) {
              selectedProviderKey = savedProvider.key;
              // 해당 프로바이더의 모델 목록에서 저장된 모델 찾기
              const savedModel = providersData[savedProvider.key].models.find((m: any) => 
                (typeof m === 'string' ? m : m.id) === savedSettings.model
              );
              if (savedModel) {
                selectedModelId = typeof savedModel === 'string' ? savedModel : savedModel.id;
              }
            }
          }
          
          // 저장된 설정이 없거나 유효하지 않으면 첫 번째 프로바이더 사용
          if (!selectedProviderKey) {
            const defaultProvider = availableResponse.data[0];
            selectedProviderKey = defaultProvider.key;
            
            if (providersData[defaultProvider.key] && providersData[defaultProvider.key].models.length > 0) {
              const firstModel = providersData[defaultProvider.key].models[0];
              selectedModelId = typeof firstModel === 'string' ? firstModel : firstModel.id;
            }
          }
          
          // 상태 업데이트
          setSelectedProvider(selectedProviderKey);
          setSelectedModel(selectedModelId);
          
          // 저장된 Temperature 설정도 적용
          if (savedSettings && savedSettings.temperature !== undefined) {
            setTemperature(savedSettings.temperature);
          }
          
          // console.log(`프로바이더 로드 완료: ${availableResponse.data.length}개 사용 가능, 현재: ${selectedProviderKey}`);
        } else {
          showModal('프로바이더 없음', 'API 서버에서 사용 가능한 프로바이더가 없습니다. API 서버 상태를 확인해주세요.', 'error');
        }
      } else {
        throw new Error(availableResponse.error?.message || '프로바이더 로드 실패');
      }
    } catch (error) {
      console.error('프로바이더 로드 실패:', error);
      showModal('프로바이더 로드 실패', 'API 서버에서 프로바이더 목록을 가져올 수 없습니다. 서버 상태를 확인해주세요.', 'error');
      
      // 에러 발생 시 빈 상태로 설정
      setAvailableProviders([]);
      setProviders({});
      
      // 저장된 설정이 있으면 기본값으로 설정
      const savedSettings = loadUserSettings();
      if (savedSettings) {
        setSelectedProvider(savedSettings.provider || '');
        setSelectedModel(savedSettings.model || '');
        setTemperature(savedSettings.temperature || 0.7);
      }
    } finally {
      setIsLoadingProviders(false);
    }
  };

  // 프로바이더 변경 처리
  const handleProviderChange = async (newProvider: string) => {
    setSelectedProvider(newProvider);
    
    const providerData = providers[newProvider];
    let newModel = '';
    
    if (!providerData || !providerData.models || providerData.models.length === 0) {
      try {
        // 실제 API 서버에서 해당 프로바이더의 모델 목록 가져오기
        const modelsResponse = await externalApiClient.getProviderModels(newProvider);
        
        if (modelsResponse.success && modelsResponse.data) {
          const models = modelsResponse.data;
          
          setProviders(prev => ({
            ...prev,
            [newProvider]: {
              ...prev[newProvider],
              models: models
            }
          }));
          
          if (models.length > 0) {
            newModel = typeof models[0] === 'string' ? models[0] : models[0].id;
            setSelectedModel(newModel);
          }
        }
      } catch (error) {
        console.error(`프로바이더 ${newProvider}의 모델 목록 로드 실패:`, error);
      }
    } else if (providerData.models.length > 0) {
      newModel = typeof providerData.models[0] === 'string' ? providerData.models[0] : providerData.models[0].id;
      setSelectedModel(newModel);
    }
    
    // 설정 저장
    saveUserSettings(newProvider, newModel, temperature);
  };

  // 모델 변경 처리
  const handleModelChange = (newModel: string) => {
    setSelectedModel(newModel);
    saveUserSettings(selectedProvider, newModel, temperature);
  };

  // 온도 변경 처리
  const handleTemperatureChange = (newTemperature: number) => {
    setTemperature(newTemperature);
    saveUserSettings(selectedProvider, selectedModel, newTemperature);
  };

  // A2A 관련 함수들
  const loadA2aNetworkStatus = async () => {
    try {
      setIsA2aLoading(true);
      const response = await fetch('/api/a2a?action=status');
      const data = await response.json();
      
      if (response.ok) {
        setA2aNetworkStatus(prev => ({
          ...prev,
          isConnected: true,
          connectedAgents: data.connectedAgents || 0,
          exposedTools: data.exposedTools || 0,
          networkHealth: data.health || 'good',
          lastSync: new Date()
        }));
      } else {
        setA2aNetworkStatus(prev => ({
          ...prev,
          isConnected: false,
          networkHealth: 'poor'
        }));
      }
    } catch (error) {
      console.error('A2A 네트워크 상태 로드 실패:', error);
      setA2aNetworkStatus(prev => ({
        ...prev,
        isConnected: false,
        networkHealth: 'poor'
      }));
    } finally {
      setIsA2aLoading(false);
    }
  };

  const toggleA2aNetwork = async () => {
    try {
      setIsA2aLoading(true);
      
      if (!isA2aEnabled) {
        // A2A 네트워크 활성화 - MCP 도구들을 노출
        const response = await fetch('/api/a2a?action=capabilities');
        const data = await response.json();
        
        if (response.ok) {
          setIsA2aEnabled(true);
          showToast('A2A 네트워크가 활성화되었습니다. MCP 도구들이 네트워크에 노출되었습니다.', 'success');
          await loadA2aNetworkStatus();
        } else {
          throw new Error(data.error || 'A2A 네트워크 활성화 실패');
        }
      } else {
        // A2A 네트워크 비활성화
        setIsA2aEnabled(false);
        setA2aNetworkStatus(prev => ({
          ...prev,
          isConnected: false,
          exposedTools: 0,
          networkHealth: 'poor'
        }));
        showToast('A2A 네트워크가 비활성화되었습니다.', 'info');
      }
    } catch (error) {
      console.error('A2A 네트워크 토글 실패:', error);
      showToast(`A2A 네트워크 ${isA2aEnabled ? '비활성화' : '활성화'} 실패: ${error}`, 'error');
    } finally {
      setIsA2aLoading(false);
    }
  };

  const registerA2aAgent = async () => {
    if (!newAgentEndpoint.trim() || !newAgentName.trim()) {
      showToast('에이전트 이름과 엔드포인트를 모두 입력해주세요.', 'error');
      return;
    }

    try {
      setIsA2aLoading(true);
      
      const newAgent: A2AAgent = {
        id: `external-${Date.now()}`,
        name: newAgentName.trim(),
        capabilities: [],
        endpoint: newAgentEndpoint.trim(),
        protocol: 'A2A/1.0',
        trustLevel: 'medium',
        status: 'disconnected'
      };

              const response = await fetch('/api/a2a', {
        method: 'PUT',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(newAgent)
      });

      const data = await response.json();

      if (response.ok) {
        setA2aAgents(prev => [...prev, { ...newAgent, status: 'connected' }]);
        setNewAgentEndpoint('');
        setNewAgentName('');
        showToast(`에이전트 "${newAgent.name}"이 성공적으로 등록되었습니다.`, 'success');
        await loadA2aNetworkStatus();
      } else {
        throw new Error(data.error || '에이전트 등록 실패');
      }
    } catch (error) {
      console.error('A2A 에이전트 등록 실패:', error);
      showToast(`에이전트 등록 실패: ${error}`, 'error');
    } finally {
      setIsA2aLoading(false);
    }
  };

  const testA2aConnection = async () => {
    try {
      setIsA2aLoading(true);
      showToast('A2A 네트워크 연결을 테스트하는 중...', 'info');
      
      const response = await fetch('/api/a2a');
      const data = await response.json();
      
      if (response.ok) {
        showToast(`A2A 네트워크 연결 성공! 프로토콜: ${data.protocol}`, 'success');
        await loadA2aNetworkStatus();
      } else {
        throw new Error('연결 테스트 실패');
      }
    } catch (error) {
      console.error('A2A 연결 테스트 실패:', error);
      showToast(`A2A 연결 테스트 실패: ${error}`, 'error');
    } finally {
      setIsA2aLoading(false);
    }
  };

  // MCP 설정 로드 및 자동 초기화
  const loadMCPConfig = async () => {
    setIsInitializing(true);
    try {
      const response = await fetch('/api/mcp/config');
      const data = await response.json();
      
      let configToUse: MCPConfig = {};
      
      if (data.success && data.config) {
        configToUse = data.config;
        setMcpConfig(data.config);
        setPendingConfig(data.config);
        setServerCount(Object.keys(data.config).length);
      } else {
        // 설정이 없거나 실패한 경우 빈 설정으로 시작
        console.error('MCP 설정 로드 실패:', data.error || '설정 로드 중 오류 발생');
        setMcpConfig({});
        setPendingConfig({});
        setServerCount(0);
        showToast('MCP 설정이 없습니다. 도구를 추가하거나 초기값으로 복원해주세요.', 'info');
        return;
      }
      
      // 설정이 있으면 자동 초기화
      if (configToUse && Object.keys(configToUse).length > 0) {
        try {
          showToast(`MCP 초기화 시작... ${Object.keys(configToUse).length}개의 서버를 로드하는 중입니다.`, 'info');
          
          // 로컬스토리지에서 인증 정보 가져오기 (initializeMCP와 동일한 로직)
          const authInfo = {
            userToken: localStorage.getItem('userToken'),
            apiKey: localStorage.getItem('apiKey'),
            userId: localStorage.getItem('userId'),
            username: localStorage.getItem('username'),
            userRole: localStorage.getItem('userRole')
          };

          const initResponse = await fetch('/api/mcp/initialize', {
            method: 'POST',
            headers: { 'Content-Type': 'application/json' },
            body: JSON.stringify({ 
              config: configToUse,
              authInfo: authInfo
            }),
          });
          
          const initData = await initResponse.json();
          
          if (initData.success) {
            // 초기화 완료 후 잠시 대기 (서버가 완전히 준비될 시간 제공)
            await new Promise(resolve => setTimeout(resolve, 3000));
            
            setIsInitialized(true);
            
            // 초기화 응답에 도구 정보가 포함되어 있는지 확인
            if (initData.toolsData && initData.toolsData.servers) {
              const { servers, totalToolCount } = initData.toolsData;
              console.log('도구 정보 로드 성공 (초기화 응답에서):', { servers, totalToolCount });
              setServerTools(servers);
              setTotalToolCount(totalToolCount);
              
              const serverCount = Object.keys(configToUse).length;
              showToast(`MCP 초기화 완료! ${serverCount}개의 서버, 총 ${totalToolCount}개의 도구가 로드되었습니다.`, 'success');
            } else {
              // 초기화 응답에 도구 정보가 없으면 별도 API 호출 (백업 로직)
              console.log('초기화 응답에 도구 정보 없음, 별도 API 호출');
              
              let toolsLoaded = false;
              let retryCount = 0;
              const maxRetries = 3;
              
              while (!toolsLoaded && retryCount < maxRetries) {
                try {
                  const toolsResponse = await fetch('/api/mcp/tools');
                  const toolsData = await toolsResponse.json();
                  
                  if (toolsData.success && toolsData.data) {
                    const { servers, totalToolCount } = toolsData.data;
                    console.log('도구 정보 로드 성공 (백업 API 호출):', { servers, totalToolCount });
                    setServerTools(servers);
                    setTotalToolCount(totalToolCount);
                    
                    const serverCount = Object.keys(configToUse).length;
                    showToast(`MCP 초기화 완료! ${serverCount}개의 서버, 총 ${totalToolCount}개의 도구가 로드되었습니다.`, 'success');
                    toolsLoaded = true;
                  } else {
                    console.error(`도구 정보 가져오기 실패 (시도 ${retryCount + 1}/${maxRetries}):`, toolsData.error);
                    retryCount++;
                    if (retryCount < maxRetries) {
                      await new Promise(resolve => setTimeout(resolve, 2000));
                    }
                  }
                } catch (error) {
                  console.error(`도구 정보 가져오기 실패 (시도 ${retryCount + 1}/${maxRetries}):`, error);
                  retryCount++;
                  if (retryCount < maxRetries) {
                    await new Promise(resolve => setTimeout(resolve, 2000));
                  }
                }
              }
              
              if (!toolsLoaded) {
                const serverCount = Object.keys(configToUse).length;
                showToast(`MCP 초기화 완료! ${serverCount}개의 서버가 로드되었습니다. (도구 정보 로드 실패)`, 'success');
              }
            }
          } else {
            console.error('MCP 초기화 실패:', initData.error);
            showToast(`MCP 초기화에 실패했습니다: ${initData.error}`, 'error');
          }
        } catch (initError) {
          console.error('MCP 초기화 중 오류:', initError);
          showToast('MCP 초기화 중 오류가 발생했습니다.', 'error');
        }
      } else {
        showToast('등록된 MCP 도구가 없습니다. 도구를 추가해주세요.', 'info');
      }
    } catch (error) {
      console.error('MCP 설정 로드 실패:', error);
      setMcpConfig({});
      setPendingConfig({});
      setServerCount(0);
      showToast('MCP 설정 로드에 실패했습니다.', 'error');
    } finally {
      setIsInitializing(false);
    }
  };

  // 채팅 컨테이너 자동 스크롤
  useEffect(() => {
    if (chatContainerRef.current) {
      chatContainerRef.current.scrollTop = chatContainerRef.current.scrollHeight;
    }
  }, [chatMessages]);

  // 컴포넌트 마운트 시 MCP 설정 로드 및 세션 로드
  useEffect(() => {
    // 인증 상태 확인
    const checkAuthAndLoad = async () => {
      const userToken = localStorage.getItem('userToken');
      const apiKey = localStorage.getItem('apiKey');
      const userId = localStorage.getItem('userId');
      const username = localStorage.getItem('username');
      
      // console.log('🔐 어시스턴트 페이지 인증 상태 확인:', {
      //   hasUserToken: !!userToken,
      //   hasApiKey: !!apiKey,
      //   userId: userId,
      //   username: username,
      //   userTokenPrefix: userToken?.substring(0, 20) + '...',
      //   apiKeyPrefix: apiKey?.substring(0, 10) + '...'
      // });
      
      if (!userToken && !apiKey) {
        console.warn('인증 정보가 없습니다. 로그인이 필요합니다.');
        showToast('로그인이 필요합니다.', 'error');
        return;
      }
      
      // 인증 정보가 있으면 초기화 진행
      loadMCPConfig();
      loadProviders();
      loadA2aNetworkStatus();
      
      try {
        await loadSessions();
      } catch (error) {
        console.error('세션 로드 실패:', error);
        showToast('세션을 로드할 수 없습니다. 로그인 상태를 확인해주세요.', 'error');
      }
    };
    
    checkAuthAndLoad();
  }, []);

  // A2A 네트워크 상태 주기적 업데이트
  useEffect(() => {
    if (isA2aEnabled) {
      const interval = setInterval(() => {
        loadA2aNetworkStatus();
      }, 30000); // 30초마다 상태 업데이트
      
      return () => clearInterval(interval);
    }
  }, [isA2aEnabled]);

  // MCP 설정 저장
  const saveMCPConfig = async (): Promise<boolean> => {
    try {
      const response = await fetch('/api/mcp/config', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify({ config: pendingConfig }),
      });

      const data = await response.json();
      
      if (data.success) {
        setServerCount(Object.keys(pendingConfig).length);
        showToast('MCP 설정이 저장되었습니다.', 'success');
        return true;
      } else {
        showToast(data.error || '설정 저장에 실패했습니다.', 'error');
        return false;
      }
    } catch (error) {
      console.error('MCP 설정 저장 실패:', error);
      showModal('설정 저장 실패', 'MCP 설정 저장에 실패했습니다.');
      return false;
    }
  };

  const initializeMCP = async () => {
    setIsLoading(true);
    setIsInitializing(true);
    try {
      // 로컬스토리지에서 인증 정보 가져오기
      const authInfo = {
        userToken: localStorage.getItem('userToken'),
        apiKey: localStorage.getItem('apiKey'),
        userId: localStorage.getItem('userId'),
        username: localStorage.getItem('username'),
        userRole: localStorage.getItem('userRole')
      };

      console.log('MCP 초기화 시 인증 정보:', {
        hasToken: !!authInfo.userToken,
        hasApiKey: !!authInfo.apiKey,
        username: authInfo.username,
        userId: authInfo.userId,
        userRole: authInfo.userRole
      });

      const response = await fetch('/api/mcp/initialize', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({ 
          config: pendingConfig,
          authInfo: authInfo
        }),
      });
      
      const data = await response.json();
      
      if (data.success) {
        // 초기화 완료 후 잠시 대기 (서버가 완전히 준비될 시간 제공)
        await new Promise(resolve => setTimeout(resolve, 3000));
        
        setIsInitialized(true);
        setMcpConfig(pendingConfig);
        
        // 초기화 응답에 도구 정보가 포함되어 있는지 확인
        if (data.toolsData && data.toolsData.servers) {
          const { servers, totalToolCount } = data.toolsData;
          console.log('도구 정보 로드 성공 (초기화 응답에서):', { servers, totalToolCount });
          setServerTools(servers);
          setTotalToolCount(totalToolCount);
          
          const currentServerCount = Object.keys(pendingConfig).length;
          addMessage('assistant', `✅ MCP 초기화 완료! ${currentServerCount}개의 서버, 총 ${totalToolCount}개의 도구가 로드되었습니다.`);
          showToast(`MCP 초기화 완료! ${currentServerCount}개의 서버, 총 ${totalToolCount}개의 도구가 로드되었습니다.`, 'success');
        } else {
          // 초기화 응답에 도구 정보가 없으면 별도 API 호출 (백업 로직)
          console.log('초기화 응답에 도구 정보 없음, 별도 API 호출');
          
          let toolsLoaded = false;
          let retryCount = 0;
          const maxRetries = 3;
          
          while (!toolsLoaded && retryCount < maxRetries) {
            try {
              const toolsResponse = await fetch('/api/mcp/tools');
              const toolsData = await toolsResponse.json();
              
              if (toolsData.success && toolsData.data) {
                const { servers, totalToolCount } = toolsData.data;
                console.log('도구 정보 로드 성공 (백업 API 호출):', { servers, totalToolCount });
                setServerTools(servers);
                setTotalToolCount(totalToolCount);
                
                const currentServerCount = Object.keys(pendingConfig).length;
                addMessage('assistant', `✅ MCP 초기화 완료! ${currentServerCount}개의 서버, 총 ${totalToolCount}개의 도구가 로드되었습니다.`);
                showToast(`MCP 초기화 완료! ${currentServerCount}개의 서버, 총 ${totalToolCount}개의 도구가 로드되었습니다.`, 'success');
                toolsLoaded = true;
              } else {
                console.error(`도구 정보 가져오기 실패 (시도 ${retryCount + 1}/${maxRetries}):`, toolsData.error);
                retryCount++;
                if (retryCount < maxRetries) {
                  await new Promise(resolve => setTimeout(resolve, 2000));
                }
              }
            } catch (error) {
              console.error(`도구 정보 가져오기 실패 (시도 ${retryCount + 1}/${maxRetries}):`, error);
              retryCount++;
              if (retryCount < maxRetries) {
                await new Promise(resolve => setTimeout(resolve, 2000));
              }
            }
          }
          
          if (!toolsLoaded) {
            const currentServerCount = Object.keys(pendingConfig).length;
            addMessage('assistant', data.message || `✅ MCP 초기화 완료! ${currentServerCount}개의 서버가 로드되었습니다. (도구 정보 로드 실패)`);
            showToast(`MCP 초기화 완료! ${currentServerCount}개의 서버가 로드되었습니다. (도구 정보 로드 실패)`, 'success');
          }
        }
      } else {
        addMessage('assistant', data.error || '❌ MCP 초기화에 실패했습니다.');
        showToast(`MCP 초기화 실패: ${data.error}`, 'error');
      }
    } catch (error) {
      console.error('MCP 초기화 실패:', error);
      addMessage('assistant', '❌ MCP 초기화 중 오류가 발생했습니다.');
      showToast('MCP 초기화 중 오류가 발생했습니다.', 'error');
    } finally {
      setIsLoading(false);
      setIsInitializing(false);
    }
  };

  const addTool = () => {
    try {
      if (!newToolJson.trim().startsWith('{') || !newToolJson.trim().endsWith('}')) {
        showModal('JSON 형식 오류', 'JSON은 중괄호({})로 시작하고 끝나야 합니다.', 'error');
        return;
      }

      const parsedTool = JSON.parse(newToolJson);
      
      // 스미더리(Smithery) 형식 처리: { "mcpServers": { ... } }
      let toolsToAdd = parsedTool;
      let isSmitheryFormat = false;
      
      if ("mcpServers" in parsedTool && typeof parsedTool.mcpServers === 'object') {
        toolsToAdd = parsedTool.mcpServers;
        isSmitheryFormat = true;
        console.log('스미더리 형식 감지됨:', toolsToAdd);
      }

      if (!toolsToAdd || typeof toolsToAdd !== 'object' || Object.keys(toolsToAdd).length === 0) {
        showModal('도구 추가 실패', '최소 하나 이상의 도구를 입력해주세요.', 'error');
        return;
      }

      const updatedConfig = { ...pendingConfig };
      let successTools: string[] = [];
      let failedTools: string[] = [];

      for (const [toolName, toolConfig] of Object.entries(toolsToAdd)) {
        const config = toolConfig as any;
        
        // name 필드 추가
        config.name = toolName;
        
        // transport 설정
        if ("url" in config) {
          config.transport = "sse";
        } else if (!config.transport) {
          config.transport = "stdio";
        }

        // 필수 필드 검증
        if (!config.command && !config.url) {
          showModal('도구 추가 실패', `'${toolName}' 도구 설정에는 'command' 또는 'url' 필드가 필요합니다.`, 'error');
          failedTools.push(toolName);
          continue;
        }
        
        if (config.command && !config.args) {
          showModal('도구 추가 실패', `'${toolName}' 도구 설정에는 'args' 필드가 필요합니다.`, 'error');
          failedTools.push(toolName);
          continue;
        }
        
        if (config.command && !Array.isArray(config.args)) {
          showModal('도구 추가 실패', `'${toolName}' 도구의 'args' 필드는 반드시 배열([]) 형식이어야 합니다.`, 'error');
          failedTools.push(toolName);
          continue;
        }

        updatedConfig[toolName] = config;
        successTools.push(toolName);
      }

      if (successTools.length > 0) {
        setPendingConfig(updatedConfig);
        setServerCount(Object.keys(updatedConfig).length);
        
        let message = '';
        if (isSmitheryFormat) {
          message = `🌐 스미더리에서 복사한 도구가 추가되었습니다.\n`;
        }
        
        if (successTools.length === 1) {
          message += `✅ ${successTools[0]} 도구가 추가되었습니다.\n\n적용하려면 '적용하기' 버튼을 눌러주세요.`;
        } else {
          message += `✅ 총 ${successTools.length}개 도구가 추가되었습니다:\n${successTools.join(', ')}\n\n적용하려면 '적용하기' 버튼을 눌러주세요.`;
        }
        
        if (failedTools.length > 0) {
          message += `\n\n⚠️ 실패한 도구: ${failedTools.join(', ')}`;
        }
        
        showModal('도구 추가 완료', message, 'success');
        setNewToolJson(JSON.stringify(defaultToolExample, null, 2));
      } else {
        showModal('도구 추가 실패', '모든 도구 추가에 실패했습니다. JSON 형식을 확인해주세요.', 'error');
      }
    } catch (error) {
      showModal('JSON 파싱 오류', `JSON 파싱 에러: ${error}\n\n스미더리에서 복사한 JSON을 그대로 붙여넣어 주세요.`, 'error');
    }
  };

  const removeTool = (toolName: string) => {
    const updatedConfig = { ...pendingConfig };
    delete updatedConfig[toolName];
    setPendingConfig(updatedConfig);
    setServerCount(Object.keys(updatedConfig).length);
  };

  const applyConfig = async () => {
    setIsLoading(true);
    try {
      // 현재 pendingConfig를 사용하여 저장 및 초기화
      setMcpConfig(pendingConfig);
      const success = await saveMCPConfig();
      if (success) {
        await initializeMCP();
      }
    } catch (error) {
      console.error('설정 적용 실패:', error);
      showModal('설정 적용 실패', '설정 적용에 실패했습니다.');
    } finally {
      setIsLoading(false);
    }
  };

  // 초기값으로 복원하는 함수
  const restoreDefaultConfig = async () => {
    setIsLoading(true);
    try {
      // mcp_config.orig.json 파일을 mcp_config.json으로 복사하고 설정 로드
      const response = await fetch('/api/mcp/config?original=true');
      const data = await response.json();
      
      if (data.success && data.config) {
        // 복원된 설정으로 상태 업데이트
        setMcpConfig(data.config);
        setPendingConfig(data.config);
        setServerCount(Object.keys(data.config).length);
        
        // 바로 MCP 초기화 실행
        showToast(`초기 설정으로 복원되었습니다. MCP를 초기화합니다...`, 'success');
        await initializeMCP();
      } else {
        showToast(data.error || '초기 설정 복원에 실패했습니다.', 'error');
      }
    } catch (error) {
      console.error('초기 설정 복원 실패:', error);
      showToast('초기 설정 복원에 실패했습니다.', 'error');
    } finally {
      setIsLoading(false);
    }
  };

  const addMessage = (role: 'user' | 'assistant' | 'tool', content: string, toolCalls?: ToolCall[], attachedFiles?: AttachedFile[]) => {
    setChatMessages(prev => [...prev, {
      role,
      content,
      timestamp: new Date(),
      toolCalls,
      attachedFiles
    }]);
  };

  // 응답 중지 함수
  const stopResponse = () => {
    if (abortControllerRef.current) {
      console.log('🛑 사용자 요청으로 응답 중지');
      abortControllerRef.current.abort();
    }
  };

  // 파일 관련 유틸리티 함수들
  const isImageFile = (file: File) => {
    return file.type.startsWith('image/');
  };

  const isPdfFile = (file: File) => {
    return file.type === 'application/pdf';
  };

  const isTextFile = (file: File) => {
    return file.type.startsWith('text/') || 
           file.type === 'application/json' ||
           file.type === 'application/xml' ||
           file.name.endsWith('.md') ||
           file.name.endsWith('.txt') ||
           file.name.endsWith('.json') ||
           file.name.endsWith('.xml') ||
           file.name.endsWith('.csv');
  };

  // 이미지 압축 함수
  const compressImage = async (file: File, maxWidth: number = 800, maxHeight: number = 800, quality: number = 0.7): Promise<File> => {
    return new Promise((resolve, reject) => {
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      const img = document.createElement('img');

      img.onload = () => {
        // 원본 크기
        const { width, height } = img;
        
        // 비율 유지하면서 크기 조정
        let newWidth = width;
        let newHeight = height;
        
        if (width > maxWidth || height > maxHeight) {
          const ratio = Math.min(maxWidth / width, maxHeight / height);
          newWidth = width * ratio;
          newHeight = height * ratio;
        }
        
        // Canvas 크기 설정
        canvas.width = newWidth;
        canvas.height = newHeight;
        
        // 이미지 그리기
        ctx?.drawImage(img, 0, 0, newWidth, newHeight);
        
        // Blob으로 변환
        canvas.toBlob(
          (blob) => {
            if (blob) {
              // 압축된 File 객체 생성
              const compressedFile = new File([blob], file.name, {
                type: file.type,
                lastModified: Date.now(),
              });
              
              console.log(`이미지 압축 완료: ${file.name}`);
              console.log(`  원본 크기: ${(file.size / 1024 / 1024).toFixed(2)}MB (${width}x${height})`);
              console.log(`  압축 후: ${(compressedFile.size / 1024 / 1024).toFixed(2)}MB (${newWidth}x${newHeight})`);
              console.log(`  압축률: ${((1 - compressedFile.size / file.size) * 100).toFixed(1)}%`);
              
              resolve(compressedFile);
            } else {
              reject(new Error('이미지 압축 실패'));
            }
          },
          file.type,
          quality
        );
      };
      
      img.onerror = () => reject(new Error('이미지 로드 실패'));
      img.src = URL.createObjectURL(file);
    });
  };

  // 이미지를 Base64로 변환
  const convertImageToBase64 = async (file: File): Promise<string> => {
    console.log(`📝 Base64 변환 시작: ${file.name} (${file.size} bytes)`);
    
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      reader.onload = () => {
        if (typeof reader.result === 'string') {
          console.log(`📄 FileReader 결과: ${reader.result.substring(0, 100)}...`);
          
          // 전체 data URL 반환 (서버에서 prefix 제거)
          console.log(`✅ Base64 변환 성공: ${reader.result.length} characters`);
          console.log(`📋 Base64 미리보기: ${reader.result.substring(0, 100)}...`);
          
          resolve(reader.result);
        } else {
          console.error('❌ Base64 변환 실패: 결과가 문자열이 아님');
          reject(new Error('Base64 변환 실패'));
        }
      };
      reader.onerror = (error) => {
        console.error('❌ Base64 변환 실패: 파일 읽기 오류', error);
        reject(new Error('파일 읽기 실패'));
      };
      reader.readAsDataURL(file);
    });
  };

  // PDF 텍스트 추출 함수
  const extractPdfText = async (file: File): Promise<string> => {
    try {
      // 동적으로 PDF.js 로드
      const pdfjsLib = await import('pdfjs-dist');
      
      // Worker 설정
      if (typeof window !== 'undefined') {
        pdfjsLib.GlobalWorkerOptions.workerSrc = '/pdf-worker/pdf.worker.min.js';
      }
      
      console.log('🔧 PDF.js 설정 확인:', pdfjsLib.GlobalWorkerOptions.workerSrc);
      
      const arrayBuffer = await file.arrayBuffer();
      
      // PDF 문서 로드
      const loadingTask = pdfjsLib.getDocument({
        data: arrayBuffer
      });
      
      const pdf = await loadingTask.promise;
      console.log(`📖 PDF 로드 완료: ${pdf.numPages}페이지`);
      
      let fullText = '';
      
      // 모든 페이지에서 텍스트 추출
      for (let pageNum = 1; pageNum <= pdf.numPages; pageNum++) {
        try {
          const page = await pdf.getPage(pageNum);
          const textContent = await page.getTextContent();
          const pageText = textContent.items
            .map((item: any) => {
              if (item.str && typeof item.str === 'string') {
                return item.str;
              }
              return '';
            })
            .filter(text => text.trim().length > 0)
            .join(' ')
            .replace(/\s+/g, ' ')
            .trim();
          
          if (pageText) {
            fullText += `\n[페이지 ${pageNum}]\n${pageText}\n`;
            console.log(`✅ 페이지 ${pageNum}: ${pageText.length}자 추출`);
          } else {
            console.log(`⚠️ 페이지 ${pageNum}: 텍스트 없음`);
          }
        } catch (pageError) {
          console.error(`❌ 페이지 ${pageNum} 처리 오류:`, pageError);
          const pageErrorMessage = pageError instanceof Error ? pageError.message : String(pageError);
          fullText += `\n[페이지 ${pageNum}]\n(이 페이지에서 텍스트를 추출할 수 없습니다: ${pageErrorMessage})\n`;
        }
      }
      
      // PDF 문서 정리
      pdf.destroy();
      
      if (fullText.trim()) {
        return fullText.trim();
      } else {
        return `PDF 파일 "${file.name}"에서 읽을 수 있는 텍스트를 찾을 수 없습니다. 이미지 기반 PDF이거나 보호된 문서일 수 있습니다.`;
      }
      
    } catch (error) {
      console.error('❌ PDF 텍스트 추출 실패:', error);
      
      // 오류 타입에 따른 상세 메시지
      const errorMessage = error instanceof Error ? error.message : String(error);
      if (errorMessage.includes('Worker')) {
        return `PDF 처리 환경 오류: ${file.name} - PDF.js Worker 로드에 실패했습니다. 네트워크 환경을 확인해주세요.`;
      } else if (errorMessage.includes('Invalid PDF')) {
        return `PDF 형식 오류: ${file.name} - 손상되었거나 유효하지 않은 PDF 파일입니다.`;
      } else if (errorMessage.includes('Password')) {
        return `PDF 보안 오류: ${file.name} - 패스워드로 보호된 PDF 파일입니다.`;
      } else {
        return `PDF 텍스트 추출 실패: ${file.name} - ${errorMessage || '알 수 없는 오류가 발생했습니다.'}`;
      }
    }
  };

  // 텍스트 파일 내용 읽기
  const readTextFile = async (file: File): Promise<string> => {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      reader.onload = () => {
        if (typeof reader.result === 'string') {
          resolve(reader.result);
        } else {
          reject(new Error('텍스트 파일 읽기 실패'));
        }
      };
      reader.onerror = () => reject(new Error('파일 읽기 실패'));
      reader.readAsText(file, 'utf-8');
    });
  };

  // 파일 크기 포맷팅
  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  const sendMessage = async () => {
    if (!currentMessage.trim() || !isInitialized) return;

    const userMessage = currentMessage;
    setCurrentMessage('');
    
    // 세션 ID가 없으면 새 세션 생성
    let sessionId = currentSessionId;
    if (!sessionId) {
      sessionId = await createNewSession();
    }
    
    // 1. 사용자 메시지 추가 (첨부된 파일 정보 포함)
    addMessage('user', userMessage, undefined, uploadedFiles);
    
    // 2. 빈 assistant 메시지 미리 추가 (순서 보장) - 이미지는 포함하지 않음
    const assistantMessageIndex = chatMessages.length + 1; // 사용자 메시지 다음 인덱스
    addMessage('assistant', '');
    
    setIsLoading(true);

    try {
      // AbortController 생성
      const abortController = new AbortController();
      abortControllerRef.current = abortController;

      // JSON 방식으로 데이터 전송 (FormData 타임아웃 문제 해결)
      const requestData = {
        message: userMessage,
        provider: selectedProvider,
        model: selectedModel,
        temperature: temperature,
        sessionId: sessionId,
        history: chatMessages.map(msg => ({
          role: msg.role,
          content: msg.content
        })),
        files: uploadedFiles.map((attachedFile, index) => {
          console.log(`📎 파일 ${index} 추가:`, {
            name: attachedFile.name,
            type: attachedFile.type,
            size: attachedFile.size,
            hasExtractedText: !!attachedFile.extractedText,
            hasBase64: !!attachedFile.base64,
            base64Length: attachedFile.base64?.length || 0
          });
          
          const fileData: any = {
            name: attachedFile.name,
            type: attachedFile.type,
            size: attachedFile.size
          };
          
          if (attachedFile.extractedText) {
            fileData.extractedText = attachedFile.extractedText;
          }
          
          if (attachedFile.base64) {
            fileData.base64 = attachedFile.base64;
            fileData.isImage = true;
          }
          
          return fileData;
        })
      };

      // console.log('📤 JSON 방식으로 요청 전송:', {
      //   messageLength: userMessage.length,
      //   fileCount: uploadedFiles.length,
      //   historyLength: chatMessages.length
      // });

      // JSON 방식으로 요청 전송
      const response = await fetch('/api/mcp/chat', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: JSON.stringify(requestData),
        signal: abortController.signal,
      });

      if (!response.ok) {
        // 미리 추가된 빈 메시지를 오류 메시지로 업데이트
        setChatMessages(prev => {
          const newMessages = [...prev];
          const lastAssistantIndex = newMessages.length - 1;
          if (lastAssistantIndex >= 0 && newMessages[lastAssistantIndex].role === 'assistant') {
            newMessages[lastAssistantIndex].content = '❌ 응답 처리 중 오류가 발생했습니다.';
          }
          return newMessages;
        });
        return;
      }

      // 스트리밍 응답 처리
      const reader = response.body?.getReader();
      if (!reader) {
        console.error('❌ Reader를 가져올 수 없습니다');
        // 미리 추가된 빈 메시지를 오류 메시지로 업데이트
        setChatMessages(prev => {
          const newMessages = [...prev];
          const lastAssistantIndex = newMessages.length - 1;
          if (lastAssistantIndex >= 0 && newMessages[lastAssistantIndex].role === 'assistant') {
            newMessages[lastAssistantIndex].content = '❌ 스트리밍 응답을 받을 수 없습니다.';
          }
          return newMessages;
        });
        return;
      }

      let buffer = '';
      let currentToolCalls: ToolCall[] = [];
      let chunkCount = 0;
      
      // 메시지 누적을 위한 ref 사용 (상태 업데이트 동기화 문제 해결)
      let accumulatedContent = '';

      try {
        while (true) {
          const { done, value } = await reader.read();
          chunkCount++;
          
          if (done) break;

          buffer += new TextDecoder().decode(value);
          const lines = buffer.split('\n');
          buffer = lines.pop() || '';

          for (const line of lines) {
            if (line.startsWith('data: ')) {
              const data = line.slice(6);
              
              if (data === '[DONE]') {
                break;
              }
              
              try {
                const parsed = JSON.parse(data);
                
                // 다양한 형태의 응답 처리
                let contentToAdd = '';
                if (parsed.chunk) {
                  contentToAdd = parsed.chunk;
                } else if (parsed.content) {
                  contentToAdd = parsed.content;
                } else if (parsed.message && parsed.message.content) {
                  contentToAdd = parsed.message.content;
                } else if (parsed.choices && parsed.choices[0] && parsed.choices[0].delta && parsed.choices[0].delta.content) {
                  contentToAdd = parsed.choices[0].delta.content;
                } else if (Array.isArray(parsed) && parsed.length > 0) {
                  // Anthropic 형태의 배열 응답 처리 [{"index":0,"type":"text","text":"..."}]
                  contentToAdd = parsed
                    .filter(item => item.type === 'text' && item.text)
                    .map(item => item.text)
                    .join('');
                } else if (parsed.type === 'text' && parsed.text) {
                  // Anthropic 단일 객체 형태 처리 {"type":"text","text":"..."}
                  contentToAdd = parsed.text;
                }
                
                if (parsed.error) {
                  console.error('🚨 응답 오류:', parsed.error);
                  // 오류 발생 시 미리 추가된 메시지를 오류 메시지로 업데이트
                  setChatMessages(prev => {
                    const newMessages = [...prev];
                    const lastAssistantIndex = newMessages.length - 1;
                    if (lastAssistantIndex >= 0 && newMessages[lastAssistantIndex].role === 'assistant') {
                      newMessages[lastAssistantIndex] = {
                        ...newMessages[lastAssistantIndex],
                        content: `❌ 오류: ${parsed.error}`
                      };
                    }
                    return newMessages;
                  });
                  break;
                }
                
                // 메시지 업데이트 - 누적된 내용으로 전체 교체 (중복 방지)
                if (contentToAdd) {
                  // 로컬 변수에 누적
                  accumulatedContent += contentToAdd;
                  
                  // 전체 메시지를 교체하여 중복 방지
                  setChatMessages(prev => {
                    const newMessages = [...prev];
                    const lastAssistantIndex = newMessages.length - 1;
                    
                    if (lastAssistantIndex >= 0 && newMessages[lastAssistantIndex].role === 'assistant') {
                      // 누적된 전체 내용으로 교체하되 첨부된 파일 정보는 유지
                      newMessages[lastAssistantIndex] = {
                        ...newMessages[lastAssistantIndex],
                        content: accumulatedContent,
                        toolCalls: [...currentToolCalls]
                      };
                    }
                    
                    return newMessages;
                  });
                }
                
              } catch (parseError) {
                console.error('JSON 파싱 오류:', parseError, '원본 데이터:', data);
              }
            } else if (line.startsWith('{') && line.endsWith('}')) {
                              // 도구 호출 이벤트 처리
              try {
                const eventData = JSON.parse(line);
                
                // 다양한 형태의 도구 호출 이벤트 처리
                if (eventData.type === 'tool_call_start' || eventData.tool_call_start) {
                  // 도구 호출 시작
                  const toolData = eventData.tool_call_start || eventData;
                  const newToolCall: ToolCall = {
                    name: toolData.toolName || toolData.name || toolData.tool_name,
                    args: toolData.args || toolData.arguments || toolData.input || {},
                    server: toolData.server || 'unknown',
                    isLoading: true
                  };
                  currentToolCalls.push(newToolCall);
                  
                  console.log('🔧 도구 호출 시작:', newToolCall.name, newToolCall.args, '서버:', newToolCall.server);
                  
                } else if (eventData.type === 'tool_call_end' || eventData.tool_call_end) {
                  // 도구 호출 완료
                  const toolData = eventData.tool_call_end || eventData;
                  const toolName = toolData.toolName || toolData.name || toolData.tool_name;
                  const toolCallIndex = currentToolCalls.findIndex(tc => tc.name === toolName && tc.isLoading);
                  if (toolCallIndex !== -1) {
                    currentToolCalls[toolCallIndex] = {
                      ...currentToolCalls[toolCallIndex],
                      result: toolData.result || toolData.output,
                      duration: toolData.duration || toolData.execution_time,
                      isLoading: false
                    };
                  }
                  
                  console.log('✅ 도구 호출 완료:', toolName, (toolData.duration || toolData.execution_time || 0) + 'ms');
                  
                } else if (eventData.type === 'tool_call_error' || eventData.tool_call_error) {
                  // 도구 호출 오류
                  const toolData = eventData.tool_call_error || eventData;
                  const toolName = toolData.toolName || toolData.name || toolData.tool_name;
                  const toolCallIndex = currentToolCalls.findIndex(tc => tc.name === toolName && tc.isLoading);
                  if (toolCallIndex !== -1) {
                    currentToolCalls[toolCallIndex] = {
                      ...currentToolCalls[toolCallIndex],
                      error: toolData.error || toolData.error_message,
                      duration: toolData.duration || toolData.execution_time,
                      isLoading: false
                    };
                  }
                  
                  console.log('❌ 도구 호출 오류:', toolName, toolData.error || toolData.error_message);
                }
                
                // 도구 호출 정보 업데이트
                if (currentToolCalls.length > 0) {
                  setChatMessages(prev => {
                    const newMessages = [...prev];
                    const lastAssistantIndex = newMessages.length - 1;
                    
                    if (lastAssistantIndex >= 0 && newMessages[lastAssistantIndex].role === 'assistant') {
                      newMessages[lastAssistantIndex].toolCalls = [...currentToolCalls];
                    }
                    
                    return newMessages;
                  });
                }
                
              } catch (parseError) {
                console.error('도구 호출 이벤트 파싱 오류:', parseError, '원본 라인:', line);
              }
            }
          }
        }
        
        // 응답 완료 후 세션 히스토리 업데이트 및 목록 새로고침
        if (sessionId) {
          // 사용자 메시지를 세션에 저장
          await updateSessionHistory(sessionId, {
            role: 'user',
            content: userMessage,
            timestamp: new Date(),
            attachedFiles: uploadedFiles.length > 0 ? uploadedFiles : undefined
          });
          
          // 누적된 어시스턴트 응답 내용을 세션에 저장
          if (accumulatedContent && accumulatedContent.trim()) {
            // console.log('어시스턴트 메시지 저장:', accumulatedContent.substring(0, 50) + '...');
            await updateSessionHistory(sessionId, {
              role: 'assistant',
              content: accumulatedContent,
              timestamp: new Date(),
              toolCalls: currentToolCalls.length > 0 ? currentToolCalls : undefined,
              attachedFiles: []
            });
          } else {
            console.log('누적된 어시스턴트 응답이 없습니다.');
          }
          
          // 세션 목록 즉시 새로고침
          loadSessions();
        }
        
      } finally {
        reader.releaseLock();
        
        // 파일 전송 완료 후 파일 목록 제거
        if (uploadedFiles.length > 0) {
          console.log(`🗑️ 업로드된 파일 ${uploadedFiles.length}개 제거`);
          setUploadedFiles([]);
        }
      }
    } catch (error) {
      // 중지된 요청인지 확인
      if ((error as Error).name === 'AbortError' || abortControllerRef.current?.signal.aborted) {
        console.log('🛑 요청이 사용자에 의해 중지되었습니다');
        
        // 미리 추가된 빈 메시지를 중지 메시지로 업데이트
        setChatMessages(prev => {
          const newMessages = [...prev];
          const lastAssistantIndex = newMessages.length - 1;
          if (lastAssistantIndex >= 0 && newMessages[lastAssistantIndex].role === 'assistant') {
            newMessages[lastAssistantIndex] = {
              ...newMessages[lastAssistantIndex],
              content: '🛑 응답이 중지되었습니다.'
            };
          }
          return newMessages;
        });
        
        showToast('응답이 중지되었습니다.', 'info');
      } else {
        console.error('❌ 메시지 전송 실패:', error);
        console.error('❌ 오류 상세:', {
          name: (error as Error).name,
          message: (error as Error).message,
          stack: (error as Error).stack,
          cause: (error as any).cause
        });
        
        // 미리 추가된 빈 메시지를 오류 메시지로 업데이트
        setChatMessages(prev => {
          const newMessages = [...prev];
          const lastAssistantIndex = newMessages.length - 1;
          if (lastAssistantIndex >= 0 && newMessages[lastAssistantIndex].role === 'assistant') {
            newMessages[lastAssistantIndex] = {
              ...newMessages[lastAssistantIndex],
              content: '❌ 메시지 전송에 실패했습니다.'
            };
          }
          return newMessages;
        });
        
        showToast('메시지 전송에 실패했습니다.', 'error');
      }
    } finally {
      setIsLoading(false);
      abortControllerRef.current = null;
    }
  };

  // 파일 처리 함수
  const processFile = async (file: File): Promise<AttachedFile> => {
    console.log(`🔄 파일 처리 시작: ${file.name} (${file.type})`);
    
    const attachedFile: AttachedFile = {
      file,
      name: file.name,
      type: file.type,
      size: file.size,
      isProcessing: true
    };

    try {
      if (isImageFile(file)) {
        console.log(`📸 이미지 파일 처리: ${file.name}`);
        
        // 이미지 압축
        const compressedFile = await compressImage(file);
        console.log(`✅ 이미지 압축 완료: ${file.size} → ${compressedFile.size} bytes`);
        
        // Base64 변환
        const base64 = await convertImageToBase64(compressedFile);
        console.log(`✅ Base64 변환 완료: ${base64.length} characters`);
        console.log(`📋 Base64 미리보기: ${base64.substring(0, 100)}...`);
        
        attachedFile.file = compressedFile;
        attachedFile.base64 = base64;
        attachedFile.url = URL.createObjectURL(compressedFile);
      } else if (isPdfFile(file)) {
        // PDF 텍스트 추출
        const extractedText = await extractPdfText(file);
        attachedFile.extractedText = extractedText;
      } else if (isTextFile(file)) {
        // 텍스트 파일 읽기
        const textContent = await readTextFile(file);
        attachedFile.extractedText = textContent;
      }
      
      attachedFile.isProcessing = false;
      return attachedFile;
    } catch (error) {
      console.error('파일 처리 실패:', error);
      attachedFile.isProcessing = false;
      attachedFile.processingError = error instanceof Error ? error.message : '파일 처리 중 오류가 발생했습니다.';
      return attachedFile;
    }
  };

  // 파일 업로드 핸들러
  const handleFileUpload = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = event.target.files;
    if (!files || files.length === 0) return;

    setIsProcessingFiles(true);
    
    try {
      const fileArray = Array.from(files);
      const processedFiles: AttachedFile[] = [];
      
      // 파일들을 순차적으로 처리
      for (const file of fileArray) {
        const processedFile = await processFile(file);
        processedFiles.push(processedFile);
      }
      
      setUploadedFiles(prev => [...prev, ...processedFiles]);
      
      // 파일 입력 초기화
      if (fileInputRef.current) {
        fileInputRef.current.value = '';
      }
    } catch (error) {
      console.error('파일 업로드 실패:', error);
      showToast('파일 업로드 중 오류가 발생했습니다.', 'error');
    } finally {
      setIsProcessingFiles(false);
    }
  };

  // 파일 제거
  const removeFile = (index: number) => {
    setUploadedFiles(prev => {
      const fileToRemove = prev[index];
      // URL 정리
      if (fileToRemove?.url) {
        URL.revokeObjectURL(fileToRemove.url);
      }
      return prev.filter((_, i) => i !== index);
    });
  };

  // 드래그 앤 드롭 핸들러들
  const handleDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOver(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    // 실제로 영역을 벗어났는지 확인
    const rect = e.currentTarget.getBoundingClientRect();
    const x = e.clientX;
    const y = e.clientY;
    if (x < rect.left || x > rect.right || y < rect.top || y > rect.bottom) {
      setIsDragOver(false);
    }
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  const handleDrop = async (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOver(false);
    
    const files = e.dataTransfer.files;
    if (files.length > 0) {
      setIsProcessingFiles(true);
      
      try {
        const fileArray = Array.from(files);
        const processedFiles: AttachedFile[] = [];
        
        // 파일들을 순차적으로 처리
        for (const file of fileArray) {
          const processedFile = await processFile(file);
          processedFiles.push(processedFile);
        }
        
        setUploadedFiles(prev => [...prev, ...processedFiles]);
      } catch (error) {
        console.error('파일 드롭 실패:', error);
        showToast('파일 처리 중 오류가 발생했습니다.', 'error');
      } finally {
        setIsProcessingFiles(false);
      }
    }
  };

  // 음성 처리 함수
  const processAudio = async (audioBlob: Blob, contentType: string) => {
    setIsProcessingAudio(true);
    
    const whisperApiUrl = "/api/stt";

    try {
      const res = await fetch(whisperApiUrl, {
        method: "POST",
        headers: {
          'Content-Type': contentType,
        },
        body: audioBlob,
      });

      if (res.ok) {
        const result = await res.json();
        const recognizedText = result.text || "";
        if (recognizedText.trim()) {
          setCurrentMessage(prev => prev + (prev ? ' ' : '') + recognizedText);
          showToast('음성 인식이 완료되었습니다.', 'success');
        } else {
          showToast('음성이 인식되지 않았습니다.', 'error');
        }
      } else {
        const errorText = await res.text();
        console.error(`서버 응답 오류: ${res.status} ${res.statusText}. 내용: ${errorText}`);
        showToast('음성 인식 중 오류가 발생했습니다.', 'error');
      }
    } catch (error) {
      console.error("Whisper API fetch error:", error);
      showToast('음성 인식 서버에 연결할 수 없습니다.', 'error');
    } finally {
      setIsProcessingAudio(false);
    }
  };

  // 음성 녹음 시작/중지 함수
  const toggleSpeechRecognition = async () => {
    if (isListening) {
      // 녹음 중지
      if (mediaRecorderRef.current) {
        mediaRecorderRef.current.stop();
        // 스트림의 모든 트랙을 중지하여 마이크 사용 표시를 끔
        mediaRecorderRef.current.stream.getTracks().forEach(track => track.stop());
      }
      setIsListening(false);
    } else {
      // 녹음 시작
      try {
        const stream = await navigator.mediaDevices.getUserMedia({ audio: true });
        const mediaRecorder = new MediaRecorder(stream);

        audioChunksRef.current = [];
        mediaRecorder.ondataavailable = (event) => {
          if (event.data.size > 0) {
            audioChunksRef.current.push(event.data);
          }
        };

        mediaRecorder.onstop = async () => {
          const audioBlob = new Blob(audioChunksRef.current, { type: "audio/webm" });
          await processAudio(audioBlob, "audio/webm");
        };

        mediaRecorder.start();
        mediaRecorderRef.current = mediaRecorder;
        setIsListening(true);
        showToast('음성 녹음을 시작합니다. 말씀해 주세요.', 'success');
      } catch (error) {
        console.error("getUserMedia error:", error);
        showToast('마이크에 접근할 수 없습니다. 브라우저의 마이크 권한을 확인해주세요.', 'error');
      }
    }
  };

  const clearChat = () => {
    setChatMessages([]);
    setCurrentSessionId(null);
    // 업로드된 파일들의 URL 정리
    uploadedFiles.forEach(file => {
      if (file.url) {
        URL.revokeObjectURL(file.url);
      }
    });
    setUploadedFiles([]);
  };

  // 세션 관리 함수들 추가
  const loadSessions = async (): Promise<AssistantSession[]> => {
    try {
      setIsLoadingSessions(true);
      
      // 인증 헤더 포함하여 API 요청
      const authHeaders = getAuthHeaders();
      // console.log('🔐 어시스턴트 세션 로드 - 인증 헤더:', {
      //   hasAuthorization: !!authHeaders['Authorization'],
      //   hasApiKey: !!authHeaders['X-API-Key'],
      //   authorizationPrefix: authHeaders['Authorization']?.substring(0, 20) + '...',
      //   apiKeyPrefix: authHeaders['X-API-Key']?.substring(0, 10) + '...'
      // });
      
      const response = await fetch('/api/assistant/sessions', {
        method: 'GET',
        headers: authHeaders,
      });

      if (!response.ok) {
        throw new Error(`세션 목록 조회 실패: ${response.statusText}`);
      }

      const data = await response.json();
      
      if (!data.success) {
        throw new Error(data.error || '세션 목록 조회 실패');
      }

      const assistantSessions = data.data || [];
      setSessions(assistantSessions);
      // console.log(`어시스턴트 세션 로드 완료: ${assistantSessions.length}개 세션`);
      return assistantSessions;
    } catch (error) {
      console.error('세션 로드 실패:', error);
      setSessions([]);
      return [];
    } finally {
      setIsLoadingSessions(false);
    }
  };

  // 세션 제목 생성 함수
  const generateSessionTitle = (history: any[]): string => {
    if (!history || history.length === 0) return '새 대화';
    
    const firstUserMessage = history.find(msg => msg.role === 'user' && msg.content);
    if (firstUserMessage && firstUserMessage.content) {
      let content = firstUserMessage.content.trim();
      
      // 특수 문자 제거 및 정리
      content = content.replace(/[^\w\s가-힣]/g, ' ').replace(/\s+/g, ' ').trim();
      
      // 질문 형태로 변환
      if (content.length > 30) {
        content = content.substring(0, 30) + '...';
      }
      
      // 빈 문자열이면 기본값 반환
      if (!content) {
        return '새 대화';
      }
      
      return content;
    }
    
    return '새 대화';
  };

  // 마지막 메시지 가져오기
  const getLastMessage = (history: any[]): string => {
    if (!history || history.length === 0) return '새 대화';
    
    // 마지막 assistant 메시지 우선 찾기
    const lastAssistantMessage = [...history].reverse().find(msg => msg.role === 'assistant' && msg.content);
    if (lastAssistantMessage && lastAssistantMessage.content) {
      let content = lastAssistantMessage.content.trim();
      
      // 마크다운 문법 제거
      content = content.replace(/[#*`_~\[\]()]/g, '').replace(/\s+/g, ' ').trim();
      
      if (content.length > 50) {
        content = content.substring(0, 50) + '...';
      }
      
      return content || '새 대화';
    }
    
    // assistant 메시지가 없으면 마지막 user 메시지
    const lastUserMessage = [...history].reverse().find(msg => msg.role === 'user' && msg.content);
    if (lastUserMessage && lastUserMessage.content) {
      let content = lastUserMessage.content.trim();
      content = content.replace(/[^\w\s가-힣]/g, ' ').replace(/\s+/g, ' ').trim();
      
      if (content.length > 50) {
        content = content.substring(0, 50) + '...';
      }
      
      return content || '새 대화';
    }
    
    return '새 대화';
  };

  // 새 세션 생성 - API 사용
  const createNewSession = async (): Promise<string> => {
    try {
      const authHeaders = getAuthHeaders();
      const response = await fetch('/api/assistant/sessions', {
        method: 'POST',
        headers: authHeaders,
        body: JSON.stringify({
          provider: selectedProvider,
          model: selectedModel,
          temperature: temperature,
        }),
      });

      if (!response.ok) {
        throw new Error(`세션 생성 실패: ${response.statusText}`);
      }

      const data = await response.json();
      
      if (!data.success) {
        throw new Error(data.error || '세션 생성 실패');
      }

      const newSessionId = data.data.sessionId;
      
      // 현재 대화 초기화
      setChatMessages([]);
      setCurrentSessionId(newSessionId);
      
      // 업로드된 파일들의 URL 정리
      uploadedFiles.forEach(file => {
        if (file.url) {
          URL.revokeObjectURL(file.url);
        }
      });
      setUploadedFiles([]);
      
      // console.log('새 어시스턴트 세션 생성 완료:', newSessionId);
      // 참고: 빈 세션은 목록에 표시되지 않으므로 loadSessions() 호출하지 않음
      
      return newSessionId;
    } catch (error) {
      console.error('새 세션 생성 실패:', error);
      throw new Error('세션 생성 실패: ' + (error instanceof Error ? error.message : 'Unknown error'));
    }
  };

  // 세션 로드 - localStorage 사용
  const loadSession = async (sessionId: string) => {
    try {
      console.log('어시스턴트 세션 로드 시작:', sessionId);
      
      const authHeaders = getAuthHeaders();
      const response = await fetch(`/api/assistant/sessions?sessionId=${sessionId}`, {
        method: 'GET',
        headers: authHeaders,
      });

      if (!response.ok) {
        throw new Error(`세션 로드 실패: ${response.statusText}`);
      }

      const data = await response.json();
      
      if (!data.success) {
        throw new Error(data.error || '세션 로드 실패');
      }

      const sessionData = data.data;
      
      if (sessionData) {
        // 세션 히스토리를 ChatMessage 형태로 변환
        const history = sessionData.history?.map((msg: any) => ({
          role: msg.role,
          content: msg.content,
          timestamp: new Date(msg.timestamp || new Date()),
          toolCalls: msg.toolCalls,
          attachedFiles: msg.attachedFiles
        })) || [];
        
        setChatMessages(history);
        setCurrentSessionId(sessionId);
        setSelectedProvider(sessionData.provider || 'ollama');
        setSelectedModel(sessionData.model || 'hamonize:latest');
        
        // 업로드된 파일들의 URL 정리
        uploadedFiles.forEach(file => {
          if (file.url) {
            URL.revokeObjectURL(file.url);
          }
        });
        setUploadedFiles([]);
        
        console.log('어시스턴트 세션 로드 완료:', sessionId);
      } else {
        console.error('세션을 찾을 수 없습니다:', sessionId);
      }
    } catch (error) {
      console.error('세션 로드 중 오류:', error);
      showToast('세션을 로드할 수 없습니다.', 'error');
    }
  };

  // 세션 삭제 - localStorage 사용
  const deleteSession = async (sessionId: string) => {
    try {
      // console.log('🗑️ 어시스턴트 세션 삭제 시작:', sessionId);
      
      const authHeaders = getAuthHeaders();
      const response = await fetch(`/api/assistant/sessions?sessionId=${sessionId}`, {
        method: 'DELETE',
        headers: authHeaders,
      });

      if (!response.ok) {
        throw new Error(`세션 삭제 실패: ${response.statusText}`);
      }

      const data = await response.json();
      
      if (!data.success) {
        throw new Error(data.error || '세션 삭제 실패');
      }
      
      // 세션 목록에서 제거
      setSessions(prev => prev.filter(s => s.id !== sessionId));
      
      // 현재 세션이 삭제된 세션이면 새 세션 생성
      if (currentSessionId === sessionId) {
        await createNewSession();
      }
      
      showToast('어시스턴트 세션이 삭제되었습니다.', 'success');
    } catch (error) {
      console.error('세션 삭제 중 오류:', error);
      showToast('세션 삭제 중 오류가 발생했습니다.', 'error');
    }
  };

  // 세션 히스토리 업데이트 함수 추가
  const updateSessionHistory = async (sessionId: string, newMessage: ChatMessage) => {
    try {
      // 세션 제목 생성 (첫 번째 사용자 메시지인 경우)
      let title = undefined;
      if (newMessage.role === 'user' && chatMessages.length === 0) {
        title = generateSessionTitle([newMessage]);
      }

      const authHeaders = getAuthHeaders();
      const response = await fetch('/api/assistant/sessions', {
        method: 'PUT',
        headers: authHeaders,
        body: JSON.stringify({
          sessionId,
          newMessage,
          title
        }),
      });

      if (!response.ok) {
        throw new Error(`세션 업데이트 실패: ${response.statusText}`);
      }

      const data = await response.json();
      
      if (!data.success) {
        throw new Error(data.error || '세션 업데이트 실패');
      }

      console.log(`어시스턴트 세션 ${sessionId} 업데이트 완료:`, {
        messageRole: newMessage.role,
        messageContent: newMessage.content.substring(0, 50) + '...'
      });
    } catch (error) {
      console.error('세션 히스토리 업데이트 실패:', error);
    }
  };

  return (
    <div className="h-screen bg-gray-50 dark:bg-gray-900 flex flex-col">
      {/* 헤더 */}
      <div className="bg-[var(--sidebar-bg)] shadow-sm border-b border-[var(--sidebar-border)] flex-shrink-0">
        <div className="max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6">
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-4">
              <div className="flex items-center space-x-3">
                <div className="w-12 h-12 bg-purple-600 rounded-lg flex items-center justify-center">
                  <span className="text-2xl">🔌</span>
                </div>
                <div>
                  <h3 className="text-base font-bold text-gray-900 dark:text-white">
                    AI Assistant
                  </h3>
                  <p className="text-sm text-gray-500 dark:text-gray-400">
                    MCP 도구와 A2A 네트워크를 활용한 지능형 AI 어시스턴트
                  </p>
                </div>
              </div>
            </div>
            <div className="flex items-center space-x-4">
              <div className="text-right">
                <div className="text-sm font-medium text-gray-900 dark:text-white">
                  🛠️ MCP 서버 수: {serverCount}개
                  {totalToolCount > 0 && ` | 도구 수: ${totalToolCount}개`}
                  {isA2aEnabled && (
                    <>
                      {' | '}
                      <span className="inline-flex items-center space-x-1">
                        <span>🤝 A2A</span>
                        <div className={`w-2 h-2 rounded-full ${
                          a2aNetworkStatus.isConnected 
                            ? a2aNetworkStatus.networkHealth === 'good' 
                              ? 'bg-green-500' 
                              : 'bg-yellow-500'
                            : 'bg-red-500'
                        }`}></div>
                        <span>{a2aNetworkStatus.connectedAgents}개 연결</span>
                      </span>
                    </>
                  )}
                </div>
                <div className="text-xs text-gray-500 dark:text-gray-400">
                  {isInitializing ? 'MCP 초기화 중...' : isInitialized ? 'MCP 초기화 완료' : 'MCP 초기화 필요'}
                  {isA2aEnabled && (
                    <>
                      {' | '}
                      <span className={`${
                        a2aNetworkStatus.isConnected ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'
                      }`}>
                        A2A {a2aNetworkStatus.isConnected ? '활성' : '비활성'}
                      </span>
                    </>
                  )}
                </div>
              </div>
            </div>
          </div>
        </div>
      </div>

      <div className="flex-1 max-w-7xl mx-auto px-4 sm:px-6 lg:px-8 py-6 w-full overflow-hidden">
        <div className="grid grid-cols-1 lg:grid-cols-4 gap-6 h-full min-h-0">
          {/* 왼쪽 사이드바: MCP 도구 관리 및 세션 목록 */}
          <div className="lg:col-span-1 space-y-6 overflow-y-auto pr-4">
            
            {/* 세션 목록 */}
            <div className="bg-[var(--body-bg)] rounded-lg flex-1 flex flex-col min-h-0 overflow-hidden">
              <div className="w-full px-4 py-3 flex items-center justify-between hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors flex-shrink-0">
                <div 
                  onClick={() => setSessionListCollapsed(!sessionListCollapsed)}
                  className="flex items-center gap-2 cursor-pointer flex-1"
                >
                  <p className="text-sm font-medium text-gray-900 dark:text-white">
                    💬 이전 대화 ({sessions.length})
                  </p>
                  <svg className={`w-4 h-4 transition-transform ${sessionListCollapsed ? '' : 'rotate-180'}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                  </svg>
                </div>
                <button
                  onClick={createNewSession}
                  className="px-3 py-1 text-xs bg-blue-600 hover:bg-blue-700 text-white rounded transition-colors"
                  title="새 대화 시작"
                >
                  새 대화
                </button>
              </div>
              
              {!sessionListCollapsed && (
                <div className="px-4 pb-4 border-t border-gray-200 dark:border-gray-600 flex-1 flex flex-col min-h-0">
                  {isLoadingSessions ? (
                    <div className="flex items-center justify-center py-8">
                      <div className="animate-spin rounded-full h-6 w-6 border-b-2 border-blue-600"></div>
                    </div>
                  ) : sessions.length === 0 ? (
                    <p className="text-gray-500 dark:text-gray-400 text-center py-4 text-sm">
                      아직 대화가 없습니다.
                    </p>
                  ) : (
                    <div className="space-y-2 overflow-y-auto mt-3">
                      {sessions.map((session) => (
                        <div 
                          key={session.id} 
                          className={`group flex items-center justify-between p-3 rounded border cursor-pointer transition-colors ${
                            currentSessionId === session.id 
                              ? 'bg-blue-50 dark:bg-blue-900/20 border-blue-200 dark:border-blue-700' 
                              : 'bg-white dark:bg-gray-800 border-gray-200 dark:border-gray-600 hover:bg-gray-50 dark:hover:bg-gray-700'
                          }`}
                          onClick={() => loadSession(session.id)}
                        >
                          <div className="flex-1 min-w-0">
                            <div className="text-sm font-medium text-gray-900 dark:text-white truncate">
                              {session.title}
                            </div>
                            {/* <div className="text-xs text-gray-500 dark:text-gray-400 truncate">
                              {session.lastMessage}
                            </div> */}
                            <div className="text-xs text-gray-400 dark:text-gray-500 mt-1">
                              {new Date(session.updatedAt).toLocaleDateString()}
                            </div>
                          </div>
                          <button
                            onClick={(e) => {
                              e.stopPropagation();
                              deleteSession(session.id);
                            }}
                            className="opacity-0 group-hover:opacity-100 text-red-600 hover:text-red-800 p-1 ml-2 transition-opacity"
                            title="세션 삭제"
                          >
                            🗑️
                          </button>
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              )}
            </div>
            
            {/* 프로바이더 및 모델 설정 */}
            <div className="bg-[var(--body-bg)] rounded-lg overflow-hidden">
              <button
                onClick={() => setChatSettingsCollapsed(!chatSettingsCollapsed)}
                className="w-full px-4 py-3 flex items-center justify-between text-left hover:bg-gray-100 dark:hover:bg-gray-600 transition-colors"
              >
                <h4 className="text-sm font-medium text-gray-900 dark:text-white">
                  대화 설정
                </h4>
                <svg className={`w-4 h-4 transition-transform ${chatSettingsCollapsed ? '' : 'rotate-180'}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                </svg>
              </button>
              
              {!chatSettingsCollapsed && (
                <div className="px-4 pb-4 border-t border-gray-200 dark:border-gray-600">
                  <div className="space-y-3 mt-3">
                    <div>
                      <select
                        value={selectedProvider}
                        onChange={(e) => handleProviderChange(e.target.value)}
                        className="text-sm w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-800 dark:text-white"
                        disabled={isLoadingProviders}
                      >
                        {isLoadingProviders ? (
                          <option value="">로딩 중...</option>
                        ) : (
                          availableProviders.map(provider => (
                            <option key={provider.key} value={provider.key}>
                              {provider.name}
                            </option>
                          ))
                        )}
                      </select>
                    </div>
                    <div>
                      <select
                        value={selectedModel}
                        onChange={(e) => handleModelChange(e.target.value)}
                        className="text-sm w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-800 dark:text-white"
                        disabled={isLoadingProviders}
                      >
                        {isLoadingProviders ? (
                          <option value="">로딩 중...</option>
                        ) : (
                          providers[selectedProvider]?.models.map(model => (
                            <option key={model.id} value={model.id}>
                              {model.name}
                            </option>
                          ))
                        )}
                      </select>
                    </div>
                    <div>
                      <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                        Temperature: <span className="font-bold text-blue-600 dark:text-blue-400">{temperature}</span>
                      </label>
                      <div className="relative">
                        <input
                          type="range"
                          min="0"
                          max="2"
                          step="0.1"
                          value={temperature}
                          onChange={(e) => handleTemperatureChange(parseFloat(e.target.value))}
                          className="w-full h-3 rounded-lg appearance-none cursor-pointer focus:outline-none focus:ring-2 focus:ring-blue-500"
                          style={{
                            background: `linear-gradient(to right, 
                              #3b82f6 0%, 
                              #3b82f6 ${(temperature / 2) * 100}%, 
                              #e5e7eb ${(temperature / 2) * 100}%, 
                              #e5e7eb 100%)`
                          }}
                        />
                      </div>
                    </div>
                  </div>
                </div>
              )}
            </div>
            
            {/* Smithery 웹사이트 링크 추가 */}
            <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
              <div className="flex items-center justify-between">
                <div>
                  <h4 className="text-sm text-blue-900 dark:text-blue-100">
                  <a
                  href="https://smithery.ai/"
                  target="_blank"
                  rel="noopener noreferrer"
                  className="text-base text-blue-900 dark:text-blue-100"
                  >
                    🌐 Smithery
                  </a>
                  </h4>
                  <p className="text-xs text-blue-700 dark:text-blue-300 mt-1">
                    MCP 도구를 제공하는 커뮤니티 플랫폼
                  </p>
                </div>
              </div>
            </div>

            {/* 기존 도구 추가 섹션 */}
            <div className="bg-[var(--body-bg)] rounded-lg overflow-hidden">
              <button
                onClick={() => setAddToolCollapsed(!addToolCollapsed)}
                className="w-full px-4 py-3 flex items-center justify-between text-left hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
              >
                <h4 className="text-sm font-medium text-gray-900 dark:text-white">
                  새 도구 추가
                </h4>
                <svg className={`w-4 h-4 transition-transform ${addToolCollapsed ? '' : 'rotate-180'}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                </svg>
              </button>
              
              {!addToolCollapsed && (
                <div className="px-4 pb-4 border-t border-gray-200 dark:border-gray-700">
                  <div className="space-y-3 mt-3">
                    <div>
                      <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-1">
                        도구 설정 (JSON 형식)
                      </label>
                      <textarea
                        value={newToolJson}
                        onChange={(e) => setNewToolJson(e.target.value)}
                        className="text-sm w-full h-64 px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 dark:bg-gray-800 dark:text-white"
                        placeholder={JSON.stringify(defaultToolExample, null, 2)}
                      />
                    </div>
                    <button
                      onClick={addTool}
                      className="w-full bg-blue-600 text-white py-2 px-4 rounded-md hover:bg-blue-700 transition-colors"
                    >
                      도구 추가
                    </button>
                  </div>
                </div>
              )}
            </div>
            
            {/* 등록된 도구 목록 */}
            <div className="bg-[var(--body-bg)] rounded-lg flex-1 flex flex-col min-h-0 overflow-hidden">
              <div className="w-full px-4 py-3 flex items-center justify-between hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors flex-shrink-0">
                <div 
                  onClick={() => setToolListCollapsed(!toolListCollapsed)}
                  className="flex items-center gap-2 cursor-pointer flex-1"
                >
                  <h4 className="text-sm font-medium text-gray-900 dark:text-white">
                    도구 목록
                  </h4>
                  <svg className={`w-4 h-4 transition-transform ${toolListCollapsed ? '' : 'rotate-180'}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                  </svg>
                </div>
                <div className="flex items-center gap-2">
                  <button
                    onClick={restoreDefaultConfig}
                    disabled={isLoading}
                    className="px-3 py-1 text-xs bg-gray-600 hover:bg-gray-700 disabled:bg-gray-400 text-white rounded transition-colors"
                    title="MCP 설정 초기화"
                  >
                    {isLoading ? '로딩중...' : '초기화'}
                  </button>
                  {Object.keys(pendingConfig).length > 0 && (
                    <button
                      onClick={applyConfig}
                      disabled={isLoading}
                      className="px-3 py-1 text-xs bg-purple-600 hover:bg-purple-700 disabled:bg-gray-400 text-white rounded transition-colors"
                    >
                      {isLoading ? '로딩중...' : '적용하기'}
                    </button>
                  )}
                </div>
              </div>
              
              {!toolListCollapsed && (
                <div className="px-4 pb-4 border-t border-gray-200 dark:border-gray-600 flex-1 flex flex-col min-h-0">
                  {Object.keys(pendingConfig).length === 0 ? (
                    <p className="text-gray-500 dark:text-gray-400 text-center py-4">
                      등록된 도구가 없습니다.
                    </p>
                  ) : (
                    <div className="space-y-2 overflow-y-auto mt-3">
                      {Object.entries(pendingConfig).map(([name, config]) => {
                        const serverToolCount = serverTools.find(s => s.serverName === name)?.toolCount || 0;
                        // 디버깅용 로그
                        if (process.env.NODE_ENV === 'development') {
                          console.log(`서버 "${name}" 도구 수 확인:`, {
                            serverName: name,
                            serverToolCount,
                            availableServers: serverTools.map(s => ({ name: s.serverName, count: s.toolCount }))
                          });
                        }
                        return (
                          <div key={name} className="flex items-center justify-between p-2 bg-white dark:bg-gray-800 rounded border">
                            <div className="flex-1">
                              <div className="flex items-center justify-between">
                                <span className="text-sm text-gray-900 dark:text-white">{name}</span>
                                {serverToolCount > 0 && (
                                  <span className="text-xs bg-blue-100 dark:bg-blue-900 text-blue-800 dark:text-blue-200 px-2 py-1 rounded-full">
                                    {serverToolCount}
                                  </span>
                                )}
                              </div>
                              <span className="text-sm text-gray-500 dark:text-gray-400">
                                ({config.transport || 'stdio'})
                              </span>
                            </div>
                            <button
                              onClick={() => removeTool(name)}
                              className="text-red-600 hover:text-red-800 p-1 ml-2"
                              title="도구 삭제"
                            >
                              🗑️
                            </button>
                          </div>
                        );
                      })}
                    </div>
                  )}
                </div>
              )}
            </div>

            {/* A2A 네트워크 설정 */}
            <div className="bg-[var(--body-bg)] rounded-lg overflow-hidden">
              <button
                onClick={() => setA2aCollapsed(!a2aCollapsed)}
                className="w-full px-4 py-3 flex items-center justify-between text-left hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
              >
                <div className="flex items-center space-x-2">
                  <h4 className="text-sm font-medium text-gray-900 dark:text-white">
                    🤝 A2A 네트워크
                  </h4>
                  <div className={`w-2 h-2 rounded-full ${
                    a2aNetworkStatus.isConnected 
                      ? a2aNetworkStatus.networkHealth === 'good' 
                        ? 'bg-green-500' 
                        : 'bg-yellow-500'
                      : 'bg-red-500'
                  }`}></div>
                </div>
                <svg className={`w-4 h-4 transition-transform ${a2aCollapsed ? '' : 'rotate-180'}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                </svg>
              </button>
              
              {!a2aCollapsed && (
                <div className="px-4 pb-4 border-t border-gray-200 dark:border-gray-600">
                  <div className="space-y-4 mt-3">
                    {/* A2A 네트워크 상태 */}
                    <div className="bg-white dark:bg-gray-800 rounded-lg p-3 border">
                      <h5 className="text-xs font-medium text-gray-700 dark:text-gray-300 mb-2">네트워크 상태</h5>
                      <div className="space-y-2">
                        <div className="flex justify-between items-center">
                          <span className="text-xs text-gray-600 dark:text-gray-400">연결 상태:</span>
                          <span className={`text-xs font-medium ${
                            a2aNetworkStatus.isConnected ? 'text-green-600 dark:text-green-400' : 'text-red-600 dark:text-red-400'
                          }`}>
                            {a2aNetworkStatus.isConnected ? '연결됨' : '연결 끊김'}
                          </span>
                        </div>
                        <div className="flex justify-between items-center">
                          <span className="text-xs text-gray-600 dark:text-gray-400">연결된 에이전트:</span>
                          <span className="text-xs font-medium text-gray-900 dark:text-white">
                            {a2aNetworkStatus.connectedAgents}개
                          </span>
                        </div>
                        <div className="flex justify-between items-center">
                          <span className="text-xs text-gray-600 dark:text-gray-400">노출된 도구:</span>
                          <span className="text-xs font-medium text-gray-900 dark:text-white">
                            {a2aNetworkStatus.exposedTools}개
                          </span>
                        </div>
                        {a2aNetworkStatus.lastSync && (
                          <div className="flex justify-between items-center">
                            <span className="text-xs text-gray-600 dark:text-gray-400">마지막 동기화:</span>
                            <span className="text-xs text-gray-500 dark:text-gray-400">
                              {a2aNetworkStatus.lastSync.toLocaleTimeString()}
                            </span>
                          </div>
                        )}
                      </div>
                    </div>

                    {/* A2A 네트워크 제어 */}
                    <div className="space-y-2">
                      <button
                        onClick={toggleA2aNetwork}
                        disabled={isA2aLoading}
                        className={`w-full px-3 py-2 text-sm rounded-md transition-colors ${
                          isA2aEnabled
                            ? 'bg-red-100 dark:bg-red-900/30 text-red-700 dark:text-red-300 hover:bg-red-200 dark:hover:bg-red-900/50'
                            : 'bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300 hover:bg-green-200 dark:hover:bg-green-900/50'
                        } disabled:opacity-50`}
                      >
                        {isA2aLoading ? '처리 중...' : isA2aEnabled ? 'A2A 비활성화' : 'A2A 활성화'}
                      </button>
                      
                      <button
                        onClick={testA2aConnection}
                        disabled={isA2aLoading}
                        className="w-full px-3 py-2 text-sm bg-blue-100 dark:bg-blue-900/30 text-blue-700 dark:text-blue-300 rounded-md hover:bg-blue-200 dark:hover:bg-blue-900/50 transition-colors disabled:opacity-50"
                      >
                        {isA2aLoading ? '테스트 중...' : '연결 테스트'}
                      </button>
                    </div>

                    {/* 새 에이전트 등록 */}
                    {isA2aEnabled && (
                      <div className="space-y-2">
                        <h5 className="text-xs font-medium text-gray-700 dark:text-gray-300">외부 에이전트 등록</h5>
                        <input
                          type="text"
                          placeholder="에이전트 이름"
                          value={newAgentName}
                          onChange={(e) => setNewAgentName(e.target.value)}
                          className="w-full px-2 py-1 text-xs border border-gray-300 dark:border-gray-600 rounded focus:outline-none focus:ring-1 focus:ring-blue-500 dark:bg-gray-800 dark:text-white"
                        />
                        <input
                          type="text"
                          placeholder="에이전트 엔드포인트 (http://...)"
                          value={newAgentEndpoint}
                          onChange={(e) => setNewAgentEndpoint(e.target.value)}
                          className="w-full px-2 py-1 text-xs border border-gray-300 dark:border-gray-600 rounded focus:outline-none focus:ring-1 focus:ring-blue-500 dark:bg-gray-800 dark:text-white"
                        />
                        <button
                          onClick={registerA2aAgent}
                          disabled={isA2aLoading || !newAgentName.trim() || !newAgentEndpoint.trim()}
                          className="w-full px-3 py-2 text-xs bg-purple-100 dark:bg-purple-900/30 text-purple-700 dark:text-purple-300 rounded-md hover:bg-purple-200 dark:hover:bg-purple-900/50 transition-colors disabled:opacity-50"
                        >
                          에이전트 등록
                        </button>
                      </div>
                    )}

                    {/* 연결된 에이전트 목록 */}
                    {isA2aEnabled && a2aAgents.length > 0 && (
                      <div className="space-y-2">
                        <h5 className="text-xs font-medium text-gray-700 dark:text-gray-300">연결된 에이전트</h5>
                        <div className="space-y-1 max-h-32 overflow-y-auto">
                          {a2aAgents.map((agent) => (
                            <div key={agent.id} className="flex items-center justify-between p-2 bg-white dark:bg-gray-800 rounded border text-xs">
                              <div className="flex-1 min-w-0">
                                <div className="flex items-center space-x-2">
                                  <span className="font-medium text-gray-900 dark:text-white truncate">
                                    {agent.name}
                                  </span>
                                  <div className={`w-1.5 h-1.5 rounded-full ${
                                    agent.status === 'connected' ? 'bg-green-500' :
                                    agent.status === 'error' ? 'bg-red-500' : 'bg-gray-500'
                                  }`}></div>
                                </div>
                                <div className="text-gray-500 dark:text-gray-400 truncate">
                                  {agent.endpoint}
                                </div>
                              </div>
                              <span className={`px-1.5 py-0.5 rounded text-xs ${
                                agent.trustLevel === 'high' ? 'bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300' :
                                agent.trustLevel === 'medium' ? 'bg-yellow-100 dark:bg-yellow-900/30 text-yellow-700 dark:text-yellow-300' :
                                'bg-red-100 dark:bg-red-900/30 text-red-700 dark:text-red-300'
                              }`}>
                                {agent.trustLevel}
                              </span>
                            </div>
                          ))}
                        </div>
                      </div>
                    )}
                  </div>
                </div>
              )}
            </div>
          </div>

          {/* 오른쪽: 채팅 인터페이스 */}
          <div className="lg:col-span-3 flex flex-col min-h-0">
            <div 
              className={`bg-white dark:bg-gray-800 rounded-lg shadow flex flex-col h-full max-h-[calc(90vh-12rem)] relative ${
                isDragOver ? 'ring-2 ring-blue-500 ring-opacity-50 bg-blue-50 dark:bg-blue-900/20' : ''
              }`}
              onDragEnter={handleDragEnter}
              onDragLeave={handleDragLeave}
              onDragOver={handleDragOver}
              onDrop={handleDrop}
            >
              {/* 드래그 오버레이 */}
              {isDragOver && (
                <div className="absolute inset-0 bg-blue-500 bg-opacity-10 flex items-center justify-center z-10 rounded-lg">
                  <div className="text-center">
                    <div className="text-4xl mb-2">📁</div>
                    <div className="text-lg font-semibold text-blue-600 dark:text-blue-400">
                      파일을 여기에 드롭하세요
                    </div>
                    <div className="text-sm text-gray-600 dark:text-gray-400">
                      이미지, PDF, 텍스트 파일을 지원합니다
                    </div>
                  </div>
                </div>
              )}
              {/* 채팅 헤더 */}
              <div className="px-6 py-4 border-b border-gray-200 dark:border-gray-700">
                <div className="flex items-center justify-between">
                  <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                    💬 AI 어시스턴트
                    {currentSessionId && (
                      <span className="text-sm font-normal text-gray-500 dark:text-gray-400 ml-2">
                        • {sessions.find(s => s.id === currentSessionId)?.title || '현재 세션'}
                      </span>
                    )}
                  </h3>
                  
                  {/* 토스트 메시지 (헤더 중간) */}
                  {toast.isVisible && (
                    <div className="flex-1 flex justify-center mx-4">
                      <div className={`px-4 py-2 rounded-full text-sm font-medium ${
                        toast.type === 'success' ? 'bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-200' :
                        toast.type === 'error' ? 'bg-red-100 dark:bg-red-900/30 text-red-800 dark:text-red-200' :
                        'bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-200'
                      } animate-in slide-in-from-top-2 fade-in-0 duration-300`}>
                        <div className="flex items-center space-x-2">
                          <span className="text-sm">
                            {toast.type === 'success' ? '✅' : toast.type === 'error' ? '❌' : 'ℹ️'}
                          </span>
                          <span>{toast.message}</span>
                          <button
                            onClick={closeToast}
                            className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 ml-2"
                          >
                            <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                            </svg>
                          </button>
                        </div>
                      </div>
                    </div>
                  )}
                  
                  <button
                    onClick={clearChat}
                    className="px-3 py-1 text-sm bg-gray-100 dark:bg-gray-700 text-gray-700 dark:text-gray-300 rounded hover:bg-gray-200 dark:hover:bg-gray-600 transition-colors"
                  >
                    대화 초기화
                  </button>
                </div>
              </div>

              {/* 채팅 메시지 영역 */}
              <div 
                ref={chatContainerRef}
                className="flex-1 overflow-y-auto p-6 space-y-4 min-h-0"
                style={{ minHeight: '300px', maxHeight: 'calc(100vh - 20rem)' }}
              >
                {chatMessages.length === 0 ? (
                  <div className="text-center text-gray-500 dark:text-gray-400 py-8">
                    <div className="text-4xl mb-4">🤖</div>
                    <p className="text-base mb-6">원하는 작업을 요청해보세요. AI가 도와드립니다.</p>
                    
                    {isInitializing && (
                      <p className="text-sm mt-2 text-blue-600 dark:text-blue-400">
                        MCP 도구를 초기화하는 중입니다...
                      </p>
                    )}
                    {!isInitialized && !isInitializing && (
                      <p className="text-sm mt-2 text-orange-600 dark:text-orange-400 mb-6">
                        먼저 MCP 도구를 설정하고 적용해주세요.
                      </p>
                    )}
                    
                    {/* 사용 예시 카드들 */}
                    <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 gap-4 max-w-4xl mx-auto mt-6">
                      <button
                        onClick={() => setCurrentMessage("사용 가능한 도구 목록을 보여주세요")}
                        className="p-4 bg-gradient-to-br from-purple-50 to-purple-100 dark:from-purple-900/20 dark:to-purple-800/20 border border-purple-200 dark:border-purple-700 rounded-lg hover:shadow-md transition-all duration-200 hover:scale-105 text-left"
                      >
                        <div className="text-2xl mb-2">🛠️</div>
                        <h4 className="font-semibold text-gray-900 dark:text-white mb-1">도구 목록 확인</h4>
                        <p className="text-sm text-gray-600 dark:text-gray-300">사용 가능한 모든 도구들을 확인하세요</p>
                      </button>
                      
                      <button
                        onClick={() => setCurrentMessage("웹에서 오늘 'AI 관련 최신 뉴스'를 검색해주세요")}
                        className="p-4 bg-gradient-to-br from-blue-50 to-blue-100 dark:from-blue-900/20 dark:to-blue-800/20 border border-blue-200 dark:border-blue-700 rounded-lg hover:shadow-md transition-all duration-200 hover:scale-105 text-left"
                      >
                        <div className="text-2xl mb-2">🌐</div>
                        <h4 className="font-semibold text-gray-900 dark:text-white mb-1">웹 검색</h4>
                        <p className="text-sm text-gray-600 dark:text-gray-300">최신 정보를 실시간으로 검색합니다</p>
                      </button>
                      
                      <button
                        onClick={() => setCurrentMessage("Python으로 파일 읽기 함수를 만들어주세요")}
                        className="p-4 bg-gradient-to-br from-green-50 to-green-100 dark:from-green-900/20 dark:to-green-800/20 border border-green-200 dark:border-green-700 rounded-lg hover:shadow-md transition-all duration-200 hover:scale-105 text-left"
                      >
                        <div className="text-2xl mb-2">💻</div>
                        <h4 className="font-semibold text-gray-900 dark:text-white mb-1">코드 생성</h4>
                        <p className="text-sm text-gray-600 dark:text-gray-300">다양한 언어로 코드를 생성합니다</p>
                      </button>
                      
                      <button
                        onClick={() => setCurrentMessage("문서 검색을 통해 '신혼(예비) 부부 건강검진비 지원을 신청할 수 있는 방법' 대한 문서를 분석하고 요약해주세요")}
                        className="p-4 bg-gradient-to-br from-orange-50 to-orange-100 dark:from-orange-900/20 dark:to-orange-800/20 border border-orange-200 dark:border-orange-700 rounded-lg hover:shadow-md transition-all duration-200 hover:scale-105 text-left"
                      >
                        <div className="text-2xl mb-2">📄</div>
                        <h4 className="font-semibold text-gray-900 dark:text-white mb-1">문서 검색</h4>
                        <p className="text-sm text-gray-600 dark:text-gray-300">문서를 검색하고 응답합니다</p>
                      </button>
                      
                      <button
                        onClick={() => setCurrentMessage("AI 기술 동향에 대한 보고서를 작성해주세요")}
                        className="p-4 bg-gradient-to-br from-red-50 to-red-100 dark:from-red-900/20 dark:to-red-800/20 border border-red-200 dark:border-red-700 rounded-lg hover:shadow-md transition-all duration-200 hover:scale-105 text-left"
                      >
                        <div className="text-2xl mb-2">📊</div>
                        <h4 className="font-semibold text-gray-900 dark:text-white mb-1">리포트 생성</h4>
                        <p className="text-sm text-gray-600 dark:text-gray-300">전문적인 보고서를 생성합니다</p>
                      </button>
                      
                      <button
                        onClick={() => setCurrentMessage("업로드한 이미지의 내용을 설명해주세요")}
                        className="p-4 bg-gradient-to-br from-indigo-50 to-indigo-100 dark:from-indigo-900/20 dark:to-indigo-800/20 border border-indigo-200 dark:border-indigo-700 rounded-lg hover:shadow-md transition-all duration-200 hover:scale-105 text-left"
                      >
                        <div className="text-2xl mb-2">🖼️</div>
                        <h4 className="font-semibold text-gray-900 dark:text-white mb-1">이미지 분석</h4>
                        <p className="text-sm text-gray-600 dark:text-gray-300">이미지를 분석하고 설명합니다</p>
                      </button>
                      
                      <button
                        onClick={() => setCurrentMessage("A2A 네트워크 상태를 확인하고 연결된 에이전트들을 보여주세요")}
                        className="p-4 bg-gradient-to-br from-purple-50 to-purple-100 dark:from-purple-900/20 dark:to-purple-800/20 border border-purple-200 dark:border-purple-700 rounded-lg hover:shadow-md transition-all duration-200 hover:scale-105 text-left"
                      >
                        <div className="text-2xl mb-2">🤝</div>
                        <h4 className="font-semibold text-gray-900 dark:text-white mb-1">A2A 네트워크</h4>
                        <p className="text-sm text-gray-600 dark:text-gray-300">에이전트 간 협업 네트워크 상태 확인</p>
                      </button>
                      
                      <button
                        onClick={() => setCurrentMessage("외부 AI 에이전트와 협업하여 복잡한 작업을 처리해주세요")}
                        className="p-4 bg-gradient-to-br from-teal-50 to-teal-100 dark:from-teal-900/20 dark:to-teal-800/20 border border-teal-200 dark:border-teal-700 rounded-lg hover:shadow-md transition-all duration-200 hover:scale-105 text-left"
                      >
                        <div className="text-2xl mb-2">🌐</div>
                        <h4 className="font-semibold text-gray-900 dark:text-white mb-1">분산 AI 협업</h4>
                        <p className="text-sm text-gray-600 dark:text-gray-300">여러 AI 에이전트와 협업하여 작업 처리</p>
                      </button>
                      
                      <button
                        onClick={() => setCurrentMessage("내 MCP 도구들을 A2A 네트워크에 공유하고 다른 에이전트들이 사용할 수 있게 해주세요")}
                        className="p-4 bg-gradient-to-br from-pink-50 to-pink-100 dark:from-pink-900/20 dark:to-pink-800/20 border border-pink-200 dark:border-pink-700 rounded-lg hover:shadow-md transition-all duration-200 hover:scale-105 text-left"
                      >
                        <div className="text-2xl mb-2">🔗</div>
                        <h4 className="font-semibold text-gray-900 dark:text-white mb-1">도구 공유</h4>
                        <p className="text-sm text-gray-600 dark:text-gray-300">MCP 도구를 A2A 네트워크에 공유</p>
                      </button>
                    </div>
                  </div>
                ) : (
                  chatMessages.map((message, index) => (
                    <div
                      key={index}
                      className={`flex ${message.role === 'user' ? 'justify-end' : 'justify-start'}`}
                    >
                      <div
                        className={`max-w-3xl px-4 py-2 ${
                          message.role === 'user'
                            ? 'bg-blue-600 text-white'
                            : message.role === 'tool'
                            ? 'bg-white dark:bg-gray-800 text-gray-800 dark:text-gray-200 border-l-4 border-purple-500'
                            : 'bg-white dark:bg-gray-800 text-gray-800 dark:text-gray-200'
                        }`}
                      >
                        {/* 첨부된 이미지들 표시 (사용자 메시지에만) */}
                        {message.role === 'user' && message.attachedFiles && message.attachedFiles.length > 0 && (
                          <div className="mb-3 flex flex-wrap gap-2">
                            {message.attachedFiles
                              .filter(file => file.base64 && file.type.startsWith('image/'))
                              .map((file, imgIndex) => {
                                // Base64 데이터에서 Data URL 접두어 제거 (중복 방지)
                                let cleanBase64 = file.base64 || '';
                                if (cleanBase64.startsWith('data:')) {
                                  // 이미 Data URL 형태라면 Base64 부분만 추출
                                  const base64Index = cleanBase64.indexOf('base64,');
                                  if (base64Index !== -1) {
                                    cleanBase64 = cleanBase64.substring(base64Index + 7);
                                  }
                                }
                                
                                const imageUrl = `data:${file.type};base64,${cleanBase64}`;
                                return (
                                  <div key={`${index}-img-${imgIndex}`} className="relative group">
                                    <img 
                                      src={imageUrl} 
                                      alt={file.name}
                                      className="max-w-xs max-h-48 rounded-lg object-cover border border-gray-200 dark:border-gray-600 shadow-sm hover:shadow-md transition-shadow cursor-pointer"
                                      onClick={() => {
                                        // 이미지 클릭 시 모달로 크게 보기
                                        setImageModal({
                                          isOpen: true,
                                          imageUrl: imageUrl,
                                          imageName: file.name
                                        });
                                      }}
                                      title={`${file.name} - 클릭하여 크게 보기`}
                                    />
                                    <div className="absolute bottom-1 left-1 bg-black bg-opacity-50 text-white text-xs px-2 py-1 rounded opacity-0 group-hover:opacity-100 transition-opacity">
                                      {file.name}
                                    </div>
                                  </div>
                                );
                              })}
                          </div>
                        )}
                        
                        <div className="text-sm">
                          {message.role === 'user' ? (
                            <div className="whitespace-pre-wrap">{message.content}</div>
                          ) : (
                            <McpMarkdownRenderer content={message.content} />
                          )}
                        </div>
                        
                        {/* 도구 호출 정보 표시 (app.py의 expander와 유사) */}
                        {message.toolCalls && message.toolCalls.length > 0 ? (
                          <ToolCallsDisplay toolCalls={message.toolCalls} />
                        ) : (
                          // 디버깅용: 도구 호출 정보가 없을 때 표시
                          process.env.NODE_ENV === 'development' && message.role === 'assistant' && (
                            <div className="mt-2 text-xs text-gray-500 dark:text-gray-400">
                              도구 호출 정보: {message.toolCalls?.length || 0}개
                            </div>
                          )
                        )}
                        
                        <div className="text-xs opacity-70 mt-1">
                          {message.timestamp.toLocaleTimeString()}
                        </div>
                      </div>
                    </div>
                  ))
                )}
                
                {isLoading && (
                  <div className="flex justify-start">
                    <div className="bg-gray-100 dark:bg-gray-700 px-4 py-2 rounded-lg">
                      <div className="flex items-center space-x-2">
                        <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-purple-600"></div>
                        <span className="text-sm text-gray-600 dark:text-gray-300">Thinking ...</span>
                      </div>
                    </div>
                  </div>
                )}
              </div>

              {/* 파일 업로드 영역 */}
              {uploadedFiles.length > 0 && (
                <div className="px-6 py-3 bg-gray-50 dark:bg-gray-700 border-t border-gray-200 dark:border-gray-600 flex-shrink-0">
                  <div className="space-y-2">
                    {uploadedFiles.map((attachedFile, index) => (
                      <div key={index} className="flex items-center justify-between">
                        <div className="flex items-center space-x-2">
                          <div className="flex-shrink-0">
                            {isImageFile(attachedFile.file) ? (
                              attachedFile.url ? (
                                <img 
                                  src={attachedFile.url} 
                                  alt={attachedFile.name}
                                  className="w-8 h-8 object-cover rounded"
                                />
                              ) : '🖼️'
                            ) : isPdfFile(attachedFile.file) ? '📄' : 
                              isTextFile(attachedFile.file) ? '📝' : '📎'}
                          </div>
                          <div className="flex-1 min-w-0">
                            <div className="text-sm text-gray-900 dark:text-white truncate">
                              {attachedFile.name}
                            </div>
                            <div className="text-xs text-gray-500 dark:text-gray-400">
                              {formatFileSize(attachedFile.size)}
                              {attachedFile.isProcessing && ' • 처리 중...'}
                              {attachedFile.processingError && ` • 오류: ${attachedFile.processingError}`}
                            </div>
                          </div>
                        </div>
                        <button
                          onClick={() => removeFile(index)}
                          className="text-red-500 hover:text-red-700 p-1"
                          title="파일 제거"
                        >
                          ✕
                        </button>
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {/* 입력 영역 */}
              <div className="px-6 py-4 border-t border-gray-200 dark:border-gray-700 flex-shrink-0">
                <div 
                  className="bg-gray-100 dark:bg-gray-700 rounded-2xl border border-gray-200 dark:border-gray-600 p-4 transition-all"
                >
                  {/* 도구 버튼들 */}
                  <div className="flex items-center justify-start gap-2 mb-3">
                    {/* 파일 업로드 버튼 */}
                    <button
                      onClick={() => document.getElementById('file-upload')?.click()}
                      disabled={isLoading || isInitializing || isProcessingFiles}
                      className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${
                        isLoading || isInitializing || isProcessingFiles ? 'bg-gray-100 dark:bg-gray-600 text-gray-400 dark:text-gray-500' : 
                        'bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-500'
                      }`}
                      title="파일 업로드"
                    >
                      {isProcessingFiles ? (
                        <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-current"></div>
                      ) : (
                        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13" />
                        </svg>
                      )}
                      <span>{isProcessingFiles ? '처리 중' : '파일'}</span>
                    </button>
                  </div>

                  {/* 입력창 영역 */}
                  <div className="relative">
                    <textarea
                      ref={(el) => {
                        if (el) {
                          el.style.height = 'auto';
                          el.style.height = `${Math.min(Math.max(el.scrollHeight, 48), 200)}px`;
                        }
                      }}
                      value={currentMessage}
                      onChange={(e) => {
                        setCurrentMessage(e.target.value);
                        // 자동 높이 조절
                        const target = e.target as HTMLTextAreaElement;
                        target.style.height = 'auto';
                        target.style.height = `${Math.min(Math.max(target.scrollHeight, 48), 200)}px`;
                      }}
                      onKeyDown={(e) => {
                        if (e.key === 'Enter' && !e.shiftKey) {
                          e.preventDefault();
                          sendMessage();
                        }
                      }}
                      placeholder={isInitializing ? "MCP 초기화 중..." : isInitialized ? "질문을 입력하세요..." : "MCP 도구를 먼저 설정해주세요"}
                      className="w-full px-0 py-2 pr-12 bg-transparent text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none resize-none border-0 text-base"
                      rows={1}
                      disabled={!isInitialized || isLoading || isInitializing}
                      style={{ minHeight: '48px', maxHeight: '200px' }}
                    />

                    <div className="absolute right-0 bottom-2 flex items-center space-x-2">
                      {/* 마이크 버튼 */}
                      <button
                        onClick={toggleSpeechRecognition}
                        disabled={isLoading || isInitializing || isProcessingAudio}
                        className={`p-2 rounded-lg transition-colors ${
                          isListening
                            ? 'bg-red-100 dark:bg-red-900/30 text-red-600 dark:text-red-400 animate-pulse'
                            : isProcessingAudio
                            ? 'bg-yellow-100 dark:bg-yellow-900/30 text-yellow-600 dark:text-yellow-400'
                            : 'bg-gray-100 dark:bg-gray-700 text-gray-600 dark:text-gray-400 hover:bg-gray-200 dark:hover:bg-gray-600'
                        } disabled:opacity-50 disabled:cursor-not-allowed`}
                        title={isListening ? '음성 녹음 중지' : isProcessingAudio ? '음성 처리 중' : '음성 녹음 시작'}
                      >
                        {isListening ? (
                          <svg className="w-4 h-4" fill="currentColor" viewBox="0 0 24 24">
                            <path d="M12 14c1.66 0 3-1.34 3-3V5c0-1.66-1.34-3-3-3S9 3.34 9 5v6c0 1.66 1.34 3 3 3zm-1-9c0-.55.45-1 1-1s1 .45 1 1v6c0 .55-.45 1-1 1s-1-.45-1-1V5zm6 6c0 2.76-2.24 5-5 5s-5-2.24-5-5H5c0 3.53 2.61 6.43 6 6.92V21h2v-3.08c3.39-.49 6-3.39 6-6.92h-2z"/>
                            <path d="M19 11h2c0 3.53-2.61 6.43-6 6.92V21h-2v-3.08C9.61 17.43 7 14.53 7 11H5c0 2.76 2.24 5 5 5s5-2.24 5-5z" opacity="0.3"/>
                          </svg>
                        ) : isProcessingAudio ? (
                          <svg className="w-4 h-4 animate-spin" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                          </svg>
                        ) : (
                          <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 11a7 7 0 01-7 7m0 0a7 7 0 01-7-7m7 7v4m0 0H8m4 0h4m-4-8a3 3 0 01-3-3V5a3 3 0 116 0v6a3 3 0 01-3 3z" />
                          </svg>
                        )}
                      </button>

                      {/* 전송 버튼 */}
                      <button
                        onClick={isLoading ? stopResponse : sendMessage}
                        disabled={!isInitialized || isInitializing || (!isLoading && !currentMessage.trim())}
                        className={`p-2 text-white rounded-lg focus:outline-none focus:ring-2 focus:ring-purple-500 disabled:opacity-50 disabled:cursor-not-allowed transition-colors ${
                          isLoading 
                            ? 'bg-red-600 hover:bg-red-700 focus:ring-red-500' 
                            : 'bg-purple-600 hover:bg-purple-700 focus:ring-purple-500'
                        }`}
                        title={isLoading ? '응답 중지' : '메시지 보내기'}
                      >
                      {isLoading ? (
                        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 6h12v12H6z" />
                        </svg>
                      ) : (
                        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8" />
                        </svg>
                      )}
                      </button>
                    </div>
                  </div>
                </div>

                {/* 숨겨진 파일 입력 */}
                <input
                  ref={fileInputRef}
                  type="file"
                  onChange={handleFileUpload}
                  accept="image/*,.pdf,.doc,.docx,.txt,.md,.json,.xml,.csv,.hwp,.hwpx"
                  multiple
                  className="hidden"
                  id="file-upload"
                />
              </div>
            </div>
          </div>
        </div>
      </div>
      <Modal
        isOpen={modal.isOpen}
        onClose={closeModal}
        title={modal.title}
        message={modal.message}
        type={modal.type}
      />
      <ImageModal
        isOpen={imageModal.isOpen}
        onClose={() => setImageModal({ isOpen: false, imageUrl: '', imageName: '' })}
        imageUrl={imageModal.imageUrl}
        imageName={imageModal.imageName}
      />
    </div>
  );
}