'use client';

import { useState, useEffect } from 'react';
import { useRouter } from 'next/navigation';
import { getBusinessServerUrl } from '@/config/serverConfig';
import { 
  Building2, 
  FileText, 
  Users, 
  TrendingUp, 
  Search,
  Filter,
  Download,
  Eye,
  Star,
  Calendar,
  DollarSign,
  MapPin,
  Tag,
  ArrowRight,
  RefreshCw,
  Plus,
  BarChart3,
  Clock,
  Target,
  Grid3X3,
  List
} from 'lucide-react';

interface Announcement {
  id: string;
  title: string;
  description: string;
  category: string;
  source: string;
  status: string;
  application_end: string | null;
  budget: number;
  support_amount: number;
  eligibility: {
    company_size: string[];
    industry: string[];
    location: string[];
  };
  contact: {
    department: string;
    name: string;
    phone: string | null;
    email: string | null;
  };
  view_count: number;
  created_at: string;
}

interface BusinessStats {
  overview: {
    total_active: number;
    total_all: number;
    total_companies: number;
    total_matches: number;
    total_proposals: number;
  };
  updated_at: string;
}

export default function BusinessAnalysisPage() {
  const router = useRouter();
  const [stats, setStats] = useState<BusinessStats | null>(null);
  const [announcements, setAnnouncements] = useState<Announcement[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [selectedCategory, setSelectedCategory] = useState('');
  const [isRefreshing, setIsRefreshing] = useState(false);
  const [viewMode, setViewMode] = useState<'card' | 'list'>('card');

  // 비즈니스 공고 API 서버 URL
  const BIZ_API_BASE = getBusinessServerUrl();

  // 데이터 로드
  const loadData = async () => {
    try {
      setLoading(true);
      
      // 통계 정보 로드
      const statsResponse = await fetch(`${BIZ_API_BASE}/stats`);
      if (statsResponse.ok) {
        const statsData = await statsResponse.json();
        setStats(statsData);
      }

      // 공고 목록 로드 (최근 10개)
      const announcementsResponse = await fetch(`${BIZ_API_BASE}/announcements?limit=10`);
      if (announcementsResponse.ok) {
        const announcementsData = await announcementsResponse.json();
        setAnnouncements(announcementsData.announcements || []);
      }
    } catch (error) {
      console.error('데이터 로드 실패:', error);
    } finally {
      setLoading(false);
    }
  };

  // 데이터 새로고침
  const refreshData = async () => {
    setIsRefreshing(true);
    await loadData();
    setIsRefreshing(false);
  };

  // 컴포넌트 마운트시 데이터 로드
  useEffect(() => {
    loadData();
  }, []);

  // 가격 포맷팅
  const formatPrice = (amount: number | null | undefined) => {
    if (!amount || amount === 0) {
      return '미정';
    }
    if (amount >= 100000000) {
      return `${(amount / 100000000).toFixed(1)}억원`;
    } else if (amount >= 10000) {
      return `${(amount / 10000).toFixed(0)}만원`;
    } else {
      return `${amount.toLocaleString()}원`;
    }
  };

  // 날짜 포맷팅
  const formatDate = (dateString: string) => {
    return new Date(dateString).toLocaleDateString('ko-KR');
  };

  // 카테고리별 색상
  const getCategoryColor = (category: string) => {
    const colors: { [key: string]: string } = {
      '창업': 'bg-green-100 text-green-800 dark:bg-green-900/20 dark:text-green-300',
      '수출': 'bg-blue-100 text-blue-800 dark:bg-blue-900/20 dark:text-blue-300',
      '금융': 'bg-purple-100 text-purple-800 dark:bg-purple-900/20 dark:text-purple-300',
      '인력': 'bg-orange-100 text-orange-800 dark:bg-orange-900/20 dark:text-orange-300',
      '경영': 'bg-indigo-100 text-indigo-800 dark:bg-indigo-900/20 dark:text-indigo-300',
      '기타': 'bg-gray-100 text-gray-800 dark:bg-gray-900/20 dark:text-gray-300'
    };
    return colors[category] || colors['기타'];
  };


  if (loading) {
    return (
      <div className="min-h-screen flex items-center justify-center">
        <div className="flex flex-col items-center gap-4">
          <RefreshCw className="w-8 h-8 animate-spin text-blue-600" />
          <p className="text-gray-600 dark:text-gray-400">사업분석 데이터를 불러오는 중...</p>
        </div>
      </div>
    );
  }

  return (
    <div className="min-h-screen" style={{ backgroundColor: 'var(--body-bg)', color: 'var(--text-primary)' }}>
      <div className="container mx-auto px-4 py-8 max-w-7xl">
        {/* 헤더 영역 */}
        <div className="flex justify-end items-center mb-8">
          <button
            onClick={refreshData}
            disabled={isRefreshing}
            className="flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 transition-colors"
          >
            <RefreshCw className={`w-4 h-4 ${isRefreshing ? 'animate-spin' : ''}`} />
            새로고침
          </button>
        </div>

        {/* 통계 카드 영역 */}
        {stats && (
          <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-5 gap-6 mb-8">
            <div className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">활성 공고</p>
                  <p className="text-2xl font-bold text-green-600">{stats.overview.total_active}</p>
                </div>
                <FileText className="w-8 h-8 text-green-600" />
              </div>
            </div>

            <div className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">전체 공고</p>
                  <p className="text-2xl font-bold text-blue-600">{stats.overview.total_all}</p>
                </div>
                <BarChart3 className="w-8 h-8 text-blue-600" />
              </div>
            </div>

            <div className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">등록 기업</p>
                  <p className="text-2xl font-bold text-purple-600">{stats.overview.total_companies}</p>
                </div>
                <Users className="w-8 h-8 text-purple-600" />
              </div>
            </div>

            <div className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">매칭 결과</p>
                  <p className="text-2xl font-bold text-orange-600">{stats.overview.total_matches}</p>
                </div>
                <Target className="w-8 h-8 text-orange-600" />
              </div>
            </div>

            <div className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700">
              <div className="flex items-center justify-between">
                <div>
                  <p className="text-sm font-medium text-gray-600 dark:text-gray-400">제안서</p>
                  <p className="text-2xl font-bold text-red-600">{stats.overview.total_proposals}</p>
                </div>
                <FileText className="w-8 h-8 text-red-600" />
              </div>
            </div>
          </div>
        )}

        {/* 메인 액션 버튼들 */}
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6 mb-8">
          <button 
            onClick={() => router.push('/business/announcements')}
            className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700 hover:shadow-md transition-shadow group"
          >
            <div className="flex items-center justify-between mb-4">
              <Search className="w-8 h-8 text-blue-600" />
              <ArrowRight className="w-5 h-5 text-gray-400 group-hover:text-blue-600 transition-colors" />
            </div>
            <h3 className="text-lg font-semibold mb-2">공고 탐색</h3>
            <p className="text-sm text-gray-600 dark:text-gray-400">정부 및 공공기관 사업공고를 검색하고 필터링합니다</p>
          </button>

          <button 
            onClick={() => router.push('/business/matching')}
            className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700 hover:shadow-md transition-shadow group"
          >
            <div className="flex items-center justify-between mb-4">
              <Target className="w-8 h-8 text-green-600" />
              <ArrowRight className="w-5 h-5 text-gray-400 group-hover:text-green-600 transition-colors" />
            </div>
            <h3 className="text-lg font-semibold mb-2">기업 매칭</h3>
            <p className="text-sm text-gray-600 dark:text-gray-400">기업 특성에 맞는 최적의 사업공고를 찾아드립니다</p>
          </button>

          <button 
            onClick={() => router.push('/business/proposals')}
            className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700 hover:shadow-md transition-shadow group"
          >
            <div className="flex items-center justify-between mb-4">
              <FileText className="w-8 h-8 text-purple-600" />
              <ArrowRight className="w-5 h-5 text-gray-400 group-hover:text-purple-600 transition-colors" />
            </div>
            <h3 className="text-lg font-semibold mb-2">제안서 관리</h3>
            <p className="text-sm text-gray-600 dark:text-gray-400">AI 기반 맞춤형 사업제안서를 생성하고 관리합니다</p>
          </button>

          <button 
            onClick={() => router.push('/business/dashboard')}
            className="bg-white dark:bg-gray-800 rounded-xl p-6 shadow-sm border border-gray-200 dark:border-gray-700 hover:shadow-md transition-shadow group"
          >
            <div className="flex items-center justify-between mb-4">
              <BarChart3 className="w-8 h-8 text-orange-600" />
              <ArrowRight className="w-5 h-5 text-gray-400 group-hover:text-orange-600 transition-colors" />
            </div>
            <h3 className="text-lg font-semibold mb-2">분석 대시보드</h3>
            <p className="text-sm text-gray-600 dark:text-gray-400">상세한 통계와 분석 결과를 확인합니다</p>
          </button>
        </div>

        {/* 최근 공고 미리보기 */}
        <div className="bg-white dark:bg-gray-800 rounded-xl shadow-sm border border-gray-200 dark:border-gray-700">
          <div className="p-6 border-b border-gray-200 dark:border-gray-700">
            <div className="flex items-center justify-between">
              <h2 className="text-xl font-semibold flex items-center gap-2">
                <Clock className="w-5 h-5 text-blue-600" />
                최근 공고
              </h2>
              <div className="flex items-center gap-4">
                {/* 뷰 모드 토글 버튼 */}
                <div className="flex items-center gap-2 bg-gray-100 dark:bg-gray-700 p-1 rounded-lg">
                  <button
                    onClick={() => setViewMode('list')}
                    className={`flex items-center gap-1 px-3 py-2 rounded-md text-sm font-medium transition-colors ${
                      viewMode === 'list'
                        ? 'bg-white dark:bg-gray-800 text-blue-600 shadow-sm'
                        : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
                    }`}
                  >
                    <List className="w-4 h-4" />
                    목록
                  </button>
                  <button
                    onClick={() => setViewMode('card')}
                    className={`flex items-center gap-1 px-3 py-2 rounded-md text-sm font-medium transition-colors ${
                      viewMode === 'card'
                        ? 'bg-white dark:bg-gray-800 text-blue-600 shadow-sm'
                        : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
                    }`}
                  >
                    <Grid3X3 className="w-4 h-4" />
                    카드
                  </button>
                </div>
                <button 
                  onClick={() => router.push('/business/announcements')}
                  className="text-blue-600 hover:text-blue-700 text-sm font-medium flex items-center gap-1"
                >
                  전체보기 <ArrowRight className="w-4 h-4" />
                </button>
              </div>
            </div>
          </div>
          
          <div className="p-6">
            {announcements.length > 0 ? (
              viewMode === 'list' ? (
                // 목록 뷰
                <div className="space-y-4">
                  {announcements.slice(0, 5).map((announcement) => (
                    <div 
                      key={announcement.id}
                      className="flex items-start justify-between p-4 border border-gray-200 dark:border-gray-700 rounded-lg hover:bg-gray-50 dark:hover:bg-gray-700/50 transition-colors cursor-pointer"
                      onClick={() => router.push(`/business/announcements/${announcement.id}`)}
                    >
                      <div className="flex-1">
                        <div className="flex items-start gap-3 mb-2">
                          <span className={`px-2 py-1 rounded-full text-xs font-medium ${getCategoryColor(announcement.category)}`}>
                            {announcement.category}
                          </span>
                          {announcement.application_end && (
                            <span className="text-xs text-gray-500 dark:text-gray-400 flex items-center gap-1">
                              <Calendar className="w-3 h-3" />
                              마감: {formatDate(announcement.application_end)}
                            </span>
                          )}
                        </div>
                        <h3 className="font-medium text-gray-900 dark:text-gray-100 mb-1 line-clamp-1">
                          {announcement.title}
                        </h3>
                        <p className="text-sm text-gray-600 dark:text-gray-400 line-clamp-2 mb-2">
                          {announcement.description}
                        </p>
                        <div className="flex items-center gap-4 text-xs text-gray-500 dark:text-gray-400">
                          <span className="flex items-center gap-1">
                            <DollarSign className="w-3 h-3" />
                            {formatPrice(announcement.support_amount)}
                          </span>
                          <span className="flex items-center gap-1">
                            <Eye className="w-3 h-3" />
                            {announcement.view_count}회
                          </span>
                        </div>
                      </div>
                      <ArrowRight className="w-5 h-5 text-gray-400 ml-4" />
                    </div>
                  ))}
                </div>
              ) : (
                // 카드 뷰 (4x1 배치)
                <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-3 xl:grid-cols-4 gap-4">
                  {announcements.slice(0, 8).map((announcement) => (
                    <div 
                      key={announcement.id}
                      className="bg-gray-50 dark:bg-gray-900 border border-gray-200 dark:border-gray-700 rounded-lg hover:shadow-md transition-all cursor-pointer overflow-hidden"
                      onClick={() => router.push(`/business/announcements/${announcement.id}`)}
                    >
                      {/* 카드 헤더 */}
                      <div className="p-3 border-b border-gray-200 dark:border-gray-700">
                        <div className="flex items-center justify-between mb-2">
                          <span className={`px-2 py-1 rounded-full text-xs font-medium ${getCategoryColor(announcement.category)}`}>
                            {announcement.category}
                          </span>
                          <span className="text-xs text-gray-500 dark:text-gray-400 flex items-center gap-1">
                            <Eye className="w-3 h-3" />
                            {announcement.view_count}
                          </span>
                        </div>
                        <h3 className="text-sm font-medium text-gray-900 dark:text-gray-100 line-clamp-2 min-h-[2.5rem]">
                          {announcement.title}
                        </h3>
                      </div>

                      {/* 카드 본문 */}
                      <div className="p-3">
                        <p className="text-xs text-gray-600 dark:text-gray-400 line-clamp-2 mb-3 min-h-[2.5rem]">
                          {announcement.description}
                        </p>

                        {/* 지원금 */}
                        <div className="flex items-center gap-1 mb-2">
                          <DollarSign className="w-3 h-3 text-green-600" />
                          <span className="text-xs font-medium text-green-600">{formatPrice(announcement.support_amount)}</span>
                        </div>

                        {/* 마감일 */}
                        {announcement.application_end && (
                          <div className="flex items-center gap-1">
                            <Calendar className="w-3 h-3 text-gray-500" />
                            <span className="text-xs text-gray-500 dark:text-gray-400">
                              {formatDate(announcement.application_end)}
                            </span>
                          </div>
                        )}
                      </div>
                    </div>
                  ))}
                </div>
              )
            ) : (
              <div className="text-center py-12">
                <FileText className="w-12 h-12 text-gray-400 mx-auto mb-4" />
                <p className="text-gray-600 dark:text-gray-400">등록된 공고가 없습니다.</p>
              </div>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}