'use client';

import { useState, useEffect } from 'react';
import Link from 'next/link';
import { useRouter } from 'next/navigation';

interface Project {
  id: string;
  name: string;
  description: string;
  language: string;
  framework: string;
  createdAt: string;
  updatedAt: string;
}

export default function ProjectListPage() {
  const router = useRouter();
  const [projects, setProjects] = useState<Project[]>([]);
  const [loading, setLoading] = useState(true);
  const [searchTerm, setSearchTerm] = useState('');
  const [languageFilter, setLanguageFilter] = useState('');

  // 프로젝트 데이터 로드
  useEffect(() => {
    const fetchProjects = async () => {
      try {
        // API 호출 시뮬레이션
        await new Promise(resolve => setTimeout(resolve, 1000));
        
        // 샘플 데이터
        const sampleProjects: Project[] = [
          {
            id: '1',
            name: 'Todo 앱',
            description: 'React와 TypeScript를 이용한 할 일 관리 애플리케이션',
            language: 'javascript',
            framework: 'React',
            createdAt: '2023-05-15T09:30:00Z',
            updatedAt: '2023-05-20T14:22:00Z'
          },
          {
            id: '2',
            name: '블로그 API',
            description: 'Django REST Framework를 이용한 블로그 백엔드 API',
            language: 'python',
            framework: 'Django',
            createdAt: '2023-06-02T11:15:00Z',
            updatedAt: '2023-06-10T08:47:00Z'
          },
          {
            id: '3',
            name: '쇼핑몰 서비스',
            description: 'Spring Boot를 이용한 e-커머스 백엔드 서비스',
            language: 'java',
            framework: 'Spring Boot',
            createdAt: '2023-04-18T16:20:00Z',
            updatedAt: '2023-06-22T10:12:00Z'
          }
        ];
        
        setProjects(sampleProjects);
      } catch (error) {
        console.error('프로젝트 로드 오류:', error);
      } finally {
        setLoading(false);
      }
    };
    
    fetchProjects();
  }, []);

  // 검색 및 필터링된 프로젝트 목록
  const filteredProjects = projects.filter(project => {
    const matchesSearch = searchTerm === '' || 
      project.name.toLowerCase().includes(searchTerm.toLowerCase()) || 
      project.description.toLowerCase().includes(searchTerm.toLowerCase());
    
    const matchesLanguage = languageFilter === '' || project.language === languageFilter;
    
    return matchesSearch && matchesLanguage;
  });

  // 언어별 프로젝트 수 계산
  const languageCounts = projects.reduce((acc, project) => {
    acc[project.language] = (acc[project.language] || 0) + 1;
    return acc;
  }, {} as Record<string, number>);

  // 언어명 매핑
  const languageNames: Record<string, string> = {
    javascript: 'JavaScript',
    python: 'Python',
    java: 'Java',
    csharp: 'C#',
    php: 'PHP',
    go: 'Go',
    rust: 'Rust'
  };

  // 날짜 포맷팅 함수
  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    return date.toLocaleDateString('ko-KR', {
      year: 'numeric',
      month: 'long',
      day: 'numeric'
    });
  };

  return (
    <div className="container mx-auto px-4 py-8">
      <div className="flex justify-between items-center mb-8">
        <h1 className="text-3xl font-bold">프로젝트 목록</h1>
        <Link 
          href="/code/new"
          className="px-4 py-2 bg-blue-600 text-white rounded-md hover:bg-blue-700 transition-colors"
        >
          새 프로젝트
        </Link>
      </div>
      
      <div className="grid grid-cols-1 lg:grid-cols-4 gap-6">
        {/* 필터 사이드바 */}
        <div className="lg:col-span-1">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-4 mb-4">
            <h2 className="text-lg font-semibold mb-4">검색</h2>
            <input
              type="text"
              className="w-full px-3 py-2 border border-gray-300 dark:border-gray-700 rounded-md shadow-sm focus:outline-none focus:ring-indigo-500 focus:border-indigo-500 dark:bg-gray-700 dark:text-white mb-4"
              placeholder="프로젝트 검색..."
              value={searchTerm}
              onChange={(e) => setSearchTerm(e.target.value)}
            />
            
            <h2 className="text-lg font-semibold mb-4">언어 필터</h2>
            <div className="space-y-2">
              <div className="flex items-center">
                <input
                  id="all-languages"
                  type="radio"
                  name="language-filter"
                  checked={languageFilter === ''}
                  onChange={() => setLanguageFilter('')}
                  className="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300"
                />
                <label htmlFor="all-languages" className="ml-2 block text-sm text-gray-700 dark:text-gray-300">
                  모든 언어 ({projects.length})
                </label>
              </div>
              
              {Object.entries(languageCounts).map(([lang, count]) => (
                <div key={lang} className="flex items-center">
                  <input
                    id={`language-${lang}`}
                    type="radio"
                    name="language-filter"
                    checked={languageFilter === lang}
                    onChange={() => setLanguageFilter(lang)}
                    className="h-4 w-4 text-indigo-600 focus:ring-indigo-500 border-gray-300"
                  />
                  <label htmlFor={`language-${lang}`} className="ml-2 block text-sm text-gray-700 dark:text-gray-300">
                    {languageNames[lang as keyof typeof languageNames] || lang} ({count})
                  </label>
                </div>
              ))}
            </div>
          </div>
        </div>
        
        {/* 프로젝트 목록 */}
        <div className="lg:col-span-3">
          {loading ? (
            <div className="flex justify-center py-12">
              <div className="animate-spin rounded-full h-10 w-10 border-t-2 border-b-2 border-blue-500"></div>
            </div>
          ) : filteredProjects.length > 0 ? (
            <div className="bg-white dark:bg-gray-800 rounded-lg shadow overflow-hidden">
              <ul className="divide-y divide-gray-200 dark:divide-gray-700">
                {filteredProjects.map((project) => (
                  <li key={project.id} className="p-4 hover:bg-gray-50 dark:hover:bg-gray-700 transition-colors">
                    <Link href={`/code/${project.id}`} className="block">
                      <div className="flex justify-between">
                        <h3 className="text-lg font-medium text-blue-600 dark:text-blue-400">{project.name}</h3>
                        <div className="flex space-x-2">
                          <span className="px-2 py-1 text-xs rounded-full bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300">
                            {languageNames[project.language as keyof typeof languageNames] || project.language}
                          </span>
                          <span className="px-2 py-1 text-xs rounded-full bg-blue-100 dark:bg-blue-900 text-blue-700 dark:text-blue-300">
                            {project.framework}
                          </span>
                        </div>
                      </div>
                      <p className="mt-1 text-sm text-gray-600 dark:text-gray-400">{project.description}</p>
                      <div className="mt-2 flex justify-between text-xs text-gray-500 dark:text-gray-400">
                        <span>생성: {formatDate(project.createdAt)}</span>
                        <span>최종 수정: {formatDate(project.updatedAt)}</span>
                      </div>
                    </Link>
                  </li>
                ))}
              </ul>
            </div>
          ) : (
            <div className="bg-white dark:bg-gray-800 rounded-lg shadow p-8 text-center">
              <p className="text-gray-500 dark:text-gray-400">
                {searchTerm || languageFilter ? '검색 조건에 맞는 프로젝트가 없습니다.' : '프로젝트가 없습니다. 새 프로젝트를 생성해보세요.'}
              </p>
              {(searchTerm || languageFilter) && (
                <button
                  onClick={() => {
                    setSearchTerm('');
                    setLanguageFilter('');
                  }}
                  className="mt-4 px-4 py-2 bg-gray-200 dark:bg-gray-700 rounded-md text-sm"
                >
                  필터 초기화
                </button>
              )}
            </div>
          )}
        </div>
      </div>
    </div>
  );
} 