'use client';

import { useState, useEffect, useRef, useMemo, Suspense } from 'react';
import { useRouter, useSearchParams } from 'next/navigation';
import { externalApiClient, internalApiClient } from '@/lib/apiClient';
import { CodeRenderer } from '@/components/CodeRenderer';
import { useAuth } from '@/hooks/useAuth';
import { useLanguage } from '@/contexts/LanguageContext';
import ClientOnly from '@/components/ClientOnly';
import { BackgroundJobStatus } from '@/components/BackgroundJobStatus';
import type { BackgroundJob } from '@/types/support';

interface Message {
  id: string;
  role: 'user' | 'assistant';
  content: string;
  timestamp: string;
  originalContent?: string; // 멘션된 문서가 추가되기 전 원본 질문 (히스토리 표시용)
  attachedImages?: {
    file: File;
    url: string;
    name: string;
  }[];
  downloadInfo?: Array<{
    type: string;
    filename: string;
    url: string;
    description?: string;
  }>;

}

interface ChatSession {
  id: string;
  title: string;
  lastMessage: string;
  updatedAt: string;
  createdAt: string; // 추가
  messageCount: number;
  provider?: string;
  model?: string;
  history: Message[];
}

interface ChatOptions {
  provider: string;
  model: string;
  temperature: number;
  maxTokens: number;
  enableRag: boolean;
  enableWebSearch: boolean;
  enableSmartResearch: boolean;
  ragSearchScope: 'personal' | 'shared' | 'all';
  userId: number;
  history: Message[];
}

interface RagFile {
  name: string;
  size: number;
  modified?: string;
  type?: string;
  status: 'uploading' | 'processing' | 'completed' | 'error';
  progress?: number;
  error?: string;
}

interface MentionedDocument {
  name: string;
  displayName: string;
}

interface DocumentContext {
  documentName: string;
  totalChunks: number;
  selectedChunks: any[];
  metadata: {
    fileName: string;
    chunkCount: number;
    selectedCount: number;
  };
}

interface ApiSession {
  id: string;
  type: string;
  firstAIResponse?: string;
  lastMessage?: string;
  updated_at?: string;
  messageCount?: number;
  provider?: string;
  model?: string;
  history?: Message[];
}

// 세션 목록을 머지하는 함수 (중복 id는 새로 받은 sessionList 기준으로 덮어씀)
function mergeSessions(oldSessions: ChatSession[], newSessions: ChatSession[]): ChatSession[] {
  const sessionMap = new Map<string, ChatSession>();
  oldSessions.forEach(s => sessionMap.set(s.id, s));
  newSessions.forEach(s => sessionMap.set(s.id, s)); // 새 세션이 덮어씀
  // updatedAt 기준 내림차순 정렬
  return Array.from(sessionMap.values()).sort((a, b) => b.updatedAt.localeCompare(a.updatedAt));
}

// useSearchParams를 사용하는 컴포넌트를 분리
function ChatPageContent() {
  const { user, isAuthenticated, isInitialized } = useAuth();
  const { t } = useLanguage();
  const router = useRouter();
  const searchParams = useSearchParams();
  
  // URL에서 세션 ID를 읽어와서 상태로 관리
  const [currentSessionId, setCurrentSessionId] = useState<string | null>(searchParams.get('session'));

  const [sessions, setSessions] = useState<ChatSession[]>([]);
  const [currentMessages, setCurrentMessages] = useState<Message[]>([]);
  const [inputMessage, setInputMessage] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [isLoadingSessions, setIsLoadingSessions] = useState(false);
  const [error, setError] = useState<string | null>(null);
  const [showSidebar, setShowSidebar] = useState(true);
  const [showOptions, setShowOptions] = useState(false);
  const [searchTerm, setSearchTerm] = useState('');
  const [options, setOptions] = useState<ChatOptions>({
    provider: 'ollama',
    model: 'hamonize:latest',
    temperature: 0.7,
    maxTokens: 2000,
    enableRag: false, // 문서검색은 사용자가 선택적으로 활성화
    enableWebSearch: false,
    enableSmartResearch: false,
    ragSearchScope: 'personal', // 기본값: 개인 문서만 검색
    userId: user?.id ? Number(user.id) : 0,
    history: []
  });

  // 프로바이더 및 모델 관련 상태
  const [providers, setProviders] = useState<any>({});
  
  // 백그라운드 작업 관련 상태
  const [refreshTrigger, setRefreshTrigger] = useState(0);
  const [availableProviders, setAvailableProviders] = useState<any[]>([]);
  const [isLoadingProviders, setIsLoadingProviders] = useState(false);

  const [uploadedFiles, setUploadedFiles] = useState<File[]>([]);
  const [uploadedFileUrls, setUploadedFileUrls] = useState<Map<string, string>>(new Map());
  const [isUploading, setIsUploading] = useState(false);
  const [uploadingFiles, setUploadingFiles] = useState<Set<string>>(new Set());

  const [isDragOver, setIsDragOver] = useState(false);
  const [progressMessages, setProgressMessages] = useState<string[]>([]);
  const [showDeleteModal, setShowDeleteModal] = useState(false);
  const [sessionToDelete, setSessionToDelete] = useState<string | null>(null);
  const [showDeleteAllModal, setShowDeleteAllModal] = useState(false);
  const [thinkingDots, setThinkingDots] = useState<string>('');
  const [isNewChatMode, setIsNewChatMode] = useState(true);
  
  // RAG 파일 브라우저 상태
  const [ragFiles, setRagFiles] = useState<RagFile[]>([]);
  const [isLoadingRagFiles, setIsLoadingRagFiles] = useState<boolean>(false);
  const [showRagBrowser, setShowRagBrowser] = useState<boolean>(true);
  const [isDragOverRag, setIsDragOverRag] = useState<boolean>(false);

  // 메시지 액션 상태
  const [messageReactions, setMessageReactions] = useState<Record<string, { liked?: boolean; disliked?: boolean }>>({});
  const [showToast, setShowToast] = useState<{ message: string; type: 'success' | 'error' } | null>(null);
  const [copiedMessageId, setCopiedMessageId] = useState<string | null>(null);
  const [pdfProcessingId, setPdfProcessingId] = useState<string | null>(null);
  
  // 프로필 이미지 상태
  const [userProfileImage, setUserProfileImage] = useState<string>('/images/user.png');

  // @ 멘션 관련 상태
  const [mentionedDocs, setMentionedDocs] = useState<MentionedDocument[]>([]);
  const [showDocSuggestions, setShowDocSuggestions] = useState(false);
  const [mentionQuery, setMentionQuery] = useState('');
  const [mentionStartPos, setMentionStartPos] = useState(0);
  const [selectedSuggestionIndex, setSelectedSuggestionIndex] = useState(0);

  const messagesEndRef = useRef<HTMLDivElement>(null);
  const fileInputRef = useRef<HTMLInputElement>(null);
  const ragFileInputRef = useRef<HTMLInputElement>(null);
  const textareaRef = useRef<HTMLTextAreaElement>(null);

  // 타이핑 효과 관련 상태
  const [typingMessageId, setTypingMessageId] = useState<string | null>(null);
  const [displayedContent, setDisplayedContent] = useState<string>('');
  const [isTyping, setIsTyping] = useState<boolean>(false);
  const lastTypingEndRef = useRef<number>(0);



  //(ryan) - 코드 실행 관련 상태 변수들
  const [reviewFlags, setReviewFlags] = useState<Record<string, string>>({});
  const [userPrompt, setUserPrompt] = useState<Record<string, string>>({});   // 코드 실행 전 사용자 프롬프트 저장.
  const [userCodeExecution, setUserCodeExecution] = useState<Record<string, string>>({});
  const [promptCodeMap, setPromptCodeMap] = useState<Record<string, { prompt: string, code: string }>>({});
  const [taskName, setTaskName] = useState("");
  const [showTaskSaveModal, setShowTaskSaveModal] = useState(false);
  const [taskSaveTargetMessageId, setTaskSaveTargetMessageId] = useState<string | null>(null);
  const [successFlags, setSuccessFlags] = useState<Record<string, string>>({});
  const [downloadFlags, setDownloadFlags] = useState<Record<string, string>>({});
  const [executingCodeMessageId, setExecutingCodeMessageId] = useState<string | null>(null);
  const [executingCodes, setExecutingCodes] = useState<Set<string>>(new Set());
  const [executionResults, setExecutionResults] = useState<Record<string, string>>({});  // 성공한 실행 결과 저장
  //(ryan)
  
  // 입력창 높이 초기화 함수
  const resetTextareaHeight = () => {
    const textareas = document.querySelectorAll('textarea');
    textareas.forEach(textarea => {
      textarea.style.height = 'auto';
      textarea.style.height = '48px';
    });
  };

  // 인증 확인
  useEffect(() => {
    if (isInitialized && !isAuthenticated) {
      router.push('/login');
    }
    
    // user 객체 디버그 출력
    if (user) {
      // console.log('=== 사용자 정보 디버그 ===');
      // console.log('user.id:', user.id, typeof user.id);
      // console.log('user.username:', user.username, typeof user.username);
      // console.log('user.email:', user.email, typeof user.email);
      // console.log('user.role:', user.role, typeof user.role);
      // console.log('전체 user 객체:', user);
      // console.log('========================');

      // options의 userId도 함께 출력
      // console.log('=== ChatOptions 디버그 ===');
      // console.log('options.userId:', options.userId, typeof options.userId);
      // console.log('========================');
    }
  }, [isInitialized, isAuthenticated, router, user, options]);

  // 사용자 인터랙션 기록 함수
  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
  };

  // 출처는 이미 백엔드에서 정리되어 전달됨

  // 타이핑 효과 함수 (onDone 콜백 추가)
  const typeMessage = (messageId: string, fullContent: string, speed: number = 2, onDone?: () => void) => {
    // console.log('🎯 타이핑 효과 시작:', { messageId, contentLength: fullContent.length });
    setTypingMessageId(messageId);
    setDisplayedContent('');
    setIsTyping(true);
    let currentIndex = 0;
    const typeInterval = setInterval(() => {
      if (currentIndex < fullContent.length) {
        // 한 번에 더 많은 글자를 표시하여 빠른 타이핑 효과
        const charsToAdd = Math.min(5, fullContent.length - currentIndex);
        setDisplayedContent(fullContent.slice(0, currentIndex + charsToAdd));
        currentIndex += charsToAdd;
        setTimeout(scrollToBottom, 10);
      } else {
        clearInterval(typeInterval);
        setTypingMessageId(null);
        setDisplayedContent('');
        setIsTyping(false);
        lastTypingEndRef.current = Date.now();
        // console.log('✅ 타이핑 효과 완료');
        if (onDone) onDone();
      }
    }, speed);
    return () => clearInterval(typeInterval);
  };

  useEffect(() => {
    // 메시지가 변경될 때 자동 스크롤
    const timer = setTimeout(scrollToBottom, 50);
    return () => clearTimeout(timer);
  }, [currentMessages]);

  // 로딩 상태가 변경될 때도 스크롤
  useEffect(() => {
    // if (isLoading) {
    //   const timer = setTimeout(scrollToBottom, 100);
    //   return () => clearTimeout(timer);
    // }
  }, [isLoading]);

  // URL의 세션 ID가 변경될 때마다 해당 세션의 메시지를 불러옵니다.
  useEffect(() => {
    if (isInitialized && isAuthenticated) {
      if (currentSessionId) {
        // 기존 세션을 선택하는 로직
        selectSession(currentSessionId);
      } else {
        // 새 대화 상태
        setCurrentMessages([]);
      }
      // 세션 목록은 항상 불러옵니다.
      loadSessions().then(setSessions);
    }
  }, [currentSessionId, isInitialized, isAuthenticated]);

  // 컴포넌트 마운트 시 초기 데이터 로드
  useEffect(() => {
    loadProviders();
    initializeDefaultConfig();
  }, []);

  // 인증 상태가 변경될 때 세션 목록 및 RAG 파일 로드
  useEffect(() => {
    if (isAuthenticated) {
      loadSessions().then(setSessions);
      loadRagFiles();
    }
  }, [isAuthenticated]);

  // Think... 점 애니메이션
  useEffect(() => {
    if (isLoading) {
      const interval = setInterval(() => {
        setThinkingDots(prev => (prev.length >= 3 ? '' : prev + '.'));
      }, 500);
      return () => clearInterval(interval);
    } else {
      setThinkingDots('');
    }
  }, [isLoading]);

  // user가 로드되면 options 업데이트 및 프로필 이미지 로드
  useEffect(() => {
    if (user?.id) {
      setOptions(prev => ({
        ...prev,
        userId: Number(user.id)
      }));
      // 프로필 이미지 불러오기
      fetchUserProfileImage();
    }
  }, [user]);

  // 컴포넌트 언마운트 시 URL 정리 및 타이머 정리
  useEffect(() => {
    return () => {
      // 컴포넌트 언마운트 시에만 현재 업로드 중인 파일들의 URL 정리
      // 메시지에 저장된 URL은 정리하지 않음
      uploadedFileUrls.forEach(url => {
        URL.revokeObjectURL(url);
      });

    };
  }, []); // 의존성 배열을 비워서 언마운트 시에만 실행

  // 프로필 이미지 불러오기 함수
  const fetchUserProfileImage = async () => {
    try {
      const userId = localStorage.getItem('userId');
      if (!userId) return;

      const response = await fetch(`/api/profile/image?userId=${userId}`, {
        method: 'GET',
        credentials: 'include'
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success && data.data.imageUrl) {
          setUserProfileImage(data.data.imageUrl);
        }
      }
    } catch (error) {
      console.error('프로필 이미지 불러오기 실패:', error);
    }
  };

  // 프로바이더 정보 로드
  const loadProviders = async () => {
    try {
      setIsLoadingProviders(true);
      
      // 먼저 사용 가능한 프로바이더 목록을 가져옴 (인증 불필요)
      const availableResponse = await externalApiClient.getAvailableProviders();
      
      if (availableResponse.success && availableResponse.data) {
        setAvailableProviders(availableResponse.data);
        
        // 각 프로바이더의 모델 목록 로드
        const providersData: any = {};
        for (const provider of availableResponse.data) {
          try {
            const modelsResponse = await externalApiClient.getProviderModels(provider.key);
            if (modelsResponse.success && modelsResponse.data) {
              providersData[provider.key] = {
                name: provider.name,
                requiresApiKey: provider.requiresApiKey,
                apiKeyConfigured: provider.apiKeyConfigured !== false, // 백엔드에서 제공된 값 사용
                isDynamic: provider.isDynamic,
                isAvailable: true,
                models: modelsResponse.data
              };
            }
          } catch (modelError) {
            console.warn(`${provider.key} 모델 목록 로드 실패:`, modelError);
            // 모델 목록 로드 실패해도 프로바이더는 표시
            providersData[provider.key] = {
              name: provider.name,
              requiresApiKey: provider.requiresApiKey,
              apiKeyConfigured: provider.apiKeyConfigured !== false, // 백엔드에서 제공된 값 사용
              isDynamic: provider.isDynamic,
              isAvailable: true,
              models: []
            };
          }
        }
        
        setProviders(providersData);
      }
    } catch (error: any) {
      console.error('프로바이더 정보 로드 실패:', error);
      
      // 오류 발생 시 기본 프로바이더 정보 설정 (Ollama + 기본 모델)
      const defaultModel = {
        id: 'hamonize:latest',
        name: 'hamonize:latest',
        size: 0
      };
      
      const defaultProviders = {
        ollama: {
          name: 'Ollama',
          isAvailable: true,
          requiresApiKey: false,
          apiKeyConfigured: true, // API 키가 필요없으므로 true
          isDynamic: true,
          models: [defaultModel] // 기본 모델 추가
        }
      };
      setProviders(defaultProviders);
      setAvailableProviders([
        { key: 'ollama', name: 'Ollama', requiresApiKey: false, apiKeyConfigured: true, isDynamic: true, models: [defaultModel] }
      ]);
      
      // 기본 모델로 설정 (옵션이 ollama인 경우)
      if (options.provider === 'ollama' || !options.model) {
        setOptions(prev => ({ 
          ...prev, 
          provider: 'ollama',
          model: 'hamonize:latest'
        }));
        localStorage.setItem('airun-selected-provider', 'ollama');
        localStorage.setItem('airun-selected-model', 'hamonize:latest');
      }
    } finally {
      setIsLoadingProviders(false);
    }
  };

  // 기본 프로바이더/모델 설정 (사용자별 로컬 설정)
  const initializeDefaultConfig = () => {
    // 로컬 스토리지에서 사용자 설정 불러오기 (기본값: ollama, hamonize:latest)
    const savedProvider = localStorage.getItem('airun-selected-provider') || 'ollama';
    const savedModel = localStorage.getItem('airun-selected-model') || 'hamonize:latest';
    
    setOptions(prev => ({
      ...prev,
      provider: savedProvider,
      model: savedModel
    }));
    
    // 기본값이 사용된 경우 로컬 스토리지에 저장
    if (!localStorage.getItem('airun-selected-provider')) {
      localStorage.setItem('airun-selected-provider', 'ollama');
    }
    if (!localStorage.getItem('airun-selected-model')) {
      localStorage.setItem('airun-selected-model', 'hamonize:latest');
    }
  };

  // 프로바이더 변경 시 모델 목록 업데이트
  const handleProviderChange = async (newProvider: string) => {
    // 로컬 스토리지에 선택한 프로바이더 저장
    localStorage.setItem('airun-selected-provider', newProvider);
    
    setOptions(prev => ({ ...prev, provider: newProvider }));
    
    // 해당 프로바이더의 모델 목록이 비어있거나 없으면 실시간으로 다시 로드
    const providerData = providers[newProvider];
    if (!providerData || !providerData.models || providerData.models.length === 0) {
      console.log(`프로바이더 ${newProvider}의 모델 목록이 없어서 실시간 로드를 시도합니다.`);
      
      try {
        // 해당 프로바이더의 모델 목록을 실시간으로 로드
        const modelsResponse = await externalApiClient.getProviderModels(newProvider);
        if (modelsResponse.success && modelsResponse.data && modelsResponse.data.length > 0) {
          // 프로바이더 정보 업데이트
          setProviders((prev: any) => ({
            ...prev,
            [newProvider]: {
              ...prev[newProvider],
              models: modelsResponse.data
            }
          }));
          
          // 첫 번째 모델로 자동 설정
          const newModel = modelsResponse.data[0].id;
          localStorage.setItem('airun-selected-model', newModel);
          
          setOptions(prev => ({ 
            ...prev, 
            provider: newProvider,
            model: newModel
          }));
          
          console.log(`프로바이더 ${newProvider} 모델 실시간 로드 성공:`, modelsResponse.data.length, '개');
          return;
        } else {
          console.warn(`프로바이더 ${newProvider}의 모델 목록 실시간 로드 실패:`, modelsResponse);
          
          // Ollama인 경우 기본 모델로 fallback
          if (newProvider === 'ollama') {
            const defaultModel = {
              id: 'hamonize:latest',
              name: 'hamonize:latest',
              size: 0
            };
            
            setProviders((prev: any) => ({
              ...prev,
              [newProvider]: {
                ...prev[newProvider],
                models: [defaultModel]
              }
            }));
            
            setOptions(prev => ({ 
              ...prev, 
              provider: newProvider,
              model: 'hamonize:latest'
            }));
            
            localStorage.setItem('airun-selected-model', 'hamonize:latest');
            console.log(`프로바이더 ${newProvider} 실시간 로드 실패, 기본 모델로 fallback 완료`);
            return;
          }
        }
      } catch (error) {
        console.error(`프로바이더 ${newProvider}의 모델 목록 실시간 로드 중 오류:`, error);
        
        // Ollama인 경우 기본 모델로 fallback
        if (newProvider === 'ollama') {
          const defaultModel = {
            id: 'hamonize:latest',
            name: 'hamonize:latest',
            size: 0
          };
          
          setProviders((prev: any) => ({
            ...prev,
            [newProvider]: {
              ...prev[newProvider],
              models: [defaultModel]
            }
          }));
          
          setOptions(prev => ({ 
            ...prev, 
            provider: newProvider,
            model: 'hamonize:latest'
          }));
          
          localStorage.setItem('airun-selected-model', 'hamonize:latest');
          console.log(`프로바이더 ${newProvider} 실시간 로드 오류, 기본 모델로 fallback 완료`);
          return;
        }
      }
    }
    
    // 기존 로직: 프로바이더에 모델이 있는 경우
    if (providerData && providerData.models && providerData.models.length > 0) {
      const newModel = providerData.models[0].id;
      
      // 로컬 스토리지에 선택한 모델 저장
      localStorage.setItem('airun-selected-model', newModel);
      
      setOptions(prev => ({ 
        ...prev, 
        provider: newProvider,
        model: newModel
      }));
      
      console.log(`프로바이더 변경: ${newProvider}, 모델: ${newModel}`);
    } else {
      console.warn(`프로바이더 ${newProvider}의 모델 목록이 없습니다:`, providerData);
      
      // Ollama 프로바이더인 경우 기본 모델 설정
      if (newProvider === 'ollama') {
        const defaultModel = 'hamonize:latest';
        
        // 프로바이더에 기본 모델 추가
        setProviders((prev: any) => ({
          ...prev,
          [newProvider]: {
            ...prev[newProvider],
            models: [{
              id: defaultModel,
              name: defaultModel,
              size: 0
            }]
          }
        }));
        
        setOptions(prev => ({ 
          ...prev, 
          provider: newProvider,
          model: defaultModel
        }));
        
        localStorage.setItem('airun-selected-model', defaultModel);
        console.log(`프로바이더 ${newProvider}에 기본 모델 ${defaultModel} 설정 완료`);
      } else {
        // 다른 프로바이더인 경우 빈 모델로 설정
        setOptions(prev => ({ 
          ...prev, 
          provider: newProvider,
          model: '' // 모델을 빈 문자열로 설정
        }));
      }
    }
  };

  // 새 대화 버튼 클릭 시 바로 새 대화창을 보여줍니다.
  const handleNewChatClick = async () => {
    
    // URL에서 세션 파라미터 제거
    router.push('/code');
    
    // 모든 상태 초기화
    setCurrentSessionId(null); // 세션 ID 초기화 추가
    setCurrentMessages([]);
    setUploadedFiles([]);
    setInputMessage('');
    setError(null);
    setProgressMessages([]); // 새 대화 시작 시 진행 상황 메시지 초기화
    setIsNewChatMode(true);
    
    // 새 대화 시작 시 세션 목록 새로고침
    try {
      const latestSessions = await loadSessions();
      setSessions(latestSessions);
      // console.log('✅ 새 대화 시작 - 세션 목록 새로고침 완료');
    } catch (error) {
      console.error('❌ 새 대화 시작 시 세션 목록 새로고침 실패:', error);
    }
    
    // console.log('새 대화 시작 - 모든 상태 초기화 완료');
  };

  // 세션 목록을 불러오고 최신 세션 배열만 반환 (setSessions는 호출하지 않음)
  const loadSessions = async (): Promise<ChatSession[]> => {
    
    try {
      setIsLoadingSessions(true);
      setError(null);
      const response = await internalApiClient.getSessions('code');
      if (!response || !response.success || !response.data) {
        console.error('API 응답이 유효하지 않음:', response);
        throw new Error('API 응답이 유효하지 않습니다.');
      }
      const sessionsArray: ApiSession[] = Object.values(response.data as Record<string, ApiSession>);
      const sessionList: ChatSession[] = sessionsArray
        .filter((session: ApiSession) => {
          if (!session || typeof session !== 'object') return false;
          if (!Array.isArray(session.history)) return false;
          return session.history.some(msg => msg.role === 'assistant' && msg.content && typeof msg.content === 'string' && msg.content.trim().length > 0);
        })
        .map((session: ApiSession): ChatSession => {
          // 안전하게 content를 문자열로 변환하는 함수
          const safeStringify = (content: any): string => {
            if (typeof content === 'string') return content;
            if (content === null || content === undefined) return '';
            try {
              return typeof content === 'object' ? JSON.stringify(content) : String(content);
            } catch {
              return String(content);
            }
          };

          // 사용자 메시지와 어시스턴트 메시지 필터링
          const userMessages = (session.history ?? []).filter(msg => {
            return msg && msg.role === 'user' && msg.content;
          });
          
          const assistantMessages = (session.history ?? []).filter(msg => {
            return msg && msg.role === 'assistant' && msg.content;
          });
          
          // 제목은 항상 첫 번째 사용자 질문을 기준으로 생성 (AI 응답 대신 사용자 질문 우선)
          const title = userMessages.length > 0 
            ? (() => {
                // originalContent가 있으면 그것을 우선 사용 (멘션된 문서가 포함되기 전 원본 질문)
                const content = userMessages[0].originalContent 
                  ? safeStringify(userMessages[0].originalContent)
                  : safeStringify(userMessages[0].content);
                
                // 시스템 프롬프트나 긴 내용은 간단히 처리 (원본 질문 사용 시에는 이 조건이 덜 걸림)
                if (content.length > 1000 || content.includes('다음 문서들의 내용을 바탕으로') || content.includes('검색 결과:')) {
                  return t('chat.question', '질문'); // 시스템 프롬프트인 경우 기본 제목
                }
                return content.slice(0, 30) + (content.length > 30 ? '...' : '');
              })()
            : assistantMessages.length > 0 
              ? (() => {
                  const content = safeStringify(assistantMessages[0].content);
                  return content.slice(0, 30) + (content.length > 30 ? '...' : '');
                })()
              : t('chat.newConversation', '새 대화');
            
          const lastMessage = assistantMessages.length > 0
            ? (() => {
                const content = safeStringify(assistantMessages[assistantMessages.length - 1].content);
                return content.slice(0, 50) + (content.length > 50 ? '...' : '');
              })()
            : t('chat.newConversation', '새 대화');
            
          return {
            id: session.id,
            title,
            lastMessage,
            updatedAt: session.updated_at || new Date().toISOString(),
            createdAt: new Date().toISOString(),
            messageCount: session.history?.length ?? 0,
            provider: session.provider,
            model: session.model,
            history: session.history ?? []
          };
        });
      return sessionList;
    } catch (error) {
      console.error('세션 목록 로드 실패:', error);
      setError(t('chat.sessionLoadFailed', '세션 목록을 불러오는데 실패했습니다.'));
      setTimeout(() => setError(null), 3000);
      return [];
    } finally {
      setIsLoadingSessions(false);
    }
  };

  const selectSession = async (sessionId: string) => {
    // console.log('=== [DEBUG] 선택된 세션 id:', sessionId);
    if (sessionId === currentSessionId) return;



    // URL에 세션 파라미터 반영
    const url = new URL(window.location.href);
    url.searchParams.set('session', sessionId);
    window.history.replaceState({}, '', url.toString());

    try {
      setIsNewChatMode(false);
      setIsLoading(true);
      setCurrentSessionId(sessionId);
      setProgressMessages([]); // 다른 세션 선택 시 진행 상황 메시지 초기화

      const response = await externalApiClient.getSession(sessionId);

      if (response.success && response.data) {
        const sessionData = response.data;
        if (Array.isArray(sessionData.history)) {
          // 코드 모드에서 응답 내용을 Python 코드 블록으로 포맷팅하는 함수 (기존 세션용)
          const formatCodeResponseForExistingSession = (content: string): string => {
            if (!content) return content;
            
            // console.log('🔍 기존 세션 메시지 포맷팅:', content.slice(0, 100) + '...');
            
            // 이미 마크다운 코드 블록이 있는지 확인
            if (content.includes('```python') || content.includes('```')) {
              // console.log('✅ 이미 마크다운 코드 블록이 있음');
              return content;
            }
            
            // Python 코드 패턴 감지 (더 강화된 패턴)
            const pythonPatterns = [
              /def\s+\w+\s*\(/,                    // 함수 정의
              /class\s+\w+\s*[\(:]?/,             // 클래스 정의
              /import\s+\w+/,                     // import 문
              /from\s+\w+\s+import/,              // from import 문
              /if\s+__name__\s*==\s*['""]__main__['""]:/,  // main 체크
              /print\s*\(/,                       // print 함수
              /return\s+/,                        // return 문
              /for\s+\w+\s+in\s+/,               // for 루프
              /while\s+.+:/,                      // while 루프
              /if\s+.+:/,                         // if 문
              /try\s*:/,                          // try 문
              /except\s+.*:/,                     // except 문
              /"""[\s\S]*?"""/,                   // 독스트링
              /^\s*#.*$/m,                        // 주석
              /^\s*\w+\s*=\s*.+$/m,              // 변수 할당
              /^\s*\w+\.\w+\s*\(/m,              // 메서드 호출
            ];
            
            // Python 코드 패턴이 발견되면 코드 블록으로 포맷팅
            const isPythonCode = pythonPatterns.some(pattern => {
              const match = pattern.test(content);
              if (match) {
                console.log('🐍 기존 세션에서 Python 패턴 감지:', pattern.toString());
              }
              return match;
            });
            
            if (isPythonCode) {
              // 앞뒤 공백 제거하고 코드 블록으로 감싸기
              const cleanedContent = content.trim();
              const formatted = `\`\`\`python\n${cleanedContent}\n\`\`\``;
              console.log('🎯 기존 세션 Python 코드 블록으로 포맷팅 완료');
              return formatted;
            }
            
            // 더 넓은 코드 패턴 감지 (일반적인 프로그래밍 요소)
            const generalCodePatterns = [
              /\{\s*\n/,                          // 중괄호 시작
              /\}\s*$/m,                          // 중괄호 끝
              /^\s*[a-zA-Z_]\w*\s*\(/m,          // 함수 호출
              /^\s*[a-zA-Z_]\w*\s*=\s*[^=]/m,    // 변수 할당
              /;\s*$/m,                           // 세미콜론으로 끝
              /^\s*\/\//m,                        // C++ 스타일 주석
              /^\s*\/\*/m,                        // C 스타일 주석 시작
              /\*\/\s*$/m,                        // C 스타일 주석 끝
            ];
            
            const isGeneralCode = generalCodePatterns.some(pattern => pattern.test(content));
            
            if (isGeneralCode) {
              const cleanedContent = content.trim();
              const formatted = `\`\`\`\n${cleanedContent}\n\`\`\``;
              console.log('💻 기존 세션 일반 코드 블록으로 포맷팅 완료');
              return formatted;
            }
            
            // 코드 모드에서는 패턴이 감지되지 않아도 코드 블록으로 감싸기
            // (사용자가 명시적으로 코드 생성을 요청했으므로)
            const hasCodeLikeContent = content.length > 10 && (
              content.includes('(') || 
              content.includes(')') || 
              content.includes('=') ||
              content.includes(':') ||
              content.includes('\n')
            );
            
            if (hasCodeLikeContent) {
              const cleanedContent = content.trim();
              const formatted = `\`\`\`python\n${cleanedContent}\n\`\`\``;
              console.log('🔧 기존 세션에서 코드 모드 강제 Python 코드 블록으로 포맷팅');
              return formatted;
            }
            
            console.log('📝 기존 세션에서 코드 패턴 감지되지 않음, 원본 반환');
            return content;
          };

          // user와 assistant role만 필터링하되, 시스템 프롬프트나 내부 처리 메시지 제외
          const filteredHistory = sessionData.history.filter((message: Message) => {
            // 기본 역할 필터링
            if (message.role !== 'user' && message.role !== 'assistant') {
              return false;
            }
            
            // internal 플래그가 있는 메시지 제외 (도구 호출 응답 등)
            if ((message as any).internal === true) {
              // console.log('내부 메시지 제외 (도구 호출 응답):', message.content?.slice(0, 100) + '...');
              return false;
            }
            
            // 시스템 프롬프트 필터링 제거 - 히스토리 누락 방지를 위해 모든 사용자/어시스턴트 메시지 유지
            // originalContent가 있으면 UI에서는 그것을 표시하고, 백엔드에는 전체 content 전송
            
            return true;
          }).map((message: Message) => {
            // Assistant 메시지의 경우 코드 포맷팅 적용
            if (message.role === 'assistant' && message.content) {
              const originalContent = typeof message.content === 'string' ? message.content : String(message.content);
              const formattedContent = formatCodeResponseForExistingSession(originalContent);
              
              return {
                ...message,
                content: formattedContent
              };
            }
            
            return message;
          });
          
          setCurrentMessages(filteredHistory);
        } else {
          setCurrentMessages([]);
        }
      } else {
        setError(t('chat.sessionSelectFailed', '세션을 불러오는데 실패했습니다.'));
        setCurrentMessages([]);
      }
    } catch (error) {
      setError(t('chat.sessionSelectFailed', '세션을 불러오는데 실패했습니다.'));
      setCurrentMessages([]);
    } finally {
      setIsLoading(false);
    }
  };

  // 중복 메시지 제거 함수
  const removeDuplicateMessages = (messages: Message[]): Message[] => {
    if (!messages || messages.length === 0) return [];
    
    const seen = new Set<string>();
    const uniqueMessages: Message[] = [];
    
    // 안전하게 content를 문자열로 변환하는 함수
    const safeStringify = (content: any): string => {
      if (typeof content === 'string') return content.trim();
      if (content === null || content === undefined) return '';
      try {
        return typeof content === 'object' ? JSON.stringify(content) : String(content).trim();
      } catch {
        return String(content).trim();
      }
    };
    
    // 메시지를 순서대로 처리하되, 중복 체크는 뒤에서부터
    for (let i = 0; i < messages.length; i++) {
      const message = messages[i];
      if (!message || !message.role) continue;
      
      // content를 안전하게 문자열로 변환
      const content = safeStringify(message.content);
      if (!content) continue; // 빈 내용은 건너뛰기
      
      // 메시지 고유 키 생성 (역할 + 내용의 해시)
      const messageKey = `${message.role}:${content}`;
      
      // 뒤에 동일한 메시지가 있는지 확인 (최신 메시지 우선)
      const hasDuplicateAfter = messages.slice(i + 1).some(laterMsg => {
        if (!laterMsg || !laterMsg.role) return false;
        
        const laterContent = safeStringify(laterMsg.content);
        if (!laterContent) return false;
        
        const laterKey = `${laterMsg.role}:${laterContent}`;
        return laterKey === messageKey;
      });
      
      if (!hasDuplicateAfter && !seen.has(messageKey)) {
        seen.add(messageKey);
        // content를 문자열로 정규화해서 저장하되, id 필드 보존
        uniqueMessages.push({
          ...message,
          id: message.id || `msg-${Date.now()}-${Math.random().toString(36).substring(2, 8)}`,
          content: content
        });
      }
    }
    
    return uniqueMessages;
  };

  const createNewSession = async (): Promise<string | null> => {
    // 이미 생성 중이면 중복 실행 방지
    if (isLoading) return null;

    try {
      setIsLoading(true);
      
      const sessionData = {
        provider: options.provider,
        model: options.model,
        parameters: {
          temperature: options.temperature,
          max_tokens: options.maxTokens,
        },
        history: [],
        type: 'code',
        title: t('chat.newConversation', '새 대화'),
        userId: user?.id ? Number(user.id) : 0, // 숫자 ID
        username: user?.username, // 문자열 username
      };

      // 클라이언트에서 ID를 만들지 않고 서버에 생성을 요청합니다.
      const response = await externalApiClient.createSession(sessionData);
      
      // 서버로부터 받은 ID가 유효한지 확인합니다.
      if (response && response.id) {
          const newSessionId = response.id;
          
          const newSession: ChatSession = {
            id: newSessionId,
            title: t('chat.newConversation', '새 대화'),
            lastMessage: t('chat.newConversation', '새 대화'),
            updatedAt: new Date().toISOString(),
            createdAt: new Date().toISOString(),
            messageCount: 0,
            provider: options.provider,
            model: options.model,
            history: [],
          };

          // 새로운 세션으로 상태를 업데이트합니다.
          setCurrentMessages([]);
          setSessions(prev => [newSession, ...prev]);
          
          // URL도 새로운 세션 ID로 업데이트합니다.
          const url = new URL(window.location.href);
          url.searchParams.set('session', newSessionId);
          window.history.replaceState({}, '', url.toString());

          return newSessionId;
      } else {
          // 서버에서 ID를 받지 못한 경우 에러를 발생시킵니다.
          throw new Error('API 서버에서 세션 ID를 받지 못했습니다.');
      }
    } catch (error) {
      console.error('세션 생성 실패:', error);
      setError(t('chat.newChatFailed', '새 대화를 시작하지 못했습니다. 잠시 후 다시 시도해주세요.'));
      // 실패 시 null을 반환하여 후속 작업을 막습니다.
      return null;
    } finally {
      setIsLoading(false);
    }
  };

  // 파일 업로드 처리
  const handleFileSelect = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = event.target.files;
    if (!files || files.length === 0) return;

    // 이미 업로드 중이면 중단
    if (isUploading) {
      // console.log('이미 파일 업로드 중입니다.');
      return;
    }

    const processedFiles: File[] = [];
    
    // 파일들을 순차적으로 처리 (이미지는 압축, 일반 파일은 그대로)
    for (const file of Array.from(files)) {
      if (isImageFile(file)) {
        try {
          console.log(`이미지 압축 시작: ${file.name}`);
          const compressedFile = await compressImage(file);
          processedFiles.push(compressedFile);
        } catch (error) {
          console.error(`이미지 압축 실패 (${file.name}):`, error);
          // 압축 실패 시 원본 파일 사용
          processedFiles.push(file);
        }
      } else {
        processedFiles.push(file);
      }
    }

    setUploadedFiles(prev => [...prev, ...processedFiles]);

    // 이미지 파일의 경우 썸네일 URL 생성 (압축된 파일 기준)
    const newUrls = new Map<string, string>();
    processedFiles.forEach(file => {
      if (isImageFile(file)) {
        const fileKey = `${file.name}-${file.size}-${file.lastModified}`;
        const url = URL.createObjectURL(file);
        newUrls.set(fileKey, url);
      }
    });
    
    if (newUrls.size > 0) {
      setUploadedFileUrls(prev => new Map([...prev, ...newUrls]));
    }

    // 이미지가 아닌 파일들은 백그라운드 RAG 업로드
    for (const file of processedFiles) {
      if (!file.type.startsWith('image/')) {
        await handleFileUploadBackground(file);
      }
    }

    // 파일 입력 초기화
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  const uploadFileToRag = async (file: File) => {
    const fileKey = `${file.name}-${file.size}-${file.lastModified}`;
    
    // 이미 업로드 중인 파일이면 중단
    if (uploadingFiles.has(fileKey)) {
      // console.log(`파일 ${file.name}은 이미 업로드 중입니다.`);
      return;
    }

    try {
      setIsUploading(true);
      setUploadingFiles(prev => new Set(prev).add(fileKey));
      
      // console.log('파일 업로드 시작:', {
      //   fileName: file.name,
      //   fileKey,
      //   userId: user?.username,
      //   userInfo: user
      // });
      
      // 사용자명과 함께 파일 업로드 (채팅 페이지와 동일한 방식)
      await externalApiClient.uploadFile(
        file, 
        user?.username, 
        'code', // 타겟 폴더
        (progress, status) => {
          console.log(`파일 "${file.name}" ${status}: ${progress}%`);
        },
        false // 단일 파일 업로드
      );
      // console.log(`파일 업로드 완료: ${file.name}`);
      
      // 업로드 완료 후 RAG 파일 목록 갱신
      await loadRagFiles();
    } catch (error) {
      console.error('파일 업로드 실패:', error);
      
      // 오류 메시지에서 구체적인 내용 추출
      let errorMessage = `파일 업로드 실패: ${file.name}`;
      if (error instanceof Error) {
        if (error.message.includes('이미 존재합니다')) {
          errorMessage = error.message;
        } else {
          errorMessage += ` - ${error.message}`;
        }
      }
      
      setError(errorMessage);
    } finally {
      setIsUploading(false);
      setUploadingFiles(prev => {
        const newSet = new Set(prev);
        newSet.delete(fileKey);
        return newSet;
      });
    }
  };

  const removeFile = (index: number) => {
    setUploadedFiles(prev => {
      const fileToRemove = prev[index];
      
      // 이미지 파일인 경우 생성된 URL 해제
      if (fileToRemove && isImageFile(fileToRemove)) {
        const fileKey = `${fileToRemove.name}-${fileToRemove.size}-${fileToRemove.lastModified}`;
        setUploadedFileUrls(urlMap => {
          const url = urlMap.get(fileKey);
          if (url) {
            URL.revokeObjectURL(url);
          }
          const newMap = new Map(urlMap);
          newMap.delete(fileKey);
          return newMap;
        });
      }
      
      return prev.filter((_, i) => i !== index);
    });
  };

  const isImageFile = (file: File) => {
    return file.type.startsWith('image/');
  };

  // 이미지 압축 함수 추가
  const compressImage = async (file: File, maxWidth: number = 1024, maxHeight: number = 1024, quality: number = 0.8): Promise<File> => {
    return new Promise((resolve, reject) => {
      const canvas = document.createElement('canvas');
      const ctx = canvas.getContext('2d');
      const img = new Image();

      img.onload = () => {
        // 원본 크기
        const { width, height } = img;
        
        // 비율 유지하면서 크기 조정
        let newWidth = width;
        let newHeight = height;
        
        if (width > maxWidth || height > maxHeight) {
          const ratio = Math.min(maxWidth / width, maxHeight / height);
          newWidth = width * ratio;
          newHeight = height * ratio;
        }
        
        // Canvas 크기 설정
        canvas.width = newWidth;
        canvas.height = newHeight;
        
        // 이미지 그리기
        ctx?.drawImage(img, 0, 0, newWidth, newHeight);
        
        // Blob으로 변환
        canvas.toBlob(
          (blob) => {
            if (blob) {
              // 압축된 File 객체 생성
              const compressedFile = new File([blob], file.name, {
                type: file.type,
                lastModified: Date.now(),
              });
              
              console.log(`이미지 압축 완료: ${file.name}`);
              console.log(`  원본 크기: ${(file.size / 1024 / 1024).toFixed(2)}MB (${width}x${height})`);
              console.log(`  압축 후: ${(compressedFile.size / 1024 / 1024).toFixed(2)}MB (${newWidth}x${newHeight})`);
              console.log(`  압축률: ${((1 - compressedFile.size / file.size) * 100).toFixed(1)}%`);
              
              resolve(compressedFile);
            } else {
              reject(new Error('이미지 압축 실패'));
            }
          },
          file.type,
          quality
        );
      };
      
      img.onerror = () => reject(new Error('이미지 로드 실패'));
      img.src = URL.createObjectURL(file);
    });
  };

  // 드래그앤드롭 이벤트 핸들러들
  const handleDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOver(true);
  };

  const handleDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    // 드래그 영역을 완전히 벗어났을 때만 상태 변경
    if (!e.currentTarget.contains(e.relatedTarget as Node)) {
      setIsDragOver(false);
    }
  };

  const handleDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  const handleDrop = async (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOver(false);

    const files = Array.from(e.dataTransfer.files);
    if (files.length === 0) return;

    // 이미 업로드 중이면 중단
    if (isUploading) {
      // console.log('이미 파일 업로드 중입니다.');
      return;
    }

    // console.log('드래그앤드롭으로 파일 업로드:', files.map(f => f.name));
    
    const processedFiles: File[] = [];
    
    // 파일들을 순차적으로 처리 (이미지는 압축, 일반 파일은 그대로)
    for (const file of files) {
      if (isImageFile(file)) {
        try {
          console.log(`이미지 압축 시작: ${file.name}`);
          const compressedFile = await compressImage(file);
          processedFiles.push(compressedFile);
        } catch (error) {
          console.error(`이미지 압축 실패 (${file.name}):`, error);
          // 압축 실패 시 원본 파일 사용
          processedFiles.push(file);
        }
      } else {
        processedFiles.push(file);
      }
    }
    
    setUploadedFiles(prev => [...prev, ...processedFiles]);

    // 이미지 파일의 경우 썸네일 URL 생성 (압축된 파일 기준)
    const newUrls = new Map<string, string>();
    processedFiles.forEach(file => {
      if (isImageFile(file)) {
        const fileKey = `${file.name}-${file.size}-${file.lastModified}`;
        const url = URL.createObjectURL(file);
        newUrls.set(fileKey, url);
      }
    });
    
    if (newUrls.size > 0) {
      setUploadedFileUrls(prev => new Map([...prev, ...newUrls]));
    }

    // 이미지가 아닌 파일들은 백그라운드 RAG 업로드
    for (const file of processedFiles) {
      if (!file.type.startsWith('image/')) {
        await handleFileUploadBackground(file);
      }
    }
  };

  // sendMessage에서 타이핑 효과가 끝난 후에만 loadSessions 호출
  const sendMessage = async () => {
    if ((!inputMessage.trim() && uploadedFiles.length === 0) || isLoading) return;

    // @ 멘션된 문서들 추출 (이메일 주소 제외)
    const mentionPattern = /@([^\s@]+)/g;
    const mentions: string[] = [];
    let match;
    
    while ((match = mentionPattern.exec(inputMessage)) !== null) {
      const mentionText = match[1];
      // 이메일 주소 패턴인지 확인 (도메인이 포함된 경우 제외)
      if (!mentionText.includes('.') || !mentionText.match(/^[a-zA-Z0-9._%+-]+\.[a-zA-Z]{2,}$/)) {
        mentions.push(mentionText);
      }
    }

    // 세션 ID 처리: currentSessionId 우선, 없으면 URL에서 가져오기
    let sessionId = currentSessionId || searchParams.get('session');

    const imageFiles = uploadedFiles.filter(isImageFile);
    
    // 이미지 파일들을 메시지에 첨부
    const attachedImages = imageFiles.map(file => {
      const fileKey = `${file.name}-${file.size}-${file.lastModified}`;
      const url = uploadedFileUrls.get(fileKey);
      return {
        file,
        url: url || '',
        name: file.name
      };
    }).filter(img => img.url); // URL이 있는 것만 포함
    
    const userMessage: Message = {
      id: `msg-${Date.now()}-user`,
      role: 'user',
      content: inputMessage.trim(),
      timestamp: new Date().toISOString(),
      attachedImages: attachedImages.length > 0 ? attachedImages : undefined,
    };

    setCurrentMessages(prev => [...prev, userMessage]);
    const messageText = inputMessage.trim();
    setInputMessage('');
    
    // 업로드된 파일 목록만 정리 (URL은 메시지에 저장되어 유지됨)
    setUploadedFiles([]);
    // 새로 업로드할 파일을 위해 URL 맵 초기화 (기존 URL은 메시지에 보존됨)
    setUploadedFileUrls(new Map());
    
    setMentionedDocs([]); // 멘션 상태 초기화
    setShowDocSuggestions(false);
    setMentionQuery('');
    setSelectedSuggestionIndex(0);
    resetTextareaHeight();
    setIsLoading(true);
    setError(null);

    try {
      // 기존 세션이 있으면 user 메시지를 서버에 저장
      if (sessionId) {
        await externalApiClient.updateSession(sessionId, {
          history: [...currentMessages, userMessage]
        });
      }
      
      // 첫 번째 이미지 파일 (있는 경우)
      const imageFile = imageFiles.length > 0 ? imageFiles[0] : undefined;

      // 멘션된 문서들의 전체 내용 가져오기
      let mentionedDocumentContent = '';
      let cleanMessage = inputMessage.trim();
      
      if (mentions.length > 0) {
        try {
          mentionedDocumentContent = await getMentionedDocumentsContent(mentions);
          // @ 멘션 제거한 깔끔한 질문 (이메일 주소는 제외하고 문서 멘션만 제거)
          cleanMessage = cleanMessage.replace(/@([^\s@]+)/g, (match, captured) => {
            // 이메일 주소 패턴인지 확인
            if (captured.includes('.') && captured.match(/^[a-zA-Z0-9._%+-]+\.[a-zA-Z]{2,}$/)) {
              return match; // 이메일 주소는 그대로 유지
            }
            return ''; // 문서 멘션만 제거
          }).trim();
        } catch (error) {
          console.error('멘션된 문서 내용 가져오기 실패:', error);
        }
      }
      
      // 이미지가 첨부된 경우 검색 옵션 자동 비활성화
      const hasImageFiles = imageFiles.length > 0;
      
      if (hasImageFiles) {
        console.log('🖼️ 이미지 파일 감지 - 검색 옵션 자동 비활성화:', imageFiles.map(f => f.name));
      }
      
      // 사용자 정보 확인
      const username = user?.username;
      
      // 코드 모드 옵션 설정
      const codeOptions = {
        provider: options.provider,
        model: options.model,
        parameters: {
          temperature: options.temperature,
          max_tokens: options.maxTokens,
        },
        rag: (mentions.length > 0 || hasImageFiles) ? false : options.enableRag, // 멘션 사용 시 또는 이미지 첨부 시 RAG 비활성화
        web: hasImageFiles ? false : options.enableWebSearch, // 이미지 첨부 시 웹 검색 비활성화
        enableSmartResearch: hasImageFiles ? false : options.enableSmartResearch, // 이미지 첨부 시 AI프로 비활성화
        ragSearchScope: options.ragSearchScope, // RAG 검색 범위 전달
        userId: Number(options.userId), // 정수형 ID (세션 관리용)
        username: username, // 문자열 사용자명 (RAG 검색용)
        type: 'code', // 세션 타입을 'code'로 설정
        mentionedDocumentContent: mentionedDocumentContent, // 멘션된 문서 내용 별도 전송
        mentionedDocuments: mentions, // 멘션된 문서 목록
        autoExecute: false, // 자동 실행 옵션 (필요에 따라 설정)
        hideCode: false, // 코드 숨기기 옵션
        mode: 'code', // 모드 명시
      };

      // 코드 모드 API 호출 - codeAsyncAndWait 사용
      const result = await externalApiClient.codeAsyncAndWait(
        messageText || (imageFile ? t('chat.describeImage', '이 이미지에 대해 설명해주세요.') : ''),
        sessionId || undefined, // null을 undefined로 변환
        codeOptions,
        (status: string) => {
          // 기술적 상태 메시지를 사용자 친화적 메시지로 변환
          const convertTechnicalStatus = (msg: string): string => {
            const lowerMsg = msg.toLowerCase();
            if (lowerMsg.includes('active')) {
              return '잠시만 기다려주세요...';
            }
            if (lowerMsg.includes('pending')) {
              return '요청을 처리하고 있습니다...';
            }
            if (lowerMsg.includes('processing')) {
              return '처리 중입니다...';
            }
            return msg; // 기존 메시지 그대로 반환
          };

          // 진행 상태 업데이트
          try {
            const progressData = JSON.parse(status);
            if (progressData.type === 'progress_array' && Array.isArray(progressData.messages)) {
              const convertedMessages = progressData.messages.map((msg: string) => convertTechnicalStatus(msg));
              setProgressMessages(convertedMessages);
              return;
            }
          } catch (e) {
            // JSON 파싱 실패 시 기존 로직 계속 진행
          }
          
          // 기존 로직: status에 '|'가 포함되어 있으면 여러 progress 메시지
          if (status.includes(' | ')) {
            const messages = status.split(' | ')
              .filter(msg => msg.trim().length > 0)
              .map(msg => convertTechnicalStatus(msg));
            setProgressMessages(messages);
          } else {
            setProgressMessages([convertTechnicalStatus(status)]);
          }
        },
        imageFile
      );

      let responseContent = '';

      // 안전하게 문자열로 변환하는 헬퍼 함수
      const safeStringify = (value: any): string => {
        if (typeof value === 'string') return value;
        if (value === null || value === undefined) return '';
        try {
          return typeof value === 'object' ? JSON.stringify(value) : String(value);
        } catch {
          return String(value);
        }
      };

      // 코드 모드에서 응답 내용을 Python 코드 블록으로 포맷팅하는 함수
      const formatCodeResponse = (content: string): string => {
        if (!content) return content;
        
        // console.log('🔍 formatCodeResponse 입력 내용:', content);
        
        // 이미 마크다운 코드 블록이 있는지 확인
        if (content.includes('```python') || content.includes('```')) {
          console.log('✅ 이미 마크다운 코드 블록이 있음');
          return content;
        }
        
        // Python 코드 패턴 감지 (더 강화된 패턴)
        const pythonPatterns = [
          /def\s+\w+\s*\(/,                    // 함수 정의
          /class\s+\w+\s*[\(:]?/,             // 클래스 정의
          /import\s+\w+/,                     // import 문
          /from\s+\w+\s+import/,              // from import 문
          /if\s+__name__\s*==\s*['""]__main__['""]:/,  // main 체크
          /print\s*\(/,                       // print 함수
          /return\s+/,                        // return 문
          /for\s+\w+\s+in\s+/,               // for 루프
          /while\s+.+:/,                      // while 루프
          /if\s+.+:/,                         // if 문
          /try\s*:/,                          // try 문
          /except\s+.*:/,                     // except 문
          /"""[\s\S]*?"""/,                   // 독스트링
          /^\s*#.*$/m,                        // 주석
          /^\s*\w+\s*=\s*.+$/m,              // 변수 할당
          /^\s*\w+\.\w+\s*\(/m,              // 메서드 호출
        ];
        
        // Python 코드 패턴이 발견되면 코드 블록으로 포맷팅
        const isPythonCode = pythonPatterns.some(pattern => {
          const match = pattern.test(content);
          // if (match) {
          //   console.log('🐍 Python 패턴 감지:', pattern.toString());
          // }
          return match;
        });
        
        if (isPythonCode) {
          // 앞뒤 공백 제거하고 코드 블록으로 감싸기
          const cleanedContent = content.trim();
          const formatted = `\`\`\`python\n${cleanedContent}\n\`\`\``;
          // console.log('🎯 Python 코드 블록으로 포맷팅 완료:', formatted);
          return formatted;
        }
        
        // 더 넓은 코드 패턴 감지 (일반적인 프로그래밍 요소)
        const generalCodePatterns = [
          /\{\s*\n/,                          // 중괄호 시작
          /\}\s*$/m,                          // 중괄호 끝
          /^\s*[a-zA-Z_]\w*\s*\(/m,          // 함수 호출
          /^\s*[a-zA-Z_]\w*\s*=\s*[^=]/m,    // 변수 할당
          /;\s*$/m,                           // 세미콜론으로 끝
          /^\s*\/\//m,                        // C++ 스타일 주석
          /^\s*\/\*/m,                        // C 스타일 주석 시작
          /\*\/\s*$/m,                        // C 스타일 주석 끝
        ];
        
        const isGeneralCode = generalCodePatterns.some(pattern => pattern.test(content));
        
        if (isGeneralCode) {
          const cleanedContent = content.trim();
          const formatted = `\`\`\`\n${cleanedContent}\n\`\`\``;
          // console.log('💻 일반 코드 블록으로 포맷팅 완료:', formatted);
          return formatted;
        }
        
        // 코드 모드에서는 패턴이 감지되지 않아도 코드 블록으로 감싸기
        // (사용자가 명시적으로 코드 생성을 요청했으므로)
        const hasCodeLikeContent = content.length > 10 && (
          content.includes('(') || 
          content.includes(')') || 
          content.includes('=') ||
          content.includes(':') ||
          content.includes('\n')
        );
        
        if (hasCodeLikeContent) {
          const cleanedContent = content.trim();
          const formatted = `\`\`\`python\n${cleanedContent}\n\`\`\``;
          console.log('🔧 코드 모드에서 강제로 Python 코드 블록으로 포맷팅:', formatted);
          return formatted;
        }
        
        console.log('📝 코드 패턴 감지되지 않음, 원본 반환');
        return content;
      };

      // 1. choices[0].message.content (기존 OpenAI 스타일)
      if (Array.isArray(result?.choices) && result.choices.length > 0) {
        const msg = result.choices[0].message;
        if (msg && msg.content) {
          const content = safeStringify(msg.content);
          if (content.trim().length > 0) {
            responseContent = formatCodeResponse(content);
          }
        }
      }

      // 2. history에서 마지막 assistant 메시지 content 추출 (airun/ollama 스타일)
      if (!responseContent && Array.isArray(result?.history)) {
        // history에서 role이 'assistant'인 마지막 메시지 찾기
        const lastAssistant = [...result.history].reverse().find(
          (msg) => msg && msg.role === 'assistant' && msg.content
        );
        if (lastAssistant) {
          const content = safeStringify(lastAssistant.content);
          if (content.trim().length > 0) {
            responseContent = formatCodeResponse(content);
          }
        }
      }

      // 3. 코드 모드 특화: python_code 필드 처리
      if (!responseContent && result?.python_code) {
        const pythonCode = safeStringify(result.python_code);
        if (pythonCode.trim().length > 0) {
          // Python 코드를 마크다운 코드 블록으로 포맷팅
          responseContent = `## 🐍 생성된 Python 코드\n\n\`\`\`python\n${pythonCode}\n\`\`\``;
          
          // 실행 결과가 있으면 추가
          if (result?.execution_result || result?.executionResult) {
            const execResult = result.execution_result || result.executionResult;
            if (execResult) {
              responseContent += `\n\n## 📊 실행 결과\n\n\`\`\`\n${safeStringify(execResult)}\n\`\`\``;
            }
          }
        }
      }

      // 4. 기타 백업
      if (!responseContent && result?.response) {
        const content = safeStringify(result.response);
        if (content.trim().length > 0) {
          responseContent = formatCodeResponse(content);
        }
      }
      if (!responseContent && result?.content) {
        const content = safeStringify(result.content);
        if (content.trim().length > 0) {
          responseContent = formatCodeResponse(content);
        }
      }
      if (!responseContent && result?.text) {
        const content = safeStringify(result.text);
        if (content.trim().length > 0) {
          responseContent = formatCodeResponse(content);
        }
      }
      if (!responseContent) {
        // 혹시라도 choices가 여러 개라면, 첫 번째가 아니라 두 번째, 세 번째에 있을 수도 있음
        if (Array.isArray(result?.choices)) {
          for (const choice of result.choices) {
            if (choice?.message?.content) {
              const content = safeStringify(choice.message.content);
              if (content.trim().length > 0) {
                responseContent = formatCodeResponse(content);
                break;
              }
            }
          }
        }
      }
      // 서버에서 반환된 세션 ID (새 세션인 경우 새로 생성된 ID)
      const newSessionId = result?.sessionId || result?.data?.sessionId;
      
      // console.log('🔍 세션 ID 추출 정보:');
      // console.log('  - result?.sessionId:', result?.sessionId);
      // console.log('  - result?.data?.sessionId:', result?.data?.sessionId);
      // console.log('  - newSessionId:', newSessionId);
      // console.log('  - currentSessionId:', sessionId);

      if (responseContent) {
        // 응답에서 downloadInfo 추출
        const downloadInfo = result?.downloadInfo || [];
        
        const assistantMessage: Message = {
          id: `msg-${Date.now()}-assistant`,
          role: 'assistant',
          content: '', // 타이핑 효과를 위해 초기 내용은 비워둡니다.
          timestamp: new Date().toISOString(),
          downloadInfo: downloadInfo,
        };

        // 세션 ID 처리 (새 세션인 경우 URL 및 상태 업데이트)
        const targetSessionId = newSessionId || sessionId;
        
        if (newSessionId && newSessionId !== sessionId) {
          // console.log('🆕 새 세션 생성됨:', newSessionId);
          setCurrentSessionId(newSessionId);
          router.push(`/code?session=${newSessionId}`);
        }

        // UI에 빈 어시스턴트 메시지 추가
        setCurrentMessages(prev => [...prev, assistantMessage]);
        
        // console.log('🚀 AI 응답 받음, 타이핑 효과 시작 준비:', { 
        //   assistantMessageId: assistantMessage.id, 
        //   responseContentLength: responseContent.length,
        //   responsePreview: responseContent.slice(0, 100) + '...'
        // });
        
        // 타이핑 효과 시작, 완료 후 서버에 최종 히스토리 저장
        typeMessage(assistantMessage.id, responseContent, 2, async () => {
          // console.log('📚 타이핑 완료, 서버에 히스토리 저장 시작');
          
          // 완성된 어시스턴트 메시지로 업데이트 (downloadInfo 유지)
          const completeAssistantMessage = { 
            ...assistantMessage, 
            content: responseContent,
            downloadInfo: downloadInfo
          };
          setCurrentMessages(prev => prev.map(msg => 
            msg.id === assistantMessage.id ? completeAssistantMessage : msg
          ));
          
          // 서버에 최종 히스토리 저장
          const updatedMessages = [...currentMessages, userMessage, completeAssistantMessage];
        
          if (targetSessionId) {
            try {
              const updateResponse = await externalApiClient.updateSession(targetSessionId, {
                history: updatedMessages
              });
              
              // 약간의 지연 후 세션 목록 새로고침
              await new Promise(resolve => setTimeout(resolve, 1000));
              
              const latestSessions = await loadSessions();
              setSessions(latestSessions);
              
              // 현재 세션의 제목도 업데이트
              const foundSession = latestSessions.find(s => s.id === targetSessionId);
              if (foundSession) {
                updateSessionInList(targetSessionId, foundSession.history);
              } else {
                console.warn('⚠️ 현재 세션이 목록에서 발견되지 않음:', targetSessionId);
              }
              
            } catch (error) {
              console.error('❌ 서버에 히스토리 저장 또는 세션 목록 새로고침 실패:', error);
            }
          } else {
            console.warn('⚠️ targetSessionId가 없어서 서버 저장 건너뜀');
          }
        });

      } else {
        throw new Error('AI 응답을 받지 못했습니다.');
      }
    } catch (error) {
      console.error('메시지 전송 실패:', error);
      setError(t('chat.messageSendFailed', '메시지 전송에 실패했습니다.'));
    } finally {
      setIsLoading(false);
      // 응답을 받은 진행 메시지를 지웁니다.
      setTimeout(() => {
        setProgressMessages([]);
      }, 200);
    }
  };

  const updateSessionInList = (sessionId: string, messages: Message[]) => {
    // 안전하게 content를 문자열로 변환하는 함수
    const safeStringify = (content: any): string => {
      if (typeof content === 'string') return content;
      if (content === null || content === undefined) return '';
      try {
        return typeof content === 'object' ? JSON.stringify(content) : String(content);
      } catch {
        return String(content);
      }
    };

    setSessions(prev => prev.map(session => {
      if (session.id === sessionId) {
        const lastMessage = messages.length > 0 
          ? safeStringify(messages[messages.length - 1]?.content) || t('chat.newConversation', '새 대화')
          : t('chat.newConversation', '새 대화');
        
        return {
          ...session,
          lastMessage: lastMessage.length > 50 ? lastMessage.slice(0, 50) + '...' : lastMessage,
          messageCount: messages.length,
          updatedAt: new Date().toISOString(),
          history: messages,
          title: session.title === t('chat.newConversation', '새 대화') && messages.length > 0 
            ? (() => {
                // 첫 번째 사용자 메시지를 찾아서 제목으로 사용
                const firstUserMessage = messages.find(msg => 
                  msg && msg.role === 'user' && msg.content
                );
                if (firstUserMessage) {
                  // originalContent가 있으면 그것을 우선 사용 (멘션된 문서가 포함되기 전 원본 질문)
                  const content = firstUserMessage.originalContent 
                    ? safeStringify(firstUserMessage.originalContent)
                    : safeStringify(firstUserMessage.content);
                  
                  // 시스템 프롬프트나 긴 내용은 간단히 처리 (원본 질문 사용 시에는 이 조건이 덜 걸림)
                  if (content.length > 1000 || content.includes('다음 문서들의 내용을 바탕으로') || content.includes('검색 결과:')) {
                    return t('chat.question', '질문'); // 시스템 프롬프트인 경우 기본 제목
                  }
                  return content.slice(0, 30) + (content.length > 30 ? '...' : '');
                } else {
                  // 사용자 메시지가 없으면 AI 응답을 사용
                  const firstAIResponse = messages.find(msg => 
                    msg && msg.role === 'assistant' && msg.content
                  );
                  const content = firstAIResponse?.content ? safeStringify(firstAIResponse.content) : t('chat.newConversation', '새 대화');
                  return content.slice(0, 30) + (content.length > 30 ? '...' : '');
                }
              })()
            : session.title,
        };
      }
      return session;
    }));
  };

  const handleDeleteClick = (sessionId: string) => {
    // console.log('[DEBUG] 삭제 아이콘 클릭, sessionId:', sessionId);
    setSessionToDelete(sessionId);
    setShowDeleteModal(true);
  };

  // confirmDelete에서 loadSessions의 반환값을 받아 setSessions로 강제 동기화
  const confirmDelete = async () => {
    // console.log('[DEBUG] confirmDelete 진입, sessionToDelete:', sessionToDelete);
    if (!sessionToDelete) return;
    try {
      // console.log('[DEBUG] 세션 삭제 요청:', sessionToDelete);
      const response = await externalApiClient.deleteSession(sessionToDelete);
      // console.log('[DEBUG] 세션 삭제 응답:', response);
      // 삭제 후 서버에서 최신 목록을 다시 불러오고, 최신 세션 배열을 받음
      const latestSessions = await loadSessions();
      setSessions(latestSessions); // 최신 세션 배열로 강제 동기화
      if (sessionToDelete === currentSessionId) {
        setCurrentSessionId(null);
        setCurrentMessages([]);
        const url = new URL(window.location.href);
        url.searchParams.delete('session');
        window.history.replaceState({}, '', url.toString());
        setIsNewChatMode(true);
      } else {
        const exists = latestSessions.some(s => s.id === currentSessionId);
        if (!exists) {
          setCurrentSessionId(null);
          setCurrentMessages([]);
          const url = new URL(window.location.href);
          url.searchParams.delete('session');
          window.history.replaceState({}, '', url.toString());
          setIsNewChatMode(true);
        }
      }
    } catch (error) {
      console.error('세션 삭제 실패:', error);
      setError(t('chat.deleteSessionFailed', '대화 삭제에 실패했습니다.'));
    } finally {
      setShowDeleteModal(false);
      setSessionToDelete(null);
    }
  };

  const cancelDelete = () => {
    setShowDeleteModal(false);
    setSessionToDelete(null);
  };

  // 전체 대화목록 삭제 핸들러
  const handleDeleteAllClick = () => {
    setShowDeleteAllModal(true);
  };

  const confirmDeleteAll = async () => {
    try {
      await externalApiClient.deleteAllSessions();
      
      // 상태 초기화
      setSessions([]);
      setCurrentMessages([]);
      setIsNewChatMode(true);
      
      // URL에서 세션 파라미터 제거
      const url = new URL(window.location.href);
      url.searchParams.delete('session');
      window.history.replaceState({}, '', url.toString());
      
      setShowDeleteAllModal(false);
      
      // 성공 토스트 표시
      setShowToast({ message: t('chat.allChatsDeleted', '모든 대화가 삭제되었습니다.'), type: 'success' });
      setTimeout(() => setShowToast(null), 3000);
      
    } catch (error) {
      console.error('전체 세션 삭제 실패:', error);
      setError(t('chat.deleteAllFailed', '전체 대화 삭제에 실패했습니다.'));
      
      // 에러 토스트 표시
      setShowToast({ message: t('chat.deleteAllFailed', '전체 대화 삭제에 실패했습니다.'), type: 'error' });
      setTimeout(() => setShowToast(null), 3000);
    }
  };

  const cancelDeleteAll = () => {
    setShowDeleteAllModal(false);
  };

  const handleKeyDown = (e: React.KeyboardEvent) => {
    // 문서 자동완성이 표시된 상태에서 키보드 네비게이션 처리
    if (showDocSuggestions && filteredDocsForMention.length > 0) {
      if (e.key === 'ArrowDown') {
        e.preventDefault();
        setSelectedSuggestionIndex(prev => 
          prev < filteredDocsForMention.length - 1 ? prev + 1 : 0
        );
        return;
      }
      
      if (e.key === 'ArrowUp') {
        e.preventDefault();
        setSelectedSuggestionIndex(prev => 
          prev > 0 ? prev - 1 : filteredDocsForMention.length - 1
        );
        return;
      }
      
      if (e.key === 'Enter' && !e.shiftKey) {
        e.preventDefault();
        const selectedDoc = filteredDocsForMention[selectedSuggestionIndex];
        if (selectedDoc) {
          selectDocument(selectedDoc.name);
        }
        return;
      }
      
      if (e.key === 'Escape') {
        e.preventDefault();
        setShowDocSuggestions(false);
        setMentionQuery('');
        setSelectedSuggestionIndex(0);
        return;
      }
    }
    
    // 일반 메시지 전송
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  };

  // filteredSessions를 useMemo로 감싸서 항상 updatedAt 내림차순 정렬을 보장
  const filteredSessions = useMemo(() => {
    return sessions
      .filter(session => {
        const safeTitle = typeof session.title === 'string' ? session.title : '';
        const safeLastMessage = typeof session.lastMessage === 'string' ? session.lastMessage : '';
        const safeSearchTerm = typeof searchTerm === 'string' ? searchTerm : '';
        return safeTitle.toLowerCase().includes(safeSearchTerm.toLowerCase()) ||
               safeLastMessage.toLowerCase().includes(safeSearchTerm.toLowerCase());
      })
      .sort((a, b) => {
        // createdAt 기준 내림차순 정렬
        const aTime = a.createdAt ? new Date(a.createdAt).getTime() : 0;
        const bTime = b.createdAt ? new Date(b.createdAt).getTime() : 0;
        return bTime - aTime;
      });
  }, [sessions, searchTerm]);

  const formatDate = (dateString: string) => {
    const date = new Date(dateString);
    const now = new Date();
    const diffInHours = (now.getTime() - date.getTime()) / (1000 * 60 * 60);

    if (diffInHours < 1) {
      return t('chat.justNow', '방금 전');
    } else if (diffInHours < 24) {
      return `${Math.floor(diffInHours)}시간 전`;
    } else if (diffInHours < 24 * 7) {
      return `${Math.floor(diffInHours / 24)}일 전`;
    } else {
      return date.toLocaleDateString('ko-KR');
    }
  };

  // 백그라운드 작업 완료 시 콜백
  const handleJobComplete = async (job: BackgroundJob) => {
    console.log('백그라운드 작업 완료:', job);
    
    if (job.type === 'file_upload') {
      // RAG 파일 목록 새로고침
      await loadRagFiles();
    }
  };

  // 백그라운드 파일 업로드 함수
  const handleFileUploadBackground = async (file: File) => {
    if (!user?.username) {
      setError('로그인이 필요합니다.');
      return;
    }

    try {
      // 파일을 Base64로 변환
      const fileData = await new Promise<{
        name: string;
        size: number;
        type: string;
        content: string;
      }>((resolve, reject) => {
        const reader = new FileReader();
        reader.onload = () => {
          const result = reader.result as string;
          const base64Content = result.split(',')[1]; // data:type;base64, 제거
          resolve({
            name: file.name,
            size: file.size,
            type: file.type,
            content: base64Content
          });
        };
        reader.onerror = reject;
        reader.readAsDataURL(file);
      });

      const response = await fetch('/api/support/background-jobs', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        credentials: 'include',
        body: JSON.stringify({
          type: 'file_upload',
          userId: user.username,
          filename: file.name,
          fileData: fileData,
          metadata: {
            pageType: 'code'
          }
        }),
      });

      if (response.ok) {
        const result = await response.json();
        if (result.success) {
          console.log(`파일 ${file.name} 백그라운드 업로드 시작:`, result.jobId);
          
          // 즉시 백그라운드 작업 상태 새로고침 트리거
          setRefreshTrigger(prev => prev + 1);
          
          return true;
        } else {
          throw new Error(result.error || '백그라운드 작업 생성 실패');
        }
      } else {
        const errorData = await response.json().catch(() => ({ error: '서버 오류' }));
        throw new Error(errorData.error || '백그라운드 작업 생성 실패');
      }
    } catch (error) {
      console.error('백그라운드 파일 업로드 실패:', error);
      setError(`파일 업로드 실패: ${error instanceof Error ? error.message : '알 수 없는 오류'}`);
      return false;
    }
  };

  // RAG 파일 목록 로드
  const loadRagFiles = async () => {
    if (!user?.username) {
      // console.log('사용자 정보가 없어서 RAG 파일 목록을 로드할 수 없습니다.');
      return;
    }
    
    try {
      // console.log('RAG 파일 목록 로드 시작:', user.username);
      setIsLoadingRagFiles(true);
      // 개인 문서 목록 조회 (모든 개인 페이지에서 공유)
      const response = await fetch(`/api/chat/documents?userId=${user.username}&checkFiles=true`, {
        method: 'GET',
        credentials: 'include'
      });
      const result = await response.json();
      
      // console.log('RAG 파일 목록 API 응답:', response);
      
      if (result.success) {
        // 데이터베이스 기반 응답 형식 처리
        const files = result.data || [];
        // console.log('서버에서 받은 파일 목록:', files);
        
        // 서버에서 온 파일들은 모두 완료 상태로 설정, 데이터베이스 형식을 RAG 형식으로 변환
        const completedFiles = files.map((file: any) => ({
          name: file.filename,
          size: file.filesize,
          type: file.mimetype,
          created_at: file.created_at,
          file_exists: file.file_exists,
          status: 'completed' as const
        }));
        
        // console.log('완료 상태로 변환된 파일들:', completedFiles);
        
        // 기존 처리 중인 파일들과 병합 (중복 제거)
        setRagFiles(prev => {
          // console.log('기존 파일 목록:', prev);
          const processingFiles = prev.filter((f: RagFile) => f.status === 'uploading' || f.status === 'processing');
          const existingNames = new Set(completedFiles.map((f: RagFile) => f.name));
          const uniqueProcessingFiles = processingFiles.filter((f: RagFile) => !existingNames.has(f.name));
          const finalFiles = [...uniqueProcessingFiles, ...completedFiles];
          // console.log('최종 파일 목록:', finalFiles);
          return finalFiles;
        });
      } else {
        console.error('RAG 파일 목록 로드 실패:', result.error?.message);
      }
    } catch (error) {
      console.error('RAG 파일 목록 로드 오류:', error);
    } finally {
      setIsLoadingRagFiles(false);
    }
  };

  // RAG 파일 삭제
  const deleteRagFile = async (filename: string) => {
    if (!user?.username) return;
    
    try {
      // Code 페이지 개인 문서 삭제 (데이터베이스 기반)
      const response = await fetch('/api/chat/documents', {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json',
        },
        credentials: 'include',
        body: JSON.stringify({ 
          filename: filename, 
          id: user.username
        }),
      });

      const result = await response.json();

      if (result.success) {
        console.log(`Code 페이지에서 파일 ${filename} 삭제 성공`);
        // 파일 목록 새로고침
        await loadRagFiles();
      } else {
        console.error('Code 페이지 파일 삭제 실패:', result.error?.message);
      }
    } catch (error) {
      console.error('Code 페이지 파일 삭제 오류:', error);
    }
  };;

  // RAG 파일 선택 핸들러
  const handleRagFileSelect = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = event.target.files;
    if (!files || files.length === 0) return;

    // 이미 업로드 중이면 중단
    if (isUploading) {
      // console.log('이미 파일 업로드 중입니다.');
      return;
    }

    const newFiles = Array.from(files);
    
    // RAG 파일 목록에 업로딩 상태로 추가
    const ragFiles: RagFile[] = newFiles.map(file => ({
      name: file.name,
      size: file.size,
      modified: new Date().toISOString(),
      type: file.name.split('.').pop()?.toLowerCase() || '',
      status: 'uploading',
      progress: 0
    }));

    setRagFiles(prev => [...ragFiles, ...prev]);

    // 각 파일을 순차적으로 백그라운드 업로드
    for (const file of newFiles) {
      await handleFileUploadBackground(file);
    }

    // 파일 입력 초기화
    if (ragFileInputRef.current) {
      ragFileInputRef.current.value = '';
    }
  };

  // RAG 드래그 앤 드롭 핸들러
  const handleRagDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOverRag(true);
  };

  const handleRagDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (!e.currentTarget.contains(e.relatedTarget as Node)) {
      setIsDragOverRag(false);
    }
  };

  const handleRagDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  // 폴더 내부 파일들을 재귀적으로 수집하는 함수
  const getAllFilesFromItems = async (items: DataTransferItemList): Promise<File[]> => {
    const files: File[] = [];
    
    const processEntry = async (entry: any): Promise<void> => {
      if (entry.isFile) {
        return new Promise((resolve) => {
          entry.file((file: File) => {
            // 지원되는 파일 확장자 확인
            const fileExt = file.name.split('.').pop()?.toLowerCase();
            const supportedExtensions = ['.pdf', '.txt', '.md', '.doc', '.docx', '.hwp', '.ppt', '.pptx', '.xls', '.xlsx'];
            
            if (fileExt && supportedExtensions.includes(`.${fileExt}`)) {
              // console.log(`폴더에서 파일 발견: ${file.name} (${file.size} bytes)`);
              files.push(file);
            } else {
              // console.log(`지원되지 않는 파일 형식: ${file.name}`);
            }
            resolve();
          });
        });
      } else if (entry.isDirectory) {
        const dirReader = entry.createReader();
        return new Promise((resolve) => {
          dirReader.readEntries(async (entries: any[]) => {
            for (const childEntry of entries) {
              await processEntry(childEntry);
            }
            resolve();
          });
        });
      }
    };

    // 모든 아이템 처리
    for (let i = 0; i < items.length; i++) {
      const item = items[i];
      if (item.kind === 'file') {
        const entry = item.webkitGetAsEntry();
        if (entry) {
          await processEntry(entry);
        }
      }
    }

    return files;
  };

  const handleRagDrop = async (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOverRag(false);

    console.log('드롭 이벤트 처리 시작...');
    console.log('dataTransfer.files 길이:', e.dataTransfer.files?.length);

    // 직접 dataTransfer.files 사용 (가장 안정적인 방법)
    const allFiles = Array.from(e.dataTransfer.files);
    console.log('드롭된 파일들:', allFiles.map(f => `${f.name} (${f.size} bytes, ${f.type})`));

    if (allFiles.length === 0) {
      console.log('드롭된 파일이 없습니다.');
      return;
    }

    // 지원되는 파일만 필터링
    const supportedExtensions = ['pdf', 'txt', 'md', 'doc', 'docx', 'hwp', 'hwpx', 'ppt', 'pptx', 'xls', 'xlsx'];
    
    const files = allFiles.filter(file => {
      const fileExt = file.name.split('.').pop()?.toLowerCase();
      const hasExtension = file.name.includes('.') && !file.name.endsWith('.');
      const isNotHidden = !file.name.startsWith('.');
      const isSupportedExtension = fileExt && supportedExtensions.includes(fileExt);
      const isValidFile = hasExtension && isNotHidden && isSupportedExtension;

      console.log(`파일 ${file.name}: 확장자=${fileExt}, 지원여부=${isSupportedExtension}, 결과=${isValidFile ? 'INCLUDE' : 'EXCLUDE'}`);

      return isValidFile;
    });

    console.log('필터링 후 최종 파일 목록:', files.map(f => f.name));

    if (files.length === 0) {
      console.log('업로드할 수 있는 파일이 없습니다.');
      return;
    }

    if (!user?.username) {
      alert('로그인이 필요합니다.');
      return;
    }

    console.log(`총 ${files.length}개 파일을 업로드합니다.`);

    // 각 파일을 순차적으로 업로드
    for (const file of files) {
      try {
        await externalApiClient.uploadFile(
          file, 
          user.username, 
          'code',
          (progress, status) => {
            setRagFiles(prev => {
              const index = prev.findIndex(f => f.name === file.name);
              if (index !== -1) {
                const updatedFiles = [...prev];
                updatedFiles[index] = {
                  ...updatedFiles[index],
                  status,
                  progress
                };
                return updatedFiles;
              } else {
                return [...prev, {
                  name: file.name,
                  size: file.size,
                  status,
                  progress
                }];
              }
            });
          },
          files.length > 1 // 다중 파일 업로드 여부
        );
      } catch (error) {
        console.error(`파일 업로드 실패: ${file.name}`, error);
        setRagFiles(prev => {
          const index = prev.findIndex(f => f.name === file.name);
          if (index !== -1) {
            const updatedFiles = [...prev];
            updatedFiles[index] = {
              ...updatedFiles[index],
              status: 'error',
              error: error instanceof Error ? error.message : '업로드 실패'
            };
            return updatedFiles;
          }
          return prev;
        });
      }
    }
  };

  // 파일 크기 포맷팅
  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };

  // @ 멘션 처리 함수들
  const handleInputChange = (e: React.ChangeEvent<HTMLTextAreaElement>) => {
    const value = e.target.value;
    const cursorPos = e.target.selectionStart;
    
    setInputMessage(value);
    
    // 자동 높이 조절
    const target = e.target as HTMLTextAreaElement;
    target.style.height = 'auto';
    target.style.height = `${Math.min(Math.max(target.scrollHeight, 48), 200)}px`;
    
    // @ 멘션 감지 (이메일 주소 제외)
    const textBeforeCursor = value.substring(0, cursorPos);
    const mentionMatch = textBeforeCursor.match(/(\s|^)@([^\s@]*)$/);
    
    if (mentionMatch) {
      const mentionText = mentionMatch[2]; // 두 번째 캡처 그룹 (@ 뒤의 텍스트)
      // 이메일 주소 패턴 감지 - 이미 도메인 형태가 입력되고 있으면 자동완성 비활성화
      const isEmailPattern = mentionText.includes('.') && mentionText.match(/^[a-zA-Z0-9._%+-]*\.[a-zA-Z]*$/);
      
      if (!isEmailPattern) {
        setMentionQuery(mentionText);
        // @ 위치 계산 시 공백 또는 시작 위치 고려
        const atPosition = textBeforeCursor.lastIndexOf('@');
        setMentionStartPos(atPosition);
        setShowDocSuggestions(true);
        setSelectedSuggestionIndex(0); // 자동완성 표시 시 첫 번째 항목 선택
      } else {
        setShowDocSuggestions(false);
        setMentionQuery('');
        setSelectedSuggestionIndex(0);
      }
    } else {
      setShowDocSuggestions(false);
      setMentionQuery('');
      setSelectedSuggestionIndex(0);
    }
  };

  const selectDocument = (docName: string) => {
    if (!textareaRef.current) return;
    
    const textarea = textareaRef.current;
    const cursorPos = textarea.selectionStart;
    const textBeforeCursor = inputMessage.substring(0, cursorPos);
    const textAfterCursor = inputMessage.substring(cursorPos);
    
    // @ 멘션 부분을 문서명으로 교체
    const beforeMention = textBeforeCursor.substring(0, mentionStartPos);
    const newMessage = `${beforeMention}@${docName} ${textAfterCursor}`;
    
    setInputMessage(newMessage);
    
    // 기존 멘션된 문서를 모두 제거하고 새로 선택된 문서만 설정
    const newDoc: MentionedDocument = {
      name: docName,
      displayName: docName
    };
    
    setMentionedDocs([newDoc]); // 배열을 새 문서 하나로 교체
    
    setShowDocSuggestions(false);
    setMentionQuery('');
    setSelectedSuggestionIndex(0);
    
    // 커서 위치 조정
    setTimeout(() => {
      const newCursorPos = beforeMention.length + docName.length + 2; // @ + docName + space
      textarea.setSelectionRange(newCursorPos, newCursorPos);
      textarea.focus();
    }, 0);
  };

  const removeMentionedDoc = (docName: string) => {
    setMentionedDocs(prev => prev.filter(doc => doc.name !== docName));
    
    // 입력창에서도 해당 멘션 제거
    const mentionPattern = new RegExp(`@${docName}\\s*`, 'g');
    setInputMessage(prev => prev.replace(mentionPattern, ''));
  };

  // 문서 목록 필터링
  const filteredDocsForMention = ragFiles
    .filter(file => file.status === 'completed')
    .filter(file => 
      file.name.toLowerCase().includes(mentionQuery.toLowerCase())
    );

  // 벡터 DB에서 문서의 모든 청크 가져오기
  const getDocumentChunksByName = async (documentName: string) => {
    try {
        // console.log(`문서 ${documentName}의 모든 청크를 가져옵니다.`);
      
        // 임베딩 서비스의 /chunks 엔드포인트를 사용하여 해당 문서의 모든 청크 가져오기
        const { getRagServerUrl } = await import('@/config/serverConfig');
        const ragServerUrl = getRagServerUrl();
        const response = await fetch(`${ragServerUrl}/chunks?filename=${encodeURIComponent(documentName)}`, {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json'
        }
      });

      if (!response.ok) {
        throw new Error(`임베딩 서비스 요청 실패: ${response.status}`);
      }

      const result = await response.json();
      
      if (result.success && result.chunks) {
        // 청크를 표준 형식으로 변환
        const chunks = result.chunks.map((chunk: any) => ({
          content: chunk.content,
          metadata: chunk.metadata || {},
          page_number: chunk.metadata?.page_number || chunk.index + 1,
          chunk_index: chunk.index || 0
        }));
        
        // console.log(`✅ 문서 ${documentName}에서 ${chunks.length}개 청크를 가져왔습니다.`);
        return chunks;
      } else {
        console.warn(`문서 ${documentName}에서 청크를 찾을 수 없습니다.`);
        return [];
      }
    } catch (error) {
      console.error(`문서 ${documentName} 청크 가져오기 실패:`, error);
      // 폴백: 기존 RAG 검색 방식 사용
      try {
        // console.log(`폴백: RAG 검색으로 ${documentName} 청크를 검색합니다.`);
        
        const { getApiServerUrl } = await import('@/config/serverConfig');
        const apiServerUrl = getApiServerUrl();
        const fallbackResponse = await fetch(`${apiServerUrl}/api/v1/rag/search`, {
          method: 'POST',
          headers: {
            'Content-Type': 'application/json',
            'Authorization': `Bearer ${localStorage.getItem('apiKey') || 'airun_1_e8f7a00fe827fdf1a9364311d11242c5'}`
          },
          body: JSON.stringify({
            query: documentName,
            userId: user?.id || 1,
            limit: 50
          })
        });

        if (fallbackResponse.ok) {
          const fallbackResult = await fallbackResponse.json();
          if (fallbackResult.success && fallbackResult.data?.documents) {
            const chunks = fallbackResult.data.documents.map((doc: any, index: number) => ({
              content: doc,
              metadata: fallbackResult.data.metadatas?.[index] || {},
              page_number: fallbackResult.data.metadatas?.[index]?.page_number || index + 1,
              chunk_index: index
            }));
            // console.log(`✅ 폴백으로 문서 ${documentName}에서 ${chunks.length}개 청크를 찾았습니다.`);
            return chunks;
          }
        }
      } catch (fallbackError) {
        console.error(`폴백 검색도 실패:`, fallbackError);
      }
      
      return [];
    }
  };

  // 토큰 수 추정
  const estimateTokens = (text: string) => {
    const englishChars = (text.match(/[a-zA-Z\s]/g) || []).length;
    const koreanChars = (text.match(/[가-힣]/g) || []).length;
    const otherChars = text.length - englishChars - koreanChars;
    
    return Math.ceil(englishChars / 4 + koreanChars / 2 + otherChars / 3);
  };

  // 토큰 제한으로 청크 선별
  const limitChunksByTokens = (chunks: any[], maxTokens: number) => {
    let totalTokens = 0;
    const limitedChunks: any[] = [];
    
    for (const chunk of chunks) {
      const chunkTokens = estimateTokens(chunk.content || chunk.text || '');
      if (totalTokens + chunkTokens <= maxTokens) {
        limitedChunks.push(chunk);
        totalTokens += chunkTokens;
      } else {
        break;
      }
    }
    
    return limitedChunks;
  };

  // 멘션된 문서의 전체 내용을 가져와서 텍스트로 합치기 (페이지 정보 포함)
  const getMentionedDocumentsContent = async (mentionedDocNames: string[]): Promise<string> => {
    let combinedContent = '';
    
    for (const docName of mentionedDocNames) {
      try {
        setProgressMessages([`문서 "${docName}" 내용 가져오는 중...`]);
        
        // 해당 문서의 모든 청크 가져오기
        const allChunks = await getDocumentChunksByName(docName);
        
        if (allChunks.length === 0) {
          console.warn(`문서 ${docName}을 찾을 수 없습니다.`);
          continue;
        }
        
        // 청크들을 페이지별로 그룹화
        const chunksByPage = new Map<number, any[]>();
        allChunks.forEach((chunk: any) => {
          const pageNum = chunk.metadata?.page_number || chunk.page_number || 1;
          if (!chunksByPage.has(pageNum)) {
            chunksByPage.set(pageNum, []);
          }
          chunksByPage.get(pageNum)!.push(chunk);
        });
        
        // 페이지 순서대로 정렬하여 내용 결합
        const sortedPages = Array.from(chunksByPage.keys()).sort((a, b) => a - b);
        let documentContent = '';
        
        for (const pageNum of sortedPages) {
          const pageChunks = chunksByPage.get(pageNum)!;
          const pageContent = pageChunks
            .map((chunk: any) => chunk.content || chunk.text || '')
            .join('\n\n');
          
          if (pageContent.trim()) {
            documentContent += `\n\n--- 페이지 ${pageNum} ---\n${pageContent}\n`;
          }
        }
        
        // 문서별로 구분해서 추가
        combinedContent += `\n\n=== ${docName} ===${documentContent}\n`;
        
        // console.log(`✅ 문서 ${docName} 내용 추가 완료 (${allChunks.length}개 청크, ${sortedPages.length}개 페이지)`);
        
      } catch (error) {
        console.error(`문서 ${docName} 처리 중 오류:`, error);
      }
    }
    
    return combinedContent.trim();
  };

  // 메시지 액션 함수들
  const copyToClipboard = async (content: string, messageId: string) => {
    try {
      // 최신 Clipboard API 사용
      if (navigator.clipboard && window.isSecureContext) {
        await navigator.clipboard.writeText(content);
        setShowToast({ message: t('chat.copiedToClipboard', '클립보드에 복사되었습니다'), type: 'success' });
      } else {
        // 폴백: 구형 브라우저나 비보안 컨텍스트에서 사용
        const textArea = document.createElement('textarea');
        textArea.value = content;
        textArea.style.position = 'fixed';
        textArea.style.left = '-999999px';
        textArea.style.top = '-999999px';
        document.body.appendChild(textArea);
        textArea.focus();
        textArea.select();
        
        const successful = document.execCommand('copy');
        document.body.removeChild(textArea);
        
        if (successful) {
          setShowToast({ message: t('chat.copiedToClipboard', '클립보드에 복사되었습니다'), type: 'success' });
        } else {
          throw new Error(t('chat.copyFailed', '복사 명령 실행 실패'));
        }
      }
      
      // 복사 완료 상태 표시
      setCopiedMessageId(messageId);
      setTimeout(() => setCopiedMessageId(null), 2000); // 2초 후 원래 아이콘으로 복원
      setTimeout(() => setShowToast(null), 3000);
    } catch (error) {
      console.error('클립보드 복사 실패:', error);
      setShowToast({ message: t('chat.copyFailed', '클립보드 복사에 실패했습니다. 수동으로 복사해주세요.'), type: 'error' });
      setTimeout(() => setShowToast(null), 4000);
    }
  };

  const saveToPDF = async (content: string, messageId: string) => {
    try {
      setPdfProcessingId(messageId);
              setShowToast({ message: t('chat.generatingPdf', 'PDF 생성 중...'), type: 'success' });
      
      // 브라우저의 인쇄 기능을 사용하여 PDF 생성
      const printWindow = window.open('', '_blank');
      if (!printWindow) {
        setPdfProcessingId(null);
        throw new Error(t('chat.popupBlocked', '팝업이 차단되었습니다. 팝업을 허용해주세요.'));
      }

      // 코드 블록 처리를 위한 함수
      const formatContent = (text: string) => {
        return text
          .replace(/```(\w+)?\n([\s\S]*?)```/g, '<div class="code-block"><pre><code>$2</code></pre></div>')
          .replace(/`([^`]+)`/g, '<code class="inline-code">$1</code>')
          .replace(/\*\*(.*?)\*\*/g, '<strong>$1</strong>')
          .replace(/\*(.*?)\*/g, '<em>$1</em>')
          .replace(/\n/g, '<br>');
      };

      const htmlContent = `
        <!DOCTYPE html>
        <html>
        <head>
          <meta charset="utf-8">
          <title>AI 응답 - ${new Date().toLocaleDateString()}</title>
          <style>
            @media print {
              body { margin: 0; }
              .no-print { display: none; }
            }
            body {
              font-family: 'Noto Sans KR', -apple-system, BlinkMacSystemFont, 'Segoe UI', sans-serif;
              line-height: 1.6;
              color: #333;
              max-width: 800px;
              margin: 0 auto;
              padding: 20px;
              background: white;
            }
            .header {
              border-bottom: 2px solid #e5e7eb;
              padding-bottom: 20px;
              margin-bottom: 30px;
            }
            .title {
              font-size: 24px;
              font-weight: bold;
              color: #1f2937;
              margin-bottom: 10px;
            }
            .meta {
              color: #6b7280;
              font-size: 14px;
            }
            .content {
              font-size: 16px;
              line-height: 1.8;
              word-wrap: break-word;
            }
            .code-block {
              background-color: #f8f9fa;
              border: 1px solid #e9ecef;
              border-radius: 8px;
              margin: 16px 0;
              overflow: hidden;
            }
            .code-block pre {
              margin: 0;
              padding: 16px;
              overflow-x: auto;
              font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
              font-size: 14px;
              line-height: 1.4;
            }
            .inline-code {
              background-color: #f3f4f6;
              padding: 2px 6px;
              border-radius: 4px;
              font-family: 'Monaco', 'Menlo', 'Ubuntu Mono', monospace;
              font-size: 14px;
            }
            .footer {
              margin-top: 40px;
              padding-top: 20px;
              border-top: 1px solid #e5e7eb;
              text-align: center;
              color: #6b7280;
              font-size: 12px;
            }
            .print-button {
              position: fixed;
              top: 20px;
              right: 20px;
              background: #3b82f6;
              color: white;
              border: none;
              padding: 10px 20px;
              border-radius: 6px;
              cursor: pointer;
              font-size: 14px;
              z-index: 1000;
            }
            .print-button:hover {
              background: #2563eb;
            }
            @media print {
              .print-button { display: none; }
            }
          </style>
        </head>
        <body>
          <button class="print-button no-print" onclick="window.print()">PDF로 저장</button>
          <div class="header">
            <div class="title">AI 응답</div>
            <div class="meta">
              생성일: ${new Date().toLocaleString('ko-KR')}
              <br>메시지 ID: ${messageId}
            </div>
          </div>
          <div class="content">${formatContent(content)}</div>
          <div class="footer">
                              Generated by AI 워크스페이스 - ${window.location.origin}
          </div>
          <script>
            // 자동으로 인쇄 대화상자 열기
            window.onload = function() {
              setTimeout(() => {
                window.print();
              }, 500);
            };
          </script>
        </body>
        </html>
      `;

      printWindow.document.write(htmlContent);
      printWindow.document.close();
      
      setShowToast({ message: t('chat.printWindowOpened', '인쇄 창이 열렸습니다. PDF로 저장을 선택하세요.'), type: 'success' });
      setTimeout(() => {
        setShowToast(null);
        setPdfProcessingId(null);
      }, 4000);
      
    } catch (error) {
      console.error('PDF 저장 실패:', error);
      setPdfProcessingId(null);
      setShowToast({ message: 'PDF 저장에 실패했습니다', type: 'error' });
      setTimeout(() => setShowToast(null), 3000);
    }
  };

  const handleLike = async (messageId: string) => {
    const isCurrentlyLiked = messageReactions[messageId]?.liked;
    
    // UI 상태 즉시 업데이트
    setMessageReactions(prev => ({
      ...prev,
      [messageId]: {
        ...prev[messageId],
        liked: !isCurrentlyLiked,
        disliked: false // 좋아요 누르면 싫어요 해제
      }
    }));

    // 좋아요를 새로 누른 경우에만 Langfuse에 전송
    if (!isCurrentlyLiked) {
      try {
        // 해당 메시지와 이전 사용자 메시지 찾기
        const messageIndex = currentMessages.findIndex(msg => msg.id === messageId);
        const aiMessage = currentMessages[messageIndex];
        
        // AI 메시지 이전에서 가장 가까운 사용자 메시지 찾기
        let userMessage = null;
        for (let i = messageIndex - 1; i >= 0; i--) {
          if (currentMessages[i] && currentMessages[i].role === 'user') {
            userMessage = currentMessages[i];
            break;
          }
        }

        // console.log('좋아요 클릭 - 디버그 정보:', {
        //   messageId,
        //   currentSessionId,
        //   messageIndex,
        //   aiMessage: aiMessage ? { id: aiMessage.id, role: aiMessage.role, contentLength: aiMessage.content.length } : null,
        //   userMessage: userMessage ? { id: userMessage.id, role: userMessage.role, contentLength: userMessage.content.length } : null,
        //   totalMessages: currentMessages.length
        // });

        if (aiMessage && aiMessage.role === 'assistant' && userMessage && userMessage.role === 'user') {
          const response = await externalApiClient.sendFeedbackToLangfuse({
            messageId: messageId,
            sessionId: currentSessionId || 'unknown',
            rating: 5, // 좋아요는 5점
            userMessage: userMessage.content,
            aiResponse: aiMessage.content,
            timestamp: new Date().toISOString()
          });

          // 학습 결과가 있으면 표시
          if (response.learning) {
            const learningMessage = response.learning.message;
            const suggestionsCount = response.learning.suggestionsGenerated;
            const mode = response.learning.mode;
            const cached = response.learning.cached;
            
            let statusIcon = '';
            let additionalInfo = '';
            
            switch (mode) {
              case 'cloud':
                statusIcon = '☁️';
                additionalInfo = suggestionsCount > 0 ? `(${suggestionsCount}개 패턴 학습됨)` : '';
                break;
              case 'local':
                statusIcon = '📦';
                additionalInfo = `(로컬 모드${cached ? ', 캐시됨' : ''})`;
                break;
              case 'fallback':
                statusIcon = '⚠️';
                additionalInfo = '(제한된 기능)';
                break;
              case 'basic':
                statusIcon = '💾';
                additionalInfo = '(기본 저장)';
                break;
              default:
                statusIcon = '✅';
                additionalInfo = suggestionsCount > 0 ? `(${suggestionsCount}개 패턴 학습됨)` : '';
            }
            
            setShowToast({ 
              message: `${statusIcon} ${learningMessage} ${additionalInfo}`, 
              type: 'success' 
            });
            // console.log('피드백 학습 결과:', response.learning);
          } else {
            setShowToast({ message: t('chat.feedbackSaved', '피드백이 저장되었습니다'), type: 'success' });
          }
          setTimeout(() => setShowToast(null), 4000);
        } else {
          // 메시지를 찾을 수 없는 경우 UI 상태만 업데이트하고 조용히 처리
          console.warn('메시지 쌍을 찾을 수 없어 Langfuse 전송을 건너뜁니다:', { 
            messageId, 
            aiMessage: !!aiMessage, 
            userMessage: !!userMessage,
            userMessageRole: userMessage?.role 
          });
          setShowToast({ message: t('chat.likeMarked', '좋아요가 표시되었습니다'), type: 'success' });
          setTimeout(() => setShowToast(null), 2000);
        }
      } catch (error) {
        console.error('피드백 전송 실패:', error);
        setShowToast({ message: t('chat.feedbackSaveFailed', '피드백 저장에 실패했습니다'), type: 'error' });
        setTimeout(() => setShowToast(null), 3000);
      }
    }
  };

  const handleDislike = async (messageId: string) => {
    const isCurrentlyDisliked = messageReactions[messageId]?.disliked;
    
    // UI 상태 즉시 업데이트
    setMessageReactions(prev => ({
      ...prev,
      [messageId]: {
        ...prev[messageId],
        disliked: !isCurrentlyDisliked,
        liked: false // 싫어요 누르면 좋아요 해제
      }
    }));

    // 싫어요를 새로 누른 경우에만 Langfuse에 전송
    if (!isCurrentlyDisliked) {
      try {
        // 해당 메시지와 이전 사용자 메시지 찾기
        const messageIndex = currentMessages.findIndex(msg => msg.id === messageId);
        const aiMessage = currentMessages[messageIndex];
        
        // AI 메시지 이전에서 가장 가까운 사용자 메시지 찾기
        let userMessage = null;
        for (let i = messageIndex - 1; i >= 0; i--) {
          if (currentMessages[i] && currentMessages[i].role === 'user') {
            userMessage = currentMessages[i];
            break;
          }
        }

        // console.log('싫어요 클릭 - 디버그 정보:', {
        //   messageId,
        //   currentSessionId,
        //   messageIndex,
        //   aiMessage: aiMessage ? { id: aiMessage.id, role: aiMessage.role, contentLength: aiMessage.content.length } : null,
        //   userMessage: userMessage ? { id: userMessage.id, role: userMessage.role, contentLength: userMessage.content.length } : null,
        //   totalMessages: currentMessages.length
        // });

        if (aiMessage && aiMessage.role === 'assistant' && userMessage && userMessage.role === 'user') {
          const response = await externalApiClient.sendFeedbackToLangfuse({
            messageId: messageId,
            sessionId: currentSessionId || 'unknown',
            rating: 1, // 싫어요는 1점
            userMessage: userMessage.content,
            aiResponse: aiMessage.content,
            timestamp: new Date().toISOString()
          });

          // 학습 결과가 있으면 표시
          if (response.learning) {
            const learningMessage = response.learning.message;
            const suggestionsCount = response.learning.suggestionsGenerated;
            const mode = response.learning.mode;
            const cached = response.learning.cached;
            
            let statusIcon = '';
            let additionalInfo = '';
            
            switch (mode) {
              case 'cloud':
                statusIcon = '☁️';
                additionalInfo = suggestionsCount > 0 ? `(${suggestionsCount}개 개선점 분석됨)` : '';
                break;
              case 'local':
                statusIcon = '📦';
                additionalInfo = `(로컬 모드${cached ? ', 캐시됨' : ''})`;
                break;
              case 'fallback':
                statusIcon = '⚠️';
                additionalInfo = '(제한된 기능)';
                break;
              case 'basic':
                statusIcon = '💾';
                additionalInfo = '(기본 저장)';
                break;
              default:
                statusIcon = '✅';
                additionalInfo = suggestionsCount > 0 ? `(${suggestionsCount}개 개선점 분석됨)` : '';
            }
            
            setShowToast({ 
              message: `${statusIcon} ${learningMessage} ${additionalInfo}`, 
              type: 'success' 
            });
            // console.log('피드백 학습 결과:', response.learning);
          } else {
            setShowToast({ message: t('chat.feedbackSaved', '피드백이 저장되었습니다'), type: 'success' });
          }
          setTimeout(() => setShowToast(null), 4000);
        } else {
          // 메시지를 찾을 수 없는 경우 UI 상태만 업데이트하고 조용히 처리
          console.warn('메시지 쌍을 찾을 수 없어 Langfuse 전송을 건너뜁니다:', { 
            messageId, 
            aiMessage: !!aiMessage, 
            userMessage: !!userMessage,
            userMessageRole: userMessage?.role 
          });
          setShowToast({ message: t('chat.dislikeMarked', '싫어요가 표시되었습니다'), type: 'success' });
          setTimeout(() => setShowToast(null), 2000);
        }
      } catch (error) {
        console.error('피드백 전송 실패:', error);
        setShowToast({ message: t('chat.feedbackSaveFailed', '피드백 저장에 실패했습니다'), type: 'error' });
        setTimeout(() => setShowToast(null), 3000);
      }
    }
  };

  // 파일 유형별 아이콘 반환
  const getFileIcon = (fileName: string, status: string) => {
    const extension = fileName.split('.').pop()?.toLowerCase() || '';
    const iconClass = "w-4 h-4 flex-shrink-0";
    
    // 상태별 아이콘 (업로딩/처리 중)
    if (status === 'uploading') {
      return (
        <div className="w-4 h-4 flex-shrink-0">
          <div className="animate-spin rounded-full h-4 w-4 border-2 border-blue-500 border-t-transparent"></div>
        </div>
      );
    }
    
    if (status === 'processing') {
      return (
        <div className="w-4 h-4 flex-shrink-0">
          <div className="animate-pulse rounded-full h-4 w-4 bg-yellow-500"></div>
        </div>
      );
    }
    
    if (status === 'error') {
      return (
        <svg className={`${iconClass} text-red-500`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
        </svg>
      );
    }
    
    // 파일 유형별 아이콘
    switch (extension) {
      case 'pdf':
        return (
          <svg className={`${iconClass} text-red-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M8.267 14.68c-.184 0-.308.018-.372.036v1.178c.076.018.171.023.302.023.479 0 .774-.242.774-.651 0-.366-.254-.586-.704-.586zm3.487.012c-.2 0-.33.018-.407.036v2.61c.077.018.201.018.313.018.817.006 1.349-.444 1.349-1.396.006-.83-.479-1.268-1.255-1.268z"/>
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM9.498 16.19c-.309.29-.765.42-1.296.42a2.23 2.23 0 0 1-.308-.018v1.426H7v-3.936A7.558 7.558 0 0 1 8.219 14c.557 0 .953.106 1.22.319.254.202.426.533.426.923-.001.392-.131.723-.367.948zm3.807 1.355c-.42.349-1.059.515-1.84.515-.468 0-.799-.03-1.024-.06v-3.917A7.947 7.947 0 0 1 11.66 14c.757 0 1.249.136 1.633.426.415.308.675.799.675 1.504 0 .763-.279 1.29-.663 1.615zM17 14.77h-1.532v.911H16.9v.734h-1.432v1.604h-.906V14.03H17v.74z"/>
          </svg>
        );
      case 'doc':
      case 'docx':
        return (
          <svg className={`${iconClass} text-blue-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM16 18H8v-2h8v2zm0-4H8v-2h8v2zm-3-5V3.5L18.5 9H13z"/>
          </svg>
        );
      case 'xls':
      case 'xlsx':
        return (
          <svg className={`${iconClass} text-green-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM12 18H8v-2h4v2zm4-4H8v-2h8v2zm0-4H8V8h8v2zm-3-5V3.5L18.5 9H13z"/>
          </svg>
        );
      case 'ppt':
      case 'pptx':
        return (
          <svg className={`${iconClass} text-orange-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM11 15H9v3H7v-7h4c1.1 0 2 .9 2 2s-.9 2-2 2zm2-6V3.5L18.5 9H13z"/>
            <path d="M9 11v2h2c.6 0 1-.4 1-1s-.4-1-1-1H9z"/>
          </svg>
        );
      case 'txt':
      case 'md':
        return (
          <svg className={`${iconClass} text-gray-600`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
          </svg>
        );
      case 'jpg':
      case 'jpeg':
      case 'png':
      case 'gif':
      case 'bmp':
      case 'svg':
        return (
          <svg className={`${iconClass} text-purple-600`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
          </svg>
        );
      case 'zip':
      case 'rar':
      case '7z':
      case 'tar':
      case 'gz':
        return (
          <svg className={`${iconClass} text-yellow-600`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10l-8 4" />
          </svg>
        );
      case 'mp4':
      case 'avi':
      case 'mov':
      case 'wmv':
      case 'flv':
        return (
          <svg className={`${iconClass} text-red-500`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z" />
          </svg>
        );
      case 'mp3':
      case 'wav':
      case 'flac':
      case 'aac':
        return (
          <svg className={`${iconClass} text-green-500`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19V6l12-3v13M9 19c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zm12-3c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zM9 10l12-3" />
          </svg>
        );
      default:
        return (
          <svg className={`${iconClass} text-gray-400`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
          </svg>
        );
    }
  };

  // 다운로드 링크를 감지하고 렌더링하는 함수
  const renderDownloadLinks = (content: string) => {
    // 다운로드 링크 패턴 감지 (📄 문서 생성 완료: 제목.확장자 (크기KB))
    const downloadLinkPattern = /📄 문서 생성 완료: (.+?)\.([a-zA-Z]+) \((\d+)KB\)/g;
    const matches = Array.from(content.matchAll(downloadLinkPattern));
    
    if (matches.length === 0) {
      return null;
    }

    return (
      <div className="mt-3 space-y-2">
        <div className="text-sm font-medium text-gray-700 dark:text-gray-300">
          📄 생성된 문서:
        </div>
        {matches.map((match, index) => {
          const [fullMatch, title, extension, size] = match;
          // 실제 생성된 파일명 추정 (타임스탬프 포함)
          const timestamp = new Date().toISOString().slice(0, 19).replace(/[-:]/g, '').replace('T', '_');
          const safeTitle = title.replace(/[^a-zA-Z0-9가-힣\s\-_]/g, '').trim().slice(0, 50);
          const filename = `document_${safeTitle}_${timestamp}.${extension}`;
          const downloadUrl = `/api/v1/download/${encodeURIComponent(filename)}`;
          
          return (
            <div key={index} className="flex items-center space-x-3 p-3 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-700">
              <div className="flex-shrink-0">
                {getFileIcon(`file.${extension}`, 'completed')}
              </div>
              <div className="flex-1 min-w-0">
                <div className="text-sm font-medium text-gray-900 dark:text-gray-100 truncate">
                  {title}.{extension}
                </div>
                <div className="text-xs text-gray-500 dark:text-gray-400">
                  {size}KB • {extension.toUpperCase()} 문서
                </div>
              </div>
              <div className="flex-shrink-0">
                <a
                  href={downloadUrl}
                  download
                  className="inline-flex items-center px-3 py-1.5 text-xs font-medium text-white bg-blue-600 hover:bg-blue-700 rounded-md transition-colors duration-200"
                  title="문서 다운로드"
                >
                  <svg className="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                  다운로드
                </a>
              </div>
            </div>
          );
        })}
      </div>
    );
  };

  //(ryan) - 코드 실행 관련 함수들
  
  // 코드 블록 추출 함수
  const extractCodeFromMessage = (content: string): string | null => {
    // Python 코드 블록 패턴 매칭 (더 유연한 패턴)
    const pythonCodePattern = /```(?:python|py)?\s*\n([\s\S]*?)\n```/i;
    const match = content.match(pythonCodePattern);
    if (match && match[1]) {
      const code = match[1].trim();
      // 빈 코드나 너무 짧은 코드는 제외
      return code.length > 5 ? code : null;
    }
    return null;
  };

  // 코드 실행 함수
  const executeCode = async (messageId: string, prompt: string, code: string) => {
    try {
      
      setExecutingCodeMessageId(messageId);
      setExecutingCodes(prev => new Set(prev).add(messageId));
      setProgressMessages(['🔧 코드 실행 중...']);

      const result = await externalApiClient.codeExecuteAndWait(
        code,
        {
          autoInstallPackages: true,
          useReview: true,
          sessionId: currentSessionId || undefined
        },
        (status: string) => {
          // 기술적 상태 메시지를 사용자 친화적 메시지로 변환
          const convertStatus = (msg: string): string => {
            const lowerMsg = msg.toLowerCase();
            if (lowerMsg.includes('active')) return '잠시만 기다려주세요...';
            if (lowerMsg.includes('pending')) return '요청을 처리하고 있습니다...';
            if (lowerMsg.includes('processing')) return '처리 중입니다...';
            return msg;
          };
          setProgressMessages([convertStatus(status)]);
        }
      );

      setProgressMessages([]);

      // 디버깅을 위한 로그 추가
      // console.log('코드 실행 API 응답:', result);

      // API 클라이언트 응답 구조에 맞게 처리
      if (result.isSuccess) {
        // 성공 처리 - reviewFlags에 값을 넣지 않음 (성공 시 오류 표시 안 함)
        setSuccessFlags(prev => ({ ...prev, [messageId]: 'yes' }));
        
        // 성공한 실행 결과 저장
        setExecutionResults(prev => ({ ...prev, [messageId]: result.response || '✅ 실행 완료' }));
        
        // 기존 reviewFlags에서 해당 메시지 제거 (성공 시 오류 메시지 삭제)
        setReviewFlags(prev => {
          const newFlags = { ...prev };
          delete newFlags[messageId];
          return newFlags;
        });
        
        setUserPrompt(prev => ({ ...prev, [messageId]: prompt }));
        setUserCodeExecution(prev => ({ ...prev, [messageId]: code }));
        setPromptCodeMap(prev => ({ ...prev, [messageId]: { prompt, code } }));
        
        setShowToast({ message: `✅ 코드가 성공적으로 실행되었습니다!`, type: 'success' });
      } else {
        // 실패 처리 - reviewFlags에 오류 메시지 설정
        setSuccessFlags(prev => ({ ...prev, [messageId]: 'no' }));
        setReviewFlags(prev => ({ 
          ...prev, 
          [messageId]: result.reviewFlag || result.response || '알 수 없는 오류가 발생했습니다.' 
        }));
        setUserPrompt(prev => ({ ...prev, [messageId]: prompt }));
        setUserCodeExecution(prev => ({ ...prev, [messageId]: code }));
        setPromptCodeMap(prev => ({ ...prev, [messageId]: { prompt, code } }));
        
        setShowToast({ message: '⚠️ 코드 실행 중 오류가 발생했습니다.', type: 'error' });
      }

      setTimeout(() => setShowToast(null), 3000);
    } catch (error) {
      console.error('코드 실행 실패:', error);
      
      // 네트워크 오류 등의 예외 처리
      setSuccessFlags(prev => ({ ...prev, [messageId]: 'no' }));
      setReviewFlags(prev => ({ 
        ...prev, 
        [messageId]: `실행 중 오류 발생: ${error instanceof Error ? error.message : '네트워크 오류'}` 
      }));
      
      setShowToast({ message: '❌ 코드 실행에 실패했습니다.', type: 'error' });
      setTimeout(() => setShowToast(null), 3000);
    } finally {
      setExecutingCodeMessageId(null);
      setExecutingCodes(prev => {
        const newSet = new Set(prev);
        newSet.delete(messageId);
        return newSet;
      });
      setProgressMessages([]);
    }
  };

  // 코드 재생성 함수
  const regenerateCode = async (messageId: string) => {
    try {
      const promptData = promptCodeMap[messageId];
      if (!promptData) {
        setShowToast({ message: '⚠️ 원본 프롬프트를 찾을 수 없습니다.', type: 'error' });
        setTimeout(() => setShowToast(null), 3000);
        return;
      }

      setIsLoading(true);
      setProgressMessages(['🔄 코드를 재생성하는 중...']);

      // 기존 메시지 제거하고 새 메시지 추가
      const userMessage: Message = {
        id: `user-${Date.now()}`,
        role: 'user',
        content: `${promptData.prompt}\n\n코드를 다시 작성해주세요.`,
        timestamp: new Date().toISOString()
      };

      setCurrentMessages(prev => [...prev, userMessage]);

      // 새 AI 응답 생성
      const response = await externalApiClient.codeAsyncAndWait(
        userMessage.content,
        currentSessionId || undefined,
        {
          provider: options.provider,
          model: options.model,
          temperature: options.temperature,
          maxTokens: options.maxTokens,
          enableRag: options.enableRag,
          enableWebSearch: options.enableWebSearch,
          enableSmartResearch: options.enableSmartResearch,
          userId: options.userId,
          history: []
        },
        (status: string) => {
          // 기술적 상태 메시지를 사용자 친화적 메시지로 변환
          const convertStatus = (msg: string): string => {
            const lowerMsg = msg.toLowerCase();
            if (lowerMsg.includes('active')) return '잠시만 기다려주세요...';
            if (lowerMsg.includes('pending')) return '요청을 처리하고 있습니다...';
            if (lowerMsg.includes('processing')) return '처리 중입니다...';
            return msg;
          };
          setProgressMessages([convertStatus(status)]);
        }
      );

      if (response?.response) {
        const assistantMessage: Message = {
          id: `assistant-${Date.now()}`,
          role: 'assistant',
          content: response.response,
          timestamp: new Date().toISOString()
        };

        setCurrentMessages(prev => [...prev, assistantMessage]);

        // 기존 플래그들 초기화
        setReviewFlags(prev => {
          const newFlags = { ...prev };
          delete newFlags[messageId];
          return newFlags;
        });
        setSuccessFlags(prev => {
          const newFlags = { ...prev };
          delete newFlags[messageId];
          return newFlags;
        });

        setShowToast({ message: '✅ 코드가 재생성되었습니다!', type: 'success' });
      }

      setTimeout(() => setShowToast(null), 3000);
    } catch (error) {
      console.error('코드 재생성 실패:', error);
      setShowToast({ message: '❌ 코드 재생성에 실패했습니다.', type: 'error' });
      setTimeout(() => setShowToast(null), 3000);
    } finally {
      setIsLoading(false);
      setProgressMessages([]);
    }
  };

  // 코드 다운로드 함수
  const downloadCode = async (messageId: string) => {
    try {
      const promptData = promptCodeMap[messageId];
      if (!promptData) {
        setShowToast({ message: '⚠️ 다운로드할 코드를 찾을 수 없습니다.', type: 'error' });
        setTimeout(() => setShowToast(null), 3000);
        return;
      }

      setDownloadFlags(prev => ({ ...prev, [messageId]: 'downloading' }));
      setProgressMessages(['📦 코드를 압축하는 중...']);

      const result = await externalApiClient.executeCodeSaveZip(
        { continueMode: "2" },
        promptData.code,
        (status: string) => {
          // 기술적 상태 메시지를 사용자 친화적 메시지로 변환
          const convertStatus = (msg: string): string => {
            const lowerMsg = msg.toLowerCase();
            if (lowerMsg.includes('active')) return '잠시만 기다려주세요...';
            if (lowerMsg.includes('pending')) return '요청을 처리하고 있습니다...';
            if (lowerMsg.includes('processing')) return '처리 중입니다...';
            return msg;
          };
          setProgressMessages([convertStatus(status)]);
        }
      );

      if (result.isSuccess && result.downloadUrl) {
        await externalApiClient.downloadCodeSaveZip(result.downloadUrl);
        setDownloadFlags(prev => ({ ...prev, [messageId]: 'completed' }));
        setShowToast({ message: '✅ 코드가 다운로드되었습니다!', type: 'success' });
      } else {
        setDownloadFlags(prev => ({ ...prev, [messageId]: 'error' }));
        setShowToast({ message: '❌ 코드 다운로드에 실패했습니다.', type: 'error' });
      }

      setTimeout(() => setShowToast(null), 3000);
    } catch (error) {
      console.error('코드 다운로드 실패:', error);
      setDownloadFlags(prev => ({ ...prev, [messageId]: 'error' }));
      setShowToast({ message: '❌ 코드 다운로드에 실패했습니다.', type: 'error' });
      setTimeout(() => setShowToast(null), 3000);
    } finally {
      setProgressMessages([]);
    }
  };

  // Task 저장 모달 열기
  const openTaskSaveModal = (messageId: string) => {
    setTaskSaveTargetMessageId(messageId);
    setShowTaskSaveModal(true);
    setTaskName("");
  };

  // Task 저장 취소
  const cancelTaskSave = () => {
    setShowTaskSaveModal(false);
    setTaskSaveTargetMessageId(null);
    setTaskName("");
  };

  // Task 저장 실행
  const onSaveTask = async (taskNameInput: string, messageId: string) => {
    try {
      if (!taskNameInput.trim()) {
        setShowToast({ message: '⚠️ 작업 이름을 입력해주세요.', type: 'error' });
        setTimeout(() => setShowToast(null), 3000);
        return;
      }

      const promptData = promptCodeMap[messageId];
      if (!promptData) {
        setShowToast({ message: '⚠️ 저장할 코드를 찾을 수 없습니다.', type: 'error' });
        setTimeout(() => setShowToast(null), 3000);
        return;
      }

      setProgressMessages(['💾 작업을 저장하는 중...']);

      const result = await externalApiClient.executeCodeTaskSave(
        {
          prompt: promptData.prompt,
          pythonCode: promptData.code,
          taskName: taskNameInput.trim(),
          userName: user?.username || 'Unknown'
        },
        (status: string) => {
          // 기술적 상태 메시지를 사용자 친화적 메시지로 변환
          const convertStatus = (msg: string): string => {
            const lowerMsg = msg.toLowerCase();
            if (lowerMsg.includes('active')) return '잠시만 기다려주세요...';
            if (lowerMsg.includes('pending')) return '요청을 처리하고 있습니다...';
            if (lowerMsg.includes('processing')) return '처리 중입니다...';
            return msg;
          };
          setProgressMessages([convertStatus(status)]);
        }
      );

      if (result.isSuccess) {
        setShowToast({ message: '✅ 작업이 성공적으로 저장되었습니다!', type: 'success' });
        setShowTaskSaveModal(false);
        setTaskSaveTargetMessageId(null);
        setTaskName("");
      } else {
        setShowToast({ message: '❌ 작업 저장에 실패했습니다.', type: 'error' });
      }

      setTimeout(() => setShowToast(null), 3000);
    } catch (error) {
      console.error('작업 저장 실패:', error);
      setShowToast({ message: '❌ 작업 저장에 실패했습니다.', type: 'error' });
      setTimeout(() => setShowToast(null), 3000);
    } finally {
      setProgressMessages([]);
    }
  };

  // 메시지에서 사용자 프롬프트 찾기
  const findUserPromptForMessage = (messageId: string): string => {
    const messageIndex = currentMessages.findIndex(msg => msg.id === messageId);
    if (messageIndex === -1) return '';
    
    // AI 메시지 이전에서 가장 가까운 사용자 메시지 찾기
    for (let i = messageIndex - 1; i >= 0; i--) {
      if (currentMessages[i] && currentMessages[i].role === 'user') {
        // originalContent가 있으면 그것을 우선 사용 (멘션된 문서가 포함되기 전 원본 질문)
        return currentMessages[i].originalContent || currentMessages[i].content;
      }
    }
    return '';
  };

  // CodeRenderer용 래퍼 함수들
  const handleExecuteCode = async (messageId: string, code: string, language: string) => {
    const prompt = findUserPromptForMessage(messageId);
    await executeCode(messageId, prompt, code);
  };

  const handleDownloadCode = async (messageId: string, code: string, language: string) => {
    await downloadCode(messageId);
  };

  const handleRegenerateCode = async (messageId: string) => {
    await regenerateCode(messageId);
  };
  //(ryan) - end

  // 인증 초기화 중인 경우
  if (!isInitialized) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <div 
            className="animate-spin rounded-full h-8 w-8 mx-auto mb-4"
            style={{ 
              border: '2px solid #e5e7eb', 
              borderBottomColor: '#ea580c' 
            }}
          ></div>
          <p className="text-gray-600 dark:text-gray-400">{t('chat.loading', '로딩 중...')}</p>
        </div>
      </div>
    );
  }

  // 인증되지 않은 경우 (리다이렉트 처리됨)
  if (!isAuthenticated) {
    return null;
  }

  return (
    <ClientOnly fallback={
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <div 
            className="animate-spin rounded-full h-8 w-8 mx-auto mb-4"
            style={{ 
              border: '2px solid #e5e7eb', 
              borderBottomColor: '#ea580c' 
            }}
          ></div>
          <p className="text-gray-600 dark:text-gray-400">{t('chat.loading', '로딩 중...')}</p>
        </div>
      </div>
    }>
      <div className="h-full flex" style={{ backgroundColor: 'var(--body-bg)' }}>
      {/* 사이드바 */}
              <div className={`${showSidebar ? 'w-80' : 'w-0'} h-full transition-all duration-300 border-r border-gray-200 dark:border-gray-700 flex flex-col overflow-hidden`} style={{ backgroundColor: 'var(--sidebar-bg)' }}>
        {showSidebar && (
          <>
            {/* 폴딩 메뉴 - 최상단 */}
            <div className="p-3 border-b border-gray-200 dark:border-gray-700">
              <div className="flex items-center justify-start">
                <button
                  onClick={() => setShowSidebar(false)}
                  className="p-1.5 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md"
                  title={t('chat.closeSidebar', '사이드바 닫기')}
                >
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                  </svg>
                </button>
              </div>
            </div>

            {/* 컨테이너 1: 내 문서 */}
            <div className="border-b border-gray-200 dark:border-gray-700">
              <div className="p-3">
                <div className="flex items-center justify-between mb-2">
                  <button
                    onClick={() => setShowRagBrowser(!showRagBrowser)}
                    className="flex items-center space-x-2 text-left hover:bg-gray-100 dark:hover:bg-gray-700 rounded p-1 transition-colors"
                  >
                    <svg className="w-4 h-4 text-gray-600 dark:text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                    </svg>
                    <span className="text-base font-medium text-gray-900 dark:text-white">{t('chat.myDocuments', '내 문서')}</span>
                  </button>
                  <button
                    onClick={() => setShowRagBrowser(!showRagBrowser)}
                    className="p-1 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded transition-colors"
                  >
                    <svg className={`w-4 h-4 transition-transform ${showRagBrowser ? 'rotate-180' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                    </svg>
                  </button>
                </div>

                {showRagBrowser && (
                  <div className="space-y-3">
                    {/* 드래그 앤 드롭 영역 */}
                    <div
                      data-drag-over={isDragOverRag}
                      onDragEnter={handleRagDragEnter}
                      onDragLeave={handleRagDragLeave}
                      onDragOver={handleRagDragOver}
                      onDrop={handleRagDrop}
                      onClick={() => ragFileInputRef.current?.click()}
                      className="drag-drop-zone rounded-lg p-4 text-center transition-colors cursor-pointer"
                    >
                      <svg className="w-8 h-8 mx-auto mb-2 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12" />
                      </svg>
                      <p className="text-sm text-gray-600 dark:text-gray-400 mb-1">
                        {isDragOverRag ? t('chat.dropFilesHere', 'Drop files here') : t('chat.upload', 'Upload')}
                      </p>
                    </div>
                    
                    {/* 백그라운드 작업 상태 */}
                    <div className="px-2">
                      <BackgroundJobStatus 
                        userId={user?.username} 
                        onJobComplete={handleJobComplete}
                        refreshTrigger={refreshTrigger}
                      />
                    </div>
                    
                    {/* 숨겨진 파일 입력 */}
                    <input
                      ref={ragFileInputRef}
                      type="file"
                      multiple
                      accept=".pdf,.doc,.docx,.txt,.md,.hwp,.ppt,.pptx,.xls,.xlsx"
                      onChange={handleRagFileSelect}
                      className="hidden"
                    />

                    {/* 파일 목록 */}
                    <div className="max-h-60 min-h-40 overflow-y-auto text-sm">
                      {isLoadingRagFiles ? (
                        <div className="flex justify-center items-center h-20">
                          <div className="relative">
                            <div 
                              className="animate-spin rounded-full h-4 w-4 dark:hidden"
                              style={{ 
                                border: '2px solid #e5e7eb', 
                                borderBottomColor: '#ea580c' 
                              }}
                            ></div>
                            <div 
                              className="animate-spin rounded-full h-4 w-4 hidden dark:block"
                              style={{ 
                                border: '2px solid #4b5563', 
                                borderBottomColor: '#fb923c' 
                              }}
                            ></div>
                          </div>
                        </div>
                      ) : ragFiles.length === 0 ? (
                        <div className="text-center text-xs text-gray-500 dark:text-gray-400 py-4">
                          {t('chat.noDocuments', '업로드된 문서가 없습니다')}
                        </div>
                      ) : (
                        <div className="space-y-1">
                          {ragFiles.map((file, index) => (
                            <div
                              key={`${file.name}-${index}`}
                              className="group flex flex-col p-0.5 rounded-md hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors"
                            >
                              <div className="flex items-center space-x-2">
                                {/* 파일 유형별 아이콘 */}
                                {getFileIcon(file.name, file.status)}
                                
                                <div className="flex-1 min-w-0">
                                  <p 
                                    className="file-browser-filename text-gray-700 dark:text-gray-300 truncate"
                                    title={file.name}
                                  >
                                    {file.name}
                                  </p>
                                  {/* 상태 텍스트 */}
                                  {(file.status === 'uploading' || file.status === 'processing' || file.status === 'error') && (
                                    <div className="flex items-center space-x-2 mt-1">
                                      {file.status === 'uploading' && (
                                        <span className="file-browser-status text-blue-500 text-xs">{t('chat.uploading', '업로드 중...')}</span>
                                      )}
                                      {file.status === 'processing' && (
                                        <span className="file-browser-status text-yellow-500 text-xs">{t('chat.processing', '처리 중...')}</span>
                                      )}
                                      {file.status === 'error' && (
                                        <span className="file-browser-status text-red-500 text-xs">{file.error || t('chat.error', '오류')}</span>
                                      )}
                                    </div>
                                  )}
                                </div>
                                
                                {/* 삭제 버튼 (완료된 파일만) */}
                                {file.status === 'completed' && (
                                  <button
                                    onClick={() => deleteRagFile(file.name)}
                                    className="opacity-0 group-hover:opacity-100 p-1 text-gray-400 hover:text-red-500 dark:hover:text-red-400 transition-opacity flex-shrink-0"
                                    title={t('chat.deleteFile', '파일 삭제')}
                                  >
                                    <svg className="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                    </svg>
                                  </button>
                                )}
                              </div>
                              
                              {/* 프로그레스 바 (업로딩/처리 중일 때만) */}
                              {(file.status === 'uploading' || file.status === 'processing') && (
                                <div className="mt-2">
                                  <div className="w-full bg-gray-200 dark:bg-gray-600 rounded-full h-1.5">
                                    <div 
                                      className={`h-1.5 rounded-full transition-all duration-300 ${
                                        file.status === 'uploading' ? 'bg-blue-500' : 'bg-yellow-500'
                                      }`}
                                      style={{ width: `${file.progress || 0}%` }}
                                    ></div>
                                  </div>
                                </div>
                              )}
                            </div>
                          ))}
                        </div>
                      )}
                    </div>
                  </div>
                )}
              </div>
            </div>

            {/* 새 대화 버튼 */}
            <div className="p-3 border-b border-gray-200 dark:border-gray-700">
              <button
                onClick={handleNewChatClick}
                className="w-full px-3 py-2 bg-blue-600 hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600 text-white text-sm font-medium rounded-lg transition-colors flex items-center justify-center"
              >
                <span>{t('chat.newCode', 'New Code')}</span>
              </button>
            </div>

            {/* 컨테이너 2: 대화목록 */}
            <div className="flex flex-col border-b border-gray-200 dark:border-gray-700 flex-1 min-h-0">
              <div className="p-3 border-b border-gray-200 dark:border-gray-700 flex-shrink-0">
                <div className="flex items-center justify-between mb-2">
                  <div className="flex items-center space-x-2">
                    <svg className="w-4 h-4 text-gray-600 dark:text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 20l4-16m4 4l4 4-4 4M6 16l-4-4 4-4" />
                    </svg>
                    <span className="text-base font-medium text-gray-900 dark:text-white">{t('chat.codeSessions', '코드목록')}</span>
                  </div>
                  {sessions.length > 0 && (
                    <button
                      onClick={handleDeleteAllClick}
                      className="p-1 text-gray-400 hover:text-red-500 dark:hover:text-red-400 transition-colors"
                      title={t('chat.deleteAllCodeSessions', '모든 코드 삭제')}
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                      </svg>
                    </button>
                  )}
                </div>
                
                {/* 검색 */}
                <input
                  type="text"
                  placeholder="코드 검색..."
                  value={searchTerm}
                  onChange={(e) => setSearchTerm(e.target.value)}
                  className="w-full px-2 py-1 border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none focus:ring-1 focus:ring-blue-500"
                  style={{ fontSize: '11px' }}
                />
              </div>

              {/* 세션 목록 */}
              <div className="flex-1 overflow-y-auto min-h-0">
                {isLoadingSessions ? (
                  <div className="flex justify-center items-center h-32">
                    <div className="relative">
                      <div 
                        className="animate-spin rounded-full h-6 w-6 dark:hidden"
                        style={{ 
                          border: '2px solid #e5e7eb', 
                          borderBottomColor: '#ea580c' 
                        }}
                      ></div>
                      <div 
                        className="animate-spin rounded-full h-6 w-6 hidden dark:block"
                        style={{ 
                          border: '2px solid #4b5563', 
                          borderBottomColor: '#fb923c' 
                        }}
                      ></div>
                    </div>
                  </div>
                ) : filteredSessions.length === 0 ? (
                  <div className="p-3 text-center text-xs text-gray-500 dark:text-gray-400">
                    {searchTerm ? t('chat.noResults', '검색 결과가 없습니다') : t('chat.noCodeSessions', '코드가 없습니다')}
                  </div>
                ) : (
                  <div className="space-y-0.5 p-1">
                    {filteredSessions.map((session) => (
                      <div
                        key={session.id}
                        onClick={async () => await selectSession(session.id)}
                        className={`group flex items-center justify-between px-2 py-2 rounded-md cursor-pointer transition-colors ${
                          session.id === currentSessionId
                            ? 'bg-gray-200 dark:bg-gray-700'
                            : 'hover:bg-gray-100 dark:hover:bg-gray-800'
                        }`}
                      >
                        <div className="flex-1 min-w-0 mr-2">
                          <p className="text-gray-700 dark:text-gray-300 truncate chat-sidebar-list-text">
                            {session.title || session.lastMessage}
                          </p>
                        </div>
                        <button
                          onClick={(e) => {
                            e.stopPropagation();
                            handleDeleteClick(session.id);
                          }}
                          className="opacity-0 group-hover:opacity-100 p-1 text-gray-400 hover:text-red-500 dark:hover:text-red-400 transition-opacity flex-shrink-0"
                        >
                          <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                          </svg>
                        </button>
                      </div>
                    ))}
                  </div>
                )}
              </div>
            </div>

            {/* 컨테이너 3: Settings */}
            <div className="relative p-3 space-y-3 flex-shrink-0">
              <div className="flex items-center justify-between">
                <button
                  onClick={() => setShowOptions(!showOptions)}
                  className="flex items-center space-x-2 text-left hover:bg-gray-100 dark:hover:bg-gray-700 rounded p-1 transition-colors"
                >
                  <svg className="w-4 h-4 text-gray-600 dark:text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10.325 4.317c.426-1.756 2.924-1.756 3.35 0a1.724 1.724 0 002.573 1.066c1.543-.94 3.31.826 2.37 2.37a1.724 1.724 0 001.065 2.572c1.756.426 1.756 2.924 0 3.35a1.724 1.724 0 00-1.066 2.573c.94 1.543-.826 3.31-2.37 2.37a1.724 1.724 0 00-2.572 1.065c-.426 1.756-2.924 1.756-3.35 0a1.724 1.724 0 00-2.573-1.066c-1.543.94-3.31-.826-2.37-2.37a1.724 1.724 0 00-1.065-2.572c-1.756-.426-1.756-2.924 0-3.35a1.724 1.724 0 001.066-2.573c-.94-1.543.826-3.31 2.37-2.37.996.608 2.296.07 2.572-1.065z" />
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
                  </svg>
                  <span className="text-sm font-medium text-gray-900 dark:text-white">{t('header.settings', 'Settings')}</span>
                </button>
                <button
                  onClick={() => setShowOptions(!showOptions)}
                  className="p-1 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded transition-colors"
                >
                  <svg className={`w-4 h-4 transition-transform ${showOptions ? 'rotate-180' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                  </svg>
                </button>
              </div>

                                {showOptions && (
                    <div className="absolute bottom-full left-0 right-0 mb-2 border border-gray-200 dark:border-gray-700 rounded-lg shadow-lg p-4 space-y-4 z-10" style={{ backgroundColor: 'var(--card-bg)' }}>
                  {/* 프로바이더 선택 */}
                  <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                      Provider
                    </label>
                    {isLoadingProviders ? (
                      <div className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md bg-gray-100 dark:bg-gray-600 text-gray-500 dark:text-gray-400">
                        Loading...
                      </div>
                    ) : (
                      <select
                        value={options.provider}
                        onChange={(e) => handleProviderChange(e.target.value)}
                        className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                      >
                        {availableProviders.map((provider) => (
                          <option key={provider.key} value={provider.key}>
                            {provider.name}
                            {provider.requiresApiKey && !providers[provider.key]?.apiKeyConfigured && ' (API 키 필요)'}
                          </option>
                        ))}
                      </select>
                    )}
                  </div>

                  {/* 모델 선택 */}
                  <div>
                    <div className="flex items-center justify-between mb-2">
                      <label className="block text-sm font-medium text-gray-700 dark:text-gray-300">
                        Model
                      </label>
                      {(options.provider === 'ollama' || !providers[options.provider]?.models?.length) && (
                        <button
                          onClick={() => handleProviderChange(options.provider)}
                          className="text-xs text-blue-600 dark:text-blue-400 hover:text-blue-800 dark:hover:text-blue-300 flex items-center gap-1"
                          title="모델 목록을 다시 불러옵니다"
                        >
                          <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                          </svg>
                          새로고침
                        </button>
                      )}
                    </div>
                    <select
                      value={options.model}
                      onChange={(e) => {
                        const newModel = e.target.value;
                        // 로컬 스토리지에 선택한 모델 저장
                        localStorage.setItem('airun-selected-model', newModel);
                        setOptions(prev => ({ ...prev, model: newModel }));
                      }}
                      className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md bg-white dark:bg-gray-700 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                      disabled={!providers[options.provider]?.models?.length}
                    >
                      {providers[options.provider]?.models?.length > 0 ? (
                        providers[options.provider].models.map((model: any) => (
                          <option key={model.id} value={model.id}>
                            {model.name}
                            {model.size && ` (${(model.size / 1024 / 1024 / 1024).toFixed(1)}GB)`}
                          </option>
                        ))
                      ) : options.provider === 'ollama' ? (
                        <option value="hamonize:latest">hamonize:latest (기본 모델)</option>
                      ) : (
                        <option value="">
                          {isLoadingProviders ? 'Loading...' : 'No models available'}
                        </option>
                      )}
                    </select>
                    
                    {/* API 키 상태 표시 */}
                    {providers[options.provider]?.requiresApiKey && (
                      <div className="mt-2 p-2 rounded-md bg-gray-50 dark:bg-gray-800">
                        {providers[options.provider]?.apiKeyConfigured ? (
                          <div className="flex items-center text-sm text-green-600 dark:text-green-400">
                            <svg className="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                              <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
                            </svg>
                            API Key Set
                          </div>
                        ) : (
                          <div className="flex items-center text-sm text-yellow-600 dark:text-yellow-400">
                            <svg className="w-4 h-4 mr-2" fill="currentColor" viewBox="0 0 20 20">
                              <path fillRule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clipRule="evenodd" />
                            </svg>
                            API Key Required
                          </div>
                        )}
                      </div>
                    )}
                  </div>

                  {/* Temperature 설정 */}
                  <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                      Temperature: {options.temperature}
                    </label>
                    <input
                      type="range"
                      min="0"
                      max="2"
                      step="0.1"
                      value={options.temperature}
                      onChange={(e) => setOptions(prev => ({ ...prev, temperature: parseFloat(e.target.value) }))}
                      className="w-full h-2 bg-gray-200 dark:bg-gray-600 rounded-lg appearance-none cursor-pointer slider"
                    />
                    <div className="flex justify-between mt-1">
                      <span className="temperature-label text-gray-500 dark:text-gray-400">{t('chat.accurate', '정확')}</span>
                      <span className="temperature-label text-gray-500 dark:text-gray-400">{t('chat.creative', '창의적')}</span>
                    </div>
                  </div>


                </div>
              )}
            </div>
          </>
        )}
      </div>

      {/* 메인 채팅 영역 */}
      <div className="flex-1 flex flex-col" style={{ backgroundColor: 'var(--body-bg)' }}>
        {/* 채팅 헤더 */}
        <div className="border-b border-gray-200 dark:border-gray-700 px-4 py-3" style={{ backgroundColor: 'var(--body-bg)' }}>
          <div className="flex items-center justify-between">
            <div className="flex items-center space-x-3">
              {!showSidebar && (
                <button
                  onClick={() => setShowSidebar(true)}
                  className="p-2 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md"
                  title="사이드바 열기"
                >
                  <svg className="w-5 h-5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 6h16M4 12h16M4 18h16" />
                  </svg>
                </button>
              )}
              
              {/* 현재 모델 정보 표시 */}
              <div className="flex items-center space-x-2 text-sm text-gray-600 dark:text-gray-400">
                <div className="flex items-center space-x-1">
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.75 17L9 20l-1 1h8l-1-1-.75-3M3 13h18M5 17h14a2 2 0 002-2V5a2 2 0 00-2-2H5a2 2 0 00-2 2v10a2 2 0 002 2z" />
                  </svg>
                  <span className="font-medium">{options.provider}</span>
                </div>
                <span className="text-gray-400 dark:text-gray-500">•</span>
                <div className="flex items-center space-x-1">
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19.428 15.428a2 2 0 00-1.022-.547l-2.387-.477a6 6 0 00-3.86.517l-.318.158a6 6 0 01-3.86.517L6.05 15.21a2 2 0 00-1.806.547M8 4h8l-1 1v5.172a2 2 0 00.586 1.414l5 5c1.26 1.26.367 3.414-1.415 3.414H4.828c-1.782 0-2.674-2.154-1.414-3.414l5-5A2 2 0 009 10.172V5L8 4z" />
                  </svg>
                  <span>{providers[options.provider]?.models?.find((m: any) => m.id === options.model)?.name || options.model}</span>
                </div>
              </div>
            </div>
          </div>
        </div>

        {/* 채팅 컨텐츠 영역 */}
        <div className="flex-1 flex flex-col min-h-0">
          {currentMessages.length === 0 ? (
                            /* 메시지가 없는 화면 - 중앙 입력창 */
            <div className="flex-1 flex items-center justify-center p-4">
              <div className="w-full max-w-5xl">
                <div className="text-center mb-8">
                  <h2 className="text-2xl font-bold text-gray-900 dark:text-white mb-2">
                  How can I help?
                  </h2>
                  <p className="text-gray-600 dark:text-gray-400">
                  아이디어부터 문제 해결까지, AI와 함께 시작해보세요! ✨
                  </p>
                </div>

                {/* 기능 안내 카드들 - 3x1 그리드 */}
                <div className="grid grid-cols-1 md:grid-cols-3 gap-4 mb-8">
                  <button
                    onClick={() => setInputMessage("Python으로 데이터 분석 코드를 작성해주세요")}
                    className="p-6 bg-gradient-to-br from-blue-50 to-blue-100 dark:from-blue-900/20 dark:to-blue-800/20 border border-blue-200 dark:border-blue-700 rounded-xl hover:shadow-lg transition-all duration-200 hover:scale-105 text-left group"
                  >
                    <div className="text-3xl mb-3 group-hover:scale-110 transition-transform">💻</div>
                    <h4 className="font-semibold text-gray-900 dark:text-white mb-2">코드 생성 & 실행</h4>
                    <p className="text-sm text-gray-600 dark:text-gray-300 leading-relaxed">
                      Python, JavaScript 등 다양한 언어로 코드를 작성하고 실시간으로 실행하여 결과를 확인할 수 있습니다
                    </p>
                  </button>
                  
                  <button
                    onClick={() => setInputMessage("기존 코드를 리팩토링하고 최적화해주세요")}
                    className="p-6 bg-gradient-to-br from-purple-50 to-purple-100 dark:from-purple-900/20 dark:to-purple-800/20 border border-purple-200 dark:border-purple-700 rounded-xl hover:shadow-lg transition-all duration-200 hover:scale-105 text-left group"
                  >
                    <div className="text-3xl mb-3 group-hover:scale-110 transition-transform">🔧</div>
                    <h4 className="font-semibold text-gray-900 dark:text-white mb-2">코드 분석 & 개선</h4>
                    <p className="text-sm text-gray-600 dark:text-gray-300 leading-relaxed">
                      기존 코드를 분석하여 버그를 찾고, 성능을 개선하며, 더 나은 구조로 리팩토링할 수 있습니다
                    </p>
                  </button>
                  
                  <button
                    onClick={() => setInputMessage("프로젝트 구조를 설계하고 개발 계획을 세워주세요")}
                    className="p-6 bg-gradient-to-br from-green-50 to-green-100 dark:from-green-900/20 dark:to-green-800/20 border border-green-200 dark:border-green-700 rounded-xl hover:shadow-lg transition-all duration-200 hover:scale-105 text-left group"
                  >
                    <div className="text-3xl mb-3 group-hover:scale-110 transition-transform">🏗️</div>
                    <h4 className="font-semibold text-gray-900 dark:text-white mb-2">프로젝트 설계</h4>
                    <p className="text-sm text-gray-600 dark:text-gray-300 leading-relaxed">
                      전체 프로젝트 아키텍처를 설계하고, 개발 단계별 계획을 수립하여 체계적인 개발을 지원합니다
                    </p>
                  </button>
                </div>

              {/* 업로드된 파일 목록 */}
              {uploadedFiles.length > 0 && (
                <div className="mb-4 flex flex-wrap gap-2 justify-center">
                  {uploadedFiles.map((file, index) => {
                    const isImage = isImageFile(file);
                    const fileKey = `${file.name}-${file.size}-${file.lastModified}`;
                    const imageUrl = isImage ? uploadedFileUrls.get(fileKey) : null;
                    
                    return (
                      <div key={`${file.name}-${file.size}-${index}`} className="flex items-center space-x-2 bg-gray-100 dark:bg-gray-700 px-3 py-1 rounded-full text-sm">
                        {isImage && imageUrl ? (
                          <img 
                            src={imageUrl} 
                            alt={file.name}
                            className="w-6 h-6 rounded-full object-cover"
                          />
                        ) : (
                          <span className={`w-2 h-2 rounded-full ${isImage ? 'bg-green-500' : 'bg-blue-500'}`}></span>
                        )}
                        <span className="text-gray-700 dark:text-gray-300 truncate max-w-32">{file.name}</span>
                        <button
                          onClick={() => removeFile(index)}
                          className="text-gray-500 hover:text-red-500 ml-1"
                        >
                          <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                          </svg>
                        </button>
                      </div>
                    );
                  })}
                </div>
              )}

              {/* 통합된 입력 영역 */}
              <div 
                className={`bg-gray-100 dark:bg-gray-700 rounded-2xl border border-gray-200 dark:border-gray-600 p-4 transition-all ${isDragOver ? 'ring-2 ring-blue-500 ring-opacity-50 border-blue-400' : ''}`}
                  onDragEnter={handleDragEnter}
                  onDragLeave={handleDragLeave}
                  onDragOver={handleDragOver}
                  onDrop={handleDrop}
                >
                  {/* 도구 버튼들 */}
                  <div className="flex items-center justify-start gap-2 mb-3">
                    {/* 파일 업로드 버튼 */}
                    <button
                      onClick={() => fileInputRef.current?.click()}
                      disabled={isUploading || isLoading || isLoadingSessions}
                      className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${
                        isUploading ? 'bg-gray-100 dark:bg-gray-600 text-gray-400 dark:text-gray-500' : 
                        'bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-500'
                      }`}
                      title={t('chat.fileUpload', '파일')}
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13" />
                      </svg>
                      <span>{t('chat.fileUpload', '파일')}</span>
                    </button>

                    {/* 문서검색 버튼 */}
                    <button
                      onClick={() => setOptions(prev => ({ 
                        ...prev, 
                        enableRag: !prev.enableRag,
                        enableSmartResearch: prev.enableRag ? prev.enableSmartResearch : false
                      }))}
                      disabled={isLoading || isLoadingSessions}
                      className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${
                        options.enableRag ? 
                        'bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300 ring-1 ring-green-200 dark:ring-green-700' : 
                        'bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-500'
                      }`}
                      title={t('chat.documentSearch', '문서검색')}
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                      </svg>
                      <span>{t('chat.documentSearch', '문서검색')}</span>
                    </button>

                    {/* 웹검색 버튼 */}
                    <button
                      onClick={() => setOptions(prev => ({ 
                        ...prev, 
                        enableWebSearch: !prev.enableWebSearch,
                        enableSmartResearch: prev.enableWebSearch ? prev.enableSmartResearch : false
                      }))}
                      disabled={isLoading || isLoadingSessions}
                      className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${
                        options.enableWebSearch ? 
                        'bg-blue-100 dark:bg-blue-900/30 text-blue-700 dark:text-blue-300 ring-1 ring-blue-200 dark:ring-blue-700' : 
                        'bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-500'
                      }`}
                      title={t('chat.webSearch', '웹검색')}
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 12a9 9 0 01-9 9m9-9a9 9 0 00-9-9m9 9H3m9 9v-9m0-9v9" />
                      </svg>
                      <span>{t('chat.webSearch', '웹검색')}</span>
                    </button>

                    {/* AI프로 버튼 */}
                    <button
                      onClick={() => setOptions(prev => ({ 
                        ...prev, 
                        enableSmartResearch: !prev.enableSmartResearch,
                        enableWebSearch: prev.enableSmartResearch ? prev.enableWebSearch : false,
                        enableRag: prev.enableSmartResearch ? prev.enableRag : false
                      }))}
                      disabled={isLoading || isLoadingSessions}
                      className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${
                        options.enableSmartResearch ? 
                        'bg-purple-100 dark:bg-purple-900/30 text-purple-700 dark:text-purple-300 ring-1 ring-purple-200 dark:ring-purple-700' : 
                        'bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-500'
                      }`}
                      title={t('chat.aiProTooltip', 'AI프로 - 검색, 응답 생성, 품질 검증을 자동으로 수행')}
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                      </svg>
                      <span>{t('chat.aiPro', 'AI프로')}</span>
                    </button>
                  </div>

                  {/* 멘션된 문서 표시 */}
                  {mentionedDocs.length > 0 && (
                    <div className="mb-3 flex flex-wrap gap-1">
                      {mentionedDocs.map((doc, index) => (
                        <span 
                          key={index}
                          className="inline-flex items-center px-2 py-1 bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-200 rounded-full text-xs"
                        >
                          <svg className="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                            <path d="M4 4a2 2 0 00-2 2v8a2 2 0 002 2h12a2 2 0 002-2V6a2 2 0 00-2-2H4z"/>
                          </svg>
                          {doc.displayName}
                          <button 
                            onClick={() => removeMentionedDoc(doc.name)}
                            className="ml-1 text-blue-600 hover:text-blue-800"
                          >
                            ×
                          </button>
                        </span>
                      ))}
                    </div>
                  )}

                  {/* 입력창 영역 */}
                  <div className="relative">
                    <textarea
                      ref={(el) => {
                        if (el) {
                          textareaRef.current = el;
                          el.style.height = 'auto';
                          el.style.height = `${Math.min(Math.max(el.scrollHeight, 48), 200)}px`;
                        }
                      }}
                      value={inputMessage}
                      onChange={handleInputChange}
                      onKeyDown={handleKeyDown}
                      placeholder={isLoadingSessions ? t('chat.loadingSessions', '세션을 로드하는 중...') : isDragOver ? t('chat.dropFiles', '파일을 여기에 놓으세요...') : t('chat.askAnything', '무엇이든 물어보세요...')}
                      className="w-full px-0 py-2 pr-12 bg-transparent text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none resize-none border-0 text-base"
                      rows={1}
                      disabled={isLoading || isLoadingSessions}
                      style={{ minHeight: '48px', maxHeight: '200px' }}
                    />

                    {/* 문서 자동완성 드롭다운 */}
                    {showDocSuggestions && filteredDocsForMention.length > 0 && (
                      <div className="absolute bottom-full left-0 right-0 mb-1 bg-gray-50 dark:bg-gray-800 border border-gray-300 dark:border-gray-500 rounded-lg shadow-xl max-h-60 overflow-y-auto z-10 scrollbar-thin scrollbar-thumb-gray-300 dark:scrollbar-thumb-gray-600 scrollbar-track-transparent">
                        {filteredDocsForMention.map((doc, index) => (
                          <button
                            key={index}
                            onClick={() => selectDocument(doc.name)}
                            className={`w-full text-left px-3 py-2 flex items-center space-x-2 transition-colors ${
                              index === selectedSuggestionIndex 
                                ? 'bg-blue-100 dark:bg-blue-900/40 text-blue-900 dark:text-blue-100' 
                                : 'hover:bg-gray-100 dark:hover:bg-gray-700 text-gray-800 dark:text-gray-200'
                            }`}
                          >
                            {getFileIcon(doc.name, doc.status)}
                            <span className="text-sm truncate flex-1" title={doc.name}>{doc.name}</span>
                          </button>
                        ))}
                      </div>
                    )}

                    <button
                      onClick={sendMessage}
                      disabled={(!inputMessage.trim() && uploadedFiles.length === 0) || isLoading || isLoadingSessions}
                      className="absolute right-0 bottom-2 p-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8" />
                      </svg>
                    </button>
                  </div>
                </div>

                {/* 숨겨진 파일 입력 */}
                <input
                  ref={fileInputRef}
                  type="file"
                  multiple
                  accept="image/*,.pdf,.doc,.docx,.txt,.md,.csv,.json"
                  onChange={handleFileSelect}
                  className="hidden"
                />

                {/* 오류 메시지 */}
                {error && (
                  <div className="mt-4 p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-md">
                    <p className="text-sm text-red-800 dark:text-red-200">{error}</p>
                  </div>
                )}
              </div>
            </div>
          ) : (
            /* 메시지가 있는 화면 - 분리된 레이아웃 */
            <>

              
              {/* 메시지 영역 */}
              <div className="flex-1 overflow-y-auto">
                <div className="max-w-5xl mx-auto px-4 py-6">
                  {/* 메시지 목록 */}
                  <div className="space-y-6">
                    {currentMessages.map((message, index) => (
                      <div
                        key={`${message.id}-${index}`}
                        className="flex justify-start"
                      >
                                                  <div className="flex max-w-5xl w-full flex-row">
                          {/* 아바타 */}
                          <div className="flex-shrink-0 mr-3">
                            <div className={`w-8 h-8 rounded-full flex items-center justify-center ${
                              message.role === 'user' 
                                ? 'bg-gray-200 dark:bg-gray-700 text-gray-600 dark:text-gray-300'
                                : 'bg-gray-200 dark:bg-gray-700 text-gray-600 dark:text-gray-300'
                            }`}>
                              {message.role === 'user' ? (
                                <img 
                                  src={userProfileImage} 
                                  alt="User" 
                                  className="w-7 h-7 object-cover rounded-full"
                                />
                              ) : (
                                <img 
                                  src="/images/ai.png" 
                                  alt="AI 워크스페이스" 
                                  className="w-7 h-7 object-contain"
                                />
                              )}
                            </div>
                          </div>
                          {/* 메시지 내용 */}
                          <div className="flex-1 min-w-0 text-left">
                            {message.role === 'assistant' ? (
                              <div className="text-gray-900 dark:text-white">
                                <CodeRenderer 
                                  content={
                                    typingMessageId === message.id 
                                      ? displayedContent 
                                      : message.content
                                  }
                                  messageId={message.id}
                                  onExecuteCode={handleExecuteCode}
                                  onDownloadCode={handleDownloadCode}
                                  onRegenerateCode={handleRegenerateCode}
                                  isExecuting={executingCodes.has(message.id)}
                                  executionResult={executionResults[message.id]}
                                  className=""
                                />
                                {/* 타이핑 중일 때 애니메이션 표시 */}
                                {typingMessageId === message.id && (
                                  <div className="flex items-center mt-2 text-gray-500 dark:text-gray-400 text-sm">
                                    <div className="flex space-x-1">
                                      <div className="w-2 h-2 bg-blue-500 rounded-full animate-bounce"></div>
                                      <div className="w-2 h-2 bg-blue-500 rounded-full animate-bounce" style={{ animationDelay: '0.1s' }}></div>
                                      <div className="w-2 h-2 bg-blue-500 rounded-full animate-bounce" style={{ animationDelay: '0.2s' }}></div>
                                    </div>
                                    <span className="ml-2">AI가 응답을 작성하고 있습니다...</span>
                                  </div>
                                )}
                                {/* 타이핑 중이 아닐 때만 다운로드 링크 렌더링 */}
                                {typingMessageId !== message.id && renderDownloadLinks(message.content)}
                              </div>
                            ) : (
                              <div className="text-blue-900 dark:text-blue-200">
                                {/* 첨부된 이미지들 표시 */}
                                {message.attachedImages && message.attachedImages.length > 0 && (
                                  <div className="mb-3 flex flex-wrap gap-2">
                                    {message.attachedImages.map((image, imgIndex) => (
                                      <div key={`${message.id}-img-${imgIndex}`} className="relative group">
                                        <img 
                                          src={image.url} 
                                          alt={image.name}
                                          className="max-w-xs max-h-48 rounded-lg object-cover border border-gray-200 dark:border-gray-600 shadow-sm hover:shadow-md transition-shadow cursor-pointer"
                                          onClick={() => {
                                            // 이미지 클릭 시 새 탭에서 크게 보기
                                            window.open(image.url, '_blank');
                                          }}
                                          title={`${image.name} - ${t('chat.clickToEnlarge', '클릭하여 크게 보기')}`}
                                        />
                                        <div className="absolute bottom-1 left-1 bg-black bg-opacity-50 text-white text-xs px-2 py-1 rounded opacity-0 group-hover:opacity-100 transition-opacity">
                                          {image.name}
                                        </div>
                                      </div>
                                    ))}
                                  </div>
                                )}
                                <CodeRenderer 
                                  content={message.originalContent || message.content || ' '}
                                  className=""
                                  messageId={message.id}
                                  onExecuteCode={handleExecuteCode}
                                  onDownloadCode={handleDownloadCode}
                                  onRegenerateCode={handleRegenerateCode}
                                  isExecuting={executingCodes.has(message.id)}
                                  executionResult={executionResults[message.id]}
                                />
                              </div>
                            )}
                            <p className="text-gray-500 dark:text-gray-400 mt-1 chat-sidebar-small-text">
                              {(() => {
                                const date = new Date(message.timestamp);
                                return isNaN(date.getTime()) ? '' : date.toLocaleTimeString('ko-KR');
                              })()}
                            </p>
                            {/* AI 응답 액션 버튼들 - 타이핑 완료 후에만 표시 */}
                            {message.role === 'assistant' && typingMessageId !== message.id && (
                              <div className="flex items-center justify-end space-x-2 mt-2">
                                {/* 클립보드 복사 */}
                                <button
                                  onClick={() => copyToClipboard(message.content, message.id)}
                                  className={`p-1.5 rounded transition-colors group ${
                                    copiedMessageId === message.id
                                      ? 'text-green-600 bg-green-50 dark:bg-green-900/20'
                                      : 'text-gray-400 hover:text-blue-600 dark:hover:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/20'
                                  }`}
                                  title={copiedMessageId === message.id ? t('chat.copyCompleted', '복사 완료!') : t('chat.copyToClipboard', '응답 내용을 클립보드에 복사')}
                                >
                                  {copiedMessageId === message.id ? (
                                    // 복사 완료 체크 아이콘
                                    <svg className="w-4 h-4 group-hover:scale-110 transition-transform" fill="currentColor" viewBox="0 0 24 24">
                                      <path d="M9 16.17L4.83 12l-1.42 1.41L9 19 21 7l-1.41-1.41z"/>
                                    </svg>
                                  ) : (
                                    // 기본 복사 아이콘
                                    <svg className="w-4 h-4 group-hover:scale-110 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 16H6a2 2 0 01-2-2V6a2 2 0 012-2h8a2 2 0 012 2v2m-6 12h8a2 2 0 002-2v-8a2 2 0 00-2-2h-8a2 2 0 00-2 2v8a2 2 0 002 2z" />
                                    </svg>
                                  )}
                                </button>

                                {/* PDF 저장 */}
                                <button
                                  onClick={() => saveToPDF(message.content, message.id)}
                                  disabled={pdfProcessingId === message.id}
                                  className={`p-1.5 rounded transition-colors group ${
                                    pdfProcessingId === message.id
                                      ? 'text-orange-600 bg-orange-50 dark:bg-orange-900/20 cursor-wait'
                                      : 'text-gray-400 hover:text-red-600 dark:hover:text-red-400 hover:bg-red-50 dark:hover:bg-red-900/20'
                                  }`}
                                  title={pdfProcessingId === message.id ? t('chat.generatingPdf', 'PDF 생성 중...') : t('chat.saveToPdf', '응답을 PDF로 저장 (인쇄 창 열림)')}
                                >
                                  {pdfProcessingId === message.id ? (
                                    // 로딩 스피너
                                    <svg className="w-4 h-4 animate-spin" fill="none" viewBox="0 0 24 24">
                                      <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                      <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                    </svg>
                                  ) : (
                                    // 기본 PDF 아이콘
                                    <svg className="w-4 h-4 group-hover:scale-110 transition-transform" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                                    </svg>
                                  )}
                                </button>

                                {/* 좋아요 */}
                                <button
                                  onClick={() => handleLike(message.id)}
                                  className={`p-1.5 rounded transition-colors ${
                                    messageReactions[message.id]?.liked
                                      ? 'text-green-600 bg-green-50 dark:bg-green-900/20 hover:bg-green-100 dark:hover:bg-green-900/30'
                                      : 'text-gray-400 hover:text-green-600 hover:bg-gray-100 dark:hover:bg-gray-700'
                                  }`}
                                  title={t('chat.likeMessage', '좋아요')}
                                >
                                  <svg className="w-4 h-4" fill={messageReactions[message.id]?.liked ? "currentColor" : "none"} stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M14 10h4.764a2 2 0 011.789 2.894l-3.5 7A2 2 0 0115.263 21h-4.017c-.163 0-.326-.02-.485-.06L7 20m7-10V5a2 2 0 00-2-2h-.095c-.5 0-.905.405-.905.905 0 .714-.211 1.412-.608 2.006L7 11v9m7-10h-2M7 20H5a2 2 0 01-2-2v-6a2 2 0 012-2h2.5" />
                                  </svg>
                                </button>

                                {/* 싫어요 */}
                                <button
                                  onClick={() => handleDislike(message.id)}
                                  className={`p-1.5 rounded transition-colors ${
                                    messageReactions[message.id]?.disliked
                                      ? 'text-red-600 bg-red-50 dark:bg-red-900/20 hover:bg-red-100 dark:hover:bg-red-900/30'
                                      : 'text-gray-400 hover:text-red-600 hover:bg-gray-100 dark:hover:bg-gray-700'
                                  }`}
                                  title={t('chat.dislikeMessage', '싫어요')}
                                >
                                  <svg className="w-4 h-4" fill={messageReactions[message.id]?.disliked ? "currentColor" : "none"} stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 14H5.236a2 2 0 01-1.789-2.894l3.5-7A2 2 0 018.736 3h4.018c.163 0 .326.02.485.06L17 4m-7 10v2a2 2 0 002 2h.095c.5 0 .905-.405.905-.905 0-.714.211-1.412.608-2.006L17 13V4m-7 10h2m5-10h2a2 2 0 012 2v6a2 2 0 01-2 2h-2.5" />
                                  </svg>
                                </button>
                              </div>
                            )}

                            {/* 코드 실행 관련 버튼들 - 타이핑 완료 후에만 표시 */}
                            {message.role === 'assistant' && typingMessageId !== message.id && extractCodeFromMessage(message.content) && (
                              <div className="mt-3 p-3 bg-gray-50 dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-700">
                                <div className="flex items-center justify-between mb-3">
                                  <div className="flex items-center space-x-2">
                                    <svg className="w-4 h-4 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 20l4-16m4 4l4 4-4 4M6 16l-4-4 4-4" />
                                    </svg>
                                    <span className="text-sm font-medium text-gray-700 dark:text-gray-300">코드 실행 해보고 결과를 확인하세요</span>
                                  </div>
                                  {(successFlags[message.id] === 'yes' || successFlags[message.id] === 'no') && (
                                    <div className={`flex items-center space-x-1 px-2 py-1 rounded-full text-xs ${
                                      successFlags[message.id] === 'yes' 
                                        ? 'bg-green-100 text-green-700 dark:bg-green-900/20 dark:text-green-300'
                                        : 'bg-red-100 text-red-700 dark:bg-red-900/20 dark:text-red-300'
                                    }`}>
                                      {successFlags[message.id] === 'yes' ? (
                                        <>
                                          <svg className="w-3 h-3" fill="currentColor" viewBox="0 0 20 20">
                                            <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                                          </svg>
                                          <span>실행 성공</span>
                                        </>
                                      ) : (
                                        <>
                                          <svg className="w-3 h-3" fill="currentColor" viewBox="0 0 20 20">
                                            <path fillRule="evenodd" d="M4.293 4.293a1 1 0 011.414 0L10 8.586l4.293-4.293a1 1 0 111.414 1.414L11.414 10l4.293 4.293a1 1 0 01-1.414 1.414L10 11.414l-4.293 4.293a1 1 0 01-1.414-1.414L8.586 10 4.293 5.707a1 1 0 010-1.414z" clipRule="evenodd" />
                                          </svg>
                                          <span>실행 실패</span>
                                        </>
                                      )}
                                    </div>
                                  )}
                                </div>
                                
                                {/* 1단계: 코드 실행 및 재생성 (AI 응답 직후 사용 가능) */}
                                <div className="flex flex-wrap gap-2">
                                  {/* 코드 실행 버튼 */}
                                  <button
                                    onClick={() => {
                                      const code = extractCodeFromMessage(message.content);
                                      const prompt = findUserPromptForMessage(message.id);
                                      if (code) {
                                        executeCode(message.id, prompt, code);
                                      }
                                    }}
                                    disabled={executingCodeMessageId === message.id}
                                    className={`flex items-center space-x-2 px-3 py-1.5 rounded-md text-sm font-medium transition-colors ${
                                      executingCodeMessageId === message.id
                                        ? 'bg-blue-100 text-blue-700 dark:bg-blue-900/20 dark:text-blue-300 cursor-wait'
                                        : 'bg-blue-600 text-white hover:bg-blue-700 dark:bg-blue-500 dark:hover:bg-blue-600'
                                    }`}
                                  >
                                    {executingCodeMessageId === message.id ? (
                                      <>
                                        <svg className="w-4 h-4 animate-spin" fill="none" viewBox="0 0 24 24">
                                          <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                          <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                        </svg>
                                        <span>실행 중...</span>
                                      </>
                                    ) : (
                                      <>
                                        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M14.828 14.828a4 4 0 01-5.656 0M9 10h1m4 0h1m-6 4h1m4 0h1m-7 4h8a2 2 0 002-2V8a2 2 0 00-2-2H8a2 2 0 00-2 2v8a2 2 0 002 2z" />
                                        </svg>
                                        <span>코드 실행</span>
                                      </>
                                    )}
                                  </button>

                                  {/* 코드 재생성 버튼 - AI 응답 후 항상 표시 */}
                                  <button
                                    onClick={() => {
                                      regenerateCode(message.id);
                                    }}
                                    disabled={isLoading || executingCodeMessageId === message.id}
                                    className="flex items-center space-x-2 px-3 py-1.5 bg-orange-600 !text-white hover:bg-orange-700 dark:bg-orange-500 dark:hover:bg-orange-600 rounded-md text-sm font-medium transition-colors disabled:opacity-50 disabled:cursor-not-allowed"
                                  >
                                    <svg className="w-4 h-4 !text-white" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 4v5h.582m15.356 2A8.001 8.001 0 004.582 9m0 0H9m11 11v-5h-.581m0 0a8.003 8.003 0 01-15.357-2m15.357 2H15" />
                                    </svg>
                                    <span>코드 재생성</span>
                                  </button>
                                </div>

                                {/* 2단계: 성공 후 저장 옵션 (실행 성공 시에만 표시) */}
                                {successFlags[message.id] === 'yes' && (
                                  <div className="flex flex-wrap gap-2 border-t border-gray-200 dark:border-gray-700 pt-2 mt-2">
                                    <div className="w-full mb-1">
                                      <span className="text-xs text-gray-500 dark:text-gray-400 font-medium">💾 저장 옵션</span>
                                    </div>
                                    
                                    {/* 코드 다운로드 버튼 */}
                                    {/*
                                    <button
                                      onClick={() => {
                                        downloadCode(message.id);
                                      }}
                                      disabled={downloadFlags[message.id] === 'downloading'}
                                      className={`flex items-center space-x-2 px-3 py-1.5 rounded-md text-sm font-medium transition-colors ${
                                        downloadFlags[message.id] === 'downloading'
                                          ? 'bg-green-100 text-green-700 dark:bg-green-900/20 dark:text-green-300 cursor-wait'
                                          : downloadFlags[message.id] === 'completed'
                                          ? 'bg-green-100 text-green-700 dark:bg-green-900/20 dark:text-green-300'
                                          : 'bg-green-600 text-white hover:bg-green-700 dark:bg-green-500 dark:hover:bg-green-600'
                                      }`}
                                    >
                                      {downloadFlags[message.id] === 'downloading' ? (
                                        <>
                                          <svg className="w-4 h-4 animate-spin" fill="none" viewBox="0 0 24 24">
                                            <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                            <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                          </svg>
                                          <span>다운로드 중...</span>
                                        </>
                                      ) : downloadFlags[message.id] === 'completed' ? (
                                        <>
                                          <svg className="w-4 h-4" fill="currentColor" viewBox="0 0 20 20">
                                            <path fillRule="evenodd" d="M16.707 5.293a1 1 0 010 1.414l-8 8a1 1 0 01-1.414 0l-4-4a1 1 0 011.414-1.414L8 12.586l7.293-7.293a1 1 0 011.414 0z" clipRule="evenodd" />
                                          </svg>
                                          <span>다운로드 완료</span>
                                        </>
                                      ) : (
                                        <>
                                          <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                                          </svg>
                                          <span className="!text-white">코드 다운로드</span>
                                        </>
                                      )}
                                    </button>
                                    */}

                                    {/* Task 저장 버튼 */}
                                    <button
                                      onClick={() => {
                                        openTaskSaveModal(message.id);
                                      }}
                                      className="flex items-center space-x-2 px-3 py-1.5 bg-purple-600 text-white hover:bg-purple-700 dark:bg-purple-500 dark:hover:bg-purple-600 rounded-md text-sm font-medium transition-colors"
                                    >
                                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 7H5a2 2 0 00-2 2v9a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-3m-1 4l-3 3m0 0l-3-3m3 3V4" />
                                      </svg>
                                      <span className="!text-white">작업 저장</span>
                                    </button>
                                  </div>
                                )}

                                {/* 오류 메시지 표시 */}
                                {reviewFlags[message.id] && (
                                  <div className="mt-3 p-2 bg-red-50 dark:bg-red-900/20 rounded border border-red-200 dark:border-red-700">
                                    <div className="flex items-start space-x-2">
                                      <svg className="w-4 h-4 text-red-600 dark:text-red-400 mt-0.5 flex-shrink-0" fill="currentColor" viewBox="0 0 20 20">
                                        <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
                                      </svg>
                                      <div className="flex-1">
                                        <h4 className="text-sm font-medium text-red-800 dark:text-red-200">코드 실행 오류</h4>
                                        <p className="text-sm text-red-700 dark:text-red-300 mt-1 whitespace-pre-wrap">{reviewFlags[message.id]}</p>
                                      </div>
                                    </div>
                                  </div>
                                )}

                                {/* 성공한 실행 결과 표시 */}
                                {executionResults[message.id] && (
                                  <div className="mt-3 p-3 bg-green-50 dark:bg-green-900/20 rounded-lg border border-green-200 dark:border-green-700">
                                    <div className="flex items-start space-x-2">
                                      <svg className="w-5 h-5 text-green-600 dark:text-green-400 mt-0.5 flex-shrink-0" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                                      </svg>
                                      <div className="flex-1">
                                        <h4 className="text-sm font-medium text-green-800 dark:text-green-200 mb-1">✅ 코드 실행 완료</h4>
                                        <div className="text-sm text-green-700 dark:text-green-300 font-mono bg-green-100 dark:bg-green-800/30 p-2 rounded border border-green-200 dark:border-green-600 whitespace-pre-wrap overflow-x-auto">
                                          {executionResults[message.id]}
                                        </div>
                                      </div>
                                    </div>
                                  </div>
                                )}
                              </div>
                            )}
                          </div>
                        </div>
                      </div>
                    ))}
                    {/* 로딩 중 Think... 표시 */}
                    {isLoading && (
                      <div className="flex justify-start">
                        <div className="flex max-w-5xl w-full">
                          <div className="flex-shrink-0 mr-3">
                            <div className="w-8 h-8 rounded-full bg-gray-200 dark:bg-gray-700 flex items-center justify-center">
                              <img 
                                src="/images/ai.png" 
                                alt="Hamonize AI" 
                                className="w-7 h-7 object-contain"
                              />
                            </div>
                          </div>
                          <div className="flex-1 min-w-0">
                            <div className="flex items-center space-x-2 mb-1">
                              <div className="relative">
                                <svg className="animate-spin h-6 w-6 text-gray-400 dark:text-gray-500" fill="none" viewBox="0 0 24 24">
                                  <circle className="opacity-25" cx="12" cy="12" r="10" stroke="currentColor" strokeWidth="4"></circle>
                                  <path className="opacity-75" fill="currentColor" d="M4 12a8 8 0 018-8V0C5.373 0 0 5.373 0 12h4zm2 5.291A7.962 7.962 0 014 12H0c0 3.042 1.135 5.824 3 7.938l3-2.647z"></path>
                                </svg>
                              </div>
                              <span className="text-base text-gray-600 dark:text-gray-500 opacity-60">
                                {`Think${thinkingDots}`}
                              </span>
                            </div>
                          </div>
                        </div>
                      </div>
                    )}
                    
                    {/* 진행 상황 메시지들 - 로딩 중이거나 완료 후에도 표시 */}
                    {progressMessages.length > 0 && (
                      <div className="flex justify-start mb-2">
                        <div className="flex max-w-5xl w-full">
                          <div className="flex-shrink-0 mr-3">
                            {/* 로딩 중이 아닐 때는 빈 공간 유지 */}
                            <div className="w-2 h-2"></div>
                          </div>
                          <div className="flex-1 min-w-0">
                            <div className="space-y-0.5">
                              {progressMessages.map((message, index) => (
                                <div 
                                  key={index} 
                                  className="group flex items-center space-x-2 py-0.5 px-5 transition-all duration-300 border-0"
                                >
                                  <div className={`w-1.5 h-1.5 rounded-full flex-shrink-0 ${
                                    isLoading 
                                      ? 'bg-blue-400 dark:bg-blue-300 animate-pulse' 
                                      : 'bg-green-400 dark:bg-green-300'
                                  }`}></div>
                                  <span 
                                    className="text-sm text-gray-600 dark:text-gray-500 leading-tight select-none font-medium tracking-wide opacity-60 group-hover:opacity-80 transition-opacity duration-200"
                                    style={{ 
                                      fontSize: '12px',
                                      lineHeight: '1.1'
                                    }}
                                  >
                                    {message}
                                  </span>
                                </div>
                              ))}
                            </div>
                          </div>
                        </div>
                      </div>
                    )}
                  </div>
                  <div ref={messagesEndRef} className="h-1" />
                </div>
              </div>
              


                {/* 하단 입력 영역 - 고정 */}
                <div className="flex-shrink-0 border-t border-gray-200 dark:border-gray-700" style={{ backgroundColor: 'var(--body-bg)' }}>
                <div className="max-w-5xl mx-auto px-4 py-4">


                  {/* 업로드된 파일 목록 */}
                  {uploadedFiles.length > 0 && (
                    <div className="mb-3">
                      <div className="flex flex-wrap gap-2">
                        {uploadedFiles.map((file, index) => {
                          const isImage = isImageFile(file);
                          const imageFiles = uploadedFiles.filter(isImageFile);
                          const isFirstImage = isImage && imageFiles.indexOf(file) === 0;
                          const willBeProcessed = !isImage || isFirstImage;
                          const fileKey = `${file.name}-${file.size}-${file.lastModified}`;
                          const imageUrl = isImage ? uploadedFileUrls.get(fileKey) : null;
                          
                          return (
                            <div 
                              key={`${file.name}-${file.size}-${index}`} 
                              className={`flex items-center space-x-2 px-3 py-1 rounded-full text-sm ${
                                willBeProcessed 
                                  ? 'bg-gray-100 dark:bg-gray-700' 
                                  : 'bg-yellow-50 dark:bg-yellow-900/30 border border-yellow-200 dark:border-yellow-700'
                              }`}
                            >
                              {isImage && imageUrl ? (
                                <img 
                                  src={imageUrl} 
                                  alt={file.name}
                                  className="w-6 h-6 rounded-full object-cover"
                                />
                              ) : (
                                <span className={`w-2 h-2 rounded-full ${
                                  isImage 
                                    ? (willBeProcessed ? 'bg-green-500' : 'bg-yellow-500') 
                                    : 'bg-blue-500'
                                }`}></span>
                              )}
                              <span className={`truncate max-w-32 ${
                                willBeProcessed 
                                  ? 'text-gray-700 dark:text-gray-300' 
                                  : 'text-yellow-700 dark:text-yellow-300'
                              }`}>
                                {file.name}
                                {isImage && !willBeProcessed && ' (대기)'}
                              </span>
                              <button
                                onClick={() => removeFile(index)}
                                className="text-gray-500 hover:text-red-500 ml-1"
                              >
                                <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                                </svg>
                              </button>
                            </div>
                          );
                        })}
                      </div>
                      
                      {/* 여러 이미지 업로드 시 안내 메시지 */}
                      {uploadedFiles.filter(isImageFile).length > 1 && (
                        <div className="mt-2 p-2 bg-yellow-50 dark:bg-yellow-900/20 border border-yellow-200 dark:border-yellow-700 rounded-md">
                                                      <p className="text-xs text-yellow-700 dark:text-yellow-300 flex items-center">
                              <svg className="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z" />
                              </svg>
                              {t('chat.multipleImagesWarning', '여러 이미지가 업로드되었습니다. 현재는 첫 번째 이미지만 분석됩니다.')}
                            </p>
                        </div>
                      )}
                    </div>
                  )}

                {/* 통합된 입력 영역 */}
                <div 
                  className={`bg-gray-100 dark:bg-gray-700 rounded-2xl border border-gray-200 dark:border-gray-600 p-4 transition-all ${isDragOver ? 'ring-2 ring-blue-500 ring-opacity-50 border-blue-400' : ''}`}
                  onDragEnter={handleDragEnter}
                  onDragLeave={handleDragLeave}
                  onDragOver={handleDragOver}
                  onDrop={handleDrop}
                >
                  {/* 도구 버튼들 */}
                  <div className="flex items-center justify-start gap-2 mb-3">
                    {/* 파일 업로드 버튼 */}
                    <button
                      onClick={() => fileInputRef.current?.click()}
                      disabled={isUploading || isLoading || isLoadingSessions}
                      className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${
                        isUploading ? 'bg-gray-100 dark:bg-gray-600 text-gray-400 dark:text-gray-500' : 
                        'bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-500'
                      }`}
                      title={t('chat.fileUpload', '파일')}
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15.172 7l-6.586 6.586a2 2 0 102.828 2.828l6.414-6.586a4 4 0 00-5.656-5.656l-6.415 6.585a6 6 0 108.486 8.486L20.5 13" />
                      </svg>
                      <span>{t('chat.fileUpload', '파일')}</span>
                    </button>

                    {/* 문서검색 버튼 */}
                    <button
                      onClick={() => setOptions(prev => ({ 
                        ...prev, 
                        enableRag: !prev.enableRag,
                        enableSmartResearch: prev.enableRag ? prev.enableSmartResearch : false
                      }))}
                      disabled={isLoading || isLoadingSessions}
                      className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${
                        options.enableRag ? 
                        'bg-green-100 dark:bg-green-900/30 text-green-700 dark:text-green-300 ring-1 ring-green-200 dark:ring-green-700' : 
                        'bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-500'
                      }`}
                      title={t('chat.documentSearch', '문서검색')}
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                      </svg>
                      <span>{t('chat.documentSearch', '문서검색')}</span>
                    </button>

                    {/* 웹검색 버튼 */}
                    <button
                      onClick={() => setOptions(prev => ({ 
                        ...prev, 
                        enableWebSearch: !prev.enableWebSearch,
                        enableSmartResearch: prev.enableWebSearch ? prev.enableSmartResearch : false
                      }))}
                      disabled={isLoading || isLoadingSessions}
                      className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${
                        options.enableWebSearch ? 
                        'bg-blue-100 dark:bg-blue-900/30 text-blue-700 dark:text-blue-300 ring-1 ring-blue-200 dark:ring-blue-700' : 
                        'bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-500'
                      }`}
                      title={t('chat.webSearch', '웹검색')}
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 12a9 9 0 01-9 9m9-9a9 9 0 00-9-9m9 9H3m9 9v-9m0-9v9" />
                      </svg>
                      <span>{t('chat.webSearch', '웹검색')}</span>
                    </button>

                    {/* AI프로 버튼 */}
                    <button
                      onClick={() => setOptions(prev => ({ 
                        ...prev, 
                        enableSmartResearch: !prev.enableSmartResearch,
                        enableWebSearch: prev.enableSmartResearch ? prev.enableWebSearch : false,
                        enableRag: prev.enableSmartResearch ? prev.enableRag : false
                      }))}
                      disabled={isLoading || isLoadingSessions}
                      className={`flex items-center gap-1.5 px-3 py-1.5 rounded-full text-sm font-medium transition-all ${
                        options.enableSmartResearch ? 
                        'bg-purple-100 dark:bg-purple-900/30 text-purple-700 dark:text-purple-300 ring-1 ring-purple-200 dark:ring-purple-700' : 
                        'bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-500'
                      }`}
                      title={t('chat.aiProTooltip', 'AI프로 - 검색, 응답 생성, 품질 검증을 자동으로 수행')}
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9.663 17h4.673M12 3v1m6.364 1.636l-.707.707M21 12h-1M4 12H3m3.343-5.657l-.707-.707m2.828 9.9a5 5 0 117.072 0l-.548.547A3.374 3.374 0 0014 18.469V19a2 2 0 11-4 0v-.531c0-.895-.356-1.754-.988-2.386l-.548-.547z" />
                      </svg>
                      <span>{t('chat.aiPro', 'AI프로')}</span>
                    </button>
                  </div>

                  {/* 멘션된 문서 표시 */}
                  {mentionedDocs.length > 0 && (
                    <div className="mb-3 flex flex-wrap gap-1">
                      {mentionedDocs.map((doc, index) => (
                        <span 
                          key={index}
                          className="inline-flex items-center px-2 py-1 bg-blue-100 dark:bg-blue-900/30 text-blue-800 dark:text-blue-200 rounded-full text-xs"
                        >
                          <svg className="w-3 h-3 mr-1" fill="currentColor" viewBox="0 0 20 20">
                            <path d="M4 4a2 2 0 00-2 2v8a2 2 0 002 2h12a2 2 0 002-2V6a2 2 0 00-2-2H4z"/>
                          </svg>
                          {doc.displayName}
                          <button 
                            onClick={() => removeMentionedDoc(doc.name)}
                            className="ml-1 text-blue-600 hover:text-blue-800"
                          >
                            ×
                          </button>
                        </span>
                      ))}
                    </div>
                  )}

                  {/* 입력창 영역 */}
                  <div className="relative">
                    <textarea
                      ref={(el) => {
                        if (el) {
                          textareaRef.current = el;
                          el.style.height = 'auto';
                          el.style.height = `${Math.min(Math.max(el.scrollHeight, 48), 200)}px`;
                        }
                      }}
                      value={inputMessage}
                      onChange={handleInputChange}
                      onKeyDown={handleKeyDown}
                      placeholder={isLoadingSessions ? t('chat.loadingSessions', '세션을 로드하는 중...') : isDragOver ? t('chat.dropFiles', '파일을 여기에 놓으세요...') : t('chat.askAnything', '무엇이든 물어보세요...')}
                      className="w-full px-0 py-2 pr-12 bg-transparent text-gray-900 dark:text-white placeholder-gray-500 dark:placeholder-gray-400 focus:outline-none resize-none border-0 text-base"
                      rows={1}
                      disabled={isLoading || isLoadingSessions}
                      style={{ minHeight: '48px', maxHeight: '200px' }}
                    />

                    {/* 문서 자동완성 드롭다운 */}
                    {showDocSuggestions && filteredDocsForMention.length > 0 && (
                      <div className="absolute bottom-full left-0 right-0 mb-1 bg-gray-50 dark:bg-gray-800 border border-gray-300 dark:border-gray-500 rounded-lg shadow-xl max-h-60 overflow-y-auto z-10 scrollbar-thin scrollbar-thumb-gray-300 dark:scrollbar-thumb-gray-600 scrollbar-track-transparent">
                        {filteredDocsForMention.map((doc, index) => (
                          <button
                            key={index}
                            onClick={() => selectDocument(doc.name)}
                            className={`w-full text-left px-3 py-2 flex items-center space-x-2 transition-colors ${
                              index === selectedSuggestionIndex 
                                ? 'bg-blue-100 dark:bg-blue-900/40 text-blue-900 dark:text-blue-100' 
                                : 'hover:bg-gray-100 dark:hover:bg-gray-700 text-gray-800 dark:text-gray-200'
                            }`}
                          >
                            {getFileIcon(doc.name, doc.status)}
                            <span className="text-sm truncate flex-1" title={doc.name}>{doc.name}</span>
                          </button>
                        ))}
                      </div>
                    )}

                    <button
                      onClick={sendMessage}
                      disabled={(!inputMessage.trim() && uploadedFiles.length === 0) || isLoading || isLoadingSessions}
                      className="absolute right-0 bottom-2 p-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 focus:outline-none focus:ring-2 focus:ring-blue-500 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
                    >
                      <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 19l9 2-9-18-9 18 9-2zm0 0v-8" />
                      </svg>
                    </button>
                  </div>
                </div>

                  {/* 숨겨진 파일 입력 */}
                  <input
                    ref={fileInputRef}
                    type="file"
                    multiple
                    accept="image/*,.pdf,.doc,.docx,.txt,.md,.csv,.json"
                    onChange={handleFileSelect}
                    className="hidden"
                  />

                  {/* 오류 메시지 */}
                  {error && (
                    <div className="mt-4 p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-md">
                      <p className="text-sm text-red-800 dark:text-red-200">{error}</p>
                    </div>
                  )}
                </div>
              </div>
            </>
          )}
        </div>
      </div>

              {/* 삭제 확인 모달 */}
        {showDeleteModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="delete-modal" style={{ backgroundColor: 'var(--card-bg)' }}>
            <div className="delete-modal-header">
              <div className="delete-modal-icon">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z" />
                </svg>
              </div>
              <div className="delete-modal-text">
                <h3 className="delete-modal-title">
                  {t('chat.deleteConversation', '대화 삭제')}
                </h3>
                <p className="delete-modal-subtitle">
                  {t('chat.actionCannotBeUndone', '이 작업은 되돌릴 수 없습니다.')}
                </p>
              </div>
            </div>
            
            <p className="delete-modal-content">
              {t('chat.deleteConversationConfirm', '선택한 대화를 영구적으로 삭제하시겠습니까? 대화 내용과 모든 메시지가 삭제됩니다.')}
            </p>
            
            <div className="delete-modal-buttons">
              <button
                onClick={cancelDelete}
                className="delete-modal-button delete-modal-button-cancel"
              >
                Cancel
              </button>
              <button
                onClick={confirmDelete}
                className="delete-modal-button delete-modal-button-delete"
              >
                Delete
              </button>
            </div>
          </div>
        </div>
      )}

              {/* 전체 삭제 확인 모달 */}
        {showDeleteAllModal && (
          <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
            <div className="delete-modal" style={{ backgroundColor: 'var(--card-bg)' }}>
            <div className="delete-modal-header">
              <div className="delete-modal-icon">
                <svg fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 9v2m0 4h.01m-6.938 4h13.856c1.54 0 2.502-1.667 1.732-2.5L13.732 4c-.77-.833-1.964-.833-2.732 0L4.082 16.5c-.77.833.192 2.5 1.732 2.5z" />
                </svg>
              </div>
              <div className="delete-modal-text">
                <h3 className="delete-modal-title">
                  {t('chat.deleteAllConversations', '모든 대화 삭제')}
                </h3>
                <p className="delete-modal-subtitle">
                  {t('chat.actionCannotBeUndone', '이 작업은 되돌릴 수 없습니다.')}
                </p>
              </div>
            </div>
            
            <p className="delete-modal-content">
              {t('chat.deleteAllConversationsConfirm', '모든 대화를 영구적으로 삭제하시겠습니까? 전체 대화 목록과 모든 메시지가 삭제됩니다.')}
            </p>
            
            <div className="delete-modal-buttons">
              <button
                onClick={cancelDeleteAll}
                className="delete-modal-button delete-modal-button-cancel"
              >
                Cancel
              </button>
              <button
                onClick={confirmDeleteAll}
                className="delete-modal-button delete-modal-button-delete"
              >
                Delete
              </button>
            </div>
          </div>
        </div>
      )}

      {/* 토스트 알림 */}
      {showToast && (
        <div className="fixed top-4 right-4 z-50">
          <div className={`px-4 py-3 rounded-lg shadow-lg flex items-center space-x-2 ${
            showToast.type === 'success' 
              ? 'bg-green-100 dark:bg-green-900/30 text-green-800 dark:text-green-200 border border-green-200 dark:border-green-700'
              : 'bg-red-100 dark:bg-red-900/30 text-red-800 dark:text-red-200 border border-red-200 dark:border-red-700'
          }`}>
            {showToast.type === 'success' ? (
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zm3.707-9.293a1 1 0 00-1.414-1.414L9 10.586 7.707 9.293a1 1 0 00-1.414 1.414l2 2a1 1 0 001.414 0l4-4z" clipRule="evenodd" />
              </svg>
            ) : (
              <svg className="w-5 h-5" fill="currentColor" viewBox="0 0 20 20">
                <path fillRule="evenodd" d="M10 18a8 8 0 100-16 8 8 0 000 16zM8.707 7.293a1 1 0 00-1.414 1.414L8.586 10l-1.293 1.293a1 1 0 101.414 1.414L10 11.414l1.293 1.293a1 1 0 001.414-1.414L11.414 10l1.293-1.293a1 1 0 00-1.414-1.414L10 8.586 8.707 7.293z" clipRule="evenodd" />
              </svg>
            )}
            <span className="text-sm font-medium">{showToast.message}</span>
          </div>
        </div>
      )}

      {/* Task 저장 모달 */}
      {showTaskSaveModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow-xl max-w-md w-full mx-4">
            <div className="p-6">
              <div className="flex items-center mb-4">
                <svg className="w-6 h-6 text-purple-600 dark:text-purple-400 mr-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 7H5a2 2 0 00-2 2v9a2 2 0 002 2h14a2 2 0 002-2V9a2 2 0 00-2-2h-3m-1 4l-3 3m0 0l-3-3m3 3V4" />
                </svg>
                <h3 className="text-lg font-semibold text-gray-900 dark:text-white">
                  작업 저장
                </h3>
              </div>
              
              <p className="text-sm text-gray-600 dark:text-gray-400 mb-4">
                코드와 프롬프트를 작업으로 저장합니다. 나중에 다시 사용할 수 있습니다.
              </p>
              
              <div className="mb-4">
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  작업 이름
                </label>
                <input
                  type="text"
                  value={taskName}
                  onChange={(e) => setTaskName(e.target.value)}
                  placeholder="예: 데이터 분석 스크립트"
                  className="w-full px-3 py-2 border border-gray-300 dark:border-gray-600 rounded-md shadow-sm focus:outline-none focus:ring-2 focus:ring-purple-500 focus:border-purple-500 dark:bg-gray-700 dark:text-white"
                  autoFocus
                />
              </div>
              
              <div className="flex justify-end space-x-3">
                <button
                  onClick={cancelTaskSave}
                  className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 bg-gray-100 dark:bg-gray-600 hover:bg-gray-200 dark:hover:bg-gray-500 rounded-md transition-colors"
                >
                  취소
                </button>
                <button
                  onClick={() => {
                    if (taskSaveTargetMessageId) {
                      onSaveTask(taskName, taskSaveTargetMessageId);
                    }
                  }}
                  disabled={!taskName.trim()}
                  className="px-4 py-2 text-sm font-medium text-white bg-purple-600 hover:bg-purple-700 disabled:bg-gray-400 disabled:cursor-not-allowed rounded-md transition-colors"
                >
                  저장
                </button>
              </div>
            </div>
          </div>
        </div>
      )}
    </div>
    </ClientOnly>
  );
} 

// 메인 컴포넌트를 Suspense로 감싸기 - 개선된 버전
export default function CodePage() {
  return (
    <Suspense fallback={
      <div className="flex items-center justify-center h-screen">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-2 text-gray-600 dark:text-gray-400">코드 모드를 로딩하고 있습니다...</p>
        </div>
      </div>
    }>
      <ChatPageContent />
    </Suspense>
  );
}