'use client';

import React, { useState, useEffect, useRef, useCallback, useMemo, Suspense } from 'react';
import { flushSync } from 'react-dom';
import { useRouter, useSearchParams } from 'next/navigation';
import { externalApiClient } from '@/lib/apiClient';
import { DocumentRenderer } from '@/components/DocumentRenderer';
import { useAuth } from '@/hooks/useAuth';
import { getAuthHeaders } from '@/utils/api';
import ClientOnly from '@/components/ClientOnly';
import { PdfLoader, PdfHighlighter, Tip, Highlight, Popup } from "react-pdf-highlighter";
import dynamic from "next/dynamic";
import PdfHighlightViewer from "@/components/PdfHighlightViewer";
import PdfIframeViewer from "@/components/PdfIframeViewer"; // 새로 만든 컴포넌트 import
import { FileText, Terminal, MessageCircle, X, Settings } from 'lucide-react';
import { Button } from "@/components/ui/button";
import { BackgroundJobStatus } from '@/components/BackgroundJobStatus';
import type { BackgroundJob } from '@/types/support';
import TableSchemaEditor from '@/components/TableSchemaEditor';
import { getApiServerUrl, getReportServerUrl } from '@/config/serverConfig';

interface RagFile {
  name: string;
  size: number;
  modified?: string;
  type?: string;
  status: 'uploading' | 'processing' | 'completed' | 'error';
  progress?: number;
  error?: string;
}

// 프로바이더 관련 인터페이스들 (agent 페이지에서 가져옴)
interface ProviderData {
  name: string;
  requiresApiKey: boolean;
  apiKeyConfigured: boolean;
  isDynamic: boolean;
  isAvailable: boolean;
  models: Model[];
}

interface Model {
  id: string;
  name: string;
  size?: number;
}

interface AvailableProvider {
  key: string;
  name: string;
  requiresApiKey: boolean;
  apiKeyConfigured: boolean;
  isDynamic: boolean;
  models?: Model[];
}



// 문서(리포트) 메타데이터 타입
interface ReportMetadata {
  uuid: string;
  project_hash: string;
  executive_summary: string;
  timestamp: string;
  output_file_path: string;
  status: 'processing' | 'completed' | 'failed';
  source: 'cache' | 'output';
  file_name?: string;
}

// 문서 섹션 타입 정의
interface DocumentSection {
  id: string;
  title: string;
  content: string;
  startLine?: number;
  endLine?: number;
}

interface TemplateOption {
  name: string;
  path: string;
  description?: string;
  metadata?: any;
  sections?: number;
  lastModified?: string;
  value?: string;
  // 기타 필요한 필드만 유지
}

// 완료된 문서의 파일명을 깔끔한 제목으로 변환하는 함수
const cleanFileName = (fileName: string): string => {
  if (!fileName) return fileName;

  // 파일 확장자 제거
  let cleanName = fileName.replace(/\.(pdf|hwpx?|docx?|pptx?)$/i, '');

  // 날짜/시간 패턴 제거 (예: _20250702_120713, _20241225_143022)
  cleanName = cleanName.replace(/_\d{8}_\d{6}$/, '');

  // bizplan_ 접두사 제거
  cleanName = cleanName.replace(/^bizplan_/, '');

  // business_plan_ 접두사 제거
  cleanName = cleanName.replace(/^business_plan_/, '');

  // 언더스코어를 공백으로 변경
  cleanName = cleanName.replace(/_/g, ' ');

  // 첫 글자 대문자로 변경
  cleanName = cleanName.charAt(0).toUpperCase() + cleanName.slice(1);

  return cleanName || fileName; // 변환 결과가 빈 문자열이면 원본 반환
};

/**
 * 사용자 요청사항에서 메타데이터를 제거하는 함수
 */
const removeMetadataFromContent = (content: string): string => {
  if (!content) return content;

  // 메타데이터 패턴들을 제거
  const cleanContent = content
    .replace(/^#\s*template:\s*.*$/gm, '') // # template: xxx
    .replace(/^#\s*pagebreak:\s*.*$/gm, '') // # pagebreak: xxx
    .replace(/^#\s*prompt:\s*$/gm, '') // # prompt: (빈 라인)
    .trim();

  return cleanContent;
};

// 최적화된 Textarea 컴포넌트 (메모이제이션)
const OptimizedTextarea = React.memo(({
  value,
  onChange,
  placeholder,
  className,
  disabled,
  onDragEnter,
  onDragLeave,
  onDragOver,
  onDrop,
  isDragOver
}: {
  value: string;
  onChange: (e: React.ChangeEvent<HTMLTextAreaElement>) => void;
  placeholder: string;
  className: string;
  disabled: boolean;
  onDragEnter: (e: React.DragEvent) => void;
  onDragLeave: (e: React.DragEvent) => void;
  onDragOver: (e: React.DragEvent) => void;
  onDrop: (e: React.DragEvent) => void;
  isDragOver: boolean;
}) => {
  return (
    <div className="relative">
      <div
        className={`relative ${isDragOver ? 'ring-2 ring-blue-500 ring-opacity-50' : ''}`}
        onDragEnter={onDragEnter}
        onDragLeave={onDragLeave}
        onDragOver={onDragOver}
        onDrop={onDrop}
      >
        <textarea
          value={value}
          onChange={onChange}
          placeholder={placeholder}
          className={className}
          disabled={disabled}
        />
        {isDragOver && (
          <div className="absolute inset-0 flex items-center justify-center bg-blue-100 dark:bg-blue-900/30 border-2 border-dashed border-blue-400 rounded-lg pointer-events-none">
            <div className="text-center">
              <svg className="w-8 h-8 mx-auto mb-2 text-blue-500" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
              </svg>
              <p className="text-sm font-medium text-blue-700 dark:text-blue-300">텍스트 파일을 놓으세요</p>
              <p className="text-xs text-blue-600 dark:text-blue-400">.txt, .md, .csv, .json 등</p>
            </div>
          </div>
        )}
      </div>
    </div>
  );
});

OptimizedTextarea.displayName = 'OptimizedTextarea';

// 청크 기반 무한 스크롤 문서 렌더러 (극강 성능 최적화)
const InfiniteScrollDocumentRenderer = React.memo(({ content }: { content: string }) => {
  const [loadedChunks, setLoadedChunks] = useState<number>(1); // 처음에는 첫 번째 청크만
  const [isLoading, setIsLoading] = useState(false);
  const containerRef = useRef<HTMLDivElement>(null);
  const loadTriggerRef = useRef<HTMLDivElement>(null);

  // 문서를 청크로 분할 (약 2000자 또는 20개 문단씩)
  const chunkSize = 2000;
  const chunks = useMemo(() => {
    if (!content) return [];

    // 문단 기준으로 우선 분할 시도
    const paragraphs = content.split(/\n\s*\n/);
    const result = [];
    let currentChunk = '';

    for (const paragraph of paragraphs) {
      if (currentChunk.length + paragraph.length > chunkSize && currentChunk.length > 0) {
        result.push(currentChunk.trim());
        currentChunk = paragraph;
      } else {
        currentChunk += (currentChunk ? '\n\n' : '') + paragraph;
      }
    }

    if (currentChunk.trim()) {
      result.push(currentChunk.trim());
    }

    // 빈 청크 제거 및 최소 크기 보장
    return result.filter(chunk => chunk.length > 0);
  }, [content]);

  const totalChunks = chunks.length;
  const hasMoreContent = loadedChunks < totalChunks;

  // Intersection Observer로 스크롤 감지
  useEffect(() => {
    if (!hasMoreContent) return;

    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting && !isLoading) {
          setIsLoading(true);

          // 자연스러운 로딩을 위해 약간의 지연
          setTimeout(() => {
            setLoadedChunks(prev => Math.min(prev + 1, totalChunks));
            setIsLoading(false);
          }, 100);
        }
      },
      {
        threshold: 0.1,
        rootMargin: '50px' // 50px 전에 미리 로딩
      }
    );

    if (loadTriggerRef.current) {
      observer.observe(loadTriggerRef.current);
    }

    return () => observer.disconnect();
  }, [hasMoreContent, isLoading, totalChunks]);

  // 작은 문서는 전체 렌더링
  if (content.length <= chunkSize) {
    return <DocumentRenderer content={content} />;
  }

  return (
    <div ref={containerRef} className="space-y-4">
      {/* 로드된 청크들 렌더링 */}
      {chunks.slice(0, loadedChunks).map((chunk, index) => (
        <div key={index} className="chunk-content">
          <DocumentRenderer content={chunk} />
        </div>
      ))}

      {/* 로딩 트리거 & 상태 */}
      {hasMoreContent && (
        <div ref={loadTriggerRef} className="py-8 text-center">
          {isLoading ? (
            <div className="flex items-center justify-center space-x-2">
              <div className="w-4 h-4 border-2 border-blue-500 border-t-transparent rounded-full animate-spin"></div>
              <span className="text-sm text-gray-500 dark:text-gray-400">
                더 많은 내용을 불러오는 중...
              </span>
            </div>
          ) : (
            <div className="text-sm text-gray-400 dark:text-gray-500">
              <div className="mb-2">📄 스크롤하여 더 많은 내용 보기</div>
              <div className="text-xs">
                ({loadedChunks}/{totalChunks} 청크 로드됨)
              </div>
            </div>
          )}
        </div>
      )}

      {/* 모든 내용 로드 완료 */}
      {!hasMoreContent && totalChunks > 1 && (
        <div className="py-4 text-center text-xs text-gray-400 dark:text-gray-500 border-t border-gray-200 dark:border-gray-700">
          ✅ 문서 전체 ({totalChunks}개 청크) 로드 완료
        </div>
      )}
    </div>
  );
});

InfiniteScrollDocumentRenderer.displayName = 'InfiniteScrollDocumentRenderer';

// 지연 로딩 이미지 컴포넌트
const LazyImage = React.memo(({ src, alt, className }: { src: string; alt: string; className: string }) => {
  const [isLoaded, setIsLoaded] = useState(false);
  const [isInView, setIsInView] = useState(false);
  const imgRef = useRef<HTMLImageElement>(null);

  useEffect(() => {
    const observer = new IntersectionObserver(
      ([entry]) => {
        if (entry.isIntersecting) {
          setIsInView(true);
          observer.disconnect();
        }
      },
      { threshold: 0.1 }
    );

    if (imgRef.current) {
      observer.observe(imgRef.current);
    }

    return () => observer.disconnect();
  }, []);

  if (!isInView) {
    return (
      <div
        ref={imgRef}
        className={`${className} bg-gray-200 dark:bg-gray-700 animate-pulse flex items-center justify-center`}
        style={{ minHeight: '200px' }}
      >
        <span className="text-gray-400">🖼️</span>
      </div>
    );
  }

  return (
    <img
      src={src}
      alt={alt}
      className={`${className} ${isLoaded ? 'opacity-100' : 'opacity-0'} transition-opacity duration-300`}
      onLoad={() => setIsLoaded(true)}
      onError={(e) => {
        console.error('이미지 로드 실패:', src);
        e.currentTarget.style.display = 'none';
      }}
    />
  );
});

LazyImage.displayName = 'LazyImage';

// 최적화된 문서 뷰어 컴포넌트 (가상화 + 지연 로딩)
const OptimizedDocumentViewer = React.memo(({
  activeTab,
  tabContents,
  streamingSections,
  streamingMessage,
  streamingProgress,
  sectionRefs,
  ThinkingMessage,
  DocumentRenderer,
  handleDownload
}: {
  activeTab: string | null;
  tabContents: any;
  streamingSections: any;
  streamingMessage: string;
  streamingProgress: number;
  sectionRefs: React.MutableRefObject<{ [key: string]: HTMLDivElement | null }>;
  ThinkingMessage: ({ sectionName }: { sectionName: string }) => React.ReactElement;
  DocumentRenderer: any;
  handleDownload: (filePath: string, fileName?: string) => Promise<void>;
}) => {
  if (!activeTab || !tabContents[activeTab]) return null;

  return (
    <>
      {/* 탭 내용 */}
      {tabContents[activeTab].type === 'streaming' ? (
        <div className="w-full flex-1 flex flex-col space-y-4 min-h-0">
          {/* 스트리밍 헤더 */}


          {/* 스트리밍 섹션들 */}
          <div className="space-y-4 w-full max-w-none overflow-visible">
            {Object.entries(streamingSections)
              .sort(([, a], [, b]) => ((a as any).animationDelay || 0) - ((b as any).animationDelay || 0)) // 애니메이션 순서로 정렬
              .map(([sectionName, section]: [string, any], index) => {
                console.log('🎨 섹션 렌더링:', {
                  sectionName,
                  title: section.title,
                  status: section.status,
                  isVisible: section.isVisible,
                  contentLength: section.content?.length || 0,
                  hasContent: !!(section.content && section.content.trim().length > 0),
                  animationDelay: section.animationDelay,
                  renderIndex: index
                });

                return (
                  <div
                    key={sectionName}
                    ref={(el) => { sectionRefs.current[sectionName] = el; }}
                    className={`p-4 rounded-lg border-l-4 w-full max-w-none overflow-visible transform transition-all duration-700 ease-out ${section.isVisible ? 'opacity-100 translate-y-0 scale-100' : 'opacity-0 translate-y-8 scale-95'
                      } ${section.status === 'completed'
                        ? 'bg-green-50 dark:bg-green-900/20 border-green-400 dark:border-green-500 shadow-sm'
                        : section.status === 'failed'
                          ? 'bg-red-50 dark:bg-red-900/20 border-red-400 dark:border-red-500 shadow-sm'
                          : section.status === 'thinking'
                            ? 'bg-blue-50 dark:bg-blue-900/20 border-blue-400 dark:border-blue-500 shadow-sm'
                            : 'bg-yellow-50 dark:bg-yellow-900/20 border-yellow-400 dark:border-yellow-500 shadow-sm'
                      }`}
                    style={{
                      transitionDelay: section.isVisible ? `${section.animationDelay}ms` : '0ms',
                      willChange: 'transform, opacity'
                    }}
                  >
                    <div className="flex items-center gap-2 mb-2">
                      <span className="text-lg">
                        {section.status === 'completed' ? '✅' :
                          section.status === 'failed' ? '❌' :
                            section.status === 'thinking' ? '🤔' : '⚡'}
                      </span>
                      <p className="font-semibold text-gray-900 dark:text-white">{section.title}</p>
                      <span className="text-xs text-gray-500 dark:text-gray-400">
                        {new Date(section.timestamp).toLocaleTimeString()}
                      </span>
                    </div>
                    <div className="max-w-none overflow-visible">
                      {section.status === 'thinking' ? (
                        <ThinkingMessage sectionName={sectionName} />
                      ) : (
                        <>
                          <div className="markdown-content text-sm mb-4">
                            <DocumentRenderer content={section.content} />
                          </div>

                          {/* 시각화 자료 */}
                          {section.visualization_paths && section.visualization_paths.length > 0 && (
                            <div className="w-full max-w-none overflow-visible mb-4">
                              <p className="text-sm font-semibold mb-2 text-gray-700 dark:text-gray-300 ml-4">📊 이미지</p>
                              <div className="grid grid-cols-1 md:grid-cols-2 gap-4 ml-4">
                                {section.visualization_paths
                                  .filter((imagePath: string) => imagePath.toLowerCase().endsWith('.png'))
                                  .map((imagePath: string, index: number) => (
                                    <div key={index} className="border rounded-lg p-2 bg-gray-50 dark:bg-gray-800">
                                      <img
                                        src={imagePath}
                                        alt={`${section.title} 시각화 자료 ${index + 1}`}
                                        className="w-full h-auto rounded"
                                        onError={(e) => {
                                          console.error('이미지 로드 실패:', imagePath);
                                          e.currentTarget.style.display = 'none';
                                        }}
                                      />
                                    </div>
                                  ))}
                              </div>
                            </div>
                          )}

                          {/* 테이블 데이터 */}
                          {section.table_data && (
                            <div className="w-full max-w-none overflow-visible mb-4">
                              <p className="text-sm font-semibold mb-2 text-gray-700 dark:text-gray-300 ml-4">📋 표</p>
                              <div className="overflow-x-auto ml-4">
                                <table className="min-w-full border-collapse border border-gray-300 dark:border-gray-600">
                                  <thead>
                                    <tr className="bg-gray-100 dark:bg-gray-700">
                                      {section.table_data.headers?.map((header: string, index: number) => (
                                        <th key={index} className="border border-gray-300 dark:border-gray-600 px-4 py-2 text-left text-sm font-medium text-gray-700 dark:text-gray-300">
                                          {header}
                                        </th>
                                      ))}
                                    </tr>
                                  </thead>
                                  <tbody>
                                    {section.table_data.rows?.map((row: any[], rowIndex: number) => (
                                      <tr key={rowIndex} className="hover:bg-gray-50 dark:hover:bg-gray-800">
                                        {row.map((cell: any, cellIndex: number) => (
                                          <td key={cellIndex} className="border border-gray-300 dark:border-gray-600 px-4 py-2 text-sm text-gray-700 dark:text-gray-300">
                                            {cell}
                                          </td>
                                        ))}
                                      </tr>
                                    ))}
                                  </tbody>
                                </table>
                              </div>
                            </div>
                          )}
                        </>
                      )}
                    </div>
                  </div>
                );
              })}
          </div>

          {/* 스트리밍 중 표시 */}
          {Object.keys(streamingSections).length === 0 && (
            <div className="text-center py-12 text-gray-500 dark:text-gray-400">
              <div className="text-4xl mb-4">📝</div>
              <p>문서 생성을 준비 중입니다...</p>
            </div>
          )}
        </div>
      ) : tabContents[activeTab].type === 'pdf' ? (
        <div className="w-full flex flex-col relative">
          <PdfIframeViewer filePath={tabContents[activeTab].filePath || ''} />
        </div>
      ) : tabContents[activeTab].type === 'hwpx' ? (
        <div className="w-full flex flex-col relative">
          <div className="mb-4 p-3 bg-green-50 dark:bg-green-900/20 border border-green-200 dark:border-green-800 rounded-lg">
            <div className="flex items-center gap-2 text-sm text-green-800 dark:text-green-200">
              <span>📝</span>
              <span className="font-medium">한글 문서</span>
              <span className="ml-auto text-xs bg-green-100 dark:bg-green-800 px-2 py-1 rounded">HWPX</span>
            </div>
          </div>
          <div className="w-full flex-1 flex flex-col items-center justify-center bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600" style={{ minHeight: 'calc(100vh - 200px)' }}>
            <div className="text-center p-8">
              <div className="text-6xl mb-4">📝</div>
              <h3 className="text-lg font-semibold mb-2 text-gray-700 dark:text-gray-300">한글 문서 (HWPX)</h3>
              <p className="text-gray-500 dark:text-gray-400 mb-4">브라우저에서 직접 미리보기를 지원하지 않습니다.</p>
              {tabContents[activeTab].filePath ? (
                <button
                  onClick={() => {
                    console.log('[HWPX 다운로드] 버튼 클릭:', {
                      filePath: tabContents[activeTab].filePath,
                      fileName: tabContents[activeTab].fileName,
                      activeTab,
                      tabContent: tabContents[activeTab]
                    });
                    handleDownload(tabContents[activeTab].filePath!, tabContents[activeTab].fileName);
                  }}
                  className="inline-flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                >
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                  파일 다운로드
                </button>
              ) : (
                <div>
                  <button
                    disabled
                    className="inline-flex items-center gap-2 px-4 py-2 bg-gray-400 text-white rounded-lg cursor-not-allowed"
                    title="파일 경로를 찾을 수 없습니다"
                  >
                    <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                    </svg>
                    파일을 찾을 수 없음
                  </button>
                  <p className="text-xs text-red-500 mt-2">
                    디버그: filePath = {JSON.stringify(tabContents[activeTab].filePath)}
                  </p>
                </div>
              )}
            </div>
          </div>
        </div>
      ) : tabContents[activeTab].type === 'docx' ? (
        <div className="w-full flex flex-col relative">
          <div className="mb-4 p-3 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg">
            <div className="flex items-center gap-2 text-sm text-blue-800 dark:text-blue-200">
              <span>📘</span>
              <span className="font-medium">Word 문서</span>
              <span className="ml-auto text-xs bg-blue-100 dark:bg-blue-800 px-2 py-1 rounded">DOCX</span>
            </div>
          </div>
          <div className="w-full flex-1 flex flex-col items-center justify-center bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600" style={{ minHeight: 'calc(100vh - 200px)' }}>
            <div className="text-center p-8">
              <div className="text-6xl mb-4">📘</div>
              <h3 className="text-lg font-semibold mb-2 text-gray-700 dark:text-gray-300">Word 문서 (DOCX)</h3>
              <p className="text-gray-500 dark:text-gray-400 mb-4">브라우저에서 직접 미리보기를 지원하지 않습니다.</p>
              {tabContents[activeTab].filePath ? (
                <button
                  onClick={() => {
                    console.log('[DOCX 다운로드] 버튼 클릭:', {
                      filePath: tabContents[activeTab].filePath,
                      fileName: tabContents[activeTab].fileName,
                      activeTab,
                      tabContent: tabContents[activeTab]
                    });
                    handleDownload(tabContents[activeTab].filePath!, tabContents[activeTab].fileName);
                  }}
                  className="inline-flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                >
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                  파일 다운로드
                </button>
              ) : (
                <button
                  disabled
                  className="inline-flex items-center gap-2 px-4 py-2 bg-gray-400 text-white rounded-lg cursor-not-allowed"
                  title="파일 경로를 찾을 수 없습니다"
                >
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                  파일을 찾을 수 없음
                </button>
              )}
            </div>
          </div>
        </div>
      ) : tabContents[activeTab].type === 'pptx' ? (
        <div className="w-full flex flex-col relative">
          <div className="mb-4 p-3 bg-orange-50 dark:bg-orange-900/20 border border-orange-200 dark:border-orange-800 rounded-lg">
            <div className="flex items-center gap-2 text-sm text-orange-800 dark:text-orange-200">
              <span>📊</span>
              <span className="font-medium">PowerPoint 문서</span>
              <span className="ml-auto text-xs bg-orange-100 dark:bg-orange-800 px-2 py-1 rounded">PPTX</span>
            </div>
          </div>
          <div className="w-full flex-1 flex flex-col items-center justify-center bg-gray-50 dark:bg-gray-800 rounded-lg border-2 border-dashed border-gray-300 dark:border-gray-600" style={{ minHeight: 'calc(100vh - 200px)' }}>
            <div className="text-center p-8">
              <div className="text-6xl mb-4">📊</div>
              <h3 className="text-lg font-semibold mb-2 text-gray-700 dark:text-gray-300">PowerPoint 문서 (PPTX)</h3>
              <p className="text-gray-500 dark:text-gray-400 mb-4">브라우저에서 직접 미리보기를 지원하지 않습니다.</p>
              {tabContents[activeTab].filePath ? (
                <button
                  onClick={() => {
                    console.log('[PPTX 다운로드] 버튼 클릭:', {
                      filePath: tabContents[activeTab].filePath,
                      fileName: tabContents[activeTab].fileName,
                      activeTab,
                      tabContent: tabContents[activeTab]
                    });
                    handleDownload(tabContents[activeTab].filePath!, tabContents[activeTab].fileName);
                  }}
                  className="inline-flex items-center gap-2 px-4 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 transition-colors"
                >
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                  파일 다운로드
                </button>
              ) : (
                <button
                  disabled
                  className="inline-flex items-center gap-2 px-4 py-2 bg-gray-400 text-white rounded-lg cursor-not-allowed"
                  title="파일 경로를 찾을 수 없습니다"
                >
                  <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                  </svg>
                  파일을 찾을 수 없음
                </button>
              )}
            </div>
          </div>
        </div>
      ) : (
        <div className="prose prose-sm dark:prose-invert max-w-none w-full">
          <InfiniteScrollDocumentRenderer content={tabContents[activeTab].content || ''} />
        </div>
      )}
    </>
  );
});

OptimizedDocumentViewer.displayName = 'OptimizedDocumentViewer';

// useSearchParams를 사용하는 컴포넌트를 분리
function DocumentPageContent() {
  const { user, isLoading: isAuthLoading, isAuthenticated, isInitialized } = useAuth();
  const [inputMessage, setInputMessage] = useState('');
  const [isLoading, setIsLoading] = useState(false);
  const [error, setError] = useState<string | null>(null);


  const [uploadedFiles, setUploadedFiles] = useState<File[]>([]);
  const [isUploading, setIsUploading] = useState(false);
  const [uploadingFiles, setUploadingFiles] = useState<Set<string>>(new Set());

  const [processingStatus, setProcessingStatus] = useState<string>('');

  // RAG 파일 브라우저 상태
  const [ragFiles, setRagFiles] = useState<RagFile[]>([]);
  const [isLoadingRagFiles, setIsLoadingRagFiles] = useState<boolean>(false);
  const [showRagBrowser, setShowRagBrowser] = useState<boolean>(true);
  const [isDragOverRag, setIsDragOverRag] = useState<boolean>(false);

  // 토스트 상태
  const [showToast, setShowToast] = useState<{ message: string; type: 'success' | 'error' } | null>(null);

  const fileInputRef = useRef<HTMLInputElement>(null);

  const router = useRouter();
  const searchParams = useSearchParams();

  const [rightPanelWidth, setRightPanelWidth] = useState(450); // 오른쪽 패널 폭 (px)
  const [isResizing, setIsResizing] = useState(false);
  const resizerRef = useRef<HTMLDivElement>(null);

  const [reports, setReports] = useState<ReportMetadata[]>([]);
  const [currentReportId, setCurrentReportId] = useState<string | null>(null);
  const [isLoadingReports, setIsLoadingReports] = useState<boolean>(true);
  const [reportTitles, setReportTitles] = useState<{ [uuid: string]: string }>({});
  const [activeJobs, setActiveJobs] = useState<Array<any>>([]);

  // 프로바이더 관련 상태 (agent 페이지와 동일)
  const [providers, setProviders] = useState<Record<string, any>>({});
  const [availableProviders, setAvailableProviders] = useState<any[]>([]);
  const [isLoadingProviders, setIsLoadingProviders] = useState(true);
  const [agentOptions, setAgentOptions] = useState({
    model: 'hamonize:latest',
    provider: 'ollama',
    temperature: 0.7,
    rag: false,
    web: false,
    ragSearchScope: 'personal' as 'personal' | 'shared' | 'all'
  });

  // 문서 템플릿 상태
  const [templates, setTemplates] = useState<TemplateOption[]>([]);

  // 선택된 템플릿, 프롬프트 예시 상태
  const [selectedTemplate, setSelectedTemplate] = useState<string>('');
  const [templateDesc, setTemplateDesc] = useState<string>('');

  // 파일유형(문서 형식) 상태
  const [fileFormat, setFileFormat] = useState<string>('pdf');

  // 수신 이메일 상태
  const [userEmail, setUserEmail] = useState<string>('');

  // 상태
  const [openTabs, setOpenTabs] = useState<{ uuid: string; title: string; status: string; }[]>([]);
  const [activeTab, setActiveTab] = useState<string | null>(null);
  const [tabContents, setTabContents] = useState<{ [uuid: string]: { type: string; filePath?: string; fileName?: string; fileExtension?: string; content?: string } }>({});

  // 1. 상태 추가
  const [deleteTarget, setDeleteTarget] = useState<ReportMetadata | null>(null);
  const [deleteAllTarget, setDeleteAllTarget] = useState<'processing' | 'completed' | null>(null);

  // 템플릿 업로드 관련 상태 추가
  const [isUploadingTemplate, setIsUploadingTemplate] = useState(false);
  const [templateUploadMessage, setTemplateUploadMessage] = useState<string | null>(null);
  const templateFileInputRef = useRef<HTMLInputElement>(null);

  // 테이블 스키마 에디터 관련 상태 추가
  const [showTableSchemaEditor, setShowTableSchemaEditor] = useState(false);
  const [currentTableSchema, setCurrentTableSchema] = useState<any>(null);


  const [isOpenMyDocs, setIsOpenMyDocs] = useState(true);
  const [isOpenReports, setIsOpenReports] = useState(true);
  const [isOpenTemplate, setIsOpenTemplate] = useState(true); // 템플릿 업로드: 펼쳐진 상태로 초기화
  const [isOpenSettings, setIsOpenSettings] = useState(false); // Settings: 닫힘

  // 텍스트 파일 드래그 앤 드롭 상태

  // 백그라운드 작업 관련 상태
  const [refreshTrigger, setRefreshTrigger] = useState(0);

  // 백그라운드 작업 완료 시 콜백
  const handleJobComplete = async (job: BackgroundJob) => {
    console.log('백그라운드 작업 완료:', job);

    if (job.type === 'file_upload') {
      // RAG 파일 목록 새로고침
      await loadRagFiles();
    }
  };

  // 백그라운드 파일 업로드 함수
  const handleFileUploadBackground = async (file: File) => {
    try {
      // 파일을 Base64로 변환
      const fileData = await new Promise<{
        name: string;
        size: number;
        type: string;
        content: string;
      }>((resolve, reject) => {
        const reader = new FileReader();
        reader.onload = () => {
          const result = reader.result as string;
          const base64Content = result.split(',')[1]; // data:type;base64, 제거
          resolve({
            name: file.name,
            size: file.size,
            type: file.type,
            content: base64Content
          });
        };
        reader.onerror = reject;
        reader.readAsDataURL(file);
      });

      const response = await fetch('/api/support/background-jobs', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        credentials: 'include',
        body: JSON.stringify({
          type: 'file_upload',
          userId: user?.username,
          filename: file.name,
          fileData: fileData,
          metadata: {
            pageType: 'document'
          }
        }),
      });

      if (response.ok) {
        const result = await response.json();
        if (result.success) {
          console.log(`파일 ${file.name} 백그라운드 업로드 시작:`, result.jobId);

          // 즉시 백그라운드 작업 상태 새로고침 트리거
          setRefreshTrigger(prev => prev + 1);

          // 업로드된 파일 목록에서 제거 (백그라운드에서 처리됨)
          setUploadedFiles(prev => prev.filter(f => f.name !== file.name));
        }
      }
    } catch (error) {
      console.error('백그라운드 작업 시작 실패:', error);
      setError(`${file.name} 업로드 작업을 시작할 수 없습니다.`);
    }
  };

  const [isDragOverTextarea, setIsDragOverTextarea] = useState(false);

  // 백그라운드 작업 관리 상태
  const [currentTaskId, setCurrentTaskId] = useState<string | null>(null);
  const [taskStartTime, setTaskStartTime] = useState<number | null>(null);
  const [isTaskCompleted, setIsTaskCompleted] = useState<boolean>(false);

  // 로그 뷰어 상태 추가
  const [logs, setLogs] = useState<string[]>([]);
  const [isLogOpen, setIsLogOpen] = useState(false);
  const [isLogPaused, setIsLogPaused] = useState(false);
  const logBoxRef = useRef<HTMLDivElement>(null);
  const logPollingIntervalRef = useRef<NodeJS.Timeout | null>(null);

  // 스트리밍 상태 - 탭 기반으로 계산
  const isStreamingActive = !!currentTaskId && !!taskStartTime &&
    activeTab && activeTab.startsWith('streaming-');
  const [streamingSections, setStreamingSections] = useState<{
    [sectionName: string]: {
      title: string;
      content: string;
      status: 'thinking' | 'processing' | 'completed' | 'failed';
      timestamp: string;
      // 실시간 렌더링을 위한 새 필드 추가
      isVisible: boolean;
      animationDelay: number;
      // 추가 데이터 필드
      chart_data?: any;
      table_data?: any;
      diagram_data?: any;
      flowchart_data?: any;
      gantt_data?: any;
      sequence_data?: any;
      visualization_paths?: string[];
    };
  }>({});
  const [streamingProgress, setStreamingProgress] = useState(0);

  // 타이핑 효과 제거됨 - 더 이상 사용하지 않음
  const [streamingMessage, setStreamingMessage] = useState('');
  const eventSourceRef = useRef<EventSource | null>(null);

  // 스크롤 관리용 ref
  const streamingContainerRef = useRef<HTMLDivElement>(null);
  const sectionRefs = useRef<{ [key: string]: HTMLDivElement | null }>({});

  // 순차적 애니메이션을 위한 큐 관리
  const [sectionQueue, setSectionQueue] = useState<string[]>([]);
  const [currentlyAnimating, setCurrentlyAnimating] = useState<string | null>(null);

  // 문서 수정 모드 상태 추가
  const [editMode, setEditMode] = useState<'create' | 'edit'>('create');
  const [editTargetDocument, setEditTargetDocument] = useState<ReportMetadata | null>(null);
  const [editTargetSection, setEditTargetSection] = useState<string | null>(null);
  const [documentSections, setDocumentSections] = useState<DocumentSection[]>([]);

  // 오른쪽 패널 탭 상태 추가
  const [rightPanelTab, setRightPanelTab] = useState<'create' | 'edit'>('create');

  // 각 탭별 독립적인 상태
  const [createInputMessage, setCreateInputMessage] = useState('');
  const [createSelectedTemplate, setCreateSelectedTemplate] = useState('');
  const [createFileFormat, setCreateFileFormat] = useState('pdf');
  const [createUserEmail, setCreateUserEmail] = useState('');

  const [editInputMessage, setEditInputMessage] = useState('');
  const [editFileFormat, setEditFileFormat] = useState('pdf');
  const [editUserEmail, setEditUserEmail] = useState('');
  const [editMethod, setEditMethod] = useState<'full' | 'section'>('full'); // 수정 방식 선택

  // 컨텍스트 메뉴 상태
  const [contextMenu, setContextMenu] = useState<{
    show: boolean;
    x: number;
    y: number;
    document: ReportMetadata | null;
  }>({ show: false, x: 0, y: 0, document: null });

  // 🔍 Clarification Modal 관련 상태 추가
  const [showClarificationModal, setShowClarificationModal] = useState(false);
  const [clarificationLoading, setClarificationLoading] = useState(false);
  const [clarificationData, setClarificationData] = useState<any>(null);
  const [clarificationResponses, setClarificationResponses] = useState<{ [key: string]: any }>({});
  const [clarificationStep, setClarificationStep] = useState<'analyzing' | 'questions' | 'todo' | 'confirm'>('analyzing');

  // 🤔 Think... 애니메이션 컴포넌트
  const ThinkingMessage = ({ sectionName }: { sectionName: string }) => {
    const [dots, setDots] = useState('');

    useEffect(() => {
      const interval = setInterval(() => {
        setDots(prev => {
          if (prev.length >= 3) return '';
          return prev + '.';
        });
      }, 500); // 0.5초마다 점 하나씩 추가

      return () => clearInterval(interval);
    }, []);

    return (
      <div className="text-sm text-gray-600 dark:text-gray-400 italic flex items-center gap-2">
        <div className="w-4 h-4 border-2 border-gray-400 border-t-transparent rounded-full animate-spin"></div>
        <span>Think{dots}</span>
      </div>
    );
  };

  // 🔄 자동 스크롤 함수
  const scrollToSection = (sectionName: string, behavior: 'smooth' | 'instant' = 'smooth') => {
    const sectionElement = sectionRefs.current[sectionName];
    const container = streamingContainerRef.current;

    if (sectionElement && container) {
      try {
        // 섹션이 현재 화면에 보이는지 확인
        const containerRect = container.getBoundingClientRect();
        const sectionRect = sectionElement.getBoundingClientRect();

        // 섹션이 컨테이너 영역 밖에 있거나 부분적으로만 보이는 경우 스크롤
        const isFullyVisible = sectionRect.top >= containerRect.top &&
          sectionRect.bottom <= containerRect.bottom;

        if (!isFullyVisible) {
          // 스크롤할 위치 계산 (섹션을 컨테이너 상단에서 약간 아래로 배치)
          const scrollTop = container.scrollTop + sectionRect.top - containerRect.top - 20;

          container.scrollTo({
            top: scrollTop,
            behavior: behavior
          });

          console.log(`📜 자동 스크롤: ${sectionName} 섹션으로 이동`);
        }
      } catch (error) {
        console.error(`스크롤 오류: ${sectionName}`, error);
      }
    }
  };

  // 🔄 페이지 로드 시 진행 중인 작업 복구
  useEffect(() => {
    const recoverActiveTask = async () => {
      try {
        // localStorage에서 활성 작업 정보 가져오기
        const savedTaskId = localStorage.getItem('airun_active_task_id');
        const savedTaskStartTime = localStorage.getItem('airun_active_task_start_time');

        if (savedTaskId) {
          console.log(`🔄 페이지 로드 시 활성 작업 발견: ${savedTaskId}`);

          // 작업 상태 확인
          try {
            const statusResponse = await externalApiClient.getReportStatus(savedTaskId);

            if (statusResponse.success) {
              const statusData = statusResponse.data || statusResponse;
              const status = statusData.status || statusData.state;

              // 작업이 아직 진행 중이면 복구
              if (status === 'processing' || status === 'queued') {
                console.log(`✅ 진행 중인 작업 복구: ${savedTaskId} (상태: ${status})`);

                setCurrentTaskId(savedTaskId);
                setTaskStartTime(savedTaskStartTime ? parseInt(savedTaskStartTime) : Date.now());

                // SSE 스트리밍만 재시작
                startSseStreaming(savedTaskId);

                // 상태 메시지 표시
                setProcessingStatus(`🔄 이전 작업을 복구했습니다. 문서 생성이 계속 진행 중입니다...`);

              } else if (status === 'completed') {
                console.log(`✅ 완료된 작업 정리: ${savedTaskId}`);

                // 완료된 작업 정리
                localStorage.removeItem('airun_active_task_id');
                localStorage.removeItem('airun_active_task_start_time');

                // 완료 상태 표시
                if (statusData.file_path) {
                  setProcessingStatus(`✅ 문서 생성이 완료되었습니다! 파일: ${statusData.file_path}`);
                } else {
                  setProcessingStatus(`✅ 문서 생성이 완료되었습니다!`);
                }

              } else {
                console.log(`❌ 작업 상태 확인 실패 또는 비정상 상태: ${status}`);
                // 비정상 상태는 정리
                localStorage.removeItem('airun_active_task_id');
                localStorage.removeItem('airun_active_task_start_time');
              }
            } else {
              console.log(`❌ 작업 상태 조회 실패: ${savedTaskId}`);
              // 조회 실패한 작업은 정리
              localStorage.removeItem('airun_active_task_id');
              localStorage.removeItem('airun_active_task_start_time');
            }
          } catch (error) {
            console.error('작업 상태 확인 중 오류:', error);
            // 오류 발생한 작업은 정리
            localStorage.removeItem('airun_active_task_id');
            localStorage.removeItem('airun_active_task_start_time');
          }
        }
      } catch (error) {
        console.error('활성 작업 복구 중 오류:', error);
      }
    };

    recoverActiveTask();
  }, []); // 페이지 로드 시 한 번만 실행

  // 🔄 작업 상태 저장 함수
  const saveActiveTask = (taskId: string, startTime: number) => {
    localStorage.setItem('airun_active_task_id', taskId);
    localStorage.setItem('airun_active_task_start_time', startTime.toString());
    // console.log(`💾 활성 작업 저장: ${taskId} (시작: ${new Date(startTime).toLocaleString()})`);
  };

  // 🔄 작업 상태 정리 함수
  const clearActiveTask = () => {
    setIsTaskCompleted(false); // 작업 완료 상태 초기화
    localStorage.removeItem('airun_active_task_id');
    localStorage.removeItem('airun_active_task_start_time');
    // console.log('🧹 활성 작업 정보 정리');
  };

  // 🔄 currentTaskId 변경 시 localStorage 업데이트
  useEffect(() => {
    if (currentTaskId && taskStartTime) {
      saveActiveTask(currentTaskId, taskStartTime);
    } else if (!currentTaskId) {
      clearActiveTask();
    }
  }, [currentTaskId, taskStartTime]);

  // SSE 스트리밍 시작
  const startSseStreaming = (taskId: string) => {
    console.log(`🔄 SSE 스트리밍 시작: ${taskId}`);

    // 이전 스트리밍 강제 정리 (안전한 연결을 위해)
    if (eventSourceRef.current) {
      console.log(`🔄 기존 SSE 연결 강제 종료`);
      eventSourceRef.current.close();
      eventSourceRef.current = null;
    }

    // 상태 초기화
    setStreamingSections({});
    setSectionQueue([]);
    setCurrentlyAnimating(null);
    setStreamingProgress(0);
    setStreamingMessage('문서 생성을 시작합니다...');
    setIsTaskCompleted(false); // 작업 완료 상태 초기화

    // EventSource 생성 (직접 보고서 서버 연결)
    // serverConfig에서 리포트 API 서버 주소 가져오기
    const reportApiServer = getReportServerUrl();
    
    const getStreamingUrl = (taskId: string) => {
      return `${reportApiServer}/stream/${taskId}`;
    };

    const eventSource = new EventSource(getStreamingUrl(taskId));
    eventSourceRef.current = eventSource;

    eventSource.onopen = () => {
      console.log('📡 SSE 연결 성공 - Job ID:', taskId);
      console.log('📡 SSE 요청 URL:', getStreamingUrl(taskId));
      setStreamingMessage('서버에 연결되었습니다...');
    };

    eventSource.onmessage = async (event) => {
      try {
        const data = JSON.parse(event.data);
        console.log('📨 SSE 메시지 수신:', data);
        console.log('🔍 현재 React 상태:', {
          streamingSectionsCount: Object.keys(streamingSections).length,
          isTaskCompleted,
          activeTab,
          currentTaskId,
          isStreamingActive: !!currentTaskId && !!taskStartTime && activeTab && activeTab.startsWith('streaming-')
        });

        // 빈 메시지나 잘못된 형식의 메시지 필터링
        if (!data || typeof data !== 'object') {
          console.warn('⚠️ 잘못된 SSE 메시지 형식:', event.data);
          return;
        }

        switch (data.type) {
          case 'connected':
            setStreamingMessage('서버에 연결되었습니다. 문서 생성을 시작합니다...');
            break;

          case 'progress':
            setStreamingProgress(data.progress || 0);
            setStreamingMessage(data.message || '');
            break;

          case 'section':
            console.log('📝 섹션 데이터 수신:', {
              section_name: data.section_name,
              title: data.title,
              hasContent: !!(data.content && data.content.trim().length > 0),
              contentLength: data.content?.length || 0,
              status: data.status
            });

            // 실시간 렌더링을 위해 flushSync 사용
            flushSync(() => {
              setStreamingSections(prev => {
                const isNewSection = !prev[data.section_name];
                const hasContent = data.content && data.content.trim().length > 0;
                const existingSectionsCount = Object.keys(prev).length;

                console.log('📝 섹션 업데이트:', {
                  section_name: data.section_name,
                  isNewSection,
                  hasContent,
                  existingSectionsCount
                });

                // 즉시 표시 - 실시간 효과보다 콘텐츠 가시성 우선
                let sectionStatus: 'thinking' | 'processing' | 'completed' | 'failed';
                let isVisible = true; // 모든 섹션 즉시 표시
                let animationDelay = 0;

                if (isNewSection) {
                  if (!hasContent) {
                    // 새 섹션이고 내용이 없으면 thinking 상태
                    sectionStatus = 'thinking';
                    animationDelay = existingSectionsCount * 200; // 200ms씩 지연
                  } else {
                    // 새 섹션이고 내용이 있으면 바로 completed
                    sectionStatus = 'completed';
                    animationDelay = existingSectionsCount * 200;
                  }
                } else {
                  // 기존 섹션 업데이트
                  sectionStatus = hasContent ? 'completed' : (data.status as any) || 'processing';
                  isVisible = true; // 항상 표시
                  animationDelay = prev[data.section_name]?.animationDelay || 0;
                }

                // 사용자 요청사항 섹션인 경우 메타데이터 제거
                const cleanContent = data.section_name === 'user_request' ||
                  data.section_name === 'executive_summary' ||
                  data.title?.includes('사용자 요청사항') ||
                  data.title?.includes('문서 생성 시작')
                  ? removeMetadataFromContent(data.content || '')
                  : data.content || '';

                const updatedSections = {
                  ...prev,
                  [data.section_name]: {
                    title: data.title || data.section_name || '제목 없음',
                    content: cleanContent || '',
                    status: sectionStatus || 'processing',
                    timestamp: data.timestamp || new Date().toISOString(),
                    isVisible: isVisible !== undefined ? isVisible : true,
                    animationDelay: animationDelay || 0,
                    // 추가 데이터 포함
                    chart_data: data.chart_data,
                    table_data: data.table_data,
                    diagram_data: data.diagram_data,
                    flowchart_data: data.flowchart_data,
                    gantt_data: data.gantt_data,
                    sequence_data: data.sequence_data,
                    visualization_paths: data.visualization_paths
                  }
                };

                console.log('📝 업데이트된 섹션들:', Object.keys(updatedSections));
                console.log('🎯 새로 생성된 섹션 상세:', {
                  sectionName: data.section_name,
                  title: data.title,
                  status: sectionStatus,
                  isVisible,
                  animationDelay,
                  contentLength: cleanContent?.length || 0
                });

                // 새 섹션이 추가되었고 내용이 있으면 즉시 스크롤
                if (isNewSection && hasContent) {
                  setTimeout(() => {
                    console.log('📜 자동 스크롤:', data.section_name, '섹션으로 이동');
                    scrollToSection(data.section_name, 'smooth');
                  }, animationDelay + 100); // 애니메이션 완료 후 스크롤
                }

                return updatedSections;
              });
            });

            // 진행률과 메시지도 즉시 업데이트
            flushSync(() => {
              setStreamingProgress(data.progress || 0);
              setStreamingMessage(data.title || '');
            });
            break;

          case 'status':
            // 상태 업데이트 처리
            if (data.status === 'completed') {
              // 완료 상태 처리 - 기존 'completed' 케이스와 동일한 로직
              console.log('📡 SSE에서 작업 완료 신호 수신 (status)');

              // 작업 완료 상태 설정 (재연결 방지)
              setIsTaskCompleted(true);

              // 현재 스트리밍 탭 ID
              const currentStreamingTabId = `streaming-${currentTaskId}`;

              // 문서 목록 새로고침
              if (isAuthenticated) {
                await fetchReports();
                await fetchActiveJobs();
              }

              // 새로 생성된 완료 문서 찾기 (약간의 지연 후)
              setTimeout(async () => {
                try {
                  // 최신 문서 목록을 직접 API에서 가져와서 새로운 완료 문서 확인
                  const headers = getAuthHeaders();
                  const userId = user?.username;
                  const queryParams = userId ? `?action=list&userId=${encodeURIComponent(userId)}` : '?action=list';
                  
                  const res = await fetch(`/api/document${queryParams}`, {
                    method: 'GET',
                    headers
                  });
                  
                  if (res.ok) {
                    const data = await res.json();
                    if (data.success && Array.isArray(data.files)) {
                      const newCompletedReport = data.files.find((report: any) =>
                        report.status === 'completed' &&
                        new Date(report.timestamp).getTime() > (taskStartTime || 0)
                      );

                      if (newCompletedReport) {
                        console.log(`[완료처리] 새로 완료된 문서 발견: ${newCompletedReport.uuid}`);

                        // 스트리밍 탭을 완료된 문서 탭으로 교체
                        const displayTitle = reportTitles[newCompletedReport.uuid] || newCompletedReport.file_name || newCompletedReport.project_hash;

                        setOpenTabs(prev => prev.map(tab =>
                          tab.uuid === currentStreamingTabId
                            ? { uuid: newCompletedReport.uuid, title: displayTitle, status: 'completed' }
                            : tab
                        ));

                        // 완료된 문서 내용 로드
                        try {
                          const headers = getAuthHeaders();
                          console.log(`[완료처리] 문서 내용 로드 시작: ${newCompletedReport.uuid}`);
                          const res = await fetch(`/api/document?action=view&uuid=${newCompletedReport.uuid}`, {
                            method: 'GET',
                            headers
                          });

                          if (res.ok) {
                            const documentData = await res.json();
                            console.log(`[완료처리] 문서 API 응답:`, documentData);

                            // API 응답 구조를 tabContents에 맞게 변환
                            const tabContent = {
                              type: documentData.type,
                              filePath: documentData.filePath,
                              fileName: documentData.fileName,
                              fileExtension: documentData.fileExtension,
                              content: documentData.content // inprogress 문서의 경우
                            };

                            // 탭 내용 업데이트
                            setTabContents(prev => {
                              const newContents = { ...prev };
                              delete newContents[currentStreamingTabId]; // 스트리밍 탭 내용 제거
                              newContents[newCompletedReport.uuid] = tabContent; // 완료된 문서 내용 추가
                              console.log(`[완료처리] 탭 내용 업데이트 완료. 새로운 내용:`, newContents[newCompletedReport.uuid]);
                              return newContents;
                            });
                            console.log(`[완료처리] 완료된 문서 내용 로드 완료: ${newCompletedReport.uuid}`);
                          } else {
                            const errorText = await res.text();
                            console.error(`[완료처리] 문서 내용 로드 실패: ${res.status} ${res.statusText}`, errorText);
                            // 내용 로드 실패 시 스트리밍 탭 내용만 제거
                            setTabContents(prev => {
                              const newContents = { ...prev };
                              delete newContents[currentStreamingTabId];
                              return newContents;
                            });
                          }
                        } catch (error) {
                          console.error(`[완료처리] 문서 내용 로드 중 오류:`, error);
                          // 오류 시 스트리밍 탭 내용만 제거
                          setTabContents(prev => {
                            const newContents = { ...prev };
                            delete newContents[currentStreamingTabId];
                            return newContents;
                          });
                        }

                        // 새 완료 문서로 활성 탭 전환
                        setActiveTab(newCompletedReport.uuid);

                        console.log(`[완료처리] 스트리밍 탭을 완료된 문서 탭으로 교체: ${currentStreamingTabId} → ${newCompletedReport.uuid}`);
                      } else {
                        // 완료된 문서를 찾지 못한 경우, 스트리밍 탭만 제거
                        console.log(`[완료처리] 완료된 문서를 찾지 못함, 스트리밍 탭 제거`);
                        setOpenTabs(prev => prev.filter(tab => tab.uuid !== currentStreamingTabId));
                        setTabContents(prev => {
                          const newContents = { ...prev };
                          delete newContents[currentStreamingTabId];
                          return newContents;
                        });

                        // 다른 탭으로 전환하거나 빈 상태로
                        const remainingTabs = openTabs.filter(tab => tab.uuid !== currentStreamingTabId);
                        setActiveTab(remainingTabs.length > 0 ? remainingTabs[0].uuid : null);
                      }
                    }
                  }
                } catch (error) {
                  console.error('[완료처리] 완료 문서 처리 중 오류:', error);
                }
              }, 2000); // 2초 후 처리

              // SSE에서 완료 신호를 받았을 때만 실제로 작업 완료 처리
              setProcessingStatus('✅ 문서 생성이 완료되었습니다!');
              setCurrentTaskId(null);
              setTaskStartTime(null);

              // 스트리밍 관련 상태 정리
              setStreamingSections({});
              setSectionQueue([]);
              setCurrentlyAnimating(null);
              setStreamingProgress(0);
              setStreamingMessage('');

              // 완료 처리 후 서버가 자연스럽게 연결을 닫도록 대기
              // 오류 핸들러를 조용한 핸들러로 교체
              if (eventSourceRef.current) {
                eventSourceRef.current.onerror = () => {
                  console.log('📡 SSE 연결 정상 종료 (서버에서 닫음)');
                  stopSseStreaming(true);
                };
              }

              // 10초 후 메시지 정리
              setTimeout(() => {
                setProcessingStatus('');
              }, 10000);
            } else {
              // 진행 중 상태 업데이트
              setStreamingProgress(data.progress || 0);
              setStreamingMessage(data.message || '');
            }
            break;

          case 'heartbeat':
            // 연결 유지용 heartbeat
            break;

          case 'error':
            // Job ID가 에러 메시지로 잘못 전달된 경우 필터링
            if (data.message && typeof data.message === 'string') {
              const trimmedMessage = data.message.replace(/['"]/g, '').trim();

              // Job ID가 포함된 오류는 완료된 작업에 대한 SSE 연결이므로 중단
              if (trimmedMessage.includes(currentTaskId) || trimmedMessage === currentTaskId) {
                console.log('🚫 완료된 작업에 대한 SSE 연결 감지 - 연결을 중단합니다:', trimmedMessage);
                setIsTaskCompleted(true);
                stopSseStreaming(true);

                // 스트리밍 탭을 제거하고 일반 탭으로 전환
                const currentStreamingTabId = `streaming-${currentTaskId}`;
                setOpenTabs(prev => prev.filter(tab => tab.uuid !== currentStreamingTabId));
                setTabContents(prev => {
                  const newContents = { ...prev };
                  delete newContents[currentStreamingTabId];
                  return newContents;
                });

                // 다른 탭으로 전환
                const remainingTabs = openTabs.filter(tab => tab.uuid !== currentStreamingTabId);
                if (remainingTabs.length > 0) {
                  setActiveTab(remainingTabs[0].uuid);
                } else {
                  setActiveTab('create');
                }
                return;
              }

              // Job ID 형태의 메시지는 무시 (숫자만 있는 경우)
              if (/^\d+$/.test(trimmedMessage)) {
                console.warn('⚠️ Job ID가 에러 메시지로 잘못 전달됨:', data.message, '- 무시함');
                return;
              }

              // 실제 의미있는 에러만 처리
              if (trimmedMessage.length > 0) {
                console.error('SSE 오류:', data.message);
                setStreamingMessage(`오류: ${trimmedMessage}`);
              }
            }
            break;

          case 'close':
            console.log('📡 SSE 스트림 종료');
            stopSseStreaming(true); // 정상적인 종료임을 표시
            break;

          case 'completed':
            console.log('📡 SSE에서 작업 완료 신호 수신');

            // 작업 완료 상태 설정 (재연결 방지)
            setIsTaskCompleted(true);

            // 현재 스트리밍 탭 ID
            const currentStreamingTabId = `streaming-${currentTaskId}`;

            // 문서 목록 새로고침
            if (isAuthenticated) {
              await fetchReports();
              await fetchActiveJobs();
            }

            // 새로 생성된 완료 문서 찾기 (약간의 지연 후)
            setTimeout(async () => {
              try {
                // 최신 문서 목록을 다시 가져와서 새로운 완료 문서 확인
                const updatedReports = await fetchReports();
                const newCompletedReport = reports.find(report =>
                  report.status === 'completed' &&
                  new Date(report.timestamp).getTime() > (taskStartTime || 0)
                );

                if (newCompletedReport) {
                  console.log(`[완료처리] 새로 완료된 문서 발견: ${newCompletedReport.uuid}`);

                  // 스트리밍 탭을 완료된 문서 탭으로 교체
                  const displayTitle = reportTitles[newCompletedReport.uuid] || newCompletedReport.file_name || newCompletedReport.project_hash;

                  setOpenTabs(prev => prev.map(tab =>
                    tab.uuid === currentStreamingTabId
                      ? { uuid: newCompletedReport.uuid, title: displayTitle, status: 'completed' }
                      : tab
                  ));

                  // 완료된 문서 내용 로드
                  try {
                    const headers = getAuthHeaders();
                    console.log(`[완료처리] 문서 내용 로드 시작: ${newCompletedReport.uuid}`);
                    const res = await fetch(`/api/document?action=view&uuid=${newCompletedReport.uuid}`, {
                      method: 'GET',
                      headers
                    });

                    if (res.ok) {
                      const documentData = await res.json();
                      console.log(`[완료처리] 문서 API 응답:`, documentData);

                      // API 응답 구조를 tabContents에 맞게 변환
                      const tabContent = {
                        type: documentData.type,
                        filePath: documentData.filePath,
                        fileName: documentData.fileName,
                        fileExtension: documentData.fileExtension,
                        content: documentData.content // inprogress 문서의 경우
                      };

                      // 탭 내용 업데이트
                      setTabContents(prev => {
                        const newContents = { ...prev };
                        delete newContents[currentStreamingTabId]; // 스트리밍 탭 내용 제거
                        newContents[newCompletedReport.uuid] = tabContent; // 완료된 문서 내용 추가
                        console.log(`[완료처리] 탭 내용 업데이트 완료. 새로운 내용:`, newContents[newCompletedReport.uuid]);
                        return newContents;
                      });
                      console.log(`[완료처리] 완료된 문서 내용 로드 완료: ${newCompletedReport.uuid}`);
                    } else {
                      const errorText = await res.text();
                      console.error(`[완료처리] 문서 내용 로드 실패: ${res.status} ${res.statusText}`, errorText);
                      // 내용 로드 실패 시 스트리밍 탭 내용만 제거
                      setTabContents(prev => {
                        const newContents = { ...prev };
                        delete newContents[currentStreamingTabId];
                        return newContents;
                      });
                    }
                  } catch (error) {
                    console.error(`[완료처리] 문서 내용 로드 중 오류:`, error);
                    // 오류 시 스트리밍 탭 내용만 제거
                    setTabContents(prev => {
                      const newContents = { ...prev };
                      delete newContents[currentStreamingTabId];
                      return newContents;
                    });
                  }

                  // 새 완료 문서로 활성 탭 전환
                  setActiveTab(newCompletedReport.uuid);

                  console.log(`[완료처리] 스트리밍 탭을 완료된 문서 탭으로 교체: ${currentStreamingTabId} → ${newCompletedReport.uuid}`);
                } else {
                  // 완료된 문서를 찾지 못한 경우, 스트리밍 탭만 제거
                  console.log(`[완료처리] 완료된 문서를 찾지 못함, 스트리밍 탭 제거`);
                  setOpenTabs(prev => prev.filter(tab => tab.uuid !== currentStreamingTabId));
                  setTabContents(prev => {
                    const newContents = { ...prev };
                    delete newContents[currentStreamingTabId];
                    return newContents;
                  });

                  // 다른 탭으로 전환하거나 빈 상태로
                  const remainingTabs = openTabs.filter(tab => tab.uuid !== currentStreamingTabId);
                  setActiveTab(remainingTabs.length > 0 ? remainingTabs[0].uuid : null);
                }
              } catch (error) {
                console.error('[완료처리] 완료 문서 처리 중 오류:', error);
              }
            }, 2000); // 2초 후 처리

            // SSE에서 완료 신호를 받았을 때만 실제로 작업 완료 처리
            setProcessingStatus('✅ 문서 생성이 완료되었습니다!');
            setCurrentTaskId(null);
            setTaskStartTime(null);

            // 스트리밍 관련 상태 정리
            setStreamingSections({});
            setSectionQueue([]);
            setCurrentlyAnimating(null);
            setStreamingProgress(0);
            setStreamingMessage('');

            // 완료 처리 후 서버가 자연스럽게 연결을 닫도록 대기
            // 오류 핸들러를 조용한 핸들러로 교체
            if (eventSourceRef.current) {
              eventSourceRef.current.onerror = () => {
                console.log('📡 SSE 연결 정상 종료 (서버에서 닫음)');
                stopSseStreaming(true);
              };
            }

            // 10초 후 메시지 정리
            setTimeout(() => {
              setProcessingStatus('');
            }, 10000);
            break;
        }
      } catch (error) {
        console.error('SSE 메시지 파싱 오류:', error);
        console.error('원본 데이터:', event.data);

        // 빈 메시지나 잘못된 형식의 메시지 처리
        if (event.data && typeof event.data === 'string') {
          const cleanData = event.data.replace(/['\"]/g, '').trim();

          // Job ID 형태의 메시지는 무시 (숫자만 있는 경우)
          if (!/^\d+$/.test(cleanData) && cleanData.length > 0) {
            setStreamingMessage(`서버 메시지: ${cleanData}`);
          }
        }
      }
    };

    eventSource.onerror = (error) => {
      // 작업이 완료된 상태에서는 SSE 연결 종료를 정상 상황으로 처리
      if (isTaskCompleted) {
        console.log('📡 작업 완료로 인한 SSE 연결 종료 (정상)');
        setStreamingMessage('작업이 완료되었습니다.');
        return;
      }

      // EventSource readyState 확인
      const state = eventSourceRef.current?.readyState;
      const stateText = state === 0 ? 'CONNECTING' : state === 1 ? 'OPEN' : 'CLOSED';

      // 상세한 오류 정보 로깅 (빈 객체 오류 방지)
      if (error && typeof error === 'object' && Object.keys(error).length > 0) {
        console.warn('📡 SSE 연결 오류 (상세):', error, `상태: ${stateText}, TaskID: ${taskId.slice(-8)}`);
      } else {
        console.log('📡 SSE 연결 상태 변경:', `${stateText} (${state}), TaskID: ${taskId.slice(-8)}`);
      }

      // 현재 작업과 다른 작업에 대한 SSE 연결인 경우 중단
      if (currentTaskId !== taskId) {
        console.log(`📡 다른 작업의 SSE 연결 감지 - 현재: ${currentTaskId?.slice(-8)}, 요청: ${taskId.slice(-8)} - 연결 중단`);
        if (eventSourceRef.current) {
          eventSourceRef.current.close();
          eventSourceRef.current = null;
        }
        return;
      }

      // CLOSED 상태일 때만 재연결 시도 (현재 작업에 대해서만)
      if (state === EventSource.CLOSED && currentTaskId === taskId) {
        setStreamingMessage('연결이 끊어졌습니다. 재연결을 시도합니다...');

        // 3초 후 자동 재연결 시도 (작업이 완료되지 않고 현재 작업인 경우에만)
        setTimeout(() => {
          if (!isTaskCompleted && currentTaskId === taskId && eventSourceRef.current?.readyState === EventSource.CLOSED) {
            console.log(`📡 SSE 재연결 시도: ${taskId.slice(-8)}`);
            startSseStreaming(taskId);
          }
        }, 3000); // 5초에서 3초로 단축
      }
    };
  };

  // SSE 스트리밍 중단
  const stopSseStreaming = (isNormalClose = false) => {
    if (eventSourceRef.current) {
      if (isNormalClose) {
        console.log('📡 SSE 스트리밍 정상 종료');
        // 정상적인 종료 시에는 오류 핸들러를 먼저 제거
        eventSourceRef.current.onerror = null;
      } else {
        console.log('📡 SSE 스트리밍 중단');
      }
      eventSourceRef.current.close();
      eventSourceRef.current = null;
    }

    // 애니메이션 상태 정리
    if (!isNormalClose) {
      setSectionQueue([]);
      setCurrentlyAnimating(null);
    }
  };

  // 백그라운드 작업 중단 함수 (서버에서 실제 작업 중단)
  const stopBackgroundTask = async () => {
    const taskIdForLog = currentTaskId?.slice(-8);

    if (currentTaskId) {
      try {
        // 서버에서 실제 작업 중단 (프록시 경로 사용)
        const response = await fetch(`/report/job/${currentTaskId}`, {
          method: 'DELETE'
        });

        const result = await response.json();

        if (result.success) {
          console.log(`[작업중단] 서버에서 작업이 중단되었습니다 - Job ID: ${taskIdForLog}`);
          setProcessingStatus(`🛑 작업이 중단되었습니다: ${taskIdForLog}`);
        } else {
          console.error(`[작업중단] 서버 작업 중단 실패 - Job ID: ${taskIdForLog}`, result);
          setProcessingStatus(`⚠️ 작업 중단 실패: ${taskIdForLog}`);
        }
      } catch (error) {
        console.error(`[작업중단] 작업 중단 중 오류 발생 - Job ID: ${taskIdForLog}`, error);
        setProcessingStatus(`❌ 작업 중단 중 오류 발생: ${taskIdForLog}`);
      }
    }

    // 클라이언트 측 정리
    setCurrentTaskId(null);
    setTaskStartTime(null);
    setIsTaskCompleted(false); // 작업 완료 상태 초기화

    // 스트리밍 관련 상태 정리
    setStreamingSections({});
    setSectionQueue([]);
    setCurrentlyAnimating(null);
    setStreamingProgress(0);
    setStreamingMessage('');

    stopSseStreaming(true); // 정상적인 중단임을 표시

    // 3초 후 상태 메시지 정리
    setTimeout(() => {
      setProcessingStatus('');
    }, 3000);

    console.log(`[작업중단] 클라이언트 측 정리 완료 - Job ID: ${taskIdForLog}`);
  };

  // 컴포넌트 언마운트 시 정리
  useEffect(() => {
    return () => {
      if (logPollingIntervalRef.current) {
        clearInterval(logPollingIntervalRef.current);
        // console.log('컴포넌트 언마운트로 인해 로그 폴링을 정리했습니다.');
      }
      stopSseStreaming(true); // 정상적인 언마운트임을 표시
      // console.log('컴포넌트 언마운트로 인해 SSE 스트리밍을 정리했습니다.');
    };
  }, []);

  // 로그 폴링 useEffect 추가
  useEffect(() => {
    if (!isLogOpen) {
      // 로그 창이 닫히면 폴링 중단
      if (logPollingIntervalRef.current) {
        clearInterval(logPollingIntervalRef.current);
        logPollingIntervalRef.current = null;
      }
      return;
    }

    let mounted = true;

    const fetchLogs = async () => {
      if (isLogPaused) return;
      try {
        const res = await fetch('/api/logs?file=airun-report.log&lines=50');
        const data = await res.json();
        if (data.success && mounted) {
          setLogs(data.logs || []);
        }
      } catch (error) {
        console.error('로그 폴링 오류:', error);
      }
    };

    // 즉시 한 번 실행
    fetchLogs();

    // 1.5초 간격으로 폴링 시작
    logPollingIntervalRef.current = setInterval(fetchLogs, 1500);

    return () => {
      mounted = false;
      if (logPollingIntervalRef.current) {
        clearInterval(logPollingIntervalRef.current);
        logPollingIntervalRef.current = null;
      }
    };
  }, [isLogOpen, isLogPaused]);



  // 로그 자동 스크롤
  useEffect(() => {
    if (isLogPaused) return; // 포커스/hover 중에는 자동 스크롤 중지
    if (logBoxRef.current) {
      logBoxRef.current.scrollTop = logBoxRef.current.scrollHeight;
    }
  }, [logs, isLogPaused]);

  // 템플릿 업로드 핸들러
  const handleTemplateFileChange = async (e: React.ChangeEvent<HTMLInputElement>) => {
    const file = e.target.files?.[0];
    if (!file) return;

    setIsUploadingTemplate(true);
    setTemplateUploadMessage(null);

    try {
      // externalApiClient를 사용하여 템플릿 업로드
      const response = await externalApiClient.uploadTemplateFile(file);

      if (response.success) {
        const templateName = response.data?.template?.name || file.name.replace(/(_templates)?\.json$/, '');
        setTemplateUploadMessage(`✅ 템플릿이 성공적으로 추가되었습니다. (${templateName})`);
        await fetchTemplates(); // 템플릿 목록 새로고침
        setTimeout(() => setTemplateUploadMessage(null), 3000);
      } else {
        setTemplateUploadMessage(`❌ 템플릿 업로드 실패: ${response.error?.message || '알 수 없는 오류'}`);
        setTimeout(() => setTemplateUploadMessage(null), 5000);
      }
    } catch (err: any) {
      console.error('템플릿 업로드 오류:', err);
      setTemplateUploadMessage(`❌ 업로드 중 오류 발생: ${err?.message || err}`);
      setTimeout(() => setTemplateUploadMessage(null), 5000);
    } finally {
      setIsUploadingTemplate(false);
      if (templateFileInputRef.current) templateFileInputRef.current.value = '';
    }
  };

  // 문서 목록 fetch 함수
  const fetchReports = useCallback(async () => {
    setIsLoadingReports(true);
    try {
      // 인증 헤더를 포함하여 요청 (사용자별 필터링)
      const headers = getAuthHeaders();
      // console.log('[문서목록] 요청 헤더:', headers);

      const userId = user?.username;
      const queryParams = userId ? `?action=list&userId=${encodeURIComponent(userId)}` : '?action=list';

      const res = await fetch(`/api/document${queryParams}`, {
        method: 'GET',
        headers
      });

      // console.log('[문서목록] 응답 상태:', res.status);

      if (!res.ok) {
        console.error('[문서목록] 응답 오류:', res.status, res.statusText);
        throw new Error(`HTTP ${res.status}: ${res.statusText}`);
      }

      const data = await res.json();
      // console.log('[문서목록] fetch 결과:', data);

      if (data.success && Array.isArray(data.files)) {
        setReports(data.files);

        // 각 문서의 제목 가져오기
        const titlePromises = data.files.map(async (report: ReportMetadata) => {
          // 진행중 문서(cache)는 business_info에서 제목 가져오기 시도, 실패하면 executive_summary 사용
          if (report.source === 'cache') {
            try {
              const response = await fetch(`/api/document?action=get-business-info&uuid=${report.uuid}`, {
                method: 'GET',
                headers
              });
              if (response.ok) {
                const businessData = await response.json();
                if (businessData.business_info && businessData.business_info.product_name) {
                  return { uuid: report.uuid, title: businessData.business_info.product_name };
                }
              }
            } catch (error) {
              console.warn(`제목 가져오기 실패 (${report.uuid}):`, error);
            }
            // business_info를 가져올 수 없으면 executive_summary 사용 (메타데이터 제거)
            const cleanSummary = report.executive_summary ? removeMetadataFromContent(report.executive_summary) : '';
            const fallbackTitle = cleanSummary ?
              cleanSummary.slice(0, 30) + (cleanSummary.length > 30 ? '...' : '') :
              report.project_hash;
            return { uuid: report.uuid, title: fallbackTitle };
          }

          // 완료된 문서(output)는 file_name을 깔끔하게 정리해서 사용
          const cleanTitle = report.file_name ? cleanFileName(report.file_name) : report.project_hash;
          return { uuid: report.uuid, title: cleanTitle };
        });

        const titles = await Promise.all(titlePromises);
        const titleMap = titles.reduce((acc, { uuid, title }) => {
          acc[uuid] = title;
          return acc;
        }, {} as { [uuid: string]: string });

        setReportTitles(titleMap);
      } else {
        setReports([]);
      }
    } catch (e) {
      console.error('[문서목록] fetch 오류:', e);
      setReports([]);
    } finally {
      setIsLoadingReports(false);
    }
  }, [user]);

  // 데이터베이스에서 진행 중인 작업 fetch 함수
  const fetchActiveJobs = useCallback(async () => {
    try {
      const headers = getAuthHeaders();
      const res = await fetch('/api/document?action=active-jobs', {
        method: 'GET',
        headers
      });

      if (!res.ok) {
        throw new Error(`HTTP error! status: ${res.status}`);
      }

      const data = await res.json();
      if (data.success && Array.isArray(data.jobs)) {
        // console.log(`[진행중작업] 데이터베이스에서 ${data.jobs.length}개의 진행 중인 작업 조회`);

        // 임시 작업과 데이터베이스 작업을 병합
        setActiveJobs(prev => {
          // 기존 임시 작업들 (id가 숫자인 것들)
          const tempJobs = prev.filter(job => typeof job.id === 'number');

          // 데이터베이스 작업들과 임시 작업들을 병합 (중복 제거)
          const dbJobIds = new Set(data.jobs.map((job: any) => job.job_id));
          const uniqueTempJobs = tempJobs.filter(job => !dbJobIds.has(job.job_id));

          // 임시 작업을 앞에, 데이터베이스 작업을 뒤에 배치
          const mergedJobs = [...uniqueTempJobs, ...data.jobs];

          // console.log(`[진행중작업] 임시 작업 ${uniqueTempJobs.length}개, DB 작업 ${data.jobs.length}개 병합`);
          return mergedJobs;
        });
      } else {
        console.warn('[진행중작업] 진행 중인 작업 조회 실패:', data.error);
        // 실패 시에도 임시 작업은 유지
        setActiveJobs(prev => prev.filter(job => typeof job.id === 'number'));
      }
    } catch (error) {
      console.error('[진행중작업] 진행 중인 작업 조회 오류:', error);
      // 오류 시에도 임시 작업은 유지
      setActiveJobs(prev => prev.filter(job => typeof job.id === 'number'));
    }
  }, [user]);

  // 백그라운드 작업 진행 시 문서 목록 주기적 새로고침
  useEffect(() => {
    if (!currentTaskId || !isAuthenticated) return;

    const intervalId = setInterval(async () => {
      try {
        console.log(`[문서목록새로고침] 백그라운드 작업 진행 중 - 문서 목록 새로고침 (Job ID: ${currentTaskId.slice(-8)})`);
        await fetchReports();
        await fetchActiveJobs(); // 진행 중인 작업도 함께 새로고침
      } catch (error) {
        console.error('문서 목록 새로고침 오류:', error);
      }
    }, 30000); // 30초마다 새로고침

    return () => {
      clearInterval(intervalId);
    };
  }, [currentTaskId, isAuthenticated, fetchReports, fetchActiveJobs]);

  // 문서 목록 API에서 불러오기 (사이드바) - 인증 상태 확인 후에만 실행
  useEffect(() => {
    if (isAuthenticated) {
      // console.log('[문서목록] 인증 완료 후 문서 목록 로드');
      fetchReports();
      fetchActiveJobs(); // 진행 중인 작업도 함께 로드
    }
  }, [fetchReports, fetchActiveJobs, isAuthenticated]);

  // 템플릿 목록 fetch 함수 분리
  const fetchTemplates = useCallback(async () => {
    try {
      const response = await externalApiClient.getReportTemplates();
      if (response.success && response.data?.data?.templates) {
        const templates = response.data.data.templates.map((opt: any) => ({
          ...opt,
          value: String(opt.name)
        }));
        setTemplates(templates);
      } else {
        console.warn('템플릿 목록을 불러오지 못했습니다:', response.error?.message);
        setTemplates([]);
      }
    } catch (error) {
      console.error('템플릿 목록 로드 오류:', error);
      setTemplates([]);
    }
  }, []);
  // 프로바이더 로딩 함수 (agent 페이지에서 가져옴)
  const loadProviders = useCallback(async () => {
    try {
      setIsLoadingProviders(true);
      
      // serverConfig에서 API URL 가져오기
      const apiUrl = getApiServerUrl();
      
      // 새로운 ApiClient 인스턴스를 동적으로 생성
      const dynamicApiClient = new (externalApiClient.constructor as any)(apiUrl);
      
      const availableResponse = await dynamicApiClient.getAvailableProviders();
      
      if (availableResponse.success && availableResponse.data) {
        setAvailableProviders(availableResponse.data);
        
        // 각 프로바이더의 모델 목록 로드
        const providersData: any = {};
        for (const provider of availableResponse.data) {
          try {
            const modelsResponse = await dynamicApiClient.getProviderModels(provider.key);
            if (modelsResponse.success && modelsResponse.data) {
              providersData[provider.key] = {
                name: provider.name,
                requiresApiKey: provider.requiresApiKey,
                apiKeyConfigured: provider.apiKeyConfigured !== false,
                isDynamic: provider.isDynamic,
                isAvailable: true,
                models: modelsResponse.data
              };
            }
          } catch (modelError) {
            console.warn(`모델 목록 로드 실패: ${provider.key}`, modelError);
            providersData[provider.key] = {
              name: provider.name,
              requiresApiKey: provider.requiresApiKey,
              apiKeyConfigured: provider.apiKeyConfigured !== false,
              isDynamic: provider.isDynamic,
              isAvailable: true,
              models: []
            };
          }
        }
        
        setProviders(providersData);
        
        // 현재 선택된 프로바이더가 유효한지 확인하고, 필요시 기본값으로 설정
        if (availableResponse.data.length > 0) {
          const currentProvider = agentOptions.provider;
          const isCurrentProviderValid = availableResponse.data.some((p: any) => p.key === currentProvider);
          
          if (!isCurrentProviderValid) {
            const firstProvider = availableResponse.data[0];
            const firstModel = providersData[firstProvider.key]?.models?.length > 0 
              ? providersData[firstProvider.key].models[0].id 
              : 'hamonize:latest';
            
            setAgentOptions(prev => ({
              ...prev,
              provider: firstProvider.key,
              model: firstModel
            }));
          } else {
            const currentProviderData = providersData[currentProvider];
            if (currentProviderData && currentProviderData.models && currentProviderData.models.length > 0) {
              const currentModel = agentOptions.model;
              const isCurrentModelValid = currentProviderData.models.some((m: any) => m.id === currentModel);
              
              if (!isCurrentModelValid) {
                setAgentOptions(prev => ({
                  ...prev,
                  model: currentProviderData.models[0].id
                }));
              }
            }
          }
        }
      } else {
        console.error('사용 가능한 프로바이더 응답 실패:', availableResponse);
        throw new Error(`프로바이더 API 응답 실패: ${availableResponse.error?.message || '알 수 없는 오류'}`);
      }
    } catch (error: any) {
      console.error('프로바이더 정보 로드 실패:', error);
      
      // 오류 발생 시 기본 프로바이더 정보 설정
      const defaultModel = {
        id: 'hamonize:latest',
        name: 'hamonize:latest',
        size: 0
      };
      
      const defaultProviders = {
        ollama: {
          name: 'Ollama',
          isAvailable: true,
          requiresApiKey: false,
          apiKeyConfigured: true,
          isDynamic: true,
          models: [defaultModel]
        }
      };
      setProviders(defaultProviders);
      setAvailableProviders([
        { key: 'ollama', name: 'Ollama', requiresApiKey: false, apiKeyConfigured: true, isDynamic: true, models: [defaultModel] }
      ]);
    } finally {
      setIsLoadingProviders(false);
    }
  }, [agentOptions.provider, agentOptions.model]);

  // 프로바이더 변경 핸들러 (agent 페이지에서 가져옴)
  const handleProviderChange = useCallback(async (newProvider: string) => {
    setAgentOptions(prev => ({ ...prev, provider: newProvider }));
    
    const providerData = providers[newProvider];
    if (!providerData || !providerData.models || providerData.models.length === 0) {
      try {
        const apiUrl = getApiServerUrl();
        const dynamicApiClient = new (externalApiClient.constructor as any)(apiUrl);
        const modelsResponse = await dynamicApiClient.getProviderModels(newProvider);
        if (modelsResponse.success && modelsResponse.data && modelsResponse.data.length > 0) {
          setProviders((prev) => ({
            ...prev,
            [newProvider]: {
              ...prev[newProvider],
              models: modelsResponse.data
            }
          }));
          
          const newModel = modelsResponse.data[0].id;
          setAgentOptions(prev => ({ 
            ...prev, 
            provider: newProvider,
            model: newModel
          }));
          return;
        } else if (newProvider === 'ollama') {
          const defaultModel = {
            id: 'hamonize:latest',
            name: 'hamonize:latest',
            size: 0
          };
          
          setProviders((prev) => ({
            ...prev,
            [newProvider]: {
              ...prev[newProvider],
              models: [defaultModel]
            }
          }));
          
          setAgentOptions(prev => ({ 
            ...prev, 
            provider: newProvider,
            model: 'hamonize:latest'
          }));
          return;
        }
      } catch (error) {
        console.error(`프로바이더 ${newProvider}의 모델 목록 실시간 로드 중 오류:`, error);
        
        if (newProvider === 'ollama') {
          const defaultModel = {
            id: 'hamonize:latest',
            name: 'hamonize:latest',
            size: 0
          };
          
          setProviders((prev) => ({
            ...prev,
            [newProvider]: {
              ...prev[newProvider],
              models: [defaultModel]
            }
          }));
          
          setAgentOptions(prev => ({ 
            ...prev, 
            provider: newProvider,
            model: 'hamonize:latest'
          }));
          return;
        }
      }
    }
    
    if (providerData && providerData.models && providerData.models.length > 0) {
      const newModel = providerData.models[0].id;
      setAgentOptions(prev => ({ 
        ...prev, 
        provider: newProvider,
        model: newModel
      }));
    } else if (newProvider === 'ollama') {
      const defaultModel = 'hamonize:latest';
      setAgentOptions(prev => ({ 
        ...prev, 
        provider: newProvider,
        model: defaultModel
      }));
    } else {
      setAgentOptions(prev => ({ 
        ...prev, 
        provider: newProvider,
        model: ''
      }));
    }
  }, [providers]);

  useEffect(() => {
    fetchTemplates();
    loadProviders();
  }, [fetchTemplates, loadProviders]);

  // 템플릿 선택 시 프롬프트 예시 자동 반영
  useEffect(() => {
    if (selectedTemplate) {
      // const found = templates.find(opt => opt.value === selectedTemplate);
      // if (found && found.name) {
      //   setInputMessage(found.name);
      //   setTemplateDesc(found.name);
      // }
    }
  }, [selectedTemplate, templates]);

  // 입력창 높이 초기화 함수
  const resetTextareaHeight = () => {
    const textareas = document.querySelectorAll('textarea');
    textareas.forEach(textarea => {
      textarea.style.height = 'auto';
      textarea.style.height = '48px';
    });
  };

  // 인증 확인
  useEffect(() => {
    if (isInitialized && !isAuthenticated) {
      router.push('/login');
    }
  }, [isInitialized, isAuthenticated, router]);



  useEffect(() => {
    // 문서 페이지에서는 세션 로드 대신 문서 목록 로드
    if (isInitialized && isAuthenticated) {
      // console.log('[문서페이지] 인증 완료 - 문서 목록 로드');
      // 문서 목록은 이미 다른 useEffect에서 로드됨
    }
  }, [isInitialized, isAuthenticated]);



  useEffect(() => {
    // 클라이언트 사이드에서만 실행되도록 보장
    if (typeof window === 'undefined') return;

    try {
      // URL에서 document 관련 파라미터만 처리
      const documentId = searchParams.get('document');
      if (documentId && reports.length > 0) {
        // console.log('[문서페이지] URL에서 문서 ID 감지:', documentId);
        // 문서 ID가 있으면 해당 문서를 열기
        const report = reports.find(r => r.uuid === documentId);
        if (report) {
          // 문서 탭 열기 로직을 직접 실행
          setOpenTabs(prev => {
            if (prev.find(tab => tab.uuid === report.uuid)) return prev;
            return [...prev, {
              uuid: report.uuid,
              title: report.executive_summary ? removeMetadataFromContent(report.executive_summary).slice(0, 30) + '...' : '문서',
              status: report.status
            }];
          });
          setActiveTab(report.uuid);
        }
      }
    } catch (error) {
      console.warn('[문서페이지] URL 파라미터 처리 중 오류:', error);
      // searchParams 접근 실패 시 조용히 무시
    }
  }, [searchParams, reports]);

  // 인증 상태가 변경될 때 RAG 파일 로드 (문서 페이지용)
  useEffect(() => {
    if (isAuthenticated) {
      // console.log('[문서페이지] 인증 완료 - RAG 파일 로드');
      loadRagFiles();
    }
  }, [isAuthenticated]);

  // 문서 내용이 로딩된 후 섹션 파싱 (수정 모드용)
  useEffect(() => {
    if (editTargetDocument && tabContents[editTargetDocument.uuid]?.content) {
      const content = tabContents[editTargetDocument.uuid].content;
      if (typeof content === 'string') {
        const sections = parseDocumentSections(content);
        setDocumentSections(sections);
        console.log(`[useEffect-섹션파싱] ${editTargetDocument.uuid}: ${sections.length}개 섹션 파싱 완료`);
      }
    }
  }, [editTargetDocument, tabContents]);

  // 리사이저 드래그 이벤트 핸들러
  useEffect(() => {
    if (!isResizing) return;
    const handleMouseMove = (e: MouseEvent) => {
      // 전체 윈도우 기준으로 오른쪽 패널의 width 계산
      const minWidth = 280;
      const maxWidth = 700;
      const newWidth = Math.max(minWidth, Math.min(maxWidth, window.innerWidth - e.clientX));
      setRightPanelWidth(newWidth);
    };
    const handleMouseUp = () => setIsResizing(false);
    window.addEventListener('mousemove', handleMouseMove);
    window.addEventListener('mouseup', handleMouseUp);
    return () => {
      window.removeEventListener('mousemove', handleMouseMove);
      window.removeEventListener('mouseup', handleMouseUp);
    };
  }, [isResizing]);

  // 🔄 새로운 섹션이 보여질 때 자동 스크롤
  useEffect(() => {
    const sectionNames = Object.keys(streamingSections);

    if (sectionNames.length > 0) {
      // 가장 최근에 visible이 된 섹션 찾기
      const visibleSections = sectionNames.filter(name =>
        streamingSections[name].isVisible
      );

      if (visibleSections.length > 0) {
        // 가장 최근에 추가된 visible 섹션으로 스크롤
        const latestVisible = visibleSections.reduce((latest, current) => {
          const currentTimestamp = new Date(streamingSections[current].timestamp).getTime();
          const latestTimestamp = new Date(streamingSections[latest].timestamp).getTime();
          return currentTimestamp > latestTimestamp ? current : latest;
        });

        // 섹션이 완전히 렌더링된 후 스크롤
        setTimeout(() => {
          scrollToSection(latestVisible, 'smooth');
        }, 200);
      }
    }
  }, [streamingSections]);

  // 중복 스크롤 로직 제거 - 위의 useEffect에서 처리

  // 전역 클릭 이벤트로 컨텍스트 메뉴 닫기
  useEffect(() => {
    const handleGlobalClick = () => {
      if (contextMenu.show) {
        closeContextMenu();
      }
    };

    if (contextMenu.show) {
      document.addEventListener('click', handleGlobalClick);
      document.addEventListener('contextmenu', handleGlobalClick);
    }

    return () => {
      document.removeEventListener('click', handleGlobalClick);
      document.removeEventListener('contextmenu', handleGlobalClick);
    };
  }, [contextMenu.show]);

  // 타이핑 효과 제거됨 - 콘텐츠를 바로 표시

  // 새 문서 버튼 클릭 처리 (문서 페이지용)
  const handleNewChatClick = async () => {
    if (isLoading) return; // 문서 생성 중에는 새 문서 금지
    console.log('=== 새 문서 버튼 클릭 시작 ===');

    // 스트리밍 상태 정리
    stopSseStreaming(true);
    setCurrentTaskId(null);
    setTaskStartTime(null);
    setStreamingSections({});
    setStreamingProgress(0);
    setStreamingMessage('');
    setProcessingStatus('');

    // URL에서 document 파라미터 제거
    const url = new URL(window.location.href);
    url.searchParams.delete('document');
    window.history.replaceState({}, '', url.toString());

    // 문서 생성 관련 상태 초기화
    setUploadedFiles([]);
    setError(null);

    // 모든 탭의 상태 초기화
    setCreateInputMessage('');
    setCreateSelectedTemplate('');
    setCreateFileFormat('pdf');
    setCreateUserEmail('');
    setEditInputMessage('');
    setEditFileFormat('pdf');
    setEditUserEmail('');

    // 생성 탭으로 전환
    setRightPanelTab('create');

    resetTextareaHeight();

    // 활성 탭 닫기
    setOpenTabs([]);
    setActiveTab(null);
    setTabContents({});

    console.log('=== 새 문서 버튼 클릭 완료 ===');
  };

  const loadSessions = async () => {
    // 문서 페이지에서는 세션 로드 대신 문서 목록 로드
    // console.log('[문서페이지] 세션 로드 요청 - 문서 목록으로 대체');
    return;
  };

  const selectSession = async (sessionId: string) => {
    // 문서 페이지에서는 세션 선택 대신 문서 선택
    // console.log('[문서페이지] 세션 선택 요청 - 문서 선택으로 대체:', sessionId);
    return;
  };

  const createNewSession = async () => {
    // 문서 페이지에서는 새 세션 생성 대신 새 문서 모드
    // console.log('[문서페이지] 새 세션 생성 요청 - 새 문서 모드로 대체');
    setCreateInputMessage('');
    setCreateSelectedTemplate('');
    setCreateFileFormat('pdf');
    setCreateUserEmail('');
    setRightPanelTab('create');
    return 'new-document';
  };

  // 파일 업로드 처리 (백그라운드 작업으로 변경)
  const handleFileSelect = async (event: React.ChangeEvent<HTMLInputElement>) => {
    const files = event.target.files;
    if (!files || files.length === 0) return;
    if (isUploading) return;

    if (!user?.username) {
      setError('로그인이 필요합니다.');
      return;
    }

    const newFiles = Array.from(files);
    setUploadedFiles(prev => [...prev, ...newFiles]);

    // 백그라운드 작업으로 파일 업로드
    for (const file of newFiles) {
      if (!file.type.startsWith('image/')) {
        await handleFileUploadBackground(file);
      }
    }

    // 파일 입력 초기화
    if (fileInputRef.current) {
      fileInputRef.current.value = '';
    }
  };

  const uploadFileToRag = async (file: File) => {
    const fileKey = `${file.name}-${file.size}-${file.lastModified}`;

    // 이미 업로드 중인 파일이면 중단
    if (uploadingFiles.has(fileKey)) {
      // console.log(`파일 ${file.name}은 이미 업로드 중입니다.`);
      return;
    }

    try {
      setIsUploading(true);
      setUploadingFiles(prev => new Set(prev).add(fileKey));

      // console.log('파일 업로드 시작:', {
      //   fileName: file.name,
      //   fileKey,
      //   userId: user?.username,
      //   userInfo: user
      // });

      // 사용자명과 함께 파일 업로드
      await externalApiClient.uploadFile(file, user?.username);
      console.log(`파일 업로드 완료: ${file.name}`);
    } catch (error) {
      console.error('파일 업로드 실패:', error);
      setError(`파일 업로드 실패: ${file.name}`);
    } finally {
      setIsUploading(false);
      setUploadingFiles(prev => {
        const newSet = new Set(prev);
        newSet.delete(fileKey);
        return newSet;
      });
    }
  };

  const removeFile = (index: number) => {
    setUploadedFiles(prev => prev.filter((_, i) => i !== index));
  };

  const isImageFile = (file: File) => {
    return file.type.startsWith('image/');
  };

  // 다운로드 처리 함수 (GET 방식)
  const handleDownload = async (filePath: string, fileName?: string) => {
    if (!filePath) {
      setError('다운로드할 파일 경로를 찾을 수 없습니다.');
      setTimeout(() => setError(null), 3000);
      return;
    }

    try {
      console.log(`[다운로드] 파일 다운로드 시작:`, { filePath, fileName });

      // 다운로드 시작 알림
      setShowToast({ message: '파일 다운로드를 시작합니다...', type: 'success' });
      setTimeout(() => setShowToast(null), 2000);

      // 인증 헤더 추가
      const headers = getAuthHeaders();

      // GET 요청으로 파일 다운로드
      console.log(`[다운로드] GET 요청:`, { url: filePath });
      const response = await fetch(filePath, {
        method: 'GET',
        headers: headers
      });

      console.log(`[다운로드] 응답:`, { 
        status: response.status, 
        statusText: response.statusText,
        contentType: response.headers.get('content-type')
      });

      if (!response.ok) {
        throw new Error(`파일을 찾을 수 없습니다 (${response.status})`);
      }

      // Blob으로 변환하여 다운로드
      const blob = await response.blob();
      const blobUrl = window.URL.createObjectURL(blob);
      
      const link = document.createElement('a');
      link.href = blobUrl;
      link.download = fileName || 'download';
      link.style.display = 'none';
      
      document.body.appendChild(link);
      link.click();
      document.body.removeChild(link);
      
      // 메모리 정리
      window.URL.revokeObjectURL(blobUrl);

      console.log(`[다운로드] 완료:`, { fileName, size: blob.size });
      setShowToast({ message: '파일 다운로드가 완료되었습니다.', type: 'success' });
      setTimeout(() => setShowToast(null), 2000);

    } catch (error) {
      console.error('[다운로드] 실패:', error);
      
      const errorMessage = error instanceof Error ? error.message : '파일 다운로드 중 오류가 발생했습니다.';
      setError(`다운로드 실패: ${errorMessage}`);
      setTimeout(() => setError(null), 5000);
    }
  };

  // 🔍 Clarification 관련 함수들
  const startClarificationProcess = async () => {
    const currentInputMessage = rightPanelTab === 'create' ? createInputMessage : editInputMessage;
    const currentFileFormat = rightPanelTab === 'create' ? createFileFormat : editFileFormat;
    const currentSelectedTemplate = rightPanelTab === 'create' ? createSelectedTemplate : '';

    if (!currentInputMessage.trim()) return;

    try {
      setClarificationLoading(true);
      setShowClarificationModal(true);
      setClarificationStep('analyzing');
      setClarificationData(null);
      setClarificationResponses({});

      console.log('🔍 Clarification 분석 시작...');

      // 사용자 요청 분석 API 호출
      const analysisResponse = await fetch('/api/document', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...getAuthHeaders()
        },
        body: JSON.stringify({
          action: 'analyze',
          prompt: currentInputMessage,
          template: currentSelectedTemplate || 'auto',
          format: currentFileFormat,
          username: user?.username
        })
      });

      if (!analysisResponse.ok) {
        throw new Error(`분석 요청 실패: ${analysisResponse.status}`);
      }

      const analysisResult = await analysisResponse.json();
      console.log('✅ Clarification 분석 완료:', analysisResult);

      setClarificationData(analysisResult);

      // 🎯 분석 결과에 따른 조건부 처리
      if (!analysisResult.questions || analysisResult.questions.length === 0) {
        // 질문이 없으면 바로 문서 생성 시작
        console.log('📝 추가 정보가 충분합니다. 바로 문서 생성을 시작합니다.');
        setClarificationStep('confirm');

        // 자동으로 문서 생성 시작
        setTimeout(async () => {
          await proceedToDirectGeneration(analysisResult);
        }, 1500); // 1.5초 후 자동 진행
      } else {
        // 질문이 있으면 질문 단계로 이동
        setClarificationStep('questions');

        // 기본 답변들 초기화
        const defaultResponses: { [key: string]: any } = {};
        analysisResult.questions?.forEach((q: any) => {
          if (q.type === 'multiselect') {
            defaultResponses[q.id] = [];
          } else {
            defaultResponses[q.id] = '';
          }
        });
        setClarificationResponses(defaultResponses);
      }

    } catch (error) {
      console.error('❌ Clarification 분석 실패:', error);
      setError(error instanceof Error ? error.message : 'Clarification 분석 중 오류가 발생했습니다.');
      setShowClarificationModal(false);
    } finally {
      setClarificationLoading(false);
    }
  };

  const handleClarificationResponse = (questionId: string, answer: any) => {
    setClarificationResponses(prev => ({
      ...prev,
      [questionId]: answer
    }));
  };

  const proceedToTodoReview = () => {
    setClarificationStep('todo');
  };

  const proceedToDirectGeneration = async (analysisResult: any) => {
    try {
      console.log('🚀 질문 없이 바로 문서 생성 시작...');

      const currentInputMessage = rightPanelTab === 'create' ? createInputMessage : editInputMessage;
      const currentFileFormat = rightPanelTab === 'create' ? createFileFormat : editFileFormat;
      const currentUserEmail = rightPanelTab === 'create' ? createUserEmail : editUserEmail;

      // 향상된 보고서 생성 요청 (빈 응답 배열로)
      const enhancedRequest = {
        original_prompt: currentInputMessage,
        clarification_responses: [], // 빈 배열
        confirmed_template: analysisResult?.suggested_template || 'simple',
        format: currentFileFormat,
        email: currentUserEmail?.trim() || undefined,
        username: user?.username,
        use_cache: true
      };

      const generateResponse = await fetch('/api/document', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...getAuthHeaders()
        },
        body: JSON.stringify({
          action: 'generate-enhanced',
          ...enhancedRequest
        })
      });

      if (!generateResponse.ok) {
        throw new Error(`문서 생성 요청 실패: ${generateResponse.status}`);
      }

      const result = await generateResponse.json();
      console.log('📄 문서 생성 작업 시작:', result);

      // 모달 닫고 일반 문서 생성 플로우로 전환
      setShowClarificationModal(false);

      if (result.job_id) {
        // 기존 문서 생성 로직과 동일하게 처리
        if (rightPanelTab === 'create') {
          setCreateInputMessage('');
          setCreateUserEmail('');
        } else {
          setEditInputMessage('');
          setEditUserEmail('');
        }

        startSseStreaming(result.job_id);
        setCurrentTaskId(result.job_id);
        saveActiveTask(result.job_id, Date.now());
      }

    } catch (error) {
      console.error('❌ 직접 문서 생성 실패:', error);
      setError(error instanceof Error ? error.message : '문서 생성 중 오류가 발생했습니다.');
      setShowClarificationModal(false);
    }
  };

  const confirmAndGenerate = async () => {
    try {
      setClarificationLoading(true);

      // 응답들을 API 형식으로 변환
      const responses = Object.entries(clarificationResponses).map(([questionId, answer]) => ({
        question_id: questionId,
        answer: answer
      }));

      const currentInputMessage = rightPanelTab === 'create' ? createInputMessage : editInputMessage;
      const currentFileFormat = rightPanelTab === 'create' ? createFileFormat : editFileFormat;
      const currentUserEmail = rightPanelTab === 'create' ? createUserEmail : editUserEmail;

      // 향상된 보고서 생성 요청
      const enhancedRequest = {
        original_prompt: currentInputMessage,
        clarification_responses: responses,
        confirmed_template: clarificationData?.suggested_template || 'simple',
        format: currentFileFormat,
        email: currentUserEmail?.trim() || undefined,
        username: user?.username,
        use_cache: true
      };

      console.log('📝 향상된 보고서 생성 요청:', enhancedRequest);

      const generateResponse = await fetch('/api/document', {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          ...getAuthHeaders()
        },
        body: JSON.stringify({
          action: 'generate-enhanced',
          ...enhancedRequest
        })
      });

      if (!generateResponse.ok) {
        throw new Error(`향상된 보고서 생성 실패: ${generateResponse.status}`);
      }

      const result = await generateResponse.json();
      console.log('✅ 향상된 보고서 생성 시작:', result);

      // 모달 닫기
      setShowClarificationModal(false);

      // 기존 sendMessage와 동일한 후속 처리
      const jobId = result.data?.job_id || result.job_id;
      if (!jobId) {
        throw new Error('Job ID를 받지 못했습니다.');
      }

      // 백그라운드 작업 상태 설정
      setCurrentTaskId(jobId);
      setTaskStartTime(Date.now());

      // 스트리밍용 탭 생성
      const streamingTabId = `streaming-${jobId}`;
      const cleanMessage = currentInputMessage.trim() ? removeMetadataFromContent(currentInputMessage.trim()) : '';
      const messageTitle = cleanMessage ? cleanMessage.slice(0, 25) : `작업 ${jobId.slice(-8)}`;
      const streamingTitle = `[향상됨] ${messageTitle}... (진행중)`;

      setOpenTabs(prev => {
        if (prev.find(tab => tab.uuid === streamingTabId)) return prev;
        return [...prev, { uuid: streamingTabId, title: streamingTitle, status: 'processing' }];
      });
      setActiveTab(streamingTabId);

      setTabContents(prev => ({
        ...prev,
        [streamingTabId]: {
          type: 'streaming',
          jobId: jobId,
          content: ''
        }
      }));

      // SSE 스트리밍 시작
      startSseStreaming(jobId);

      // UI 상태 초기화
      if (rightPanelTab === 'create') {
        setCreateInputMessage('');
      } else {
        setEditInputMessage('');
      }
      setUploadedFiles([]);
      setIsLoading(false);

      // 진행 중인 작업 목록 업데이트
      await fetchActiveJobs();

    } catch (error) {
      console.error('❌ 향상된 보고서 생성 실패:', error);
      setError(error instanceof Error ? error.message : '향상된 보고서 생성 중 오류가 발생했습니다.');
    } finally {
      setClarificationLoading(false);
    }
  };

  // sendMessage에서 externalApiClient를 사용하여 리포트 생성
  const sendMessage = async () => {
    // 현재 활성 탭에 따라 적절한 상태 사용
    const currentInputMessage = rightPanelTab === 'create' ? createInputMessage : editInputMessage;
    const currentSelectedTemplate = rightPanelTab === 'create' ? createSelectedTemplate : '';
    const currentFileFormat = rightPanelTab === 'create' ? createFileFormat : editFileFormat;
    const currentUserEmail = rightPanelTab === 'create' ? createUserEmail : editUserEmail;

    if ((!currentInputMessage.trim() && uploadedFiles.length === 0) || isLoading) return;

    setIsLoading(true);
    setError(null);

    // 모드에 따른 상태 메시지
    const isEditMode = rightPanelTab === 'edit';
    let statusMessage = '📄 문서 생성 요청을 시작합니다...';
    if (isEditMode) {
      if (editMethod === 'section' && editTargetSection) {
        statusMessage = `✂️ "${editTargetSection}" 섹션 부분 수정을 시작합니다...`;
      } else {
        statusMessage = '🔄 전체 문서 재생성을 시작합니다...';
      }
    }
    setProcessingStatus(statusMessage);

    try {
      let finalPrompt = currentInputMessage.trim();

      // 📝 편집 모드: 기존 문서 내용과 수정 요청을 결합
      if (isEditMode && editTargetDocument) {
        const existingContent = tabContents[editTargetDocument.uuid]?.content || '';

        if (editMethod === 'section' && editTargetSection && documentSections.length > 0) {
          // 섹션별 부분 수정: 선택된 섹션만 교체
          const targetSection = documentSections.find(section => section.title === editTargetSection);
          if (targetSection) {
            finalPrompt = `기존 문서에서 "${editTargetSection}" 섹션만을 다음 요청에 따라 수정하고, 나머지 모든 섹션은 기존 그대로 유지해주세요.

[전체 문서]
${existingContent}

[수정할 섹션: ${editTargetSection}]
${targetSection.content}

[수정 요청사항]
${currentInputMessage.trim()}

※ 중요: "${editTargetSection}" 섹션만 수정하고, 다른 모든 섹션과 내용은 원본 그대로 유지해주세요.`;
          }
        } else {
          // 전체 문서 재생성: 모든 내용을 고려하여 새로운 문서 생성
          if (editTargetSection && documentSections.length > 0) {
            // 특정 섹션을 중점적으로 수정하되 전체 재생성
            const targetSection = documentSections.find(section => section.title === editTargetSection);
            finalPrompt = `다음은 기존 문서입니다:

${existingContent}

===

"${editTargetSection}" 섹션을 중점적으로 다음과 같이 수정해주세요:
${currentInputMessage.trim()}

전체 문서의 일관성과 완성도를 유지하면서 요청된 수정사항을 반영한 완전한 새 문서를 생성해주세요.`;
          } else {
            // 전체 문서 수정
            finalPrompt = `다음은 기존 문서입니다:

${existingContent}

===

다음과 같이 이 문서를 수정해주세요:
${currentInputMessage.trim()}

기존 문서의 구조와 다른 내용은 유지하면서 요청된 부분만 수정하여 완전한 문서를 생성해주세요.`;
          }
        }

        console.log('[문서수정] 결합된 프롬프트:', {
          mode: rightPanelTab,
          targetDocument: editTargetDocument?.uuid,
          targetSection: editTargetSection,
          originalPromptLength: currentInputMessage.trim().length,
          finalPromptLength: finalPrompt.length
        });
      }

      // 리포트 생성 요청 옵션 (섹션별 수정 메타데이터 포함)
      const reportOptions = {
        prompt: finalPrompt,
        template: isEditMode ? '' : (currentSelectedTemplate || 'simple'), // 편집 모드에서는 템플릿 사용 안함
        format: currentFileFormat,
        email: currentUserEmail?.trim() || undefined,
        userId: user?.username,
        ragSearchScope: 'personal', // RAG 검색 범위 (기본값: 개인 문서)

        // 📝 섹션별 수정을 위한 핵심 메타데이터
        modification_mode: (isEditMode && editMethod === 'section' ? 'partial' : 'full') as 'full' | 'partial',
        existing_document_content: isEditMode && editTargetDocument ? (tabContents[editTargetDocument.uuid]?.content || '') : undefined,
        sections_to_modify: isEditMode && editMethod === 'section' && editTargetSection ? [editTargetSection] : undefined,
        preserve_formatting: true, // 기존 문서 서식 유지

        // 편집 모드 관련 메타데이터 (백엔드 확장용)
        editMethod: isEditMode ? editMethod : undefined,
        editTargetDocument: isEditMode && editTargetDocument ? editTargetDocument.uuid : undefined,
        editTargetSection: isEditMode && editTargetSection ? editTargetSection : undefined,
        
        // AI 모델 설정
        provider: agentOptions.provider,
        model: agentOptions.model,
      };

      console.log('[문서처리] 요청 시작:', {
        mode: rightPanelTab,
        promptLength: finalPrompt.length,
        template: reportOptions.template,
        format: reportOptions.format
      });

      // 1. 먼저 생성 요청을 보내서 job_id를 받아옴
      const generateResponse = await externalApiClient.generateReport(reportOptions);
      console.log('[문서생성] 생성 요청 응답:', generateResponse);

      if (!generateResponse.success) {
        throw new Error(generateResponse.error?.message || '문서 생성 요청 실패');
      }

      // 2. 서버에서 받은 실제 job_id 사용
      const responseData = generateResponse.data || generateResponse;
      let jobId = responseData?.job_id || responseData?.jobId;
      if (!jobId) {
        console.error('[문서생성] job_id를 찾을 수 없음:', { generateResponse, responseData });
        throw new Error('서버에서 job_id를 받지 못했습니다.');
      }

      console.log('[문서생성] 원본 응답 확인:', { generateResponse, responseData, jobId });

      // 🔧 Job ID 정리: externalApiClient에서 조합된 형태라면 백엔드 API를 직접 호출하여 실제 Job ID 확인
      // Job ID는 큐에서 생성된 것을 그대로 사용 (변경하지 않음)
      console.log('[문서생성] 큐에서 생성된 Job ID 사용:', jobId);

      console.log('[문서생성] 최종 job_id:', jobId);

      // 3. 백그라운드 작업 상태 설정
      setCurrentTaskId(jobId);
      setTaskStartTime(Date.now());

      // 즉시 UI 피드백을 위해 임시로 activeJobs에 추가
      let tempJobTitle: string;
      if (isEditMode && editTargetDocument) {
        const docTitle = reportTitles[editTargetDocument.uuid] || editTargetDocument.file_name || editTargetDocument.project_hash.slice(0, 8);
        if (editMethod === 'section' && editTargetSection) {
          tempJobTitle = `✂️ ${docTitle} (${editTargetSection} 부분수정)`;
        } else {
          tempJobTitle = `🔄 ${docTitle} (전체재생성)`;
        }
      } else {
        const cleanMessage = removeMetadataFromContent(currentInputMessage.trim());
        tempJobTitle = cleanMessage.slice(0, 50) + (cleanMessage.length > 50 ? '...' : '');
      }

      const tempJob = {
        id: Date.now(), // 임시 ID
        job_id: jobId,
        title: tempJobTitle,
        status: 'processing' as const,
        progress: 0,
        created_at: new Date().toISOString(),
        updated_at: new Date().toISOString(),
        user_id: user?.username || null,
        username: user?.username || 'admin'
      };

      setActiveJobs(prev => {
        // 이미 같은 job_id가 있으면 추가하지 않음
        if (prev.find(job => job.job_id === jobId)) return prev;
        return [tempJob, ...prev];
      });

      // 사용자에게 백그라운드 작업 시작을 알림
      let taskType = '문서 생성';
      if (isEditMode) {
        if (editMethod === 'section' && editTargetSection) {
          taskType = `"${editTargetSection}" 섹션 부분 수정`;
        } else {
          taskType = '문서 전체 재생성';
        }
      }
      const backgroundMessage = currentUserEmail?.trim()
        ? `${taskType}이 백그라운드에서 시작되었습니다.\n완료되면 ${currentUserEmail}로 이메일이 발송됩니다.\n\n⏱️ 복잡한 문서는 시간이 소요될 수 있습니다.`
        : `${taskType}이 백그라운드에서 시작되었습니다.\n완료되면 왼쪽 문서 목록에 나타납니다.\n\n⏱️ 복잡한 문서는 시간이 소요될 수 있습니다.`;

      setProcessingStatus(`✅ ${backgroundMessage}\n\n🆔 작업 ID: ${jobId.slice(-8)}\n🔄 진행상태를 실시간으로 확인하고 있습니다...`);

      // 4. 스트리밍용 탭 생성
      const streamingTabId = `streaming-${jobId}`;

      // 메타데이터가 제거된 제목 사용
      let messageTitle: string;
      if (isEditMode && editTargetDocument) {
        const docTitle = reportTitles[editTargetDocument.uuid] || editTargetDocument.file_name || editTargetDocument.project_hash.slice(0, 8);
        if (editMethod === 'section' && editTargetSection) {
          messageTitle = `✂️ ${docTitle} (${editTargetSection} 부분수정)`;
        } else {
          messageTitle = `🔄 ${docTitle} (전체재생성)`;
        }
      } else {
        const cleanMessage = currentInputMessage.trim() ? removeMetadataFromContent(currentInputMessage.trim()) : '';
        messageTitle = cleanMessage ? cleanMessage.slice(0, 25) : `작업 ${jobId.slice(-8)}`;
      }

      const streamingTitle = `${messageTitle}... (진행중)`;

      // 스트리밍 탭 추가
      setOpenTabs(prev => {
        if (prev.find(tab => tab.uuid === streamingTabId)) return prev;
        return [...prev, { uuid: streamingTabId, title: streamingTitle, status: 'processing' }];
      });
      setActiveTab(streamingTabId);

      // 스트리밍 탭 내용 설정
      setTabContents(prev => ({
        ...prev,
        [streamingTabId]: {
          type: 'streaming',
          jobId: jobId,
          content: ''
        }
      }));

      // 5. SSE 스트리밍 시작 - jobId를 직접 전달
      console.log(`[문서생성] SSE 스트리밍 시작 - Job ID: ${jobId}`);
      startSseStreaming(jobId);

      // 6. 진행 중인 작업 목록 즉시 업데이트 시도
      try {
        await fetchActiveJobs();
        console.log('[문서생성] 진행 중인 작업 목록 즉시 업데이트 완료');
      } catch (error) {
        console.error('[문서생성] 진행 중인 작업 목록 즉시 업데이트 실패:', error);
      }

      // 7. 추가 업데이트 (데이터베이스 등록 지연 고려)
      setTimeout(async () => {
        try {
          await fetchActiveJobs();
          console.log('[문서생성] 진행 중인 작업 목록 지연 업데이트 완료');
        } catch (error) {
          console.error('[문서생성] 진행 중인 작업 목록 지연 업데이트 실패:', error);
        }
      }, 1000); // 1초 후 재업데이트

      // 8. UI 즉시 해제 (백그라운드 작업이므로)
      setIsLoading(false);

      // 현재 탭에 따라 적절한 상태 초기화
      if (rightPanelTab === 'create') {
        setCreateInputMessage('');
      } else {
        setEditInputMessage('');
      }
      setUploadedFiles([]);

      // 9. 편집 모드였다면 생성 모드로 자동 복귀
      if (isEditMode) {
        setTimeout(() => {
          setRightPanelTab('create');
          enterCreateMode();
        }, 1000); // 1초 후 자동 복귀
      }

    } catch (error) {
      const taskType = isEditMode ? '문서 수정' : '문서 생성';
      console.error(`[${taskType}] 오류:`, error);
      setError(error instanceof Error ? error.message : `${taskType} 요청 중 오류가 발생했습니다.`);
      setProcessingStatus(`❌ ${taskType} 요청 중 오류가 발생했습니다.`);
      setCurrentTaskId(null);
      setTaskStartTime(null);
      setIsLoading(false);
    }
  };

  const handleKeyPress = (e: React.KeyboardEvent) => {
    if (e.key === 'Enter' && !e.shiftKey) {
      e.preventDefault();
      sendMessage();
    }
  };

  // RAG 파일 목록 로드
  const loadRagFiles = async () => {
    if (!user?.username) {
      console.log('사용자 정보가 없어서 RAG 파일 목록을 로드할 수 없습니다.');
      return;
    }

    try {
      // console.log('RAG 파일 목록 로드 시작:', user.username);
      setIsLoadingRagFiles(true);
      // 개인 문서 목록 조회 (모든 개인 페이지에서 공유)
      const response = await fetch(`/api/chat/documents?userId=${user.username}&checkFiles=true`, {
        method: 'GET',
        credentials: 'include'
      });
      const result = await response.json();

      // console.log('RAG 파일 목록 API 응답:', result);

      if (result.success) {
        // 데이터베이스 기반 응답 형식 처리
        const files = result.data || [];
        // console.log('서버에서 받은 파일 목록:', files);

        // 서버에서 온 파일들은 모두 완료 상태로 설정, 데이터베이스 형식을 RAG 형식으로 변환
        const completedFiles = files.map((file: any) => ({
          name: file.filename,
          size: file.filesize,
          type: file.mimetype,
          created_at: file.created_at,
          file_exists: file.file_exists,
          status: 'completed' as const
        }));

        // console.log('완료 상태로 변환된 파일들:', completedFiles);

        // 기존 처리 중인 파일들과 병합 (중복 제거)
        setRagFiles(prev => {
          // console.log('기존 파일 목록:', prev);
          const processingFiles = prev.filter((f: RagFile) => f.status === 'uploading' || f.status === 'processing');
          const existingNames = new Set(completedFiles.map((f: RagFile) => f.name));
          const uniqueProcessingFiles = processingFiles.filter((f: RagFile) => !existingNames.has(f.name));
          const finalFiles = [...uniqueProcessingFiles, ...completedFiles];
          // console.log('최종 파일 목록:', finalFiles);
          return finalFiles;
        });
      } else {
        console.error('RAG 파일 목록 로드 실패:', result.error?.message);
      }
    } catch (error) {
      console.error('RAG 파일 목록 로드 오류:', error);
    } finally {
      setIsLoadingRagFiles(false);
    }
  };;

  // RAG 파일 삭제
  const deleteRagFile = async (filename: string) => {
    if (!user?.username) return;

    try {
      // Document 페이지 전용 문서 삭제 (데이터베이스 기반)
      const response = await fetch('/api/chat/documents', {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json',
        },
        credentials: 'include',
        body: JSON.stringify({ 
          filename: filename, 
          id: user.username
        }),
      });

      const result = await response.json();

      if (result.success) {
        console.log(`Document 페이지에서 파일 ${filename} 삭제 성공`);
        // 파일 목록 새로고침
        await loadRagFiles();
      } else {
        console.error('Document 페이지 파일 삭제 실패:', result.error?.message);
      }
    } catch (error) {
      console.error('Document 페이지 파일 삭제 오류:', error);
    }
  };;

  // RAG 드래그 앤 드롭 핸들러
  const handleRagDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOverRag(true);
  };

  const handleRagDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (!e.currentTarget.contains(e.relatedTarget as Node)) {
      setIsDragOverRag(false);
    }
  };

  const handleRagDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  // 폴더 내부 파일들을 재귀적으로 수집하는 함수
  const getAllFilesFromItems = async (items: DataTransferItemList): Promise<File[]> => {
    const files: File[] = [];

    const processEntry = async (entry: any): Promise<void> => {
      if (entry.isFile) {
        return new Promise((resolve) => {
          entry.file((file: File) => {
            // 지원되는 파일 확장자 확인
            const fileExt = file.name.split('.').pop()?.toLowerCase();
            const supportedExtensions = ['.pdf', '.txt', '.md', '.doc', '.docx', '.hwp', '.ppt', '.pptx', '.xls', '.xlsx'];

            if (fileExt && supportedExtensions.includes(`.${fileExt}`)) {
              console.log(`폴더에서 파일 발견: ${file.name} (${file.size} bytes)`);
              files.push(file);
            } else {
              console.log(`지원되지 않는 파일 형식: ${file.name}`);
            }
            resolve();
          });
        });
      } else if (entry.isDirectory) {
        const dirReader = entry.createReader();
        return new Promise((resolve) => {
          dirReader.readEntries(async (entries: any[]) => {
            for (const childEntry of entries) {
              await processEntry(childEntry);
            }
            resolve();
          });
        });
      }
    };

    // 모든 아이템 처리
    for (let i = 0; i < items.length; i++) {
      const item = items[i];
      if (item.kind === 'file') {
        const entry = item.webkitGetAsEntry();
        if (entry) {
          await processEntry(entry);
        }
      }
    }

    return files;
  };

  const handleRagDrop = async (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOverRag(false);

    console.log('드롭 이벤트 처리 시작...');
    console.log('dataTransfer.files 길이:', e.dataTransfer.files?.length);

    // 직접 dataTransfer.files 사용 (가장 안정적인 방법)
    const allFiles = Array.from(e.dataTransfer.files);
    console.log('드롭된 파일들:', allFiles.map(f => `${f.name} (${f.size} bytes, ${f.type})`));

    if (allFiles.length === 0) {
      console.log('드롭된 파일이 없습니다.');
      return;
    }

    // 지원되는 파일만 필터링
    const supportedExtensions = ['pdf', 'txt', 'md', 'doc', 'docx', 'hwp', 'hwpx', 'ppt', 'pptx', 'xls', 'xlsx'];
    
    const files = allFiles.filter(file => {
      const fileExt = file.name.split('.').pop()?.toLowerCase();
      const hasExtension = file.name.includes('.') && !file.name.endsWith('.');
      const isNotHidden = !file.name.startsWith('.');
      const isSupportedExtension = fileExt && supportedExtensions.includes(fileExt);
      const isValidFile = hasExtension && isNotHidden && isSupportedExtension;

      console.log(`파일 ${file.name}: 확장자=${fileExt}, 지원여부=${isSupportedExtension}, 결과=${isValidFile ? 'INCLUDE' : 'EXCLUDE'}`);

      return isValidFile;
    });

    console.log('필터링 후 최종 파일 목록:', files.map(f => f.name));

    if (files.length === 0) {
      console.log('업로드할 수 있는 파일이 없습니다.');
      return;
    }

    if (!user?.username) {
      setError('로그인이 필요합니다.');
      return;
    }

    console.log(`총 ${files.length}개 파일을 업로드합니다.`);

    // 각 파일을 순차적으로 업로드
    for (const file of files) {
      try {
        await externalApiClient.uploadFile(
          file, 
          user.username, 
          'ai-document',
          (progress, status) => {
            setRagFiles(prev => {
              const index = prev.findIndex(f => f.name === file.name);
              if (index !== -1) {
                const updatedFiles = [...prev];
                updatedFiles[index] = {
                  ...updatedFiles[index],
                  status,
                  progress
                };
                return updatedFiles;
              } else {
                return [...prev, {
                  name: file.name,
                  size: file.size,
                  status,
                  progress
                }];
              }
            });
          },
          files.length > 1 // 다중 파일 업로드 여부
        );
      } catch (error) {
        console.error(`파일 업로드 실패: ${file.name}`, error);
        setError(`${file.name} 업로드에 실패했습니다.`);
        setRagFiles(prev => {
          const index = prev.findIndex(f => f.name === file.name);
          if (index !== -1) {
            const updatedFiles = [...prev];
            updatedFiles[index] = {
              ...updatedFiles[index],
              status: 'error',
              error: error instanceof Error ? error.message : '업로드 실패'
            };
            return updatedFiles;
          }
          return prev;
        });
      }
    }
  };

  // 텍스트 파일 드래그 앤 드롭 핸들러
  const handleTextareaDragEnter = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOverTextarea(true);
  };

  const handleTextareaDragLeave = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    if (!e.currentTarget.contains(e.relatedTarget as Node)) {
      setIsDragOverTextarea(false);
    }
  };

  const handleTextareaDragOver = (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
  };

  const handleTextareaDrop = async (e: React.DragEvent) => {
    e.preventDefault();
    e.stopPropagation();
    setIsDragOverTextarea(false);

    const files = Array.from(e.dataTransfer.files);
    if (files.length === 0) return;

    // 첫 번째 파일만 처리
    const file = files[0];

    // 텍스트 파일인지 확인 (확장자 및 MIME 타입 체크)
    const textExtensions = ['.txt', '.md', '.csv', '.json', '.xml', '.html', '.css', '.js', '.ts', '.py', '.java', '.cpp', '.c', '.h', '.sql', '.log'];
    const fileName = file.name.toLowerCase();
    const isTextFile = textExtensions.some(ext => fileName.endsWith(ext)) ||
      file.type.startsWith('text/') ||
      file.type === 'application/json' ||
      file.type === 'application/xml';

    if (!isTextFile) {
      setError('텍스트 파일만 지원됩니다. (.txt, .md, .csv, .json 등)');
      setTimeout(() => setError(null), 3000);
      return;
    }

    // 파일 크기 체크 (10MB 제한)
    const maxSize = 10 * 1024 * 1024; // 10MB
    if (file.size > maxSize) {
      setError('파일 크기가 너무 큽니다. (최대 10MB)');
      setTimeout(() => setError(null), 3000);
      return;
    }

    try {
      // 파일 내용 읽기
      const text = await file.text();

      // 현재 활성 탭에 따라 적절한 상태 사용
      const currentInputMessage = rightPanelTab === 'create' ? createInputMessage : editInputMessage;
      const setCurrentInputMessage = rightPanelTab === 'create' ? setCreateInputMessage : setEditInputMessage;

      // 기존 내용이 있으면 확인
      if (currentInputMessage.trim()) {
        const shouldReplace = window.confirm('기존 내용을 덮어쓰시겠습니까?\n\n취소를 누르면 기존 내용 뒤에 추가됩니다.');
        if (shouldReplace) {
          setCurrentInputMessage(text);
        } else {
          setCurrentInputMessage(prev => prev + '\n\n' + text);
        }
      } else {
        setCurrentInputMessage(text);
      }

      // 성공 메시지
      setShowToast({ message: `파일 내용이 성공적으로 입력되었습니다. (${file.name})`, type: 'success' });
      setTimeout(() => setShowToast(null), 3000);

    } catch (error) {
      console.error('파일 읽기 오류:', error);
      setError('파일을 읽는 중 오류가 발생했습니다.');
      setTimeout(() => setError(null), 3000);
    }
  };

  // 별도 핸들러로 조건부 로직 완전 제거 (최고 성능)
  const handleCreateInputChange = useCallback((e: React.ChangeEvent<HTMLTextAreaElement>) => {
    setCreateInputMessage(e.target.value);
  }, []);

  const handleEditInputChange = useCallback((e: React.ChangeEvent<HTMLTextAreaElement>) => {
    setEditInputMessage(e.target.value);
  }, []);

  // 현재 모드에 따른 핸들러 선택 (메모이제이션)
  const currentTextareaHandler = useMemo(() =>
    rightPanelTab === 'create' ? handleCreateInputChange : handleEditInputChange,
    [rightPanelTab, handleCreateInputChange, handleEditInputChange]
  );

  // 메모이제이션된 placeholder
  const textareaPlaceholder = useMemo(() =>
    rightPanelTab === 'create'
      ? "작성하고 싶은 내용을 입력하세요... (또는 텍스트 파일을 드래그하세요)"
      : "수정하고 싶은 내용을 구체적으로 입력하세요... (예: '2장에 새로운 그래프 추가', '결론 부분을 더 강화')"
    , [rightPanelTab]);

  // 메모이제이션된 className (transition 제거로 성능 최적화)
  const textareaClassName = useMemo(() => {
    const baseClass = 'w-full min-h-[250px] max-h-60 p-3 border rounded-lg bg-gray-50 dark:bg-gray-800 text-gray-900 dark:text-white placeholder-gray-400 dark:placeholder-gray-500 focus:outline-none focus:ring-2 focus:ring-blue-500 resize-y text-sm';
    const borderClass = isDragOverTextarea
      ? 'border-blue-400 bg-blue-50 dark:bg-blue-900/20'
      : 'border-gray-300 dark:border-gray-600';
    return `${baseClass} ${borderClass}`;
  }, [isDragOverTextarea]);

  // 파일 크기 포맷팅
  const formatFileSize = (bytes: number) => {
    if (bytes === 0) return '0 Bytes';
    const k = 1024;
    const sizes = ['Bytes', 'KB', 'MB', 'GB'];
    const i = Math.floor(Math.log(bytes) / Math.log(k));
    return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
  };



  // 파일 유형별 아이콘 반환
  const getFileIcon = (fileName: string, status: string) => {
    const extension = fileName.split('.').pop()?.toLowerCase() || '';
    const iconClass = "w-4 h-4 flex-shrink-0";

    // 상태별 아이콘 (업로딩/처리 중)
    if (status === 'uploading') {
      return (
        <div className="w-4 h-4 flex-shrink-0">
          <div className="animate-spin rounded-full h-4 w-4 border-2 border-blue-500 border-t-transparent"></div>
        </div>
      );
    }

    if (status === 'processing') {
      return (
        <div className="w-4 h-4 flex-shrink-0">
          <div className="animate-pulse rounded-full h-4 w-4 bg-yellow-500"></div>
        </div>
      );
    }

    if (status === 'error') {
      return (
        <svg className={`${iconClass} text-red-500`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4m0 4h.01M21 12a9 9 0 11-18 0 9 9 0 0118 0z" />
        </svg>
      );
    }

    // 파일 유형별 아이콘
    switch (extension) {
      case 'pdf':
        return (
          <svg className={`${iconClass} text-red-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M8.267 14.68c-.184 0-.308.018-.372.036v1.178c.076.018.171.023.302.023.479 0 .774-.242.774-.651 0-.366-.254-.586-.704-.586zm3.487.012c-.2 0-.33.018-.407.036v2.61c.077.018.201.018.313.018.817.006 1.349-.444 1.349-1.396.006-.83-.479-1.268-1.255-1.268z" />
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM9.498 16.19c-.309.29-.765.42-1.296.42a2.23 2.23 0 0 1-.308-.018v1.426H7v-3.936A7.558 7.558 0 0 1 8.219 14c.557 0 .953.106 1.22.319.254.202.426.533.426.923-.001.392-.131.723-.367.948zm3.807 1.355c-.42.349-1.059.515-1.84.515-.468 0-.799-.03-1.024-.06v-3.917A7.947 7.947 0 0 1 11.66 14c.757 0 1.249.136 1.633.426.415.308.675.799.675 1.504 0 .763-.279 1.29-.663 1.615zM17 14.77h-1.532v.911H16.9v.734h-1.432v1.604h-.906V14.03H17v.74z" />
          </svg>
        );
      case 'doc':
      case 'docx':
        return (
          <svg className={`${iconClass} text-blue-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM16 18H8v-2h8v2zm0-4H8v-2h8v2zm-3-5V3.5L18.5 9H13z" />
          </svg>
        );
      case 'xls':
      case 'xlsx':
        return (
          <svg className={`${iconClass} text-green-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM12 18H8v-2h4v2zm4-4H8v-2h8v2zm0-4H8V8h8v2zm-3-5V3.5L18.5 9H13z" />
          </svg>
        );
      case 'ppt':
      case 'pptx':
        return (
          <svg className={`${iconClass} text-orange-600`} fill="currentColor" viewBox="0 0 24 24">
            <path d="M14 2H6a2 2 0 0 0-2 2v16a2 2 0 0 0 2 2h12a2 2 0 0 0 2-2V8l-6-6zM11 15H9v3H7v-7h4c1.1 0 2 .9 2 2s-.9 2-2 2zm2-6V3.5L18.5 9H13z" />
            <path d="M9 11v2h2c.6 0 1-.4 1-1s-.4-1-1-1H9z" />
          </svg>
        );
      case 'txt':
      case 'md':
        return (
          <svg className={`${iconClass} text-gray-600`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
          </svg>
        );
      case 'jpg':
      case 'jpeg':
      case 'png':
      case 'gif':
      case 'bmp':
      case 'svg':
        return (
          <svg className={`${iconClass} text-purple-600`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16l4.586-4.586a2 2 0 012.828 0L16 16m-2-2l1.586-1.586a2 2 0 012.828 0L20 14m-6-6h.01M6 20h12a2 2 0 002-2V6a2 2 0 00-2-2H6a2 2 0 00-2 2v12a2 2 0 002 2z" />
          </svg>
        );
      case 'zip':
      case 'rar':
      case '7z':
      case 'tar':
      case 'gz':
        return (
          <svg className={`${iconClass} text-yellow-600`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M20 7l-8-4-8 4m16 0l-8 4m8-4v10l-8 4m0-10L4 7m8 4v10M4 7v10l8 4" />
          </svg>
        );
      case 'mp4':
      case 'avi':
      case 'mov':
      case 'wmv':
      case 'flv':
        return (
          <svg className={`${iconClass} text-red-500`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 10l4.553-2.276A1 1 0 0121 8.618v6.764a1 1 0 01-1.447.894L15 14M5 18h8a2 2 0 002-2V8a2 2 0 00-2-2H5a2 2 0 00-2 2v8a2 2 0 002 2z" />
          </svg>
        );
      case 'mp3':
      case 'wav':
      case 'flac':
      case 'aac':
        return (
          <svg className={`${iconClass} text-green-500`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19V6l12-3v13M9 19c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zm12-3c0 1.105-1.343 2-3 2s-3-.895-3-2 1.343-2 3-2 3 .895 3 2zM9 10l12-3" />
          </svg>
        );
      default:
        return (
          <svg className={`${iconClass} text-gray-400`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
            <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
          </svg>
        );
    }
  };

  // 인증 초기화 중인 경우
  if (!isInitialized) {
    return (
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <div
            className="animate-spin rounded-full h-8 w-8 mx-auto mb-4"
            style={{
              border: '2px solid #e5e7eb',
              borderBottomColor: '#ea580c'
            }}
          ></div>
          <p className="text-gray-600 dark:text-gray-400">로딩 중...</p>
        </div>
      </div>
    );
  }

  // 인증되지 않은 경우 (리다이렉트 처리됨)
  if (!isAuthenticated) {
    return null;
  }

  // 사이드바 클릭
  const handleSidebarDocClick = async (report: ReportMetadata) => {
    // reportTitles에서 제목 가져오기, 없으면 fallback 사용
    const displayTitle = reportTitles[report.uuid] || report.file_name || report.project_hash;

    console.log(`[완료문서클릭] 문서 탭 생성 및 수정 모드 활성화: ${report.uuid}`);

    // 1. 탭 생성 및 활성화
    setOpenTabs(prev => {
      if (prev.find(tab => tab.uuid === report.uuid)) return prev;
      return [...prev, { uuid: report.uuid, title: displayTitle, status: report.status }];
    });
    setActiveTab(report.uuid);

    // 2. 문서 내용 로드
    let documentData = tabContents[report.uuid];
    if (!documentData) {
      try {
        const headers = getAuthHeaders();
        const res = await fetch(`/api/document?action=view&uuid=${report.uuid}`, {
          method: 'GET',
          headers
        });

        if (!res.ok) {
          console.error('문서 조회 실패:', res.status, res.statusText);
          return;
        }

        documentData = await res.json();

        // API 응답 구조를 tabContents에 맞게 변환
        const tabContent = {
          type: documentData.type,
          filePath: documentData.filePath,
          fileName: documentData.fileName,
          fileExtension: documentData.fileExtension,
          content: documentData.content // inprogress 문서의 경우
        };

        console.log(`[완료문서클릭] API 응답 변환 완료:`, {
          uuid: report.uuid,
          apiResponse: documentData,
          transformedContent: tabContent,
          hasFilePath: !!tabContent.filePath
        });

        setTabContents(prev => ({ ...prev, [report.uuid]: tabContent }));
      } catch (error) {
        console.error('문서 조회 오류:', error);
        return;
      }
    }

    // 3. 수정 탭 활성화 (섹션 파싱은 useEffect에서 자동 처리)
    enterEditMode(report);
  };



  // 탭 닫기
  const handleCloseTab = (uuid: string) => {
    console.log(`[탭닫기] 탭 닫기: ${uuid}`);

    // 닫히는 탭이 현재 수정 중인 문서인지 확인
    const isEditingThisDocument = editTargetDocument?.uuid === uuid;

    setOpenTabs(prev => prev.filter(tab => tab.uuid !== uuid));
    setTabContents(prev => {
      const copy = { ...prev };
      delete copy[uuid];
      return copy;
    });

    if (activeTab === uuid) {
      // 다른 탭으로 포커스
      const idx = openTabs.findIndex(tab => tab.uuid === uuid);
      const nextTab = openTabs[idx + 1] || openTabs[idx - 1] || null;
      setActiveTab(nextTab ? nextTab.uuid : null);
    }

    // 수정 중이던 문서의 탭이 닫혔으면 생성 모드로 전환
    if (isEditingThisDocument) {
      console.log('[탭닫기] 수정 중이던 문서 탭이 닫혔으므로 생성 모드로 전환');
      enterCreateMode();
    }
  };

  // 전체삭제 함수
  const handleDeleteAllReports = async (status: 'processing' | 'completed') => {
    try {
      const filteredReports = reports.filter(r => r.status === status);

      if (filteredReports.length === 0) {
        console.log(`삭제할 ${status === 'processing' ? '진행중인 프로젝트' : '완료된 문서'}가 없습니다.`);
        return;
      }

      console.log(`전체삭제 시도: ${status} 상태의 ${filteredReports.length}개 문서`);

      const headers = {
        'Content-Type': 'application/json',
        ...getAuthHeaders()
      };

      let successCount = 0;
      let failCount = 0;

      // 각 문서를 개별적으로 삭제
      for (const report of filteredReports) {
        try {
          let res;
          if (report.source === 'cache') {
            res = await fetch('/api/document', {
              method: 'POST',
              headers,
              body: JSON.stringify({ action: 'delete', uuid: report.uuid, source: 'cache' })
            });
          } else if (report.source === 'output') {
            res = await fetch('/api/document', {
              method: 'POST',
              headers,
              body: JSON.stringify({ action: 'delete', uuid: report.uuid, source: 'output' })
            });
          }

          if (res && res.ok) {
            successCount++;
            // 삭제된 문서가 열린 탭에 있으면 닫기
            setOpenTabs(prev => prev.filter(tab => tab.uuid !== report.uuid));
            setTabContents(prev => {
              const copy = { ...prev };
              delete copy[report.uuid];
              return copy;
            });
            if (activeTab === report.uuid) {
              setActiveTab(null);
            }
          } else {
            failCount++;
            console.error(`문서 삭제 실패: ${report.uuid}`, res?.statusText);
          }
        } catch (error) {
          failCount++;
          console.error(`문서 삭제 중 오류: ${report.uuid}`, error);
        }
      }

      // 문서 목록 새로고침
      if (isAuthenticated) {
        await fetchReports();
      }

      // 결과 로그 (alert 제거)
      if (successCount > 0) {
        if (failCount > 0) {
          console.log(`${status === 'processing' ? '진행중인 프로젝트' : '완료된 문서'} ${successCount}개가 삭제되었습니다. (실패: ${failCount}개)`);
        } else {
          console.log(`${status === 'processing' ? '진행중인 프로젝트' : '완료된 문서'} ${successCount}개가 모두 삭제되었습니다.`);
        }
      } else {
        console.error('문서 삭제에 실패했습니다.');
      }

    } catch (error) {
      console.error('전체삭제 중 오류:', error);
    } finally {
      setDeleteAllTarget(null);
    }
  };



  // 📝 문서 섹션 파싱 함수
  const parseDocumentSections = (content: string): DocumentSection[] => {
    if (!content) return [];

    const lines = content.split('\n');
    const sections: DocumentSection[] = [];

    let currentSection: DocumentSection | null = null;

    for (let index = 0; index < lines.length; index++) {
      const line = lines[index];
      // 헤딩 라인 감지 (# ## ### 등)
      const headingMatch = line.match(/^(#{1,6})\s+(.+)$/);

      if (headingMatch) {
        // 이전 섹션 완료
        if (currentSection) {
          const sectionToFinish = currentSection as DocumentSection;
          sectionToFinish.endLine = index - 1;
          sections.push(sectionToFinish);
        }

        // 새 섹션 시작
        const level = headingMatch[1].length;
        const title = headingMatch[2].trim();
        const id = title.toLowerCase()
          .replace(/[^a-z0-9가-힣\s]/g, '')
          .replace(/\s+/g, '-')
          .substring(0, 50);

        currentSection = {
          id: `section-${level}-${id}-${index}`,
          title,
          content: line,
          startLine: index,
        };
      } else if (currentSection) {
        // 현재 섹션에 내용 추가
        currentSection.content += '\n' + line;
      }
    }

    // 마지막 섹션 처리
    if (currentSection) {
      const finalSection = currentSection as DocumentSection;
      finalSection.endLine = lines.length - 1;
      sections.push(finalSection);
    }

    return sections;
  };

  // 📝 수정 모드로 전환
  const enterEditMode = (document: ReportMetadata, section?: string) => {
    console.log(`[수정모드] 문서 수정 모드 진입: ${document.uuid}`, { section });

    setEditMode('edit');
    setEditTargetDocument(document);
    setEditTargetSection(section || null);
    setEditMethod('full'); // 수정 방식 초기화

    // 수정 탭으로 전환
    setRightPanelTab('edit');

    // 수정 탭의 프롬프트 영역 초기화
    if (section) {
      setEditInputMessage(`"${section}" 섹션을 다음과 같이 수정해주세요:\n\n`);
      setEditMethod('section'); // 섹션이 지정되면 섹션 수정 모드로
    } else {
      setEditInputMessage(`다음 부분을 수정해주세요:\n\n`);
      setEditMethod('full'); // 기본은 전체 수정 모드
    }

    // 섹션 파싱은 useEffect에서 자동으로 처리됨
  };

  // 📝 새 문서 모드로 전환
  const enterCreateMode = () => {
    console.log('[수정모드] 새 문서 생성 모드로 전환');

    setEditMode('create');
    setEditTargetDocument(null);
    setEditTargetSection(null);
    setDocumentSections([]);
    setEditMethod('full'); // 수정 방식 초기화

    // 생성 탭으로 전환
    setRightPanelTab('create');

    // 생성 탭 상태 초기화
    setCreateInputMessage('');
    setCreateSelectedTemplate('');
  };

  // 컨텍스트 메뉴 핸들러
  const handleDocumentRightClick = (e: React.MouseEvent, document: ReportMetadata) => {
    e.preventDefault();
    e.stopPropagation();

    // 화면 경계 내에서 메뉴 위치 조정
    const menuWidth = 180;
    const menuHeight = 100;
    let x = e.clientX;
    let y = e.clientY;

    if (x + menuWidth > window.innerWidth) {
      x = window.innerWidth - menuWidth - 10;
    }
    if (y + menuHeight > window.innerHeight) {
      y = window.innerHeight - menuHeight - 10;
    }

    setContextMenu({
      show: true,
      x,
      y,
      document
    });
  };

  // 컨텍스트 메뉴 닫기
  const closeContextMenu = () => {
    setContextMenu({ show: false, x: 0, y: 0, document: null });
  };

  // 컨텍스트 메뉴에서 수정 모드로 전환
  const enterEditModeFromContext = (document: ReportMetadata) => {
    closeContextMenu();
    enterEditMode(document);
  };

  // 테이블 스키마 에디터 관련 함수들
  const openTableSchemaEditor = (schema: any = null) => {
    setCurrentTableSchema(schema);
    setShowTableSchemaEditor(true);
  };

  const closeTableSchemaEditor = () => {
    setShowTableSchemaEditor(false);
    setCurrentTableSchema(null);
  };

  const handleTableSchemaChange = (schema: any) => {
    setCurrentTableSchema(schema);
    console.log('테이블 스키마 업데이트:', schema);
    // 여기서 스키마를 저장하거나 적용하는 로직을 추가할 수 있습니다
  };



  return (
    <ClientOnly fallback={
      <div className="flex items-center justify-center min-h-screen">
        <div className="text-center">
          <div
            className="animate-spin rounded-full h-8 w-8 mx-auto mb-4"
            style={{
              border: '2px solid #e5e7eb',
              borderBottomColor: '#ea580c'
            }}
          ></div>
          <p className="text-gray-600 dark:text-gray-400">로딩 중...</p>
        </div>
      </div>
    }>
      {/* 3분할 레이아웃: 좌(사이드바) - 중(문서) - 우(프롬프트) */}
      <div className="h-screen flex" style={{ backgroundColor: 'var(--body-bg)' }}>
        {/* --- 왼쪽: 사이드바(기존) --- */}
        <div
          className={`w-80 h-screen border-r border-gray-200 dark:border-gray-900 flex flex-col overflow-y-auto`}
          style={{ backgroundColor: 'var(--sidebar-bg)' }}
        >


          {/* 컨테이너 1: 내 문서 */}
          <div className="border-b border-gray-200 dark:border-gray-900 p-2">
            {/* 내 문서 헤더 - 메뉴 스타일 */}
            <div className="flex items-center justify-between px-4 py-2 border-b border-gray-300 dark:border-gray-600 font-bold text-sm text-gray-900 dark:text-white select-none" style={{ backgroundColor: 'var(--card-bg)' }}>
              <div className="flex items-center gap-2">
                <svg className="w-4 h-4 text-gray-600 dark:text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                </svg>
                내 문서
              </div>
              <button
                onClick={() => setShowRagBrowser(!showRagBrowser)}
                className="p-1 text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 rounded transition-colors"
                style={{ backgroundColor: 'transparent' }}
                onMouseEnter={(e) => e.currentTarget.style.backgroundColor = 'var(--card-bg)'}
                onMouseLeave={(e) => e.currentTarget.style.backgroundColor = 'transparent'}
              >
                <svg className={`w-4 h-4 transition-transform ${showRagBrowser ? 'rotate-180' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                </svg>
              </button>
            </div>

              <div className="" style={{ backgroundColor: 'var(--body-bg)', borderRadius: '8px' }}>
              {showRagBrowser && (
                <div className="space-y-3 min-h-[380px] max-h-80 flex flex-col pt-2 pl-2 pr-2">
                  {/* 드래그 앤 드롭 영역 */}
                  <div
                    onDragEnter={handleRagDragEnter}
                    onDragLeave={handleRagDragLeave}
                    onDragOver={handleRagDragOver}
                    onDrop={handleRagDrop}
                    className={`border-2 border-dashed rounded-lg p-4 text-center transition-colors ${isDragOverRag
                      ? 'border-blue-400 bg-blue-50 dark:bg-blue-900/20'
                      : 'border-gray-300 dark:border-gray-600 hover:border-gray-400 dark:hover:border-gray-500'
                      }`}
                  >
                    <svg className="w-8 h-8 mx-auto mb-2 text-gray-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 16a4 4 0 01-.88-7.903A5 5 0 1115.9 6L16 6a5 5 0 011 9.9M15 13l-3-3m0 0l-3 3m3-3v12" />
                    </svg>
                    <input
                      type="file"
                      multiple
                      ref={fileInputRef}
                      onChange={handleFileSelect}
                      className="hidden"
                    />
                    <button
                      type="button"
                      className="mt-2 px-3 py-1 bg-blue-500 text-white rounded hover:bg-blue-600 text-sm"
                      onClick={() => fileInputRef.current?.click()}
                    >
                      파일 업로드
                    </button>
                    <div className="text-xs text-gray-500 mt-2">또는 파일을 이곳에 드래그하세요</div>
                  </div>
                  {/* 백그라운드 작업 상태 */}
                  <div className="px-2">
                    <BackgroundJobStatus
                      userId={user?.username}
                      onJobComplete={handleJobComplete}
                      refreshTrigger={refreshTrigger}
                    />
                  </div>
                  {/* 파일 목록 */}
                  <div className="max-h-80 overflow-y-auto pb-5 text-sm">
                    {isLoadingRagFiles ? (
                      <div className="flex justify-center items-center h-20">
                        <div className="relative">
                          <div
                            className="animate-spin rounded-full h-4 w-4 dark:hidden"
                            style={{
                              border: '2px solid #e5e7eb',
                              borderBottomColor: '#ea580c'
                            }}
                          ></div>
                          <div
                            className="animate-spin rounded-full h-4 w-4 hidden dark:block"
                            style={{
                              border: '2px solid #4b5563',
                              borderBottomColor: '#fb923c'
                            }}
                          ></div>
                        </div>
                      </div>
                    ) : ragFiles.length === 0 ? (
                      <div className="text-center text-sm text-gray-500 dark:text-gray-400 py-4">
                        업로드된 문서가 없습니다
                      </div>
                    ) : (
                      <div className="space-y-1">
                        {ragFiles.map((file, index) => (
                          <div
                            key={`${file.name}-${index}`}
                            className="group flex flex-col p-1.5 rounded-md transition-colors hover:bg-opacity-10"
                            style={{ backgroundColor: 'transparent' }}
                            onMouseEnter={(e) => e.currentTarget.style.backgroundColor = 'var(--card-bg)'}
                            onMouseLeave={(e) => e.currentTarget.style.backgroundColor = 'transparent'}
                          >
                            <div className="flex items-center space-x-2">
                              {/* 파일 유형별 아이콘 */}
                              {getFileIcon(file.name, file.status)}

                              <div className="flex-1 min-w-0">
                                <p
                                  className="file-browser-filename text-gray-700 dark:text-gray-300 truncate"
                                  title={file.name}
                                >
                                  {file.name}
                                </p>
                                {/* 상태 텍스트 */}
                                {(file.status === 'uploading' || file.status === 'processing' || file.status === 'error') && (
                                  <div className="flex items-center space-x-2 mt-1">
                                    {file.status === 'uploading' && (
                                      <span className="file-browser-status text-blue-500 text-xs">업로드 중...</span>
                                    )}
                                    {file.status === 'processing' && (
                                      <span className="file-browser-status text-yellow-500 text-xs">처리 중...</span>
                                    )}
                                    {file.status === 'error' && (
                                      <span className="file-browser-status text-red-500 text-xs">{file.error || '오류'}</span>
                                    )}
                                  </div>
                                )}
                              </div>

                              {/* 삭제 버튼 (완료된 파일만) */}
                              {file.status === 'completed' && (
                                <button
                                  onClick={() => deleteRagFile(file.name)}
                                  className="opacity-0 group-hover:opacity-100 p-1 text-gray-400 hover:text-red-500 dark:hover:text-red-400 transition-opacity flex-shrink-0"
                                  title="파일 삭제"
                                >
                                  <svg className="w-3.5 h-3.5" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                  </svg>
                                </button>
                              )}
                            </div>

                            {/* 프로그레스 바 (업로딩/처리 중일 때만) */}
                            {(file.status === 'uploading' || file.status === 'processing') && (
                              <div className="mt-2">
                                <div className="w-full bg-gray-200 dark:bg-gray-600 rounded-full h-1.5">
                                  <div
                                    className={`h-1.5 rounded-full transition-all duration-300 ${file.status === 'uploading' ? 'bg-blue-500' : 'bg-yellow-500'
                                      }`}
                                    style={{ width: `${file.progress || 0}%` }}
                                  ></div>
                                </div>
                              </div>
                            )}
                          </div>
                        ))}
                      </div>
                    )}
                  </div>
                </div>
              )}
            </div>
          </div>

          {/* 스크롤 영역: 진행중 작업 + 문서목록 + 템플릿 업로드 */}
          <div className="flex flex-col overflow-y-auto max-h-[calc(100vh-300px)]">
            {/* 컨테이너 1: 진행 중인 작업 (데이터베이스) */}
            <div className="flex flex-col border-b border-gray-200 dark:border-gray-700 p-2">
              {/* 진행 중인 작업 헤더 */}
              <div className="flex items-center px-4 py-2 bg-orange-100 dark:bg-orange-900/30 border-b border-orange-300 dark:border-orange-600 font-bold text-sm text-orange-900 dark:text-orange-200 select-none">
                <div className="w-2 h-2 bg-orange-500 rounded-full animate-pulse mr-2"></div>
                <svg className="w-4 h-4 mr-2 text-orange-600 dark:text-orange-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 8v4l3 3m6-3a9 9 0 11-18 0 9 9 0 0118 0z" />
                </svg>
                진행 중인 작업
              </div>
              <div className="p-3" style={{ backgroundColor: 'var(--body-bg)', borderRadius: '8px' }}>
                {/* 데이터베이스에서 가져온 진행 중인 작업들 (현재 사용자만) */}
                {activeJobs
                  .sort((a, b) => {
                    // created_at 또는 updated_at 기준 내림차순 정렬 (최신 순)
                    const timeA = new Date(a.updated_at || a.created_at || 0).getTime();
                    const timeB = new Date(b.updated_at || b.created_at || 0).getTime();
                    return timeB - timeA;
                  })
                  .map((job) => {
                    // 메타데이터 제거 후 제목 생성
                    const cleanSummary = job.executive_summary ? removeMetadataFromContent(job.executive_summary) : '';
                    const displayTitle = job.title || cleanSummary.slice(0, 50) || job.job_id;
                    const shortTitle = displayTitle.length > 25 ? displayTitle.slice(0, 25) + '...' : displayTitle;
                    const isCurrentTask = currentTaskId === job.job_id;

                    return (
                      <div
                        key={job.job_id}
                        className={`mb-2 p-3 border rounded-lg cursor-pointer transition-colors ${isCurrentTask && isStreamingActive
                          ? 'bg-blue-50 dark:bg-blue-900/20 border-blue-200 dark:border-blue-700 ring-2 ring-blue-300 dark:ring-blue-600'
                          : 'bg-yellow-50 dark:bg-yellow-900/20 border-yellow-200 dark:border-yellow-700 hover:bg-yellow-100 dark:hover:bg-yellow-900/30'
                          }`}
                        onClick={async () => {
                          try {
                            console.log(`[진행중작업클릭] 작업 상태 확인 중: ${job.job_id}`);

                            // 0. 같은 작업이면 중복 처리 방지
                            if (currentTaskId === job.job_id) {
                              console.log(`[진행중작업클릭] 이미 연결된 작업입니다: ${job.job_id.slice(-8)}`);
                              return;
                            }

                            // 1. 먼저 작업 상태 확인 (프록시 경로 사용)
                            const statusResponse = await fetch(`/report/status/${job.job_id}`);
                            const statusData = await statusResponse.json();

                            if (statusData.success && (statusData.status === 'processing' || statusData.status === 'queued')) {
                              console.log(`[진행중작업클릭] 유효한 진행 중인 작업 확인됨: ${job.job_id}`);

                              // 2. 기존 SSE 연결 정리 (중요!)
                              if (currentTaskId && currentTaskId !== job.job_id) {
                                console.log(`[진행중작업클릭] 기존 SSE 연결 정리: ${currentTaskId.slice(-8)} → ${job.job_id.slice(-8)}`);
                                stopSseStreaming(true); // 정상적인 종료로 처리
                              }

                              // 3. 스트리밍 탭이 이미 있는지 확인
                              const streamingTabId = `streaming-${job.job_id}`;
                              const existingTab = openTabs.find(tab => tab.uuid === streamingTabId);

                              if (existingTab) {
                                // 기존 스트리밍 탭으로 전환
                                console.log(`[진행중작업클릭] 기존 스트리밍 탭으로 전환: ${streamingTabId}`);
                                setActiveTab(streamingTabId);
                              } else {
                                // 새 스트리밍 탭 생성
                                console.log(`[진행중작업클릭] 새 스트리밍 탭 생성: ${streamingTabId}`);
                                // 메타데이터가 제거된 제목 사용
                                const cleanSummary = job.executive_summary ? removeMetadataFromContent(job.executive_summary) : '';
                                const jobTitle = cleanSummary ? cleanSummary.slice(0, 25) : `작업 ${job.job_id.slice(-8)}`;
                                const streamingTitle = `${jobTitle}... (진행중)`;

                                setOpenTabs(prev => [...prev, { uuid: streamingTabId, title: streamingTitle, status: 'processing' }]);
                                setActiveTab(streamingTabId);

                                // 스트리밍 탭 내용 설정
                                setTabContents(prev => ({
                                  ...prev,
                                  [streamingTabId]: {
                                    type: 'streaming',
                                    jobId: job.job_id,
                                    content: ''
                                  }
                                }));
                              }

                              // 4. 진행 중인 작업으로 상태 설정
                              setCurrentTaskId(job.job_id);
                              setTaskStartTime(Date.now());

                              // 5. 잠깐 대기 후 SSE 스트리밍 시작 (안전한 연결을 위해)
                              setTimeout(() => {
                                console.log(`[진행중작업클릭] SSE 스트리밍 시작: ${job.job_id.slice(-8)}`);
                                startSseStreaming(job.job_id);
                              }, 100);

                              // 6. 상태 메시지 업데이트
                              setProcessingStatus(`🔄 진행 중인 작업에 연결했습니다: ${job.job_id.slice(-8)}`);

                            } else if (statusData.success && statusData.status === 'completed') {
                              console.log(`[진행중작업클릭] 완료된 작업: ${job.job_id}`);

                              // 완료된 작업은 문서 목록 새로고침
                              await fetchReports();
                              await fetchActiveJobs();

                              setProcessingStatus(`✅ 작업이 이미 완료되었습니다: ${job.job_id.slice(-8)}`);

                              // 3초 후 메시지 정리
                              setTimeout(() => {
                                setProcessingStatus('');
                              }, 3000);

                            } else {
                              console.warn(`[진행중작업클릭] 작업 상태 불명 또는 오류: ${job.job_id}`, statusData);
                              setProcessingStatus(`⚠️ 작업 상태를 확인할 수 없습니다: ${job.job_id.slice(-8)}`);

                              // 3초 후 메시지 정리
                              setTimeout(() => {
                                setProcessingStatus('');
                              }, 3000);
                            }
                          } catch (error) {
                            console.error('[진행중작업클릭] 작업 상태 확인 중 오류:', error);
                            setProcessingStatus(`❌ 작업 상태 확인 중 오류 발생: ${job.job_id.slice(-8)}`);

                            // 3초 후 메시지 정리
                            setTimeout(() => {
                              setProcessingStatus('');
                            }, 3000);
                          }
                        }}
                        title={displayTitle}
                      >
                        <div className="flex items-center justify-between mb-2">
                          <div className="flex items-center gap-2">
                            {isCurrentTask && isStreamingActive ? (
                              <div className="animate-spin w-4 h-4 border-2 border-blue-600 border-t-transparent rounded-full"></div>
                            ) : (
                              <div className="w-2 h-2 bg-yellow-500 rounded-full animate-pulse"></div>
                            )}
                            <span className={`text-sm font-medium ${isCurrentTask && isStreamingActive
                              ? 'text-blue-800 dark:text-blue-200'
                              : 'text-yellow-800 dark:text-yellow-200'
                              }`}>
                              {isCurrentTask && isStreamingActive ? '문서 작성 중' :
                                job.status === 'processing' ? '진행 중' :
                                  job.status === 'queued' ? '대기 중' : job.status}
                            </span>
                          </div>
                          <span className="text-xs text-gray-500 dark:text-gray-400">
                            {job.created_at ? new Date(job.created_at).toLocaleDateString() : ''}
                          </span>
                        </div>
                        <div className="text-sm text-gray-600 dark:text-gray-300 truncate">
                          {shortTitle}
                        </div>
                        <div className="text-xs text-gray-500 dark:text-gray-400 mt-1">
                          진행률: {job.progress || 0}%
                        </div>
                        {isCurrentTask && isStreamingActive && (
                          <div className="mt-2 text-xs text-blue-600 dark:text-blue-400 flex items-center gap-1">
                            <div className="w-1 h-1 bg-blue-500 rounded-full animate-pulse"></div>
                            실시간 진행 상황 표시 중
                          </div>
                        )}
                      </div>
                    );
                  })}

                {/* 진행 중인 작업이 없는 경우 */}
                {!currentTaskId && activeJobs.length === 0 && (
                  <div className="text-center py-6 text-gray-500 dark:text-gray-400">
                    <div className="text-2xl mb-2">📝</div>
                    <p className="text-xs">진행 중인 작업이 없습니다</p>
                  </div>
                )}
              </div>
            </div>

            {/* 컨테이너 2: 문서(리포트) 목록 */}
            <div className="flex flex-col border-b border-gray-200 dark:border-gray-700 p-2">
              {/* 요청 문서 목록 헤더 - 메뉴 스타일 */}
              <div
                className="flex items-center px-4 py-2 border-b border-gray-300 dark:border-gray-600 font-bold text-sm text-gray-900 dark:text-white select-none cursor-pointer"
                style={{ backgroundColor: 'var(--card-bg)' }}
                onMouseEnter={(e) => e.currentTarget.style.backgroundColor = 'var(--sidebar-bg)'}
                onMouseLeave={(e) => e.currentTarget.style.backgroundColor = 'var(--card-bg)'}
                onClick={() => setIsOpenReports(v => !v)}
              >
                <svg className="w-4 h-4 mr-2 text-gray-600 dark:text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                </svg>
                프로젝트 목록
                <button
                  className="p-1 ml-auto text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 rounded transition-colors"
                  style={{ backgroundColor: 'transparent' }}
                  onMouseEnter={(e) => e.currentTarget.style.backgroundColor = 'var(--card-bg)'}
                  onMouseLeave={(e) => e.currentTarget.style.backgroundColor = 'transparent'}
                  tabIndex={-1}
                  type="button"
                  onClick={e => { e.stopPropagation(); setIsOpenReports(v => !v); }}
                  aria-label="폴딩 토글"
                >
                  <svg className={`w-4 h-4 transition-transform ${isOpenReports ? 'rotate-180' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                  </svg>
                </button>
              </div>
              {isOpenReports && (
                <div className="flex-1 overflow-y-auto min-h-0 pb-2" style={{ backgroundColor: 'var(--body-bg)', borderRadius: '8px' }}>
                  {isLoadingReports ? (
                    <div className="flex justify-center items-center h-32">
                      <div className="relative">
                        <div
                          className="animate-spin rounded-full h-6 w-6 dark:hidden"
                          style={{
                            border: '2px solid #e5e7eb',
                            borderBottomColor: '#ea580c'
                          }}
                        ></div>
                        <div
                          className="animate-spin rounded-full h-6 w-6 hidden dark:block"
                          style={{
                            border: '2px solid #4b5563',
                            borderBottomColor: '#fb923c'
                          }}
                        ></div>
                      </div>
                    </div>
                  ) : reports.length === 0 ? (
                    <div className="p-3 text-center text-xs text-gray-500 dark:text-gray-400">
                      문서가 없습니다
                    </div>
                  ) : (
                    <div className="space-y-0.5 p-1 min-h-[30vh]">
                      {['processing', 'completed'].map(status => (
                        <div key={status}>
                          <div className="pl-2 flex items-center justify-between text-sm font-bold border border-gray-300 dark:border-gray-600 rounded px-3 py-2 bg-transparent text-gray-700 dark:text-gray-200 cursor-default select-none mb-2 mt-4">
                            <span>{status === 'processing' ? '프로젝트' : status === 'completed' ? '완료된 문서' : '실패'}</span>
                            {reports.filter(r => r.status === status).length > 0 && (
                              <button
                                onClick={() => setDeleteAllTarget(status as 'processing' | 'completed')}
                                className="text-xs px-2 py-1 bg-red-500 hover:bg-red-600 text-white rounded transition-colors ml-2"
                                title={`모든 ${status === 'processing' ? '진행중인 프로젝트' : '완료된 문서'} 삭제`}
                              >
                                전체삭제
                              </button>
                            )}
                          </div>
                          {reports.filter(r => r.status === status).length === 0 ? (
                            <div className="text-xs text-gray-400 dark:text-gray-600 px-2 py-1 text-center rounded-md mx-2 my-1" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
                              {status === 'processing' ? '📝 진행중인 문서가 없습니다' : '📄 완료된 문서가 없습니다'}
                            </div>
                          ) : (
                            reports
                              .filter(r => r.status === status)
                              .sort((a, b) => {
                                // 1차: timestamp 기준 내림차순 정렬 (최신 순)
                                const timeA = new Date(a.timestamp || 0).getTime();
                                const timeB = new Date(b.timestamp || 0).getTime();

                                // timestamp가 유효한 경우 timestamp로 정렬
                                if (timeA > 0 && timeB > 0) {
                                  return timeB - timeA;
                                }

                                // timestamp가 없는 경우 project_hash(uuid)로 정렬 (최신 것이 보통 더 긴 해시값)
                                if (timeA === 0 && timeB === 0) {
                                  return (b.project_hash || b.uuid || '').localeCompare(a.project_hash || a.uuid || '');
                                }

                                // 하나만 timestamp가 있는 경우, timestamp가 있는 것을 우선
                                return timeB > 0 ? 1 : -1;
                              })
                              .map((report) => {
                                let dateStr = '날짜 정보 없음';
                                if (report.timestamp) {
                                  const d = new Date(report.timestamp);
                                  dateStr = isNaN(d.getTime()) ? '날짜 정보 없음' : `${d.getFullYear()}.${(d.getMonth() + 1).toString().padStart(2, '0')}.${d.getDate().toString().padStart(2, '0')}`;
                                }
                                const fullName = reportTitles[report.uuid] || report.file_name || report.project_hash;
                                const shortName = fullName.length > 20 ? fullName.slice(0, 20) + '…' : fullName;





                                return (
                                  <div key={report.uuid + (report.file_name || '')} className="flex flex-col">
                                    <div
                                      onClick={() => handleSidebarDocClick(report)}
                                      onContextMenu={(e) => handleDocumentRightClick(e, report)}
                                      className={`group flex items-center px-2 py-2 rounded-md cursor-pointer transition-colors w-full
                                          ${activeTab === report.uuid
                                          ? 'text-blue-900 dark:text-white'
                                          : 'text-gray-900 dark:text-white'}
                                        `}
                                      style={{ 
                                        backgroundColor: activeTab === report.uuid ? 'var(--card-bg)' : 'transparent'
                                      }}
                                      onMouseEnter={(e) => {
                                        if (activeTab !== report.uuid) {
                                          e.currentTarget.style.backgroundColor = 'var(--card-bg)';
                                        }
                                      }}
                                      onMouseLeave={(e) => {
                                        if (activeTab !== report.uuid) {
                                          e.currentTarget.style.backgroundColor = 'transparent';
                                        }
                                      }}
                                    >
                                      <span className="mr-2 text-sm">{getFileIcon(report.file_name || '', 'completed')}</span>
                                      <div className="flex-1 min-w-0">
                                        <div className="flex items-center">
                                          <span className="text-xs truncate" title={fullName}>
                                            {shortName}
                                          </span>
                                        </div>
                                        {dateStr !== '날짜 정보 없음' && (
                                          <div className="text-xs text-gray-500 dark:text-gray-400 mt-0.5">
                                            {dateStr}
                                          </div>
                                        )}
                                      </div>
                                      <button
                                        onClick={e => {
                                          e.stopPropagation();
                                          setDeleteTarget(report);
                                        }}
                                        className="opacity-0 group-hover:opacity-100 ml-2 p-1 text-gray-400 hover:text-red-500 dark:hover:text-red-400 transition-all flex-shrink-0"
                                        title="문서 삭제"
                                      >
                                        <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                                        </svg>
                                      </button>
                                    </div>
                                  </div>
                                );
                              })
                          )}
                        </div>
                      ))}
                    </div>
                  )}
                </div>
              )}
            </div>

            {/* 템플릿 업로드 (요청 문서 목록 바로 아래) */}
            <div className="p-2">
              {/* 템플릿 업로드 헤더 - 메뉴 스타일 */}
              <div
                className="px-4 py-2 border-t border-b border-gray-300 dark:border-gray-600 font-bold text-sm text-gray-900 dark:text-white flex items-center gap-2 select-none cursor-pointer"
                style={{ backgroundColor: 'var(--card-bg)' }}
                onMouseEnter={(e) => e.currentTarget.style.backgroundColor = 'var(--sidebar-bg)'}
                onMouseLeave={(e) => e.currentTarget.style.backgroundColor = 'var(--card-bg)'}
                onClick={() => setIsOpenTemplate(v => !v)}
              >
                <svg className="w-5 h-5 text-gray-600 dark:text-gray-300" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M4 16v1a3 3 0 003 3h10a3 3 0 003-3v-1m-4-8l-4-4m0 0L8 8m4-4v12" />
                </svg>
                템플릿 업로드
                <button
                  className="p-1 ml-auto text-gray-500 hover:text-gray-700 dark:text-gray-400 dark:hover:text-gray-200 rounded transition-colors"
                  style={{ backgroundColor: 'transparent' }}
                  onMouseEnter={(e) => e.currentTarget.style.backgroundColor = 'var(--card-bg)'}
                  onMouseLeave={(e) => e.currentTarget.style.backgroundColor = 'transparent'}
                  tabIndex={-1}
                  type="button"
                  onClick={e => { e.stopPropagation(); setIsOpenTemplate(v => !v); }}
                  aria-label="폴딩 토글"
                >
                  <svg className={`w-4 h-4 transition-transform ${isOpenTemplate ? 'rotate-180' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                    <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 9l-7 7-7-7" />
                  </svg>
                </button>
              </div>
              {isOpenTemplate && (
                <div className="p-3 border-t-0 border-gray-200 dark:border-gray-700 min-h-[80px] max-h-60 flex flex-col" style={{ backgroundColor: 'var(--body-bg)', borderRadius: '8px' }}>
                  <div className="flex items-center gap-2">
                    <input
                      ref={templateFileInputRef}
                      type="file"
                      accept="application/json,.json"
                      onChange={handleTemplateFileChange}
                      className="block w-full text-xs text-gray-700 dark:text-gray-200 file:mr-2 file:py-1 file:px-2 file:rounded file:border-0 file:text-xs file:bg-blue-50 file:text-blue-700 hover:file:bg-blue-100"
                      disabled={isUploadingTemplate}
                    />
                  </div>
                  {isUploadingTemplate && (
                    <div className="mt-2 text-xs text-blue-500">업로드 중...</div>
                  )}
                  {templateUploadMessage && (
                    <div className={`mt-2 text-xs ${templateUploadMessage.startsWith('✅') ? 'text-green-600' : 'text-red-500'}`}>{templateUploadMessage}</div>
                  )}
                </div>
              )}
            </div>
          </div>

        </div>

        {/* --- 중앙: 문서 미리보기/에디터 (탭) --- */}
        <div
          className="flex-1 flex flex-col min-w-0 border-r border-gray-200 dark:border-gray-700 relative h-full"
          style={{ 
            maxHeight: "calc(100vh - 64px)",
            backgroundColor: 'var(--body-bg)'
          }}
        >
          {/* 탭 바 */}
          <div className="flex border-b border-gray-200 dark:border-gray-700 text-xs" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
            {openTabs.map(tab => {
              // 탭의 파일 유형 아이콘 결정
              const getTabIcon = (uuid: string) => {
                const content = tabContents[uuid];
                if (!content) return '📄';

                switch (content.type) {
                  case 'pdf': return '📄';
                  case 'hwpx': return '📝';
                  case 'docx': return '📘';
                  case 'pptx': return '📊';
                  case 'streaming': return '📄'; // 스트리밍 문서
                  default: return '📝'; // 기타 문서 (마크다운)
                }
              };

              return (
                <div
                  key={tab.uuid}
                  className={`flex items-center px-3 py-2 cursor-pointer select-none border-r border-gray-200 dark:border-gray-700
                    ${activeTab === tab.uuid ? 'font-bold' : ''}`}
                  style={{ 
                    backgroundColor: activeTab === tab.uuid ? 'var(--card-bg)' : 'var(--sidebar-bg)'
                  }}
                  onClick={() => {
                    console.log(`[탭클릭] 탭 전환: ${tab.uuid}`);

                    // 스트리밍 탭이 아닌 경우에만 스트리밍 상태 정리
                    if (!tab.uuid.startsWith('streaming-')) {
                      console.log(`[탭클릭] 완료된 문서 탭으로 전환 - 스트리밍 상태 정리`);
                      stopSseStreaming(true);
                      setCurrentTaskId(null);
                      setTaskStartTime(null);
                      setStreamingSections({});
                      setStreamingProgress(0);
                      setStreamingMessage('');
                      setProcessingStatus('');
                    } else {
                      console.log(`[탭클릭] 스트리밍 탭으로 전환 - 스트리밍 상태 유지`);
                    }

                    setActiveTab(tab.uuid);
                  }}
                  title={tab.title}
                >
                  <span className="mr-1">{getTabIcon(tab.uuid)}</span>
                  <span className="truncate max-w-[100px]">{tab.title}</span>
                  <button
                    onClick={e => { e.stopPropagation(); handleCloseTab(tab.uuid); }}
                    className="ml-2 text-gray-400 hover:text-red-500"
                    title="닫기"
                  >×</button>
                </div>
              );
            })}
          </div>
          {/* 탭 내용 - 스크롤바 위치 안정화 */}
          <div
            ref={streamingContainerRef}
            className="flex-1 overflow-y-auto p-8 flex flex-col items-start justify-start relative max-w-none"
            style={{
              scrollbarGutter: 'stable',
              overflowAnchor: 'none'
            }}
          >
            {/* 탭 내용 - 최적화된 컴포넌트 */}
            <OptimizedDocumentViewer
              activeTab={activeTab}
              tabContents={tabContents}
              streamingSections={streamingSections}
              streamingMessage={streamingMessage}
              streamingProgress={streamingProgress}
              sectionRefs={sectionRefs}
              ThinkingMessage={ThinkingMessage}
              DocumentRenderer={DocumentRenderer}
              handleDownload={handleDownload}
            />
            {!activeTab && !isStreamingActive && (
              <div className="w-full text-center text-gray-400 dark:text-gray-500 py-24">
                <h3 className="text-xl font-bold mb-2">선택된 문서가 없습니다.</h3>
                <p>문서를 생성하거나, 기존 문서를 선택하세요.</p>
              </div>
            )}
          </div>
        </div>

        {/* --- 리사이저 바 --- */}
        <div
          ref={resizerRef}
          onMouseDown={() => setIsResizing(true)}
          className={`w-2 cursor-col-resize select-none bg-gray-200 dark:bg-gray-700 hover:bg-blue-400 dark:hover:bg-blue-500 transition-colors duration-150 z-10`}
          style={{ cursor: isResizing ? 'col-resize' : 'ew-resize' }}
        >
          <div className="h-full w-1 mx-auto bg-gray-400 dark:bg-gray-600 rounded" style={{ marginTop: 8, marginBottom: 8 }} />
        </div>

        {/* --- 오른쪽: 프롬프트/AI 입력 패널 --- */}
        <div
          className="flex flex-col border-l border-gray-200 dark:border-gray-700"
          style={{ 
            width: rightPanelWidth, 
            minWidth: 280, 
            maxWidth: 700,
            backgroundColor: 'var(--sidebar-bg)'
          }}
        >
          <div className="flex-1 flex flex-col">
            {/* 탭 헤더 */}
            <div className="flex border-b border-gray-200 dark:border-gray-700" style={{ backgroundColor: 'var(--card-bg)' }}>
              <button
                onClick={() => {
                  setRightPanelTab('create');
                  // 생성 탭으로 전환 시 수정 관련 상태 초기화
                  setEditMode('create');
                  setEditTargetDocument(null);
                  setEditTargetSection(null);
                  setEditMethod('full');
                }}
                className={`flex-1 px-4 py-3 text-sm font-medium transition-colors border-b-2 ${
                  rightPanelTab === 'create'
                    ? 'text-blue-700 dark:text-blue-300 border-blue-600 dark:border-blue-400 shadow-sm'
                    : 'text-gray-700 dark:text-gray-300 border-transparent hover:text-gray-900 dark:hover:text-gray-100'
                }`}
                style={{
                  backgroundColor: rightPanelTab === 'create' 
                    ? 'var(--card-bg)' 
                    : 'transparent'
                }}
                onMouseEnter={(e) => {
                  if (rightPanelTab !== 'create') {
                    e.currentTarget.style.backgroundColor = 'var(--sidebar-bg)';
                  }
                }}
                onMouseLeave={(e) => {
                  if (rightPanelTab !== 'create') {
                    e.currentTarget.style.backgroundColor = 'transparent';
                  }
                }}
              >
                🆕 문서 생성
              </button>
              <button
                onClick={() => {
                  setRightPanelTab('edit');
                  // 수정 방식을 기본값으로 초기화
                  setEditMethod('full');
                  // 수정 탭으로 전환할 때 현재 활성 문서가 있으면 자동으로 수정 모드로 설정
                  if (activeTab && reports.length > 0) {
                    const activeReport = reports.find(r => r.uuid === activeTab);
                    if (activeReport && !editTargetDocument) {
                      setEditTargetDocument(activeReport);
                      setEditMode('edit');
                      // 섹션 파싱은 useEffect에서 자동으로 처리됨
                    }
                  }
                }}
                className={`flex-1 px-4 py-3 text-sm font-medium transition-colors border-b-2 ${
                  rightPanelTab === 'edit'
                    ? 'text-orange-700 dark:text-orange-300 border-orange-600 dark:border-orange-400 shadow-sm'
                    : 'text-gray-700 dark:text-gray-300 border-transparent hover:text-gray-900 dark:hover:text-gray-100'
                }`}
                style={{
                  backgroundColor: rightPanelTab === 'edit' 
                    ? 'var(--card-bg)' 
                    : 'transparent'
                }}
                onMouseEnter={(e) => {
                  if (rightPanelTab !== 'edit') {
                    e.currentTarget.style.backgroundColor = 'var(--sidebar-bg)';
                  }
                }}
                onMouseLeave={(e) => {
                  if (rightPanelTab !== 'edit') {
                    e.currentTarget.style.backgroundColor = 'transparent';
                  }
                }}
              >
                📝 문서 수정
              </button>
            </div>

            {/* 탭 컨텐츠 영역 */}
            <div className="flex-1 flex flex-col px-4 py-4">


              {/* 수정 모드 전용 정보 표시 */}
              {rightPanelTab === 'edit' && editMode === 'edit' && editTargetDocument && (
                <div className="mb-4 p-3 bg-orange-50 dark:bg-orange-900/20 border border-orange-200 dark:border-orange-800 rounded-lg">
                  <div className="text-sm text-orange-800 dark:text-orange-200">
                    <div className="font-medium mb-1">📄 수정 대상 문서</div>
                    <div className="text-xs">
                      {reportTitles[editTargetDocument.uuid] || editTargetDocument.file_name || editTargetDocument.project_hash}
                    </div>
                    {editTargetSection && (
                      <div className="text-xs mt-1 opacity-75">
                        섹션: {editTargetSection}
                      </div>
                    )}
                    <div className="text-xs mt-2 flex gap-2">
                      <button
                        onClick={() => {
                          setEditTargetSection(null);
                          setEditMethod('full');
                        }}
                        className="px-2 py-1 bg-orange-100 dark:bg-orange-800 hover:bg-orange-200 dark:hover:bg-orange-700 rounded text-orange-800 dark:text-orange-200"
                      >
                        전체 문서 수정
                      </button>
                      {documentSections.length > 0 && (
                        <select
                          value={editTargetSection || ''}
                          onChange={(e) => {
                            const selectedSection = e.target.value || null;
                            setEditTargetSection(selectedSection);
                            // 섹션이 선택되면 자동으로 section 모드로, 해제되면 full 모드로
                            if (selectedSection) {
                              setEditMethod('section');
                            } else {
                              setEditMethod('full');
                            }
                          }}
                          className="px-2 py-1 text-xs bg-orange-100 dark:bg-orange-800 text-orange-800 dark:text-orange-200 rounded border-none"
                        >
                          <option value="">섹션 선택</option>
                          {documentSections.map((section) => (
                            <option key={section.id} value={section.title}>
                              {section.title}
                            </option>
                          ))}
                        </select>
                      )}
                    </div>
                  </div>
                </div>
              )}


              {/* 템플릿/문서유형/이메일 입력 영역 */}
              <div className="grid grid-cols-1 gap-3 mb-4">
                {/* 템플릿 선택 - 생성 탭에서만 표시 */}
                {rightPanelTab === 'create' && (
                  <div className="space-y-2">
                    <div className="grid grid-cols-[64px_1fr] items-center gap-4">
                      <label className="text-sm font-medium text-gray-700 dark:text-gray-300 min-w-[56px] w-[64px] text-left">템플릿</label>
                      <select
                        value={rightPanelTab === 'create' ? createSelectedTemplate : ''}
                        onChange={e => {
                          if (rightPanelTab === 'create') {
                            setCreateSelectedTemplate(e.target.value);
                          }
                        }}
                        className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                        style={{ backgroundColor: 'var(--sidebar-bg)' }}
                        disabled={templates.length === 0}
                      >
                        {templates.length === 0 ? (
                          <option value="">로딩 중...</option>
                        ) : (
                          <>
                            <option value="">템플릿 선택</option>
                            {templates.map(opt => (
                              <option key={opt.value} value={opt.value}>
                                {opt.name}{opt.description ? ` - ${opt.description}` : ''}
                              </option>
                            ))}
                          </>
                        )}
                      </select>
                    </div>

                    {/* 테이블 스키마 에디터 버튼 - 새로운 템플릿이 선택되었을 때만 표시 */}
                    {(createSelectedTemplate === 'weekly_report' || createSelectedTemplate === 'daily_system_check') && (
                      <div className="flex justify-end">
                        <Button
                          variant="outline"
                          size="sm"
                          onClick={() => openTableSchemaEditor()}
                          className="flex items-center space-x-2"
                        >
                          <Settings className="w-4 h-4" />
                          <span>테이블 스키마 편집</span>
                        </Button>
                      </div>
                    )}
                  </div>
                )}

                {/* 수정 모드 안내 - 수정 탭에서만 표시 */}
                {rightPanelTab === 'edit' && (
                  <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-3">
                    <div className="flex items-start gap-2 mb-3">
                      <span className="text-blue-500 text-lg">ℹ️</span>
                      <div className="text-sm text-blue-700 dark:text-blue-300">
                        <p className="font-medium mb-1">문서 수정 방식 선택</p>
                        <p className="text-xs leading-relaxed mb-3">
                          수정 방식을 선택하여 원하는 형태로 문서를 수정할 수 있습니다.
                        </p>
                      </div>
                    </div>

                    {/* 수정 방식 선택 라디오 버튼 */}
                    <div className="space-y-2">
                      <label className="flex items-start gap-3 cursor-pointer p-2 rounded-md hover:bg-blue-100 dark:hover:bg-blue-800/30 transition-colors">
                        <input
                          type="radio"
                          name="editMethod"
                          value="full"
                          checked={editMethod === 'full'}
                          onChange={() => setEditMethod('full')}
                          className="mt-0.5 accent-blue-600"
                        />
                        <div className="text-xs">
                          <div className="font-medium text-blue-800 dark:text-blue-200">🔄 전체 문서 재생성</div>
                          <div className="text-blue-600 dark:text-blue-300">기존 문서를 참고하여 완전히 새로운 문서 생성 (일관성 보장)</div>
                        </div>
                      </label>

                      <label className="flex items-start gap-3 cursor-pointer p-2 rounded-md hover:bg-blue-100 dark:hover:bg-blue-800/30 transition-colors">
                        <input
                          type="radio"
                          name="editMethod"
                          value="section"
                          checked={editMethod === 'section'}
                          onChange={() => setEditMethod('section')}
                          className="mt-0.5 accent-blue-600"
                          disabled={!editTargetSection}
                        />
                        <div className="text-xs">
                          <div className={`font-medium ${!editTargetSection ? 'text-gray-400 dark:text-gray-500' : 'text-blue-800 dark:text-blue-200'}`}>
                            ✂️ 선택된 섹션만 수정
                          </div>
                          <div className={`${!editTargetSection ? 'text-gray-400 dark:text-gray-500' : 'text-blue-600 dark:text-blue-300'}`}>
                            선택된 섹션만 교체하여 다른 부분은 그대로 유지 {!editTargetSection && '(섹션 선택 필요)'}
                          </div>
                        </div>
                      </label>
                    </div>
                  </div>
                )}

                {/* 문서유형(파일 형식) 선택 */}
                <div className="grid grid-cols-[64px_1fr] items-center gap-4">
                  <label className="text-sm font-medium text-gray-700 dark:text-gray-300 min-w-[56px] w-[64px] text-left">
                    {rightPanelTab === 'create' ? '문서 유형' : '출력 형식'}
                  </label>
                  <div className="flex flex-wrap gap-2">
                    {[
                      { value: 'pdf', label: 'PDF' },
                      { value: 'hwpx', label: 'HWPX' },
                      { value: 'docx', label: 'DOCX' },
                      { value: 'pptx', label: 'PPTX' },
                    ].map(opt => (
                      <label key={opt.value} className={`flex items-center px-3 py-1.5 rounded cursor-pointer border transition-colors text-xs font-medium
                      ${(rightPanelTab === 'create' ? createFileFormat : editFileFormat) === opt.value
                          ? 'bg-blue-600 border-blue-600 text-white'
                          : 'text-gray-700 dark:text-gray-200 border-gray-300 dark:border-gray-600'}
                    `}
                      style={{ 
                        backgroundColor: (rightPanelTab === 'create' ? createFileFormat : editFileFormat) === opt.value 
                          ? 'var(--blue-600)' 
                          : 'var(--sidebar-bg)'
                      }}
                      onMouseEnter={(e) => {
                        if ((rightPanelTab === 'create' ? createFileFormat : editFileFormat) !== opt.value) {
                          e.currentTarget.style.backgroundColor = 'var(--card-bg)';
                        }
                      }}
                      onMouseLeave={(e) => {
                        if ((rightPanelTab === 'create' ? createFileFormat : editFileFormat) !== opt.value) {
                          e.currentTarget.style.backgroundColor = 'var(--sidebar-bg)';
                        }
                      }}>
                        <input
                          type="radio"
                          name="fileFormat"
                          value={opt.value}
                          checked={(rightPanelTab === 'create' ? createFileFormat : editFileFormat) === opt.value}
                          onChange={() => {
                            if (rightPanelTab === 'create') {
                              setCreateFileFormat(opt.value);
                            } else {
                              setEditFileFormat(opt.value);
                            }
                          }}
                          className="mr-2 accent-blue-600"
                        />
                        {opt.label}
                      </label>
                    ))}
                  </div>
                </div>
                {/* 수신 이메일 입력 (선택사항) */}
                <div className="grid grid-cols-[64px_1fr] items-start gap-4">
                  <label className="text-sm font-medium text-gray-700 dark:text-gray-300 min-w-[56px] w-[64px] text-left pt-2">
                    수신 이메일
                    <span className="block text-xs text-gray-400 font-normal">(선택사항)</span>
                  </label>
                  <div className="space-y-1">
                    <input
                      type="email"
                      value={rightPanelTab === 'create' ? createUserEmail : editUserEmail}
                      onChange={e => {
                        if (rightPanelTab === 'create') {
                          setCreateUserEmail(e.target.value);
                        } else {
                          setEditUserEmail(e.target.value);
                        }
                      }}
                      placeholder="your@email.com (선택사항)"
                      className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                      style={{ backgroundColor: 'var(--sidebar-bg)' }}
                    />
                    <p className="text-xs text-gray-500 dark:text-gray-400">
                      📧 {rightPanelTab === 'create' ? '문서 생성' : '문서 수정'} 완료 시 이메일 알림 (선택사항)
                    </p>
                  </div>
                </div>
                {/* 사용자 요청사항 입력 */}
                <div className="grid grid-cols-1 gap-2">
                  <label className="text-sm font-medium text-gray-700 dark:text-gray-300">
                    {rightPanelTab === 'create' ? '📝 문서 내용 작성 요청' : '🔧 문서 수정 요청사항'}
                  </label>
                  <OptimizedTextarea
                    value={rightPanelTab === 'create' ? createInputMessage : editInputMessage}
                    onChange={currentTextareaHandler}
                    placeholder={textareaPlaceholder}
                    className={textareaClassName}
                    disabled={isLoading}
                    onDragEnter={handleTextareaDragEnter}
                    onDragLeave={handleTextareaDragLeave}
                    onDragOver={handleTextareaDragOver}
                    onDrop={handleTextareaDrop}
                    isDragOver={isDragOverTextarea}
                  />
                </div>

                {/* 예제 버튼들 */}
                <div className="mb-4">
                  <h4 className="text-sm font-semibold mb-2 text-gray-700 dark:text-gray-300">💡 요청 예제</h4>
                  <div className="grid grid-cols-4 gap-1.5 mb-3">
                    <button
                      onClick={() => {
                        const example = `AI 기반 스마트팩토리 최적화 시스템 개발 사업계획서를 작성해주세요.

우리 회사는 제조업 공정 최적화를 위한 AI 솔루션을 개발하고 있습니다. 주요 기술은 다음과 같습니다:
- 실시간 공정 데이터 수집 및 분석
- 머신러닝 기반 예측 모델링
- 디지털 트윈 기술 활용
- 실시간 모니터링 및 제어 시스템

목표 시장은 국내 중소 제조업체이며, 현재 베타 서비스를 통해 3개 기업에서 파일럿 테스트를 진행 중입니다.`;
                        if (rightPanelTab === 'create') {
                          setCreateInputMessage(example);
                        } else {
                          setEditInputMessage(example);
                        }
                      }}
                      className="px-2 py-1 text-xs hover:bg-gray-100 dark:hover:bg-gray-800 text-gray-700 dark:text-gray-300 rounded border border-gray-200 dark:border-gray-700 transition-colors"
                      style={{ backgroundColor: 'var(--body-bg)' }}
                    >
                      사업계획서
                    </button>
                    
                    <button
                      onClick={() => {
                        const example = `MZ세대를 타겟으로 한 친환경 패션 브랜드의 디지털 마케팅 전략 보고서를 작성해주세요.

브랜드 현황:
- 브랜드명: EcoStyle
- 주요 제품: 재활용 소재를 활용한 캐주얼 의류 및 액세서리
- 타겟: 20-30대 환경 의식이 높은 MZ세대
- 현재 매출: 월 평균 5천만원 (온라인 채널 80%)

마케팅 목표:
- 6개월 내 매출 50% 증가
- 인스타그램 팔로워 10만명 달성
- 브랜드 인지도 30% 향상`;
                        if (rightPanelTab === 'create') {
                          setCreateInputMessage(example);
                        } else {
                          setEditInputMessage(example);
                        }
                      }}
                      className="px-2 py-1 text-xs hover:bg-gray-100 dark:hover:bg-gray-800 text-gray-700 dark:text-gray-300 rounded border border-gray-200 dark:border-gray-700 transition-colors"
                      style={{ backgroundColor: 'var(--body-bg)' }}
                    >
                      마케팅 전략
                    </button>
                    
                    <button
                      onClick={() => {
                        const example = `메타버스 플랫폼의 사용자 만족도에 영향을 미치는 요인에 관한 연구 논문을 작성해주세요.

연구 목적:
메타버스 플랫폼의 사용자 만족도에 영향을 미치는 주요 요인을 파악하고, 이를 통해 메타버스 플랫폼 개발 및 운영을 위한 실무적 시사점을 도출하고자 합니다.

연구 방법:
- 국내 주요 메타버스 플랫폼 사용자 300명 대상 설문조사 실시
- 기술수용모델(TAM)을 기반으로 연구모형 설계`;
                        if (rightPanelTab === 'create') {
                          setCreateInputMessage(example);
                        } else {
                          setEditInputMessage(example);
                        }
                      }}
                      className="px-2 py-1 text-xs hover:bg-gray-100 dark:hover:bg-gray-800 text-gray-700 dark:text-gray-300 rounded border border-gray-200 dark:border-gray-700 transition-colors"
                      style={{ backgroundColor: 'var(--body-bg)' }}
                    >
                      학술논문
                    </button>
                    
                    <button
                      onClick={() => {
                        const example = `AI 기반 개인화 학습 플랫폼 개발 사업계획서를 작성해주세요.

사업 개요:
- 사업명: AI 튜터 - 개인 맞춤형 학습 플랫폼
- 목적: AI 기술을 활용한 1:1 맞춤형 학습 서비스 제공
- 개발 기간: 2024.07 - 2025.06 (12개월)
- 총 사업비: 5억원

주요 내용:
1. AI 기반 학습자 수준 진단
2. 개인별 맞춤 학습 경로 제시
3. 실시간 학습 피드백 제공`;
                        if (rightPanelTab === 'create') {
                          setCreateInputMessage(example);
                        } else {
                          setEditInputMessage(example);
                        }
                      }}
                      className="px-2 py-1 text-xs hover:bg-gray-100 dark:hover:bg-gray-800 text-gray-700 dark:text-gray-300 rounded border border-gray-200 dark:border-gray-700 transition-colors"
                      style={{ backgroundColor: 'var(--body-bg)' }}
                    >
                      간단한 계획
                    </button>
                  </div>
                </div>

                {/* 프로바이더/모델 선택 */}
                <div className="mb-4 p-3 rounded-lg border border-gray-200 dark:border-gray-700" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
                  <h4 className="text-sm font-semibold mb-3 text-gray-800 dark:text-gray-200">🤖 AI 설정</h4>
                  <div className="grid grid-cols-2 gap-3">
                    <div>
                      <label className="block text-xs font-medium mb-1 text-gray-700 dark:text-gray-300">프로바이더</label>
                      <select
                        value={agentOptions?.provider || 'ollama'}
                        onChange={(e) => handleProviderChange?.(e.target.value)}
                        className="text-xs w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md"
                        style={{ backgroundColor: 'var(--sidebar-bg)' }}
                        disabled={isLoadingProviders}
                      >
                        {isLoadingProviders ? (
                          <option value="">프로바이더 로딩 중...</option>
                        ) : (
                          <>
                            {availableProviders?.map((provider) => (
                              <option key={provider.key} value={provider.key}>
                                {provider.name}
                              </option>
                            ))}
                          </>
                        )}
                      </select>
                    </div>

                    <div>
                      <label className="block text-xs font-medium mb-1 text-gray-700 dark:text-gray-300">모델</label>
                      <select
                        value={agentOptions?.model || 'hamonize:latest'}
                        onChange={(e) => setAgentOptions?.({...agentOptions, model: e.target.value})}
                        className="text-xs w-full p-2 border border-gray-300 dark:border-gray-600 rounded-md"
                        style={{ backgroundColor: 'var(--sidebar-bg)' }}
                        disabled={isLoadingProviders || !providers?.[agentOptions?.provider || 'ollama']?.models?.length}
                      >
                        {!providers?.[agentOptions?.provider || 'ollama']?.models?.length ? (
                          <option value="">모델 로딩 중...</option>
                        ) : (
                          <>
                            {providers?.[agentOptions?.provider || 'ollama']?.models?.map((model: Model) => (
                              <option key={model.id} value={model.id}>
                                {model.name}
                              </option>
                            ))}
                          </>
                        )}
                      </select>
                    </div>
                  </div>
                  
                  <div className="mt-3">
                    <label className="block text-xs font-medium mb-1 text-gray-700 dark:text-gray-300">
                      Temperature: {agentOptions?.temperature || 0.7}
                    </label>
                    <div className="flex items-center space-x-2">
                      <span className="text-xs text-gray-500">0</span>
                      <input
                        type="range"
                        min="0"
                        max="2"
                        step="0.1"
                        value={agentOptions?.temperature || 0.7}
                        onChange={(e) => setAgentOptions?.({...agentOptions, temperature: parseFloat(e.target.value)})}
                        className="flex-1 h-1.5 bg-gray-200 rounded-lg appearance-none cursor-pointer dark:bg-gray-700"
                      />
                      <span className="text-xs text-gray-500">2</span>
                    </div>
                  </div>
                </div>

                {/* 문서 생성 버튼들 */}
                <div className="space-y-2">
                  {/* 🔍 Clarification 버튼 (생성 모드에서만 표시) */}
                  {rightPanelTab === 'create' && (
                    <button
                      onClick={startClarificationProcess}
                      disabled={
                        !createInputMessage.trim() ||
                        isLoading ||
                        clarificationLoading
                      }
                      className="w-full py-3 bg-purple-600 text-white rounded-lg font-bold text-base hover:bg-purple-700 disabled:opacity-50"
                    >
                      {clarificationLoading
                        ? '🔍 요청 분석 중...'
                        : '🔍 스마트 분석 후 생성'
                      }
                    </button>
                  )}

                  {/* 기존 즉시 생성 버튼 */}
                  <button
                    onClick={sendMessage}
                    disabled={
                      ((rightPanelTab === 'create' ? !createInputMessage.trim() : !editInputMessage.trim()) &&
                        uploadedFiles.length === 0) ||
                      isLoading
                    }
                    className="w-full py-3 bg-blue-600 text-white rounded-lg font-bold text-base hover:bg-blue-700 disabled:opacity-50"
                  >
                    {isLoading
                      ? (rightPanelTab === 'create' ? '문서 생성 시작 중...' : '문서 수정 시작 중...')
                      : (rightPanelTab === 'create' ? '📄 즉시 문서 생성' : '📝 문서 수정 시작')
                    }
                  </button>
                </div>

                {/* 백그라운드 작업 상태 표시 */}
                {processingStatus && (
                  <div className="mb-4 p-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-700 rounded-lg">
                    <div className="flex items-start justify-between">
                      <div className="flex-1">
                        <p className="text-sm text-blue-800 dark:text-blue-200 whitespace-pre-line">
                          {processingStatus}
                        </p>
                      </div>
                    </div>
                  </div>
                )}

                {/* 작업 로그 뷰어 */}
                <div className="mt-4">
                  <button
                    className="flex items-center gap-2 text-sm font-semibold mb-2 focus:outline-none select-none text-gray-700 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white transition-colors"
                    onClick={() => setIsLogOpen(v => !v)}
                    aria-expanded={isLogOpen}
                    aria-controls="api-log-box"
                  >
                    <svg className={`w-4 h-4 transition-transform ${isLogOpen ? 'rotate-90' : ''}`} fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 5l7 7-7 7" />
                    </svg>
                    📄 Logs
                    {!isLogOpen && logs.length > 0 && (
                      <span className="ml-1 px-2 py-0.5 text-xs bg-blue-100 dark:bg-blue-900/30 text-blue-700 dark:text-blue-300 rounded-full">
                        {logs.length}
                      </span>
                    )}
                  </button>
                  {isLogOpen && (
                    <div
                      id="api-log-box"
                      ref={logBoxRef}
                      className="
                      rounded-md p-3 h-32 overflow-auto border
                      border-gray-300 dark:border-gray-600
                      bg-gray-50 text-gray-800
                      dark:bg-gray-900 dark:text-green-400
                      text-xs font-mono
                      resize-y
                      min-h-[80px] max-h-[400px]
                    "
                      onMouseEnter={() => setIsLogPaused(true)}
                      onMouseLeave={() => setIsLogPaused(false)}
                      onFocus={() => setIsLogPaused(true)}
                      onBlur={() => setIsLogPaused(false)}
                      tabIndex={0}
                    >
                      {logs.length === 0 ? (
                        <div className="text-gray-400 dark:text-gray-500 text-center py-4">
                          로그가 없습니다
                        </div>
                      ) : (
                        logs.map((line, i) => (
                          <div key={i} className="mb-1 break-all leading-relaxed">
                            {line}
                          </div>
                        ))
                      )}
                    </div>
                  )}
                </div>

                {/* 오류 메시지 */}
                {error && (
                  <div className="mt-2 p-2 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded text-red-800 dark:text-red-200 text-sm">
                    {error}
                  </div>
                )}
                {/* 업로드된 파일 목록 */}
                {uploadedFiles.length > 0 && (
                  <div className="mt-4">
                    <div className="flex flex-wrap gap-2">
                      {uploadedFiles.map((file, idx) => (
                        <div key={`${file.name}-${file.size}-${idx}`} className="flex items-center space-x-2 bg-gray-100 dark:bg-gray-700 px-3 py-1 rounded-full text-xs">
                          <span className="truncate max-w-24 text-gray-700 dark:text-gray-300">{file.name}</span>
                          <button onClick={() => removeFile(idx)} className="text-gray-400 hover:text-red-500">
                            <svg className="w-3 h-3" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                            </svg>
                          </button>
                        </div>
                      ))}
                    </div>
                  </div>
                )}
              </div>
            </div>
          </div>
        </div>
      </div>
      {/* (기존 삭제 모달/토스트 등은 그대로 유지) */}
      {deleteTarget && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/40">
          <div className="rounded-lg p-6 shadow-lg w-[400px] max-w-[90vw]" style={{ backgroundColor: 'var(--card-bg)' }}>
            <div className="font-bold text-lg mb-2 text-gray-900 dark:text-white">삭제 확인</div>
            <div className="mb-4 text-gray-700 dark:text-gray-300">
              <span className="font-semibold">
                {(() => {
                  // 1. reportTitles에서 제목 가져오기
                  const reportTitle = reportTitles[deleteTarget.uuid];
                  if (reportTitle && reportTitle.length > 0) {
                    return reportTitle.length > 50 ? reportTitle.slice(0, 50) + '...' : reportTitle;
                  }

                  // 2. file_name 사용
                  if (deleteTarget.file_name && deleteTarget.file_name.length > 0) {
                    return deleteTarget.file_name.length > 50 ? deleteTarget.file_name.slice(0, 50) + '...' : deleteTarget.file_name;
                  }

                  // 3. executive_summary를 짧게 잘라서 사용
                  if (deleteTarget.executive_summary && deleteTarget.executive_summary.length > 0) {
                    const cleanSummary = removeMetadataFromContent(deleteTarget.executive_summary);
                    return cleanSummary.length > 50 ? cleanSummary.slice(0, 50) + '...' : cleanSummary;
                  }

                  // 4. 마지막 대안으로 project_hash 사용
                  return deleteTarget.project_hash.slice(0, 20) + '...';
                })()}
              </span>
              <div className="text-base pt-2 text-gray-700 dark:text-gray-300">
                삭제된 자료는 복구할 수 없습니다. 정말 삭제 하시겠습니까?
              </div>
            </div>
            <div className="flex justify-end gap-2">
              <button
                className="px-4 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-gray-200 hover:bg-gray-300 dark:hover:bg-gray-600"
                onClick={() => setDeleteTarget(null)}
              >취소</button>
              <button
                className="px-4 py-1 rounded bg-red-600 text-white hover:bg-red-700"
                onClick={async () => {
                  // 삭제 API 호출
                  let res;
                  const headers = {
                    'Content-Type': 'application/json',
                    ...getAuthHeaders()
                  };

                  if (deleteTarget.source === 'cache') {
                    res = await fetch('/api/document', {
                      method: 'POST',
                      headers,
                      body: JSON.stringify({ action: 'delete', uuid: deleteTarget.uuid, source: 'cache' })
                    });
                  } else if (deleteTarget.source === 'output') {
                    res = await fetch('/api/document', {
                      method: 'POST',
                      headers,
                      body: JSON.stringify({ action: 'delete', uuid: deleteTarget.uuid, source: 'output' })
                    });
                  }
                  setDeleteTarget(null);
                  if (res && res.ok && isAuthenticated) {
                    fetchReports(); // 목록만 갱신

                    // 삭제된 문서가 열린 탭에 있으면 닫기
                    setOpenTabs(prev => prev.filter(tab => tab.uuid !== deleteTarget.uuid));
                    setTabContents(prev => {
                      const copy = { ...prev };
                      delete copy[deleteTarget.uuid];
                      return copy;
                    });
                    if (activeTab === deleteTarget.uuid) {
                      // 다른 탭으로 포커스(없으면 null)
                      const idx = openTabs.findIndex(tab => tab.uuid === deleteTarget.uuid);
                      const nextTab = openTabs[idx + 1] || openTabs[idx - 1] || null;
                      setActiveTab(nextTab ? nextTab.uuid : null);
                    }
                  }
                }}
              >삭제</button>
            </div>
          </div>
        </div>
      )}

      {/* 전체삭제 확인 다이얼로그 */}
      {deleteAllTarget && (
        <div className="fixed inset-0 z-50 flex items-center justify-center bg-black/40">
          <div className="rounded-lg p-6 shadow-lg w-[400px] max-w-[90vw]" style={{ backgroundColor: 'var(--card-bg)' }}>
            <div className="font-bold text-lg mb-2 text-gray-900 dark:text-white">전체삭제 확인</div>
            <div className="mb-4 text-gray-700 dark:text-gray-300">
              정말 모든 <span className="font-semibold">
                {deleteAllTarget === 'processing' ? '진행중인 프로젝트' : '완료된 문서'}
              </span>를 삭제하시겠습니까?
              <br />
              <span className="text-sm text-red-600 dark:text-red-400 mt-2 block">
                총 {reports.filter(r => r.status === deleteAllTarget).length}개의 문서가 삭제됩니다.
              </span>
            </div>
            <div className="flex justify-end gap-2">
              <button
                className="px-4 py-1 rounded bg-gray-200 dark:bg-gray-700 text-gray-800 dark:text-gray-200 hover:bg-gray-300 dark:hover:bg-gray-600"
                onClick={() => setDeleteAllTarget(null)}
              >취소</button>
              <button
                className="px-4 py-1 rounded bg-red-600 text-white hover:bg-red-700"
                onClick={() => handleDeleteAllReports(deleteAllTarget)}
              >전체삭제</button>
            </div>
          </div>
        </div>
      )}

      {/* 컨텍스트 메뉴 */}
      {contextMenu.show && contextMenu.document && (
        <div
          className="fixed z-50 border border-gray-200 dark:border-gray-700 rounded-lg shadow-lg py-2 min-w-[160px]"
          style={{
            left: contextMenu.x,
            top: contextMenu.y,
            backgroundColor: 'var(--card-bg)'
          }}
          onClick={(e) => e.stopPropagation()}
        >
          <div className="px-3 py-1 text-xs font-medium text-gray-500 dark:text-gray-400 border-b border-gray-200 dark:border-gray-700 mb-1">
            {reportTitles[contextMenu.document.uuid] || contextMenu.document.file_name || contextMenu.document.project_hash.slice(0, 20)}
          </div>

          <button
            onClick={() => handleSidebarDocClick(contextMenu.document!)}
            className="w-full px-3 py-2 text-left text-sm text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 transition-colors flex items-center gap-2"
          >
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M15 12a3 3 0 11-6 0 3 3 0 016 0z" />
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M2.458 12C3.732 7.943 7.523 5 12 5c4.478 0 8.268 2.943 9.542 7-1.274 4.057-5.064 7-9.542 7-4.477 0-8.268-2.943-9.542-7z" />
            </svg>
            문서 보기
          </button>

          <button
            onClick={() => enterEditModeFromContext(contextMenu.document!)}
            className="w-full px-3 py-2 text-left text-sm text-orange-600 dark:text-orange-400 hover:bg-orange-50 dark:hover:bg-orange-900/20 transition-colors flex items-center gap-2"
          >
            <svg className="w-4 h-4" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M11 5H6a2 2 0 00-2 2v11a2 2 0 002 2h11a2 2 0 002-2v-5m-1.414-9.414a2 2 0 112.828 2.828L11.828 15H9v-2.828l8.586-8.586z" />
            </svg>
            수정 모드로 전환
          </button>
        </div>
      )}

      {/* 🔍 Clarification Modal */}
      {showClarificationModal && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4 pt-20">
          <div className="rounded-xl shadow-2xl max-w-4xl w-full max-h-[85vh] overflow-hidden mt-4" style={{ backgroundColor: 'var(--card-bg)' }}>
            {/* 모달 헤더 */}
            <div className="flex items-center justify-between p-6 border-b border-gray-200 dark:border-gray-600">
              <div>
                <h2 className="text-xl font-bold text-gray-900 dark:text-white">
                  🔍 스마트 문서 생성 도우미
                </h2>
                <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                  더 완성도 높은 문서를 위해 몇 가지 질문에 답해주세요
                </p>
              </div>
              <button
                onClick={() => setShowClarificationModal(false)}
                className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-300"
                disabled={clarificationLoading}
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            </div>

            {/* 진행 단계 표시 */}
            <div className="px-6 py-4" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
              <div className="flex items-center space-x-4">
                <div className={`flex items-center space-x-2 ${clarificationStep === 'analyzing' ? 'text-blue-600' : clarificationStep === 'questions' || clarificationStep === 'todo' || clarificationStep === 'confirm' ? 'text-green-600' : 'text-gray-400'}`}>
                  <div className={`w-3 h-3 rounded-full ${clarificationStep === 'analyzing' ? 'bg-blue-600 animate-pulse' : clarificationStep === 'questions' || clarificationStep === 'todo' || clarificationStep === 'confirm' ? 'bg-green-600' : 'bg-gray-400'}`}></div>
                  <span className="text-sm font-medium">분석</span>
                </div>
                <div className="flex-1 h-px bg-gray-300 dark:bg-gray-600"></div>
                <div className={`flex items-center space-x-2 ${clarificationStep === 'questions' ? 'text-blue-600' : clarificationStep === 'todo' || clarificationStep === 'confirm' ? 'text-green-600' : 'text-gray-400'}`}>
                  <div className={`w-3 h-3 rounded-full ${clarificationStep === 'questions' ? 'bg-blue-600 animate-pulse' : clarificationStep === 'todo' || clarificationStep === 'confirm' ? 'bg-green-600' : 'bg-gray-400'}`}></div>
                  <span className="text-sm font-medium">질문</span>
                </div>
                <div className="flex-1 h-px bg-gray-300 dark:bg-gray-600"></div>
                <div className={`flex items-center space-x-2 ${clarificationStep === 'todo' ? 'text-blue-600' : clarificationStep === 'confirm' ? 'text-green-600' : 'text-gray-400'}`}>
                  <div className={`w-3 h-3 rounded-full ${clarificationStep === 'todo' ? 'bg-blue-600 animate-pulse' : clarificationStep === 'confirm' ? 'bg-green-600' : 'bg-gray-400'}`}></div>
                  <span className="text-sm font-medium">확인</span>
                </div>
              </div>
            </div>

            {/* 모달 내용 */}
            <div className="p-6 overflow-y-auto max-h-[60vh]">
              {/* 분석 중 */}
              {clarificationStep === 'analyzing' && (
                <div className="text-center py-12">
                  <div className="inline-block animate-spin rounded-full h-12 w-12 border-b-2 border-blue-600 mb-4"></div>
                  <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-2">
                    요청을 분석하고 있습니다...
                  </h3>
                  <p className="text-gray-600 dark:text-gray-400">
                    AI가 귀하의 요청을 분석하여 부족한 정보를 식별하고 있습니다.
                  </p>
                </div>
              )}

              {/* 질문 단계 */}
              {clarificationStep === 'questions' && clarificationData && (
                <div className="space-y-6">
                  {/* 분석 결과 요약 카드 */}
                  <div className="bg-gradient-to-r from-blue-50 to-indigo-50 dark:from-blue-900/20 dark:to-indigo-900/20 p-5 rounded-xl border border-blue-200 dark:border-blue-700">
                    <div className="flex items-center mb-3">
                      <div className="w-8 h-8 bg-blue-100 dark:bg-blue-800 rounded-lg flex items-center justify-center mr-3">
                        <svg className="w-4 h-4 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                          <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 19v-6a2 2 0 00-2-2H5a2 2 0 00-2 2v6a2 2 0 002 2h2a2 2 0 002-2zm0 0V9a2 2 0 012-2h2a2 2 0 012 2v10m-6 0a2 2 0 002 2h2a2 2 0 00-2-2m0 0V5a2 2 0 012-2h2a2 2 0 012 2v14a2 2 0 01-2 2h-2a2 2 0 01-2-2z" />
                        </svg>
                      </div>
                      <h3 className="font-semibold text-blue-900 dark:text-blue-100">
                        분석 결과
                      </h3>
                    </div>
                    <div className="grid grid-cols-2 gap-4 text-sm">
                      <div className="flex items-center space-x-2">
                        <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-blue-100 dark:bg-blue-800 text-blue-800 dark:text-blue-200">
                          문서 유형
                        </span>
                        <span className="font-medium text-gray-900 dark:text-gray-100">{clarificationData.analysis?.document_type || '일반 문서'}</span>
                      </div>
                      <div className="flex items-center space-x-2">
                        <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-green-100 dark:bg-green-800 text-green-800 dark:text-green-200">
                          추천 템플릿
                        </span>
                        <span className="font-medium text-gray-900 dark:text-gray-100">{clarificationData.suggested_template}</span>
                      </div>
                      <div className="flex items-center space-x-2">
                        <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-purple-100 dark:bg-purple-800 text-purple-800 dark:text-purple-200">
                          신뢰도
                        </span>
                        <span className="font-medium text-gray-900 dark:text-gray-100">{Math.round(clarificationData.confidence_score * 100)}%</span>
                      </div>
                      <div className="flex items-center space-x-2">
                        <span className="inline-flex items-center px-2 py-1 rounded-full text-xs font-medium bg-orange-100 dark:bg-orange-800 text-orange-800 dark:text-orange-200">
                          예상 섹션
                        </span>
                        <span className="font-medium text-gray-900 dark:text-gray-100">{clarificationData.estimated_sections?.length || 0}개</span>
                      </div>
                    </div>
                  </div>

                  {/* 질문 카드들 */}
                  <div className="space-y-4">
                    <h3 className="text-lg font-medium text-gray-900 dark:text-white flex items-center">
                      <span className="w-6 h-6 bg-blue-100 dark:bg-blue-800 text-blue-600 dark:text-blue-300 rounded-full flex items-center justify-center text-sm font-medium mr-2">
                        ?
                      </span>
                      추가 정보가 필요합니다
                    </h3>

                    {clarificationData.questions?.map((question: any, index: number) => (
                      <div key={question.id} className="p-5 rounded-xl border border-gray-200 dark:border-gray-600 shadow-sm hover:shadow-md transition-shadow" style={{ backgroundColor: 'var(--card-bg)' }}>
                        <div className="flex items-start space-x-3 mb-4">
                          <div className="flex-shrink-0 w-6 h-6 bg-blue-100 dark:bg-blue-800 text-blue-600 dark:text-blue-300 rounded-full flex items-center justify-center text-sm font-medium">
                            {index + 1}
                          </div>
                          <div className="flex-1">
                            <h4 className="text-sm font-medium text-gray-900 dark:text-white mb-1">
                              {question.question}
                              {question.required && <span className="text-red-500 ml-1">*</span>}
                            </h4>
                            {question.placeholder && (
                              <p className="text-xs text-gray-500 dark:text-gray-400">{question.placeholder}</p>
                            )}
                          </div>
                        </div>

                        {/* 텍스트 입력 */}
                        {question.type === 'text' && (
                          <textarea
                            value={clarificationResponses[question.id] || ''}
                            onChange={(e) => handleClarificationResponse(question.id, e.target.value)}
                            placeholder={question.placeholder}
                            className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg text-gray-900 dark:text-white resize-none focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                            style={{ backgroundColor: 'var(--sidebar-bg)' }}
                            rows={3}
                          />
                        )}

                        {/* 선택 입력 - 태그 버튼 스타일 */}
                        {question.type === 'select' && (
                          <div className="space-y-3">
                            <div className="flex flex-wrap gap-2">
                              {question.options?.map((option: string) => (
                                <button
                                  key={option}
                                  onClick={() => handleClarificationResponse(question.id, option)}
                                  className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${clarificationResponses[question.id] === option
                                    ? 'bg-blue-600 text-white shadow-sm'
                                    : 'bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-500'
                                    }`}
                                >
                                  {option}
                                </button>
                              ))}
                            </div>
                          </div>
                        )}

                        {/* 다중 선택 - 태그 버튼 스타일 */}
                        {question.type === 'multiselect' && (
                          <div className="space-y-3">
                            <div className="flex flex-wrap gap-2">
                              {question.options?.map((option: string) => {
                                const isSelected = (clarificationResponses[question.id] || []).includes(option);
                                return (
                                  <button
                                    key={option}
                                    onClick={() => {
                                      const currentValues = clarificationResponses[question.id] || [];
                                      if (isSelected) {
                                        handleClarificationResponse(question.id, currentValues.filter((v: string) => v !== option));
                                      } else {
                                        handleClarificationResponse(question.id, [...currentValues, option]);
                                      }
                                    }}
                                    className={`px-4 py-2 rounded-lg text-sm font-medium transition-colors ${isSelected
                                      ? 'bg-blue-600 text-white shadow-sm'
                                      : 'bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 hover:bg-gray-200 dark:hover:bg-gray-500'
                                      }`}
                                  >
                                    {option}
                                  </button>
                                );
                              })}
                            </div>
                            {(clarificationResponses[question.id] || []).length > 0 && (
                              <div className="text-xs text-gray-500 dark:text-gray-400">
                                {(clarificationResponses[question.id] || []).length}개 선택됨
                              </div>
                            )}
                          </div>
                        )}

                        {/* 숫자 입력 */}
                        {question.type === 'number' && (
                          <input
                            type="number"
                            value={clarificationResponses[question.id] || ''}
                            onChange={(e) => handleClarificationResponse(question.id, e.target.value)}
                            placeholder={question.placeholder}
                            className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-transparent transition-colors"
                            style={{ backgroundColor: 'var(--sidebar-bg)' }}
                          />
                        )}

                        {/* 날짜 입력 */}
                        {question.type === 'date' && (
                          <input
                            type="date"
                            value={clarificationResponses[question.id] || ''}
                            onChange={(e) => handleClarificationResponse(question.id, e.target.value)}
                            className="w-full p-3 border border-gray-300 dark:border-gray-600 rounded-lg text-gray-900 dark:text-white"
                            style={{ backgroundColor: 'var(--sidebar-bg)' }}
                          />
                        )}
                      </div>
                    ))}
                  </div>
                </div>
              )}

              {/* Todo 리스트 검토 */}
              {clarificationStep === 'todo' && clarificationData && (
                <div className="space-y-6">
                  <div>
                    <h3 className="text-lg font-medium text-gray-900 dark:text-white mb-4">
                      📋 생성될 문서 작업 목록
                    </h3>
                    <p className="text-gray-600 dark:text-gray-400 mb-6">
                      다음 작업들이 순서대로 수행됩니다. 확인 후 문서 생성을 시작하세요.
                    </p>
                  </div>

                  <div className="rounded-lg p-4" style={{ backgroundColor: 'var(--sidebar-bg)' }}>
                    <ul className="space-y-3">
                      {clarificationData.todo_list?.map((task: string, index: number) => (
                        <li key={index} className="flex items-start space-x-3">
                          <div className="flex-shrink-0 w-6 h-6 bg-blue-100 dark:bg-blue-900 text-blue-600 dark:text-blue-300 rounded-full flex items-center justify-center text-sm font-medium">
                            {index + 1}
                          </div>
                          <span className="text-gray-700 dark:text-gray-300">{task}</span>
                        </li>
                      ))}
                    </ul>
                  </div>

                  <div className="bg-green-50 dark:bg-green-900/20 p-4 rounded-lg">
                    <h4 className="font-medium text-green-900 dark:text-green-100 mb-2">
                      ✅ 예상 결과
                    </h4>
                    <ul className="text-sm text-green-700 dark:text-green-300 space-y-1">
                      <li>• 총 {clarificationData.estimated_sections?.length || 0}개 섹션으로 구성</li>
                      <li>• {clarificationData.analysis?.complexity_level === 'complex' ? '상세한' : clarificationData.analysis?.complexity_level === 'simple' ? '간단한' : '보통'} 수준의 문서</li>
                      <li>• 차트, 표, 다이어그램 등 시각적 요소 포함</li>
                      <li>• 완성까지 예상 시간: 2-5분</li>
                    </ul>
                  </div>
                </div>
              )}

              {/* 자동 진행 확인 단계 */}
              {clarificationStep === 'confirm' && clarificationData && (
                <div className="text-center py-12">
                  <div className="inline-flex items-center justify-center w-16 h-16 bg-green-100 dark:bg-green-900 rounded-full mb-6">
                    <svg className="w-8 h-8 text-green-600 dark:text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12l2 2 4-4m6 2a9 9 0 11-18 0 9 9 0 0118 0z" />
                    </svg>
                  </div>
                  <h3 className="text-xl font-medium text-gray-900 dark:text-white mb-4">
                    ✅ 분석이 완료되었습니다!
                  </h3>
                  <p className="text-gray-600 dark:text-gray-400 mb-6">
                    입력하신 정보가 충분하여 추가 질문 없이 바로 문서 생성을 시작합니다.
                  </p>

                  {/* 분석 결과 요약 */}
                  <div className="bg-green-50 dark:bg-green-900/20 p-6 rounded-lg mb-6 text-left">
                    <h4 className="font-medium text-green-900 dark:text-green-100 mb-3">
                      📊 분석 결과
                    </h4>
                    <div className="grid grid-cols-2 gap-4 text-sm">
                      <div>
                        <span className="text-green-700 dark:text-green-300">문서 유형:</span>
                        <span className="ml-2 font-medium">{clarificationData.analysis?.document_type || '일반 문서'}</span>
                      </div>
                      <div>
                        <span className="text-green-700 dark:text-green-300">추천 템플릿:</span>
                        <span className="ml-2 font-medium">{clarificationData.suggested_template || 'simple'}</span>
                      </div>
                      <div>
                        <span className="text-green-700 dark:text-green-300">신뢰도:</span>
                        <span className="ml-2 font-medium">{Math.round((clarificationData.confidence_score || 0.7) * 100)}%</span>
                      </div>
                      <div>
                        <span className="text-green-700 dark:text-green-300">예상 섹션:</span>
                        <span className="ml-2 font-medium">{clarificationData.estimated_sections?.length || 0}개</span>
                      </div>
                    </div>
                  </div>

                  <div className="flex items-center justify-center space-x-2 text-blue-600 dark:text-blue-400">
                    <div className="animate-spin rounded-full h-5 w-5 border-b-2 border-blue-600"></div>
                    <span className="text-sm">곧 문서 생성이 시작됩니다...</span>
                  </div>
                </div>
              )}
            </div>

            {/* 모달 푸터 */}
            <div className="flex items-center justify-between p-6 border-t border-gray-200 dark:border-gray-600">
              <div className="text-sm text-gray-500 dark:text-gray-400">
                {clarificationStep === 'questions' && clarificationData && (
                  `${clarificationData.questions?.length || 0}개 질문 중 ${Object.values(clarificationResponses).filter(v => v && v !== '').length}개 답변 완료`
                )}
                {clarificationStep === 'confirm' && (
                  '분석 완료 - 자동으로 문서 생성이 시작됩니다'
                )}
              </div>

              <div className="flex space-x-3">
                {clarificationStep === 'questions' && (
                  <>
                    <button
                      onClick={() => setShowClarificationModal(false)}
                      className="px-4 py-2 text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200"
                      disabled={clarificationLoading}
                    >
                      건너뛰기
                    </button>
                    <button
                      onClick={proceedToTodoReview}
                      className="px-6 py-2 bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50"
                      disabled={clarificationLoading}
                    >
                      다음 단계
                    </button>
                  </>
                )}

                {clarificationStep === 'todo' && (
                  <>
                    <button
                      onClick={() => setClarificationStep('questions')}
                      className="px-4 py-2 text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200"
                      disabled={clarificationLoading}
                    >
                      이전
                    </button>
                    <button
                      onClick={confirmAndGenerate}
                      className="px-6 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 disabled:opacity-50 flex items-center space-x-2"
                      disabled={clarificationLoading}
                    >
                      {clarificationLoading ? (
                        <>
                          <div className="animate-spin rounded-full h-4 w-4 border-b-2 border-white"></div>
                          <span>생성 중...</span>
                        </>
                      ) : (
                        <>
                          <span>🚀</span>
                          <span>문서 생성 시작</span>
                        </>
                      )}
                    </button>
                  </>
                )}
              </div>
            </div>
          </div>
        </div>
      )}

      {/* 테이블 스키마 에디터 */}
      {showTableSchemaEditor && (
        <TableSchemaEditor
          schema={currentTableSchema}
          onSchemaChange={handleTableSchemaChange}
          onClose={closeTableSchemaEditor}
          isVisible={showTableSchemaEditor}
        />
      )}
    </ClientOnly>
  );
}

// 메인 컴포넌트를 Suspense로 감싸기 - 개선된 버전
export default function DocumentPage() {
  return (
    <Suspense fallback={
      <div className="flex items-center justify-center h-screen">
        <div className="text-center">
          <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600 mx-auto"></div>
          <p className="mt-2 text-gray-600 dark:text-gray-400">페이지를 로딩하고 있습니다...</p>
        </div>
      </div>
    }>
      <DocumentPageContent />
    </Suspense>
  );
}