import React, { useState, useEffect, useCallback, useRef } from 'react';
import type { BackgroundJob, BackgroundJobStatus as BackgroundJobStatusType } from '@/types/support';

interface BackgroundJobStatusProps {
  userId?: string;
  onJobComplete?: (job: BackgroundJob) => void;
  refreshTrigger?: number;
}

// 간단한 아이콘 컴포넌트들
const SpinnerIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <path d="M21 12a9 9 0 1 1-6.219-8.56"/>
  </svg>
);

const CheckIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <polyline points="20 6 9 17 4 12"/>
  </svg>
);

const XIcon = (props: React.SVGProps<SVGSVGElement>) => (
  <svg {...props} xmlns="http://www.w3.org/2000/svg" width="16" height="16" viewBox="0 0 24 24" fill="none" stroke="currentColor" strokeWidth="2" strokeLinecap="round" strokeLinejoin="round">
    <line x1="18" y1="6" x2="6" y2="18"/>
    <line x1="6" y1="6" x2="18" y2="18"/>
  </svg>
);

export const BackgroundJobStatus: React.FC<BackgroundJobStatusProps> = ({ 
  userId, 
  onJobComplete,
  refreshTrigger
}) => {
  const [jobStatus, setJobStatus] = useState<BackgroundJobStatusType | null>(null);
  const [isLoading, setIsLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [autoDeleteTimers, setAutoDeleteTimers] = useState<Map<string, NodeJS.Timeout>>(new Map());
  
  // useRef로 폴링 상태 관리
  const pollingIntervalRef = useRef<NodeJS.Timeout | null>(null);
  const isPollingRef = useRef<boolean>(false);

  const handleDeleteJob = async (jobId: string) => {
    try {
      // 타이머가 있으면 제거
      const timer = autoDeleteTimers.get(jobId);
      if (timer) {
        clearTimeout(timer);
        setAutoDeleteTimers(prev => {
          const newMap = new Map(prev);
          newMap.delete(jobId);
          return newMap;
        });
      }
      
      const response = await fetch(`/api/support/background-jobs?jobId=${jobId}`, {
        method: 'DELETE',
        credentials: 'include'
      });
      
      if (response.ok) {
        fetchJobStatus();
      }
    } catch (err) {
      console.error('작업 삭제 실패:', err);
    }
  };

  // 이전 작업 상태를 추적하기 위한 ref
  const previousJobsRef = useRef<BackgroundJob[]>([]);

  const fetchJobStatus = useCallback(async () => {
    try {
      const url = userId ? `/api/support/background-jobs?userId=${userId}` : '/api/support/background-jobs';
      // console.log('[BackgroundJobStatus] 작업 상태 조회:', { url, userId });
      const response = await fetch(url, {
        credentials: 'include'
      });
      
      if (response.ok) {
        const result = await response.json();
        // console.log('[BackgroundJobStatus] 작업 상태 응답:', result);
        
        if (result.success) {
          const newJobs = result.data.jobs || [];
          
          // 완료된 작업 확인 및 콜백 호출
          if (onJobComplete && previousJobsRef.current.length > 0) {
            const completedJobs = newJobs.filter((job: BackgroundJob) => 
              job.status === 'completed' && 
              !previousJobsRef.current.some(prevJob => prevJob.id === job.id && prevJob.status === 'completed')
            );
            
            completedJobs.forEach((job: BackgroundJob) => {
              onJobComplete(job);
              
              // 완료된 작업을 3초 후 자동 삭제
              const timer = setTimeout(() => {
                handleDeleteJob(job.id);
                setAutoDeleteTimers(prev => {
                  const newMap = new Map(prev);
                  newMap.delete(job.id);
                  return newMap;
                });
              }, 3000);
              
              setAutoDeleteTimers(prev => {
                const newMap = new Map(prev);
                newMap.set(job.id, timer);
                return newMap;
              });
            });
          }
          
          // 이전 작업 상태 업데이트
          previousJobsRef.current = newJobs;
          
          setJobStatus(result.data);
          setError(null);
          
          // 폴링 제어 로직
          const hasActiveJobs = newJobs.some((job: BackgroundJob) => 
            job.status === 'pending' || job.status === 'running'
          );
          
          if (hasActiveJobs && !isPollingRef.current) {
            startPolling();
          } else if (!hasActiveJobs && isPollingRef.current) {
            stopPolling();
          }
        } else {
          setError(result.error || '작업 상태 조회 실패');
        }
      } else {
        setError('서버 오류');
      }
    } catch (err) {
      setError('네트워크 오류');
      console.error('백그라운드 작업 상태 조회 실패:', err);
    } finally {
      setIsLoading(false);
    }
  }, [userId, onJobComplete]);

  // 스마트 폴링 시작/중지 함수
  const startPolling = useCallback(() => {
    if (isPollingRef.current) return;
    
    const interval = setInterval(() => {
      fetchJobStatus();
    }, 1000);
    
    pollingIntervalRef.current = interval;
    isPollingRef.current = true;
  }, [fetchJobStatus]);

  const stopPolling = useCallback(() => {
    if (pollingIntervalRef.current) {
      clearInterval(pollingIntervalRef.current);
      pollingIntervalRef.current = null;
    }
    isPollingRef.current = false;
  }, []);

  // 초기 로드
  useEffect(() => {
    fetchJobStatus();
  }, [fetchJobStatus]);

  // refreshTrigger가 변경될 때마다 즉시 새로고침
  useEffect(() => {
    if (refreshTrigger) {
      fetchJobStatus();
    }
  }, [refreshTrigger, fetchJobStatus]);

  // 컴포넌트 언마운트 시 타이머 및 폴링 정리
  useEffect(() => {
    return () => {
      autoDeleteTimers.forEach((timer) => clearTimeout(timer));
      if (pollingIntervalRef.current) {
        clearInterval(pollingIntervalRef.current);
      }
    };
  }, [autoDeleteTimers]);

  const getJobTypeText = (type: string) => {
    switch (type) {
      case 'file_upload':
        return '임베딩';
      case 'ticket_sync':
        return '티켓 동기화';
      default:
        return '작업';
    }
  };

  const getStatusIcon = (status: string) => {
    switch (status) {
      case 'pending':
      case 'running':
        return <SpinnerIcon className="animate-spin text-blue-500" />;
      case 'completed':
        return <CheckIcon className="text-green-500" />;
      case 'failed':
        return <XIcon className="text-red-500" />;
      default:
        return <SpinnerIcon className="text-gray-500" />;
    }
  };

  // 작업이 없거나 로딩 중이면 아무것도 표시하지 않음
  if (isLoading || error) {
    // console.log('[BackgroundJobStatus] 렌더링 안함 - 로딩 중이거나 에러:', { isLoading, error });
    return null;
  }

  // userId가 없고 작업도 없으면 표시하지 않음
  if (!userId && (!jobStatus || jobStatus.jobs.length === 0)) {
    // console.log('[BackgroundJobStatus] 렌더링 안함 - userId 없고 작업도 없음');
    return null;
  }

  // 작업이 없으면 표시하지 않음
  if (!jobStatus || jobStatus.jobs.length === 0) {
    // console.log('[BackgroundJobStatus] 렌더링 안함 - 작업 없음');
    return null;
  }

  // 활성 작업만 표시 (완료된 작업은 잠시만 표시)
  const activeJobs = jobStatus.jobs.filter(job => 
    job.status === 'pending' || 
    job.status === 'running' || 
    (job.status === 'completed' && autoDeleteTimers.has(job.id)) ||
    job.status === 'failed'
  );

  if (activeJobs.length === 0) {
    return null;
  }

  return (
    <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-md p-2 mb-2">
      <div className="space-y-1">
        {activeJobs.map((job) => (
          <div
            key={job.id}
            className="flex items-center gap-2 text-xs"
          >
            {getStatusIcon(job.status)}
            <span className="text-gray-700 dark:text-gray-300 truncate flex-1">
              {getJobTypeText(job.type)}: {job.filename}
            </span>
            {job.status === 'completed' && (
              <span className="text-xs text-green-600 dark:text-green-400">완료</span>
            )}
            {job.status === 'failed' && (
              <span className="text-xs text-red-600 dark:text-red-400">실패</span>
            )}
          </div>
        ))}
      </div>
    </div>
  );
}; 