import React, { memo, useState } from 'react';
import ReactMarkdown from 'react-markdown';
import remarkGfm from 'remark-gfm';
import remarkBreaks from 'remark-breaks';
import Prism from 'prismjs';
import 'prismjs/components/prism-python';
import 'prismjs/components/prism-jsx';
import 'prismjs/components/prism-tsx';
import 'prismjs/components/prism-typescript';
import 'prismjs/components/prism-json';
import 'prismjs/components/prism-bash';
import 'prismjs/components/prism-markdown';
import 'prismjs/themes/prism-tomorrow.css';
import { getApiServerUrl } from '@/config/serverConfig';

interface ChatMarkdownRendererProps {
  content: string;
  className?: string;
  downloadInfo?: any[]; // 다운로드 정보를 별도 prop으로 받을 수 있도록 추가
}

// ChatPage 전용 코드블록 컴포넌트
const ChatCodeBlock = memo(function ChatCodeBlock({ code, language = 'plaintext' }: { code: string; language?: string }) {
  const [isCopied, setIsCopied] = useState(false);
  
  // Prism 하이라이트
  const highlighted = Prism.highlight(code, Prism.languages[language] || Prism.languages.plaintext, language);
  
  const highlightedLines = highlighted.split('\n');
  const codeLines = code.split('\n');

  // 안전한 클립보드 복사 함수
  const copyToClipboard = async (text: string) => {
    try {
      // Clipboard API가 사용 가능한 경우
      if (navigator.clipboard && navigator.clipboard.writeText) {
        await navigator.clipboard.writeText(text);
        return true;
      }
      
      // fallback: document.execCommand 사용
      const textArea = document.createElement('textarea');
      textArea.value = text;
      textArea.style.position = 'fixed';
      textArea.style.left = '-999999px';
      textArea.style.top = '-999999px';
      document.body.appendChild(textArea);
      textArea.focus();
      textArea.select();
      
      const successful = document.execCommand('copy');
      document.body.removeChild(textArea);
      
      if (successful) {
        return true;
      }
      
      // 마지막 fallback: 사용자에게 알림
      alert('클립보드 복사에 실패했습니다. 코드를 수동으로 복사해주세요.');
      return false;
    } catch (error) {
      console.error('클립보드 복사 오류:', error);
      alert('클립보드 복사에 실패했습니다. 코드를 수동으로 복사해주세요.');
      return false;
    }
  };

  // 복사 버튼 클릭 핸들러
  const handleCopyClick = async () => {
    const success = await copyToClipboard(code);
    if (success) {
      setIsCopied(true);
      // 2초 후 원래 상태로 복원
      setTimeout(() => {
        setIsCopied(false);
      }, 2000);
    }
  };

  return (
    <div className="relative group rounded-lg my-4 border overflow-hidden border-gray-200 bg-gray-50 dark:border-gray-700 dark:bg-gray-900">
      {/* 언어 표시 */}
      {language && language !== 'plaintext' && (
        <div className="px-3 py-1 text-xs font-medium border-b bg-gray-100 text-gray-600 border-gray-200 dark:bg-gray-800 dark:text-gray-300 dark:border-gray-700">
          {language}
        </div>
      )}
      
      {/* 복사 버튼 */}
      <button
        className={`absolute top-2 right-2 z-10 px-2 py-1 text-xs rounded transition-colors ${
          isCopied 
            ? 'bg-green-600 text-white' 
            : 'bg-gray-200 text-gray-600 hover:bg-blue-600 hover:text-white dark:bg-gray-800 dark:text-gray-300'
        }`}
        onClick={handleCopyClick}
        title="코드 복사"
      >
        {isCopied ? '복사됨' : '복사'}
      </button>
      
      {/* 코드 내용 */}
      <div className="relative">
        <pre className="p-4 overflow-x-auto text-base">
          <code>
            {highlightedLines.map((line: string, index: number) => (
              <div key={index} className="flex">
                <span className="inline-block w-8 text-right text-gray-400 dark:text-gray-600 mr-4 select-none flex-shrink-0">
                  {index + 1}
                </span>
                <span 
                  className="flex-1 min-w-0"
                  dangerouslySetInnerHTML={{ __html: line || '&nbsp;' }}
                />
              </div>
            ))}
          </code>
        </pre>
      </div>
    </div>
  );
});

export const ChatMarkdownRenderer: React.FC<ChatMarkdownRendererProps> = ({ 
  content, 
  className = "",
  downloadInfo: propDownloadInfo // 컴포넌트 내부에서 사용할 다운로드 정보 prop
}) => {

  // 참조 정보 추출 및 content 분리
  const referenceInfoMatch = content.match(/<!-- REFERENCE_INFO:([\s\S]*?) -->/);
  const downloadInfoMatch = content.match(/<!-- DOWNLOAD_INFO:([\s\S]*?) -->/);
  let downloadInfo: any[] = [];
  let cleanContent = content;
  
  // prop으로 전달된 다운로드 정보가 있으면 우선 사용
  if (propDownloadInfo && propDownloadInfo.length > 0) {
    downloadInfo = propDownloadInfo;
    cleanContent = content.replace(/<!-- REFERENCE_INFO:[\s\S]*? -->/, '').trim();
    // console.log('✅ [ChatMarkdownRenderer] prop downloadInfo 사용:', { 
    //   downloadInfo,
    //   count: downloadInfo.length,
    //   items: downloadInfo.map((item: any) => ({
    //     filename: item.filename,
    //     title: item.title,
    //     source: item.source,
    //     url: item.url
    //   }))
    // });
  }
  
  // 새로운 REFERENCE_INFO 형식 처리
  if (!downloadInfo.length && referenceInfoMatch) {
    try {
      const referenceInfo = JSON.parse(referenceInfoMatch[1]);
      downloadInfo = referenceInfo.downloadInfo || [];
      cleanContent = content.replace(/<!-- REFERENCE_INFO:[\s\S]*? -->/, '').trim();
      // console.log('✅ [ChatMarkdownRenderer] referenceInfo 파싱 성공:', { downloadInfo });
    } catch (error) {
      console.error('❌ [ChatMarkdownRenderer] referenceInfo 파싱 오류:', error);
    }
  }
  
  // 기존 DOWNLOAD_INFO 형식 호환성 유지
  if (!downloadInfo.length && downloadInfoMatch) {
    try {
      downloadInfo = JSON.parse(downloadInfoMatch[1]);
      cleanContent = content.replace(/<!-- DOWNLOAD_INFO:[\s\S]*? -->/, '').trim();
      // console.log('✅ [ChatMarkdownRenderer] downloadInfo 파싱 성공:', downloadInfo);
    } catch (error) {
      console.error('❌ [ChatMarkdownRenderer] downloadInfo 파싱 오류:', error);
    }
  }
  
  // JSON 형식의 cited_documents 추출 및 제거
  // 패턴: {"cited_documents": [...]} (백틱, 작은따옴표 코드 블록, 일반 JSON 모두 지원)
  const citedDocPattern = /```json\s*\n?\s*\{"cited_documents"\s*:\s*\[[^\]]*\]\}\s*\n?\s*```|'''json\s*\n?\s*\{"cited_documents"\s*:\s*\[[^\]]*\]\}\s*\n?\s*'''|\{"cited_documents"\s*:\s*\[[^\]]*\]\}/g;
  const citedMatches = cleanContent.match(citedDocPattern);
  
  if (citedMatches) {
    citedMatches.forEach(match => {
      try {
        // 코드 블록에서 JSON 추출
        let jsonString = match;
        if (match.includes('```json')) {
          // 백틱 코드 블록에서 JSON 부분만 추출
          jsonString = match.replace(/```json\s*\n?\s*/, '').replace(/\s*\n?\s*```/, '');
        } else if (match.includes("'''json")) {
          // 작은따옴표 코드 블록에서 JSON 부분만 추출
          jsonString = match.replace(/'''json\s*\n?\s*/, '').replace(/\s*\n?\s*'''/, '');
        }
        
        // JSON 파싱
        const citedJson = JSON.parse(jsonString);
        if (citedJson.cited_documents && Array.isArray(citedJson.cited_documents)) {
          // cited_documents를 downloadInfo 형식으로 변환
          // 이미 downloadInfo에 있는 웹 소스는 제외하고 실제 문서만 추가
          const citedDownloadInfo = citedJson.cited_documents
            .filter((filename: string) => {
              // HTTP/HTTPS URL은 제외 (웹 검색 결과가 잘못 포함된 경우)
              if (filename.startsWith('http://') || filename.startsWith('https://')) {
                console.log('⚠️ cited_documents에서 URL 제거:', filename);
                return false;
              }
              // 이미 웹 소스로 등록된 항목인지 확인
              const filenameWithoutPdf = filename.replace(/\.pdf$/i, '');
              // 웹 소스와 중복되는지 더 정확하게 체크
              const isDuplicate = downloadInfo.some((item: any) => {
                if (item.source === 'web') {
                  // 웹 항목의 제목이나 파일명과 비교
                  const webTitle = item.title || item.filename || '';
                  // PDF 확장자를 제거한 파일명과 웹 제목 비교
                  return webTitle.includes(filenameWithoutPdf) || 
                         filenameWithoutPdf.includes(webTitle) ||
                         webTitle === filenameWithoutPdf ||
                         // Instagram 같은 특수 케이스 처리
                         (filenameWithoutPdf.includes('Instagram') && webTitle.includes('instagram.com'));
                }
                return false;
              });
              return !isDuplicate;
            })
            .map((filename: string) => ({
              filename: filename,
              source: 'rag',
              score: 1.0
            }));
          
          // 기존 downloadInfo와 병합 (중복 제거)
          const seenFilenames = new Set(downloadInfo.map((info: any) => info.filename));
          for (const info of citedDownloadInfo) {
            if (!seenFilenames.has(info.filename)) {
              downloadInfo.push(info);
              seenFilenames.add(info.filename);
            }
          }
          
          console.log('✅ [ChatMarkdownRenderer] cited_documents 추출 및 변환:', citedJson.cited_documents);
        }
        
        // JSON 블록을 content에서 제거
        cleanContent = cleanContent.replace(match, '').trim();
      } catch (error) {
        console.error('❌ [ChatMarkdownRenderer] cited_documents 파싱 오류:', error);
      }
    });
  }
  
  // [REF]내용[/REF] 태그 처리 - 숫자나 URL 모두 제거
  cleanContent = cleanContent.replace(/\[REF\].*?\[\/REF\]/g, '');
  
  // 디버깅: content에 참조 정보가 있는지 확인
  // if (content.includes('REFERENCE_INFO:') || content.includes('DOWNLOAD_INFO:')) {
  //   console.log('🔍 [ChatMarkdownRenderer] 참조 정보 포함된 content 발견');
  //   console.log('📄 Content 길이:', content.length);
  //   console.log('📄 downloadInfo 개수:', downloadInfo.length);

  // }

  // 출처 섹션이 있는지 확인
  const hasReferencesSection = cleanContent.includes('# 출처(References)') || cleanContent.includes('## 출처')|| cleanContent.includes('출처(References)');

  return (
    <div className={`markdown-content text-base ${className}`} style={{ fontFamily: 'Spoqa Han Sans Neo' }}>
      <style dangerouslySetInnerHTML={{
        __html: `
          .markdown-content li p {
            margin-bottom: 0 !important;
            margin-top: 0 !important;
            display: inline !important;
          }
          .markdown-content li {
            display: list-item !important;
            margin-bottom: 0.25rem !important;
          }
          .markdown-content ul li p,
          .markdown-content ol li p {
            margin: 0 !important;
            display: inline !important;
          }
        `
      }} />
      <ReactMarkdown
        remarkPlugins={[remarkGfm, remarkBreaks]}
        unwrapDisallowed={true}
        components={{
        // 강조 텍스트 (**text**)
        strong: ({ children }) => (
          <strong className="font-bold text-gray-900 dark:text-gray-100">
            {children}
          </strong>
        ),
        // 기울임 텍스트 (*text*)
        em: ({ children }) => (
          <em className="italic text-blue-600 dark:text-blue-400">
            {children}
          </em>
        ),
        // 인라인 코드 (`code`)
        code: ({ children, className }) => {
          const match = /language-(\w+)/.exec(className || '');
          const language = match ? match[1] : '';
          
          if (language) {
            return <ChatCodeBlock code={String(children)} language={language} />;
          }
          
          return (
            <code className="bg-gray-100 dark:bg-gray-800 text-gray-800 dark:text-gray-200 px-1 py-0.5 rounded text-base font-mono">
              {children}
            </code>
          );
        },
        // 단락
        p: ({ children }) => (
          <p className="mb-3 leading-relaxed text-gray-900 dark:text-gray-100 break-words text-base">
            {children}
          </p>
        ),
        // 링크
        a: ({ href, children }) => {
          if (!href) return <span>{children}</span>;
          
          // rag/download 관련 링크는 일반 텍스트로 처리 (별도 다운로드 섹션이 있으므로)
          if (href.includes('rag/download') || href.includes('/download?filename=') || href.includes('filename=')) {
            // 중첩된 링크에서 텍스트만 추출
            const textContent = React.Children.toArray(children).map(child => {
              if (typeof child === 'string') return child;
              if (React.isValidElement(child) && (child as any).props?.children) {
                return React.Children.toArray((child as any).props.children).join('');
              }
              return '';
            }).join('');
            
            return <span className="text-gray-600 dark:text-gray-400">{textContent}</span>;
          }
          
          // 내부 링크 처리
          if (href.startsWith('#')) {
            return (
              <a
                href={href}
                className="text-blue-600 dark:text-blue-400 hover:underline"
              >
                {children}
              </a>
            );
          }
          

                    
            return (
            <a
              href={href}
              target="_blank"
              rel="noopener noreferrer"
              className="text-blue-600 dark:text-blue-400 hover:underline break-words break-all"
              style={{ wordBreak: 'break-all', overflowWrap: 'anywhere' }}
            >
              {children}
            </a>
          );
        },
        // 리스트
        ul: ({ children }) => (
          <ul className="list-disc list-outside my-3 ml-6 space-y-2 text-base text-gray-900 dark:text-gray-100">
            {children}
          </ul>
        ),
        ol: ({ children }) => (
          <ol className="list-decimal list-outside my-3 ml-6 space-y-2 text-base text-gray-900 dark:text-gray-100">
            {children}
          </ol>
        ),
        li: ({ children }) => {
          return <li className="leading-relaxed">{children}</li>;
        },
        // 제목
        h1: ({ children }) => {
          // 출처(References) 제목인지 확인
          const childrenText = React.Children.toArray(children).join('').toLowerCase();
          const isReferences = childrenText.includes('출처') || childrenText.includes('references');
          
          if (isReferences) {
            return (
              <h3 className="text-lg font-bold text-gray-900 dark:text-white my-3 flex items-center space-x-2 pt-10">
                <svg className="w-5 h-5 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                </svg>
                <span>{children}</span>
              </h3>
            );
          }
          
          return (
            <h1 className="text-xl font-bold text-gray-900 dark:text-white my-3">
              {children}
            </h1>
          );
        },
        h2: ({ children }) => (
          <h2 className="text-lg font-bold text-gray-900 dark:text-white mt-8 my-2">
            {children}
          </h2>
        ),
        h3: ({ children }) => (
          <h4 className="text-base font-bold text-gray-900 dark:text-white mt-8 my-2">
            {children}
          </h4>
        ),
        // 인용문
        blockquote: ({ children }) => (
          <blockquote className="border-l-4 border-gray-300 dark:border-gray-600 pl-4 my-2 italic text-gray-600 dark:text-gray-400 text-base">
            {children}
          </blockquote>
        ),
        // 구분선
        hr: () => (
          <hr className="border-gray-300 dark:border-gray-600 my-4" />
        ),

        // 표 - 개선된 스타일링
        table: ({ children }) => (
          <div className="overflow-x-auto my-4 rounded-lg border border-gray-300 dark:border-gray-600 shadow-sm">
            <table className="min-w-full divide-y divide-gray-200 dark:divide-gray-700">
              {children}
            </table>
          </div>
        ),
        thead: ({ children }) => (
          <thead className="bg-gray-50 dark:bg-gray-800">
            {children}
          </thead>
        ),
        tbody: ({ children }) => (
          <tbody className="bg-white dark:bg-gray-900 divide-y divide-gray-200 dark:divide-gray-700">
            {children}
          </tbody>
        ),
        tr: ({ children }) => (
          <tr className="hover:bg-gray-50 dark:hover:bg-gray-800 transition-colors">
            {children}
          </tr>
        ),
        th: ({ children }) => (
          <th className="px-6 py-3 text-left text-sm font-medium text-gray-500 dark:text-gray-300 uppercase tracking-wider">
            {children}
          </th>
        ),
        td: ({ children }) => {
          // children을 문자열로 변환하여 <br> 태그 처리
          const processChildren = (children: any): any => {
            if (typeof children === 'string') {
              // <br> 태그를 실제 줄바꿈으로 변환
              return children.split('<br>').map((part, index, array) => (
                <React.Fragment key={index}>
                  {part}
                  {index < array.length - 1 && <br />}
                </React.Fragment>
              ));
            }
            if (Array.isArray(children)) {
              return children.map((child, index) => (
                <React.Fragment key={index}>
                  {processChildren(child)}
                </React.Fragment>
              ));
            }
            return children;
          };

          return (
            <td className="px-6 py-4 text-sm text-gray-900 dark:text-gray-100 break-words">
              {processChildren(children)}
            </td>
          );
        },
        }}
      >
        {cleanContent}
      </ReactMarkdown>
      
      {/* 다운로드 섹션 렌더링 - 웹 링크와 문서 구분 */}
      {downloadInfo.length > 0 && (
        <div className="mt-6 pt-4 border-t border-gray-200 dark:border-gray-700">
          {/* 출처 제목 추가 */}
          <h2 className="text-lg font-bold text-gray-900 dark:text-white mb-4 flex items-center space-x-2">
            <svg className="w-5 h-5 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
              <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M9 12h6m-6 4h6m2 5H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
            </svg>
            <span>출처</span>
          </h2>
          <div className="space-y-1">
            {downloadInfo.map((item, index) => {
              // source 필드로 웹과 문서 구분
              // console.log(`📋 Item ${index}:`, { 
              //   filename: item.filename, 
              //   source: item.source, 
              //   url: item.url,
              //   pages: item.pages 
              // });
              
              // source 필드를 기준으로 웹 링크인지 문서인지 판단
              const isWebLink = item.source === 'web';
              
              if (isWebLink && item.url) {
                // 웹 링크는 일반 링크로 표시
                return (
                  <div key={index} className="flex items-center pl-5 border border-blue-200 dark:border-blue-700">
                    <svg className="w-5 h-5 text-blue-600 dark:text-blue-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M21 12a9 9 0 01-9 9m9-9a9 9 0 00-9-9m9 9H3m9 9a9 9 0 01-9-9m9 9c1.657 0 3-4.03 3-9s-1.343-9-3-9m0 18c-1.657 0-3-4.03-3-9s1.343-9 3-9m-9 9a9 9 0 019-9" />
                    </svg>
                    <div>
                      <a
                        href={item.url}
                        target="_blank"
                        rel="noopener noreferrer"
                        className="pl-2 text-sm font-medium text-blue-600 dark:text-blue-400 hover:underline"
                      >
                        {item.title || item.filename}
                      </a>
                    </div>
                  </div>
                );
              } else {
                // 문서는 다운로드 버튼으로 표시
                const apiServerUrl = getApiServerUrl();
                const downloadUrl = `${apiServerUrl}/rag/download?filename=${encodeURIComponent(item.filename)}`;
                
                return (
                  <div key={index} className="flex items-center justify-between pl-3 p-2 bg-gray-50 dark:bg-gray-800 rounded-lg border border-gray-200 dark:border-gray-600">
                    <div className="flex items-center space-x-3">
                      <svg className="w-5 h-5 text-green-600 dark:text-green-400" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M7 21h10a2 2 0 002-2V9.414a1 1 0 00-.293-.707l-5.414-5.414A1 1 0 0012.586 3H7a2 2 0 00-2 2v14a2 2 0 002 2z" />
                        <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M16 7h-4V3" />
                      </svg>
                      <div>
                        <div className="text-sm font-medium text-gray-900 dark:text-gray-100">
                          {item.filename}
                          {item.pages && item.pages.length > 0 && (
                            <span className="ml-2 text-gray-500 dark:text-gray-400">
                              (페이지: {item.pages.join(', ')})
                            </span>
                          )}
                        </div>
                      </div>
                    </div>
                    <button
                    onClick={async () => {
                      try {
                        console.log('[다운로드] 시작:', item.filename);
                        
                        const response = await fetch(downloadUrl, {
                          method: 'GET',
                          headers: {
                            'X-API-Key': localStorage.getItem('apiKey') || 'airun_1_3d85009c98964579c622eded69997b16',
                            'Authorization': `Bearer ${localStorage.getItem('userToken') || ''}`
                          }
                        });
                        
                        if (!response.ok) {
                          const errorData = await response.json().catch(() => ({ error: { message: '파일을 찾을 수 없습니다.' } }));
                          throw new Error(errorData.error?.message || '파일을 찾을 수 없습니다.');
                        }
                        
                        const blob = await response.blob();
                        const url = window.URL.createObjectURL(blob);
                        const a = document.createElement('a');
                        a.href = url;
                        a.download = item.filename;
                        document.body.appendChild(a);
                        a.click();
                        window.URL.revokeObjectURL(url);
                        document.body.removeChild(a);
                        
                        console.log('[다운로드] 완료:', item.filename);
                      } catch (error) {
                        console.error('다운로드 오류:', error);
                        alert('다운로드에 실패했습니다: ' + (error instanceof Error ? error.message : String(error)));
                      }
                    }}
                    className="inline-flex items-center px-3 py-1.5 text-xs font-medium text-white bg-blue-600 hover:bg-blue-700 rounded-md transition-colors duration-200"
                    title={`${item.filename} 다운로드`}
                  >
                    <svg className="w-3 h-3 mr-1" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M12 10v6m0 0l-3-3m3 3l3-3m2 8H7a2 2 0 01-2-2V5a2 2 0 012-2h5.586a1 1 0 01.707.293l5.414 5.414a1 1 0 01.293.707V19a2 2 0 01-2 2z" />
                    </svg>
                    다운로드
                  </button>
                </div>
              );
              }
            })}
          </div>
        </div>
      )}


    </div>
  );
}; 