'use client';

import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Dialog, DialogContent, DialogHeader, DialogTitle } from '@/components/ui/dialog';
import { Upload, Trash2, Plus, Globe, FileText, Database, BookOpen } from 'lucide-react';

interface Dataset {
  id: string;
  name: string;
  type: string;
  size: number;
  created: string;
  status: string;
}

export default function LLMDatasetTab() {
  const [datasets, setDatasets] = useState<Dataset[]>([]);
  const [loading, setLoading] = useState(false);
  const [showCreateModal, setShowCreateModal] = useState(false);
  const [createType, setCreateType] = useState<string>('');

  useEffect(() => {
    loadDatasets();
  }, []);

  const loadDatasets = async () => {
    try {
      setLoading(true);
      const response = await fetch('/api/llm/datasets');
      const data = await response.json();
      if (data.success) {
        setDatasets(data.datasets || []);
      }
    } catch (error) {
      console.error('데이터셋 로드 실패:', error);
    } finally {
      setLoading(false);
    }
  };

  const handleDeleteDataset = async (id: string) => {
    if (!confirm('정말로 이 데이터셋을 삭제하시겠습니까?')) return;
    
    try {
      const response = await fetch(`/api/llm/datasets/${id}`, {
        method: 'DELETE'
      });
      const data = await response.json();
      if (data.success) {
        setDatasets(datasets.filter(d => d.id !== id));
      }
    } catch (error) {
      console.error('데이터셋 삭제 실패:', error);
    }
  };

  const handleCreateDataset = (type: string) => {
    setCreateType(type);
    setShowCreateModal(true);
  };

  return (
    <div className="space-y-6">
      {/* 데이터셋 생성 섹션 */}
      <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
        <div className="flex justify-between items-center mb-4">
          <h4 className="text-base font-semibold text-gray-900 dark:text-white">
            📚 데이터셋 생성
          </h4>
        </div>
        <div className="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
          <Button 
            onClick={() => handleCreateDataset('upload')} 
            className="flex flex-col items-center justify-center p-6 h-auto bg-gradient-to-br from-blue-50 to-blue-100 dark:from-blue-900/20 dark:to-blue-800/20 border border-blue-200 dark:border-blue-700 hover:from-blue-100 hover:to-blue-200 dark:hover:from-blue-800/30 dark:hover:to-blue-700/30 transition-all"
            variant="outline"
          >
            <Upload className="h-8 w-8 mb-2 text-blue-600 dark:text-blue-400" />
            <span className="font-medium text-blue-900 dark:text-blue-200">파일 업로드</span>
            <span className="text-xs text-blue-700 dark:text-blue-300 mt-1">JSON/CSV 파일</span>
          </Button>
          
          <Button 
            onClick={() => handleCreateDataset('web')} 
            className="flex flex-col items-center justify-center p-6 h-auto bg-gradient-to-br from-green-50 to-green-100 dark:from-green-900/20 dark:to-green-800/20 border border-green-200 dark:border-green-700 hover:from-green-100 hover:to-green-200 dark:hover:from-green-800/30 dark:hover:to-green-700/30 transition-all"
            variant="outline"
          >
            <Globe className="h-8 w-8 mb-2 text-green-600 dark:text-green-400" />
            <span className="font-medium text-green-900 dark:text-green-200">웹 크롤링</span>
            <span className="text-xs text-green-700 dark:text-green-300 mt-1">웹 콘텐츠 수집</span>
          </Button>
          
          <Button 
            onClick={() => handleCreateDataset('rag')} 
            className="flex flex-col items-center justify-center p-6 h-auto bg-gradient-to-br from-purple-50 to-purple-100 dark:from-purple-900/20 dark:to-purple-800/20 border border-purple-200 dark:border-purple-700 hover:from-purple-100 hover:to-purple-200 dark:hover:from-purple-800/30 dark:hover:to-purple-700/30 transition-all"
            variant="outline"
          >
            <Database className="h-8 w-8 mb-2 text-purple-600 dark:text-purple-400" />
            <span className="font-medium text-purple-900 dark:text-purple-200">RAG 변환</span>
            <span className="text-xs text-purple-700 dark:text-purple-300 mt-1">RAG 데이터 활용</span>
          </Button>
          
          <Button 
            onClick={() => handleCreateDataset('instruction')} 
            className="flex flex-col items-center justify-center p-6 h-auto bg-gradient-to-br from-orange-50 to-orange-100 dark:from-orange-900/20 dark:to-orange-800/20 border border-orange-200 dark:border-orange-700 hover:from-orange-100 hover:to-orange-200 dark:hover:from-orange-800/30 dark:hover:to-orange-700/30 transition-all"
            variant="outline"
          >
            <BookOpen className="h-8 w-8 mb-2 text-orange-600 dark:text-orange-400" />
            <span className="font-medium text-orange-900 dark:text-orange-200">문서 기반</span>
            <span className="text-xs text-orange-700 dark:text-orange-300 mt-1">문서에서 생성</span>
          </Button>
        </div>
      </div>

      {/* 데이터셋 목록 섹션 */}
      <div className="bg-[var(--card-bg)] rounded-xl shadow-sm border border-[var(--card-border)] p-6">
        <h4 className="text-base font-semibold text-gray-900 dark:text-white mb-4">
          📂 데이터셋 목록
        </h4>
        <div>
          {loading ? (
            <div className="text-center py-4 text-[var(--text-secondary)]">데이터셋을 불러오는 중...</div>
          ) : datasets.length === 0 ? (
            <div className="text-center py-8 text-[var(--text-muted)]">
              <Database className="h-12 w-12 mx-auto mb-4 opacity-50" />
              <p>생성된 데이터셋이 없습니다.</p>
              <p className="text-sm">위의 버튼을 사용해 새 데이터셋을 생성하세요.</p>
            </div>
          ) : (
            <div className="space-y-2">
              {datasets.map((dataset) => (
                <div key={dataset.id} className="flex items-center justify-between p-4 border rounded-lg">
                  <div className="flex-1">
                    <div className="font-medium">{dataset.name}</div>
                    <div className="text-sm text-[var(--text-secondary)]">
                      타입: {dataset.type} • 크기: {dataset.size} • 생성일: {dataset.created}
                    </div>
                  </div>
                  <div className="flex items-center gap-2">
                    <span className={`px-2 py-1 text-xs rounded-full ${
                      dataset.status === 'ready' ? 'bg-green-100 text-green-800' :
                      dataset.status === 'processing' ? 'bg-yellow-100 text-yellow-800' :
                      'bg-gray-100 text-gray-800'
                    }`}>
                      {dataset.status}
                    </span>
                    <Button
                      variant="ghost"
                      size="sm"
                      onClick={() => handleDeleteDataset(dataset.id)}
                      className="text-red-600 hover:text-red-800"
                    >
                      <Trash2 className="h-4 w-4" />
                    </Button>
                  </div>
                </div>
              ))}
            </div>
          )}
        </div>
      </div>

      <Dialog open={showCreateModal} onOpenChange={setShowCreateModal}>
        <DialogContent className="max-w-2xl">
          <DialogHeader>
            <DialogTitle>새 데이터셋 생성</DialogTitle>
          </DialogHeader>
          <CreateDatasetForm type={createType} onClose={() => setShowCreateModal(false)} onSuccess={loadDatasets} />
        </DialogContent>
      </Dialog>
    </div>
  );
}

interface CreateDatasetFormProps {
  type: string;
  onClose: () => void;
  onSuccess: () => void;
}

function CreateDatasetForm({ type, onClose, onSuccess }: CreateDatasetFormProps) {
  const [formData, setFormData] = useState({
    name: '',
    description: '',
    url: '',
    file: null as File | null,
    promptTemplate: '',
    ragQuery: ''
  });
  const [loading, setLoading] = useState(false);

  const handleSubmit = async (e: React.FormEvent) => {
    e.preventDefault();
    setLoading(true);

    try {
      let endpoint = '';
      let body: any = {};

      switch (type) {
        case 'upload':
          if (!formData.file) return;
          const uploadFormData = new FormData();
          uploadFormData.append('file', formData.file);
          uploadFormData.append('name', formData.name);
          uploadFormData.append('description', formData.description);

          const uploadResponse = await fetch('/api/llm/datasets', {
            method: 'POST',
            body: uploadFormData
          });
          break;

        case 'web':
          body = { type: 'web-crawl', url: formData.url, name: formData.name };
          break;

        case 'rag':
          body = { type: 'rag-conversion', query: formData.ragQuery, name: formData.name };
          break;

        case 'instruction':
          body = { type: 'document-instruction', template: formData.promptTemplate, name: formData.name };
          break;
      }

      if (type !== 'upload') {
        await fetch('/api/llm/datasets', {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(body)
        });
      }

      onSuccess();
      onClose();
    } catch (error) {
      console.error('데이터셋 생성 실패:', error);
    } finally {
      setLoading(false);
    }
  };

  return (
    <form onSubmit={handleSubmit} className="space-y-4">
      <div>
        <Label htmlFor="name">데이터셋 이름</Label>
        <Input
          id="name"
          value={formData.name}
          onChange={(e) => setFormData(prev => ({ ...prev, name: e.target.value }))}
          required
        />
      </div>

      <div>
        <Label htmlFor="description">설명</Label>
        <Textarea
          id="description"
          value={formData.description}
          onChange={(e) => setFormData(prev => ({ ...prev, description: e.target.value }))}
          rows={3}
        />
      </div>

      {type === 'upload' && (
        <div>
          <Label htmlFor="file">파일 선택</Label>
          <Input
            id="file"
            type="file"
            accept=".txt,.json,.csv,.jsonl"
            onChange={(e) => setFormData(prev => ({ ...prev, file: e.target.files?.[0] || null }))}
            required
          />
        </div>
      )}

      {type === 'web' && (
        <div>
          <Label htmlFor="url">웹 URL</Label>
          <Input
            id="url"
            type="url"
            value={formData.url}
            onChange={(e) => setFormData(prev => ({ ...prev, url: e.target.value }))}
            placeholder="https://example.com"
            required
          />
        </div>
      )}

      {type === 'rag' && (
        <div>
          <Label htmlFor="ragQuery">RAG 검색 쿼리</Label>
          <Input
            id="ragQuery"
            value={formData.ragQuery}
            onChange={(e) => setFormData(prev => ({ ...prev, ragQuery: e.target.value }))}
            placeholder="검색할 내용을 입력하세요"
            required
          />
        </div>
      )}

      {type === 'instruction' && (
        <div>
          <Label htmlFor="promptTemplate">프롬프트 템플릿</Label>
          <Textarea
            id="promptTemplate"
            value={formData.promptTemplate}
            onChange={(e) => setFormData(prev => ({ ...prev, promptTemplate: e.target.value }))}
            placeholder="문서를 기반으로 질문과 답변을 생성하는 템플릿"
            rows={4}
            required
          />
        </div>
      )}

      <div className="flex justify-end gap-2 pt-4">
        <Button type="button" variant="outline" onClick={onClose}>취소</Button>
        <Button type="submit" disabled={loading}>
          {loading ? '생성 중...' : '생성하기'}
        </Button>
      </div>
    </form>
  );
}