'use client';

import React, { useState, useEffect } from 'react';
import { Button } from '@/components/ui/button';
import { Input } from '@/components/ui/input';
import { Label } from '@/components/ui/label';
import { Textarea } from '@/components/ui/textarea';
import { Select, SelectContent, SelectItem, SelectTrigger, SelectValue } from '@/components/ui/select';
import { Card, CardContent, CardHeader, CardTitle } from '@/components/ui/card';
import { Slider } from '@/components/ui/slider';
import { Badge } from '@/components/ui/badge';
import { Tabs, TabsContent, TabsList, TabsTrigger } from '@/components/ui/tabs';
import { Send, RotateCcw, Copy, Save, MessageSquare, Settings, History } from 'lucide-react';

interface Model {
  id: string;
  name: string;
  type: 'full' | 'lora';
  status: 'active' | 'inactive';
}

interface ChatMessage {
  role: 'user' | 'assistant';
  content: string;
  timestamp: string;
}

interface ChatSession {
  id: string;
  name: string;
  messages: ChatMessage[];
  model: string;
  createdAt: string;
}

export default function LLMPlaygroundTab() {
  const [models, setModels] = useState<Model[]>([]);
  const [selectedModel, setSelectedModel] = useState<string>('');
  const [currentSession, setCurrentSession] = useState<ChatSession | null>(null);
  const [sessions, setSessions] = useState<ChatSession[]>([]);
  const [message, setMessage] = useState('');
  const [loading, setLoading] = useState(false);
  const [config, setConfig] = useState({
    temperature: 0.7,
    maxTokens: 150,
    topP: 0.9,
    topK: 50
  });

  useEffect(() => {
    loadModels();
    loadSessions();
  }, []);

  const loadModels = async () => {
    try {
      const response = await fetch('/api/llm/models');
      const data = await response.json();
      if (data.success) {
        const activeModels = data.models?.filter((m: Model) => m.status === 'active') || [];
        setModels(activeModels);
        if (activeModels.length > 0 && !selectedModel) {
          setSelectedModel(activeModels[0].id);
        }
      }
    } catch (error) {
      console.error('모델 로드 실패:', error);
    }
  };

  const loadSessions = async () => {
    try {
      const response = await fetch('/api/llm/playground/sessions');
      const data = await response.json();
      if (data.success) {
        setSessions(data.sessions || []);
      }
    } catch (error) {
      console.error('세션 로드 실패:', error);
    }
  };

  const createNewSession = () => {
    const newSession: ChatSession = {
      id: Date.now().toString(),
      name: `New Chat ${sessions.length + 1}`,
      messages: [],
      model: selectedModel,
      createdAt: new Date().toISOString()
    };
    setCurrentSession(newSession);
    setSessions([newSession, ...sessions]);
  };

  const sendMessage = async () => {
    if (!message.trim() || !selectedModel || loading) return;

    const userMessage: ChatMessage = {
      role: 'user',
      content: message,
      timestamp: new Date().toISOString()
    };

    let session = currentSession;
    if (!session) {
      session = {
        id: Date.now().toString(),
        name: message.slice(0, 30) + '...',
        messages: [],
        model: selectedModel,
        createdAt: new Date().toISOString()
      };
      setCurrentSession(session);
      setSessions([session, ...sessions]);
    }

    const updatedMessages = [...session.messages, userMessage];
    const updatedSession = { ...session, messages: updatedMessages };
    setCurrentSession(updatedSession);
    setSessions(sessions.map(s => s.id === session.id ? updatedSession : s));

    setMessage('');
    setLoading(true);

    try {
      const response = await fetch('/api/llm/playground/chat', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify({
          model: selectedModel,
          message: message,
          history: session.messages,
          config
        })
      });

      const data = await response.json();
      
      if (data.success) {
        const assistantMessage: ChatMessage = {
          role: 'assistant',
          content: data.response,
          timestamp: new Date().toISOString()
        };

        const finalMessages = [...updatedMessages, assistantMessage];
        const finalSession = { ...updatedSession, messages: finalMessages };
        setCurrentSession(finalSession);
        setSessions(sessions.map(s => s.id === session.id ? finalSession : s));
      }
    } catch (error) {
      console.error('메시지 전송 실패:', error);
    } finally {
      setLoading(false);
    }
  };

  const clearSession = () => {
    if (currentSession) {
      setCurrentSession({ ...currentSession, messages: [] });
    }
  };

  const copyMessage = (content: string) => {
    navigator.clipboard.writeText(content);
  };

  const saveSession = async () => {
    if (!currentSession) return;

    try {
      await fetch('/api/llm/playground/sessions', {
        method: 'POST',
        headers: { 'Content-Type': 'application/json' },
        body: JSON.stringify(currentSession)
      });
    } catch (error) {
      console.error('세션 저장 실패:', error);
    }
  };

  return (
    <div className="h-full flex">
      {/* 사이드바 */}
      <div className="w-80 border-r border-[var(--card-border)] bg-[var(--sidebar-bg)] flex flex-col">
        <div className="p-4 border-b border-[var(--card-border)]">
          <Button onClick={createNewSession} className="w-full flex items-center gap-2">
            <MessageSquare className="h-4 w-4" />
            새 대화
          </Button>
        </div>

        <div className="flex-1 overflow-auto p-2">
          <div className="space-y-1">
            {sessions.map((session) => (
              <div
                key={session.id}
                onClick={() => setCurrentSession(session)}
                className={`p-3 rounded-lg cursor-pointer transition-colors ${
                  currentSession?.id === session.id ? 'bg-[var(--accent-blue)] bg-opacity-10' : 'hover:bg-[var(--card-bg)]'
                }`}
              >
                <div className="font-medium text-sm truncate text-[var(--text-primary)]">{session.name}</div>
                <div className="text-xs text-[var(--text-muted)]">
                  {session.messages.length}개 메시지 • {new Date(session.createdAt).toLocaleDateString()}
                </div>
              </div>
            ))}
          </div>
        </div>

        {/* 설정 패널 */}
        <div className="p-4 border-t border-[var(--card-border)] bg-[var(--card-bg)]">
          <Tabs defaultValue="model" className="w-full">
            <TabsList className="grid w-full grid-cols-2 bg-[var(--sidebar-bg)] border border-[var(--card-border)]">
              <TabsTrigger 
                value="model" 
                className="data-[state=active]:bg-[var(--card-bg)] data-[state=active]:text-[var(--text-primary)] data-[state=active]:shadow-sm text-[var(--text-secondary)]"
              >
                모델
              </TabsTrigger>
              <TabsTrigger 
                value="params" 
                className="data-[state=active]:bg-[var(--card-bg)] data-[state=active]:text-[var(--text-primary)] data-[state=active]:shadow-sm text-[var(--text-secondary)]"
              >
                파라미터
              </TabsTrigger>
            </TabsList>
            
            <TabsContent value="model" className="space-y-3 mt-3">
              <div>
                <Label className="text-xs text-[var(--text-primary)] font-medium">모델 선택</Label>
                <Select value={selectedModel} onValueChange={setSelectedModel}>
                  <SelectTrigger className="h-8 bg-[var(--card-bg)] border-[var(--card-border)] text-[var(--text-primary)]">
                    <SelectValue placeholder="모델 선택" />
                  </SelectTrigger>
                  <SelectContent>
                    {models.map(model => (
                      <SelectItem key={model.id} value={model.id}>
                        <div className="flex items-center gap-2">
                          <Badge variant={model.type === 'lora' ? 'secondary' : 'default'} className="text-xs">
                            {model.type.toUpperCase()}
                          </Badge>
                          <span className="text-sm">{model.name}</span>
                        </div>
                      </SelectItem>
                    ))}
                  </SelectContent>
                </Select>
              </div>
            </TabsContent>
            
            <TabsContent value="params" className="space-y-3 mt-3">
              <div>
                <Label className="text-xs text-[var(--text-primary)] font-medium">Temperature: {config.temperature}</Label>
                <Slider
                  value={[config.temperature]}
                  onValueChange={([value]) => setConfig(prev => ({ ...prev, temperature: value }))}
                  max={2}
                  min={0}
                  step={0.1}
                  className="mt-1"
                />
              </div>
              
              <div>
                <Label className="text-xs text-[var(--text-primary)] font-medium">Max Tokens: {config.maxTokens}</Label>
                <Slider
                  value={[config.maxTokens]}
                  onValueChange={([value]) => setConfig(prev => ({ ...prev, maxTokens: value }))}
                  max={500}
                  min={50}
                  step={25}
                  className="mt-1"
                />
              </div>

              <div>
                <Label className="text-xs text-[var(--text-primary)] font-medium">Top P: {config.topP}</Label>
                <Slider
                  value={[config.topP]}
                  onValueChange={([value]) => setConfig(prev => ({ ...prev, topP: value }))}
                  max={1}
                  min={0}
                  step={0.1}
                  className="mt-1"
                />
              </div>
            </TabsContent>
          </Tabs>
        </div>
      </div>

      {/* 메인 채팅 영역 */}
      <div className="flex-1 flex flex-col">
        {!currentSession ? (
          <div className="flex-1 flex items-center justify-center bg-[var(--body-bg)]">
            <div className="text-center">
              <MessageSquare className="h-12 w-12 mx-auto mb-4 text-[var(--text-muted)]" />
              <h3 className="text-lg font-semibold mb-2 text-[var(--text-primary)]">LLM 플레이그라운드</h3>
              <p className="text-[var(--text-secondary)] mb-4">학습된 모델과 대화를 시작해보세요.</p>
              <Button onClick={createNewSession} className="flex items-center gap-2">
                <MessageSquare className="h-4 w-4" />
                새 대화 시작
              </Button>
            </div>
          </div>
        ) : (
          <>
            {/* 채팅 헤더 */}
            <div className="border-b border-[var(--card-border)] p-4 flex items-center justify-between bg-[var(--card-bg)]">
              <div>
                <h3 className="font-semibold text-[var(--text-primary)]">{currentSession.name}</h3>
                <p className="text-sm text-[var(--text-secondary)]">
                  {models.find(m => m.id === selectedModel)?.name || '모델 선택'}
                </p>
              </div>
              <div className="flex gap-2">
                <Button variant="outline" size="sm" onClick={clearSession}>
                  <RotateCcw className="h-4 w-4" />
                  초기화
                </Button>
                <Button variant="outline" size="sm" onClick={saveSession}>
                  <Save className="h-4 w-4" />
                  저장
                </Button>
              </div>
            </div>

            {/* 메시지 목록 */}
            <div className="flex-1 overflow-auto p-4 space-y-4 bg-[var(--body-bg)]">
              {currentSession.messages.length === 0 ? (
                <div className="text-center text-[var(--text-muted)] py-8">
                  <p>대화를 시작해보세요!</p>
                </div>
              ) : (
                currentSession.messages.map((msg, index) => (
                  <div key={index} className={`flex ${msg.role === 'user' ? 'justify-end' : 'justify-start'}`}>
                    <div className={`max-w-[70%] p-4 rounded-lg relative group shadow-sm ${
                      msg.role === 'user' 
                        ? 'bg-blue-500 text-white ml-4' 
                        : 'bg-[var(--card-bg)] text-[var(--text-primary)] border border-[var(--card-border)] mr-4'
                    }`}>
                      <div className="whitespace-pre-wrap">{msg.content}</div>
                      <div className={`text-xs mt-2 opacity-70 ${
                        msg.role === 'user' ? 'text-blue-100' : 'text-[var(--text-muted)]'
                      }`}>
                        {new Date(msg.timestamp).toLocaleTimeString()}
                      </div>
                      <Button
                        variant="ghost"
                        size="sm"
                        onClick={() => copyMessage(msg.content)}
                        className={`absolute top-2 right-2 opacity-0 group-hover:opacity-100 transition-opacity h-6 w-6 p-1 ${
                          msg.role === 'user' ? 'hover:bg-blue-600' : 'hover:bg-[var(--card-hover)]'
                        }`}
                      >
                        <Copy className="h-3 w-3" />
                      </Button>
                    </div>
                  </div>
                ))
              )}
              
              {loading && (
                <div className="flex justify-start">
                  <div className="bg-[var(--card-bg)] border border-[var(--card-border)] p-4 rounded-lg shadow-sm mr-4">
                    <div className="flex items-center gap-2">
                      <div className="animate-pulse w-2 h-2 bg-[var(--text-muted)] rounded-full"></div>
                      <div className="animate-pulse w-2 h-2 bg-[var(--text-muted)] rounded-full" style={{animationDelay: '0.2s'}}></div>
                      <div className="animate-pulse w-2 h-2 bg-[var(--text-muted)] rounded-full" style={{animationDelay: '0.4s'}}></div>
                    </div>
                    <div className="text-xs text-[var(--text-muted)] mt-2">AI가 응답하고 있습니다...</div>
                  </div>
                </div>
              )}
            </div>

            {/* 입력 영역 */}
            <div className="border-t border-[var(--card-border)] p-4 bg-[var(--card-bg)] shadow-sm">
              <div className="flex gap-3">
                <Textarea
                  value={message}
                  onChange={(e) => setMessage(e.target.value)}
                  placeholder="메시지를 입력하세요..."
                  className="flex-1 min-h-[40px] max-h-32 bg-[var(--body-bg)] border-[var(--card-border)] text-[var(--text-primary)] placeholder:text-[var(--text-muted)] resize-none"
                  onKeyDown={(e) => {
                    if (e.key === 'Enter' && !e.shiftKey) {
                      e.preventDefault();
                      sendMessage();
                    }
                  }}
                />
                <Button 
                  onClick={sendMessage} 
                  disabled={!message.trim() || loading || !selectedModel}
                  className="flex items-center gap-2 bg-blue-500 hover:bg-blue-600 text-white px-4 py-2 h-auto"
                >
                  <Send className="h-4 w-4" />
                  전송
                </Button>
              </div>
              <div className="text-xs text-[var(--text-muted)] mt-2">
                Enter로 전송, Shift+Enter로 줄바꿈
              </div>
            </div>
          </>
        )}
      </div>
    </div>
  );
}