'use client';

import { FC, useState, useEffect, useRef } from 'react';
import { useParams, useRouter } from 'next/navigation';
import Link from 'next/link';
import ChatInput from '@/components/chat/ChatInput';
import ChatMessages from '@/components/chat/ChatMessages';
import ChatScrollButtons from '@/components/chat/ChatScrollButtons';
import { API_SERVER, getAuthHeaders, apiRequest, streamChatRequest, closeWebSocketConnection, syncChatRequest } from '@/utils/api';
import { Session } from '@/types/session';
import { Message } from '@/types/message';

interface ChatUIProps {
  onSendMessage?: (message: string, useRag: boolean, useWebSearch: boolean) => Promise<void>;
  isNewChat?: boolean;
  sessionId?: string;
}

interface SessionType {
  id: string;
  title: string;
  useRag: boolean;
  useWebSearch: boolean;
  [key: string]: any;
}

// 삭제 확인 모달 컴포넌트
const DeleteConfirmModal: FC<{
  isOpen: boolean;
  onClose: () => void;
  onConfirm: () => void;
  title: string;
}> = ({ isOpen, onClose, onConfirm, title }) => {
  if (!isOpen) return null;

  return (
    <div className="fixed inset-0 z-50 flex items-center justify-center">
      <div className="fixed inset-0 bg-black bg-opacity-50" onClick={onClose}></div>
      <div className="relative bg-white dark:bg-gray-800 rounded-lg p-6 max-w-sm w-full mx-4">
        <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100 mb-4">
          대화 삭제 확인
        </h3>
        <p className="text-sm text-gray-500 dark:text-gray-400 mb-4">
          다음 대화를 삭제하시겠습니까?<br />
          "{title}"
        </p>
        <div className="flex justify-end gap-3">
          <button
            onClick={onClose}
            className="px-4 py-2 text-sm font-medium text-gray-700 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-gray-700 rounded-md transition-colors"
          >
            취소
          </button>
          <button
            onClick={onConfirm}
            className="px-4 py-2 text-sm font-medium text-white bg-red-600 hover:bg-red-700 rounded-md transition-colors"
          >
            삭제
          </button>
        </div>
      </div>
    </div>
  );
};

export const ChatUI: FC<ChatUIProps> = ({ onSendMessage, isNewChat: initialIsNewChat = false, sessionId = '' }) => {
  const router = useRouter();
  const params = useParams();
  const [localChatId, setLocalChatId] = useState(sessionId);
  const chatId = !initialIsNewChat && localChatId ? localChatId : '';

  const [messages, setMessages] = useState<Message[]>([]);
  const [session, setSession] = useState<Session | null>(null);
  const [isLoading, setIsLoading] = useState(false);
  const [isFetching, setIsFetching] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [useRag, setUseRag] = useState(false);
  const [useWebSearch, setUseWebSearch] = useState(false);
  const [enableSmartResearch, setEnableSmartResearch] = useState(false);
  const [chatHistory, setChatHistory] = useState<Array<{id: string, title: string}>>([]);
  const [isNewChat, setIsNewChat] = useState(initialIsNewChat);
  
  // 스크롤 관련 상태 및 참조
  const messagesContainerRef = useRef<HTMLDivElement>(null);
  const messagesStartRef = useRef<HTMLDivElement>(null);
  const messagesEndRef = useRef<HTMLDivElement>(null);
  const [isAtBottom, setIsAtBottom] = useState(true);
  const [isAtTop, setIsAtTop] = useState(true);
  const [isOverflowing, setIsOverflowing] = useState(false);

  // 스트리밍 상태 관리
  const [isStreaming, setIsStreaming] = useState(false);
  const [streamContent, setStreamContent] = useState<string>('');
  const streamMessageRef = useRef<Message | null>(null);
  
  // 로컬 스토리지에서 설정 불러오기
  useEffect(() => {
    if (typeof window !== 'undefined') {
      const savedUseRag = localStorage.getItem('useRag');
      const savedUseWebSearch = localStorage.getItem('useWebSearch');
      const savedEnableSmartResearch = localStorage.getItem('enableSmartResearch');
      
      if (savedUseRag !== null) {
        setUseRag(savedUseRag === 'true');
      }
      
      if (savedUseWebSearch !== null) {
        setUseWebSearch(savedUseWebSearch === 'true');
      }
      
      if (savedEnableSmartResearch !== null) {
        setEnableSmartResearch(savedEnableSmartResearch === 'true');
      }
    }
  }, []);
  
  // 설정 변경시 로컬 스토리지에 저장
  useEffect(() => {
    if (typeof window !== 'undefined') {
      localStorage.setItem('useRag', useRag.toString());
      localStorage.setItem('useWebSearch', useWebSearch.toString());
      localStorage.setItem('enableSmartResearch', enableSmartResearch.toString());
    }
  }, [useRag, useWebSearch, enableSmartResearch]);

  // 채팅 목록 가져오기
  // 채팅 목록 가져오기
  const fetchChatList = async () => {
    try {
      console.log('채팅 목록 가져오기 시작...');
      
      // API 서버에서 직접 세션 목록 가져오기
      const response = await apiRequest<Record<string, any>>('/sessions', {
        method: 'GET'
      });
      
      console.log('채팅 목록 응답:', response);
      
      if (response.success && response.data) {
        // API 서버 응답이 객체 형태일 경우 배열로 변환
        const sessions = Object.entries(response.data).map(([id, session]: [string, any]) => ({
          id,
          title: session.history && session.history.length > 0 
            ? session.history[0].content.substring(0, 30) + (session.history[0].content.length > 30 ? '...' : '')
            : '새 대화',
          updatedAt: session.last_active || session.created_at || new Date().toISOString(),
          type: session.type // type 필드 추가
        }));
        
        // type이 'code'가 아닌 세션만 필터링
        const filteredSessions = sessions.filter(session => session.type !== 'code');
        
        // 최신 대화가 위로 오도록 정렬
        filteredSessions.sort((a, b) => new Date(b.updatedAt).getTime() - new Date(a.updatedAt).getTime());
        
        console.log('채팅 목록 불러오기 성공, 항목 수:', filteredSessions.length);
        setChatHistory(filteredSessions.slice(0, 20)); // 최대 20개까지 표시
      } else {
        console.log('응답 데이터 형식이 올바르지 않습니다:', response);
        setChatHistory([]);
      }
    } catch (err) {
      console.error('채팅 목록 가져오기 실패:', err);
      setChatHistory([]);
    }
  };

  // 세션 및 메시지 데이터 불러오기
  const fetchSessionData = async () => {
    setIsFetching(true);
    setError(null);
    
    try {
      // API 서버에서 직접 세션 정보 가져오기
      const sessionResponse = await apiRequest<any>(`/sessions/${chatId}`, {
        method: 'GET'
      });
      
      if (!sessionResponse.success || !sessionResponse.data) {
        throw new Error(sessionResponse.error?.message || '세션 정보를 불러오는데 실패했습니다.');
      }
      
      // 세션 데이터 처리
      const sessionData = sessionResponse.data;
      setSession({
        id: chatId,
        title: sessionData.history && sessionData.history.length > 0 
          ? sessionData.history[0].content.substring(0, 30) + (sessionData.history[0].content.length > 30 ? '...' : '')
          : '새 대화',
        createdAt: sessionData.created_at || new Date().toISOString(),
        updatedAt: sessionData.last_active || new Date().toISOString(),
        useRag: false,
        useWebSearch: false,
        messageCount: sessionData.history?.length || 0,
        provider: sessionData.provider || 'gemini',
        model: sessionData.model
      });
      
      // 대화 내역 처리 (세션 데이터에 이미 포함되어 있음)
      if (sessionData.history && Array.isArray(sessionData.history)) {
        const formattedMessages = sessionData.history.map((msg: any, index: number) => ({
          id: `msg-${index}-${Date.now()}`,
          content: msg.content,
          role: msg.role,
          createdAt: msg.timestamp || new Date().toISOString()
        }));
        setMessages(formattedMessages);
      } else {
        setMessages([]);
      }
    } catch (err) {
      console.error('Error fetching chat data:', err);
      setError(err instanceof Error ? err.message : '알 수 없는 오류가 발생했습니다.');
    } finally {
      setIsFetching(false);
    }
  };

  const [deleteModalOpen, setDeleteModalOpen] = useState(false);
  const [sessionToDelete, setSessionToDelete] = useState<{id: string, title: string} | null>(null);

  // 세션 삭제 처리
  const handleDeleteSession = async (sessionId: string, title: string, e: React.MouseEvent) => {
    e.preventDefault();
    e.stopPropagation();
    
    setSessionToDelete({ id: sessionId, title });
    setDeleteModalOpen(true);
  };

  const confirmDelete = async () => {
    if (!sessionToDelete) return;
    
    try {
      const response = await apiRequest<any>(`/sessions/${sessionToDelete.id}`, {
        method: 'DELETE'
      });
      
      if (response.success) {
        console.log('대화 삭제 성공:', sessionToDelete.id);
        
        // 현재 보고 있는 대화가 삭제된 경우 새 대화 페이지로 이동
        if (sessionToDelete.id === chatId) {
          router.push('/chat/new');
        } else {
          // 목록에서만 제거
          setChatHistory(prev => prev.filter(chat => chat.id !== sessionToDelete.id));
        }
      } else {
        console.error('대화 삭제 실패:', response.error);
      }
    } catch (err) {
      console.error('대화 삭제 중 오류 발생:', err);
    } finally {
      setDeleteModalOpen(false);
      setSessionToDelete(null);
    }
  };

  // isNewChat일 때 환영 메시지 표시, 아닐 때 세션 데이터 불러오기
  useEffect(() => {
    if (isNewChat) {
      setMessages([{
        id: `welcome-${Date.now()}`,
        content: "안녕하세요! 어떻게 도와드릴까요?",
        role: 'assistant',
        createdAt: new Date().toISOString()
      }]);
      setSession(null);
      setError(null);
      setIsFetching(false);
    } else if (chatId) {
      // 채팅 ID가 있을 때만 세션 데이터 불러오기
      fetchSessionData();
    } else {
      // 세션 ID가 없는 경우에도 로딩 상태 종료
      setIsFetching(false);
    }
    fetchChatList();
  }, [chatId, isNewChat]);

  // 스크롤 관련 핸들러
  const handleScroll = () => {
    if (!messagesContainerRef.current) return;
    
    const { scrollTop, scrollHeight, clientHeight } = messagesContainerRef.current;
    
    setIsAtBottom(scrollHeight - scrollTop - clientHeight < 50);
    setIsAtTop(scrollTop < 50);
    setIsOverflowing(scrollHeight > clientHeight);
  };

  const scrollToBottom = () => {
    messagesEndRef.current?.scrollIntoView({ behavior: 'smooth' });
    setIsAtBottom(true);
  };

  const scrollToTop = () => {
    messagesStartRef.current?.scrollIntoView({ behavior: 'smooth' });
    setIsAtTop(true);
  };

  // 메시지가 추가될 때 스크롤 맨 아래로 이동
  useEffect(() => {
    if (isAtBottom) {
      scrollToBottom();
    }
  }, [messages]);

  useEffect(() => {
    // 컴포넌트 마운트시 웹소켓 연결 해제 (새로 연결하기 위해)
    closeWebSocketConnection();
    
    // 컴포넌트 언마운트시 웹소켓 연결 해제
    return () => {
      closeWebSocketConnection();
    };
  }, []);

  const sendMessage = async (content: string, options: any = {}) => {
    let assistantMessage: Message;
    let loadingDots = '';
    let loadingInterval: NodeJS.Timeout | null = null;
    
    try {
        if (!content.trim()) return;

        setError(null);
        setIsLoading(true);

        // 사용자 메시지 추가
        const userMessage: Message = {
            id: `user-${Date.now()}`,
            content,
            role: 'user',
            createdAt: new Date().toISOString()
        };
        setMessages((prev) => [...prev, userMessage]);

        // 채팅 옵션 설정
        const chatOptions = {
            ...options,
            sessionId: chatId,
            userId: options.userId,
            rag: useRag,
            web: useWebSearch,
            enableSmartResearch: enableSmartResearch
        };

        // AI 응답 메시지 추가 (로딩 상태로)
        assistantMessage = {
            id: `assistant-${Date.now()}`,
            content: 'Thinking',
            role: 'assistant',
            createdAt: new Date().toISOString(),
            isLoading: true
        };
        setMessages((prev) => [...prev, assistantMessage]);

        // 로딩 애니메이션 시작
        loadingInterval = setInterval(() => {
            loadingDots = loadingDots.length >= 3 ? '' : loadingDots + '.';
            setMessages((prev) =>
                prev.map((msg) =>
                    msg.id === assistantMessage.id
                        ? { ...msg, content: `Thinking${loadingDots}` }
                        : msg
                )
            );
        }, 500);

        // 동기식 채팅 요청
        const response = await syncChatRequest(content, chatOptions);
        console.log('API 응답:', response); // 디버깅용 로그

        if (response.success && response.data) {
            // 응답 메시지 업데이트
            const responseContent = response.data.data?.raw?.content || '';
            setMessages((prev) =>
                prev.map((msg) =>
                    msg.id === assistantMessage.id
                        ? { ...msg, content: responseContent, isLoading: false }
                        : msg
                )
            );

            // 새 세션 ID 설정 (새 채팅인 경우)
            const newSessionId = response.data.data?.sessionId;
            if (initialIsNewChat && newSessionId) {
                setLocalChatId(newSessionId);
                router.push(`/chat/${newSessionId}`);

                // 세션 정보 업데이트
                setSession({
                    id: newSessionId,
                    title: content.substring(0, 30) + (content.length > 30 ? '...' : ''),
                    createdAt: new Date().toISOString(),
                    updatedAt: new Date().toISOString(),
                    useRag: options.useRag || false,
                    useWebSearch: options.useWebSearch || false,
                    messageCount: 2,
                    provider: 'default',
                    model: 'default'
                });
            }

            // 채팅 목록 새로고침
            fetchChatList();
        } else {
            throw new Error(response.error?.message || '채팅 응답을 받지 못했습니다.');
        }
    } catch (error) {
        console.error('메시지 전송 중 에러:', error);
        setMessages((prev) =>
            prev.map((msg) =>
                msg.id === assistantMessage.id
                    ? { ...msg, content: error instanceof Error ? error.message : '알 수 없는 오류가 발생했습니다.', isLoading: false, error: true }
                    : msg
            )
        );
    } finally {
        setIsLoading(false);
        if (loadingInterval) {
            clearInterval(loadingInterval);
        }
    }
};

  return (
    <div className="h-full w-full flex">
      {/* 삭제 확인 모달 */}
      <DeleteConfirmModal
        isOpen={deleteModalOpen}
        onClose={() => {
          setDeleteModalOpen(false);
          setSessionToDelete(null);
        }}
        onConfirm={confirmDelete}
        title={sessionToDelete?.title || ''}
      />

      {/* 왼쪽 사이드바 */}
      <div className="hidden sm:flex flex-col w-64 bg-gray-900 h-full chat-sidebar">
        <div className="p-4">
          <Link 
            href="/chat/new" 
            className="w-full flex items-center justify-center gap-2 py-2 px-4 rounded-md bg-blue-600 hover:bg-blue-700 !text-white transition-colors"
          >
            <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" viewBox="0 0 20 20" fill="currentColor">
              <path fillRule="evenodd" d="M10 3a1 1 0 00-1 1v5H4a1 1 0 100 2h5v5a1 1 0 102 0v-5h5a1 1 0 100-2h-5V4a1 1 0 00-1-1z" clipRule="evenodd" />
            </svg>
            <span>새 대화</span>
          </Link>
        </div>
        
        {/* 전역 설정 */}
        <div className="px-4 py-2 mb-4">
          <div className="text-xs uppercase text-gray-500 font-semibold mb-2">설정</div>
          <label className="flex items-center text-xs text-gray-300 mb-2">
            <input
              type="checkbox"
              checked={useRag}
              onChange={(e) => setUseRag(e.target.checked)}
              className="form-checkbox h-3.5 w-3.5 mr-2 text-blue-600 bg-gray-800 border-0"
            />
            RAG 활성화
          </label>
          <label className="flex items-center text-xs text-gray-300">
            <input
              type="checkbox"
              checked={useWebSearch}
              onChange={(e) => setUseWebSearch(e.target.checked)}
              className="form-checkbox h-3.5 w-3.5 mr-2 text-blue-600 bg-gray-800 border-0"
            />
            웹 검색 활성화
          </label>
        </div>
        
        <div className="px-2 flex-1 overflow-hidden flex flex-col">
          <div className="text-xs uppercase text-gray-500 font-semibold px-2 py-1 flex justify-between items-center">
            <span>최근 대화</span>
            <span className="text-gray-500 text-xs">{chatHistory.length}개</span>
          </div>
          
          <div className="space-y-1 mt-1 overflow-y-auto flex-1">
            {chatHistory.length > 0 ? (
              chatHistory.map((chat) => (
                <div key={chat.id} className="group relative">
                  <Link
                    href={`/chat/${chat.id}`}
                    className={`flex items-center px-2 py-1.5 rounded-md ${
                      chat.id === chatId 
                        ? 'bg-gray-800/50 text-white' 
                        : 'text-gray-300 hover:bg-gray-800/30'
                    } transition-colors truncate text-xs`}
                  >
                    <svg xmlns="http://www.w3.org/2000/svg" className="h-3 w-3 mr-2 flex-shrink-0" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M8 10h.01M12 10h.01M16 10h.01M9 16H5a2 2 0 01-2-2V6a2 2 0 012-2h14a2 2 0 012 2v8a2 2 0 01-2 2h-5l-5 5v-5z" />
                    </svg>
                    <span className="truncate text-xs">{chat.title}</span>
                  </Link>
                  <button
                    onClick={(e) => handleDeleteSession(chat.id, chat.title, e)} 
                    className="absolute right-1 top-1/2 -translate-y-1/2 p-1 text-gray-500 hover:text-red-500 opacity-0 group-hover:opacity-100 transition-opacity"
                    title="대화 삭제"
                  >
                    <svg xmlns="http://www.w3.org/2000/svg" className="h-3 w-3" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                      <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M19 7l-.867 12.142A2 2 0 0116.138 21H7.862a2 2 0 01-1.995-1.858L5 7m5 4v6m4-6v6m1-10V4a1 1 0 00-1-1h-4a1 1 0 00-1 1v3M4 7h16" />
                    </svg>
                  </button>
                </div>
              ))
            ) : (
              <div className="text-gray-500 text-xs px-2 py-1">대화 기록이 없습니다</div>
            )}
          </div>
        </div>
      </div>
      
      {/* 메인 컨텐츠 */}
      <div className="flex-1 flex flex-col relative bg-gray-900">
        {/* 스크롤 버튼 */}
        <div className="absolute left-4 top-2.5 z-10 flex justify-center">
          <ChatScrollButtons
            isAtTop={isAtTop}
            isAtBottom={isAtBottom}
            isOverflowing={isOverflowing}
            scrollToTop={scrollToTop}
            scrollToBottom={scrollToBottom}
          />
        </div>

        {/* 헤더 */}
        <div className="bg-gray-900 flex h-14 min-h-[56px] w-full items-center justify-center">
          <div className="absolute left-4 sm:hidden">
            <Link
              href="/chat/list"
              className="p-2 text-gray-500 dark:text-gray-400 hover:text-blue-600 transition-colors"
              title="뒤로 가기"
            >
              <svg xmlns="http://www.w3.org/2000/svg" className="h-5 w-5" fill="none" viewBox="0 0 24 24" stroke="currentColor">
                <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M10 19l-7-7m0 0l7-7m-7 7h18" />
              </svg>
            </Link>
          </div>
        </div>

        {/* 메시지 목록 */}
        <div
          ref={messagesContainerRef}
          className="flex-1 w-full overflow-auto bg-gray-900 pb-4"
          onScroll={handleScroll}
        >
          <div ref={messagesStartRef} />

          <div className="w-full max-w-4xl mx-auto px-4">
            <ChatMessages
              messages={messages}
              isLoading={isFetching && messages.length === 0}
              error={error}
              isStreaming={isStreaming}
            />
          </div>

          <div ref={messagesEndRef} />
        </div>

        {/* 입력 폼 */}
        <div className="w-full bg-gray-900 px-4 py-3">
          <div className="max-w-4xl mx-auto">
            <ChatInput
              onSendMessage={sendMessage}
              isLoading={isLoading}
              useRag={useRag}
              setUseRag={setUseRag}
              useWebSearch={useWebSearch}
              setUseWebSearch={setUseWebSearch}
              isStreaming={isStreaming}
            />
          </div>
        </div>
      </div>
    </div>
  );
};

export default ChatUI; 