'use client';

import { useState, useEffect } from 'react';
import { externalApiClient as apiClient } from '@/lib/apiClient';
import type { NodeData } from '@/types/flowai';
import {
  InputNodeConfig,
  ProcessNodeConfig,
  OutputNodeConfig,
  ConditionNodeConfig,
  LoopNodeConfig
} from './nodes';
import ProcessNodeConfigWithTest from './nodes/ProcessNodeConfigWithTest';

interface ConfigPanelProps {
  node: NodeData;
  onNodeUpdate: (nodeId: string, updates: Partial<NodeData>) => void;
  onNodeDelete?: (nodeId: string) => void;
  onClose: () => void;
}

export default function ConfigPanel({ node, onNodeUpdate, onNodeDelete, onClose }: ConfigPanelProps) {
  const [localData, setLocalData] = useState(node.data);
  const [activeTab, setActiveTab] = useState<'settings' | 'test'>('settings');
  
  // Test functionality states
  const [testInput, setTestInput] = useState('');
  const [testOutput, setTestOutput] = useState('');
  const [isTestRunning, setIsTestRunning] = useState(false);
  const [testError, setTestError] = useState('');

  // 노드 데이터가 변경되면 로컬 상태 업데이트
  useEffect(() => {
    setLocalData(node.data);
  }, [node]);

  const updateData = (key: string, value: any) => {
    const newData = { ...localData, [key]: value };
    setLocalData(newData);
    
    // 실시간 자동 저장 - 즉시 캔버스에 반영
    onNodeUpdate(node.id, {
      data: newData,
    });
  };



  const renderNodeConfig = () => {
    switch (node.type) {
      case 'input':
        return <InputNodeConfig node={node} onNodeUpdate={onNodeUpdate} />;
      case 'process':
        return <ProcessNodeConfigWithTest node={node} onNodeUpdate={onNodeUpdate} />;
      case 'output':
        return <OutputNodeConfig node={node} onNodeUpdate={onNodeUpdate} />;
      case 'condition':
        return <ConditionNodeConfig node={node} onNodeUpdate={onNodeUpdate} />;
      case 'loop':
        return <LoopNodeConfig node={node} onNodeUpdate={onNodeUpdate} />;
      default:
        return <div className="text-gray-500">지원되지 않는 노드 타입입니다.</div>;
    }
  };

  return (
    <div className="h-full flex flex-col bg-white dark:bg-gray-800">
      {/* 상단 탭 헤더 */}
      <div className="flex items-center justify-between border-b border-gray-200 dark:border-gray-700">
        <div className="flex bg-gray-50 dark:bg-gray-900">
          <button
            onClick={() => setActiveTab('settings')}
            className={`flex items-center space-x-2 px-6 py-3 text-sm font-medium transition-all duration-200 ${
              activeTab === 'settings'
                ? 'text-blue-600 dark:text-blue-400 bg-white dark:bg-gray-800 border-b-2 border-blue-600 dark:border-blue-400'
                : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-100 hover:bg-gray-100 dark:hover:bg-gray-800'
            }`}
          >
            <span>⚙️</span>
            <span>설정</span>
          </button>
          <button
            onClick={() => setActiveTab('test')}
            className={`flex items-center space-x-2 px-6 py-3 text-sm font-medium transition-all duration-200 ${
              activeTab === 'test'
                ? 'text-blue-600 dark:text-blue-400 bg-white dark:bg-gray-800 border-b-2 border-blue-600 dark:border-blue-400'
                : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-100 hover:bg-gray-100 dark:hover:bg-gray-800'
            }`}
          >
            <span>🧪</span>
            <span>테스트</span>
          </button>
        </div>
        <button
          onClick={onClose}
          className="p-2 mr-4 hover:bg-gray-50 dark:hover:bg-gray-700 rounded-xl border border-transparent hover:border-gray-200 dark:hover:border-gray-600 transition-all duration-150 hover:scale-[1.02]"
        >
          <span className="text-gray-500 dark:text-gray-400">✕</span>
        </button>
      </div>

      {/* 설정 내용 */}
      <div className="flex-1 overflow-y-auto p-4 space-y-6">
        {/* 기본 정보 */}
        <div className="space-y-4">
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              노드 이름
            </label>
            <input
              type="text"
              value={localData.label || ''}
              onChange={(e) => updateData('label', e.target.value)}
              className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              설명
            </label>
            <textarea
              value={localData.description || ''}
              onChange={(e) => updateData('description', e.target.value)}
              className="w-full px-3 py-2 bg-transparent border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150 h-20 resize-none"
            />
          </div>
        </div>

        {/* 노드별 설정 */}
        <div>
          <h4 className="text-sm font-medium text-gray-700 dark:text-gray-300 mb-4">
            상세 설정
          </h4>
          {renderNodeConfig()}
        </div>
      </div>

      {/* 하단 액션 버튼 */}
      <div className="p-4 border-t border-gray-200 dark:border-gray-700 bg-gray-50 dark:bg-gray-800">
        <div className="flex space-x-2">
          <button
            onClick={onClose}
            className="flex-1 px-4 py-2.5 text-sm font-medium rounded-xl border border-gray-200 dark:border-gray-700 bg-transparent text-gray-600 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-gray-800 hover:scale-[1.02] transition-all duration-150"
          >
            닫기
          </button>

          {onNodeDelete && (
            <button
              onClick={() => {
                if (confirm('이 노드를 삭제하시겠습니까?')) {
                  onNodeDelete(node.id);
                  onClose();
                }
              }}
              className="px-4 py-2.5 text-sm font-medium rounded-xl border border-red-200 dark:border-red-700 bg-transparent text-red-600 dark:text-red-400 hover:bg-red-50 dark:hover:bg-red-900/20 hover:scale-[1.02] transition-all duration-150"
            >
              노드 삭제
            </button>
          )}
        </div>
      </div>
    </div>
  );
}