'use client';

import type { ExecutionResult } from '@/types/flowai';

interface ExecutionPanelProps {
  results: ExecutionResult[];
  isExecuting: boolean;
  onClose: () => void;
}

export default function ExecutionPanel({ results, isExecuting, onClose }: ExecutionPanelProps) {
  const getStatusIcon = (status: ExecutionResult['status']) => {
    switch (status) {
      case 'pending':
        return '⏳';
      case 'running':
        return '▶️';
      case 'success':
        return '✅';
      case 'error':
        return '❌';
      case 'skipped':
        return '⏭️';
      default:
        return '❓';
    }
  };

  const getStatusColor = (status: ExecutionResult['status']) => {
    switch (status) {
      case 'pending':
        return 'text-gray-500 bg-gray-100 dark:bg-gray-700';
      case 'running':
        return 'text-blue-500 bg-blue-100 dark:bg-blue-900';
      case 'success':
        return 'text-green-500 bg-green-100 dark:bg-green-900';
      case 'error':
        return 'text-red-500 bg-red-100 dark:bg-red-900';
      case 'skipped':
        return 'text-yellow-500 bg-yellow-100 dark:bg-yellow-900';
      default:
        return 'text-gray-500 bg-gray-100 dark:bg-gray-700';
    }
  };

  const formatDuration = (startTime?: string, endTime?: string) => {
    if (!startTime) return '-';
    if (!endTime) return isExecuting ? '실행 중...' : '-';
    
    const start = new Date(startTime).getTime();
    const end = new Date(endTime).getTime();
    const duration = (end - start) / 1000;
    
    if (duration < 1) {
      return `${Math.round(duration * 1000)}ms`;
    } else if (duration < 60) {
      return `${duration.toFixed(1)}초`;
    } else {
      return `${Math.floor(duration / 60)}분 ${Math.round(duration % 60)}초`;
    }
  };

  const totalDuration = results.length > 0 && results.every(r => r.endTime) 
    ? results.reduce((total, result) => {
        if (result.startTime && result.endTime) {
          const start = new Date(result.startTime).getTime();
          const end = new Date(result.endTime).getTime();
          return total + (end - start);
        }
        return total;
      }, 0) / 1000
    : null;

  const successCount = results.filter(r => r.status === 'success').length;
  const errorCount = results.filter(r => r.status === 'error').length;

  return (
    <div className="h-full flex flex-col bg-white dark:bg-gray-800">
      {/* 헤더 */}
      <div className="flex items-center justify-between p-4 border-b border-gray-200 dark:border-gray-700">
        <div className="flex items-center space-x-2">
          <span className="text-lg">📊</span>
          <h3 className="text-lg font-medium text-gray-900 dark:text-gray-100">
            실행 결과
          </h3>
          {isExecuting && (
            <div className="flex items-center space-x-1">
              <div className="w-2 h-2 bg-blue-500 rounded-full animate-pulse"></div>
              <span className="text-xs text-blue-500">실행 중</span>
            </div>
          )}
        </div>
        <button
          onClick={onClose}
          className="p-2 hover:bg-gray-50 dark:hover:bg-gray-700 rounded-xl border border-transparent hover:border-gray-200 dark:hover:border-gray-600 transition-all duration-150 hover:scale-[1.02]"
        >
          <span className="text-gray-500 dark:text-gray-400">✕</span>
        </button>
      </div>

      {/* 요약 정보 */}
      {results.length > 0 && (
        <div className="p-4 bg-gray-50 dark:bg-gray-700 border-b border-gray-200 dark:border-gray-600">
          <div className="grid grid-cols-2 gap-4 text-sm">
            <div className="space-y-1">
              <div className="text-gray-500 dark:text-gray-400">총 노드</div>
              <div className="font-medium text-gray-900 dark:text-gray-100">
                {results.length}개
              </div>
            </div>
            <div className="space-y-1">
              <div className="text-gray-500 dark:text-gray-400">실행 시간</div>
              <div className="font-medium text-gray-900 dark:text-gray-100">
                {totalDuration ? `${totalDuration.toFixed(1)}초` : '계산 중...'}
              </div>
            </div>
            <div className="space-y-1">
              <div className="text-gray-500 dark:text-gray-400">성공</div>
              <div className="font-medium text-green-600">
                {successCount}개
              </div>
            </div>
            <div className="space-y-1">
              <div className="text-gray-500 dark:text-gray-400">실패</div>
              <div className="font-medium text-red-600">
                {errorCount}개
              </div>
            </div>
          </div>
        </div>
      )}

      {/* 실행 결과 목록 */}
      <div className="flex-1 overflow-y-auto min-h-0 pb-4">
        {results.length === 0 ? (
          <div className="flex items-center justify-center h-32 text-gray-500 dark:text-gray-400">
            {isExecuting ? '실행 준비 중...' : '실행 결과가 없습니다'}
          </div>
        ) : (
          <div className="divide-y divide-gray-200 dark:divide-gray-700">
            {results.map((result, index) => (
              <div key={`execution-${index}-${result.nodeId}-${result.startTime || Date.now()}`} className="p-4">
                <div className="flex items-start justify-between mb-3">
                  <div className="flex items-center space-x-3">
                    <div className={`flex items-center justify-center w-8 h-8 rounded-full ${getStatusColor(result.status)}`}>
                      <span className="text-sm">{getStatusIcon(result.status)}</span>
                    </div>
                    <div>
                      <div className="font-medium text-gray-900 dark:text-gray-100">
                        노드 {result.nodeId.slice(-8)}
                      </div>
                      <div className="text-xs text-gray-500 dark:text-gray-400">
                        {formatDuration(result.startTime, result.endTime)}
                      </div>
                    </div>
                  </div>
                  <div className={`px-2 py-1 rounded-full text-xs font-medium ${
                    result.status === 'success' 
                      ? 'bg-green-100 text-green-800 dark:bg-green-900 dark:text-green-200'
                      : result.status === 'error'
                      ? 'bg-red-100 text-red-800 dark:bg-red-900 dark:text-red-200'
                      : result.status === 'running'
                      ? 'bg-blue-100 text-blue-800 dark:bg-blue-900 dark:text-blue-200'
                      : 'bg-gray-100 text-gray-800 dark:bg-gray-700 dark:text-gray-300'
                  }`}>
                    {result.status === 'pending' && '대기 중'}
                    {result.status === 'running' && '실행 중'}
                    {result.status === 'success' && '성공'}
                    {result.status === 'error' && '실패'}
                    {result.status === 'skipped' && '건너뜀'}
                  </div>
                </div>

                {/* 출력 결과 */}
                {result.output && (
                  <div className="mb-3">
                    <div className="text-xs text-gray-500 dark:text-gray-400 mb-1">출력:</div>
                    <div className="bg-gray-50 dark:bg-gray-700 rounded-lg p-3 max-h-64 overflow-y-auto">
                      <pre className="text-xs text-gray-700 dark:text-gray-300 whitespace-pre-wrap break-words">
                        {typeof result.output === 'string' 
                          ? result.output
                          : JSON.stringify(result.output, null, 2)
                        }
                      </pre>
                    </div>
                  </div>
                )}

                {/* 에러 메시지 */}
                {result.error && (
                  <div className="mb-3">
                    <div className="text-xs text-red-500 mb-1">오류:</div>
                    <div className="bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg p-3 max-h-32 overflow-y-auto">
                      <pre className="text-xs text-red-700 dark:text-red-300 whitespace-pre-wrap break-words">
                        {result.error}
                      </pre>
                    </div>
                  </div>
                )}

                {/* 시간 정보 */}
                {(result.startTime || result.endTime) && (
                  <div className="text-xs text-gray-500 dark:text-gray-400 space-y-1">
                    {result.startTime && (
                      <div>시작: {new Date(result.startTime).toLocaleTimeString()}</div>
                    )}
                    {result.endTime && (
                      <div>완료: {new Date(result.endTime).toLocaleTimeString()}</div>
                    )}
                  </div>
                )}
              </div>
            ))}
          </div>
        )}
      </div>

      {/* 하단 액션 버튼 */}
      {results.length > 0 && !isExecuting && (
        <div className="flex-shrink-0 p-4 pb-6 border-t border-gray-200 dark:border-gray-700 bg-white dark:bg-gray-800">
          <div className="flex space-x-2">
            <button
              onClick={() => {
                // 결과를 JSON으로 다운로드
                const data = JSON.stringify(results, null, 2);
                const blob = new Blob([data], { type: 'application/json' });
                const url = URL.createObjectURL(blob);
                const a = document.createElement('a');
                a.href = url;
                a.download = `workflow-execution-${Date.now()}.json`;
                document.body.appendChild(a);
                a.click();
                document.body.removeChild(a);
                URL.revokeObjectURL(url);
              }}
              className="flex-1 px-4 py-2.5 text-sm font-medium rounded-xl border border-blue-200 dark:border-blue-700 bg-transparent text-blue-600 dark:text-blue-400 hover:bg-blue-50 dark:hover:bg-blue-900/20 hover:scale-[1.02] transition-all duration-150"
            >
              결과 다운로드
            </button>
            <button
              onClick={() => {
                // 결과 초기화
                window.location.reload();
              }}
              className="px-4 py-2.5 text-sm font-medium rounded-xl border border-gray-200 dark:border-gray-700 bg-transparent text-gray-600 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-gray-800 hover:scale-[1.02] transition-all duration-150"
            >
              초기화
            </button>
          </div>
        </div>
      )}
    </div>
  );
}