'use client';

import { useState } from 'react';
import type { NodeType, WorkflowData } from '@/types/flowai';
import { 
  FileInput, 
  Cpu, 
  FileOutput, 
  GitBranch, 
  RotateCcw,
  Menu
} from 'lucide-react';

interface HeaderProps {
  onNodeAdd: (type: NodeType) => void;
  onToggleSidebar: () => void;
  isExecuting: boolean;
  isSidebarCollapsed: boolean;
  nodesCount: number;
  workflowName?: string;
  onWorkflowNameUpdate?: (newName: string) => void;
}

const nodeTypes: { 
  type: NodeType; 
  label: string; 
  icon: React.ComponentType<{ className?: string }>; 
  description: string;
}[] = [
  { 
    type: 'input', 
    label: '데이터 입력', 
    icon: FileInput, 
    description: '사용자 입력 및 외부 데이터 소스 연결'
  },
  { 
    type: 'process', 
    label: 'AI 처리', 
    icon: Cpu, 
    description: 'AI 모델을 통한 콘텐츠 생성 및 데이터 처리'
  },
  { 
    type: 'output', 
    label: '결과 출력', 
    icon: FileOutput, 
    description: '처리된 결과를 파일이나 화면으로 출력'
  },
  { 
    type: 'condition', 
    label: '조건 분기', 
    icon: GitBranch, 
    description: '조건에 따른 워크플로우 분기 처리'
  },
  { 
    type: 'loop', 
    label: '반복 처리', 
    icon: RotateCcw, 
    description: '지정된 조건에 따른 반복 실행'
  },
];

export default function Header({
  onNodeAdd,
  onToggleSidebar,
  isExecuting,
  isSidebarCollapsed,
  nodesCount,
  workflowName,
  onWorkflowNameUpdate,
}: HeaderProps) {
  const [isEditing, setIsEditing] = useState(false);
  const [editingName, setEditingName] = useState('');


  return (
    <header className="h-24 border-b px-6 relative z-20" style={{ backgroundColor: 'var(--sidebar-bg)', borderColor: 'var(--sidebar-border)' }}>
      <div className="flex items-center h-full">
        {/* 왼쪽 - 사이드바 토글 + 워크플로우 이름 */}
        <div className="flex items-center space-x-4 min-w-0">
          {/* 사이드바 토글 */}
          <button
            onClick={onToggleSidebar}
            className="p-2 text-gray-600 hover:text-gray-800 dark:text-gray-400 dark:hover:text-gray-200 hover:bg-gray-50 dark:hover:bg-gray-800 rounded-lg transition-all duration-150 hover:scale-[1.02]"
            title={isSidebarCollapsed ? '사이드바 열기' : '사이드바 닫기'}
          >
            <Menu className="w-5 h-5" />
          </button>

          {/* 워크플로우 이름 */}
          {workflowName && (
            <div className="flex items-center">
              {isEditing ? (
                <input
                  type="text"
                  value={editingName}
                  onChange={(e) => setEditingName(e.target.value)}
                  onBlur={() => {
                    if (editingName.trim() && editingName.trim() !== workflowName) {
                      onWorkflowNameUpdate?.(editingName.trim());
                    }
                    setIsEditing(false);
                  }}
                  onKeyDown={(e) => {
                    if (e.key === 'Enter') {
                      if (editingName.trim() && editingName.trim() !== workflowName) {
                        onWorkflowNameUpdate?.(editingName.trim());
                      }
                      setIsEditing(false);
                    } else if (e.key === 'Escape') {
                      setEditingName(workflowName);
                      setIsEditing(false);
                    }
                  }}
                  className="text-lg font-semibold bg-white dark:bg-gray-800 rounded px-2 py-1 text-gray-800 dark:text-gray-200 focus:outline-none focus:ring-2 focus:ring-blue-500"
                  autoFocus
                />
              ) : (
                <button
                  onClick={() => {
                    setEditingName(workflowName);
                    setIsEditing(true);
                  }}
                  className="text-lg font-semibold text-gray-800 dark:text-gray-200 hover:text-blue-600 dark:hover:text-blue-400 transition-colors cursor-pointer rounded px-2 py-1"
                  title="워크플로우 이름 수정"
                >
                  {workflowName}
                </button>
              )}
            </div>
          )}
        </div>

        {/* 중앙 - 노드 추가 버튼들 */}
        <div className="flex-1 flex items-center justify-center">
          <div className="flex items-center space-x-4 py-2">
            {nodeTypes.map((nodeType) => {
              const IconComponent = nodeType.icon;
              return (
                <button
                  key={nodeType.type}
                  onClick={() => onNodeAdd(nodeType.type)}
                  className={`group flex flex-col items-center justify-center w-20 h-20 text-xs font-medium rounded-2xl border transition-all duration-200 hover:scale-[1.05] active:scale-[0.95] shadow-sm hover:shadow-lg
                    ${nodeType.type === 'input' ? 'flowai-node-input-button' : 
                      nodeType.type === 'process' ? 'flowai-node-process-button' :
                      nodeType.type === 'output' ? 'flowai-node-output-button' :
                      nodeType.type === 'condition' ? 'flowai-node-condition-button' :
                      'flowai-node-loop-button'
                    }`}
                  title={nodeType.description}
                  draggable
                  onDragStart={(e) => {
                    e.dataTransfer.setData('application/reactflow', nodeType.type);
                    e.dataTransfer.effectAllowed = 'move';
                  }}
                >
                  <div className="mb-1 group-hover:scale-110 transition-transform duration-200">
                    <IconComponent className="w-6 h-6" />
                  </div>
                  <span className="text-xs font-semibold text-center leading-tight">{nodeType.label}</span>
                </button>
              );
            })}
          </div>
        </div>

        {/* 오른쪽 - 상태 표시 */}
        <div className="flex items-center space-x-3">
          <div className="text-sm text-gray-600 dark:text-gray-400">
            노드: {nodesCount}개
          </div>
          {isExecuting && (
            <div className="flex items-center space-x-2 text-sm text-blue-600 dark:text-blue-400">
              <div className="w-3 h-3 border-2 border-blue-600 dark:border-blue-400 border-t-transparent rounded-full animate-spin"></div>
              <span>실행 중</span>
            </div>
          )}
        </div>
      </div>
    </header>
  );
}