'use client';

import { useState } from 'react';
import type { NodeType, NodeTemplate } from '@/types/flowai';

interface NodePanelProps {
  onNodeDragStart: (type: NodeType) => void;
  onNodeDragEnd: () => void;
  onNodeClick?: (type: NodeType) => void;
}

export default function NodePanel({ onNodeDragStart, onNodeDragEnd, onNodeClick }: NodePanelProps) {
  const [activeTab, setActiveTab] = useState<'nodes' | 'templates'>('nodes');
  const [selectedCategory, setSelectedCategory] = useState<string>('입력');

  // 노드 템플릿 정의 (api_server.js 엔드포인트 기반)
  const nodeTemplates: NodeTemplate[] = [
    // Input 노드들
    {
      type: 'input',
      label: '텍스트 입력',
      description: '사용자 텍스트 입력',
      icon: '📝',
      category: '입력',
      defaultConfig: {
        inputType: 'text',
        placeholder: '텍스트를 입력하세요',
      },
    },
    {
      type: 'input',
      label: '파일 업로드',
      description: '문서 파일 업로드',
      icon: '📄',
      category: '입력',
      defaultConfig: {
        inputType: 'file',
        allowedTypes: ['pdf', 'docx', 'hwp', 'txt'],
      },
    },
    
    // Process 노드들 - API 엔드포인트 기반
    {
      type: 'process',
      label: 'AI 채팅',
      description: 'GPT/Claude 등 AI 모델과 대화',
      icon: '💬',
      category: 'AI',
      defaultConfig: {
        processType: 'chat',
        chatConfig: {
          model: 'gpt-4',
          temperature: 0.7,
          maxTokens: 2000,
        }
      },
    },
    {
      type: 'process',
      label: '코드 생성',
      description: 'AI를 활용한 코드 생성',
      icon: '💻',
      category: 'AI',
      defaultConfig: {
        processType: 'code',
        codeConfig: {
          model: 'gpt-4',
          language: 'javascript',
          framework: '',
        }
      },
    },
    {
      type: 'process',
      label: '보고서 생성',
      description: 'PDF/워드 보고서 자동 생성',
      icon: '📊',
      category: '문서',
      defaultConfig: {
        processType: 'report',
        reportConfig: {
          template: 'simple',
          format: 'pdf',
          includeCharts: true,
        }
      },
    },
    
    // RAG 관련 노드들
    {
      type: 'process',
      label: 'RAG 검색',
      description: '지식베이스에서 정보 검색',
      icon: '🔍',
      category: 'RAG',
      defaultConfig: {
        processType: 'rag-search',
        ragConfig: {
          max_results: 5,
          similarity_threshold: 0.7,
        }
      },
    },
    {
      type: 'process',
      label: 'RAG 문서 추가',
      description: '지식베이스에 문서 추가',
      icon: '📚',
      category: 'RAG',
      defaultConfig: {
        processType: 'rag-add',
        ragConfig: {}
      },
    },
    {
      type: 'process',
      label: 'RAG 문서 목록',
      description: '지식베이스 문서 목록 조회',
      icon: '📋',
      category: 'RAG',
      defaultConfig: {
        processType: 'rag-list',
      },
    },
    {
      type: 'process',
      label: 'RAG 초기화',
      description: '지식베이스 초기화/재구축',
      icon: '⚡',
      category: 'RAG',
      defaultConfig: {
        processType: 'rag-init',
      },
    },
    
    // 웹 검색
    {
      type: 'process',
      label: '웹 검색',
      description: '웹에서 실시간 정보 검색',
      icon: '🌐',
      category: '웹검색',
      defaultConfig: {
        processType: 'web-search',
        webSearchConfig: {
          engine: 'auto',
          maxResults: 5,
        }
      },
    },
    
    // 제어 노드들
    {
      type: 'process',
      label: '데이터 변환',
      description: 'JSON, CSV 등 데이터 변환',
      icon: '🔄',
      category: '유틸리티',
      defaultConfig: {
        processType: 'data-transform',
      },
    },
    {
      type: 'condition',
      label: '조건 분기',
      description: '조건에 따른 분기 처리',
      icon: '🔀',
      category: '제어',
      defaultConfig: {
        processType: 'condition',
        condition: '',
        operator: 'equals',
      },
    },
    {
      type: 'loop',
      label: '반복 처리',
      description: '지정된 횟수만큼 반복',
      icon: '🔁',
      category: '제어',
      defaultConfig: {
        processType: 'loop',
        loopCount: 3,
        loopVariable: 'item',
      },
    },
    
    // Output 노드들
    {
      type: 'output',
      label: '텍스트 출력',
      description: '결과를 텍스트로 표시',
      icon: '📋',
      category: '출력',
      defaultConfig: {
        outputFormat: 'text',
      },
    },
    {
      type: 'output',
      label: 'JSON 출력',
      description: '결과를 JSON 형식으로 출력',
      icon: '📄',
      category: '출력',
      defaultConfig: {
        outputFormat: 'json',
      },
    },
  ];



  // 출력 형식 템플릿
  const outputTemplates = [
    {
      type: 'output' as NodeType,
      label: 'PDF 출력',
      description: 'PDF 파일로 결과 저장',
      icon: '📄',
      category: '출력',
      defaultConfig: {
        outputFormat: 'pdf',
        filename: 'output.pdf',
        template: 'default',
      },
    },
    {
      type: 'output' as NodeType,
      label: 'Word 문서',
      description: 'Word 문서로 결과 저장',
      icon: '📝',
      category: '출력',
      defaultConfig: {
        outputFormat: 'docx',
        filename: 'output.docx',
        template: 'default',
      },
    },
    {
      type: 'output' as NodeType,
      label: '프레젠테이션',
      description: 'PowerPoint 파일로 결과 저장',
      icon: '📊',
      category: '출력',
      defaultConfig: {
        outputFormat: 'pptx',
        filename: 'output.pptx',
        template: 'default',
      },
    },
    {
      type: 'output' as NodeType,
      label: '한글 문서',
      description: '한글 파일로 결과 저장',
      icon: '🇰🇷',
      category: '출력',
      defaultConfig: {
        outputFormat: 'hwpx',
        filename: 'output.hwpx',
        template: 'default',
      },
    },
    {
      type: 'output' as NodeType,
      label: '웹 페이지',
      description: 'HTML 파일로 결과 저장',
      icon: '🌐',
      category: '출력',
      defaultConfig: {
        outputFormat: 'html',
        filename: 'output.html',
        template: 'default',
      },
    },
  ];

  // 모든 템플릿 결합
  const allTemplates = nodeTemplates;

  // 워크플로우 템플릿 정의
  const workflowTemplates = [
    {
      id: 'blog-writer',
      name: '블로그 글 작성기',
      description: '주제를 입력하면 SEO 최적화된 블로그 글을 자동 생성',
      icon: '📝',
      category: '콘텐츠',
      nodes: [
        { type: 'input', position: { x: 50, y: 100 }, config: { inputType: 'text', placeholder: '블로그 주제를 입력하세요' } },
        { type: 'process', position: { x: 300, y: 100 }, config: { processType: 'text-generation', model: 'gpt-4', prompt: 'SEO 최적화된 블로그 글을 작성해주세요. 제목, 메타 설명, 본문을 포함하여 작성해주세요.' } },
        { type: 'output', position: { x: 550, y: 100 }, config: { outputFormat: 'markdown', filename: 'blog-post' } }
      ]
    },
    {
      id: 'report-generator',
      name: '보고서 생성기',
      description: '데이터를 분석하여 전문적인 보고서를 자동 생성',
      icon: '📊',
      category: '비즈니스',
      nodes: [
        { type: 'input', position: { x: 50, y: 100 }, config: { inputType: 'file', placeholder: '데이터 파일을 업로드하세요' } },
        { type: 'process', position: { x: 300, y: 100 }, config: { processType: 'chat', prompt: '업로드된 문서를 요약해주세요' } },
        { type: 'process', position: { x: 300, y: 250 }, config: { processType: 'report', reportConfig: { template: 'simple', format: 'pdf' } } },
        { type: 'output', position: { x: 550, y: 175 }, config: { outputFormat: 'pdf', filename: 'report', template: 'professional' } }
      ]
    },
    {
      id: 'social-media-content',
      name: '소셜미디어 콘텐츠 생성기',
      description: '하나의 콘텐츠로 여러 SNS 플랫폼용 게시물을 생성',
      icon: '📱',
      category: '마케팅',
      nodes: [
        { type: 'input', position: { x: 50, y: 100 }, config: { inputType: 'text', placeholder: '콘텐츠 주제를 입력하세요' } },
        { type: 'process', position: { x: 300, y: 50 }, config: { processType: 'chat', prompt: '인스타그램용 캐주얼한 게시물을 작성해주세요' } },
        { type: 'process', position: { x: 300, y: 150 }, config: { processType: 'chat', prompt: '링크드인용 전문적인 게시물을 작성해주세요' } },
        { type: 'output', position: { x: 550, y: 100 }, config: { outputFormat: 'html', filename: 'social-media-posts' } }
      ]
    },
    {
      id: 'code-reviewer',
      name: '코드 리뷰어',
      description: '코드를 분석하여 품질 개선 제안을 생성',
      icon: '💻',
      category: '개발',
      nodes: [
        { type: 'input', position: { x: 50, y: 100 }, config: { inputType: 'file', placeholder: '소스코드 파일을 업로드하세요' } },
        { type: 'process', position: { x: 300, y: 100 }, config: { processType: 'code', prompt: '다음 코드를 분석하여 품질 개선 제안을 해주세요' } },
        { type: 'condition', position: { x: 550, y: 100 }, config: { condition: 'critical_issues > 0', operator: 'greater' } },
        { type: 'output', position: { x: 800, y: 100 }, config: { outputFormat: 'markdown', filename: 'code-review' } }
      ]
    },
    {
      id: 'research-assistant',
      name: '리서치 어시스턴트',
      description: '주제를 조사하여 종합적인 리서치 보고서를 생성',
      icon: '🔍',
      category: '연구',
      nodes: [
        { type: 'input', position: { x: 50, y: 100 }, config: { inputType: 'text', placeholder: '리서치 주제를 입력하세요' } },
        { type: 'process', position: { x: 300, y: 50 }, config: { processType: 'web-search', maxResults: 10 } },
        { type: 'process', position: { x: 300, y: 150 }, config: { processType: 'rag-search', scope: 'personal' } },
        { type: 'process', position: { x: 550, y: 100 }, config: { processType: 'chat', prompt: '수집된 정보를 종합하여 리서치 보고서를 작성해주세요' } },
        { type: 'output', position: { x: 800, y: 100 }, config: { outputFormat: 'docx', filename: 'research-report' } }
      ]
    }
  ];

  // 카테고리별 템플릿 필터링
  const getFilteredTemplates = () => {
    return allTemplates.filter(template => template.category === selectedCategory);
  };

  // 카테고리 목록 추출
  const categories = Array.from(new Set(allTemplates.map(template => template.category)));

  // 드래그 시작 - 간단한 방식으로 변경
  const handleDragStart = (e: React.DragEvent, template: NodeTemplate) => {
    console.log('🔥 NodePanel drag start:', template);
    e.dataTransfer.setData('application/reactflow', template.type);
    e.dataTransfer.effectAllowed = 'move';
    onNodeDragStart(template.type);
  };

  // 드래그 끝
  const handleDragEnd = () => {
    console.log('🔥 NodePanel drag end');
    onNodeDragEnd();
  };

  // 노드 클릭으로 추가
  const handleNodeClick = (template: NodeTemplate) => {
    console.log('🔥 NodePanel click:', template);
    if (onNodeClick) {
      onNodeClick(template.type);
    }
  };

  return (
    <div className="flex flex-col h-full p-4">
      {/* 탭 헤더 */}
      <div className="flex mb-4 bg-gray-100 dark:bg-gray-700 rounded-lg p-1">
        <button
          onClick={() => setActiveTab('nodes')}
          className={`flex-1 px-3 py-2 text-sm font-medium rounded-md transition-colors ${
            activeTab === 'nodes'
              ? 'bg-white dark:bg-gray-600 text-gray-900 dark:text-gray-100 shadow-sm'
              : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
          }`}
        >
          노드
        </button>
        <button
          onClick={() => setActiveTab('templates')}
          className={`flex-1 px-3 py-2 text-sm font-medium rounded-md transition-colors ${
            activeTab === 'templates'
              ? 'bg-white dark:bg-gray-600 text-gray-900 dark:text-gray-100 shadow-sm'
              : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-200'
          }`}
        >
          템플릿
        </button>
      </div>

      {/* 노드 탭 */}
      {activeTab === 'nodes' && (
        <>
          {/* 카테고리 선택 */}
          <div className="mb-4">
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              카테고리
            </label>
            <select
              value={selectedCategory}
              onChange={(e) => setSelectedCategory(e.target.value)}
              className="w-full px-3 py-2 text-sm bg-white dark:bg-gray-700 border border-gray-300 dark:border-gray-600 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-transparent"
            >
              {categories.map(category => (
                <option key={category} value={category}>
                  {category}
                </option>
              ))}
            </select>
          </div>

          {/* 노드 템플릿 목록 */}
          <div className="flex-1 overflow-y-auto">
            <div className="space-y-2">
              {getFilteredTemplates().map((template, index) => (
                <div
                  key={`${template.type}-${index}`}
                  draggable={true}
                  onDragStart={(e) => handleDragStart(e, template)}
                  onDragEnd={handleDragEnd}
                  onClick={() => handleNodeClick(template)}
                  className="group flex items-start p-3 bg-white dark:bg-gray-700 border border-gray-200 dark:border-gray-600 rounded-lg cursor-pointer hover:border-blue-300 dark:hover:border-blue-600 hover:bg-blue-50 dark:hover:bg-blue-900/20 transition-all"
                >
                  <div className="flex-shrink-0 w-10 h-10 flex items-center justify-center bg-gray-100 dark:bg-gray-600 rounded-lg mr-3 group-hover:bg-blue-100 dark:group-hover:bg-blue-800 transition-colors">
                    <span className="text-base">{template.icon}</span>
                  </div>
                  <div className="flex-1 min-w-0">
                    <p className="font-medium text-gray-900 dark:text-gray-100 truncate">
                      {template.label}
                    </p>
                    <p className="text-xs text-gray-500 dark:text-gray-400 leading-relaxed mt-1">
                      {template.description}
                    </p>
                  </div>
                </div>
              ))}
            </div>
          </div>
        </>
      )}

      {/* 템플릿 탭 */}
      {activeTab === 'templates' && (
        <div className="flex-1 overflow-y-auto">
          <div className="space-y-3">
            {workflowTemplates.map((template) => (
              <div
                key={template.id}
                className="p-4 bg-white dark:bg-gray-700 border border-gray-200 dark:border-gray-600 rounded-lg hover:border-blue-300 dark:hover:border-blue-600 hover:bg-blue-50 dark:hover:bg-blue-900/20 transition-all cursor-pointer"
                onClick={() => {
                  // 템플릿 적용 이벤트 발생
                  window.dispatchEvent(new CustomEvent('applyWorkflowTemplate', { 
                    detail: template 
                  }));
                }}
              >
                <div className="flex items-start">
                  <div className="flex-shrink-0 w-10 h-10 flex items-center justify-center bg-gradient-to-br from-blue-100 to-blue-200 dark:from-blue-900 dark:to-blue-800 rounded-lg mr-3">
                    <span className="text-lg">{template.icon}</span>
                  </div>
                  <div className="flex-1 min-w-0">
                    <h3 className="font-semibold text-gray-900 dark:text-gray-100 truncate">
                      {template.name}
                    </h3>
                    <p className="text-sm text-gray-500 dark:text-gray-400 leading-relaxed mt-1">
                      {template.description}
                    </p>
                    <div className="flex items-center mt-2">
                      <span className="inline-flex items-center px-2 py-1 text-xs font-medium bg-gray-100 dark:bg-gray-600 text-gray-700 dark:text-gray-300 rounded">
                        {template.category}
                      </span>
                      <span className="ml-2 text-xs text-gray-400 dark:text-gray-500">
                        {template.nodes.length}개 노드
                      </span>
                    </div>
                  </div>
                </div>
              </div>
            ))}
          </div>
        </div>
      )}

      {/* 도움말 */}
      <div className="mt-4 p-4 bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg">
        <div className="flex items-start">
          <span className="text-blue-500 mr-2 text-lg">💡</span>
          <div className="text-sm text-blue-700 dark:text-blue-300">
            <p className="font-medium mb-2">
              {activeTab === 'nodes' ? '노드 사용법' : '템플릿 사용법'}
            </p>
            {activeTab === 'nodes' ? (
              <ul className="space-y-1 text-blue-600 dark:text-blue-400 text-xs">
                <li>• 노드를 캔버스로 드래그</li>
                <li>• 노드를 연결하여 워크플로우 구성</li>
                <li>• 노드 클릭으로 설정 변경</li>
                <li>• Delete 키로 삭제</li>
              </ul>
            ) : (
              <ul className="space-y-1 text-blue-600 dark:text-blue-400 text-xs">
                <li>• 템플릿 클릭으로 즉시 적용</li>
                <li>• 기본 워크플로우가 자동 구성</li>
                <li>• 필요에 따라 노드 추가/수정</li>
                <li>• 초보자도 쉽게 시작 가능</li>
              </ul>
            )}
          </div>
        </div>
      </div>
    </div>
  );
}