import { useState } from 'react';
import type { NodeConfigProps } from './types';
import type { OutputFormat } from '@/types/flowai';

export default function OutputNodeConfig({ node, onNodeUpdate }: NodeConfigProps) {
  const [localConfig, setLocalConfig] = useState(node.data.config || {});
  const [localData, setLocalData] = useState(node.data);

  const updateConfig = (key: string, value: any) => {
    const newConfig = { ...localConfig, [key]: value };
    setLocalConfig(newConfig);
    onNodeUpdate(node.id, {
      data: {
        ...localData,
        config: newConfig,
      },
    });
  };

  const updateData = (key: string, value: any) => {
    const newData = { ...localData, [key]: value };
    setLocalData(newData);
    onNodeUpdate(node.id, {
      data: newData,
    });
  };

  return (
    <div className="space-y-4">
      <div>
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          출력 형식
        </label>
        <select
          value={localData.outputFormat || 'text'}
          onChange={(e) => updateData('outputFormat', e.target.value as OutputFormat)}
          className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
        >
          <option value="text">텍스트</option>
          <option value="pdf">PDF</option>
          <option value="docx">Word 문서</option>
          <option value="pptx">PowerPoint</option>
          <option value="hwpx">한글 문서</option>
          <option value="markdown">Markdown</option>
          <option value="html">HTML</option>
          <option value="json">JSON</option>
          <option value="csv">CSV</option>
        </select>
      </div>

      <div>
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          파일명
        </label>
        <input
          type="text"
          value={localConfig.filename || 'output'}
          onChange={(e) => updateConfig('filename', e.target.value)}
          placeholder="출력 파일명"
          className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
        />
      </div>

      {(localData.outputFormat === 'pdf' || 
        localData.outputFormat === 'docx' || 
        localData.outputFormat === 'pptx' || 
        localData.outputFormat === 'hwpx') && (
        <div>
          <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
            템플릿
          </label>
          <select
            value={localConfig.template || 'default'}
            onChange={(e) => updateConfig('template', e.target.value)}
            className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
          >
            <option value="default">기본 템플릿</option>
            <option value="professional">전문 문서</option>
            <option value="academic">학술 논문</option>
            <option value="report">보고서</option>
            <option value="presentation">프레젠테이션</option>
          </select>
        </div>
      )}

      <div className="flex items-center">
        <input
          type="checkbox"
          id="autoDownload"
          checked={localConfig.autoDownload || false}
          onChange={(e) => updateConfig('autoDownload', e.target.checked)}
          className="w-4 h-4 text-blue-600 bg-gray-100 dark:bg-gray-700 border-gray-300 dark:border-gray-600 rounded focus:ring-blue-500 focus:ring-2 mr-3"
        />
        <label htmlFor="autoDownload" className="text-sm font-medium text-gray-700 dark:text-gray-300">
          자동 다운로드
        </label>
      </div>

      <div className="flex items-center">
        <input
          type="checkbox"
          id="showInBrowser"
          checked={localConfig.showInBrowser || true}
          onChange={(e) => updateConfig('showInBrowser', e.target.checked)}
          className="w-4 h-4 text-blue-600 bg-gray-100 dark:bg-gray-700 border-gray-300 dark:border-gray-600 rounded focus:ring-blue-500 focus:ring-2 mr-3"
        />
        <label htmlFor="showInBrowser" className="text-sm font-medium text-gray-700 dark:text-gray-300">
          브라우저에서 미리보기
        </label>
      </div>
    </div>
  );
}