import { useState, useEffect } from 'react';
import { externalApiClient as apiClient } from '@/lib/apiClient';
import type { NodeConfigProps, Provider, Model } from './types';
import type { ProcessType } from '@/types/flowai';

export default function ProcessNodeConfigWithTest({ node, onNodeUpdate }: NodeConfigProps) {
  const [localData, setLocalData] = useState(node.data);
  const [providers, setProviders] = useState<Provider[]>([]);
  const [availableModels, setAvailableModels] = useState<Model[]>([]);
  const [isLoadingProviders, setIsLoadingProviders] = useState(false);
  const [isLoadingModels, setIsLoadingModels] = useState(false);
  
  // Test functionality states
  const [testInput, setTestInput] = useState('');
  const [testOutput, setTestOutput] = useState('');
  const [isTestRunning, setIsTestRunning] = useState(false);
  const [testError, setTestError] = useState('');
  const [activeTab, setActiveTab] = useState<'settings' | 'test'>('settings');

  const updateData = (key: string, value: any) => {
    const newData = { ...localData, [key]: value };
    setLocalData(newData);
    onNodeUpdate(node.id, {
      data: newData,
    });
  };

  // 프로바이더 목록 로딩
  useEffect(() => {
    loadProviders();
  }, []);

  // 프로바이더 변경 시 모델 목록 로딩
  useEffect(() => {
    if (localData.provider) {
      loadModelsForProvider(localData.provider);
    }
  }, [localData.provider]);

  const loadProviders = async () => {
    try {
      setIsLoadingProviders(true);
      const response = await apiClient.getProviders();
      
      if (response.success) {
        const data = response.data;
        const providersFromData = data ? Object.keys(data).map(key => ({
          key: key,
          name: data[key].name || key.charAt(0).toUpperCase() + key.slice(1),
          requiresApiKey: data[key].requiresApiKey || false,
          apiKeyConfigured: data[key].apiKeyConfigured !== false,
          isDynamic: data[key].isDynamic || false,
          isAvailable: true,
          models: []
        })) : [];
        
        setProviders(providersFromData);
        
        if (localData.provider) {
          loadModelsForProvider(localData.provider);
        }
      }
    } catch (error) {
      console.error('프로바이더 로딩 실패:', error);
    } finally {
      setIsLoadingProviders(false);
    }
  };

  const loadModelsForProvider = async (providerKey: string) => {
    try {
      setIsLoadingModels(true);
      const response = await apiClient.getProviderModels(providerKey);
      
      if (response.success) {
        const models = response.data || [];
        setAvailableModels(models);
      } else {
        setAvailableModels([]);
      }
    } catch (error) {
      console.error(`${providerKey} 모델 로딩 실패:`, error);
      setAvailableModels([]);
    } finally {
      setIsLoadingModels(false);
    }
  };

  // Test node functionality
  const testNode = async () => {
    setIsTestRunning(true);
    setTestError('');
    setTestOutput('');

    try {
      // Prepare test data based on processType
      const testData = {
        nodeId: node.id,
        type: node.type,
        data: {
          ...localData,
          testInput: testInput
        }
      };

      // Call test endpoint
      const response = await apiClient.testNode(testData);
      
      if (response.success) {
        setTestOutput(JSON.stringify(response.data, null, 2));
      } else {
        setTestError(response.error?.message || '테스트 실행 실패');
      }
    } catch (error: any) {
      setTestError(error.message || '테스트 중 오류 발생');
    } finally {
      setIsTestRunning(false);
    }
  };

  const renderProcessTypeSpecificConfig = () => {
    switch (localData.processType) {
      case 'chat':
        return (
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                프로바이더
              </label>
              <select
                key={`provider-${localData.provider || 'none'}`}
                value={localData.provider || ''}
                onChange={(e) => {
                  const newProvider = e.target.value;
                  const newData = { ...localData, provider: newProvider, model: '' };
                  setLocalData(newData);
                  onNodeUpdate(node.id, { data: newData });
                  setAvailableModels([]);
                  if (newProvider) {
                    loadModelsForProvider(newProvider);
                  }
                }}
                className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
                disabled={isLoadingProviders}
              >
                <option value="">프로바이더를 선택하세요</option>
                {isLoadingProviders ? (
                  <option>로딩 중...</option>
                ) : (
                  providers.map((provider) => (
                    <option key={provider.key} value={provider.key}>
                      {provider.name}
                    </option>
                  ))
                )}
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                모델
              </label>
              <select
                value={localData.model || ''}
                onChange={(e) => updateData('model', e.target.value)}
                className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
                disabled={isLoadingModels || !localData.provider}
              >
                <option value="">모델을 선택하세요</option>
                {isLoadingModels ? (
                  <option>로딩 중...</option>
                ) : (
                  availableModels.map((model) => (
                    <option key={model.id} value={model.id}>
                      {model.name || model.id}
                    </option>
                  ))
                )}
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                Temperature ({localData.chatConfig?.temperature || 0.7})
              </label>
              <input
                type="range"
                min="0"
                max="2"
                step="0.1"
                value={localData.chatConfig?.temperature || 0.7}
                onChange={(e) => updateData('chatConfig', { ...localData.chatConfig, temperature: parseFloat(e.target.value) })}
                className="w-full h-2 bg-gray-200 rounded-lg appearance-none cursor-pointer dark:bg-gray-700"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                최대 토큰 수
              </label>
              <input
                type="number"
                min="100"
                max="8000"
                value={localData.chatConfig?.maxTokens || 2000}
                onChange={(e) => updateData('chatConfig', { ...localData.chatConfig, maxTokens: parseInt(e.target.value) })}
                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                AI 지시사항 (System Prompt)
              </label>
              <textarea
                value={localData.chatConfig?.systemPrompt || ''}
                onChange={(e) => updateData('chatConfig', { ...localData.chatConfig, systemPrompt: e.target.value })}
                rows={3}
                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                placeholder="AI에게 역할과 행동 방식을 지시하세요 (예: 당신은 전문 번역가입니다. 정확하고 자연스러운 번역을 제공하세요)"
              />
            </div>
          </div>
        );
      
      case 'code':
        return (
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                프로바이더
              </label>
              <select
                key={`provider-${localData.provider || 'none'}`}
                value={localData.provider || ''}
                onChange={(e) => {
                  const newProvider = e.target.value;
                  const newData = { ...localData, provider: newProvider, model: '' };
                  setLocalData(newData);
                  onNodeUpdate(node.id, { data: newData });
                  setAvailableModels([]);
                  if (newProvider) {
                    loadModelsForProvider(newProvider);
                  }
                }}
                className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
                disabled={isLoadingProviders}
              >
                <option value="">프로바이더를 선택하세요</option>
                {isLoadingProviders ? (
                  <option>로딩 중...</option>
                ) : (
                  providers.map((provider) => (
                    <option key={provider.key} value={provider.key}>
                      {provider.name}
                    </option>
                  ))
                )}
              </select>
            </div>

            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                모델
              </label>
              <select
                value={localData.model || ''}
                onChange={(e) => updateData('model', e.target.value)}
                className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
                disabled={isLoadingModels || !localData.provider}
              >
                <option value="">모델을 선택하세요</option>
                {isLoadingModels ? (
                  <option>로딩 중...</option>
                ) : (
                  availableModels.map((model) => (
                    <option key={model.id} value={model.id}>
                      {model.name || model.id}
                    </option>
                  ))
                )}
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                프로그래밍 언어
              </label>
              <select
                value={localData.codeConfig?.language || 'javascript'}
                onChange={(e) => updateData('codeConfig', { ...localData.codeConfig, language: e.target.value })}
                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              >
                <option value="javascript">JavaScript</option>
                <option value="python">Python</option>
                <option value="typescript">TypeScript</option>
                <option value="java">Java</option>
                <option value="cpp">C++</option>
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                프레임워크/라이브러리
              </label>
              <input
                type="text"
                value={localData.codeConfig?.framework || ''}
                onChange={(e) => updateData('codeConfig', { ...localData.codeConfig, framework: e.target.value })}
                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                placeholder="예: React, Django, Spring"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                AI 지시사항 (System Prompt)
              </label>
              <textarea
                value={localData.codeConfig?.systemPrompt || ''}
                onChange={(e) => updateData('codeConfig', { ...localData.codeConfig, systemPrompt: e.target.value })}
                rows={3}
                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                placeholder="코드 생성 스타일을 지시하세요 (예: 깔끔하고 주석이 포함된 코드를 작성하세요. 모든 함수에 타입 힌트를 추가하세요)"
              />
            </div>
          </div>
        );

      case 'report':
        return (
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                보고서 템플릿
              </label>
              <select
                value={localData.reportConfig?.template || 'simple'}
                onChange={(e) => updateData('reportConfig', { ...localData.reportConfig, template: e.target.value })}
                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              >
                <option value="simple">간단한 보고서</option>
                <option value="bizplan">사업계획서</option>
                <option value="proposal">제안서</option>
                <option value="thesis">논문</option>
                <option value="marketing">마케팅 보고서</option>
                <option value="startup">스타트업 보고서</option>
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                출력 형식
              </label>
              <select
                value={localData.reportConfig?.format || 'pdf'}
                onChange={(e) => updateData('reportConfig', { ...localData.reportConfig, format: e.target.value })}
                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              >
                <option value="pdf">PDF</option>
                <option value="docx">Word 문서</option>
                <option value="pptx">PowerPoint</option>
                <option value="hwpx">한글 문서</option>
              </select>
            </div>
            <div className="flex items-center">
              <input
                type="checkbox"
                id="includeCharts"
                checked={localData.reportConfig?.includeCharts !== false}
                onChange={(e) => updateData('reportConfig', { ...localData.reportConfig, includeCharts: e.target.checked })}
                className="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300 rounded"
              />
              <label htmlFor="includeCharts" className="ml-2 block text-sm text-gray-700 dark:text-gray-300">
                차트 포함
              </label>
            </div>
          </div>
        );

      case 'rag-search':
      case 'rag-add':
      case 'rag-list':
      case 'rag-init':
        return (
          <div className="space-y-4">
            {localData.processType === 'rag-search' && (
              <div>
                <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                  검색 쿼리
                </label>
                <input
                  type="text"
                  value={localData.ragConfig?.query || ''}
                  onChange={(e) => updateData('ragConfig', { ...localData.ragConfig, query: e.target.value })}
                  className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
                  placeholder="검색할 내용을 입력하세요"
                />
              </div>
            )}
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                최대 결과 수
              </label>
              <input
                type="number"
                min="1"
                max="20"
                value={localData.ragConfig?.max_results || 5}
                onChange={(e) => updateData('ragConfig', { ...localData.ragConfig, max_results: parseInt(e.target.value) })}
                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              />
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                유사도 임계값
              </label>
              <input
                type="number"
                min="0"
                max="1"
                step="0.1"
                value={localData.ragConfig?.similarity_threshold || 0.7}
                onChange={(e) => updateData('ragConfig', { ...localData.ragConfig, similarity_threshold: parseFloat(e.target.value) })}
                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              />
            </div>
          </div>
        );

      case 'web-search':
        return (
          <div className="space-y-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                검색 엔진
              </label>
              <select
                value={localData.webSearchConfig?.engine || 'auto'}
                onChange={(e) => updateData('webSearchConfig', { ...localData.webSearchConfig, engine: e.target.value })}
                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              >
                <option value="auto">자동</option>
                <option value="google">구글</option>
                <option value="naver">네이버</option>
                <option value="daum">다음</option>
              </select>
            </div>
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                최대 결과 수
              </label>
              <input
                type="number"
                min="1"
                max="50"
                value={localData.webSearchConfig?.maxResults || 5}
                onChange={(e) => updateData('webSearchConfig', { ...localData.webSearchConfig, maxResults: parseInt(e.target.value) })}
                className="w-full px-3 py-2 text-sm border border-gray-300 dark:border-gray-600 rounded-md focus:ring-2 focus:ring-blue-500 focus:border-transparent bg-white dark:bg-gray-700 text-gray-900 dark:text-gray-100"
              />
            </div>
          </div>
        );

      default:
        return null;
    }
  };

  const renderSettings = () => (
    <div className="space-y-4">
      {/* 기본 설정 */}
      <div>
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          처리 타입
        </label>
        <select
          value={localData.processType || ''}
          onChange={(e) => updateData('processType', e.target.value as ProcessType)}
          className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
        >
          <option value="">처리 타입을 선택하세요</option>
          <option value="chat">채팅/대화 생성</option>
          <option value="code">코드 생성</option>
          <option value="report">보고서 생성</option>
          <option value="rag-search">RAG 문서 검색</option>
          <option value="rag-add">RAG 문서 추가</option>
          <option value="rag-list">RAG 문서 목록</option>
          <option value="rag-init">RAG 초기화</option>
          <option value="web-search">웹 검색</option>
          <option value="data-transform">데이터 변환</option>
          <option value="condition">조건문</option>
          <option value="loop">반복문</option>
        </select>
      </div>



      {/* 처리 타입별 상세 설정 */}
      {localData.processType && (
        <div className="border-t border-gray-200 dark:border-gray-700 pt-4">
          <h4 className="font-medium text-gray-700 dark:text-gray-300 mb-4">
            {localData.processType} 설정
          </h4>
          {renderProcessTypeSpecificConfig()}
        </div>
      )}
    </div>
  );

  const renderTestPanel = () => (
    <div className="h-full flex flex-col">
      {/* 테스트 헤더 */}
      <div className="flex items-center justify-between p-4 bg-gray-50 dark:bg-gray-900 border-b border-gray-200 dark:border-gray-700">
        <span className="text-sm font-medium text-gray-700 dark:text-gray-300">
          노드 테스트
        </span>
        <button
          onClick={testNode}
          disabled={isTestRunning || !localData.processType}
          className={`px-4 py-1.5 text-sm font-medium rounded-lg transition-all duration-150 ${
            isTestRunning || !localData.processType
              ? 'bg-gray-300 dark:bg-gray-700 text-gray-500 cursor-not-allowed'
              : 'bg-blue-500 hover:bg-blue-600 text-white hover:scale-[1.02]'
          }`}
        >
          {isTestRunning ? (
            <span className="flex items-center">
              <span className="animate-spin mr-2">⚡</span>
              실행 중...
            </span>
          ) : (
            '▶ Test 실행'
          )}
        </button>
      </div>

      {/* Input/Output panels */}
      <div className="flex-1 grid grid-cols-2 divide-x divide-gray-200 dark:divide-gray-700">
        {/* Input panel */}
        <div className="flex flex-col">
          <div className="px-4 py-3 bg-gray-100 dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
            <span className="text-xs font-medium text-gray-600 dark:text-gray-400 uppercase tracking-wider">
              INPUT
            </span>
          </div>
          <div className="flex-1 p-4">
            <textarea
              value={testInput}
              onChange={(e) => setTestInput(e.target.value)}
              placeholder="테스트 입력 데이터를 입력하세요..."
              className="w-full h-full px-3 py-2 bg-transparent border border-gray-200 dark:border-gray-700 rounded-lg text-sm text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 resize-none font-mono"
            />
          </div>
        </div>

        {/* Output panel */}
        <div className="flex flex-col">
          <div className="px-4 py-3 bg-gray-100 dark:bg-gray-800 border-b border-gray-200 dark:border-gray-700">
            <span className="text-xs font-medium text-gray-600 dark:text-gray-400 uppercase tracking-wider">
              OUTPUT
            </span>
          </div>
          <div className="flex-1 p-4 overflow-auto">
            {testError ? (
              <div className="p-3 bg-red-50 dark:bg-red-900/20 border border-red-200 dark:border-red-800 rounded-lg">
                <span className="text-sm text-red-600 dark:text-red-400">{testError}</span>
              </div>
            ) : testOutput ? (
              <pre className="text-sm text-gray-900 dark:text-gray-100 font-mono whitespace-pre-wrap">
                {testOutput}
              </pre>
            ) : (
              <span className="text-sm text-gray-500 dark:text-gray-400">
                테스트를 실행하면 결과가 여기에 표시됩니다
              </span>
            )}
          </div>
        </div>
      </div>
    </div>
  );

  return (
    <div className="h-full flex flex-col bg-white dark:bg-gray-800">
      {/* 상단 탭 네비게이션 */}
      <div className="flex border-b border-gray-200 dark:border-gray-700 bg-gray-50 dark:bg-gray-900">
        <button
          onClick={() => setActiveTab('settings')}
          className={`flex-1 px-6 py-3 text-sm font-medium transition-all duration-200 ${
            activeTab === 'settings'
              ? 'text-blue-600 dark:text-blue-400 bg-white dark:bg-gray-800 border-b-2 border-blue-600 dark:border-blue-400'
              : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-100 hover:bg-gray-100 dark:hover:bg-gray-800'
          }`}
        >
          노드 설정
        </button>
        <button
          onClick={() => setActiveTab('test')}
          className={`flex-1 px-6 py-3 text-sm font-medium transition-all duration-200 ${
            activeTab === 'test'
              ? 'text-blue-600 dark:text-blue-400 bg-white dark:bg-gray-800 border-b-2 border-blue-600 dark:border-blue-400'
              : 'text-gray-600 dark:text-gray-400 hover:text-gray-900 dark:hover:text-gray-100 hover:bg-gray-100 dark:hover:bg-gray-800'
          }`}
        >
          테스트
        </button>
      </div>

      {/* 콘텐츠 영역 */}
      <div className="flex-1 overflow-auto">
        {activeTab === 'settings' ? (
          <div className="p-6">
            {renderSettings()}
          </div>
        ) : (
          renderTestPanel()
        )}
      </div>
    </div>
  );
}
