/**
 * A2A 프로토콜 사용 예시 및 테스트 클라이언트
 */

import type { A2AMessage, A2AAgent } from '../A2AAdapter';

export class A2ATestClient {
  private baseUrl: string;
  private agentId: string;

  constructor(baseUrl: string = 'http://localhost:3000', agentId: string = 'test-client') {
    this.baseUrl = baseUrl;
    this.agentId = agentId;
  }

  /**
   * A2A 네트워크 상태 확인
   */
  async ping(): Promise<any> {
    const response = await fetch(`${this.baseUrl}/api/a2a?action=ping`);
    return await response.json();
  }

  /**
   * A2A 에이전트 능력 목록 조회
   */
  async getCapabilities(): Promise<any> {
    const response = await fetch(`${this.baseUrl}/api/a2a?action=capabilities`);
    return await response.json();
  }

  /**
   * A2A 메시지 전송
   */
  async sendMessage(message: Partial<A2AMessage>): Promise<any> {
    const fullMessage: A2AMessage = {
      id: this.generateMessageId(),
      from: this.agentId,
      to: message.to || 'airun-mcp-agent',
      type: message.type || 'request',
      method: message.method,
      params: message.params,
      timestamp: new Date(),
      ...message
    };

    const response = await fetch(`${this.baseUrl}/api/a2a/message`, {
      method: 'POST',
      headers: {
        'Content-Type': 'application/json',
        'A2A-Protocol': '1.0'
      },
      body: JSON.stringify(fullMessage)
    });

    return await response.json();
  }

  /**
   * 외부 A2A 에이전트 등록
   */
  async registerAgent(agent: A2AAgent): Promise<any> {
    const response = await fetch(`${this.baseUrl}/api/a2a/agents`, {
      method: 'PUT',
      headers: {
        'Content-Type': 'application/json'
      },
      body: JSON.stringify(agent)
    });

    return await response.json();
  }

  private generateMessageId(): string {
    return `msg_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
  }
}

/**
 * A2A 프로토콜 사용 예시들
 */
export class A2AUsageExamples {
  private client: A2ATestClient;

  constructor(client: A2ATestClient) {
    this.client = client;
  }

  /**
   * 예시 1: AI MCP 도구 호출
   */
  async example1_CallMCPTool(): Promise<void> {
    console.log('=== 예시 1: MCP 도구 호출 ===');
    
    try {
      const response = await this.client.sendMessage({
        method: 'mcp.web_search',
        params: {
          query: 'A2A 프로토콜 최신 동향'
        }
      });
      
      console.log('웹 검색 결과:', response);
    } catch (error) {
      console.error('MCP 도구 호출 실패:', error);
    }
  }

  /**
   * 예시 2: Ollama 에이전트와 협업
   */
  async example2_CollaborateWithOllama(): Promise<void> {
    console.log('=== 예시 2: Ollama 에이전트 협업 ===');
    
    try {
      // 1. Ollama 에이전트에게 텍스트 생성 요청
      const response = await this.client.sendMessage({
        to: 'ollama-hamonize:latest',
        method: 'generate',
        params: {
          prompt: 'A2A 프로토콜의 장점을 설명해주세요',
          max_tokens: 500
        }
      });
      
      console.log('Ollama 생성 결과:', response);
    } catch (error) {
      console.error('Ollama 협업 실패:', error);
    }
  }

  /**
   * 예시 3: 외부 에이전트 등록 및 활용
   */
  async example3_RegisterExternalAgent(): Promise<void> {
    console.log('=== 예시 3: 외부 에이전트 등록 ===');
    
    try {
      // 1. 외부 에이전트 등록
      const externalAgent: A2AAgent = {
        id: 'external-translator',
        name: 'Translation Agent',
        capabilities: ['translation', 'language-detection'],
        endpoint: 'https://api.example.com/translate',
        protocol: 'http',
        trustLevel: 'trusted'
      };

      const registerResult = await this.client.registerAgent(externalAgent);
      console.log('에이전트 등록 결과:', registerResult);

      // 2. 등록된 에이전트에게 번역 요청
      const translateResponse = await this.client.sendMessage({
        to: 'external-translator',
        method: 'translate',
        params: {
          text: 'Hello, how are you?',
          target_language: 'ko'
        }
      });
      
      console.log('번역 결과:', translateResponse);
    } catch (error) {
      console.error('외부 에이전트 활용 실패:', error);
    }
  }

  /**
   * 예시 4: 분산 AI 워크플로우
   */
  async example4_DistributedWorkflow(): Promise<void> {
    console.log('=== 예시 4: 분산 AI 워크플로우 ===');
    
    try {
      // 1. 데이터 수집 (MCP 웹 검색)
      const searchResult = await this.client.sendMessage({
        method: 'mcp.web_search',
        params: { query: 'AI 시장 동향 2024' }
      });

      // 2. 데이터 분석 (Ollama)
      const analysisResult = await this.client.sendMessage({
        to: 'ollama-hamonize:latest',
        method: 'analyze',
        params: {
          data: searchResult.result,
          task: '시장 동향 분석 및 요약'
        }
      });

      // 3. 결과 문서화 (MCP 문서 분석 도구)
      const documentResult = await this.client.sendMessage({
        method: 'mcp.document_analysis',
        params: {
          content: analysisResult.result,
          format: 'report'
        }
      });

      console.log('분산 워크플로우 최종 결과:', documentResult);
    } catch (error) {
      console.error('분산 워크플로우 실패:', error);
    }
  }

  /**
   * 모든 예시 실행
   */
  async runAllExamples(): Promise<void> {
    console.log('🚀 A2A 프로토콜 예시 실행 시작\n');
    
    // 네트워크 상태 확인
    const status = await this.client.ping();
    console.log('A2A 네트워크 상태:', status);
    
    // 능력 목록 조회
    const capabilities = await this.client.getCapabilities();
    console.log('사용 가능한 능력:', capabilities);
    
    console.log('\n');
    
    await this.example1_CallMCPTool();
    console.log('\n');
    
    await this.example2_CollaborateWithOllama();
    console.log('\n');
    
    await this.example3_RegisterExternalAgent();
    console.log('\n');
    
    await this.example4_DistributedWorkflow();
    
    console.log('\n🎉 모든 예시 실행 완료');
  }
}

/**
 * 사용법 예시 실행 함수
 */
export async function runA2AExamples(): Promise<void> {
  const client = new A2ATestClient();
  const examples = new A2AUsageExamples(client);
  
  await examples.runAllExamples();
}

/**
 * Ollama를 A2A 에이전트로 사용하는 구체적인 예시
 */
export class OllamaA2AIntegration {
  private client: A2ATestClient;

  constructor(client: A2ATestClient) {
    this.client = client;
  }

  /**
   * Ollama 모델들을 A2A 네트워크에 등록
   */
  async registerOllamaModels(): Promise<void> {
    const ollamaModels = [
      'hamonize:latest',
      'airun-think:latest', 
      'airun-vision:latest'
    ];

    for (const model of ollamaModels) {
      const agent: A2AAgent = {
        id: `ollama-${model}`,
        name: `Ollama ${model}`,
        capabilities: this.getModelCapabilities(model),
        endpoint: process.env.NEXT_PUBLIC_OLLAMA_PROXY_SERVER || 'http://localhost:11434',
        protocol: 'http',
        trustLevel: 'private'
      };

      try {
        await this.client.registerAgent(agent);
        console.log(`✅ Ollama 모델 등록 완료: ${model}`);
      } catch (error) {
        console.error(`❌ Ollama 모델 등록 실패: ${model}`, error);
      }
    }
  }

  /**
   * 모델별 능력 정의
   */
  private getModelCapabilities(model: string): string[] {
    const baseCapabilities = ['text-generation', 'conversation', 'reasoning'];
    
    if (model.includes('vision')) {
      return [...baseCapabilities, 'image-analysis', 'visual-reasoning'];
    } else if (model.includes('think')) {
      return [...baseCapabilities, 'complex-reasoning', 'problem-solving'];
    } else {
      return [...baseCapabilities, 'general-chat', 'code-generation'];
    }
  }

  /**
   * Ollama 모델들 간 협업 예시
   */
  async demonstrateOllamaCollaboration(): Promise<void> {
    console.log('=== Ollama 모델들 간 협업 데모 ===');

    try {
      // 1. airun-think로 문제 분석
      const analysisResult = await this.client.sendMessage({
        to: 'ollama-airun-think:latest',
        method: 'analyze',
        params: {
          problem: '회사의 AI 도입 전략을 수립해야 합니다. 어떤 요소들을 고려해야 할까요?'
        }
      });

      console.log('1단계 - 문제 분석:', analysisResult);

      // 2. airun-chat으로 구체적인 계획 수립
      const planResult = await this.client.sendMessage({
        to: 'ollama-hamonize:latest',
        method: 'generate',
        params: {
          prompt: `다음 분석 결과를 바탕으로 구체적인 AI 도입 계획을 수립해주세요: ${analysisResult.result}`,
          max_tokens: 1000
        }
      });

      console.log('2단계 - 계획 수립:', planResult);

      // 3. 결과를 MCP 도구로 문서화
      const docResult = await this.client.sendMessage({
        method: 'mcp.document_analysis',
        params: {
          content: planResult.result,
          format: 'structured_report'
        }
      });

      console.log('3단계 - 문서화:', docResult);

    } catch (error) {
      console.error('Ollama 협업 데모 실패:', error);
    }
  }
}

// 브라우저 환경에서 사용할 수 있는 글로벌 함수
if (typeof window !== 'undefined') {
  (window as any).runA2AExamples = runA2AExamples;
} 