import type { User } from "../types/user";

// API 서버 기본 URL - Next.js 내부 API 라우트 우선 사용
const API_BASE_URL = '/api';

// API 요청 타입 정의
type ApiMethod = 'GET' | 'POST' | 'PUT' | 'DELETE';

interface ApiRequestOptions {
  method?: ApiMethod;
  headers?: Record<string, string>;
  body?: any;
  params?: Record<string, string>;
}

// API 응답 타입 정의
interface ApiResponse<T = any> {
  success: boolean;
  data?: T;
  error?: {
    code?: string;
    message: string;
    details?: any;
  };
  timestamp?: string;
}

// API 클라이언트 클래스
export class ApiClient {
  private apiKey: string = '';
  private baseUrl: string;

  constructor(baseUrl: string = API_BASE_URL) {
    this.baseUrl = baseUrl;
    // console.log(`[DEBUG-API-CLIENT] ApiClient 생성됨 - baseUrl: ${this.baseUrl}`);
  }

  // API 키 설정
  setApiKey(apiKey: string): void {
    this.apiKey = apiKey;
  }

  // 브라우저 환경인지 확인
  private isBrowser(): boolean {
    return typeof window !== 'undefined';
  }

  // 로컬 스토리지에서 API 키 가져오기
  private getApiKeyFromStorage(): string | null {
    if (!this.isBrowser()) return null;
    return localStorage.getItem('apiKey');
  }

  // 기본 헤더 생성
  getHeaders(): Record<string, string> {
    const headers: Record<string, string> = {
      'Content-Type': 'application/json',
    };

    // 1. 직접 설정된 API 키 사용
    if (this.apiKey) {
      headers['X-API-Key'] = this.apiKey;
    } 
    // 2. 로컬 스토리지에서 API 키 가져오기
    else {
      const storageApiKey = this.getApiKeyFromStorage();
      if (storageApiKey) {
        headers['X-API-Key'] = storageApiKey;
      }
    }

    // 토큰도 함께 전송 (인증 토큰이 있는 경우)
    if (this.isBrowser()) {
      const token = localStorage.getItem('userToken');
      if (token) {
        headers['Authorization'] = `Bearer ${token}`;
      }
    }

    return headers;
  }

  // API 요청 함수
  async request<T = any>(
    endpoint: string,
    options: ApiRequestOptions = {}
  ): Promise<ApiResponse<T>> {
    const {
      method = 'GET',
      headers = {},
      body,
      params = {},
    } = options;

    // URL 매개변수 처리
    const queryParams = new URLSearchParams();
    Object.entries(params).forEach(([key, value]) => {
      queryParams.append(key, value);
    });

    const queryString = queryParams.toString();
    const url = `${this.baseUrl}${endpoint}${queryString ? `?${queryString}` : ''}`;
    
    // console.log(`[DEBUG-API-REQUEST] ${method} ${url} (baseUrl: ${this.baseUrl}, endpoint: ${endpoint})`);

    try {
      const response = await fetch(url, {
        method,
        headers: {
          ...this.getHeaders(),
          ...headers,
        },
        body: body ? JSON.stringify(body) : undefined,
      });

      const data = await response.json();

      if (!response.ok) {
        return {
          success: false,
          error: {
            code: data.error?.code || `HTTP_ERROR_${response.status}`,
            message: data.error?.message || response.statusText,
            details: data.error?.details,
          },
          timestamp: new Date().toISOString(),
        };
      }

      return data as ApiResponse<T>;
    } catch (error) {
      return {
        success: false,
        error: {
          code: 'NETWORK_ERROR',
          message: error instanceof Error ? error.message : '네트워크 오류가 발생했습니다.',
        },
        timestamp: new Date().toISOString(),
      };
    }
  }

  // 편의 메서드: GET
  async get<T = any>(endpoint: string, params?: Record<string, string>): Promise<ApiResponse<T>> {
    return this.request<T>(endpoint, { method: 'GET', params });
  }

  // 편의 메서드: POST
  async post<T = any>(endpoint: string, body?: any): Promise<ApiResponse<T>> {
    return this.request<T>(endpoint, { method: 'POST', body });
  }

  // 편의 메서드: PUT
  async put<T = any>(endpoint: string, body?: any): Promise<ApiResponse<T>> {
    return this.request<T>(endpoint, { method: 'PUT', body });
  }

  // 편의 메서드: DELETE
  async delete<T = any>(endpoint: string): Promise<ApiResponse<T>> {
    return this.request<T>(endpoint, { method: 'DELETE' });
  }

  // 세션 관리
  async getSessions(type?: string, page?: number, limit?: number): Promise<ApiResponse<any>> {
    // 외부 API 서버를 사용하도록 변경
    if (this.baseUrl === '/api') {
      // 내부 API 클라이언트인 경우 외부 API 서버를 호출
      const externalClient = new ApiClient(getApiUrl());
      return await externalClient.getSessions(type, page, limit);
    }

    let url = type ? `/sessions/type/${type}` : '/sessions';

    // 페이지네이션 파라미터 추가
    const params = new URLSearchParams();
    if (page !== undefined) params.append('page', page.toString());
    if (limit !== undefined) params.append('limit', limit.toString());

    if (params.toString()) {
      url += `?${params.toString()}`;
    }

    const fullUrl = `${this.baseUrl}${url}`;
    
    try {
      const response = await fetch(fullUrl, {
        method: 'GET',
        headers: this.getHeaders(),
      });

      const data = await response.json();

      if (!response.ok) {
        return {
          success: false,
          error: {
            code: data.error?.code || `HTTP_ERROR_${response.status}`,
            message: data.error?.message || response.statusText,
            details: data.error?.details,
          },
          timestamp: new Date().toISOString(),
        };
      }

      // API 응답이 이미 ApiResponse 형식인 경우
      if (data && typeof data === 'object' && 'success' in data) {
        return data as ApiResponse<any>;
      }

      // API 응답이 배열인 경우 ApiResponse 형식으로 변환
      return {
        success: true,
        data: Array.isArray(data) ? data : (data?.sessions || data?.data || []),
        timestamp: new Date().toISOString(),
      };
    } catch (error) {
      return {
        success: false,
        error: {
          code: 'NETWORK_ERROR',
          message: error instanceof Error ? error.message : '네트워크 오류가 발생했습니다.',
        },
        timestamp: new Date().toISOString(),
      };
    }
  }

  async getSession(sessionId: string): Promise<ApiResponse<any>> {
    const url = `/sessions/${sessionId}`;
    const fullUrl = `${this.baseUrl}${url}`;
    
    try {
      const response = await fetch(fullUrl, {
        method: 'GET',
        headers: this.getHeaders(),
      });

      const data = await response.json();

      if (!response.ok) {
        return {
          success: false,
          error: {
            code: data.error?.code || `HTTP_ERROR_${response.status}`,
            message: data.error?.message || response.statusText,
            details: data.error?.details,
          },
          timestamp: new Date().toISOString(),
        };
      }

      // API 응답이 이미 ApiResponse 형식인 경우
      if (data && typeof data === 'object' && 'success' in data) {
        // 데이터가 있는 경우에만 정규화 처리
        if (data.data) {
          data.data = this.normalizeSessionData(data.data, sessionId);
        }
        return data;
      }

      // 세션 데이터가 직접 반환된 경우 ApiResponse 형식으로 변환
      return {
        success: true,
        data: this.normalizeSessionData(data, sessionId),
        timestamp: new Date().toISOString(),
      };
    } catch (error) {
      console.error('세션 조회 실패:', error);
      return {
        success: false,
        error: {
          code: 'NETWORK_ERROR',
          message: error instanceof Error ? error.message : '세션을 불러오는데 실패했습니다.',
        },
        timestamp: new Date().toISOString(),
      };
    }
  }

  // 세션 데이터 정규화 헬퍼 함수
  private normalizeSessionData(data: any, sessionId: string) {
    // 기본 세션 데이터 구조 정의
    const normalizedData = {
      id: data.id || sessionId,
      type: data.type || 'chat',
      history: [],
      provider: data.provider,
      model: data.model,
      lastMessage: data.lastMessage,
      updated_at: data.updated_at || new Date().toISOString(),
      messageCount: 0
    };

    // 히스토리 데이터 정규화
    if (Array.isArray(data.history)) {
      normalizedData.history = data.history.map((msg: any) => ({
        id: msg.id || `msg-${Date.now()}-${Math.random().toString(36).substring(2, 8)}`,
        role: msg.role || 'user',
        content: typeof msg.content === 'string' ? msg.content : JSON.stringify(msg.content || ''),
        originalContent: typeof msg.originalContent === 'string' ? msg.originalContent : msg.originalContent ? JSON.stringify(msg.originalContent) : null,
        timestamp: msg.timestamp || new Date().toISOString(),
        downloadInfo: msg.downloadInfo || undefined,  // Preserve downloadInfo if it exists
        attachedImages: msg.attachedImages || undefined,  // Preserve attachedImages if it exists
        thinking: msg.thinking || undefined  // Preserve thinking content if it exists
      })).filter((msg: any) => {
        // 안전하게 content 타입 체크
        const content = typeof msg.content === 'string' ? msg.content : String(msg.content || '');
        return content.trim() !== '';
      });
    }

    // 메시지 수 계산
    normalizedData.messageCount = normalizedData.history.length || data.messageCount || 0;

    return normalizedData;
  }

  async createSession(sessionData: any): Promise<any> {
    const response = await fetch(`${this.baseUrl}/sessions`, {
      method: 'POST',
      headers: this.getHeaders(),
      body: JSON.stringify(sessionData),
    });

    if (!response.ok) {
      throw new Error(`세션 생성 실패: ${response.statusText}`);
    }

    return response.json();
  }

  async updateSession(sessionId: string, sessionData: any): Promise<any> {
    const response = await fetch(`${this.baseUrl}/sessions/${sessionId}`, {
      method: 'PUT',
      headers: this.getHeaders(),
      body: JSON.stringify(sessionData),
    });

    if (!response.ok) {
      throw new Error(`세션 업데이트 실패: ${response.statusText}`);
    }

    return response.json();
  }

  async deleteSession(sessionId: string): Promise<any> {
    const response = await fetch(`${this.baseUrl}/sessions/${sessionId}`, {
      method: 'DELETE',
      headers: this.getHeaders(),
    });

    if (!response.ok) {
      throw new Error(`세션 삭제 실패: ${response.statusText}`);
    }

    return response.json();
  }

  // 전체 세션 삭제
  async deleteAllSessions(): Promise<any> {
    const response = await fetch(`${this.baseUrl}/sessions`, {
      method: 'DELETE',
      headers: this.getHeaders(),
    });

    if (!response.ok) {
      throw new Error(`전체 세션 삭제 실패: ${response.statusText}`);
    }

    return response.json();
  }

  // 파일 업로드 (RAG용) - 임베딩 완료까지 추적
  async uploadFile(
    file: File,
    userId?: string,
    targetFolder?: string,
    onProgress?: (progress: number, status: 'uploading' | 'processing' | 'completed') => void,
    ragModeOrMultiple?: string | boolean,
    projectName?: string
  ): Promise<any> {
    const formData = new FormData();
    formData.append('file', file);

    // 사용자 ID가 제공된 경우 추가 (API 서버는 user_id 필드를 사용)
    if (userId) {
      formData.append('user_id', userId);
    }

    // 타겟 폴더가 제공된 경우 추가
    if (targetFolder) {
      formData.append('target_folder', targetFolder);
    }

    // 프로젝트명이 제공된 경우 추가
    if (projectName) {
      formData.append('project_name', projectName);
    }

    // RAG 모드가 문자열로 제공된 경우 추가
    if (typeof ragModeOrMultiple === 'string') {
      formData.append('rag_mode', ragModeOrMultiple);

      // RAG_MODE가 'fast'인 경우 추가 설정들 포함
      if (ragModeOrMultiple === 'fast') {
        formData.append('rag_pdf_backend', 'pymupdf');
        formData.append('rag_process_text_only', 'yes');
        formData.append('rag_generate_image_embeddings', 'no');
        formData.append('rag_enable_graphrag', 'no');
        formData.append('use_enhanced_embedding', 'no');
      }
      // RAG_MODE가 'rich'인 경우 추가 설정들 포함
      else if (ragModeOrMultiple === 'rich') {
        formData.append('rag_pdf_backend', 'pdfplumber');
        formData.append('rag_process_text_only', 'no');
        formData.append('rag_generate_image_embeddings', 'yes');
        formData.append('rag_enable_graphrag', 'yes');
        formData.append('use_enhanced_embedding', 'yes');
      }
    }

    const headers = this.getHeaders();
    // FormData 사용 시 Content-Type 헤더 제거 (브라우저가 자동 설정)
    delete headers['Content-Type'];

    // 5분 타임아웃 설정 (임베딩 과정이 오래 걸릴 수 있음)
    const controller = new AbortController();
    const timeoutId = setTimeout(() => controller.abort(), 5 * 60 * 1000); // 5분

    try {
      // 1단계: 파일 업로드 시작
      onProgress?.(10, 'uploading');
      
      const response = await fetch(`${this.baseUrl}/rag/upload`, {
        method: 'POST',
        headers: headers,
        body: formData,
        signal: controller.signal,
      });

      if (!response.ok) {
        const errorText = await response.text();
        throw new Error(`파일 업로드 실패: ${response.statusText} - ${errorText}`);
      }

      const result = await response.json();
      
      // API 응답의 success 필드도 확인
      if (result && result.success === false) {
        const errorMessage = result.error?.message || result.error || '파일 업로드에 실패했습니다.';
        throw new Error(errorMessage);
      }

      // 2단계: 파일 업로드 완료, 임베딩 처리 시작
      onProgress?.(50, 'processing');
      
      // 3단계: 임베딩 완료 대기 (채팅 페이지 로직 참고)
      const isMultipleFiles = typeof ragModeOrMultiple === 'boolean' ? ragModeOrMultiple : false;
      await this.waitForEmbeddingCompletion(file.name, userId || 'admin', onProgress, isMultipleFiles);
      
      return result;
    } catch (error) {
      if (error instanceof Error && error.name === 'AbortError') {
        throw new Error('파일 업로드가 5분 후 타임아웃되었습니다. 파일 크기를 확인하거나 다시 시도해주세요.');
      }
      throw error;
    } finally {
      clearTimeout(timeoutId);
    }
  }

  // 임베딩 완료 대기 함수 (개선된 진행형 폴링)
  private async waitForEmbeddingCompletion(
    fileName: string, 
    userId: string,
    onProgress?: (progress: number, status: 'uploading' | 'processing' | 'completed') => void,
    isMultipleFiles: boolean = false
  ): Promise<void> {
    let checkCount = 0;
    let consecutiveFailures = 0;
    
    // 다중 파일일 경우 더 짧은 타임아웃 적용
    const maxChecks = isMultipleFiles ? 60 : 120; // 다중파일: 60초, 단일파일: 120초
    const maxConsecutiveFailures = 5;

    const checkEmbeddingStatus = async (): Promise<boolean> => {
      try {
        const ragResponse = await fetch(`${this.baseUrl}/rag/embedding-status?fileName=${encodeURIComponent(fileName)}&userId=${userId}`, {
          method: 'GET',
          headers: this.getHeaders(),
        });
        
        if (ragResponse.ok) {
          const ragData = await ragResponse.json();
          consecutiveFailures = 0; // 성공 시 실패 카운터 리셋
          
          if (ragData.success && ragData.data) {
            const fileEmbedded = ragData.data.embedded;
            
            if (fileEmbedded) {
              console.log(`파일 ${fileName} RAG 임베딩 완료`);
              onProgress?.(100, 'completed');
              return true; // 완료됨
            } else {
              // 진행 중 - 진행률 업데이트 (70-95% 사이에서 점진적 증가)
              const progressIncrement = Math.min(checkCount * 2, 25); // 최대 25% 증가
              const progress = Math.min(70 + progressIncrement, 95);
              onProgress?.(progress, 'processing');
              console.log(`파일 ${fileName} 임베딩 처리 중... (체크 ${checkCount + 1}회)`);
              return false; // 진행 중
            }
          }
        } else {
          consecutiveFailures++;
          console.warn(`임베딩 상태 확인 응답 오류 (${ragResponse.status}): ${checkCount + 1}회`);
          return false;
        }
      } catch (error) {
        consecutiveFailures++;
        console.error(`임베딩 상태 확인 실패 (${consecutiveFailures}회):`, error);
        return false;
      }
      return false;
    };

    // 진행형 폴링 간격 (초기: 500ms → 점진적으로 증가 → 최대 3초)
    const getPollingInterval = (checkCount: number): number => {
      if (checkCount < 10) return 500;   // 처음 10회: 0.5초
      if (checkCount < 20) return 1000;  // 11-20회: 1초
      if (checkCount < 40) return 2000;  // 21-40회: 2초
      return 3000;                       // 41회 이후: 3초
    };

    // 임베딩 완료 폴링
    while (checkCount < maxChecks && consecutiveFailures < maxConsecutiveFailures) {
      const isCompleted = await checkEmbeddingStatus();
      if (isCompleted) {
        return; // 임베딩 완료
      }
      
      checkCount++;
      const pollingInterval = getPollingInterval(checkCount);
      await new Promise(resolve => setTimeout(resolve, pollingInterval));
    }

    // 타임아웃 또는 연속 실패
    if (consecutiveFailures >= maxConsecutiveFailures) {
      console.warn(`파일 ${fileName} 임베딩 상태 확인 실패 (연속 ${consecutiveFailures}회 실패)`);
    } else {
      const timeoutDuration = isMultipleFiles ? "60초" : "120초";
      console.warn(`파일 ${fileName} 임베딩 완료 대기 타임아웃 (${timeoutDuration})`);
    }
    
    // 타임아웃이어도 업로드는 성공했다고 간주
    onProgress?.(100, 'completed');
  }

  // 비동기 채팅 요청 (큐 처리) - 이미지 지원 추가
  async chatAsync(message: string, sessionId?: string, options?: any, imageFile?: File): Promise<{ jobId: string }> {
    // 사용자 username과 userId 가져오기
    const username = this.isBrowser() ? localStorage.getItem('username') : null;
    const userId = this.isBrowser() ? localStorage.getItem('userId') : null;
    
    const requestBody: any = {
      prompt: message, // API 서버는 'prompt' 필드를 사용
      options: {
        provider: options?.provider || 'openai',
        model: options?.model || 'gpt-4',
        parameters: options?.parameters || {},
        rag: options?.rag || false,
        web: options?.web || false,
        ragSearchScope: options?.ragSearchScope || 'personal', // RAG 검색 범위 추가
        userId: options?.userId || (userId ? parseInt(userId) : null), // 세션 관리용 정수형 ID
        username: options?.username || username, // RAG 검색용 문자열 username
        sessionId: sessionId || undefined,
        enableSmartResearch: options?.enableSmartResearch === true,
        // 멘션 기능 관련 필드 추가
        documentContexts: options?.documentContexts || [],
        mentionedDocuments: options?.mentionedDocuments || [],
        // 기타 필요한 옵션 추가
      }
    };

    // 이미지가 있는 경우 base64로 변환하여 추가
    if (imageFile) {
      const base64Image = await this.fileToBase64(imageFile);
      requestBody.options.image = base64Image;
    }

    // console.log('API 요청 enableSmartResearch:', requestBody.options.enableSmartResearch);

    const response = await fetch(`${this.baseUrl}/chat`, {
      method: 'POST',
      headers: this.getHeaders(),
      body: JSON.stringify(requestBody),
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`비동기 채팅 요청 실패: ${response.statusText} - ${errorText}`);
    }

    const result = await response.json();
    
    // 큐 미들웨어는 직접 jobId를 반환
    if (result.success && result.jobId) {
      return { jobId: result.jobId };
    } else {
      throw new Error(result.message || '작업 ID를 받지 못했습니다.');
    }
  }

  // 채팅 작업 상태 확인
  async getChatStatus(jobId: string): Promise<any> {
    // console.log(`[DEBUG-API-CLIENT] 상태 확인 요청: ${this.baseUrl}/chat/status/${jobId}`);
    // console.log(`[DEBUG-API-CLIENT] baseUrl: ${this.baseUrl}`);
    const response = await fetch(`${this.baseUrl}/chat/status/${jobId}`, {
      method: 'GET',
      headers: this.getHeaders(),
    });

    if (!response.ok) {
      const errorText = await response.text();
      console.error(`[DEBUG-API-CLIENT] 상태 확인 실패: ${response.status} ${response.statusText} - ${errorText}`);
      throw new Error(`채팅 상태 확인 실패: ${response.statusText} - ${errorText}`);
    }

    const result = await response.json();
    // console.log(`[DEBUG-API-CLIENT] 상태 확인 응답:`, result);
    
    // API 서버의 standardResponse 형식 처리
    if (result.success === false) {
      throw new Error(result.error?.message || '상태 확인 실패');
    }
    
    return result.data || result;
  }

  // 동기 채팅 (기존 방식 유지) - 이미지 지원 추가
  async chatSync(message: string, sessionId?: string, options?: any, imageFile?: File): Promise<any> {
    // 사용자 username과 userId 가져오기
    const username = this.isBrowser() ? localStorage.getItem('username') : null;
    const userId = this.isBrowser() ? localStorage.getItem('userId') : null;
    
    const requestBody: any = {
      prompt: message, // API 서버는 'prompt' 필드를 사용
      options: {
        provider: options?.provider || 'openai',
        model: options?.model || 'gpt-4',
        parameters: options?.parameters || {},
        rag: options?.rag || false,
        web: options?.web || false,
        ragSearchScope: options?.ragSearchScope || 'personal', // RAG 검색 범위 추가
        userId: options?.userId || (userId ? parseInt(userId) : null), // 세션 관리용 정수형 ID
        username: options?.username || username, // RAG 검색용 문자열 username
        sessionId: sessionId || undefined,
        // 멘션 기능 관련 필드 추가
        documentContexts: options?.documentContexts || [],
        mentionedDocuments: options?.mentionedDocuments || [],
        // 기타 필요한 옵션 추가
      }
    };

    // 이미지가 있는 경우 base64로 변환하여 추가
    if (imageFile) {
      const base64Image = await this.fileToBase64(imageFile);
      requestBody.options.image = base64Image;
    }

    const response = await fetch(`${this.baseUrl}/chat/sync`, {
      method: 'POST',
      headers: this.getHeaders(),
      body: JSON.stringify(requestBody),
    });

    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`동기 채팅 실패: ${response.statusText} - ${errorText}`);
    }

    return response.json();
  }

  // 비동기 채팅 완료까지 대기하는 헬퍼 메서드 - 이미지 지원 추가
  async chatAsyncAndWait(
    message: string, 
    sessionId?: string, 
    options?: any,
    onProgress?: (status: string) => void,
    imageFile?: File
  ): Promise<any> {
    try {
      // console.log('=== 비동기 채팅 시작 ===', { message, sessionId, options });
      
      // 1. 비동기 채팅 요청
      const { jobId } = await this.chatAsync(message, sessionId, options, imageFile);
      // console.log('작업 ID 받음:', jobId);
      
      // 2. 상태 폴링
      return new Promise((resolve, reject) => {
        let pollCount = 0;
        const maxPolls = 180; // 최대 3분 대기 (복잡한 요청 고려)
        
        const checkStatus = async () => {
          try {
            pollCount++;
            // console.log(`[DEBUG-POLLING] 상태 확인 ${pollCount}/${maxPolls}:`, jobId);
            
            const statusResponse = await this.getChatStatus(jobId);
            // console.log('[DEBUG-API-CLIENT] 상태 응답 전체:', JSON.stringify(statusResponse, null, 2));
            
            // API 서버는 'state' 필드를 사용 (getChatStatus에서 이미 .data 추출했으므로 statusResponse에서 직접 확인)
            let state = statusResponse.state || statusResponse.status;
            
            // state가 없지만 choices가 있으면 완료된 것으로 간주
            if (!state && statusResponse.choices && Array.isArray(statusResponse.choices)) {
              // console.log('[DEBUG-API-CLIENT] state 없지만 choices 배열 발견 - 완료로 간주');
              state = 'completed';
            }
            
            // console.log('[DEBUG-API-CLIENT] 현재 상태:', state, '| 원본 state:', statusResponse.data?.state || statusResponse.state || statusResponse.status);
            
            if (onProgress) {
              // progress 메시지가 있으면 우선적으로 사용 (getChatStatus에서 이미 .data 추출했으므로 statusResponse에서 직접 확인)
              const progressArray = statusResponse.progress;
              if (progressArray && Array.isArray(progressArray) && progressArray.length > 0) {
                // console.log('[DEBUG-API-CLIENT] Progress 메시지 수신:', progressArray);
                // progress 메시지들을 JSON 형태로 전달하여 프론트엔드에서 배열로 처리 가능하게 함
                const progressData = {
                  type: 'progress_array',
                  messages: progressArray,
                  count: progressArray.length
                };
                // console.log('[DEBUG-API-CLIENT] Progress 데이터 전달:', progressData);
                onProgress(JSON.stringify(progressData));
              } else {
                // 진행 상황에 따른 상세 메시지 제공
                let progressMessage = state;
                if (state === 'pending' || state === 'queued') {
                  progressMessage = `대기 중... (${pollCount}초 경과)`;
                } else if (state === 'processing' || state === 'running') {
                  // 시간에 따른 단계별 메시지
                  if (pollCount < 5) {
                    progressMessage = `🔧 Finding tools...`;
                  } else if (pollCount < 10) {
                    progressMessage = `🧠 Analyzing...`;
                  } else if (pollCount < 20) {
                    progressMessage = `🌐 Searching...`;
                  } else if (pollCount < 30) {
                    progressMessage = `📚 Searching...`;
                  } else if (pollCount < 50) {
                    progressMessage = `🤖 Generating...`;
                  } else {
                    progressMessage = `📊 Generating... (${pollCount}초 경과)`;
                  }
                } else if (state === 'thinking') {
                  progressMessage = `🧠 Thinking...`;
                } else {
                  progressMessage = `🧠 Thinking... (${state})`;
                }
                // console.log('chatAsyncAndWait 진행 상황 콜백 호출:', progressMessage);
                onProgress(progressMessage);
              }
            }
            
            if (state === 'completed') {
              // console.log('작업 완료! 응답 데이터:', statusResponse);
              // console.log('[DEBUG-API-CLIENT] 완료 응답 구조 분석:', {
              //   hasStatusResponse: !!statusResponse,
              //   statusResponseKeys: statusResponse ? Object.keys(statusResponse) : [],
              //   hasContent: !!(statusResponse?.content),
              //   contentType: typeof statusResponse?.content,
              //   contentLength: statusResponse?.content?.length || 0,
              //   contentPreview: statusResponse?.content ? statusResponse.content.substring(0, 100) + '...' : 'no content'
              // });
              
              // 완료된 경우 결과 반환
              // getChatStatus에서 이미 .data를 추출해서 반환하므로 statusResponse가 실제 데이터
              
              // if (statusResponse?.data?.history && Array.isArray(statusResponse.data.history)) {
              //   // history 배열에서 마지막 assistant 메시지 찾기
              //   const lastAssistantMessage = statusResponse.data.history
              //     .filter((msg: any) => msg.role === 'assistant')
              //     .pop();
                
              //   if (lastAssistantMessage?.content) {
              //     const response = lastAssistantMessage.content;
              //     // console.log('[DEBUG-API-CLIENT] AI 응답 추출 성공 (data.history):', response.substring(0, 100) + '...');
              //     // sessionId도 포함하여 반환
              //     resolve({ 
              //       response,
              //       sessionId: statusResponse?.data?.sessionId || statusResponse?.sessionId
              //     });
              //     return;
              //   }
              // }
              
              if (statusResponse?.data?.choices?.[0]?.message?.content) {
                const response = statusResponse.data.choices[0].message.content;
                // console.log('[DEBUG-API-CLIENT] AI 응답 추출 성공 (data.choices):', response.substring(0, 100) + '...');
                resolve({ 
                  response,
                  sessionId: statusResponse?.data?.sessionId || statusResponse?.sessionId,
                  downloadInfo: statusResponse?.data?.downloadInfo || statusResponse?.downloadInfo || []
                });
              } else if (statusResponse?.history && Array.isArray(statusResponse.history)) {
                // 이전 버전 호환성: 직접 history 접근
                const lastAssistantMessage = statusResponse.history
                  .filter((msg: any) => msg.role === 'assistant')
                  .pop();
                
                if (lastAssistantMessage?.content) {
                  const response = lastAssistantMessage.content;
                  // console.log('[DEBUG-API-CLIENT] AI 응답 추출 성공 (direct history):', response.substring(0, 100) + '...');
                  resolve({ 
                    response,
                    sessionId: statusResponse?.sessionId,
                    downloadInfo: statusResponse?.downloadInfo || []
                  });
                  return;
                }
              } else if (statusResponse?.choices?.[0]?.message?.content) {
                // 이전 버전 호환성: 직접 choices 접근
                const response = statusResponse.choices[0].message.content;
                // console.log('[DEBUG-API-CLIENT] AI 응답 추출 성공 (direct choices):', response.substring(0, 100) + '...');
                resolve({ 
                  response,
                  sessionId: statusResponse?.sessionId,
                  downloadInfo: statusResponse?.downloadInfo || []
                });
              } else if (statusResponse?.content) {
                const response = statusResponse.content;
                // console.log('[DEBUG-API-CLIENT] AI 응답 추출 성공 (content):', response.substring(0, 100) + '...');
                resolve({ 
                  response,
                  sessionId: statusResponse?.sessionId,
                  downloadInfo: statusResponse?.downloadInfo || []
                });
              } else if (statusResponse?.response) {
                // console.log('[DEBUG-API-CLIENT] response 필드에서 응답 반환:', statusResponse.response);
                resolve({ 
                  response: statusResponse.response,
                  sessionId: statusResponse?.sessionId,
                  downloadInfo: statusResponse?.downloadInfo || []
                });
              } else {
                // console.log('[DEBUG-API-CLIENT] 전체 응답 반환:', statusResponse);
                // sessionId가 있으면 최상위로 올림
                const responseWithSessionId = {
                  ...statusResponse,
                  sessionId: statusResponse?.sessionId,
                  downloadInfo: statusResponse?.downloadInfo || []
                };
                resolve(responseWithSessionId);
              }
            } else if (state === 'failed') {
              const errorMsg = statusResponse.message || statusResponse.error?.message || '채팅 처리 실패';
              console.error('작업 실패:', errorMsg);
              reject(new Error(errorMsg));
            } else if (pollCount >= maxPolls) {
              console.error('폴링 시간 초과');
              reject(new Error(`AI 응답 시간이 초과되었습니다 (${Math.floor(maxPolls/60)}분 대기). 요청이 매우 복잡하거나 서버가 바쁜 상태입니다. 잠시 후 다시 시도하거나 더 간단한 질문으로 나누어 요청해주세요.`));
            } else {
              // 적응형 폴링: 초기 20초는 빠르게, 이후는 정상 속도
              let pollInterval = 1000; // 기본 1초
              if (pollCount <= 20) {
                pollInterval = 500; // 처음 20초는 500ms로 빠르게
              }
              
              // console.log(`${pollInterval}ms 후 다시 확인...`);
              
              // 1분마다 대기 상황 알림
              if (pollCount % 60 === 0 && pollCount > 0) {
                              //                 // console.log(`${Math.floor(pollCount/60)}분째 대기 중... 상태: ${state}`);
                if (onProgress) {
                  onProgress(`${Math.floor(pollCount/60)}분째 대기 중입니다. 복잡한 요청을 처리하고 있습니다...`);
                }
              }
              
              setTimeout(checkStatus, pollInterval);
            }
          } catch (error) {
            console.error('상태 확인 중 오류:', error);
            
            // 사용자 친화적인 오류 메시지로 변환
            let friendlyError = error;
            if (error instanceof Error) {
              const errorMessage = error.message.toLowerCase();
              
              if (errorMessage.includes('json: cannot unmarshal') || 
                  errorMessage.includes('ollama request failed')) {
                friendlyError = new Error('AI 서버가 일시적으로 응답하지 않습니다. 잠시 후 다시 시도해주세요.');
              } else if (errorMessage.includes('network error') || 
                         errorMessage.includes('fetch')) {
                friendlyError = new Error('네트워크 연결에 문제가 있습니다. 인터넷 연결을 확인해주세요.');
              } else if (errorMessage.includes('timeout')) {
                friendlyError = new Error('응답 시간이 초과되었습니다. 다시 시도해주세요.');
              }
            }
            
            reject(friendlyError);
          }
        };
        
        checkStatus();
      });
    } catch (error) {
      console.error('비동기 채팅 요청 실패:', error);
      throw error;
    }
  }

  // 실시간 로그 스트리밍을 통한 진행 상황 추적
  async chatAsyncWithRealTimeProgress(
    message: string, 
    sessionId?: string, 
    options?: any,
    onProgress?: (status: string, details?: any) => void,
    imageFile?: File
  ): Promise<any> {
    try {
      // console.log('=== 실시간 진행 상황 추적 채팅 시작 ===', { message, sessionId, options });
      
      // 1. 비동기 채팅 요청
      const { jobId } = await this.chatAsync(message, sessionId, options, imageFile);
      // console.log('작업 ID 받음:', jobId);
      
      // 초기 진행 상황 알림
      if (onProgress) {
        onProgress('🚀 작업이 시작되었습니다...');
      }
      
      // 2. 실시간 로그 스트리밍 연결 (선택적)
      let eventSource: EventSource | null = null;
      let isCompleted = false;
      let sseConnected = false;
      
      // SSE 연결 시도 (실패해도 계속 진행)
      if (this.isBrowser()) {
        try {
          const sseUrl = `${this.baseUrl}/logs/stream?sessionId=${jobId}`;
          // console.log('SSE 연결 시도:', sseUrl);
          
          eventSource = new EventSource(sseUrl);
          
          eventSource.onopen = () => {
            // console.log('SSE 연결 성공');
            sseConnected = true;
          };
          
          eventSource.onmessage = (event) => {
            try {
              const data = JSON.parse(event.data);
              // console.log('SSE 메시지 수신:', data);
              
              if (data.type === 'log' && data.message) {
                const logMessage = data.message;
                // console.log('실시간 로그:', logMessage);
                
                // 로그 메시지에서 진행 상황 파싱
                const progressInfo = this.parseProgressFromLog(logMessage);
                if (progressInfo && onProgress) {
                  // console.log('진행 상황 업데이트:', progressInfo.message);
                  onProgress(progressInfo.message, progressInfo);
                }
              }
            } catch (error) {
              console.error('로그 파싱 오류:', error);
            }
          };
          
          eventSource.onerror = (error) => {
            console.warn('SSE 연결 오류 (계속 진행):', error);
            sseConnected = false;
          };
          
        } catch (sseError) {
          console.warn('SSE 연결 실패 (계속 진행):', sseError);
          sseConnected = false;
        }
      }
      
      // 3. 상태 폴링 (기존 로직 유지)
      return new Promise((resolve, reject) => {
        let pollCount = 0;
        const maxPolls = 300; // 최대 5분 대기
        
        const checkStatus = async () => {
          try {
            pollCount++;
            console.log(`상태 확인 ${pollCount}/${maxPolls}:`, jobId);
            
            // 첫 번째 폴링에서 즉시 진행 상황 업데이트
            if (pollCount === 1 && onProgress) {
              onProgress('⏳ 대기열에서 순서를 기다리고 있습니다...');
            }
            
            const statusResponse = await this.getChatStatus(jobId);
            // console.log('상태 응답:', statusResponse);
            
            const state = statusResponse.state || statusResponse.status;
            // console.log('현재 상태:', state);
            
            if (state === 'completed') {
              isCompleted = true;
              
              // SSE 연결 정리
              if (eventSource) {
                eventSource.close();
              }
              
              // console.log('작업 완료! 응답 데이터:', statusResponse);
              
              // 완료된 경우 결과 반환
              if (statusResponse.result?.choices?.[0]?.message?.content) {
                const response = statusResponse.result.choices[0].message.content;
                console.log('AI 응답 추출 성공 (result.choices):', response);
                resolve({ response });
              } else if (statusResponse.result?.data?.choices?.[0]?.message?.content) {
                const response = statusResponse.result.data.choices[0].message.content;
                console.log('AI 응답 추출 성공 (result.data.choices):', response);
                resolve({ response });
              } else if (statusResponse.data?.choices?.[0]?.message?.content) {
                const response = statusResponse.data.choices[0].message.content;
                console.log('AI 응답 추출 성공 (data.choices):', response);
                resolve({ response });
              } else if (statusResponse.result) {
                console.log('result 필드에서 응답 반환:', statusResponse.result);
                resolve(statusResponse.result);
              } else if (statusResponse.data) {
                console.log('data 필드에서 응답 반환:', statusResponse.data);
                resolve(statusResponse.data);
              } else if (statusResponse.response) {
                console.log('response 필드에서 응답 반환:', statusResponse.response);
                resolve({ response: statusResponse.response });
              } else {
                console.log('전체 응답 반환:', statusResponse);
                resolve(statusResponse);
              }
            } else if (state === 'failed') {
              isCompleted = true;
              
              // SSE 연결 정리
              if (eventSource) {
                eventSource.close();
              }
              
              const errorMsg = statusResponse.message || statusResponse.error?.message || '채팅 처리 실패';
              console.error('작업 실패:', errorMsg);
              reject(new Error(errorMsg));
            } else if (pollCount >= maxPolls) {
              isCompleted = true;
              
              // SSE 연결 정리
              if (eventSource) {
                eventSource.close();
              }
              
              console.error('폴링 시간 초과');
              reject(new Error(`AI 응답 시간이 초과되었습니다 (${Math.floor(maxPolls/60)}분 대기). 요청이 매우 복잡하거나 서버가 바쁜 상태입니다. 잠시 후 다시 시도하거나 더 간단한 질문으로 나누어 요청해주세요.`));
            } else {
              // 아직 처리 중이면 1초 후 다시 확인
              console.log('1초 후 다시 확인...');
              
              // 진행 상황 업데이트 (SSE와 관계없이 항상 제공)
              if (onProgress) {
                let progressMessage = state;
                if (state === 'pending' || state === 'queued') {
                  progressMessage = `⏳ 대기열에서 순서를 기다리고 있습니다... (${pollCount}초 경과)`;
                } else if (state === 'processing' || state === 'running') {
                  // 시간에 따른 단계별 메시지
                  if (pollCount < 10) {
                    // progressMessage = `🔧 사용 가능한 도구를 찾는 중...`;
                  } else if (pollCount < 20) {
                    progressMessage = `🧠 AI프로를 시작합니다...`;
                  } else if (pollCount < 40) {
                    progressMessage = `🌐 웹에서 관련 정보를 검색하고 있습니다...`;
                  } else if (pollCount < 50) {
                    progressMessage = `📚 내부 문서에서 관련 정보를 검색하고 있습니다...`;
                  } else if (pollCount < 70) {
                    progressMessage = `🤖 수집된 정보를 바탕으로 응답을 생성하고 있습니다...`;
                  } else {
                    progressMessage = `📊 응답 품질을 검증하고 있습니다... (${pollCount}초 경과)`;
                  }
                } else {
                  // 알 수 없는 상태인 경우 기본 메시지
                  progressMessage = `🔄 작업을 처리하고 있습니다... (상태: ${state}, ${pollCount}초 경과)`;
                }
                // console.log('진행 상황 콜백 호출:', progressMessage);
                onProgress(progressMessage);
              }
              
              // 1분마다 대기 상황 알림
              if (pollCount % 60 === 0 && pollCount > 0) {
                console.log(`${Math.floor(pollCount/60)}분째 대기 중... 상태: ${state}`);
                if (onProgress) {
                  onProgress(`${Math.floor(pollCount/60)}분째 대기 중입니다. 복잡한 요청을 처리하고 있습니다...`);
                }
              }
              
              setTimeout(checkStatus, 1000);
            }
          } catch (error) {
            console.error('상태 확인 중 오류:', error);
            
            // SSE 연결 정리
            if (eventSource) {
              eventSource.close();
            }
            
            // 사용자 친화적인 오류 메시지로 변환
            let friendlyError = error;
            if (error instanceof Error) {
              const errorMessage = error.message.toLowerCase();
              
              if (errorMessage.includes('json: cannot unmarshal') || 
                  errorMessage.includes('ollama request failed')) {
                friendlyError = new Error('AI 서버가 일시적으로 응답하지 않습니다. 잠시 후 다시 시도해주세요.');
              } else if (errorMessage.includes('network error') || 
                         errorMessage.includes('fetch')) {
                friendlyError = new Error('네트워크 연결에 문제가 있습니다. 인터넷 연결을 확인해주세요.');
              } else if (errorMessage.includes('timeout')) {
                friendlyError = new Error('응답 시간이 초과되었습니다. 다시 시도해주세요.');
              }
            }
            
            reject(friendlyError);
          }
        };
        
        checkStatus();
      });
    } catch (error) {
      console.error('실시간 진행 상황 추적 채팅 요청 실패:', error);
      throw error;
    }
  }

  // 로그 메시지에서 진행 상황 파싱
  private parseProgressFromLog(logMessage: string): { message: string; step?: string; details?: any } | null {
    try {
      // 도구 찾기 단계
      // if (logMessage.includes('🔧 사용 가능한 도구를 찾는 중')) {
      //   return {
      //     message: '🔧 사용 가능한 도구를 찾는 중...',
      //     step: 'finding_tools'
      //   };
      // }
      
      // AI프로 리서치 시작
      if (logMessage.includes('🧠 === AI프로 리서치 시작 ===')) {
        return {
          message: '🧠 AI프로 리서치를 시작합니다...',
          step: 'ai_pro_start'
        };
      }
      
      // 웹 검색 단계
      if (logMessage.includes('🌐 웹 검색 수행 중') || logMessage.includes('🔍 웹 검색 시작')) {
        return {
          message: '🌐 웹에서 관련 정보를 검색하고 있습니다...',
          step: 'web_search'
        };
      }
      
      // 웹 검색 완료
      if (logMessage.includes('✅ 웹 검색 완료')) {
        const match = logMessage.match(/(\d+)개 결과/);
        if (match && match[1] !== '0') {
          return {
            message: `✅ 웹 검색 완료 (${match[1]}개 결과 발견)`,
            step: 'web_search_complete'
          };
        } else {
          return {
            message: `✅ 웹 검색 완료`,
            step: 'web_search_complete'
          };
        }
      }
      
      // 문서 검색 단계
      if (logMessage.includes('📚 문서 검색 수행 중') || logMessage.includes('🔍 지식 검색 시작')) {
        return {
          message: '📚 내부 문서에서 관련 정보를 검색하고 있습니다...',
          step: 'document_search'
        };
      }
      
      // 문서 검색 완료
      if (logMessage.includes('✅ 문서 검색 완료')) {
        const match = logMessage.match(/(\d+)개 결과/);
        if (match && match[1] !== '0') {
          return {
            message: `✅ 문서 검색 완료 (${match[1]}개 결과 발견)`,
            step: 'document_search_complete'
          };
        } else {
          return {
            message: `✅ 문서 검색 완료`,
            step: 'document_search_complete'
          };
        }
      }
      
      // 응답 생성 단계
      if (logMessage.includes('🤖 응답 생성 중')) {
        return {
          message: '🤖 수집된 정보를 바탕으로 응답을 생성하고 있습니다...',
          step: 'generating_response'
        };
      }
      
      // 응답 생성 완료
      if (logMessage.includes('✅ 응답 검증...')) {
        const match = logMessage.match(/\((\d+)자\)/);
        const charCount = match ? match[1] : '';
        return {
          message: `✅ 응답 검증...${charCount ? ` (${charCount}자)` : ''}`,
          step: 'response_complete'
        };
      }
      
      // 품질 검증 단계
      if (logMessage.includes('📊 품질 검증 중')) {
        return {
          message: '📊 응답 품질을 검증하고 있습니다...',
          step: 'quality_check'
        };
      }
      
      // 품질 점수
      if (logMessage.includes('📈 품질 점수')) {
        const match = logMessage.match(/(\d+)\/(\d+)/);
        if (match) {
          return {
            message: `📈 품질 검증 완료 (점수: ${match[1]}/${match[2]})`,
            step: 'quality_score'
          };
        }
      }
      
      // AI프로 완료
      if (logMessage.includes('🏁 === AI프로 리서치 완료 ===')) {
        return {
          message: '🏁 AI프로 리서치가 완료되었습니다!',
          step: 'ai_pro_complete'
        };
      }
      
      // 작업 완료
      if (logMessage.includes('✅ AI프로 리서치 완료 - 최종 응답 반환')) {
        return {
          message: '✅ 모든 작업이 완료되었습니다. 응답을 준비하고 있습니다...',
          step: 'final_complete'
        };
      }
      
      // 작업 큐 완료
      if (logMessage.includes('[chat-request-queue] 작업 완료')) {
        return {
          message: '🎉 작업이 성공적으로 완료되었습니다!',
          step: 'queue_complete'
        };
      }
      
      return null;
    } catch (error) {
      console.error('로그 파싱 오류:', error);
      return null;
    }
  }

  // 프로바이더 관련 메서드들
  async getProviders(): Promise<ApiResponse<any>> {
    return this.get('/providers');
  }

  async getAvailableProviders(): Promise<ApiResponse<any>> {
    return this.get('/providers/available');
  }

  async getProviderModels(provider: string): Promise<ApiResponse<any>> {
    return this.get(`/providers/${provider}/models`);
  }

  async validateProvider(provider: string): Promise<ApiResponse<any>> {
    return this.get(`/providers/${provider}/validate`);
  }

  async getCurrentProviderConfig(): Promise<ApiResponse<any>> {
    return this.get('/providers/current');
  }

  // RAG 파일 관리
  async getRagFiles(userId: string): Promise<ApiResponse<any>> {
    return this.get('/rag/files', { userId });
  }

  // 관리자용 전체 RAG 폴더 구조 조회
  async getRagAllUsers(): Promise<ApiResponse<any>> {
    const username = localStorage.getItem('username') || '';
    const userRole = localStorage.getItem('userRole') || '';
    return this.request('/rag/files/all-users', {
      method: 'GET',
      headers: {
        'x-username': username,
        'x-user-role': userRole
      }
    });
  }

  async deleteRagFile(filename: string, userId: string): Promise<ApiResponse<any>> {
    return this.request('/rag/file', {
      method: 'DELETE',
      body: { filename, userId }
    });
  }

  // 리포트 생성 (비동기) - 섹션별 수정 지원 확장
  async generateReport(options: {
    prompt: string;
    template?: string;
    format?: string;
    email?: string;
    username?: string;
    // 섹션별 수정 기능을 위한 추가 필드들
    modification_mode?: 'full' | 'partial';
    existing_document_content?: string;
    existing_document_path?: string;
    sections_to_modify?: string[];
    preserve_formatting?: boolean;
    // 편집 모드 관련 메타데이터
    editMethod?: 'full' | 'section';
    editTargetDocument?: string;
    editTargetSection?: string;
    ragSearchScope?: string;
    userId?: string;
    // AI 모델 설정
    provider?: string;
    model?: string;
  }): Promise<ApiResponse<any>> {
    return this.post('/report', options);
  }

  // 리포트 생성 상태 확인
  async getReportStatus(jobId: string): Promise<ApiResponse<any>> {
    return this.get(`/report/status/${encodeURIComponent(jobId)}`);
  }

  // 리포트 템플릿 목록 조회
  async getReportTemplates(): Promise<ApiResponse<any>> {
    // 🚀 가벼운 템플릿 목록 API 사용 (빠른 로딩)
    return this.get('/report/templates-simple');
  }

  // 리포트 생성 비동기 요청 및 대기 (폴링)
  async generateReportAndWait(
    options: {
      prompt: string;
      template?: string;
      format?: string;
      email?: string;
      username?: string;
    },
    onProgress?: (status: string, details?: any) => void
  ): Promise<any> {
    try {
      console.log('[리포트생성] 요청:', options);
      
      // 1. 리포트 생성 요청
      const response = await this.generateReport(options);
      console.log('[리포트생성] 응답:', response);
      
      if (!response.success) {
        throw new Error(response.error?.message || '리포트 생성 요청 실패');
      }

      // 응답 데이터 구조 확인 - response.data 또는 response 직접 사용
      const data = response.data || response;
      
      // 2-1. 즉시 완료된 경우 (file_path가 있는 경우)
      if (data?.file_path) {
        console.log('[리포트생성] 즉시 완료됨:', data);
        onProgress?.('completed', data);
        
        return {
          success: true,
          file_path: data.file_path,
          format: data.format || options.format,
          template: data.template || options.template,
          project_hash: data.project_hash,
          message: data.message || '리포트가 생성되었습니다.'
        };
      }
      
      // 2-2. 큐에 추가된 경우 (jobId가 있는 경우)
      const jobId = data?.jobId;
      if (!jobId) {
        console.log('[리포트생성] 응답 구조 디버그:', { response, data, hasJobId: !!data?.jobId, hasFilePath: !!data?.file_path });
        throw new Error('예상치 못한 응답 형식입니다. jobId 또는 file_path가 필요합니다.');
      }

      console.log('[리포트생성] 큐에 추가됨, jobId:', jobId);
      onProgress?.('queued', { jobId });

      // 3. 상태 폴링
      return new Promise((resolve, reject) => {
        let pollCount = 0;
        const maxPolls = 1200; // 최대 1시간 (3초 * 1200 = 3600초 = 1시간)
        
        const checkStatus = async () => {
          try {
            pollCount++;
            console.log(`[리포트생성] 상태 확인 ${pollCount}/${maxPolls} - jobId: ${jobId}`);
            
            const statusResponse = await this.getReportStatus(jobId);
            console.log(`[리포트생성] 상태 응답:`, statusResponse);
            
            // 응답 구조 확인: statusResponse.data 또는 statusResponse 직접 사용
            const data = statusResponse.data || statusResponse;
            const state = data.state || data.status;
            
            console.log(`[리포트생성] 현재 상태: ${state}`);

            if (onProgress) {
              onProgress(state, data);
            }

            if (state === 'completed') {
              console.log(`[리포트생성] 완료됨:`, data);
              resolve({
                success: true,
                file_path: data.file_path,
                format: data.format,
                template: data.template,
                project_hash: data.project_hash || jobId,
                message: data.message || '리포트가 생성되었습니다.'
              });
            } else if (state === 'failed') {
              console.log(`[리포트생성] 실패:`, data.error);
              reject(new Error(data.error || '리포트 생성 실패'));
            } else {
              // 진행 중 (waiting, active, queued, etc.)
              console.log(`[리포트생성] 진행 중... (${state})`);
              if (pollCount >= maxPolls) {
                reject(new Error('리포트 생성 시간 초과 (1시간 경과)'));
                return;
              }
              
              // 적응형 폴링 간격: 처음 5분은 3초, 이후는 10초 간격
              const pollInterval = pollCount <= 100 ? 3000 : 10000; // 100회(5분) 후부터는 10초 간격
              setTimeout(checkStatus, pollInterval);
            }
          } catch (error) {
            console.error(`[리포트생성] 상태 확인 중 오류:`, error);
            if (pollCount >= maxPolls) {
              reject(new Error('리포트 생성 시간 초과 (1시간 경과)'));
            } else {
              // 오류 발생 시에도 적응형 간격 적용
              const pollInterval = pollCount <= 100 ? 3000 : 10000;
              setTimeout(checkStatus, pollInterval);
            }
          }
        };

        // 첫 번째 상태 확인을 2초 후에 시작 (큐 처리 시간 고려)
        setTimeout(checkStatus, 2000);
      });
    } catch (error) {
      console.error('[리포트생성] 오류:', error);
      throw error;
    }
  }

  // 템플릿 파일 업로드
  async uploadTemplateFile(file: File): Promise<ApiResponse<any>> {
    const formData = new FormData();
    formData.append('file', file);

    const headers = this.getHeaders();
    // FormData를 사용할 때는 Content-Type을 제거해야 브라우저가 자동으로 설정
    delete headers['Content-Type'];

    const url = `${this.baseUrl}/template/upload`;

    try {
      const response = await fetch(url, {
        method: 'POST',
        headers,
        body: formData,
      });

      const data = await response.json();

      if (!response.ok) {
        return {
          success: false,
          error: {
            code: data.error?.code || `HTTP_ERROR_${response.status}`,
            message: data.error?.message || response.statusText,
            details: data.error?.details,
          },
          timestamp: new Date().toISOString(),
        };
      }

      return data as ApiResponse<any>;
    } catch (error) {
      return {
        success: false,
        error: {
          code: 'NETWORK_ERROR',
          message: error instanceof Error ? error.message : '네트워크 오류가 발생했습니다.',
        },
        timestamp: new Date().toISOString(),
      };
    }
  }

  // 백엔드 피드백 시스템을 통한 Langfuse 피드백 전송
  async sendFeedbackToLangfuse(feedbackData: {
    messageId: string;
    sessionId: string;
    rating: number;
    userMessage: string;
    aiResponse: string;
    timestamp: string;
  }): Promise<any> {
    try {
      console.log('백엔드 피드백 시스템으로 전송할 데이터:', feedbackData);

      // 백엔드의 피드백 API 호출
      const response = await this.post('/feedback', {
        messageId: feedbackData.messageId,
        sessionId: feedbackData.sessionId,
        rating: feedbackData.rating,
        userMessage: feedbackData.userMessage,
        aiResponse: feedbackData.aiResponse,
        timestamp: feedbackData.timestamp
      });

      if (response.success) {
        console.log('백엔드 피드백 전송 성공:', response.data);
        return response.data;
      } else {
        throw new Error(response.error?.message || '피드백 전송 실패');
      }
    } catch (error) {
      console.error('백엔드 피드백 전송 실패:', error);
      throw error;
    }
  }

  // 파일을 base64로 변환하는 헬퍼 메서드
  private fileToBase64(file: File): Promise<string> {
    return new Promise((resolve, reject) => {
      const reader = new FileReader();
      reader.readAsDataURL(file);
      reader.onload = () => {
        const result = reader.result as string;
        // data:image/jpeg;base64, 부분을 제거하지 않고 전체를 반환
        resolve(result);
      };
      reader.onerror = error => reject(error);
    });
  }

  // 코드 생성 비동기 요청
  async codeAsync(message: string, sessionId?: string, options?: any, imageFile?: File): Promise<{ jobId: string }> {
    const username = this.isBrowser() ? localStorage.getItem('username') : null;
    const userId = this.isBrowser() ? localStorage.getItem('userId') : null;
    const requestBody: any = {
      prompt: message,
      sessionId,
      provider: options?.provider || 'openai',
      model: options?.model || 'gpt-4',
      parameters: options?.parameters || {},
      rag: options?.rag || false,
      web: options?.web || false,
      ragSearchScope: options?.ragSearchScope || 'personal', // RAG 검색 범위 추가
      userId: options?.userId || (userId ? parseInt(userId) : null), // 세션 관리용 정수형 ID
      username: options?.username || username // RAG 검색용 문자열 username
    };
    if (imageFile) {
      const base64Image = await this.fileToBase64(imageFile);
      requestBody.options.image = base64Image;
    }
    const response = await fetch(`${this.baseUrl}/code`, {
      method: 'POST',
      headers: this.getHeaders(),
      body: JSON.stringify(requestBody),
    });
    if (!response.ok) {
      const errorText = await response.text();
      throw new Error(`비동기 코드 요청 실패: ${response.statusText} - ${errorText}`);
    }
    const result = await response.json();
    if (result.success && result.jobId) {
      return { jobId: result.jobId };
    } else {
      throw new Error(result.message || '작업 ID를 받지 못했습니다.');
    }
  }

  // 코드 생성 비동기 요청 및 대기 (폴링)
  async codeAsyncAndWait(
    message: string,
    sessionId?: string,
    options?: any,
    onProgress?: (status: string) => void,
    imageFile?: File
  ): Promise<any> {
    try {
      // 1. 비동기 코드 생성 요청
      const job = await this.codeAsync(message, sessionId, options, imageFile);
      const jobId = typeof job === 'object' && job !== null ? job.jobId : job;
      if (!jobId) throw new Error('jobId가 올바르게 반환되지 않았습니다.');
      // console.log('codeAsync 반환값:', jobId);

      // 2. 상태 폴링 (chatAsyncAndWait와 동일하게)
      return new Promise((resolve, reject) => {
        let pollCount = 0;
        const maxPolls = 180;
        const checkStatus = async () => {
          try {
            pollCount++;
            const statusResponse = await this.get(`/code/status/${jobId}`);
            // data가 null이거나 아예 없으면 즉시 종료
            if (!statusResponse.data || statusResponse.data === null) {
              reject(new Error(`작업 상태를 찾을 수 없습니다. (data가 null)\n실제 응답: ${JSON.stringify(statusResponse)}`));
              return;
            }
            // 만약 state/status가 없지만 결과 데이터가 있으면 completed로 간주
            const hasResultData = !!(statusResponse.data.data && (
              (statusResponse.data.data.choices && statusResponse.data.data.choices.length > 0) ||
              statusResponse.data.data.python_code
            ));
            
            let state = statusResponse.data.state ?? statusResponse.data.status;

            if ((state === undefined || state === null) && hasResultData) {
              // 강제로 completed로 처리
              if (onProgress && typeof state === 'string') onProgress('completed');
              // 아래 completed 처리로 이동
              state = 'completed';
            }

            if (state === undefined || state === null) {
              reject(new Error(`작업 상태(state/status)가 undefined입니다. 백엔드 응답을 확인하세요.\n실제 응답: ${JSON.stringify(statusResponse)}`));
              return;
            }
            if (onProgress && typeof state === 'string') onProgress(state);

            if (state === 'completed') {
              // 핵심 데이터
              const core = statusResponse.data.data || statusResponse.data.result || statusResponse.data;

              let responseContent = null;
              if (core?.choices?.[0]?.message?.content) {
                responseContent = core.choices[0].message.content;
              } else if (core?.python_code) {
                responseContent = core.python_code;
              } else if (core?.response) {
                responseContent = core.response;
              } else if (typeof core === 'string') {
                responseContent = core;
              } else if (core?.content) {
                responseContent = core.content;
              } else if (core) {
                responseContent = JSON.stringify(core);
              }

              if (responseContent) {
                // ryan
                // resolve({ response: responseContent });
                const res = await externalApiClient.get("/config/USE_REVIEW");
                if (res.success && res.data) {
                    const reviewFlag = res.data.value;
                    // console.log("🔑 get config use_review value:", reviewFlag);
                }
                
                // downloadInfo와 webReferences 추출
                const downloadInfo = core?.downloadInfo || statusResponse.data?.downloadInfo || [];
                const webReferences = core?.webReferences || statusResponse.data?.webReferences || [];
                
                resolve({
                    response: responseContent,
                    reviewFlag: res.data.value,
                    message: message,
                    sessionId: statusResponse.data.data?.sessionId || statusResponse.data.sessionId,
                    downloadInfo: downloadInfo,
                    webReferences: webReferences
                });
                // end-ryan
              } else {
                resolve(statusResponse.data);
              }
            } else if (state === 'failed') {
              reject(new Error(statusResponse.data.error || '코드 생성 실패'));
            } else if (state === 'cancelled') {
              reject(new Error('코드 생성이 취소되었습니다'));
            } else if (pollCount >= maxPolls) {
              reject(new Error('AI 응답 시간이 초과되었습니다.'));
            } else {
              setTimeout(checkStatus, 1000);
            }
          } catch (error) {
            reject(error);
          }
        };
        checkStatus();
      });
    } catch (error) {
      throw error;
    }
  }

  // ryan -- code execute and wait
  codeExecuteAndWait(
    // prompt: string,
    pythonCode: string,
    options: {
        autoInstallPackages?: boolean;
        useReview?: boolean;
        sessionId?: string;
    } = {},
    onProgress?: (status: string) => void
  ): Promise<{
      response: string;
      reviewFlag: string;
      isSuccess: boolean;
      message?: string;
      isPromptCode?: string;
  }> {
      return new Promise(async (resolve, reject) => {
          try {
              // console.log("================== codeExecuteAndWait 호출 ==================", "sessionId:", options.sessionId);

              // 1) 요청 전송
              onProgress?.("코드 실행 요청 중…");
              const res = await fetch(`${this.baseUrl}/code/execute`, {
                  method: "POST",
                  headers: this.getHeaders(),
                  body: JSON.stringify({
                      python_code: pythonCode,
                      options: {
                          autoInstallPackages: options.autoInstallPackages,
                          useReview: options.useReview,
                          sessionId: options.sessionId
                      }
                  })
              });

              // 2) 응답 수신
              onProgress?.("⬇️ 코드 실행 응답 수신 중…");
              if (!res.ok) {
                  return reject(new Error(`HTTP ${res.status}`));
              }

              // 3) 페이로드 언베벨
              const envelope = (await res.json()) as {
                  success: boolean;
                  data: {
                      stdout: string;
                      stderr: string;
                      code: number;
                      scriptPath?: string;
                      improved_success?: boolean;
                  };
                  error: any;
                  timestamp: string;
              };

              if (!envelope.success) {
                  return reject(new Error(envelope.error || "codeExecute failed"));
              }

              const execResult = envelope.data;
              onProgress?.("✅ 코드 실행 완료");

              // // 4) 결과 가공
              const isSuccess = !execResult.stderr;
              const message = isSuccess ? "success" : execResult.stderr.trim();

              // 실패 시에만 reviewFlag에 오류 메시지 설정
              const reviewFlag = isSuccess ? "" : execResult.stderr.trim();

              // 5) 최종 resolve
              resolve({
                  response: isSuccess ? `실행 결과: ${execResult.stdout.trim()}` : `실행 오류:\n${execResult.stderr.trim()}`,
                  reviewFlag,
                  isSuccess: isSuccess ? true : false,
                  message,
                  isPromptCode: pythonCode
              });

          } catch (err) {
              console.error("[codeExecuteAndWait] 에러:", err);
              reject(err);
          }
      });
  }
  // end  ryan -- code execute and wait


  // ryan -- code save and wait
  executeCodeSaveZip(
    options: {
      continueMode?: string;
    } = {},
    pythonCode: string,
    onProgress?: (status: string) => void
  ): Promise<{
      isSuccess: boolean;
      msg: string;
      downloadUrl?: string;
  }> {
    return new Promise(async (resolve, reject) => {
      try {
        // 1) 요청 전송
        onProgress?.("📤 코드 실행 요청 중…");
        const res = await fetch(`${this.baseUrl}/code/saveCode`, {
          method: "POST",
          headers: this.getHeaders(),
          body: JSON.stringify({
            options: {
              continueMode: options.continueMode
            },
            python_code: pythonCode
          })
        });

        // 2) 응답 수신
        onProgress?.("⬇️ 코드 실행 응답 수신 중…");
        if (!res.ok) {
          return reject(new Error(`HTTP ${res.status}`));
        }

        const envelope = await res.json();
        if (!envelope.success) {
          return reject(new Error(envelope.msg || "codeExecute failed"));
        }

        onProgress?.("✅ 코드 실행 완료");

        // 5) 최종 resolve
        resolve({
          isSuccess: envelope.success,
          msg: envelope.data?.msg,                 // 파일 경로
          downloadUrl: envelope.data?.downloadUrl // 다운로드 URL
        });

      } catch (err) {
        console.error("[codeExecuteAndWait] 에러:", err);
        reject(err);
      }
    });
  }
  // end  ryan -- code execute and wait

  // ryan -- executeCodeSaveZip download
  async downloadCodeSaveZip(downloadUrl: string): Promise<void> {
    try {

      // console.log("다운로드 URL:", downloadUrl);
      // console.log("this.getHeaders()==="+ JSON.stringify(this.getHeaders()));
      const response = await fetch(`${this.baseUrl}${downloadUrl}`, {
        method: 'GET',
        headers: this.getHeaders(),
      });
      // console.log("다운로드 응답 상태:", JSON.stringify(response));
      if (!response.ok) {
        throw new Error(`다운로드 실패: HTTP ${response.status}`);
      }
      const blob = await response.blob();
      const url = window.URL.createObjectURL(blob);
      const a = document.createElement('a'); 
      a.href = url;
      a.download = 'code.zip'; // 다운로드할 파일 이름
      document.body.appendChild(a);
      a.click(); // 다운로드 시작
      a.remove(); // 다운로드 후 링크 제거
      window.URL.revokeObjectURL(url); // 메모리 해제
      // console.log("코드 ZIP 파일 다운로드 완료");
    } catch (error) {
      console.error("코드 ZIP 파일 다운로드 중 오류 발생:", error);
      throw new Error(`코드 ZIP 파일 다운로드 실패: ${error instanceof Error ? error.message : '알 수 없는 오류'}`);
    }
  }
  //end ryan -- executeCodeSaveZip download

  // ryan -- task save

  executeCodeTaskSave(
    taskData: { prompt: string; pythonCode: string; taskName: string; userName: string; },
    onProgress?: (status: string) => void
  ): Promise<{
      isSuccess: boolean;
      msg: string;
      taskId: string;
      taskName: string;
      prompt: string;
      code: string;
  }> {
    return new Promise(async (resolve, reject) => {
      try {
        // console.log("executeCodeTaskSave 호출:", `${this.baseUrl}/tasks`);
        // 1) 요청 전송
        // console.log("exe---task username :", taskData.userName);
        onProgress?.("📤 코드 실행 요청 중…");
        const res = await fetch(`${this.baseUrl}/tasks`, {
          method: "POST",
          headers: this.getHeaders(),
          body: JSON.stringify({
            name: taskData.taskName,
            prompt: taskData.prompt,
            code: taskData.pythonCode,
            userName: taskData.userName
          })
        });
        
        // 2) 응답 수신
        onProgress?.("⬇️ 코드 실행 응답 수신 중…");
        if (!res.ok) {
          return reject(new Error(`HTTP ${res.status}`));
        }

        const envelope = await res.json();
        if (!envelope.success || !envelope.data?.success) {
          return reject(new Error(envelope.message || "taskSave failed"));
        }

        onProgress?.("✅ 코드 실행 완료");

        // task 객체를 직접 반환
        const task = envelope.data.task;

        resolve({
          isSuccess: true,
          msg: "타스크가 성공적으로 저장되었습니다.",
          taskId: task.id,
          taskName: task.name,
          prompt: task.prompt,
          code: task.code,
        });

      } catch (err) {
        console.error("[codeExecuteAndWait] 에러:", err);
        reject(err);
      }
    });
  }
  // end  ryan -- task save


  // ryan -- code task save
  codeTaskSaveAndWait(
    taskname: string,
    prompt?: string,
    pythonCode?: string,
    sessionId?: string,
    onProgress?: (status: string) => void
  ): Promise<{
    response: string;
    reviewFlag: string;
    isSuccess: boolean;
    message?: string;
    isPromptCode?: string;
  }> {
    return new Promise(async (resolve, reject) => {
        try {
            // 1) 요청 전송
            onProgress?.("코드 실행 요청 중…");
            const res = await fetch(`${this.baseUrl}/tasks`, {
                method: "POST",
                headers: this.getHeaders(),
                body: JSON.stringify({
                    name: taskname,
                    prompt: prompt,
                    code: pythonCode
                    // sessionId: options.sessionId
                })
            });

            // 2) 응답 수신
            onProgress?.("⬇️ 코드 실행 응답 수신 중…");
            if (!res.ok) {
                return reject(new Error(`HTTP ${res.status}`));
            }

            // 3) 페이로드 언베벨
            const envelope = (await res.json()) as {
                success: boolean;
                data: {
                    stdout: string;
                    stderr: string;
                    code: number;
                    scriptPath?: string;
                    improved_success?: boolean;
                };
                error: any;
                timestamp: string;
            };

            if (!envelope.success) {
                return reject(new Error(envelope.error || "codeExecute failed"));
            }

            const execResult = envelope.data;
            onProgress?.("✅ 코드 실행 완료");

            // // 4) 결과 가공
            const isSuccess = !execResult.stderr;
            const message = isSuccess ? "success" : execResult.stderr.trim();

            // 성공이면 'y', 실패면 오류 메시지
            const reviewFlag = !isSuccess || execResult.improved_success ? "yes" : "no";

            // 5) 최종 resolve
            resolve({
                response: isSuccess ? `실행 결과: ${execResult.stdout.trim()}` : `실행 오류:\n${execResult.stderr.trim()}`,
                reviewFlag,
                isSuccess: isSuccess ? true : false,
                message,
                isPromptCode: pythonCode
            });

        } catch (err) {
            console.error("[codeExecuteAndWait] 에러:", err);
            reject(err);
        }
    });
  }
  // end  ryan -- code task save

  // 사용자 목록 조회 (검색/페이지네이션 지원)
  async getUsers(params?: { search?: string; page?: number; pageSize?: number }): Promise<ApiResponse<{ users: User[]; total: number; page: number; pageSize: number }>> {
    const query: Record<string, string> = {};
    if (params?.search) query.search = params.search;
    if (params?.page) query.page = String(params.page);
    if (params?.pageSize) query.pageSize = String(params.pageSize);
    return this.get<{ users: User[]; total: number; page: number; pageSize: number }>(
      '/admin/users',
      query
    );
  }

  // 사용자 생성
  async createUser(user: Partial<User>): Promise<ApiResponse<User>> {
    return this.post<User>('/admin/users', user);
  }

  // 사용자 수정
  async updateUser(userId: string, user: Partial<User>): Promise<ApiResponse<User>> {
    return this.put<User>(`/admin/users/${userId}`, user);
  }

  // 사용자 삭제
  async deleteUser(userId: string): Promise<ApiResponse<null>> {
    return this.delete<null>(`/admin/users/${userId}`);
  }

  // 사용자 API 키 재생성 (단일키 정책: 기존 모든 키 비활성화 후 새로 발급)
  async regenerateUserApiKey(userId: string): Promise<ApiResponse<{ key: string }>> {
    const res = await this.post<{ key: string }>(`/admin/api-keys`, { userId });
    return res;
  }

  // 로고 관련 메서드 (Next.js API 라우트 사용)
  async getLogo(): Promise<ApiResponse<{ logoUrl: string | null }>> {
    try {
      const response = await fetch('/api/config/logo', {
        method: 'GET',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      const data = await response.json();
      return data as ApiResponse<{ logoUrl: string | null }>;
    } catch (error: any) {
      return {
        success: false,
        error: {
          code: 'NETWORK_ERROR',
          message: error.message || 'Network request failed',
        },
        timestamp: new Date().toISOString(),
      };
    }
  }

  async uploadLogo(file: File): Promise<ApiResponse<{ logoUrl: string }>> {
    try {
      const formData = new FormData();
      formData.append('logo', file);

      const response = await fetch('/api/config/logo', {
        method: 'POST',
        body: formData,
      });

      const data = await response.json();
      return data as ApiResponse<{ logoUrl: string }>;
    } catch (error: any) {
      return {
        success: false,
        error: {
          code: 'NETWORK_ERROR',
          message: error.message || 'Network request failed',
        },
        timestamp: new Date().toISOString(),
      };
    }
  }

  async deleteLogo(): Promise<ApiResponse<{ message: string }>> {
    try {
      const response = await fetch('/api/config/logo', {
        method: 'DELETE',
        headers: {
          'Content-Type': 'application/json',
        },
      });

      const data = await response.json();
      return data as ApiResponse<{ message: string }>;
    } catch (error: any) {
      return {
        success: false,
        error: {
          code: 'NETWORK_ERROR',
          message: error.message || 'Network request failed',
        },
        timestamp: new Date().toISOString(),
      };
    }
  }

  // FlowAI node testing
  async testNode(nodeData: any): Promise<ApiResponse<any>> {
    return this.post('/flowai/test-node', nodeData);
  }
}

// API URL을 동적으로 결정하는 함수
function getApiUrl(): string {
  // 환경 변수 우선 사용 (클라이언트에서는 NEXT_PUBLIC_ 접두사가 있는 것만 접근 가능)
  const apiUrl = process.env.NEXT_PUBLIC_API_SERVER_URL || 
                 process.env.NEXT_PUBLIC_API_URL || 
                 'http://localhost:5500/api/v1';
  
  // 디버깅을 위한 로그
  // console.log('[DEBUG] getApiUrl() 호출됨');
  // console.log('[DEBUG] NEXT_PUBLIC_API_SERVER_URL:', process.env.NEXT_PUBLIC_API_SERVER_URL);
  // console.log('[DEBUG] 최종 API URL:', apiUrl);
  
  return apiUrl;
}

// 내부 API 클라이언트 (Next.js API Route)
export const internalApiClient = new ApiClient('/api');

// 외부 API 클라이언트 (실제 API 서버)
export const externalApiClient = new ApiClient(getApiUrl());

// 기본 API 클라이언트 (default export)
const apiClient = new ApiClient();
export default apiClient;

// 설정 관련 API
export async function getConfig(key: string): Promise<any> {
  const res = await fetch(`/api/settings?key=${encodeURIComponent(key)}`, {
    method: 'GET',
    headers: {
      'Content-Type': 'application/json'
    },
    credentials: 'include' // 쿠키 포함하여 전송
  });

  if (!res.ok) {
    const error = await res.json();
    throw new Error(error.error?.message || '설정을 가져오는데 실패했습니다.');
  }

  const data = await res.json();
  return data.data.value;
}

export async function setConfig(key: string, value: any): Promise<any> {
  const res = await fetch('/api/settings', {
    method: 'POST',
    headers: {
      'Content-Type': 'application/json'
    },
    credentials: 'include', // 쿠키 포함하여 전송
    body: JSON.stringify({ key, value })
  });

  if (!res.ok) {
    const error = await res.json();
    throw new Error(error.error?.message || '설정을 저장하는데 실패했습니다.');
  }

  const data = await res.json();
  return data.data;
}

// 기술 지원 관련 API
const headers = {
  'Content-Type': 'application/json',
};

const handleResponse = async (response: Response) => {
  if (!response.ok) {
    const errorData = await response.json().catch(() => ({ message: 'An unknown error occurred' }));
    throw new Error(errorData.message || 'API request failed');
  }
  return response.json();
};

export const supportApi = {
  getTickets: async (params: {
    userName: string;
    userRole: string;
    page: number;
    limit: number;
    searchTerm: string;
  }) => {
    const queryParams = new URLSearchParams({
        userName: params.userName,
        userRole: params.userRole,
        page: String(params.page),
        limit: String(params.limit),
        searchTerm: params.searchTerm,
    });
    const response = await fetch(`/api/support/issues?${queryParams.toString()}`);
    return handleResponse(response);
  },

  getTicket: async (id: string) => {
    const response = await fetch(`/api/support/issues/${id}`);
    return handleResponse(response);
  },

  createTicket: async (formData: FormData) => {
    const response = await fetch('/api/support/issues', {
      method: 'POST',
      body: formData,
    });
    return handleResponse(response);
  },

  addReply: async (ticketId: string, replyData: FormData) => {
    const response = await fetch(`/api/support/issues/${ticketId}`, {
      method: 'POST',
      body: replyData,
    });
    return handleResponse(response);
  },

  updateTicket: async (ticketId: string, data: any) => {
    const response = await fetch(`/api/support/issues/${ticketId}`, {
      method: 'PUT',
      headers,
      body: JSON.stringify(data),
    });
    return handleResponse(response);
  },
  
  uploadSupportDocuments: async (formData: FormData) => {
    const response = await fetch('/api/support/upload', {
        method: 'POST',
        body: formData,
    });
    return handleResponse(response);
  },
};
