import { requireAuth } from './auth.js';

// 리뷰 관련 유틸리티 함수들
async function getUseReview(reqAPI) {
    return await reqAPI('getconfig', { key: 'USE_REVIEW' }) || 'NO';
}

async function performResultReview(reqAPI) { 
    return (await getUseReview(reqAPI)) === 'yes'; 
}

export { getUseReview, performResultReview };

export async function logsPage(chatMessages, translations, reqAPI, getText, showCustomDialog) {
    // 사용자 인증 체크
    const hasAccess = await requireAuth(reqAPI);
    if (!hasAccess) {
        return;
    }

    // 입력 컨테이너 숨기기
    const inputContainer = document.querySelector('.input-container');
    inputContainer.style.display = 'none';

    let currentPage = 1;
    const itemsPerPage = 10;
    let totalLogs = [];

    chatMessages.innerHTML = `
        <div class="page-container">
            <div class="page-header">
                <h2 class="page-title">${getText('logs', 'title')}</h2>
                <p class="page-description">${getText('logs', 'description')}</p>
                <div class="page-actions">
                    <button id="clearLogs" class="btn-delete">${getText('logs', 'clearLogs')}</button>
                </div>
            </div>
            <div id="logsList"></div>
            <div class="pagination" id="pagination">
                <button id="prevPage" class="btn-execute">&lt; ${getText('common', 'previous')}</button>
                <span id="pageInfo"></span>
                <button id="nextPage" class="btn-execute">${getText('common', 'next')} &gt;</button>
            </div>
        </div>
    `;

    // Clear Logs 버튼 클릭 이벤트 핸들러
    document.getElementById('clearLogs').addEventListener('click', async () => {
        const confirmed = await showCustomDialog({
            title: getText('dialog', 'clearLogsTitle'),
            message: getText('dialog', 'clearLogsMessage'),
            okText: getText('dialog', 'clearLogsOk'),
            cancelText: getText('dialog', 'clearLogsCancel')
        });
        if (confirmed) {
            try {
                await reqAPI('clearExecutionLogs');
                await refreshLogs();
            } catch (error) {
                await reqAPI('errnotify', getText('logs', 'clearError'));
            }
        }
    });

    // 로그 목록 새로고침 함수
    async function refreshLogs() {
        const logsList = document.getElementById('logsList');
        const pagination = document.getElementById('pagination');
        const prevPageBtn = document.getElementById('prevPage');
        const nextPageBtn = document.getElementById('nextPage');
        const pageInfo = document.getElementById('pageInfo');
        
        try {
            totalLogs = await reqAPI('getExecutionLogs') || [];
            
            logsList.innerHTML = '';
            
            const totalLogsCount = document.createElement('div');
            totalLogsCount.className = 'total-logs-count';
            totalLogsCount.innerHTML = `${getText('logs', 'totalLogs')}<span>${totalLogs.length}</span>`;
            logsList.appendChild(totalLogsCount);

            if (totalLogs.length === 0) {
                logsList.appendChild(createNoLogsMessage());
                pagination.style.display = 'none';
                return;
            }

            const totalPages = Math.ceil(totalLogs.length / itemsPerPage);
            const startIndex = (currentPage - 1) * itemsPerPage;
            const endIndex = startIndex + itemsPerPage;
            const currentLogs = totalLogs.slice(startIndex, endIndex);

            currentLogs.forEach(log => {
                const timestamp = new Date(log.timestamp).toLocaleString();
                const statusClass = log.success ? 'success' : 'error';
                const statusText = log.success ? getText('logs', 'success') : getText('logs', 'failed');
                
                const logItem = document.createElement('div');
                logItem.className = `log-item ${statusClass}`;
                
                // 사용자 질의와 코드를 분리
                let userQuery = '';
                let codeContent = '';
                let content = '';

                try {
                    const userQueryPrefix = `${getText('logs', 'userQuery')}:`;
                    const codePrefix = `${getText('logs', 'executedCode')}:`;
                    
                    // content 처리
                    content = log.content || '';
                    if (typeof content === 'object') {
                        content = JSON.stringify(content, null, 2);
                    }

                    // 사용자 질의 추출
                    userQuery = '';
                    if (log.prompt) {
                        userQuery = log.prompt;
                    } else if (log.userQuery) {
                        userQuery = log.userQuery;
                    } else if (content.includes(userQueryPrefix)) {
                        const parts = content.split(`\n\n`);
                        userQuery = parts[0].replace(userQueryPrefix, '').trim();
                    } else if (typeof content === 'string' && content.includes('"prompt":')) {
                        try {
                            const jsonContent = JSON.parse(content);
                            userQuery = jsonContent.prompt || jsonContent.command || '';
                        } catch (e) {
                            // JSON 파싱 실패 무시
                        }
                    }
                    
                    // 코드 내용 추출
                    if (content.includes(codePrefix)) {
                        const parts = content.split(`\n\n${codePrefix}`);
                        codeContent = parts[1] ? parts[1].trim() : '';
                    }
                } catch (error) {
                    console.error('Error processing log content:', error);
                    content = getText('logs', 'errorProcessingContent');
                }
                
                logItem.innerHTML = `
                    <div class="log-header">
                        <div class="log-time">${timestamp}</div>
                        <span class="log-status">${statusText}</span>
                    </div>
                    <div class="log-content">
                        <div class="log-section">
                            <div class="section-title">${getText('logs', 'query')}:</div>
                            <div class="section-content">${userQuery ? userQuery : getText('logs', 'noQuery')}</div>
                        </div>
                        ${codeContent ? `
                            <div class="log-section">
                                <div class="section-title">${getText('logs', 'executedCode')}:</div>
                                <div class="section-content">${codeContent}</div>
                            </div>
                        ` : ''}
                        <div class="log-section">
                            <div class="section-title">${getText('logs', 'result')}:</div>
                            <div class="section-content">${
                                typeof log.result === 'string' ? log.result : 
                                typeof log.result === 'object' ? JSON.stringify(log.result, null, 2) :
                                getText('logs', 'noOutput')
                            }</div>
                        </div>
                    </div>`;
                
                logsList.appendChild(logItem);
            });

            pageInfo.textContent = `${currentPage} / ${totalPages}`;
            prevPageBtn.disabled = currentPage === 1;
            nextPageBtn.disabled = currentPage === totalPages;
            pagination.style.display = totalPages > 1 ? 'flex' : 'none';
        } catch (error) {
            console.error('Failed to load logs:', error);
            // error.message가 객체인 경우를 처리
            let errorMessage = error?.message;
            if (typeof errorMessage === 'object') {
                errorMessage = JSON.stringify(errorMessage);
            }
            logsList.innerHTML = `<div class="error-message">${getText('dialog', 'error')}: ${errorMessage || 'Unknown error'}</div>`;
            pagination.style.display = 'none';
        }
    }

    function createNoLogsMessage() {
        const noLogsDiv = document.createElement('div');
        noLogsDiv.className = 'no-logs';
        noLogsDiv.innerText = getText('logs', 'noLogs');
        return noLogsDiv;
    }

    // 페이지네이션 이벤트 핸들러
    document.getElementById('prevPage').addEventListener('click', () => {
        if (currentPage > 1) {
            currentPage--;
            refreshLogs();
        }
    });

    document.getElementById('nextPage').addEventListener('click', () => {
        const totalPages = Math.ceil(totalLogs.length / itemsPerPage);
        if (currentPage < totalPages) {
            currentPage++;
            refreshLogs();
        }
    });

    // 초기 로그 목록 로드
    await refreshLogs();
} 