import { requireAuth } from './auth.js';

// usagePage 함수를 export
export async function usagePage(chatMessages, translations) {
    // 사용자 인증 체크
    const hasAccess = await requireAuth(reqAPI);
    if (!hasAccess) {
        return;
    }

    // Chart.js 로드
    if (!window.Chart) {
        await new Promise((resolve, reject) => {
            // date-fns 먼저 로드
            const dateFnsScript = document.createElement('script');
            dateFnsScript.src = './static/index.min.js';
            dateFnsScript.onload = () => {
                // Chart.js 로드
                const chartScript = document.createElement('script');
                chartScript.src = './static/chart.umd.min.js';
                chartScript.onload = () => {
                    // date-fns 어댑터 로드
                    const adapterScript = document.createElement('script');
                    adapterScript.src = './static/chartjs-adapter-date-fns.bundle.min.js';
                    adapterScript.onload = resolve;
                    adapterScript.onerror = reject;
                    document.head.appendChild(adapterScript);
                };
                chartScript.onerror = reject;
                document.head.appendChild(chartScript);
            };
            dateFnsScript.onerror = reject;
            document.head.appendChild(dateFnsScript);
        });
    }

    // html2pdf.js 로드
    if (!window.html2pdf) {
        await new Promise((resolve, reject) => {
            const html2pdfScript = document.createElement('script');
            html2pdfScript.src = './static/html2pdf.bundle.min.js';
            html2pdfScript.onload = resolve;
            html2pdfScript.onerror = reject;
            document.head.appendChild(html2pdfScript);
        });
    }

    // CSS 파일 로드
    if (!document.querySelector('link[href="./pages/usage.css"]')) {
        const link = document.createElement('link');
        link.rel = 'stylesheet';
        link.href = './pages/usage.css';
        document.head.appendChild(link);
    }

    // 입력 컨테이너 숨기기
    const inputContainer = document.querySelector('.input-container');
    inputContainer.style.display = 'none';

    // 페이지 컨테이너 생성
    chatMessages.innerHTML = `
        <div class="page-container usage-page">
            <div class="page-header">
                <div class="header-content">
                    <h2 class="page-title">${translations?.usage?.title || 'AI Usage Statistics'}</h2>
                    <p class="page-description">${translations?.usage?.description || 'Monitor AI provider usage and costs'}</p>
                </div>
            </div>
            
            <!-- 전체 요약 섹션 -->
            <div class="usage-summary-section">
                <div class="summary-cards">
                    <div class="summary-card total-cost">
                        <h3>${translations?.usage?.totalCost || 'Total Cost'}</h3>
                        <div class="amount" id="totalCost">$0.00</div>
                        <div class="trend" id="costTrend"></div>
                    </div>
                    <div class="summary-card total-tokens">
                        <h3>${translations?.usage?.totalTokens || 'Total Tokens'}</h3>
                        <div class="amount" id="totalTokens">0</div>
                        <div class="trend" id="tokenTrend"></div>
                    </div>
                    <div class="summary-card monthly-cost">
                        <h3>${translations?.usage?.monthlyCost || 'Monthly Cost'}</h3>
                        <div class="amount" id="monthlyTotalCost">$0.00</div>
                        <div class="trend" id="monthlyCostTrend"></div>
                    </div>
                    <div class="summary-card monthly-tokens">
                        <h3>${translations?.usage?.monthlyTokens || 'Monthly Tokens'}</h3>
                        <div class="amount" id="monthlyTotalTokens">0</div>
                        <div class="trend" id="monthlyTokenTrend"></div>
                    </div>
                </div>
            </div>

            <!-- 제공자별 분석 섹션 -->
            <div class="provider-analysis-section">
                <h3>${translations?.usage?.providerAnalysis || 'Provider Analysis'}</h3>
                <div class="provider-analysis-container">
                    <div class="provider-cards" id="providerBreakdown">
                        <!-- 제공자별 카드가 여기에 동적으로 추가됨 -->
                    </div>
                    <!-- 제공자별 비용 분포 도넛 차트 추가 -->
                    <div class="provider-cost-distribution">
                        <h3>${translations?.usage?.costDistribution || 'Cost Distribution by Provider'}</h4>
                        <canvas id="providerCostChart"></canvas>
                    </div>
                </div>
            </div>

            <!-- 월별 추세 비교 -->
            <div class="monthly-trend-section">
                <h3>${translations?.usage?.monthlyTrend || 'Monthly Usage Trend'}</h3>
                <canvas id="monthlyTrendChart"></canvas>
            </div>

            <!-- 시간대별 사용량 히트맵 -->
            <div class="usage-heatmap-section">
                <h3>${translations?.usage?.timeDistribution || 'Usage Distribution by Time'}</h3>
                <div class="heatmap-container">
                    <canvas id="usageHeatmap"></canvas>
                </div>
            </div>

            <!-- AI 호출 상세 내역 -->
            <div class="recent-requests-section">
                <div class="section-header">
                    <h3>${translations?.usage?.callDetails || 'AI Call Details'}</h3>
                    <button class="download-btn" onclick="downloadCallDetails()">
                        ${translations?.usage?.csv?.downloadCallDetails || 'Download Call Details'}
                    </button>
                </div>
                <div class="table-container">
                    <table class="usage-table">
                        <thead>
                            <tr>
                                <th style="width: 15%">${translations?.usage?.timestamp || 'Timestamp'}</th>
                                <th style="width: 10%">${translations?.usage?.provider || 'Provider'}</th>
                                <th style="width: 25%" class="model-column">${translations?.usage?.model || 'Model'}</th>
                                <th style="width: 10%" class="right-align">${translations?.usage?.promptTokens || 'Request Tokens'}</th>
                                <th style="width: 10%" class="right-align">${translations?.usage?.completionTokens || 'Response Tokens'}</th>
                                <th style="width: 10%" class="right-align">${translations?.usage?.promptCost || 'Request Cost'}</th>
                                <th style="width: 10%" class="right-align">${translations?.usage?.completionCost || 'Response Cost'}</th>
                                <th style="width: 10%" class="right-align">${translations?.usage?.totalCost || 'Total Cost'}</th>
                            </tr>
                        </thead>
                        <tbody id="usageTableBody">
                            <!-- 사용 기록이 여기에 동적으로 추가됨 -->
                        </tbody>
                    </table>
                </div>
                <div class="pagination" id="usageTablePagination">
                    <!-- 페이지네이션 컨트롤이 여기에 추가됨 -->
                </div>
            </div>

            <!-- 세션별 사용량 -->
            <div class="prompt-history-section">
                <div class="section-header">
                    <h3>${translations?.usage?.promptHistory || 'Session Usage'}</h3>
                    <button class="download-btn" onclick="downloadSessionUsage()">
                        ${translations?.usage?.csv?.downloadSessionUsage || 'Download Session Usage'}
                    </button>
                </div>
                <div class="prompt-list" id="promptList">
                    <!-- 세션별 사용량이 동적으로 추가됨 -->
                </div>
                <div class="pagination" id="promptListPagination">
                    <!-- 페이지네이션 컨트롤이 여기에 추가됨 -->
                </div>
            </div>

            <!-- 세션별 사용량 분포 차트 추가 -->
            <div class="session-bubble-section">
                <h3>${translations?.usage?.sessionBubble || 'Usage Distribution by Session'}</h3>
                <div class="session-bubble-container">
                    <canvas id="sessionBubbleChart"></canvas>
                </div>
            </div>            

            <!-- 추가 시각화 섹션 -->
            <div class="additional-visualizations">
                <!-- 토큰 사용량 비교 -->
                <div class="token-comparison-section">
                    <h3>${translations?.usage?.tokenComparison || 'Token Usage Comparison by Model'}</h3>
                    <canvas id="tokenComparisonChart"></canvas>
                </div>

                <!-- 비용 효율성 분석 -->
                <div class="cost-efficiency-section">
                    <h3>${translations?.usage?.costEfficiency || 'Cost Efficiency by Model'}</h3>
                    <canvas id="costEfficiencyChart"></canvas>
                </div>
            </div>
        </div>
    `;

    try {
        const taskId = getUnique();
        const usageData = await getUsageData();
        await updateUsageDisplay(usageData, translations);
        
        // 세션별 사용량 업데이트
        const promptList = document.getElementById('promptList');
        
        // 세션별로 데이터 그룹화
        const sessionData = {};
        usageData.forEach(record => {
            const sessionId = record.sessionId || 'default';
            
            // 세션 데이터 초기화 (첫 번째 프롬프트가 있는 경우만)
            if (!sessionData[sessionId] && record.prompt && 
                !record.prompt.startsWith('I executed') && 
                !record.prompt.startsWith('\n')) {
                sessionData[sessionId] = {
                    timestamp: record.timestamp,
                    firstPrompt: record.prompt,
                    calls: 0,
                    totalTokens: 0,
                    providers: new Set()
                };
            }

            // 세션이 존재하는 경우 모든 레코드의 데이터를 누적
            if (sessionData[sessionId]) {
                sessionData[sessionId].calls++;
                sessionData[sessionId].totalTokens += (record.promptTokens || 0) + (record.completionTokens || 0);
                sessionData[sessionId].providers.add(record.provider);
            }
        });

        // 전역 변수에 세션 데이터 저장
        currentSessionData = sessionData;
        
        // 세션별 사용량 업데이트
        updatePromptList(usageData);
    } catch (error) {
        console.error('Error loading usage data:', error);
        showError(translations?.usage?.loadError || 'Failed to load usage data');
    }
}

// 데이터 로드 및 표시 함수
async function getUsageData() {
    const taskId = getUnique();
    return new Promise((resolve, reject) => {
        const timeout = setTimeout(() => {
            reject(new Error('Request timeout'));
        }, 10000);

        window.electronAPI.receive('response', (response) => {
            if (response.taskId === taskId) {
                clearTimeout(timeout);
                if (response.error) {
                    reject(new Error(response.error));
                } else {
                    resolve(response.arg);
                }
            }
        });
        
        window.electronAPI.send('request', {
            mode: 'getUsageData',
            taskId: taskId,
            arg: {}
        });
    });
}

// 전체 통계 업데이트
async function updateUsageDisplay(usageData, translations) {
    // console.log('Updating usage display with data:', usageData);
    if (!Array.isArray(usageData)) {
        console.error('Usage data is not an array:', usageData);
        return;
    }
    if (usageData.length === 0) {
        console.log('No usage data available');
        return;
    }

    try {
        // console.log('Updating total stats...');
        updateTotalStats(usageData);
        
        // console.log('Updating provider breakdown...');
        updateProviderBreakdown(usageData, translations);
        
        // console.log('Updating usage table...');
        await updateUsageTable(usageData, translations);
        
        // console.log('Updating provider cost chart...');
        updateProviderCostChart(usageData);
        
        // console.log('Updating usage heatmap...');
        updateUsageHeatmap(usageData);
        
        // console.log('Updating token comparison chart...');
        updateTokenComparisonChart(usageData);
        
        // console.log('Updating cost efficiency chart...');
        await updateCostEfficiencyChart(usageData);
        
        // console.log('Updating monthly trend chart...');
        updateMonthlyTrendChart(usageData);
        
        // 세션 버블 차트는 마지막에 업데이트
        // console.log('Updating session bubble chart...');
        if (Array.isArray(usageData) && usageData.length > 0) {
            updateSessionBubbleChart(usageData);
        }
        
        // console.log('All updates completed successfully');
    } catch (error) {
        console.error('Error in updateUsageDisplay:', error);
    }
}

// 총계 통계 업데이트
function updateTotalStats(usageData) {
    if (!Array.isArray(usageData) || usageData.length === 0) {
        document.getElementById('totalCost').textContent = '$0.00';
        document.getElementById('totalTokens').textContent = '0';
        document.getElementById('monthlyTotalCost').textContent = '$0.00';
        document.getElementById('monthlyTotalTokens').textContent = '0';
        return;
    }

    const totalCost = usageData.reduce((sum, record) => {
        // vllm 또는 ollama의 경우 무료로 처리
        if (record.provider === 'vllm' || record.provider === 'ollama') {
            return sum;
        }
        
        // 비용이 직접 지정된 경우
        if (record.cost) {
            return sum + parseFloat(record.cost);
        }
        
        // 토큰 수를 기반으로 비용 계산
        const promptTokens = parseInt(record.promptTokens) || 0;
        const completionTokens = parseInt(record.completionTokens) || 0;
        
        // 다른 제공자의 경우 기본 가격 적용 (예: $0.0001 per token)
        const cost = (promptTokens + completionTokens) * 0.0001;
        return sum + cost;
    }, 0);
    
    const totalTokens = usageData.reduce((sum, record) => {
        const promptTokens = parseInt(record.promptTokens) || 0;
        const completionTokens = parseInt(record.completionTokens) || 0;
        return sum + promptTokens + completionTokens;
    }, 0);
    
    document.getElementById('totalCost').textContent = `$${totalCost.toFixed(2)}`;
    document.getElementById('totalTokens').textContent = totalTokens.toLocaleString();
    
    // 월별 통계 계산
    const currentDate = new Date();
    const currentMonth = currentDate.getMonth();
    const currentYear = currentDate.getFullYear();
    
    const monthlyData = usageData.filter(record => {
        const recordDate = new Date(record.timestamp);
        return recordDate.getMonth() === currentMonth && 
               recordDate.getFullYear() === currentYear;
    });
    
    const monthlyTotalCost = monthlyData.reduce((sum, record) => {
        // vllm 또는 ollama의 경우 무료로 처리
        if (record.provider === 'vllm' || record.provider === 'ollama') {
            return sum;
        }
        
        // 비용이 직접 지정된 경우
        if (record.cost) {
            return sum + parseFloat(record.cost);
        }
        
        // 토큰 수를 기반으로 비용 계산
        const promptTokens = parseInt(record.promptTokens) || 0;
        const completionTokens = parseInt(record.completionTokens) || 0;
        
        // 다른 제공자의 경우 기본 가격 적용 (예: $0.0001 per token)
        const cost = (promptTokens + completionTokens) * 0.0001;
        return sum + cost;
    }, 0);
    
    const monthlyTotalTokens = monthlyData.reduce((sum, record) => {
        const promptTokens = parseInt(record.promptTokens) || 0;
        const completionTokens = parseInt(record.completionTokens) || 0;
        return sum + promptTokens + completionTokens;
    }, 0);
    
    document.getElementById('monthlyTotalCost').textContent = `$${monthlyTotalCost.toFixed(2)}`;
    document.getElementById('monthlyTotalTokens').textContent = monthlyTotalTokens.toLocaleString();
}

// 제공자별 분석 업데이트
function updateProviderBreakdown(usageData, translations) {
    if (!Array.isArray(usageData) || usageData.length === 0) {
        document.getElementById('providerBreakdown').innerHTML = `
            <div class="no-data-message">${translations?.usage?.noData || 'No usage data available'}</div>
        `;
        return;
    }

    const providerStats = {};
    usageData.forEach(record => {
        // provider가 없거나 빈 문자열인 경우에만 'Unknown'으로 처리
        const provider = (!record.provider || record.provider.trim() === '') ? 'Unknown' : record.provider;
        
        if (!providerStats[provider]) {
            providerStats[provider] = {
                cost: 0,
                promptTokens: 0,
                completionTokens: 0,
                requests: 0
            };
        }
        const stats = providerStats[provider];
        
        // vllm 또는 ollama의 경우 무료로 처리
        if (provider === 'vllm' || provider === 'ollama') {
            stats.cost += 0;
        } else {
            // 비용이 직접 지정된 경우
            if (record.cost) {
                stats.cost += parseFloat(record.cost);
            } else {
                // 토큰 수를 기반으로 비용 계산
                const promptTokens = parseInt(record.promptTokens) || 0;
                const completionTokens = parseInt(record.completionTokens) || 0;
                
                // 다른 제공자의 경우 기본 가격 적용 (예: $0.0001 per token)
                stats.cost += (promptTokens + completionTokens) * 0.0001;
            }
        }
        
        stats.promptTokens += parseInt(record.promptTokens) || 0;
        stats.completionTokens += parseInt(record.completionTokens) || 0;
        stats.requests++;
    });

    // 디버깅을 위한 로그 추가
    console.log('Provider Stats:', providerStats);

    const breakdownHtml = Object.entries(providerStats)
        .map(([provider, stats]) => `
            <div class="provider-card">
                <h4>${provider}</h4>
                <div class="provider-stats-list">
                    <div class="stat-row">
                        <span class="stat-label">${translations?.usage?.charts?.providerStats?.cost || 'Cost:'}</span>
                        <span class="stat-value">$${stats.cost.toFixed(2)}</span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">${translations?.usage?.charts?.providerStats?.requestTokens || 'Request Tokens:'}</span>
                        <span class="stat-value">${stats.promptTokens.toLocaleString()}</span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">${translations?.usage?.charts?.providerStats?.responseTokens || 'Response Tokens:'}</span>
                        <span class="stat-value">${stats.completionTokens.toLocaleString()}</span>
                    </div>
                    <div class="stat-row">
                        <span class="stat-label">${translations?.usage?.charts?.providerStats?.requests || 'Requests:'}</span>
                        <span class="stat-value">${stats.requests}</span>
                    </div>
                </div>
            </div>
        `).join('');

    document.getElementById('providerBreakdown').innerHTML = breakdownHtml;
}

// 유니크 ID 생성 함수
function getUnique() {
    return Date.now().toString(36) + Math.random().toString(36).substr(2);
}

// 오류 메시지 표시 함수
function showError(message) {
    const container = document.querySelector('.page-container');
    if (container) {
        container.innerHTML = `
            <div class="error-message">
                <i class="fas fa-exclamation-circle"></i>
                <p>${message}</p>
            </div>
        `;
    }
}

// 페이지네이션 상태 변수
let currentTablePage = 1;
let currentPromptPage = 1;
const itemsPerPage = 20;
let currentUsageData = null;
let currentTranslations = null;
let currentSessionData = null;

// 페이지 변경 핸들러 함수들
window.handleTablePageChange = function(page) {
    currentTablePage = page;
    updateUsageTable(currentUsageData, currentTranslations);
};

window.handlePromptPageChange = function(page) {
    currentPromptPage = page;
    updatePromptList(currentSessionData);
};

// CSV 다운로드 함수들
window.downloadCallDetails = async function() {
    if (!currentUsageData) return;

    const notification = createProgressNotification(translations?.usage?.generatingCSV || 'CSV 파일 생성 중...');
    
    try {
        const headers = [
            translations?.usage?.csv?.timestamp || 'Timestamp',
            translations?.usage?.csv?.provider || 'Provider',
            translations?.usage?.csv?.model || 'Model',
            translations?.usage?.csv?.requestTokens || 'Request Tokens',
            translations?.usage?.csv?.responseTokens || 'Response Tokens',
            translations?.usage?.csv?.requestCost || 'Request Cost',
            translations?.usage?.csv?.responseCost || 'Response Cost',
            translations?.usage?.csv?.totalCost || 'Total Cost'
        ];

        const csvRows = [headers.join(',')];
        const batchSize = 1000;
        const totalRecords = currentUsageData.length;
        
        for (let i = 0; i < totalRecords; i += batchSize) {
            const batch = currentUsageData.slice(i, i + batchSize);
            const batchPromises = batch.map(async record => {
                const price = await getPriceForModel(record.provider, record.model);
                const promptCost = (record.promptTokens / 1000) * price.prompt;
                const completionCost = (record.completionTokens / 1000) * price.completion;
                const totalCost = promptCost + completionCost;

                return [
                    new Date(record.timestamp).toLocaleString(),
                    record.provider,
                    record.model,
                    record.promptTokens,
                    record.completionTokens,
                    promptCost.toFixed(4),
                    completionCost.toFixed(4),
                    totalCost.toFixed(4)
                ].map(value => `"${value}"`).join(',');
            });

            const batchRows = await Promise.all(batchPromises);
            csvRows.push(...batchRows);
            await new Promise(resolve => setTimeout(resolve, 0));
        }

        const csvContent = '\ufeff' + csvRows.join('\n');
        downloadCSVFile(csvContent, 'ai_call_details.csv');
        showSuccessNotification(notification);
    } catch (error) {
        console.error('Error generating CSV:', error);
        showErrorNotification(notification);
    }
};

window.downloadSessionUsage = async function() {
    if (!currentSessionData) return;

    const notification = createProgressNotification(translations?.usage?.generatingCSV || 'CSV 파일 생성 중...');
    
    try {
        const headers = [
            translations?.usage?.csv?.timestamp || 'Timestamp',
            translations?.usage?.csv?.firstPrompt || 'First Prompt',
            translations?.usage?.csv?.totalCalls || 'Total Calls',
            translations?.usage?.csv?.totalTokens || 'Total Tokens',
            translations?.usage?.csv?.providers || 'Providers'
        ];

        const csvRows = [headers.join(',')];
        const sessionEntries = Object.entries(currentSessionData);
        const batchSize = 100;
        const totalSessions = sessionEntries.length;

        for (let i = 0; i < totalSessions; i += batchSize) {
            const batch = sessionEntries.slice(i, i + batchSize);
            const batchRows = batch.map(([sessionId, data]) => [
                formatTimestamp(data.timestamp),
                `"${(data.firstPrompt || '').replace(/"/g, '""')}"`,
                data.calls,
                data.totalTokens,
                `"${Array.from(data.providers).join(', ')}"`
            ].join(','));

            csvRows.push(...batchRows);
            await new Promise(resolve => setTimeout(resolve, 0));
        }

        const csvContent = '\ufeff' + csvRows.join('\n');
        downloadCSVFile(csvContent, 'session_usage.csv');
        showSuccessNotification(notification);
    } catch (error) {
        console.error('Error generating CSV:', error);
        showErrorNotification(notification);
    }
};

// 알림 관련 유틸리티 함수들
function createProgressNotification(message) {
    const notification = document.createElement('div');
    notification.className = 'pdf-notification';
    notification.textContent = message;
    document.body.appendChild(notification);
    return notification;
}

function showSuccessNotification(notification) {
    notification.textContent = translations?.usage?.csvDownloadComplete || 'CSV 파일이 다운로드되었습니다';
    setTimeout(() => {
        notification.style.transform = 'translateX(120%)';
        setTimeout(() => notification.remove(), 300);
    }, 2000);
}

function showErrorNotification(notification) {
    notification.style.backgroundColor = 'var(--error-color)';
    notification.textContent = translations?.usage?.csvGenerationFailed || 'CSV 파일 생성에 실패했습니다';
    setTimeout(() => {
        notification.style.transform = 'translateX(120%)';
        setTimeout(() => notification.remove(), 300);
    }, 3000);
}

function downloadCSVFile(content, filename) {
    const blob = new Blob([content], { type: 'text/csv;charset=utf-8;' });
    const url = URL.createObjectURL(blob);
    const link = document.createElement('a');
    link.setAttribute('href', url);
    link.setAttribute('download', filename);
    link.style.visibility = 'hidden';
    document.body.appendChild(link);
    link.click();
    document.body.removeChild(link);
    URL.revokeObjectURL(url);
}

// 모델별 가격 정보 조회
async function getPriceForModel(provider, model) {
    try {
        const taskId = getUnique();
        return new Promise((resolve, reject) => {
            const timeout = setTimeout(() => {
                reject(new Error('Request timeout'));
            }, 10000);

            window.electronAPI.receive('response', (response) => {
                if (response.taskId === taskId) {
                    clearTimeout(timeout);
                    if (response.error) {
                        reject(new Error(response.error));
                    } else {
                        const prices = response.arg;
                        const modelPrices = prices[provider]?.[model];
                        resolve(modelPrices || { prompt: 0, completion: 0 });
                    }
                }
            });
            
            window.electronAPI.send('request', {
                mode: 'loadPrice',
                taskId: taskId,
                arg: {}
            });
        });
    } catch (error) {
        console.error('Error getting price data:', error);
        return { prompt: 0, completion: 0 };
    }
}

// 사용 내역 테이블 업데이트 함수 수정
async function updateUsageTable(usageData, translations) {
    currentUsageData = usageData;
    currentTranslations = translations;
    
    const tableBody = document.getElementById('usageTableBody');
    const paginationContainer = document.getElementById('usageTablePagination');
    
    // 전체 페이지 수 계산
    const totalPages = Math.ceil(usageData.length / itemsPerPage);
    
    // 현재 페이지의 데이터만 선택
    const startIndex = (currentTablePage - 1) * itemsPerPage;
    const endIndex = startIndex + itemsPerPage;
    const pageData = usageData
        .sort((a, b) => new Date(b.timestamp) - new Date(a.timestamp))
        .slice(startIndex, endIndex);

    // 테이블 행 생성
    const tableRows = await Promise.all(pageData.map(async record => {
        const price = await getPriceForModel(record.provider, record.model);
        const promptCost = (record.promptTokens / 1000) * price.prompt;
        const completionCost = (record.completionTokens / 1000) * price.completion;
        const totalCost = promptCost + completionCost;

        return `
            <tr>
                <td>${formatTimestamp(record.timestamp)}</td>
                <td>${record.provider}</td>
                <td class="model-column" title="${record.model}">${record.model}</td>
                <td class="right-align">${record.promptTokens.toLocaleString()}</td>
                <td class="right-align">${record.completionTokens.toLocaleString()}</td>
                <td class="right-align">$${promptCost.toFixed(4)}</td>
                <td class="right-align">$${completionCost.toFixed(4)}</td>
                <td class="right-align">$${totalCost.toFixed(4)}</td>
            </tr>
        `;
    }));

    tableBody.innerHTML = tableRows.join('');
    
    // 페이지네이션 컨트롤 업데이트
    updatePagination(paginationContainer, totalPages, currentTablePage, 'handleTablePageChange');
}

// 시간 포맷팅 함수 추가
function formatTimestamp(timestamp) {
    const date = new Date(timestamp);
    const year = date.getFullYear();
    const month = String(date.getMonth() + 1).padStart(2, '0');
    const day = String(date.getDate()).padStart(2, '0');
    const hours = String(date.getHours()).padStart(2, '0');
    const minutes = String(date.getMinutes()).padStart(2, '0');
    
    return `${year}-${month}-${day} ${hours}:${minutes}`;
}

// 세션별 사용량 업데이트 함수 수정
function updatePromptList(usageData) {
    try {
        const promptList = document.getElementById('promptList');
        const paginationContainer = document.getElementById('promptListPagination');
        
        if (!promptList || !paginationContainer) {
            console.error('Prompt list or pagination container not found');
            return;
        }

        // 세션 데이터가 없는 경우 처리
        if (!currentSessionData || Object.keys(currentSessionData).length === 0) {
            promptList.innerHTML = `<div class="no-data-message">${translations?.usage?.noData || 'No session data available'}</div>`;
            paginationContainer.innerHTML = '';
            return;
        }

        // 세션 데이터를 배열로 변환
        const sessionEntries = Object.entries(currentSessionData)
            .sort(([, a], [, b]) => new Date(b.timestamp) - new Date(a.timestamp));
        
        // 전체 페이지 수 계산
        const totalPages = Math.ceil(sessionEntries.length / itemsPerPage);
        
        // 현재 페이지의 데이터만 선택
        const startIndex = (currentPromptPage - 1) * itemsPerPage;
        const endIndex = startIndex + itemsPerPage;
        const pageData = sessionEntries.slice(startIndex, endIndex);
        
        // 프롬프트 리스트 HTML 생성
        promptList.innerHTML = pageData
            .map(([sessionId, data]) => `
                <div class="prompt-item">
                    <div class="prompt-header">
                        <div class="prompt-timestamp">${formatTimestamp(data.timestamp)}</div>
                        <div class="prompt-stats">
                            <span class="stat-item">${translations?.usage?.charts?.sessionStats?.calls || 'Calls'}: ${data.calls}</span>
                            <span class="stat-item">${translations?.usage?.charts?.sessionStats?.totalTokens || 'Total Tokens'}: ${data.totalTokens.toLocaleString()}</span>
                            <span class="stat-item">${translations?.usage?.charts?.sessionStats?.providers || 'Providers'}: ${Array.from(data.providers).join(', ')}</span>
                        </div>
                    </div>
                    <div class="prompt-content">${data.firstPrompt}</div>
                </div>
            `).join('');
        
        // 페이지네이션 컨트롤 업데이트
        updatePagination(paginationContainer, totalPages, currentPromptPage, 'handlePromptPageChange');
    } catch (error) {
        console.error('Error updating prompt list:', error);
        showError(translations?.usage?.updateError || 'Failed to update prompt list');
    }
}

// 페이지네이션 컨트롤 업데이트 함수
function updatePagination(container, totalPages, currentPage, handlerName) {
    const maxVisiblePages = 5;
    let pagesHtml = '';
    
    // 이전 페이지 버튼
    pagesHtml += `
        <button 
            ${currentPage === 1 ? 'disabled' : ''} 
            onclick="window.${handlerName}(${currentPage - 1})">
            ${translations?.usage?.pagination?.previous || 'Previous'}
        </button>
    `;
    
    // 페이지 번호 버튼
    let startPage = Math.max(1, currentPage - Math.floor(maxVisiblePages / 2));
    let endPage = Math.min(totalPages, startPage + maxVisiblePages - 1);
    
    if (endPage - startPage + 1 < maxVisiblePages) {
        startPage = Math.max(1, endPage - maxVisiblePages + 1);
    }
    
    for (let i = startPage; i <= endPage; i++) {
        pagesHtml += `
            <button 
                class="${i === currentPage ? 'active' : ''}"
                onclick="window.${handlerName}(${i})">
                ${i}
            </button>
        `;
    }
    
    // 다음 페이지 버튼
    pagesHtml += `
        <button 
            ${currentPage === totalPages ? 'disabled' : ''} 
            onclick="window.${handlerName}(${currentPage + 1})">
            ${translations?.usage?.pagination?.next || 'Next'}
        </button>
    `;
    
    // 페이지 정보 표시
    pagesHtml += `
        <span class="page-info">
            ${currentPage} / ${totalPages} ${translations?.usage?.pagination?.page || 'Page'}
        </span>
    `;

    // 컨테이너에 HTML 설정
    container.innerHTML = pagesHtml;
}

// 제공자별 비용 분포 도넛 차트 업데이트
function updateProviderCostChart(usageData) {
    try {
        // 데이터 유효성 검사
        if (!Array.isArray(usageData) || usageData.length === 0) {
            const canvas = document.getElementById('providerCostChart');
            const ctx = canvas.getContext('2d');
            
            // 기존 차트 제거
            const existingChart = Chart.getChart(canvas);
            if (existingChart) {
                existingChart.destroy();
            }

            // 데이터 없음 메시지 표시
            new Chart(ctx, {
                type: 'doughnut',
                data: {
                    labels: [],
                    datasets: [{
                        data: []
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: true,
                    aspectRatio: 1
                }
            });

            // 에러 메시지 표시
            const container = canvas.parentElement;
            let errorMsg = container.querySelector('.error-message');
            if (!errorMsg) {
                errorMsg = document.createElement('div');
                errorMsg.className = 'error-message';
                errorMsg.style.position = 'absolute';
                errorMsg.style.top = '50%';
                errorMsg.style.left = '50%';
                errorMsg.style.transform = 'translate(-50%, -50%)';
                errorMsg.style.textAlign = 'center';
                errorMsg.style.color = '#666';
                container.style.position = 'relative';
                container.appendChild(errorMsg);
            }
            errorMsg.textContent = translations?.usage?.charts?.noData || 'No usage data available';
            return;
        }

        // 기존 에러 메시지 제거
        const container = document.getElementById('providerCostChart').parentElement;
        const errorMsg = container.querySelector('.error-message');
        if (errorMsg) {
            errorMsg.remove();
        }

        const providerCosts = {};
        usageData.forEach(record => {
            if (!providerCosts[record.provider]) {
                providerCosts[record.provider] = 0;
            }
            providerCosts[record.provider] += record.cost;
        });

        // 데이터가 없는 경우 처리
        if (Object.keys(providerCosts).length === 0) {
            throw new Error('No provider cost data available');
        }

        const canvas = document.getElementById('providerCostChart');
        const ctx = canvas.getContext('2d');

        const existingChart = Chart.getChart(canvas);
        if (existingChart) {
            existingChart.destroy();
        }

        new Chart(ctx, {
            type: 'doughnut',
            data: {
                labels: Object.keys(providerCosts),
                datasets: [{
                    data: Object.values(providerCosts),
                    backgroundColor: [
                        'rgb(255, 99, 132)',
                        'rgb(54, 162, 235)',
                        'rgb(255, 206, 86)',
                        'rgb(75, 192, 192)',
                        'rgb(153, 102, 255)'
                    ]
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: true,
                aspectRatio: 1,
                layout: {
                    padding: {
                        top: 20,
                        bottom: 40,
                        left: 20,
                        right: 20
                    }
                },
                plugins: {
                    legend: {
                        position: 'bottom',
                        labels: {
                            padding: 20,
                            usePointStyle: true,
                            pointStyle: 'circle',
                            boxWidth: 10,
                            font: {
                                size: 12
                            }
                        },
                        maxWidth: 120,
                        align: 'start'
                    },
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const value = context.raw;
                                const total = context.dataset.data.reduce((a, b) => a + b, 0);
                                const percentage = ((value / total) * 100).toFixed(1);
                                return `${context.label}: $${value.toFixed(2)} (${percentage}%)`;
                            }
                        }
                    }
                }
            }
        });
    } catch (error) {
        console.error('Error updating provider cost chart:', error);
        showError(translations?.usage?.charts?.updateError || 'Failed to update provider cost chart');
    }
}

// 시간대별 사용량 히트맵 업데이트
function updateUsageHeatmap(usageData) {
    try {
        const canvas = document.getElementById('usageHeatmap');
        if (!canvas) {
            console.error('Usage heatmap canvas not found');
            return;
        }

        // 데이터 유효성 검사
        if (!Array.isArray(usageData) || usageData.length === 0) {
            const ctx = canvas.getContext('2d');
            
            // 기존 차트 제거
            const existingChart = Chart.getChart(canvas);
            if (existingChart) {
                existingChart.destroy();
            }

            // 데이터 없음 메시지 표시
            new Chart(ctx, {
                type: 'scatter',
                data: {
                    datasets: [{
                        data: []
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        x: {
                            display: false
                        },
                        y: {
                            display: false
                        }
                    }
                }
            });

            // 에러 메시지 표시
            const container = canvas.parentElement;
            if (container) {
                let errorMsg = container.querySelector('.error-message');
                if (!errorMsg) {
                    errorMsg = document.createElement('div');
                    errorMsg.className = 'error-message';
                    errorMsg.style.position = 'absolute';
                    errorMsg.style.top = '50%';
                    errorMsg.style.left = '50%';
                    errorMsg.style.transform = 'translate(-50%, -50%)';
                    errorMsg.style.textAlign = 'center';
                    errorMsg.style.color = '#666';
                    container.style.position = 'relative';
                    container.appendChild(errorMsg);
                }
                errorMsg.textContent = translations?.usage?.charts?.noData || 'No usage data available';
            }
            return;
        }

        // 요일/시간별 호출 횟수와 발생 일자 수를 저장할 배열
        const heatmapData = Array(7).fill().map(() => Array(24).fill(0));
        const dayCount = Array(7).fill().map(() => Array(24).fill().map(() => new Map()));
        
        const days = [
            translations?.usage?.days?.sun || 'Sun',
            translations?.usage?.days?.mon || 'Mon',
            translations?.usage?.days?.tue || 'Tue',
            translations?.usage?.days?.wed || 'Wed',
            translations?.usage?.days?.thu || 'Thu',
            translations?.usage?.days?.fri || 'Fri',
            translations?.usage?.days?.sat || 'Sat'
        ];

        // 데이터 누적 및 일자 수 계산
        usageData.forEach(record => {
            const date = new Date(record.timestamp);
            const day = date.getDay();
            const hour = date.getHours();
            
            heatmapData[day][hour]++;
            // 같은 날짜의 동일 시간대는 한 번만 카운트
            const dateKey = date.toDateString();
            if (!dayCount[day][hour].has(dateKey)) {
                dayCount[day][hour].set(dateKey, true);
            }
        });

        const ctx = canvas.getContext('2d');
        const existingChart = Chart.getChart(canvas);
        if (existingChart) {
            existingChart.destroy();
        }

        // 히트맵 데이터 변환 (평균 계산)
        const data = [];
        let maxValue = 0;
        
        // 평균값 계산 및 최대값 찾기
        heatmapData.forEach((dayData, dayIndex) => {
            dayData.forEach((value, hour) => {
                if (value > 0) {
                    const count = dayCount[dayIndex][hour].size;
                    const average = count > 0 ? value / count : 0;
                    if (average > 0) {
                        data.push({
                            x: hour,
                            y: dayIndex,
                            v: average,
                            total: value,
                            days: count
                        });
                        maxValue = Math.max(maxValue, average);
                    }
                }
            });
        });

        new Chart(ctx, {
            type: 'scatter',
            data: {
                datasets: [{
                    data: data,
                    backgroundColor: function(context) {
                        const value = context.raw.v;
                        const alpha = Math.max(0.2, value / maxValue);
                        return `rgba(75, 192, 192, ${alpha})`;
                    },
                    pointRadius: 15,
                    pointHoverRadius: 20
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                plugins: {
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const v = context.raw;
                                return [
                                    `${days[v.y]} ${v.x}${translations?.usage?.charts?.timeUnit || 'h'}`,
                                    `평균: ${v.v.toFixed(1)} ${translations?.usage?.charts?.requestCount || 'Calls'}`,
                                    `총 ${v.total}회 / ${v.days}일`
                                ];
                            }
                        }
                    },
                    legend: {
                        display: false
                    }
                },
                scales: {
                    x: {
                        type: 'linear',
                        min: 0,
                        max: 23,
                        ticks: {
                            stepSize: 1,
                            callback: value => `${value}${translations?.usage?.charts?.timeUnit || 'h'}`
                        },
                        grid: {
                            display: false
                        }
                    },
                    y: {
                        type: 'linear',
                        min: -0.5,
                        max: 6.5,
                        ticks: {
                            stepSize: 1,
                            callback: value => days[value]
                        },
                        grid: {
                            display: false
                        }
                    }
                }
            }
        });
    } catch (error) {
        console.error('Error updating usage heatmap:', error);
        showError(translations?.usage?.charts?.updateError || 'Failed to update usage heatmap');
    }
}

// 토큰 사용량 비교 차트 업데이트
function updateTokenComparisonChart(usageData) {
    try {
        const canvas = document.getElementById('tokenComparisonChart');
        if (!canvas) {
            console.error('Token comparison chart canvas not found');
            return;
        }

        // 데이터 유효성 검사
        if (!Array.isArray(usageData) || usageData.length === 0) {
            const ctx = canvas.getContext('2d');
            
            // 기존 차트 제거
            const existingChart = Chart.getChart(canvas);
            if (existingChart) {
                existingChart.destroy();
            }

            // 데이터 없음 메시지 표시
            new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: [],
                    datasets: [{
                        data: []
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        x: {
                            display: false
                        },
                        y: {
                            display: false
                        }
                    }
                }
            });

            // 에러 메시지 표시
            const container = canvas.parentElement;
            if (container) {
                let errorMsg = container.querySelector('.error-message');
                if (!errorMsg) {
                    errorMsg = document.createElement('div');
                    errorMsg.className = 'error-message';
                    errorMsg.style.position = 'absolute';
                    errorMsg.style.top = '50%';
                    errorMsg.style.left = '50%';
                    errorMsg.style.transform = 'translate(-50%, -50%)';
                    errorMsg.style.textAlign = 'center';
                    errorMsg.style.color = '#666';
                    container.style.position = 'relative';
                    container.appendChild(errorMsg);
                }
                errorMsg.textContent = translations?.usage?.charts?.noData || 'No usage data available';
            }
            return;
        }

        const modelStats = {};
        usageData.forEach(record => {
            const modelKey = `${record.provider}-${record.model}`;
            if (!modelStats[modelKey]) {
                modelStats[modelKey] = {
                    promptTokens: 0,
                    completionTokens: 0
                };
            }
            modelStats[modelKey].promptTokens += record.promptTokens || 0;
            modelStats[modelKey].completionTokens += record.completionTokens || 0;
        });

        // 데이터가 없는 경우 처리
        if (Object.keys(modelStats).length === 0) {
            throw new Error('No token comparison data available');
        }

        const ctx = canvas.getContext('2d');
        const existingChart = Chart.getChart(canvas);
        if (existingChart) {
            existingChart.destroy();
        }

        new Chart(ctx, {
            type: 'bar',
            data: {
                labels: Object.keys(modelStats),
                datasets: [
                    {
                        label: translations?.usage?.charts?.tokenComparison?.requestTokens || 'Request Tokens',
                        data: Object.values(modelStats).map(stat => stat.promptTokens),
                        backgroundColor: 'rgba(255, 99, 132, 0.8)'
                    },
                    {
                        label: translations?.usage?.charts?.tokenComparison?.responseTokens || 'Response Tokens',
                        data: Object.values(modelStats).map(stat => stat.completionTokens),
                        backgroundColor: 'rgba(54, 162, 235, 0.8)'
                    }
                ]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    x: {
                        stacked: true
                    },
                    y: {
                        stacked: true,
                        beginAtZero: true
                    }
                },
                plugins: {
                    legend: {
                        position: 'bottom',
                        labels: {
                            padding: 20,
                            usePointStyle: true,
                            pointStyle: 'circle',
                            boxWidth: 10,
                            font: {
                                size: 12
                            }
                        }
                    }
                }
            }
        });
    } catch (error) {
        console.error('Error updating token comparison chart:', error);
        showError(translations?.usage?.charts?.updateError || 'Failed to update token comparison chart');
    }
}

// 비용 효율성 분석 차트 업데이트
async function updateCostEfficiencyChart(usageData) {
    try {
        const canvas = document.getElementById('costEfficiencyChart');
        if (!canvas) {
            console.error('Cost efficiency chart canvas not found');
            return;
        }

        // 데이터 유효성 검사
        if (!Array.isArray(usageData) || usageData.length === 0) {
            const ctx = canvas.getContext('2d');
            
            // 기존 차트 제거
            const existingChart = Chart.getChart(canvas);
            if (existingChart) {
                existingChart.destroy();
            }

            // 데이터 없음 메시지 표시
            new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: [],
                    datasets: [{
                        data: []
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        x: {
                            display: false
                        },
                        y: {
                            display: false
                        }
                    }
                }
            });

            // 에러 메시지 표시
            const container = canvas.parentElement;
            if (container) {
                let errorMsg = container.querySelector('.error-message');
                if (!errorMsg) {
                    errorMsg = document.createElement('div');
                    errorMsg.className = 'error-message';
                    errorMsg.style.position = 'absolute';
                    errorMsg.style.top = '50%';
                    errorMsg.style.left = '50%';
                    errorMsg.style.transform = 'translate(-50%, -50%)';
                    errorMsg.style.textAlign = 'center';
                    errorMsg.style.color = '#666';
                    container.style.position = 'relative';
                    container.appendChild(errorMsg);
                }
                errorMsg.textContent = translations?.usage?.charts?.noData || 'No usage data available';
            }
            return;
        }

        const modelStats = {};
        for (const record of usageData) {
            const modelKey = `${record.provider}-${record.model}`;
            if (!modelStats[modelKey]) {
                modelStats[modelKey] = {
                    totalTokens: 0,
                    totalCost: 0
                };
            }
            const totalTokens = (record.promptTokens || 0) + (record.completionTokens || 0);
            modelStats[modelKey].totalTokens += totalTokens;
            modelStats[modelKey].totalCost += record.cost || 0;
        }

        // 데이터가 없는 경우 처리
        if (Object.keys(modelStats).length === 0) {
            throw new Error('No cost efficiency data available');
        }

        const costPerToken = Object.entries(modelStats)
            .filter(([_, stats]) => stats.totalTokens > 0)  // 토큰이 0인 경우 제외
            .map(([model, stats]) => ({
                model,
                efficiency: stats.totalCost / stats.totalTokens * 1000 // Cost per 1000 tokens
            }));

        const ctx = canvas.getContext('2d');
        const existingChart = Chart.getChart(canvas);
        if (existingChart) {
            existingChart.destroy();
        }

        new Chart(ctx, {
            type: 'bar',
            data: {
                labels: costPerToken.map(item => item.model),
                datasets: [{
                    label: translations?.usage?.charts?.costEfficiency?.costPerThousandTokens || 'Cost per 1000 Tokens',
                    data: costPerToken.map(item => item.efficiency),
                    backgroundColor: 'rgba(75, 192, 192, 0.8)'
                }]
            },
            options: {
                responsive: true,
                scales: {
                    y: {
                        beginAtZero: true,
                        title: {
                            display: true,
                            text: translations?.usage?.charts?.costEfficiency?.yAxisLabel || 'Cost (USD/1000 Tokens)'
                        }
                    }
                }
            }
        });
    } catch (error) {
        console.error('Error updating cost efficiency chart:', error);
        showError(translations?.usage?.charts?.updateError || 'Failed to update cost efficiency chart');
    }
}

// 월별 추세 비교 차트 업데이트
function updateMonthlyTrendChart(usageData) {
    try {
        const canvas = document.getElementById('monthlyTrendChart');
        if (!canvas) {
            console.error('Monthly trend chart canvas not found');
            return;
        }

        // 데이터 유효성 검사
        if (!Array.isArray(usageData) || usageData.length === 0) {
            const ctx = canvas.getContext('2d');
            
            // 기존 차트 제거
            const existingChart = Chart.getChart(canvas);
            if (existingChart) {
                existingChart.destroy();
            }

            // 데이터 없음 메시지 표시
            new Chart(ctx, {
                type: 'bar',
                data: {
                    labels: [],
                    datasets: [{
                        data: []
                    }]
                },
                options: {
                    responsive: true,
                    maintainAspectRatio: false,
                    plugins: {
                        legend: {
                            display: false
                        }
                    },
                    scales: {
                        x: {
                            display: false
                        },
                        y: {
                            display: false
                        }
                    }
                }
            });

            // 에러 메시지 표시
            const container = canvas.parentElement;
            if (container) {
                let errorMsg = container.querySelector('.error-message');
                if (!errorMsg) {
                    errorMsg = document.createElement('div');
                    errorMsg.className = 'error-message';
                    errorMsg.style.position = 'absolute';
                    errorMsg.style.top = '50%';
                    errorMsg.style.left = '50%';
                    errorMsg.style.transform = 'translate(-50%, -50%)';
                    errorMsg.style.textAlign = 'center';
                    errorMsg.style.color = '#666';
                    container.style.position = 'relative';
                    container.appendChild(errorMsg);
                }
                errorMsg.textContent = translations?.usage?.charts?.noData || 'No usage data available';
            }
            return;
        }

        const monthlyStats = {};
        
        usageData.forEach(record => {
            const date = new Date(record.timestamp);
            const monthKey = `${date.getFullYear()}-${String(date.getMonth() + 1).padStart(2, '0')}`;
            
            if (!monthlyStats[monthKey]) {
                monthlyStats[monthKey] = {
                    cost: 0,
                    tokens: 0
                };
            }
            
            monthlyStats[monthKey].cost += record.cost || 0;
            monthlyStats[monthKey].tokens += (record.promptTokens || 0) + (record.completionTokens || 0);
        });

        const ctx = canvas.getContext('2d');
        const existingChart = Chart.getChart(canvas);
        if (existingChart) {
            existingChart.destroy();
        }

        const months = Object.keys(monthlyStats).sort();

        new Chart(ctx, {
            type: 'bar',
            data: {
                labels: months,
                datasets: [
                    {
                        label: translations?.usage?.charts?.monthlyTrend?.cost || 'Cost',
                        data: months.map(month => monthlyStats[month].cost),
                        yAxisID: 'cost',
                        type: 'line',
                        borderColor: 'rgb(255, 99, 132)',
                        backgroundColor: 'rgba(255, 99, 132, 0.5)'
                    },
                    {
                        label: translations?.usage?.charts?.monthlyTrend?.tokens || 'Tokens',
                        data: months.map(month => monthlyStats[month].tokens),
                        yAxisID: 'tokens',
                        backgroundColor: 'rgba(54, 162, 235, 0.5)'
                    }
                ]
            },
            options: {
                responsive: true,
                scales: {
                    cost: {
                        type: 'linear',
                        position: 'left',
                        title: {
                            display: true,
                            text: translations?.usage?.charts?.monthlyTrend?.costAxis || 'Cost (USD)'
                        }
                    },
                    tokens: {
                        type: 'linear',
                        position: 'right',
                        title: {
                            display: true,
                            text: translations?.usage?.charts?.monthlyTrend?.tokensAxis || 'Tokens'
                        }
                    }
                },
                plugins: {
                    tooltip: {
                        callbacks: {
                            label: function(context) {
                                const value = context.raw;
                                if (context.dataset.yAxisID === 'cost') {
                                    return `${context.dataset.label}: $${value.toFixed(2)}`;
                                } else {
                                    return `${context.dataset.label}: ${value.toLocaleString()} tokens`;
                                }
                            }
                        }
                    }
                }
            }
        });
    } catch (error) {
        console.error('Error updating monthly trend chart:', error);
        showError(translations?.usage?.charts?.updateError || 'Failed to update monthly trend chart');
    }
}

// 세션 버블 차트 업데이트
function updateSessionBubbleChart(usageData) {
    try {
        const canvas = document.getElementById('sessionBubbleChart');
        if (!canvas) {
            console.error('Session bubble chart canvas not found');
            return;
        }

        // 데이터 유효성 검사
        if (!usageData || !Array.isArray(usageData)) {
            console.error('Invalid usage data:', usageData);
            const ctx = canvas.getContext('2d');
            const existingChart = Chart.getChart(canvas);
            if (existingChart) {
                existingChart.destroy();
            }

            // 데이터 없음 메시지 표시
            const container = canvas.parentElement;
            if (container) {
                let errorMsg = container.querySelector('.error-message');
                if (!errorMsg) {
                    errorMsg = document.createElement('div');
                    errorMsg.className = 'error-message';
                    errorMsg.style.position = 'absolute';
                    errorMsg.style.top = '50%';
                    errorMsg.style.left = '50%';
                    errorMsg.style.transform = 'translate(-50%, -50%)';
                    errorMsg.style.textAlign = 'center';
                    errorMsg.style.color = '#666';
                    container.style.position = 'relative';
                    container.appendChild(errorMsg);
                }
                errorMsg.textContent = translations?.usage?.charts?.noData || 'No usage data available';
            }
            return;
        }

        // 오늘 날짜의 시작과 끝 시간 설정
        const today = new Date();
        today.setHours(0, 0, 0, 0);
        const tomorrow = new Date(today);
        tomorrow.setDate(tomorrow.getDate() + 1);

        // 오늘 데이터만 필터링
        const todayData = usageData.filter(record => {
            const recordDate = new Date(record.timestamp);
            return recordDate >= today && recordDate < tomorrow;
        });

        // 세션별로 데이터 그룹화
        const sessionData = {};
        todayData.forEach(record => {
            const sessionId = record.sessionId || 'default';
            if (!sessionData[sessionId]) {
                sessionData[sessionId] = {
                    timestamp: record.timestamp,
                    calls: 0,
                    totalTokens: 0,
                    providers: new Set()
                };
            }
            sessionData[sessionId].calls++;
            sessionData[sessionId].totalTokens += (record.promptTokens || 0) + (record.completionTokens || 0);
            sessionData[sessionId].providers.add(record.provider);
        });

        const bubbleData = Object.entries(sessionData).map(([sessionId, data]) => ({
            x: new Date(data.timestamp).getTime(),
            y: data.calls,
            r: Math.sqrt(data.calls) * 3,
            calls: data.calls,
            tokens: data.totalTokens,
            providers: Array.from(data.providers).join(', ')
        }));

        const ctx = canvas.getContext('2d');
        const existingChart = Chart.getChart(canvas);
        if (existingChart) {
            existingChart.destroy();
        }

        new Chart(ctx, {
            type: 'bubble',
            data: {
                datasets: [{
                    label: 'Session',
                    data: bubbleData,
                    backgroundColor: 'rgba(75, 192, 192, 0.6)',
                    borderColor: 'rgba(75, 192, 192, 1)',
                    borderWidth: 1,
                    hoverBackgroundColor: 'rgba(75, 192, 192, 0.8)',
                    hoverBorderColor: 'rgba(75, 192, 192, 1)',
                    hoverBorderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    x: {
                        type: 'linear',
                        position: 'bottom',
                        min: today.getTime(),
                        max: tomorrow.getTime(),
                        ticks: {
                            stepSize: 3600000, // 1시간 간격
                            callback: function(value) {
                                const hour = new Date(value).getHours();
                                return `${hour}${translations?.usage?.charts?.timeUnit || 'h'}`;
                            }
                        },
                        title: {
                            display: true,
                            text: translations?.usage?.charts?.sessionBubble?.time || 'Time'
                        }
                    },
                    y: {
                        title: {
                            display: true,
                            text: translations?.usage?.charts?.sessionBubble?.calls || 'Number of Calls'
                        },
                        beginAtZero: true
                    }
                },
                plugins: {
                    tooltip: {
                        callbacks: {
                            title: function(context) {
                                const date = new Date(context[0].raw.x);
                                const hour = date.getHours();
                                return `${hour}${translations?.usage?.charts?.timeUnit || 'h'}`;
                            },
                            label: function(context) {
                                const data = context.raw;
                                return [
                                    `${translations?.usage?.charts?.sessionStats?.calls || 'Calls'}: ${data.calls}`,
                                    `${translations?.usage?.charts?.sessionStats?.totalTokens || 'Total Tokens'}: ${data.tokens.toLocaleString()}`,
                                    `${translations?.usage?.charts?.sessionStats?.providers || 'Providers'}: ${data.providers}`
                                ];
                            }
                        }
                    },
                    legend: {
                        display: false
                    }
                }
            }
        });
    } catch (error) {
        console.error('Error updating session bubble chart:', error);
        showError(translations?.usage?.charts?.updateError || 'Failed to update session bubble chart');
    }
}

// PDF 다운로드 함수
window.downloadPDF = async function() {
    const element = document.querySelector('.page-container');
    
    // PDF 생성 시작을 알림
    const notification = document.createElement('div');
    notification.className = 'pdf-notification';
    notification.textContent = translations?.usage?.generatingPDF || 'Generating PDF...';
    document.body.appendChild(notification);

    try {
        // 모든 차트 업데이트 함수 재실행
        const usageData = await getUsageData();
        await updateProviderCostChart(usageData);
        await updateUsageHeatmap(usageData);
        await updateTokenComparisonChart(usageData);
        await updateCostEfficiencyChart(usageData);
        await updateMonthlyTrendChart(usageData);
        await updateSessionBubbleChart(usageData);

        // 차트가 완전히 렌더링될 때까지 대기
        await new Promise(resolve => setTimeout(resolve, 1000));

        // PDF 생성 옵션
        const opt = {
            margin: 15,
            filename: `${translations?.usage?.aiUsageStats || 'AI_Usage_Stats'}_${new Date().toLocaleDateString()}.pdf`,
            image: { type: 'png', quality: 1.0 },
            html2canvas: { 
                scale: 2,
                useCORS: true,
                logging: true,
                backgroundColor: '#ffffff',
                windowWidth: 1200,
                onclone: function(clonedDoc) {
                    const clonedElement = clonedDoc.querySelector('.page-container');
                    if (clonedElement) {
                        clonedElement.style.width = '1200px';
                        clonedElement.style.margin = '0 auto';
                        clonedElement.style.backgroundColor = '#ffffff';
                        clonedElement.style.padding = '20px';
                        
                        // 클론된 문서의 모든 차트 캔버스를 이미지로 변환
                        const canvases = clonedElement.querySelectorAll('canvas');
                        canvases.forEach(canvas => {
                            const originalCanvas = document.querySelector(`canvas#${canvas.id}`);
                            if (originalCanvas) {
                                const img = document.createElement('img');
                                img.src = originalCanvas.toDataURL('image/png', 1.0);
                                img.style.width = '100%';
                                img.style.height = 'auto';
                                img.style.maxHeight = '400px';
                                canvas.parentNode.replaceChild(img, canvas);
                            }
                        });
                    }
                }
            },
            jsPDF: { 
                unit: 'mm', 
                format: 'a4', 
                orientation: 'portrait',
                compress: true
            }
        };

        // PDF 생성
        await html2pdf().set(opt).from(element).save();

        notification.textContent = translations?.usage?.pdfDownloadComplete || 'PDF Download Complete';
        setTimeout(() => notification.remove(), 2000);
    } catch (error) {
        console.error('PDF Generation Error:', error);
        notification.textContent = translations?.usage?.pdfGenerationFailed || 'PDF Generation Failed';
        notification.style.backgroundColor = 'var(--error-color)';
        setTimeout(() => notification.remove(), 3000);
    }
};