#!/usr/bin/env python3
# -*- coding: utf-8 -*-

import os
import sys
import argparse
from pdf2image import convert_from_path
import pytesseract
from PIL import Image
import tempfile
import shutil

def extract_text_from_pdf(pdf_path, output_file=None, lang='kor'):
    """
    PDF 파일에서 텍스트를 추출합니다.
    
    Args:
        pdf_path (str): PDF 파일 경로
        output_file (str, optional): 추출된 텍스트를 저장할 파일 경로
        lang (str, optional): OCR 언어 (기본값: 'kor' - 한국어)
    
    Returns:
        str: 추출된 텍스트
    """
    print(f"PDF 파일 '{pdf_path}'에서 텍스트 추출 중...")
    
    # 임시 디렉토리 생성
    temp_dir = tempfile.mkdtemp()
    try:
        # PDF를 이미지로 변환
        images = convert_from_path(pdf_path)
        
        full_text = ""
        
        # 각 페이지에서 텍스트 추출
        for i, image in enumerate(images):
            print(f"페이지 {i+1}/{len(images)} 처리 중...")
            
            # 이미지에서 텍스트 추출
            text = pytesseract.image_to_string(image, lang=lang)
            full_text += f"\n\n--- 페이지 {i+1} ---\n\n"
            full_text += text
        
        # 결과 출력
        if output_file:
            with open(output_file, 'w', encoding='utf-8') as f:
                f.write(full_text)
            print(f"추출된 텍스트가 '{output_file}'에 저장되었습니다.")
        
        return full_text
    
    finally:
        # 임시 디렉토리 삭제
        shutil.rmtree(temp_dir)

def process_directory(input_dir, output_dir=None, lang='kor'):
    """
    디렉토리 내의 모든 PDF 파일을 처리합니다.
    
    Args:
        input_dir (str): 입력 디렉토리 경로
        output_dir (str, optional): 출력 디렉토리 경로 (None인 경우 입력 디렉토리와 동일)
        lang (str, optional): OCR 언어 (기본값: 'kor' - 한국어)
    """
    if not os.path.isdir(input_dir):
        print(f"오류: '{input_dir}'는 유효한 디렉토리가 아닙니다.")
        return
    
    # 출력 디렉토리가 지정되지 않은 경우 입력 디렉토리와 동일하게 설정
    if output_dir is None:
        output_dir = input_dir
    
    # 출력 디렉토리가 존재하지 않으면 생성
    if not os.path.exists(output_dir):
        os.makedirs(output_dir)
    
    # PDF 파일 목록 가져오기
    pdf_files = [f for f in os.listdir(input_dir) if f.lower().endswith('.pdf')]
    
    if not pdf_files:
        print(f"경고: '{input_dir}' 디렉토리에 PDF 파일이 없습니다.")
        return
    
    print(f"{len(pdf_files)}개의 PDF 파일을 처리합니다...")
    
    for pdf_file in pdf_files:
        input_path = os.path.join(input_dir, pdf_file)
        output_path = os.path.join(output_dir, os.path.splitext(pdf_file)[0] + '.txt')
        
        extract_text_from_pdf(input_path, output_path, lang)

def main():
    parser = argparse.ArgumentParser(description='PDF 이미지에서 한국어 텍스트를 추출합니다.')
    parser.add_argument('input', help='입력 PDF 파일 또는 디렉토리 경로')
    parser.add_argument('-o', '--output', help='출력 텍스트 파일 또는 디렉토리 경로')
    parser.add_argument('-l', '--lang', default='kor', help='OCR 언어 (기본값: kor)')
    
    args = parser.parse_args()
    
    # 입력이 디렉토리인 경우
    if os.path.isdir(args.input):
        # -o 옵션이 없는 경우 입력 디렉토리와 동일한 경로에 저장
        if args.output is None:
            process_directory(args.input, args.input, args.lang)
        else:
            process_directory(args.input, args.output, args.lang)
    # 입력이 파일인 경우
    elif os.path.isfile(args.input) and args.input.lower().endswith('.pdf'):
        # -o 옵션이 없는 경우 같은 경로에 같은 파일명.txt로 저장
        if args.output is None:
            output_file = os.path.splitext(args.input)[0] + '.txt'
            extract_text_from_pdf(args.input, output_file, args.lang)
        else:
            extract_text_from_pdf(args.input, args.output, args.lang)
    else:
        print(f"오류: '{args.input}'는 유효한 PDF 파일 또는 디렉토리가 아닙니다.")

if __name__ == "__main__":
    if len(sys.argv) == 1:
        # 인수가 없으면 대화형 모드로 실행
        print("PDF 이미지에서 한국어 텍스트 추출 도구")
        print("1. 단일 PDF 파일 처리")
        print("2. 디렉토리 내 모든 PDF 파일 처리")
        
        mode = input("모드 선택 (1 또는 2): ")
        
        if mode == "1":
            # 단일 파일 처리
            pdf_path = input("PDF 파일 경로를 입력하세요: ")
            output_path = input("출력 텍스트 파일 경로를 입력하세요 (빈칸으로 두면 입력 파일과 같은 경로에 .txt로 저장): ")
            lang = input("OCR 언어를 입력하세요 (기본값: kor): ") or "kor"
            
            if not output_path:
                # 출력 경로가 지정되지 않은 경우 입력 파일과 같은 경로에 .txt로 저장
                output_path = os.path.splitext(pdf_path)[0] + '.txt'
            
            if os.path.isfile(pdf_path) and pdf_path.lower().endswith('.pdf'):
                text = extract_text_from_pdf(pdf_path, output_path, lang)
                print(f"추출된 텍스트가 '{output_path}'에 저장되었습니다.")
            else:
                print(f"오류: '{pdf_path}'는 유효한 PDF 파일이 아닙니다.")
        
        elif mode == "2":
            # 디렉토리 처리
            input_dir = input("PDF 파일이 있는 디렉토리 경로를 입력하세요: ")
            output_dir = input("출력 텍스트 파일을 저장할 디렉토리 경로를 입력하세요 (빈칸으로 두면 입력 디렉토리와 동일): ")
            lang = input("OCR 언어를 입력하세요 (기본값: kor): ") or "kor"
            
            if not output_dir:
                # 출력 디렉토리가 지정되지 않은 경우 입력 디렉토리와 동일하게 설정
                output_dir = input_dir
            
            process_directory(input_dir, output_dir, lang)
        
        else:
            print("오류: 잘못된 모드를 선택했습니다.")
    else:
        main() 