// 도구 로더 - 시스템 시작 시 도구들을 자동으로 로드
import toolRegistry from './tools-registry.js';
import builtinTools from './builtin-tools.js';
import path from 'path';
import { fileURLToPath } from 'url';

const __filename = fileURLToPath(import.meta.url);
const __dirname = path.dirname(__filename);

class ToolLoader {
    constructor() {
        this.loaded = false;
        this.loadedTools = new Set();
    }

    /**
     * 모든 도구 로드 (시스템 초기화 시 호출)
     */
    async loadAllTools() {
        if (this.loaded) {
            // console.log('[ToolLoader] 도구들이 이미 로드되었습니다.');
            return;
        }

        // console.log('[ToolLoader] 도구 로딩 시작...');
        
        try {
            // 1. 기본 내장 도구들 로드
            await this.loadBuiltinTools();
            
            // 2. 커스텀 도구 디렉토리 로드
            await this.loadCustomTools();
            
            // 3. 사용자 정의 도구 디렉토리 로드 (선택적)
            await this.loadUserTools();
            
            this.loaded = true;
            // console.log(`[ToolLoader] 총 ${this.loadedTools.size}개 도구 로드 완료`);
            
            // 로드된 도구 목록 출력
            this.printLoadedTools();
            
        } catch (error) {
            // console.error('[ToolLoader] 도구 로딩 중 오류:', error);
            throw error;
        }
    }

    /**
     * 기본 내장 도구들 로드
     */
    async loadBuiltinTools() {
        // console.log('[ToolLoader] 기본 내장 도구 로드 중...');
        
        try {
            let loadedCount = 0;
            for (const tool of builtinTools) {
                try {
                    toolRegistry.registerTool(tool);
                    this.loadedTools.add(tool.name);
                    loadedCount++;
                } catch (error) {
                    // console.error(`기본 도구 로드 실패 ${tool.name}:`, error.message);
                }
            }
            
            // console.log(`[ToolLoader] 기본 도구 ${loadedCount}개 로드됨`);
        } catch (error) {
            // console.error('[ToolLoader] 기본 도구 로드 실패:', error);
        }
    }

    /**
     * 커스텀 도구들 로드
     */
    async loadCustomTools() {
        const customToolsDir = path.join(__dirname, 'custom-tools');
        
        try {
            const loaded = await toolRegistry.loadToolsFromDirectory(customToolsDir);
            // console.log(`[ToolLoader] 커스텀 도구 ${loaded}개 로드됨`);
            
            // 로드된 도구 이름들을 추적
            const tools = toolRegistry.getTools();
            tools.forEach(toolName => this.loadedTools.add(toolName));
            
        } catch (error) {
            // console.warn('[ToolLoader] 커스텀 도구 로드 실패 (디렉토리가 없거나 오류):', error.message);
        }
    }

    /**
     * 사용자 정의 도구들 로드
     */
    async loadUserTools() {
        const userToolsDir = path.join(process.cwd(), 'user-tools');
        
        try {
            const loaded = await toolRegistry.loadToolsFromDirectory(userToolsDir);
            if (loaded > 0) {
                // console.log(`[ToolLoader] 사용자 도구 ${loaded}개 로드됨`);
                
                // 로드된 도구 이름들을 추적
                const tools = toolRegistry.getTools();
                tools.forEach(toolName => this.loadedTools.add(toolName));
            }
        } catch (error) {
            // 사용자 도구 디렉토리는 선택사항이므로 경고만 출력
            // console.log('[ToolLoader] 사용자 도구 디렉토리 없음 (선택사항)');
        }
    }

    /**
     * 특정 파일에서 도구 로드
     */
    async loadToolsFromFile(filePath) {
        try {
            const loaded = await toolRegistry.loadToolsFromFile(filePath);
            // console.log(`[ToolLoader] ${filePath}에서 ${loaded}개 도구 로드됨`);
            
            // 새로 로드된 도구들을 추적
            const tools = toolRegistry.getTools();
            tools.forEach(toolName => this.loadedTools.add(toolName));
            
            return loaded;
        } catch (error) {
            // console.error(`[ToolLoader] 파일 로드 실패 ${filePath}:`, error);
            throw error;
        }
    }

    /**
     * 도구 다시 로드
     */
    async reloadTools() {
        // console.log('[ToolLoader] 도구 다시 로드 중...');
        
        // 기존 도구들 제거 (기본 도구 제외)
        const currentTools = Array.from(this.loadedTools);
        for (const toolName of currentTools) {
            const toolInfo = toolRegistry.getToolInfo(toolName);
            if (toolInfo && toolInfo.category !== 'builtin') {
                toolRegistry.unregisterTool(toolName);
                this.loadedTools.delete(toolName);
            }
        }
        
        this.loaded = false;
        await this.loadAllTools();
    }

    /**
     * 로드된 도구 목록 출력
     */
    printLoadedTools() {
        // console.log('\n=== 로드된 도구 목록 ===');
        const toolsByCategory = toolRegistry.getToolsByCategory();
        
        for (const [category, tools] of Object.entries(toolsByCategory)) {
            // console.log(`\n📁 ${category.toUpperCase()}:`);
            tools.forEach(tool => {
                // console.log(`  • ${tool.name} (v${tool.version}) - ${tool.description}`);
            });
        }
        // console.log('\n=========================\n');
    }

    /**
     * 도구 통계 반환
     */
    getToolStats() {
        const toolsByCategory = toolRegistry.getToolsByCategory();
        const stats = {
            total: this.loadedTools.size,
            categories: Object.keys(toolsByCategory).length,
            byCategory: {}
        };
        
        for (const [category, tools] of Object.entries(toolsByCategory)) {
            stats.byCategory[category] = tools.length;
        }
        
        return stats;
    }

    /**
     * 도구 검색
     */
    searchTools(query) {
        const allTools = toolRegistry.getTools();
        const results = [];
        
        for (const toolName of allTools) {
            const toolInfo = toolRegistry.getToolInfo(toolName);
            if (toolInfo) {
                const searchText = `${toolInfo.name} ${toolInfo.description} ${toolInfo.category}`.toLowerCase();
                if (searchText.includes(query.toLowerCase())) {
                    results.push(toolInfo);
                }
            }
        }
        
        return results;
    }
}

// 글로벌 도구 로더 인스턴스
const toolLoader = new ToolLoader();

export { ToolLoader, toolLoader };
export default toolLoader; 