"use client";
import { useEffect, useState } from "react";
import type { User } from "../../../types/user";
import { internalApiClient } from "@/lib/apiClient";
import { getAuthHeaders } from '@/utils/api';
import UserTable from "./UserTable";
import UserForm from "./UserForm";
import UserStats from "./UserStats";

const PAGE_SIZE_OPTIONS = [10, 20, 30, 50];

export default function AdminUsersPage() {
  const [users, setUsers] = useState<User[]>([]);
  const [loading, setLoading] = useState(true);
  const [error, setError] = useState<string | null>(null);
  const [showForm, setShowForm] = useState(false);
  const [editUser, setEditUser] = useState<User | null>(null);
  const [formLoading, setFormLoading] = useState(false);
  const [deleteUser, setDeleteUser] = useState<User | null>(null);
  const [deleteLoading, setDeleteLoading] = useState(false);
  const [showUserStats, setShowUserStats] = useState<User | null>(null);

  // 시스템 프롬프트 관련 상태
  const [showSystemPromptModal, setShowSystemPromptModal] = useState(false);
  const [editSystemPromptUser, setEditSystemPromptUser] = useState<User | null>(null);
  const [systemPrompt, setSystemPrompt] = useState<string>('');
  const [isLoadingSystemPrompt, setIsLoadingSystemPrompt] = useState(false);
  const [systemPromptProvider, setSystemPromptProvider] = useState<string>('');
  const [isSystemPromptCustom, setIsSystemPromptCustom] = useState(false);
  // 페이지네이션/검색 상태
  const [page, setPage] = useState(1);
  const [pageSize, setPageSize] = useState(10);
  const [search, setSearch] = useState("");
  const [total, setTotal] = useState(0);

  // 사용자 목록 불러오기
  const fetchUsers = async () => {
    setLoading(true);
    setError(null);
    const res = await internalApiClient.getUsers({ search, page, pageSize });
    if (res.success) {
      setUsers(res.data?.users || []);
      setTotal(res.data?.total || 0);
    } else {
      setUsers([]);
      setTotal(0);
      setError(res.error?.message || "사용자 목록을 불러오지 못했습니다.");
    }
    setLoading(false);
  };

  useEffect(() => {
    fetchUsers();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [search, page, pageSize]);

  // 검색 입력 변경 시 1페이지로 이동
  const handleSearchChange = (e: React.ChangeEvent<HTMLInputElement>) => {
    setSearch(e.target.value);
    setPage(1);
  };
  // 페이지네이션 핸들러
  const handlePageChange = (newPage: number) => {
    setPage(newPage);
  };
  const handlePageSizeChange = (e: React.ChangeEvent<HTMLSelectElement>) => {
    setPageSize(Number(e.target.value));
    setPage(1);
  };

  // 사용자 추가/수정
  const handleFormSubmit = async (data: Partial<User> & { password?: string }) => {
    setFormLoading(true);
    setError(null);
    try {
      if (editUser) {
        // 수정
        const cleanData = {
          name: data.name,
          email: data.email,
          role: data.role,
          status: data.status,
          password: data.password,
        };
        const res = await internalApiClient.updateUser(String(editUser.id), cleanData);
        if (!res.success) {
          setError(res.error?.message || '수정 실패');
          return;
        }
      } else {
        // 추가
        const cleanData = {
          username: data.username,
          password: data.password,
          name: data.name,
          role: data.role,
          status: data.status,
          email: data.email,
        };
        const res = await internalApiClient.createUser(cleanData);
        
        if (!res.success) {
          const dbDetail = res.error?.details?.detail;
          // 이메일 중복 시도일 경우 따로 메시지 처리
          if (dbDetail && dbDetail.includes('already exists')) {
            setError('이미 존재하는 이메일입니다.');
          } else {
            setError(res.error?.message || '사용자 생성에 실패했습니다.');
          }
        
          return;
        }
      }
      setShowForm(false);
      setEditUser(null);
      fetchUsers();
    } finally {
      setFormLoading(false);
    }
  };

  // 사용자 삭제
  const handleDelete = async () => {
    if (!deleteUser) return;
    setDeleteLoading(true);
    try {
      const res = await internalApiClient.deleteUser(String(deleteUser.id));
      if (!res.success) throw new Error(res.error?.message);
      setDeleteUser(null);
      fetchUsers();
    } catch (err) {
      setError((err as Error).message);
    } finally {
      setDeleteLoading(false);
    }
  };

  // 시스템 프롬프트 불러오기 (기존 API 활용)
  const fetchUserSystemPrompt = async (user: User) => {
    try {
      setIsLoadingSystemPrompt(true);
      const authHeaders = getAuthHeaders();
      const { getApiServerUrl } = await import('@/config/serverConfig');
      const apiServerUrl = getApiServerUrl();

      console.log(`관리자가 사용자 ${user.username} (ID: ${user.id})의 시스템 프롬프트를 조회 중...`);

      // 관리자로서 특정 사용자의 시스템 프롬프트를 조회
      const response = await fetch(`${apiServerUrl}/user/system-prompt?mode=chat&userId=${user.id}`, {
        method: 'GET',
        headers: {
          ...authHeaders
        },
        credentials: 'include'
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success && data.data) {
          setSystemPrompt(data.data.content || '');
          setSystemPromptProvider(data.data.provider || '');
          setIsSystemPromptCustom(data.data.isCustom || false);
        } else {
          // 사용자별 프롬프트가 없으면 빈 상태로 설정
          setSystemPrompt('');
          setSystemPromptProvider('');
          setIsSystemPromptCustom(false);
        }
      } else {
        console.error('시스템 프롬프트 불러오기 실패:', response.status);
        setSystemPrompt('');
        setSystemPromptProvider('');
        setIsSystemPromptCustom(false);
      }
    } catch (error) {
      console.error('시스템 프롬프트 불러오기 실패:', error);
      setSystemPrompt('');
      setSystemPromptProvider('');
      setIsSystemPromptCustom(false);
    } finally {
      setIsLoadingSystemPrompt(false);
    }
  };

  // 시스템 프롬프트 저장 (기존 API 활용)
  const saveUserSystemPrompt = async () => {
    if (!editSystemPromptUser) return;

    try {
      setIsLoadingSystemPrompt(true);
      const authHeaders = getAuthHeaders();
      const { getApiServerUrl } = await import('@/config/serverConfig');
      const apiServerUrl = getApiServerUrl();

      console.log(`관리자가 사용자 ${editSystemPromptUser.username} (ID: ${editSystemPromptUser.id})의 시스템 프롬프트를 저장 중...`);

      // 관리자로서 특정 사용자의 시스템 프롬프트를 저장
      const response = await fetch(`${apiServerUrl}/user/system-prompt?userId=${editSystemPromptUser.id}`, {
        method: 'PUT',
        headers: {
          'Content-Type': 'application/json',
          ...authHeaders
        },
        credentials: 'include',
        body: JSON.stringify({ content: systemPrompt })
      });

      if (response.ok) {
        const data = await response.json();
        if (data.success) {
          setShowSystemPromptModal(false);
          setEditSystemPromptUser(null);
          setError(null);
        } else {
          throw new Error(data.error?.message || '시스템 프롬프트 저장 실패');
        }
      } else {
        throw new Error('서버 응답 오류');
      }
    } catch (error) {
      console.error('시스템 프롬프트 저장 실패:', error);
      setError(error instanceof Error ? error.message : '시스템 프롬프트 저장에 실패했습니다.');
    } finally {
      setIsLoadingSystemPrompt(false);
    }
  };

  // 시스템 프롬프트 편집 시작
  const handleEditSystemPrompt = async (user: User) => {
    setEditSystemPromptUser(user);
    setShowSystemPromptModal(true);
    await fetchUserSystemPrompt(user);
  };

  return (
    <div className="w-[98vw] max-w-[1200px] mx-auto p-4">
      <div className="flex justify-between items-center mb-4">
        <h2 className="text-base font-bold">사용자 관리</h2>
        <button
          className="px-4 py-2 bg-blue-600 text-white rounded hover:bg-blue-700"
          onClick={() => {
            setEditUser(null);
            setShowForm(true);
          }}
        >
          사용자 추가
        </button>
      </div>
      
      {/* 페이지 설명 */}
      <div className="mb-6 p-4 bg-blue-50 dark:bg-blue-900/20 rounded-lg border border-blue-200 dark:border-blue-700">
        <div className="flex items-start gap-3">
          <div className="w-5 h-5 rounded-full bg-blue-500 flex items-center justify-center flex-shrink-0 mt-0.5">
            <svg className="w-3 h-3 text-white" fill="currentColor" viewBox="0 0 16 16">
              <path d="M8 16A8 8 0 1 0 8 0a8 8 0 0 0 0 16zm.93-9.412-1 4.705c-.07.34.029.533.304.533.194 0 .487-.07.686-.246l-.088.416c-.287.346-.92.598-1.465.598-.703 0-1.002-.422-.808-1.319l.738-3.468c.064-.293.006-.399-.287-.47l-.451-.081.082-.381 2.29-.287zM8 5.5a1 1 0 1 1 0-2 1 1 0 0 1 0 2z"/>
            </svg>
          </div>
          <div>
            <p className="text-sm font-semibold text-blue-800 dark:text-blue-200 mb-1">
              사용자를 관리하고 사용자별 통계 기능을 제공합니다.
            </p>
            <p className="text-sm text-blue-700 dark:text-blue-300">
              사용자 목록에서 <span className="font-medium text-blue-600 dark:text-blue-400">사용자 아이디</span>를 클릭하면 해당 사용자의 <span className="font-medium">AI 사용통계</span>, <span className="font-medium">API 사용통계</span>, <span className="font-medium">최근 활동</span>을 확인할 수 있습니다.
            </p>
          </div>
        </div>
      </div>
      
      {/* 검색 */}
      <div className="mb-4 flex gap-2 items-center">
        <input
          type="text"
          placeholder="아이디, 이름, 이메일 검색"
          value={search}
          onChange={handleSearchChange}
          className="border border-gray-300 dark:border-gray-600 rounded px-3 py-2 w-64 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition"
        />
        <select value={pageSize} onChange={handlePageSizeChange} className="border border-gray-300 dark:border-gray-600 rounded px-2 py-1 bg-white dark:bg-gray-800 text-gray-900 dark:text-gray-100 transition">
          {PAGE_SIZE_OPTIONS.map((n) => (
            <option key={n} value={n}>{n}개씩</option>
          ))}
        </select>
        <span className="ml-2 text-sm text-gray-500">총 {total}명</span>
      </div>
      {loading ? (
        <div>로딩 중...</div>
      ) : (
        <>
          <UserTable
            users={users}
            onEdit={(user) => {
              setEditUser(user);
              setShowForm(true);
            }}
            onDelete={(user) => setDeleteUser(user)}
            onRegenApiKey={fetchUsers}
            onShowStats={(user) => setShowUserStats(user)}
            onEditSystemPrompt={handleEditSystemPrompt}
          />
          {/* 페이지네이션 */}
          <div className="flex justify-center items-center gap-2 mt-4">
            <button
              className="px-2 py-1 border rounded"
              onClick={() => handlePageChange(page - 1)}
              disabled={page === 1}
            >
              이전
            </button>
            <span>
              {page} / {Math.max(1, Math.ceil(total / pageSize))}
            </span>
            <button
              className="px-2 py-1 border rounded"
              onClick={() => handlePageChange(page + 1)}
              disabled={page >= Math.ceil(total / pageSize)}
            >
              다음
            </button>
          </div>
        </>
      )}
      {/* 사용자 추가/수정 폼 모달 */}
      {showForm && (
        <div className="fixed inset-0 bg-black bg-opacity-30 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow-lg p-6 w-full max-w-md">
            <h2 className="text-lg font-semibold mb-4">{editUser ? "사용자 수정" : "사용자 추가"}</h2>
            <UserForm
              initialData={editUser || {}}
              onSubmit={handleFormSubmit}
              onCancel={() => {
                setShowForm(false);
                setEditUser(null);
                setError(null);
              }}
              loading={formLoading}
            />
            {error && (
              <div className="mt-4 px-4 py-2 bg-red-100 border border-red-400 text-red-700 font-semibold rounded text-center">
                {error}
              </div>
            )}
          </div>
        </div>
      )}
      {/* 삭제 확인 다이얼로그 */}
      {deleteUser && (
        <div className="fixed inset-0 bg-black bg-opacity-30 flex items-center justify-center z-50">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow-lg p-6 w-full max-w-sm">
            <h2 className="text-lg font-semibold mb-4">사용자 삭제</h2>
            <p className="mb-4">정말로 <b>{deleteUser.username}</b> 사용자를 삭제하시겠습니까?</p>
            <div className="flex gap-2 justify-end">
              <button
                className="px-4 py-2 bg-gray-200 rounded"
                onClick={() => setDeleteUser(null)}
                disabled={deleteLoading}
              >
                취소
              </button>
              <button
                className="px-4 py-2 bg-red-600 text-white rounded"
                onClick={handleDelete}
                disabled={deleteLoading}
              >
                삭제
              </button>
            </div>
          </div>
        </div>
      )}
      {/* 사용자 통계 - 테이블 아래 표시 */}
      {showUserStats && (
        <UserStats
          userId={String(showUserStats.id)}
          username={showUserStats.username}
          onClose={() => setShowUserStats(null)}
        />
      )}

      {/* 시스템 프롬프트 편집 모달 */}
      {showSystemPromptModal && editSystemPromptUser && (
        <div className="fixed inset-0 bg-black bg-opacity-50 flex items-center justify-center z-50 p-4">
          <div className="bg-white dark:bg-gray-800 rounded-lg shadow-xl w-full max-w-6xl max-h-[90vh] overflow-hidden">
            {/* 모달 헤더 */}
            <div className="flex items-center justify-between p-6 border-b border-gray-200 dark:border-gray-700">
              <div>
                <h2 className="text-xl font-semibold text-gray-900 dark:text-white">
                  시스템 프롬프트 편집 - {editSystemPromptUser.username}
                </h2>
                {systemPromptProvider && (
                  <p className="text-sm text-gray-600 dark:text-gray-400 mt-1">
                    프로바이더: {systemPromptProvider.toUpperCase()}
                    {isSystemPromptCustom ? ' (사용자 정의)' : ' (기본 템플릿)'}
                  </p>
                )}
              </div>
              <button
                onClick={() => {
                  setShowSystemPromptModal(false);
                  setEditSystemPromptUser(null);
                  setError(null);
                }}
                className="text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors"
              >
                <svg className="w-6 h-6" fill="none" stroke="currentColor" viewBox="0 0 24 24">
                  <path strokeLinecap="round" strokeLinejoin="round" strokeWidth={2} d="M6 18L18 6M6 6l12 12" />
                </svg>
              </button>
            </div>

            {/* 모달 내용 */}
            <div className="p-6 overflow-y-auto max-h-[70vh]">
              {isLoadingSystemPrompt ? (
                <div className="flex items-center justify-center py-12">
                  <div className="animate-spin rounded-full h-8 w-8 border-b-2 border-blue-600"></div>
                  <span className="ml-3 text-gray-600 dark:text-gray-400">로딩 중...</span>
                </div>
              ) : (
                <div className="space-y-4">
                  <div>
                    <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                      프롬프트 내용
                    </label>
                    <textarea
                      value={systemPrompt}
                      onChange={(e) => setSystemPrompt(e.target.value)}
                      className="w-full h-96 px-4 py-3 text-sm border border-gray-300 dark:border-gray-600 rounded-lg bg-white dark:bg-gray-800 text-gray-900 dark:text-white focus:ring-2 focus:ring-blue-500 focus:border-transparent resize-none font-mono"
                      placeholder="시스템 프롬프트를 입력하세요..."
                    />
                  </div>

                  {!isSystemPromptCustom && (
                    <div className="bg-blue-50 dark:bg-blue-900/20 border border-blue-200 dark:border-blue-800 rounded-lg p-4">
                      <div className="flex items-start">
                        <div className="flex-shrink-0">
                          <svg className="w-5 h-5 text-blue-400" fill="currentColor" viewBox="0 0 20 20">
                            <path fillRule="evenodd" d="M18 10a8 8 0 100-16 8 8 0 000 16zm-7-4a1 1 0 11-2 0 1 1 0 012 0zM9 9a1 1 0 000 2v3a1 1 0 001 1h1a1 1 0 100-2v-3a1 1 0 00-1-1H9z" clipRule="evenodd" />
                          </svg>
                        </div>
                        <div className="ml-3">
                          <h3 className="text-sm font-medium text-blue-800 dark:text-blue-200">
                            기본 템플릿 사용 중
                          </h3>
                          <div className="mt-2 text-sm text-blue-700 dark:text-blue-300">
                            <p>현재 {systemPromptProvider.toUpperCase()} 프로바이더의 기본 템플릿을 사용하고 있습니다. 수정하면 사용자별 정의 프롬프트로 저장됩니다.</p>
                          </div>
                        </div>
                      </div>
                    </div>
                  )}
                </div>
              )}
            </div>

            {/* 모달 푸터 */}
            <div className="flex items-center justify-end space-x-3 p-6 border-t border-gray-200 dark:border-gray-700">
              <button
                onClick={() => {
                  setShowSystemPromptModal(false);
                  setEditSystemPromptUser(null);
                  setError(null);
                }}
                className="px-4 py-2 text-sm text-gray-600 dark:text-gray-400 hover:text-gray-800 dark:hover:text-gray-200 transition-colors"
              >
                취소
              </button>
              <button
                onClick={saveUserSystemPrompt}
                disabled={isLoadingSystemPrompt}
                className="px-4 py-2 text-sm bg-blue-600 text-white rounded-lg hover:bg-blue-700 disabled:opacity-50 disabled:cursor-not-allowed transition-colors"
              >
                {isLoadingSystemPrompt ? '저장 중...' : '저장'}
              </button>
            </div>

            {/* 에러 메시지 표시 */}
            {error && (
              <div className="mx-6 mb-6 px-4 py-2 bg-red-100 border border-red-400 text-red-700 font-semibold rounded text-center">
                {error}
              </div>
            )}
          </div>
        </div>
      )}
    </div>
  );
} 