import { NextRequest, NextResponse } from 'next/server';
import { withTokenAuth } from '../../middlewares';

const API_SERVER = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';

export async function POST(req: NextRequest) {
  return withTokenAuth(req, async (req, user) => {
    try {
      const body = await req.json();
      const { currentPassword, newPassword } = body;

      // 입력 유효성 검사
      if (!currentPassword || !newPassword) {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '현재 비밀번호와 새 비밀번호가 필요합니다.'
            }
          },
          { status: 400 }
        );
      }

      // 새 비밀번호 길이 검증
      if (newPassword.length < 8) {
        return NextResponse.json(
          {
            success: false,
            error: {
              message: '새 비밀번호는 최소 8자 이상이어야 합니다.'
            }
          },
          { status: 400 }
        );
      }

      console.log('API 서버에 비밀번호 변경 요청:', `${API_SERVER}/auth/change-password`);

      // API 서버에 비밀번호 변경 요청
      const response = await fetch(`${API_SERVER}/auth/change-password`, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
          'Authorization': `Bearer ${req.cookies.get('auth_token')?.value}`
        },
        body: JSON.stringify({
          currentPassword,
          newPassword
        })
      });

      if (!response.ok) {
        console.error('API 서버 응답 오류:', {
          status: response.status,
          statusText: response.statusText
        });
        
        let errorMessage = '비밀번호 변경에 실패했습니다.';
        try {
          const errorData = await response.json();
          if (errorData.error?.message) {
            errorMessage = errorData.error.message;
          }
        } catch (parseError) {
          console.error('응답 파싱 오류:', parseError);
        }
        
        return NextResponse.json(
          {
            success: false,
            error: {
              message: errorMessage
            }
          },
          { status: response.status }
        );
      }

      const data = await response.json();
      console.log('비밀번호 변경 성공');
      
      return NextResponse.json({
        success: true,
        message: '비밀번호가 성공적으로 변경되었습니다.'
      });
    } catch (error) {
      console.error('비밀번호 변경 중 오류:', error);
      return NextResponse.json(
        {
          success: false,
          error: {
            message: error instanceof Error ? error.message : '비밀번호 변경 중 오류가 발생했습니다.'
          }
        },
        { status: 500 }
      );
    }
  });
} 