import { NextRequest, NextResponse } from 'next/server';
import { loginToApiServer, setTokenCookie } from '../utils';

export async function POST(request: NextRequest) {
  try {
    const body = await request.json();
    const { username, password } = body;
    
    // 입력 유효성 검사
    if (!username || !password) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '아이디와 비밀번호가 필요합니다.'
          }
        }, 
        { status: 400 }
      );
    }
    
    // API 서버에 로그인 요청
    const loginResult = await loginToApiServer(username, password);
    
    if (!loginResult) {
      return NextResponse.json(
        {
          success: false,
          error: {
            message: '아이디 또는 비밀번호가 올바르지 않습니다.'
          }
        },
        { status: 401 }
      );
    }
    
    // API 키가 없는 경우 처리
    if (!loginResult.apiKey) {
      console.warn('⚠️ API 서버 응답에 API 키가 없습니다. 기본값 사용');
      loginResult.apiKey = 'default-api-key';
    } else {
      console.log('✅ API 서버로부터 API 키를 받았습니다');
    }
    
    // 응답 생성 - apiKeys 배열로 변환하여 클라이언트와 일관성 유지
    const response = NextResponse.json({
      success: true,
      data: {
        user: loginResult.user,
        token: loginResult.token,
        apiKeys: [{ key: loginResult.apiKey }] // 클라이언트에서 기대하는 배열 형태로 변환
      }
    });
    
    // 토큰을 쿠키에 저장
    setTokenCookie(response, loginResult.token);
    
    return response;
  } catch (error) {
    console.error('Login error:', error);
    return NextResponse.json(
      {
        success: false,
        error: {
          message: '로그인 처리 중 오류가 발생했습니다.'
        }
      },
      { status: 500 }
    );
  }
} 