import { NextRequest, NextResponse } from 'next/server';
import fs from 'fs';
import path from 'path';
import os from 'os';
import { exec } from 'child_process';
import { promisify } from 'util';
import { withAnyAuth } from '../middlewares';

const execAsync = promisify(exec);

// 문서/리포트 목록 조회
export async function GET(req: NextRequest) {
  const { searchParams } = new URL(req.url);
  const action = searchParams.get('action');
  const uuid = searchParams.get('uuid');
  const filePath = searchParams.get('filePath');
  const template = searchParams.get('template');
  const jobId = searchParams.get('jobId');

  // 목록 조회 - 사용자 인증 필요
  if (action === 'list') {
    return withAnyAuth(req, async (req, user, authType) => {
      try {
        const username = user?.username;
        if (!username) {
          return NextResponse.json(
            { success: false, error: '사용자 정보를 찾을 수 없습니다.' },
            { status: 401 }
          );
        }

        // console.log(`[문서목록] 사용자별 문서 조회: ${username}`);

        // 1. 진행중 문서 (캐시) - 사용자별 필터링
        const cacheBaseDir = path.join(os.homedir(), '.airun', 'cache', 'report');
        let cacheFiles: Array<any> = [];
        
        try {
          // 사용자별 캐시 디렉토리 확인
          const userCacheDir = path.join(cacheBaseDir, username);
          if (fs.existsSync(userCacheDir)) {
            const projectDirs = fs.readdirSync(userCacheDir);
            // console.log(`[문서목록] 사용자 ${username}의 캐시 폴더 개수:`, projectDirs.length);
            
            for (const dir of projectDirs) {
              const metaPath = path.join(userCacheDir, dir, '000_metadata.json');
              if (fs.existsSync(metaPath)) {
                const meta = JSON.parse(fs.readFileSync(metaPath, 'utf-8'));
                
                // timestamp가 없거나 빈 문자열인 경우 메타데이터 파일의 수정 시간 사용
                let timestamp = meta.timestamp;
                if (!timestamp) {
                  try {
                    const metaStats = fs.statSync(metaPath);
                    timestamp = metaStats.mtime.toISOString();
                  } catch (e) {
                    timestamp = new Date().toISOString(); // 폴백
                  }
                }
                
                cacheFiles.push({
                  uuid: dir,
                  project_hash: dir,
                  executive_summary: meta.executive_summary || '',
                  timestamp: timestamp,
                  status: meta.status || 'processing',
                  output_file_path: meta.output_file_path || '',
                  source: 'cache',
                  userId: username,
                });
              }
            }
          }
          
          // 사용자별 디렉토리가 없는 경우, 기존 방식으로 폴백 (메타데이터에서 사용자 확인)
          if (cacheFiles.length === 0 && fs.existsSync(cacheBaseDir)) {
            const allProjectDirs = fs.readdirSync(cacheBaseDir);
            for (const dir of allProjectDirs) {
              // 사용자 디렉토리가 아닌 경우 건너뛰기
              if (dir === username) continue;
              
              const metaPath = path.join(cacheBaseDir, dir, '000_metadata.json');
              if (fs.existsSync(metaPath)) {
                const meta = JSON.parse(fs.readFileSync(metaPath, 'utf-8'));
                // 메타데이터에서 사용자 확인
                if (meta.userId === username || meta.user_id === username) {
                  
                  // timestamp가 없거나 빈 문자열인 경우 메타데이터 파일의 수정 시간 사용
                  let timestamp = meta.timestamp;
                  if (!timestamp) {
                    try {
                      const metaStats = fs.statSync(metaPath);
                      timestamp = metaStats.mtime.toISOString();
                    } catch (e) {
                      timestamp = new Date().toISOString(); // 폴백
                    }
                  }
                  
                  cacheFiles.push({
                    uuid: dir,
                    project_hash: dir,
                    executive_summary: meta.executive_summary || '',
                    timestamp: timestamp,
                    status: meta.status || 'processing',
                    output_file_path: meta.output_file_path || '',
                    source: 'cache',
                    userId: username,
                  });
                }
              }
            }
          }
        } catch (e) {
          console.error(`[문서목록] 사용자 ${username}의 캐시 처리 오류:`, e);
        }

        // 2. 완료 문서 (output) - 사용자별 필터링
        const outputBaseDir = path.resolve(process.cwd(), '../../output');
        let outputFiles: Array<any> = [];
        
        try {
          // 사용자별 출력 디렉토리 확인
          const userOutputDir = path.join(outputBaseDir, username);
          // console.log(`[문서목록] 사용자 ${username}의 출력 디렉토리 확인:`, userOutputDir);
          // console.log(`[문서목록] 사용자 ${username}의 출력 디렉토리 존재 여부:`, fs.existsSync(userOutputDir));
          
          if (fs.existsSync(userOutputDir)) {
            const outputFolders = fs.readdirSync(userOutputDir);
            // console.log(`[문서목록] 사용자 ${username}의 출력 폴더 개수:`, outputFolders.length);
            // console.log(`[문서목록] 사용자 ${username}의 출력 폴더 목록:`, outputFolders);
            
                          for (const folder of outputFolders) {
                const folderPath = path.join(userOutputDir, folder);
                // console.log(`[문서목록] 폴더 처리 중:`, folder, '경로:', folderPath);
                
                if (fs.statSync(folderPath).isDirectory()) {
                  const files = fs.readdirSync(folderPath).filter(f =>
                    /\.(pdf|docx|hwpx|pptx)$/i.test(f)
                  );
                  // console.log(`[문서목록] 폴더 ${folder}의 문서 파일:`, files);
                  
                  for (const file of files) {
                  outputFiles.push({
                    uuid: folder,
                    project_hash: folder,
                    executive_summary: '',
                    timestamp: fs.statSync(path.join(folderPath, file)).mtime.toISOString(),
                    status: 'completed',
                    output_file_path: `/api/document?action=view-file&filePath=${encodeURIComponent(path.join(folderPath, file))}`,
                    source: 'output',
                    file_name: file,
                    userId: username,
                  });
                }
              }
            }
          }
          
          // 사용자별 디렉토리가 없는 경우, 기존 방식으로 폴백 (메타데이터에서 사용자 확인)
          if (outputFiles.length === 0 && fs.existsSync(outputBaseDir)) {
            const allOutputFolders = fs.readdirSync(outputBaseDir);
            for (const folder of allOutputFolders) {
              // 사용자 디렉토리는 이미 위에서 처리했으므로 건너뛰기
              if (folder === username) continue;
              
              const folderPath = path.join(outputBaseDir, folder);
              if (fs.statSync(folderPath).isDirectory()) {
                // 메타데이터 파일 확인
                const metaPath = path.join(folderPath, '000_metadata.json');
                if (fs.existsSync(metaPath)) {
                  const meta = JSON.parse(fs.readFileSync(metaPath, 'utf-8'));
                  if (meta.userId === username || meta.user_id === username) {
                    const files = fs.readdirSync(folderPath).filter(f =>
                      /\.(pdf|docx|hwpx|pptx)$/i.test(f)
                    );
                    for (const file of files) {
                      outputFiles.push({
                        uuid: folder,
                        project_hash: folder,
                        executive_summary: meta.executive_summary || '',
                        timestamp: fs.statSync(path.join(folderPath, file)).mtime.toISOString(),
                        status: 'completed',
                        output_file_path: `/api/document?action=view-file&filePath=${encodeURIComponent(path.join(folderPath, file))}`,
                        source: 'output',
                        file_name: file,
                        userId: username,
                      });
                    }
                  }
                }
              }
            }
          }
        } catch (e) {
          // console.error(`[문서목록] 사용자 ${username}의 출력 처리 오류:`, e);
        }

        // 합쳐서 반환하고 timestamp 기준으로 정렬 (최신 문서가 맨 위에 오도록)
        const allFiles = [...cacheFiles, ...outputFiles];
        
        // timestamp 기준으로 내림차순 정렬 (최신 문서가 맨 위에)
        allFiles.sort((a, b) => {
          const timeA = new Date(a.timestamp || 0).getTime();
          const timeB = new Date(b.timestamp || 0).getTime();
          return timeB - timeA; // 내림차순 (최신이 위)
        });
        
        const totalFiles = allFiles.length;
        // console.log(`[문서목록] 사용자 ${username}의 최종 files:`, totalFiles);
        
        return NextResponse.json({ 
          success: true, 
          files: allFiles,
          userId: username,
          total: totalFiles
        });
      } catch (error) {
        console.error('[문서목록] 사용자별 필터링 오류:', error);
        return NextResponse.json(
          { success: false, error: '문서 목록 조회 중 오류가 발생했습니다.' },
          { status: 500 }
        );
      }
    });
  }

  // 데이터베이스에서 진행 중인 작업 조회
  if (action === 'active-jobs') {
    return withAnyAuth(req, async (req, user, authType) => {
      try {
        const username = user?.username;
        if (!username) {
          return NextResponse.json(
            { success: false, error: '사용자 정보를 찾을 수 없습니다.' },
            { status: 401 }
          );
        }

        // console.log(`[진행중작업] 데이터베이스에서 사용자별 진행 중인 작업 조회: ${username}`);

        // 인증 헤더 설정
        const headers: Record<string, string> = {
          'Content-Type': 'application/json'
        };

        if (authType === 'token') {
          const token = req.cookies.get('auth_token')?.value;
          if (token) {
            headers['Authorization'] = `Bearer ${token}`;
          } else {
            return NextResponse.json(
              {
                success: false,
                error: '인증 토큰이 필요합니다.'
              },
              { status: 401 }
            );
          }
        } else if (authType === 'apikey') {
          const apiKey = req.headers.get('x-api-key');
          if (apiKey) {
            headers['X-API-Key'] = apiKey;
          } else {
            return NextResponse.json(
              {
                success: false,
                error: 'API 키가 필요합니다.'
              },
              { status: 401 }
            );
          }
        }

        // 스트리밍 서버에서 실제 진행 중인 작업 조회 요청
        const STREAMING_SERVER = 'http://127.0.0.1:5620';
        let userActiveJobs: any[] = [];
        
        try {
          // console.log(`[진행중작업] 스트리밍 서버 연결 시도: ${STREAMING_SERVER}/jobs`);
          
          const controller = new AbortController();
          const timeoutId = setTimeout(() => controller.abort(), 5000); // 5초 타임아웃
          
          const response = await fetch(`${STREAMING_SERVER}/jobs`, {
            method: 'GET',
            headers: {
              'Content-Type': 'application/json'
            },
            signal: controller.signal
          });

          clearTimeout(timeoutId);

          if (!response.ok) {
            const errorText = await response.text();
            // console.warn(`[진행중작업] 스트리밍 서버 호출 실패: ${response.status} ${errorText}`);
            // console.log(`[진행중작업] 빈 배열 반환으로 폴백`);
          } else {
            const data = await response.json();
            
            // 현재 사용자의 작업만 필터링 (processing 및 queued 상태)
            userActiveJobs = (data.jobs || [])
              .filter((job: any) => 
                job.username === username && 
                (job.status === 'processing' || job.status === 'queued')
              )
              .map((job: any) => ({
                id: job.job_id,
                job_id: job.job_id,
                title: job.title || job.request?.prompt || '제목 없음',
                status: job.status,
                progress: job.progress || 0,
                created_at: job.created_at,
                updated_at: job.updated_at,
                user_id: job.username,
                username: job.username,
                template: job.template,
                output_format: job.output_format
              }));

            // console.log(`[진행중작업] 스트리밍 서버에서 사용자 ${username}의 진행 중인 작업 ${userActiveJobs.length}개 조회`);
          }
        } catch (error: any) {
          if (error.name === 'AbortError') {
            console.warn(`[진행중작업] 스트리밍 서버 연결 타임아웃: ${STREAMING_SERVER}`);
          } else {
            console.warn(`[진행중작업] 스트리밍 서버 연결 실패: ${error.message}`);
          }
          console.log(`[진행중작업] 스트리밍 서버에 연결할 수 없어 빈 배열 반환`);
          // 스트리밍 서버 연결 실패 시 빈 배열로 폴백 (오류가 아님)
        }

        return NextResponse.json({
          success: true,
          jobs: userActiveJobs,
          userId: username,
          total: userActiveJobs.length
        });
      } catch (error) {
        console.error('[진행중작업] 데이터베이스 조회 오류:', error);
        return NextResponse.json(
          { success: false, error: '진행 중인 작업 조회 중 오류가 발생했습니다.' },
          { status: 500 }
        );
      }
    });
  }

  // 프로젝트 섹션 목록 조회
  if (action === 'get-sections' && uuid) {
    return withAnyAuth(req, async (req, user, authType) => {
      try {
        const username = user?.username;
        if (!username) {
          return NextResponse.json(
            { success: false, error: '사용자 정보를 찾을 수 없습니다.' },
            { status: 401 }
          );
        }

        console.log(`[섹션조회] 사용자 ${username}의 프로젝트 ${uuid} 섹션 조회 시작`);

        // 1. 사용자별 캐시 디렉토리에서 프로젝트 폴더 확인
        let projectPath = path.join(os.homedir(), '.airun', 'cache', 'report', username, uuid);
        
        if (!fs.existsSync(projectPath)) {
          // 기존 방식으로 폴백 (메타데이터에서 사용자 확인)
          const legacyProjectPath = path.join(os.homedir(), '.airun', 'cache', 'report', uuid);
          if (fs.existsSync(legacyProjectPath)) {
            const metaPath = path.join(legacyProjectPath, '000_metadata.json');
            if (fs.existsSync(metaPath)) {
              const meta = JSON.parse(fs.readFileSync(metaPath, 'utf-8'));
              if (meta.userId === username || meta.user_id === username) {
                projectPath = legacyProjectPath;
              } else {
                return NextResponse.json({ 
                  success: false, 
                  error: '해당 프로젝트에 대한 접근 권한이 없습니다.' 
                }, { status: 403 });
              }
            } else {
              return NextResponse.json({ 
                success: false, 
                error: '프로젝트를 찾을 수 없습니다.' 
              }, { status: 404 });
            }
          } else {
            return NextResponse.json({ 
              success: false, 
              error: '프로젝트를 찾을 수 없습니다.' 
            }, { status: 404 });
          }
        }

        // 2. 프로젝트 폴더에서 파일 목록 가져오기
        const files = fs.readdirSync(projectPath);
        console.log(`[섹션조회] 프로젝트 ${uuid}에서 ${files.length}개 파일 발견`);

        // 3. content 파일들만 필터링 (*-content-*.json)
        const contentFiles = files.filter(filename => {
          // 메타데이터 파일 제외 (000_로 시작)
          if (filename.startsWith('000_')) return false;
          
          // content 파일만 포함 (차트, 테이블, 다이어그램 파일 제외)
          if (!filename.includes('-content-')) return false;
          
          // JSON 파일만
          if (!filename.endsWith('.json')) return false;
          
          return true;
        });

        console.log(`[섹션조회] ${contentFiles.length}개 content 파일 발견:`, contentFiles);

        // 4. 파일 내용 읽어서 섹션 데이터 구성
        const sections = [];
        for (const filename of contentFiles) {
          try {
            const filePath = path.join(projectPath, filename);
            const fileContent = JSON.parse(fs.readFileSync(filePath, 'utf-8'));
            
            // 파일명에서 섹션 정보 추출 (01-01-섹션명-하위섹션명-content-1.json)
            const parts = filename.split('-');
            let sectionTitle = filename.replace('.json', '');
            
            if (parts.length >= 4) {
              // 섹션명과 하위섹션명 합치기 (예: 사업개요-시장분석)
              sectionTitle = `${parts[2]}-${parts[3]}`.replace(/_/g, ' ');
            }
            
            sections.push({
              id: filename.replace('.json', ''),
              title: fileContent.section_name && fileContent.subsection_name 
                ? `${fileContent.section_name}-${fileContent.subsection_name}`
                : sectionTitle,
              content: fileContent.content || '',
              filename: filename,
              created_at: fileContent.created_at,
              section_name: fileContent.section_name,
              subsection_name: fileContent.subsection_name
            });
          } catch (error) {
            console.error(`[섹션조회] 파일 ${filename} 읽기 실패:`, error);
            // 파일 읽기 실패한 경우는 건너뛰기
            continue;
          }
        }

        // 5. 파일명 순서대로 정렬 (01-01, 01-02, 02-01 ...)
        sections.sort((a, b) => {
          const getOrderFromFilename = (filename: string) => {
            const parts = filename.split('-');
            if (parts.length < 2) return '99-99';
            return `${parts[0].padStart(2, '0')}-${parts[1].padStart(2, '0')}`;
          };
          
          const aOrder = getOrderFromFilename(a.filename);
          const bOrder = getOrderFromFilename(b.filename);
          return aOrder.localeCompare(bOrder);
        });

        console.log(`[섹션조회] ${sections.length}개 섹션 조회 완료:`, sections.map(s => ({ title: s.title, filename: s.filename })));

        return NextResponse.json({
          success: true,
          sections: sections,
          total: sections.length
        });
      } catch (error) {
        console.error('[섹션조회] 섹션 목록 조회 오류:', error);
        return NextResponse.json(
          { success: false, error: '섹션 목록 조회 중 오류가 발생했습니다.' },
          { status: 500 }
        );
      }
    });
  }
  
  // 리포트 미리보기 - 사용자별 경로 확인
  if (action === 'view' && uuid) {
    return withAnyAuth(req, async (req, user, authType) => {
      try {
        const username = user?.username;
        if (!username) {
          return NextResponse.json(
            { success: false, error: '사용자 정보를 찾을 수 없습니다.' },
            { status: 401 }
          );
        }

        // console.log(`[문서미리보기] 사용자 ${username}의 문서 조회: ${uuid}`);

        // 1. 사용자별 output 폴더에서 uuid와 일치하는 폴더 확인
        const userOutputDir = path.join(path.resolve(process.cwd(), '../../output'), username);
        const folderPath = path.join(userOutputDir, uuid);
        
        if (fs.existsSync(folderPath) && fs.statSync(folderPath).isDirectory()) {
          // 결과 파일(PDF 등) 찾기
          const files = fs.readdirSync(folderPath).filter(f => /\.(pdf|docx|hwpx|pptx)$/i.test(f));
          if (files.length > 0) {
            const file = files[0];
            const fileExt = path.extname(file).toLowerCase();
            
            // 파일 확장자에 따라 타입 결정
            let fileType = 'pdf';
            if (fileExt === '.pdf') {
              fileType = 'pdf';
            } else if (fileExt === '.hwpx' || fileExt === '.hwp') {
              fileType = 'hwpx';
            } else if (fileExt === '.docx' || fileExt === '.doc') {
              fileType = 'docx';
            } else if (fileExt === '.pptx' || fileExt === '.ppt') {
              fileType = 'pptx';
            }
            
            // 한글 파일명 문제 해결을 위해 UUID 기반 방식 사용
            return NextResponse.json({
              success: true,
              type: fileType,
              filePath: `/api/document?action=view-file&fileUuid=${uuid}&fileName=${encodeURIComponent(file)}`,
              fileName: file,
              fileExtension: fileExt
            });
          }
        }
        
        // 2. 사용자별 진행중 문서: 캐시 폴더에서 내용 결합
        const userCacheDir = path.join(os.homedir(), '.airun', 'cache', 'report', username, uuid);
        if (fs.existsSync(userCacheDir)) {
          const mergedContent = mergeInProgressDocument(userCacheDir);
          return NextResponse.json({ success: true, type: 'inprogress', content: mergedContent });
        }
        
        // 3. 기존 방식으로 폴백 (메타데이터에서 사용자 확인)
        const legacyCacheDir = path.join(os.homedir(), '.airun', 'cache', 'report', uuid);
        if (fs.existsSync(legacyCacheDir)) {
          const metaPath = path.join(legacyCacheDir, '000_metadata.json');
          if (fs.existsSync(metaPath)) {
            const meta = JSON.parse(fs.readFileSync(metaPath, 'utf-8'));
            if (meta.userId === username || meta.user_id === username) {
              const mergedContent = mergeInProgressDocument(legacyCacheDir);
              return NextResponse.json({ success: true, type: 'inprogress', content: mergedContent });
            }
          }
        }
        
        return NextResponse.json(
          { success: false, error: '문서를 찾을 수 없거나 접근 권한이 없습니다.' },
          { status: 404 }
        );
      } catch (error) {
        console.error('[문서미리보기] 사용자별 접근 오류:', error);
        return NextResponse.json(
          { success: false, error: '문서 미리보기 중 오류가 발생했습니다.' },
          { status: 500 }
        );
      }
    });
  }
  // 파일 미리보기/다운로드 (해시 기반)
  const fileHash = searchParams.get('fileHash');
  if (action === 'view-file' && fileHash) {
    try {
      let tempPath = fileHashMap.get(fileHash);
      
      // 해시 맵에 없거나 파일이 존재하지 않으면 원본 파일 찾아서 재생성
      if (!tempPath || !fs.existsSync(tempPath)) {
        console.log(`[view-file] 해시 ${fileHash}에 대한 파일을 찾을 수 없음, 원본 파일 검색 중...`);
        
        const recoveredPath = findAndRecreateHash(fileHash);
        if (recoveredPath && fs.existsSync(recoveredPath)) {
          tempPath = recoveredPath;
          console.log(`[view-file] 해시 ${fileHash} 복구 성공:`, recoveredPath);
        } else {
          console.error(`[view-file] 해시 ${fileHash}에 대한 원본 파일을 찾을 수 없음`);
          return NextResponse.json({ 
            success: false, 
            error: '파일을 찾을 수 없습니다. 문서를 다시 생성해주세요.',
            fileHash: fileHash
          }, { status: 404 });
        }
      }
      
      // 임시 파일에서 읽기 (한글 경로 문제 없음)
      const fileBuffer = await fs.promises.readFile(tempPath);
      
      // 파일 확장자를 안전하게 추출
      const fileName = path.basename(tempPath);
      const ext = fileName.includes('.') ? '.' + fileName.split('.').pop()?.toLowerCase() : '';
      
      let contentType = 'application/octet-stream';
      // 이미지 파일 타입 설정
      if (ext === '.png') contentType = 'image/png';
      else if (ext === '.jpg' || ext === '.jpeg') contentType = 'image/jpeg';
      else if (ext === '.gif') contentType = 'image/gif';
      else if (ext === '.svg') contentType = 'image/svg+xml';
      else if (ext === '.bmp') contentType = 'image/bmp';
      else if (ext === '.webp') contentType = 'image/webp';
      else if (ext === '.ico') contentType = 'image/x-icon';
      
      console.log('[view-file] 해시 기반 파일 읽기 성공:', {
        fileHash,
        tempPath,
        size: fileBuffer.length,
        contentType
      });
      
      return new NextResponse(fileBuffer, {
        status: 200,
        headers: {
          'Content-Type': contentType,
          'Content-Disposition': `inline; filename="${fileName}"`,
          'Cache-Control': 'public, max-age=31536000, immutable',
          'Access-Control-Allow-Origin': '*',
          'Access-Control-Allow-Methods': 'GET',
          'Access-Control-Allow-Headers': 'Content-Type'
        }
      });
    } catch (e: any) {
      console.error('[view-file] 해시 기반 파일 읽기 오류:', e);
      return NextResponse.json({ 
        success: false, 
        error: `파일 처리 오류: ${e.message}`,
        fileHash: fileHash
      }, { status: 500 });
    }
  }
  
    // UUID 기반 파일 처리 (한글 파일명 문제 해결) - 인증 필요
  const fileUuid = searchParams.get('fileUuid');
  const fileName = searchParams.get('fileName');
  if (action === 'view-file' && fileUuid && fileName) {
    return withAnyAuth(req, async (req, user, authType) => {
      try {
        const username = user?.username;
        if (!username) {
          return NextResponse.json(
            { success: false, error: '사용자 정보를 찾을 수 없습니다.' },
            { status: 401 }
          );
        }

        const decodedFileName = decodeURIComponent(fileName);
        console.log('[view-file] UUID 기반 파일 요청:', { fileUuid, fileName: decodedFileName, username });
        
        // 사용자별 output 폴더에서 파일 찾기
        const userOutputDir = path.join(path.resolve(process.cwd(), '../../output'), username);
        const folderPath = path.join(userOutputDir, fileUuid);
        const filePath = path.join(folderPath, decodedFileName);
        
        console.log('[view-file] 파일 경로 구성:', {
          userOutputDir,
          folderPath,
          filePath,
          exists: fs.existsSync(filePath)
        });
        
        // 파일 존재 여부 확인
        if (!fs.existsSync(filePath)) {
          console.error('[view-file] 파일이 존재하지 않음:', filePath);
          return NextResponse.json({ 
            success: false, 
            error: '파일을 찾을 수 없습니다.',
            filePath: filePath
          }, { status: 404 });
        }
        
        // 파일 읽기
        const fileBuffer = await fs.promises.readFile(filePath);
        console.log('[view-file] 파일 읽기 성공, 크기:', fileBuffer.length);
        
        // 파일 확장자를 안전하게 추출
        const ext = decodedFileName.includes('.') ? '.' + decodedFileName.split('.').pop()?.toLowerCase() : '';
        let contentType = 'application/octet-stream';
        
        // 파일 타입 설정
        if (ext === '.pdf') contentType = 'application/pdf';
        else if (ext === '.hwpx') contentType = 'application/octet-stream';
        else if (ext === '.docx') contentType = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document';
        else if (ext === '.pptx') contentType = 'application/vnd.openxmlformats-officedocument.presentationml.presentation';
        
        // 안전한 파일명 생성 (한글 제거)
        const safeFileName = 'document' + ext;
        
        return new NextResponse(fileBuffer, {
          status: 200,
          headers: {
            'Content-Type': contentType,
            'Content-Disposition': `inline; filename="${safeFileName}"`,
            'Cache-Control': 'public, max-age=31536000, immutable',
            'Access-Control-Allow-Origin': '*',
            'Access-Control-Allow-Methods': 'GET',
            'Access-Control-Allow-Headers': 'Content-Type'
          }
        });
      } catch (e: any) {
        console.error('[view-file] UUID 기반 파일 처리 오류:', e);
        return NextResponse.json({ 
          success: false, 
          error: `파일 처리 오류: ${e.message}`,
          fileUuid: fileUuid,
          fileName: fileName
        }, { status: 500 });
      }
    });
  }

  // 파일 미리보기/다운로드 (기존 경로 기반 - 한글 파일명 문제 해결)
  if (action === 'view-file' && filePath) {
    try {
      const absPath = decodeURIComponent(filePath);
      // console.log('[view-file] 파일 접근 시도:', {
      //   originalPath: filePath,
      //   decodedPath: absPath,
      //   exists: fs.existsSync(absPath)
      // });
      
      // 파일 존재 여부 확인
      if (!fs.existsSync(absPath)) {
        console.error('[view-file] 파일이 존재하지 않음:', absPath);
        return NextResponse.json({ 
          success: false, 
          error: '파일을 찾을 수 없습니다.',
          filePath: absPath
        }, { status: 404 });
      }
      
      // 한글 파일명 문제를 해결하기 위해 Node.js 기본 기능 사용
      let fileBuffer: Buffer;
      try {
        // Node.js fs.readFile을 직접 사용 - Buffer로 읽어서 인코딩 문제 회피
        fileBuffer = await fs.promises.readFile(absPath);
        // console.log('[view-file] 파일 읽기 성공, 크기:', fileBuffer.length);
      } catch (readError: any) {
        console.error('[view-file] 파일 읽기 실패:', readError.message);
        
        // 폴백: child_process로 cp 명령어 사용
        try {
          const tempFileName = `temp_${Date.now()}_${Math.random().toString(36).substr(2, 9)}`;
          const tempPath = `/tmp/${tempFileName}`;
          
          // cp 명령어로 파일 복사 (쉘 이스케이프 처리)
          const escapedPath = absPath.replace(/'/g, "'\"'\"'");
          await execAsync(`cp '${escapedPath}' '${tempPath}'`, {
            maxBuffer: 50 * 1024 * 1024 // 50MB 제한
          });
          
          // 임시 파일에서 읽기
          fileBuffer = await fs.promises.readFile(tempPath);
          
          // 임시 파일 정리
          await fs.promises.unlink(tempPath);
          
          // console.log('[view-file] 폴백 방식으로 파일 읽기 성공, 크기:', fileBuffer.length);
        } catch (fallbackError: any) {
          console.error('[view-file] 폴백 방식도 실패:', fallbackError.message);
          return NextResponse.json({ 
            success: false, 
            error: `파일을 읽을 수 없습니다: ${fallbackError.message}`,
            filePath: absPath
          }, { status: 500 });
        }
      }
      
      // 파일 확장자를 안전하게 추출 (한글 경로 문제 회피)
      const fileName = absPath.split('/').pop() || '';
      const ext = fileName.includes('.') ? '.' + fileName.split('.').pop()?.toLowerCase() : '';
      let contentType = 'application/octet-stream';
      
      // 이미지 파일 타입 설정
      if (ext === '.png') contentType = 'image/png';
      else if (ext === '.jpg' || ext === '.jpeg') contentType = 'image/jpeg';
      else if (ext === '.gif') contentType = 'image/gif';
      else if (ext === '.svg') contentType = 'image/svg+xml';
      else if (ext === '.bmp') contentType = 'image/bmp';
      else if (ext === '.webp') contentType = 'image/webp';
      else if (ext === '.ico') contentType = 'image/x-icon';
      // 문서 파일 타입 설정
      else if (ext === '.pdf') contentType = 'application/pdf';
      else if (ext === '.docx') contentType = 'application/vnd.openxmlformats-officedocument.wordprocessingml.document';
      else if (ext === '.pptx') contentType = 'application/vnd.openxmlformats-officedocument.presentationml.presentation';
      else if (ext === '.hwpx') contentType = 'application/octet-stream';
      // 텍스트 파일 타입 설정
      else if (ext === '.txt') contentType = 'text/plain';
      else if (ext === '.md') contentType = 'text/markdown';
      else if (ext === '.html') contentType = 'text/html';
      else if (ext === '.css') contentType = 'text/css';
      else if (ext === '.js') contentType = 'application/javascript';
      else if (ext === '.json') contentType = 'application/json';
      else if (ext === '.xml') contentType = 'application/xml';

      // 파일명을 안전하게 추출 (한글 경로 문제 회피)
      const safeFileName = fileName || 'file';
      
      return new NextResponse(fileBuffer, {
        status: 200,
        headers: {
          'Content-Type': contentType,
          'Content-Disposition': `inline; filename="${safeFileName}"`,
          'Cache-Control': 'public, max-age=31536000, immutable', // 1년 캐시
          'Access-Control-Allow-Origin': '*',
          'Access-Control-Allow-Methods': 'GET',
          'Access-Control-Allow-Headers': 'Content-Type'
        }
      });
    } catch (e: any) {
      console.error('[view-file] 전체 처리 오류:', {
        filePath: filePath,
        absPath: decodeURIComponent(filePath),
        error: e.message,
        stack: e.stack
      });
      return NextResponse.json({ 
        success: false, 
        error: `파일 처리 오류: ${e.message}`,
        filePath: decodeURIComponent(filePath)
      }, { status: 500 });
    }
  }
  // 템플릿/예제 목록 제공
  if (action === 'template' && !template) {
    try {
      const API_SERVER = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';
      const res = await fetch(`${API_SERVER}/report/templates`, {
        headers: {
          'accept': 'application/json',
          'X-API-Key': process.env.AIRUN_API_KEY || 'airun_1_e8f7a00fe827fdf1a9364311d11242c5',
        },
      });
      if (!res.ok) {
        const text = await res.text();
        return NextResponse.json({ success: false, error: `템플릿 목록 API 오류: ${res.status} ${res.statusText} - ${text}` }, { status: res.status });
      }
      const data = await res.json();
      if (data.success && data.data && data.data.data && Array.isArray(data.data.data.templates)) {
        const templates = data.data.data.templates.map((t: any) => ({
          value: t.name,
          label: t.name,
          ...t,
        }));
        return NextResponse.json({ success: true, templates });
      } else {
        return NextResponse.json({ success: false, error: '템플릿 목록을 불러오지 못했습니다.' }, { status: 500 });
      }
    } catch (e: any) {
      return NextResponse.json({ success: false, error: `템플릿 목록 API 호출 실패: ${e.message}` }, { status: 500 });
    }
  }
  // 작업 로그/상태 확인
  if (action === 'log') {
    // 1. log_file 파라미터가 있으면, 해당 경로의 파일을 직접 읽어서 반환
    const logFileParam = searchParams.get('log_file');
    if (logFileParam) {
      let log = '';
      let found = false;
      try {
        log = fs.readFileSync(logFileParam, 'utf-8');
        found = true;
      } catch (e) {}
      if (found) {
        return NextResponse.json({ success: true, log_file: logFileParam, log });
      } else {
        return NextResponse.json({ success: false, log_file: logFileParam, log: '', error: '로그파일 없음' });
      }
    }

    // 2. jobId만 있으면, 상태 API에서 log_file을 받아서 그 경로의 파일을 읽어서 반환
    if (jobId) {
      // 상태 API에서 log_file 경로 조회
      try {
        const API_SERVER = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';
        const statusRes = await fetch(`${API_SERVER}/report/status/${encodeURIComponent(jobId)}`, {
          headers: { 
            'X-API-Key': process.env.AIRUN_API_KEY || 'airun_1_e8f7a00fe827fdf1a9364311d11242c5', 
            'accept': 'application/json' 
          },
        });
        if (!statusRes.ok) {
          return NextResponse.json({ success: false, error: '상태 API 호출 실패' });
        }
        const response = await statusRes.json();
        let logFile = null;
        if (response && response.data && response.data.log_file) {
          logFile = response.data.log_file;
        } else if (response && response.log_file) {
          logFile = response.log_file;
        }
        if (logFile) {
          let log = '';
          let found = false;
          try {
            log = fs.readFileSync(logFile, 'utf-8');
            found = true;
          } catch (e) {}
          if (found) {
            return NextResponse.json({ success: true, log_file: logFile, log });
          } else {
            return NextResponse.json({ success: false, log_file: logFile, log: '', error: '로그파일 없음' });
          }
        } else {
          return NextResponse.json({ success: false, error: '상태 API에서 log_file을 찾을 수 없습니다.' });
        }
      } catch (e) {
        return NextResponse.json({ success: false, error: '상태 API 호출 실패' });
      }
    }

    // log_file, jobId 둘 다 없으면
    return NextResponse.json({ success: false, error: 'log_file 또는 jobId가 필요합니다.' }, { status: 400 });
  }

  // business_info.json 가져오기 - 사용자별 경로 지원
  if (action === 'get-business-info') {
    return withAnyAuth(req, async (req, user, authType) => {
      try {
        const uuid = searchParams.get('uuid');
        if (!uuid) {
          return NextResponse.json({ error: 'UUID 필요' }, { status: 400 });
        }

        const username = user?.username;
        if (!username) {
          return NextResponse.json(
            { success: false, error: '사용자 정보를 찾을 수 없습니다.' },
            { status: 401 }
          );
        }

        // 1. 사용자별 캐시 디렉토리에서 찾기
        const userCacheDir = path.join(os.homedir(), '.airun', 'cache', 'report', username, uuid);
        let businessInfoPath = path.join(userCacheDir, '000_business_info.json');
        
        if (fs.existsSync(businessInfoPath)) {
          try {
            const businessInfo = JSON.parse(fs.readFileSync(businessInfoPath, 'utf-8'));
            return NextResponse.json(businessInfo);
          } catch (error) {
            console.error('비즈니스 정보 파일 읽기 오류:', error);
            return NextResponse.json({ error: '비즈니스 정보 파일 읽기 실패' }, { status: 500 });
          }
        }

        // 2. 기존 방식으로 폴백 (메타데이터에서 사용자 확인)
        const legacyCacheDir = path.join(os.homedir(), '.airun', 'cache', 'report', uuid);
        businessInfoPath = path.join(legacyCacheDir, '000_business_info.json');
        
        if (fs.existsSync(businessInfoPath)) {
          // 메타데이터에서 사용자 확인
          const metaPath = path.join(legacyCacheDir, '000_metadata.json');
          if (fs.existsSync(metaPath)) {
            try {
              const meta = JSON.parse(fs.readFileSync(metaPath, 'utf-8'));
              if (meta.userId === username || meta.user_id === username) {
                const businessInfo = JSON.parse(fs.readFileSync(businessInfoPath, 'utf-8'));
                return NextResponse.json(businessInfo);
              } else {
                return NextResponse.json({ 
                  error: '해당 문서에 대한 접근 권한이 없습니다.' 
                }, { status: 403 });
              }
            } catch (error) {
              console.error('비즈니스 정보 파일 읽기 오류:', error);
              return NextResponse.json({ error: '비즈니스 정보 파일 읽기 실패' }, { status: 500 });
            }
          }
        }

        return NextResponse.json({ error: '비즈니스 정보 파일을 찾을 수 없습니다' }, { status: 404 });
      } catch (error) {
        console.error('비즈니스 정보 조회 오류:', error);
        return NextResponse.json(
          { success: false, error: '비즈니스 정보 조회 중 오류가 발생했습니다.' },
          { status: 500 }
        );
      }
    });
  }

  return NextResponse.json({ success: false, error: 'Invalid request' }, { status: 400 });
}

// 파일 해시 맵 저장소 (메모리 기반, 실제로는 Redis나 DB 사용 권장)
const fileHashMap = new Map<string, string>();

// 해시 맵 복구 함수 (서버 재시작 시 기존 해시 복구)
function recoverHashMap() {
  try {
    const tempDir = '/tmp/airun_images';
    if (fs.existsSync(tempDir)) {
      const files = fs.readdirSync(tempDir);
      for (const file of files) {
        const filePath = path.join(tempDir, file);
        const hash = file.split('.')[0]; // 파일명에서 해시 추출
        if (hash && fs.existsSync(filePath)) {
          fileHashMap.set(hash, filePath);
        }
      }
      console.log(`[recoverHashMap] ${fileHashMap.size}개 해시 복구 완료`);
    }
  } catch (e) {
    console.error('[recoverHashMap] 해시 맵 복구 오류:', e);
  }
}

// 서버 시작 시 해시 맵 복구
recoverHashMap();

// 원본 파일을 찾아서 해시를 재생성하는 함수
function findAndRecreateHash(targetHash: string): string | null {
  try {
    // 캐시 디렉토리에서 해당 해시의 원본 파일 찾기
    const cacheBaseDir = path.join(os.homedir(), '.airun', 'cache', 'report');
    
    if (!fs.existsSync(cacheBaseDir)) {
      return null;
    }
    
    const searchForImageFiles = (dir: string): string | null => {
      try {
        const items = fs.readdirSync(dir);
        for (const item of items) {
          const itemPath = path.join(dir, item);
          const stat = fs.statSync(itemPath);
          
          if (stat.isDirectory()) {
            const result = searchForImageFiles(itemPath);
            if (result) return result;
          } else if (stat.isFile()) {
            // 이미지 파일인지 확인
            const ext = path.extname(item).toLowerCase();
            if (['.png', '.jpg', '.jpeg', '.svg'].includes(ext)) {
              // 파일 내용으로 해시 생성해서 비교
              try {
                const crypto = require('crypto');
                const fileContent = fs.readFileSync(itemPath);
                const hash = crypto.createHash('md5').update(fileContent).digest('hex');
                
                if (hash === targetHash) {
                  // 해시가 일치하면 다시 캐싱
                  const recreatedHash = createImageHash(itemPath, item);
                  if (recreatedHash === targetHash) {
                    return fileHashMap.get(targetHash) || null;
                  }
                }
              } catch (e) {
                // 파일 읽기 실패 시 건너뛰기
                continue;
              }
            }
          }
        }
        return null;
      } catch (e) {
        return null;
      }
    };
    
    return searchForImageFiles(cacheBaseDir);
  } catch (e) {
    console.error('[findAndRecreateHash] 원본 파일 찾기 오류:', e);
    return null;
  }
}

// 진행중 문서 내용 결합 함수
function mergeInProgressDocument(cacheDir: string): string {
  let result = '';
  // 1. business_info.json에서 product_name을 제목으로 사용
  let documentTitle = '문서 제목 없음';
  
  const businessInfoPath = path.join(cacheDir, '000_business_info.json');
  if (fs.existsSync(businessInfoPath)) {
    try {
      const businessInfo = JSON.parse(fs.readFileSync(businessInfoPath, 'utf-8'));
      if (businessInfo.business_info && businessInfo.business_info.product_name) {
        documentTitle = businessInfo.business_info.product_name;
      }
    } catch {}
  }
  
  result += `# ${documentTitle}\n\n`;
  // 2. 섹션/하위섹션 content 파일들 정렬 (000_로 시작하는 메타데이터 파일 제외)
  let files = fs.readdirSync(cacheDir)
    .filter(f => f.endsWith('.json') && f.includes('content') && !f.startsWith('000_'))
    .sort();
  for (const file of files) {
    const filePath = path.join(cacheDir, file);
    try {
      const data = JSON.parse(fs.readFileSync(filePath, 'utf-8'));
      if (data.section_name && data.subsection_name) {
        result += `## ${data.section_name} - ${data.subsection_name}\n`;
      }
      if (data.content) {
        result += `${data.content}\n\n`;
      }
      // 3. 표 데이터(csv) 결합
      // 파일명 패턴: ...-table-*.csv (000_로 시작하는 파일 제외)
      const basePattern = file.replace(/content.*\.json$/, '');
      const tableFiles = fs.readdirSync(cacheDir)
        .filter(f => f.startsWith(basePattern) && f.includes('table') && f.endsWith('.csv') && !f.startsWith('000_'));
      for (const tableFile of tableFiles) {
        const tablePath = path.join(cacheDir, tableFile);
        try {
          const csvContent = fs.readFileSync(tablePath, 'utf-8');
          const rows = csvContent.split(/\r?\n/).filter(Boolean).map(row => row.split(','));
          if (rows.length > 0) {
            // 마크다운 테이블 변환
            const header = '| ' + rows[0].join(' | ') + ' |\n';
            const separator = '| ' + rows[0].map(() => '---').join(' | ') + ' |\n';
            const body = rows.slice(1).map(r => '| ' + r.join(' | ') + ' |').join('\n') + '\n';
            result += `\n${header}${separator}${body}\n`;
          }
        } catch {}
      }
      // 4. 이미지 결합 (png, jpg, jpeg, svg) - 해시 기반 캐싱 사용 (000_로 시작하는 파일 제외)
      const imageExts = ['.png', '.jpg', '.jpeg', '.svg'];
      const imageFiles = fs.readdirSync(cacheDir)
        .filter(f => f.startsWith(basePattern) && imageExts.some(ext => f.endsWith(ext)) && !f.startsWith('000_'));
      for (const imageFile of imageFiles) {
        try {
          // 한글 파일명 문제 해결을 위해 해시 기반 캐싱
          const originalPath = path.join(cacheDir, imageFile);
          const fileHash = createImageHash(originalPath, imageFile);
          
          // 해시 기반 API 경로 사용
          const imageApiPath = `/api/document?action=view-file&fileHash=${fileHash}`;
          result += `\n![이미지](${imageApiPath})\n`;
        } catch (e) {
          console.error('[mergeInProgressDocument] 이미지 처리 오류:', e);
        }
      }
    } catch {}
  }
  return result;
}

// 파일을 해시 기반으로 캐싱하는 함수 (한글 파일명 문제 해결)
function createImageHash(originalPath: string, fileName: string): string {
  try {
    // 파일 존재 여부 확인
    if (!fs.existsSync(originalPath)) {
      console.error('[createImageHash] 원본 파일이 존재하지 않음:', originalPath);
      return '';
    }
    
    // 파일 내용 기반 해시 생성
    const crypto = require('crypto');
    const fileContent = fs.readFileSync(originalPath);
    const hash = crypto.createHash('md5').update(fileContent).digest('hex');
    
    // 파일 확장자 추출
    const ext = fileName.includes('.') ? '.' + fileName.split('.').pop()?.toLowerCase() : '.png';
    const hashedFileName = `${hash}${ext}`;
    
    // 임시 디렉토리에 ASCII 파일명으로 복사
    const tempDir = '/tmp/airun_images';
    if (!fs.existsSync(tempDir)) {
      fs.mkdirSync(tempDir, { recursive: true });
    }
    
    const tempPath = path.join(tempDir, hashedFileName);
    
    // 파일이 이미 존재하지 않으면 복사
    if (!fs.existsSync(tempPath)) {
      try {
        fs.copyFileSync(originalPath, tempPath);
        console.log('[createImageHash] 파일 복사 완료:', { originalPath, tempPath, hash });
      } catch (copyError) {
        console.error('[createImageHash] 파일 복사 실패:', copyError);
        return '';
      }
    }
    
    // 해시 맵에 저장
    fileHashMap.set(hash, tempPath);
    
    return hash;
  } catch (e) {
    console.error('[createImageHash] 해시 생성 오류:', e);
    return '';
  }
}

// 문서/리포트 생성, 변환, 수정, 삭제 등
export async function POST(req: NextRequest) {
  return withAnyAuth(req, async (req, user, authType) => {
    try {
      const body = await req.json();
      const { action, prompt, template, outputType, email, username } = body;

      // 사용자 정보 확인
      const authenticatedUsername = user?.username;
      if (!authenticatedUsername) {
        return NextResponse.json(
          { success: false, error: '사용자 정보를 찾을 수 없습니다.' },
          { status: 401 }
        );
      }

      // 요청 body의 username보다 인증된 사용자 정보를 우선 사용
      const finalUsername = authenticatedUsername;

      // console.log(`[문서생성] 사용자 ${finalUsername}의 요청:`, { action, prompt: prompt?.substring(0, 50) + '...', template, outputType });

      // 🔍 Clarification 분석 액션 추가
      if (action === 'analyze') {
        try {
          const { prompt, template, format } = body;
          
          // 인증 헤더 설정
          const headers: Record<string, string> = {
            'Content-Type': 'application/json'
          };

          if (authType === 'token') {
            const token = req.cookies.get('auth_token')?.value;
            if (token) {
              headers['Authorization'] = `Bearer ${token}`;
            } else {
              return NextResponse.json(
                { success: false, error: '인증 토큰이 필요합니다.' },
                { status: 401 }
              );
            }
          } else if (authType === 'apikey') {
            const apiKey = req.headers.get('x-api-key');
            if (apiKey) {
              headers['X-API-Key'] = apiKey;
            } else {
              return NextResponse.json(
                { success: false, error: 'API 키가 필요합니다.' },
                { status: 401 }
              );
            }
          }

          // Report Server의 /analyze 엔드포인트 호출
          const REPORT_SERVER = 'http://127.0.0.1:5620';
          
          console.log(`[Clarification] Report Server 연결 시도: ${REPORT_SERVER}/analyze`);
          
          let response: Response;
          let analysisResult: any;
          let timeoutId: NodeJS.Timeout | undefined;
          
          try {
            // 타임아웃 설정 (30초)
            const controller = new AbortController();
            timeoutId = setTimeout(() => controller.abort(), 30000);
            
            console.log(`[Clarification] fetch 요청 시작...`);
            
            response = await fetch(`${REPORT_SERVER}/analyze`, {
              method: 'POST',
              headers,
              body: JSON.stringify({
                prompt,
                template: template || 'auto',
                format: format || 'pdf',
                username: finalUsername,
                model: 'airun-chat:latest' // JSON 형식 응답을 위해 airun-chat 모델 사용
              }),
              signal: controller.signal
            });
            
            clearTimeout(timeoutId);
            console.log(`[Clarification] fetch 응답 받음:`, response.status, response.statusText);

            if (!response.ok) {
              const errorData = await response.json().catch(() => ({}));
              console.error(`[Clarification] 분석 요청 실패:`, response.status, errorData);
              return NextResponse.json({ 
                success: false, 
                error: errorData.detail || '요청 분석에 실패했습니다.',
                username: finalUsername
              }, { status: response.status });
            }
            
            console.log(`[Clarification] 응답 JSON 파싱 시작...`);
            analysisResult = await response.json();
            console.log(`[Clarification] 응답 JSON 파싱 완료:`, analysisResult ? '성공' : '실패');
          } catch (fetchError: any) {
            if (timeoutId) clearTimeout(timeoutId);
            
            console.error(`[Clarification] fetch 에러 발생:`, fetchError.name, fetchError.message);
            
            if (fetchError.name === 'AbortError') {
              console.error(`[Clarification] Report Server 연결 타임아웃: ${REPORT_SERVER}`);
              return NextResponse.json({ 
                success: false, 
                error: 'Report Server 연결이 시간 초과되었습니다. 서버가 실행 중인지 확인해주세요.',
                username: finalUsername
              }, { status: 503 });
            }
            
            console.error(`[Clarification] Report Server 연결 실패:`, fetchError.message);
            return NextResponse.json({ 
              success: false, 
              error: 'Report Server에 연결할 수 없습니다. 서버가 실행 중인지 확인해주세요.',
              username: finalUsername
            }, { status: 503 });
          }

          console.log(`[Clarification] 사용자 ${finalUsername} 분석 완료`);

          return NextResponse.json({ 
            success: true, 
            ...analysisResult,
            username: finalUsername
          });

        } catch (e: any) {
          console.error(`[Clarification] 사용자 ${finalUsername} 분석 오류:`, e);
          return NextResponse.json({ 
            success: false, 
            error: `요청 분석 중 오류가 발생했습니다: ${e.message}`,
            username: finalUsername
          }, { status: 500 });
        }
      }

      // 🔍 향상된 문서 생성 액션 추가
      if (action === 'generate-enhanced') {
        try {
          const { 
            original_prompt, 
            clarification_responses, 
            confirmed_template, 
            format, 
            email, 
            use_cache 
          } = body;
          
          // 인증 헤더 설정
          const headers: Record<string, string> = {
            'Content-Type': 'application/json'
          };

          if (authType === 'token') {
            const token = req.cookies.get('auth_token')?.value;
            if (token) {
              headers['Authorization'] = `Bearer ${token}`;
            } else {
              return NextResponse.json(
                { success: false, error: '인증 토큰이 필요합니다.' },
                { status: 401 }
              );
            }
          } else if (authType === 'apikey') {
            const apiKey = req.headers.get('x-api-key');
            if (apiKey) {
              headers['X-API-Key'] = apiKey;
            } else {
              return NextResponse.json(
                { success: false, error: 'API 키가 필요합니다.' },
                { status: 401 }
              );
            }
          }

          // Report Server의 /generate-enhanced 엔드포인트 호출
          const REPORT_SERVER = 'http://127.0.0.1:5620';
          
          let response: Response;
          let enhancedResult: any;
          let timeoutId: NodeJS.Timeout | undefined;
          
          try {
            // 타임아웃 설정 (30초)
            const controller = new AbortController();
            timeoutId = setTimeout(() => controller.abort(), 30000);
            
            response = await fetch(`${REPORT_SERVER}/generate-enhanced`, {
              method: 'POST',
              headers,
              body: JSON.stringify({
                original_prompt,
                clarification_responses,
                confirmed_template: confirmed_template || 'simple',
                format: format || 'pdf',
                email,
                username: finalUsername,
                use_cache: use_cache || true,
                model: 'airun-chat:latest' // JSON 형식 응답을 위해 airun-chat 모델 사용
              }),
              signal: controller.signal
            });
            
            clearTimeout(timeoutId);

            if (!response.ok) {
              const errorData = await response.json().catch(() => ({}));
              console.error(`[Clarification] 향상된 생성 요청 실패:`, response.status, errorData);
              return NextResponse.json({ 
                success: false, 
                error: errorData.detail || '향상된 문서 생성에 실패했습니다.',
                username: finalUsername
              }, { status: response.status });
            }
            
            enhancedResult = await response.json();
          } catch (fetchError: any) {
            if (timeoutId) clearTimeout(timeoutId);
            
            if (fetchError.name === 'AbortError') {
              console.error(`[Clarification] Report Server 연결 타임아웃: ${REPORT_SERVER}`);
              return NextResponse.json({ 
                success: false, 
                error: 'Report Server 연결이 시간 초과되었습니다. 서버가 실행 중인지 확인해주세요.',
                username: finalUsername
              }, { status: 503 });
            }
            
            console.error(`[Clarification] Report Server 연결 실패:`, fetchError.message);
            return NextResponse.json({ 
              success: false, 
              error: 'Report Server에 연결할 수 없습니다. 서버가 실행 중인지 확인해주세요.',
              username: finalUsername
            }, { status: 503 });
          }

          console.log(`[Clarification] 사용자 ${finalUsername} 향상된 생성 시작`);

          return NextResponse.json({ 
            success: true, 
            ...enhancedResult,
            username: finalUsername
          });

        } catch (e: any) {
          console.error(`[Clarification] 사용자 ${finalUsername} 향상된 생성 오류:`, e);
          return NextResponse.json({ 
            success: false, 
            error: `향상된 문서 생성 중 오류가 발생했습니다: ${e.message}`,
            username: finalUsername
          }, { status: 500 });
        }
      }

      if (action === 'create') {
        try {
          // 인증 헤더 설정
          const headers: Record<string, string> = {
            'Content-Type': 'application/json'
          };

          if (authType === 'token') {
            const token = req.cookies.get('auth_token')?.value;
            if (token) {
              headers['Authorization'] = `Bearer ${token}`;
            } else {
              return NextResponse.json(
                {
                  success: false,
                  error: '인증 토큰이 필요합니다.'
                },
                { status: 401 }
              );
            }
          } else if (authType === 'apikey') {
            const apiKey = req.headers.get('x-api-key');
            if (apiKey) {
              headers['X-API-Key'] = apiKey;
            } else {
              return NextResponse.json(
                {
                  success: false,
                  error: 'API 키가 필요합니다.'
                },
                { status: 401 }
              );
            }
          }

          const API_SERVER = process.env.NEXT_PUBLIC_API_SERVER_URL || 'http://localhost:5500/api/v1';
          const res = await fetch(`${API_SERVER}/report`, {
            method: 'POST',
            headers,
            body: JSON.stringify({
              prompt,
              template: template || 'simple',
              format: outputType || 'pdf',
              email,
              username: finalUsername, // 사용자명 추가
            }),
          });
          
          // console.log(`[문서생성] 사용자 ${finalUsername} 문서 생성 fetch 응답:`, res.status, res.statusText);
          let data;
          try {
            data = await res.json();
          } catch (jsonErr) {
            console.error(`[문서생성] 사용자 ${finalUsername} fetch JSON 파싱 실패:`, jsonErr);
            data = { error: 'JSON 파싱 실패', detail: jsonErr };
          }
          
          if (!res.ok) {
            console.error(`[문서생성] 사용자 ${finalUsername} API 호출 실패:`, data);
            return NextResponse.json({ 
              success: false, 
              message: data.error?.message || '문서 생성 실패',
              username: finalUsername
            }, { status: 500 });
          }
          // console.log(`[문서생성] 사용자 ${finalUsername} data:`, data);
          
          // project_hash가 있으면 그 값을 uuid로 반환
          let projectHash = data.project_hash;
          if (!projectHash && (data.data && data.data.project_hash)) {
            projectHash = data.data.project_hash;
          }
          const jobId = data.jobId || (data.data && data.data.jobId);
          
          // project_hash가 있으면(즉, 워커가 이미 시작된 경우)만 반환
          if (data.success && projectHash) {
            return NextResponse.json({ 
              success: true, 
              uuid: projectHash, 
              project_hash: projectHash,
              username: finalUsername
            });
          }
          
          // project_hash가 없고, jobId만 있으면(큐에 들어간 직후) 오직 jobId만 반환
          if (data.success && jobId) {
            return NextResponse.json({ 
              success: true, 
              jobId, 
              status: 'queued', 
              message: '작업이 큐에 추가되었습니다.',
              username: finalUsername
            });
          }
          
          return NextResponse.json({ 
            success: false, 
            message: data.message || '문서 생성 실패',
            username: finalUsername
          }, { status: 500 });
        } catch (e: any) {
          console.error(`[문서생성] 사용자 ${finalUsername} API 오류:`, e, e?.stack);
          return NextResponse.json({ 
            success: false, 
            message: `문서 생성 API 오류: ${e.message || e}`,
            username: finalUsername
          }, { status: 500 });
        }
      }
      
      // 파일 변환 (오피스/한글 → PDF)
      if (action === 'convert') {
        // TODO: officeFilePath 변환, 변환된 pdfFilePath 반환
        return NextResponse.json({ success: true, pdfFilePath: '/stub/path/to/converted.pdf' });
      }
      
      // 리포트 수정
      if (action === 'update') {
        // TODO: uuid, prompt 등으로 리포트 수정
        return NextResponse.json({ success: true });
      }
      
      // 리포트/문서 삭제 - 사용자별 권한 확인
      if (action === 'delete') {
        try {
          if (body.source === 'cache' && body.uuid) {
            // 사용자별 캐시 디렉토리에서 삭제
            const userCacheDir = path.join(os.homedir(), '.airun', 'cache', 'report', finalUsername, body.uuid);
            if (fs.existsSync(userCacheDir)) {
              await fs.promises.rm(userCacheDir, { recursive: true, force: true });
              // console.log(`[문서삭제] 사용자 ${finalUsername}의 캐시 문서 삭제 완료:`, body.uuid);
              return NextResponse.json({ success: true, message: '진행중 문서가 삭제되었습니다.' });
            }
            
            // 기존 방식으로 폴백 (메타데이터에서 사용자 확인)
            const legacyCacheDir = path.join(os.homedir(), '.airun', 'cache', 'report', body.uuid);
            if (fs.existsSync(legacyCacheDir)) {
              const metaPath = path.join(legacyCacheDir, '000_metadata.json');
              if (fs.existsSync(metaPath)) {
                const meta = JSON.parse(fs.readFileSync(metaPath, 'utf-8'));
                if (meta.userId === finalUsername || meta.user_id === finalUsername) {
                  await fs.promises.rm(legacyCacheDir, { recursive: true, force: true });
                  // console.log(`[문서삭제] 사용자 ${finalUsername}의 레거시 캐시 문서 삭제 완료:`, body.uuid);
                  return NextResponse.json({ success: true, message: '진행중 문서가 삭제되었습니다.' });
                } else {
                  return NextResponse.json({ 
                    success: false, 
                    error: '해당 문서에 대한 삭제 권한이 없습니다.' 
                  }, { status: 403 });
                }
              }
            }
            
            return NextResponse.json({ 
              success: false, 
              error: '삭제할 문서를 찾을 수 없습니다.' 
            }, { status: 404 });
          }
          
          if (body.source === 'output' && body.uuid) {
            // 사용자별 출력 디렉토리에서 삭제
            const userOutputDir = path.join(path.resolve(process.cwd(), '../../output'), finalUsername, body.uuid);
            if (fs.existsSync(userOutputDir)) {
              await fs.promises.rm(userOutputDir, { recursive: true, force: true });
              // console.log(`[문서삭제] 사용자 ${finalUsername}의 출력 문서 삭제 완료:`, body.uuid);
              return NextResponse.json({ success: true, message: '완료된 문서가 삭제되었습니다.' });
            }
            
            // 기존 방식으로 폴백 (메타데이터에서 사용자 확인)
            const legacyOutputDir = path.join(path.resolve(process.cwd(), '../../output'), body.uuid);
            if (fs.existsSync(legacyOutputDir)) {
              const metaPath = path.join(legacyOutputDir, '000_metadata.json');
              if (fs.existsSync(metaPath)) {
                const meta = JSON.parse(fs.readFileSync(metaPath, 'utf-8'));
                if (meta.userId === finalUsername || meta.user_id === finalUsername) {
                  await fs.promises.rm(legacyOutputDir, { recursive: true, force: true });
                  // console.log(`[문서삭제] 사용자 ${finalUsername}의 레거시 출력 문서 삭제 완료:`, body.uuid);
                  return NextResponse.json({ success: true, message: '완료된 문서가 삭제되었습니다.' });
                } else {
                  return NextResponse.json({ 
                    success: false, 
                    error: '해당 문서에 대한 삭제 권한이 없습니다.' 
                  }, { status: 403 });
                }
              }
            }
            
            return NextResponse.json({ 
              success: false, 
              error: '삭제할 문서를 찾을 수 없습니다.' 
            }, { status: 404 });
          }
          
          return NextResponse.json({ 
            success: false, 
            error: '잘못된 삭제 요청입니다.' 
          }, { status: 400 });
        } catch (e: any) {
          console.error(`[문서삭제] 사용자 ${finalUsername} 삭제 오류:`, e);
          return NextResponse.json({ 
            success: false, 
            error: `문서 삭제 중 오류가 발생했습니다: ${e.message}` 
          }, { status: 500 });
        }
      }

      return NextResponse.json({ 
        success: false, 
        error: '지원하지 않는 작업입니다.' 
      }, { status: 400 });
    } catch (error) {
      console.error('[문서API] 전체 처리 오류:', error);
      return NextResponse.json(
        { success: false, error: '요청 처리 중 오류가 발생했습니다.' },
        { status: 500 }
      );
    }
  });
}