import { NextRequest, NextResponse } from 'next/server';

const RAG_API_BASE_URL = process.env.NEXT_PUBLIC_RAG_API_SERVER || 'http://localhost:5600';

export async function GET() {
  try {
    console.log('[RAG API] 데이터셋 목록 요청 시작');
    console.log('[RAG API] RAG_API_BASE_URL:', RAG_API_BASE_URL);
    console.log('[RAG API] 요청 URL:', `${RAG_API_BASE_URL}/datasets/list`);
    
    const response = await fetch(`${RAG_API_BASE_URL}/datasets/list`);
    console.log('[RAG API] 응답 상태:', response.status, response.statusText);
    
    if (!response.ok) {
      console.error('[RAG API] HTTP 오류:', response.status, response.statusText);
      const errorText = await response.text();
      console.error('[RAG API] 오류 내용:', errorText);
      
      return NextResponse.json(
        { 
          success: false, 
          error: `RAG 서버 오류: ${response.status} ${response.statusText}`,
          details: errorText
        },
        { status: response.status }
      );
    }
    
    const data = await response.json();
    console.log('[RAG API] 성공적으로 데이터 받음:', Object.keys(data));
    
    return NextResponse.json(data);
  } catch (error) {
    console.error('[RAG API] 데이터셋 목록 호출 실패:', error);
    console.error('[RAG API] 오류 타입:', error instanceof Error ? error.name : typeof error);
    console.error('[RAG API] 오류 메시지:', error instanceof Error ? error.message : String(error));
    
    // 네트워크 연결 오류 체크
    const isNetworkError = error instanceof Error && (
      error.message.includes('ECONNREFUSED') ||
      error.message.includes('ENOTFOUND') ||
      error.message.includes('ETIMEDOUT') ||
      error.message.includes('fetch failed')
    );
    
    return NextResponse.json(
      { 
        success: false, 
        error: isNetworkError 
          ? `RAG 서버(${RAG_API_BASE_URL})에 연결할 수 없습니다. 서버가 실행 중인지 확인하세요.`
          : '데이터셋 목록을 불러올 수 없습니다.',
        details: error instanceof Error ? error.message : String(error)
      },
      { status: isNetworkError ? 503 : 500 }
    );
  }
}

export async function POST(request: NextRequest) {
  try {
    const contentType = request.headers.get('content-type');
    
    // FormData (파일 업로드)인 경우
    if (contentType?.includes('multipart/form-data')) {
      const formData = await request.formData();
      const method = formData.get('method') as string;
      
      if (method === 'file-upload') {
        // 파일 업로드 방식
        const file = formData.get('file') as File;
        const name = formData.get('name') as string;
        const description = formData.get('description') as string;
        
        if (!file || !name) {
          return NextResponse.json(
            { success: false, error: '파일과 이름은 필수입니다.' },
            { status: 400 }
          );
        }
        
        // 파일을 RAG 서버로 전송
        const ragFormData = new FormData();
        ragFormData.append('file', file);
        ragFormData.append('name', name);
        ragFormData.append('description', description || '');
        
        const response = await fetch(`${RAG_API_BASE_URL}/datasets/from-file`, {
          method: 'POST',
          body: ragFormData
        });
        
        const data = await response.json();
        return NextResponse.json(data);
        
      } else if (method === 'document-llm') {
        // 문서 + LLM 방식
        const document = formData.get('document') as File;
        const name = formData.get('name') as string;
        const description = formData.get('description') as string;
        
        if (!document || !name) {
          return NextResponse.json(
            { success: false, error: '문서와 이름은 필수입니다.' },
            { status: 400 }
          );
        }
        
        const ragFormData = new FormData();
        ragFormData.append('document', document);
        ragFormData.append('name', name);
        ragFormData.append('description', description || '');
        
        // 프로바이더 정보 추가
        const provider = formData.get('provider') as string;
        const model = formData.get('model') as string;
        if (provider) ragFormData.append('provider', provider);
        if (model) ragFormData.append('model', model);
        
        // 사용자 지정 QA 개수 추가
        const targetQACount = formData.get('target_qa_count') as string;
        if (targetQACount) ragFormData.append('target_qa_count', targetQACount);
        
        const response = await fetch(`${RAG_API_BASE_URL}/datasets/from-document`, {
          method: 'POST',
          body: ragFormData
        });
        
        const data = await response.json();
        return NextResponse.json(data);
      }
    } else {
      // JSON 데이터인 경우
      let body;
      try {
        const text = await request.text();
        console.log('Raw request body:', text);
        
        if (!text.trim()) {
          return NextResponse.json(
            { success: false, error: '요청 본문이 비어있습니다.' },
            { status: 400 }
          );
        }
        
        body = JSON.parse(text);
      } catch (parseError) {
        console.error('JSON 파싱 오류:', parseError);
        return NextResponse.json(
          { success: false, error: '유효하지 않은 JSON 형식입니다.' },
          { status: 400 }
        );
      }
      
      const method = body.method;
      
      if (method === 'manual-input') {
        // 수동 입력 방식 - 빈 데이터셋 생성
        const response = await fetch(`${RAG_API_BASE_URL}/datasets/create-empty`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            name: body.name,
            description: body.description || ''
          })
        });
        
        const data = await response.json();
        return NextResponse.json(data);
        
      } else if (method === 'template-based') {
        // 템플릿 기반 방식
        const response = await fetch(`${RAG_API_BASE_URL}/datasets/from-template`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify(body)
        });
        
        const data = await response.json();
        return NextResponse.json(data);
        
      } else {
        // 기본값: RAG 데이터셋 생성
        const response = await fetch(`${RAG_API_BASE_URL}/datasets/from-rag`, {
          method: 'POST',
          headers: { 'Content-Type': 'application/json' },
          body: JSON.stringify({
            ...body,
            provider: body.provider || 'ollama',
            model: body.model || 'hamonize:latest'
          })
        });
        
        const data = await response.json();
        return NextResponse.json(data);
      }
    }
    
    return NextResponse.json(
      { success: false, error: '지원하지 않는 요청 형식입니다.' },
      { status: 400 }
    );
    
  } catch (error) {
    console.error('데이터셋 생성 실패:', error);
    return NextResponse.json(
      { success: false, error: '데이터셋을 생성할 수 없습니다.' },
      { status: 500 }
    );
  }
}

export async function DELETE(request: NextRequest) {
  try {
    const body = await request.json();
    const { datasetId } = body;
    
    if (!datasetId) {
      return NextResponse.json(
        { success: false, error: '데이터셋 ID가 필요합니다.' },
        { status: 400 }
      );
    }
    
    // RAG 서버에 삭제 요청 (올바른 DELETE 메서드 사용)
    const response = await fetch(`${RAG_API_BASE_URL}/datasets/${datasetId}`, {
      method: 'DELETE'
    });
    
    const result = await response.json();
    
    if (response.ok && result.success) {
      return NextResponse.json({
        success: true,
        message: '데이터셋이 성공적으로 삭제되었습니다.'
      });
    } else {
      return NextResponse.json(
        { success: false, error: result.error || '데이터셋 삭제에 실패했습니다.' },
        { status: response.status || 500 }
      );
    }
    
  } catch (error) {
    console.error('데이터셋 삭제 실패:', error);
    return NextResponse.json(
      { success: false, error: '데이터셋 삭제 중 오류가 발생했습니다.' },
      { status: 500 }
    );
  }
} 