import { useState, useEffect } from 'react';
import { externalApiClient as apiClient } from '@/lib/apiClient';
import type { NodeConfigProps, Provider, Model } from './types';
import type { ProcessType } from '@/types/flowai';

export default function ProcessNodeConfig({ node, onNodeUpdate }: NodeConfigProps) {
  const [localConfig, setLocalConfig] = useState(node.data.config || {});
  const [localData, setLocalData] = useState(node.data);
  const [providers, setProviders] = useState<Provider[]>([]);
  const [availableModels, setAvailableModels] = useState<Model[]>([]);
  const [isLoadingProviders, setIsLoadingProviders] = useState(false);
  const [isLoadingModels, setIsLoadingModels] = useState(false);

  const updateConfig = (key: string, value: any) => {
    const newConfig = { ...localConfig, [key]: value };
    setLocalConfig(newConfig);
    onNodeUpdate(node.id, {
      data: {
        ...localData,
        config: newConfig,
      },
    });
  };

  const updateData = (key: string, value: any) => {
    const newData = { ...localData, [key]: value };
    setLocalData(newData);
    onNodeUpdate(node.id, {
      data: newData,
    });
  };

  // 프로바이더 목록 로딩
  useEffect(() => {
    loadProviders();
  }, []);

  // 프로바이더 변경 시 모델 목록 로딩
  useEffect(() => {
    if (localData.provider) {
      loadModelsForProvider(localData.provider);
    }
  }, [localData.provider]);

  const loadProviders = async () => {
    try {
      setIsLoadingProviders(true);
      const response = await apiClient.getProviders();
      
      if (response.success) {
        const data = response.data;
        const providersFromData = data ? Object.keys(data).map(key => ({
          key: key,
          name: data[key].name || key.charAt(0).toUpperCase() + key.slice(1),
          requiresApiKey: data[key].requiresApiKey || false,
          apiKeyConfigured: data[key].apiKeyConfigured !== false,
          isDynamic: data[key].isDynamic || false,
          isAvailable: true,
          models: []
        })) : [];
        
        setProviders(providersFromData);
        
        if (localData.provider) {
          loadModelsForProvider(localData.provider);
        }
      }
    } catch (error) {
      console.error('프로바이더 로딩 실패:', error);
    } finally {
      setIsLoadingProviders(false);
    }
  };

  const loadModelsForProvider = async (providerKey: string) => {
    try {
      setIsLoadingModels(true);
      const response = await apiClient.getProviderModels(providerKey);
      
      if (response.success) {
        const models = response.data || [];
        setAvailableModels(models);
      } else {
        setAvailableModels([]);
      }
    } catch (error) {
      console.error(`${providerKey} 모델 로딩 실패:`, error);
      setAvailableModels([]);
    } finally {
      setIsLoadingModels(false);
    }
  };

  return (
    <div className="space-y-4">
      <div>
        <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
          처리 타입
        </label>
        <select
          value={localData.processType || ''}
          onChange={(e) => updateData('processType', e.target.value as ProcessType)}
          className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
        >
          <option value="">처리 타입을 선택하세요</option>
          <option value="text-generation">텍스트 생성</option>
          <option value="image-generation">이미지 생성</option>
          <option value="code-generation">코드 생성</option>
          <option value="document-analysis">문서 분석</option>
          <option value="web-search">웹 검색</option>
          <option value="rag-search">RAG 검색</option>
          <option value="data-transform">데이터 변환</option>
          <option value="api-call">API 호출</option>
        </select>
      </div>

      {(localData.processType === 'text-generation' || 
        localData.processType === 'code-generation' ||
        localData.processType === 'document-analysis') && (
        <>
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              프로바이더
            </label>
            <select
              key={`provider-${localData.provider || 'none'}`}
              value={localData.provider || ''}
              onChange={(e) => {
                const newProvider = e.target.value;
                const newData = { ...localData, provider: newProvider, model: '' };
                setLocalData(newData);
                onNodeUpdate(node.id, { data: newData });
                setAvailableModels([]);
                if (newProvider) {
                  loadModelsForProvider(newProvider);
                }
              }}
              className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
              disabled={isLoadingProviders}
            >
              <option value="">프로바이더를 선택하세요</option>
              {isLoadingProviders ? (
                <option>로딩 중...</option>
              ) : (
                providers.map((provider) => (
                  <option key={provider.key} value={provider.key}>
                    {provider.name}
                  </option>
                ))
              )}
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              모델
            </label>
            <select
              value={localData.model || ''}
              onChange={(e) => updateData('model', e.target.value)}
              className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
              disabled={isLoadingModels || !localData.provider}
            >
              <option value="">모델을 선택하세요</option>
              {isLoadingModels ? (
                <option>로딩 중...</option>
              ) : (
                availableModels.map((model) => (
                  <option key={model.id} value={model.id}>
                    {model.name || model.id}
                  </option>
                ))
              )}
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              프롬프트
            </label>
            <textarea
              value={localData.prompt || ''}
              onChange={(e) => updateData('prompt', e.target.value)}
              placeholder="AI에게 전달할 지시사항을 작성하세요"
              className="w-full px-3 py-2 bg-transparent border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150 h-32 resize-none"
            />
          </div>

          <div className="grid grid-cols-2 gap-4">
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                Temperature
              </label>
              <input
                type="range"
                min="0"
                max="2"
                step="0.1"
                value={localData.temperature || 0.7}
                onChange={(e) => updateData('temperature', parseFloat(e.target.value))}
                className="w-full"
              />
              <span className="text-xs text-gray-500 dark:text-gray-400">
                {localData.temperature || 0.7}
              </span>
            </div>
            
            <div>
              <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
                최대 토큰
              </label>
              <input
                type="number"
                value={localData.maxTokens || 2000}
                onChange={(e) => updateData('maxTokens', parseInt(e.target.value))}
                min="100"
                max="8000"
                className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
              />
            </div>
          </div>
        </>
      )}

      {localData.processType === 'image-generation' && (
        <>
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              이미지 크기
            </label>
            <select
              value={localConfig.size || '1024x1024'}
              onChange={(e) => updateConfig('size', e.target.value)}
              className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
            >
              <option value="256x256">256×256</option>
              <option value="512x512">512×512</option>
              <option value="1024x1024">1024×1024</option>
              <option value="1024x1792">1024×1792</option>
              <option value="1792x1024">1792×1024</option>
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              이미지 프롬프트
            </label>
            <textarea
              value={localData.prompt || ''}
              onChange={(e) => updateData('prompt', e.target.value)}
              placeholder="생성할 이미지에 대한 설명을 작성하세요"
              className="w-full px-3 py-2 bg-transparent border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150 h-24 resize-none"
            />
          </div>
        </>
      )}

      {localData.processType === 'web-search' && (
        <>
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              검색엔진
            </label>
            <select
              value={localConfig.searchEngine || 'google'}
              onChange={(e) => updateConfig('searchEngine', e.target.value)}
              className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
            >
              <option value="google">Google</option>
              <option value="bing">Bing</option>
              <option value="duckduckgo">DuckDuckGo</option>
            </select>
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              최대 결과 수
            </label>
            <input
              type="number"
              value={localConfig.maxResults || 10}
              onChange={(e) => updateConfig('maxResults', parseInt(e.target.value))}
              min="1"
              max="50"
              className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
            />
          </div>
        </>
      )}

      {localData.processType === 'api-call' && (
        <>
          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              API URL
            </label>
            <input
              type="url"
              value={localData.apiEndpoint || ''}
              onChange={(e) => updateData('apiEndpoint', e.target.value)}
              placeholder="https://api.example.com/endpoint"
              className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
            />
          </div>

          <div>
            <label className="block text-sm font-medium text-gray-700 dark:text-gray-300 mb-2">
              HTTP 메서드
            </label>
            <select
              value={localConfig.method || 'GET'}
              onChange={(e) => updateConfig('method', e.target.value)}
              className="w-full px-3 py-2 bg-white dark:bg-gray-800 border border-gray-200 dark:border-gray-700 rounded-xl text-gray-900 dark:text-gray-100 focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-all duration-150"
            >
              <option value="GET">GET</option>
              <option value="POST">POST</option>
              <option value="PUT">PUT</option>
              <option value="DELETE">DELETE</option>
              <option value="PATCH">PATCH</option>
            </select>
          </div>
        </>
      )}

      <div className="flex flex-col space-y-3">
        <label className="flex items-center cursor-pointer">
          <input
            type="checkbox"
            checked={localData.ragEnabled || false}
            onChange={(e) => updateData('ragEnabled', e.target.checked)}
            className="w-4 h-4 text-blue-600 bg-gray-100 dark:bg-gray-700 border-gray-300 dark:border-gray-600 rounded focus:ring-blue-500 focus:ring-2 mr-3"
          />
          <span className="text-sm font-medium text-gray-700 dark:text-gray-300">RAG 검색 활성화</span>
        </label>
        
        <label className="flex items-center cursor-pointer">
          <input
            type="checkbox"
            checked={localData.webSearchEnabled || false}
            onChange={(e) => updateData('webSearchEnabled', e.target.checked)}
            className="w-4 h-4 text-blue-600 bg-gray-100 dark:bg-gray-700 border-gray-300 dark:border-gray-600 rounded focus:ring-blue-500 focus:ring-2 mr-3"
          />
          <span className="text-sm font-medium text-gray-700 dark:text-gray-300">웹 검색 활성화</span>
        </label>
      </div>
    </div>
  );
}