import { checkPageAccess } from './auth.js';

export async function helpPage(chatMessages, translations, reqAPI, getText) {
    // 페이지 접근 권한 체크
    const hasAccess = await checkPageAccess(reqAPI, 'help');
    if (!hasAccess) {
        return;
    }

    // 입력 컨테이너 숨기기
    const inputContainer = document.querySelector('.input-container');
    if (inputContainer) {
        inputContainer.style.display = 'none';
    }

    try {
        // marked 라이브러리 로드
        if (typeof marked === 'undefined') {
            await new Promise((resolve, reject) => {
                const script = document.createElement('script');
                script.src = './node_modules/marked/marked.min.js';
                script.onload = resolve;
                script.onerror = () => reject(new Error('Failed to load marked library'));
                document.head.appendChild(script);
            });
        }

        // help.md 파일 내용 가져오기
        const guideContent = await reqAPI('readGuide');
        if (!guideContent) {
            throw new Error('Failed to load help content');
        }
        
        // 버전 정보 가져오기
        const vinfo = await reqAPI('versioninfo');
        const versionText = vinfo?.client || '';
        
        // 버전 정보를 제목에 추가
        const contentWithVersion = guideContent.replace(
            /^# 📚 AI.RUN*$/m,
            `# 📚 AI.RUN (v${versionText})`
        );
        
        // marked 설정
        marked.setOptions({
            breaks: true,
            gfm: true,
            headerIds: true,
            mangle: false
        });

        // 컨테이너에 help-mode 클래스 추가
        chatMessages.classList.add('help-mode');

        chatMessages.innerHTML = `
            <div class="help-container">
                <div class="help-header">
                    <h1 class="help-title">${translations?.help?.title || 'Help'}</h1>
                    <p class="help-description">${translations?.help?.description || 'Help guide for using the application'}</p>
                </div>
                <div class="help-content">
                    <div class="markdown-body">
                        ${marked.parse(contentWithVersion)}
                    </div>
                </div>
            </div>
        `;

        // 앵커 링크 처리
        const links = chatMessages.querySelectorAll('a[href^="#"]');
        links.forEach(link => {
            link.addEventListener('click', (e) => {
                e.preventDefault();
                const targetId = link.getAttribute('href').slice(1);
                const targetElement = document.getElementById(targetId);
                if (targetElement) {
                    targetElement.scrollIntoView({ behavior: 'smooth' });
                }
            });
        });
    } catch (error) {
        console.error('Error loading help content:', error);
        // error.message가 객체인 경우를 처리
        let errorMessage = error?.message;
        if (typeof errorMessage === 'object') {
            errorMessage = JSON.stringify(errorMessage);
        }
        chatMessages.innerHTML = `
            <div class="help-container">
                <div class="error-message">
                    ${getText('help', 'loadError')}<br>
                    <small>${errorMessage || 'Unknown error'}</small>
                </div>
            </div>
        `;
    }
} 